/*************************************************************************
 *
 *  $RCSfile: treecache.cxx,v $
 *
 *  $Revision: 1.76 $
 *
 *  last change: $Author: jb $ $Date: 2001/12/07 18:16:58 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <stdio.h>

#include "treecache.hxx"

#ifndef CONFIGMGR_TREEDATA_HXX
#include "treedata.hxx"
#endif
#ifndef CONFIGMGR_DISPOSETIMER_HXX
#include "disposetimer.hxx"
#endif
#ifndef CONFIGMGR_CACHEWRITESCHEDULER_HXX
#include "cachewritescheduler.hxx"
#endif
#ifndef CONFIGMGR_TIMESTAMP_HXX
#include "timestamp.hxx"
#endif
#ifndef CONFIGMGR_LOADER_HXX
#include "loader.hxx"
#endif
#ifndef CONFIGMGR_NOTIFYCALLBACK_HXX
#include "notifycallback.hxx"
#endif
#ifndef CONFIGMGR_TREEBUILDERCALLBACK_HXX
#include "treebuildercallback.hxx"
#endif
#ifndef _CONFIGMGR_SESSION_CONFIGSESSION_HXX_
#include "configsession.hxx"
#endif
#ifndef _CONFIGMGR_TREEACCESS_HXX_
#include "treeaccess.hxx"
#endif
#ifndef _CONFIGMGR_SYNCHRONIZE_HXX_
#include "synchronize.hxx"
#endif
#ifndef CONFIGMGR_XMLFORMATER_HXX
#include "xmlformater.hxx"
#endif
#ifndef _CONFIGMGR_SERVER_VERSION_HXX_
#include "server_version.hxx"
#endif

#ifndef _CONFIGMGR_TREEACTIONS_HXX_
#include "treeactions.hxx"
#endif
#ifndef CONFIGMGR_LOCALIZEDTREEACTIONS_HXX
#include "localizedtreeactions.hxx"
#endif
#ifndef CONFIGMGR_UPDATEHELPER_HXX
#include "updatehelper.hxx"
#endif

#ifndef _CONFIGMGR_TREEACCESS_HXX_
#include "treeaccess.hxx"
#endif
#ifndef CONFIGMGR_NODEPART_HXX
#include "nodepart.hxx"
#endif
#ifndef CONFIGMGR_CONFIGEXCEPT_HXX_
#include "configexcept.hxx"
#endif
#ifndef INCLUDED_CONFIGMGR_MERGECHANGE_HXX
#include "mergechange.hxx"
#endif

#ifndef _CONFIGMGR_TRACER_HXX_
#include "tracer.hxx"
#endif

#ifndef _OSL_DIAGNOSE_H_
#include <osl/diagnose.h>
#endif

#ifndef _COM_SUN_STAR_LANG_XUNOTUNNEL_HPP_
#include <com/sun/star/lang/XUnoTunnel.hpp>
#endif
#ifndef _COM_SUN_STAR_LANG_DISPOSEDEXCEPTION_HPP_
#include <com/sun/star/lang/DisposedException.hpp>
#endif
#ifndef _COM_SUN_STAR_CONTAINER_NOSUCHELEMENTEXCEPTION_HPP_
#include <com/sun/star/container/NoSuchElementException.hpp>
#endif

#ifndef _RTL_LOGFILE_HXX_
#include <rtl/logfile.hxx>
#endif


#ifndef INCLUDED_SET
#include <set>
#define INCLUDED_SET
#endif

#ifndef INCLUDED_ALGORITHM
#include <algorithm>
#define INCLUDED_ALGORITHM
#endif

#ifndef MSC_MSG_HXX
#include<msc_msg.hxx>
#endif

/*	current response formats (02.07.00)

		REQ				REPONSE					ERROR_RESPONSE
	open/getNode		<Tree>					"error"
	update				"success"				<ChangesTree>
	<Notification>		<ChangesTree>			n.a.
*/

namespace configmgr
{

    namespace uno = ::com::sun::star::uno;
    namespace lang= ::com::sun::star::lang;
    namespace sax = ::com::sun::star::xml::sax;
    namespace container = ::com::sun::star::container;
    using uno::Any;
    using uno::XInterface;
    using uno::Reference;
    using uno::Sequence;
    using lang::XMultiServiceFactory;

    namespace Path = configuration::Path;
    using configuration::AbsolutePath;
// =========================================================================

//==========================================================================
//= TreeOnDemand
//==========================================================================

TreeOnDemand::TreeOnDemand()
{
}
// -------------------------------------------------------------------------

TreeOnDemand::~TreeOnDemand()
{
}
// -------------------------------------------------------------------------

void TreeOnDemand::makeTree()
{
	osl::MutexGuard aGuard(this->mutex);
	if (!m_pTree.get())
		m_pTree.reset( new Tree );
}
// -------------------------------------------------------------------------

// -------------------------------------------------------------------------

//==========================================================================
//= TreeManager
//==========================================================================
		
OTreeDisposeScheduler* TreeManager::createDisposer(vos::ORef<OOptions> const&)
{
	// TODO: (_xOptions) retrieve the values from the options and/or m_xDefaultOptions
#ifndef _DEBUG
	const sal_uInt32 minute			= 60; // units are seconds
	// for initial debugging: use seconds instead of minutes
#else
	const sal_uInt32 minute			= 1;
#endif

	const sal_uInt32 c_nDefaultDelay	= 15 * minute; 
	const sal_uInt32 c_nDefaultInterval =  1 * minute;

	TimeInterval aDelay(c_nDefaultDelay);
	TimeInterval aInterval(c_nDefaultInterval);	

	return new OTreeDisposeScheduler(*this, aDelay, aInterval);
}

// -----------------------------------------------------------------------------
OCacheWriteScheduler* TreeManager::createCacheWriter(vos::ORef<OOptions> const&)
{
	const sal_uInt32 seconds = 1;
	const sal_uInt32 c_nDefaultInterval =  2 * seconds;

	TimeInterval aInterval(c_nDefaultInterval);	
	return new OCacheWriteScheduler(*this, aInterval);
}
// -------------------------------------------------------------------------
TreeInfo* TreeManager::requestTreeInfo(vos::ORef<OOptions> const& _xOptions, bool bCreate)
{
	// now locked!!! (was not locked)
	osl::MutexGuard aGuard(m_aTreeListMutex);

	TreeInfo* pInfo = NULL;
	TreeList::iterator i = m_aTreeList.find(_xOptions);
	if (i == m_aTreeList.end())	
	{
		if (bCreate)
		{
			if (m_pDisposer == NULL)
				m_pDisposer = createDisposer(m_xDefaultOptions);

			if (m_pCacheWriter == NULL)
				m_pCacheWriter = createCacheWriter(m_xDefaultOptions);

			CFG_TRACE_INFO("TreeManager: Adding new TreeInfo for user '%s' with locale '%s'", 
							OUSTRING2ASCII(_xOptions->getUser()), OUSTRING2ASCII(_xOptions->getLocale()) );

			pInfo = new TreeInfo(*m_pDisposer, ConfigChangeBroadcaster::newBroadcastHelper());
			m_aTreeList[_xOptions] = pInfo;
		}		
	}
	else
	{
		pInfo = (*i).second;
		OSL_ENSURE( pInfo != 0, "ERROR: TreeManager: Found Entry for NULL tree info");

		// returning non-null also guarantees that we have a OTreeDisposeScheduler:
		OSL_ENSURE( m_pDisposer != 0, "ERROR: TreeManager: Found TreeInfo but have no disposer");
	}
	return pInfo;
}

// disposing
// -------------------------------------------------------------------------
void TreeManager::disposeAll()
{	
	CFG_TRACE_INFO("TreeManager: Disposing all data" );			
	TreeList aReleaseList;
	
	if (m_pDisposer) 
	{
		osl::MutexGuard aShotGuard(m_pDisposer->getShotMutex());
		osl::MutexGuard aGuard(m_aTreeListMutex);			
		m_pDisposer->stopAndClearTasks();
		aReleaseList.swap(m_aTreeList);			 // move data out of m_aTreeList and empty m_aTreeList		
	}
	OSL_ASSERT(m_aTreeList.empty());			

	// free all the trees - not exception safe !! (i.e. disposeBroadcastHelper() must not throw)
	for (TreeList::iterator i = aReleaseList.begin(); i != aReleaseList.end(); ++i)
	{
		ConfigChangeBroadcaster::disposeBroadcastHelper((*i).second->pBroadcastHelper);		
		delete (*i).second;
		(*i).second = NULL;
	}	
}

// -------------------------------------------------------------------------
void TreeManager::dispose()
{
	CFG_TRACE_INFO("TreeManager: dispoing the treemanager" );		
	OWriteSynchronized aGuard(this);

	RTL_LOGFILE_CONTEXT_AUTHOR(aLog, "configmgr::TreeManager", "jb99855", "dispose(), disable lazy write cache.");
	m_bDisposeMode = true;						 // we are in dispose, handling of errors must be something different.

	// writing of pending updates
	disableAsync();
	
	// cleaning the cache
	disposeAll();		
}

// -------------------------------------------------------------------------
void TreeManager::disposeOne(vos::ORef<OOptions> const& _xOptions, bool bFlushUpdates)
{
	osl::ClearableMutexGuard aGuard(m_aTreeListMutex);
		
	CFG_TRACE_INFO("TreeManager: Disposing data and TreeInfo for user '%s' with locale '%s'", 
					OUSTRING2ASCII(_xOptions->getUser()), OUSTRING2ASCII(_xOptions->getLocale()) );

	m_pDisposer->clearTasks(_xOptions);
	if (!m_pCacheWriter->clearTasks(_xOptions)) // had no pending updates
		bFlushUpdates = false;

	else if (!bFlushUpdates)
		CFG_TRACE_WARNING_NI("Found orphaned Changes in the cache - Discarding.");

	TreeList::iterator i = m_aTreeList.find(_xOptions);
	if (i != m_aTreeList.end())
	{
		std::auto_ptr<TreeInfo> pDisposeInfo(i->second );		
		m_aTreeList.erase(i);

		OSL_ASSERT(pDisposeInfo.get() != NULL);
		if (pDisposeInfo.get() != NULL)
		{
			// got it out of reachability - now dispose/notify without lock
			aGuard.clear();			
			implDisposeOne(pDisposeInfo, _xOptions, bFlushUpdates);
		}
	}
	else
		CFG_TRACE_INFO_NI("- No affected TreeInfo found" );
}

// -------------------------------------------------------------------------
void TreeManager::disposeUser(vos::ORef<OOptions> const& _xUserOptions, bool bFlushUpdates)
{
	//OWriteSynchronized aReadGuard(this);
	osl::ClearableMutexGuard aGuard(m_aTreeListMutex);
	
	CFG_TRACE_INFO("TreeManager: Disposing data and TreeInfo(s) for user '%s'", 
					OUSTRING2ASCII(_xUserOptions->getUser()) );

	typedef std::vector< std::pair< vos::ORef<OOptions>, TreeInfo* > > DisposeList;
	
	DisposeList aDisposeList;
	// collect the ones to dispose
	{
		OUString sUser = _xUserOptions->getUser();
		OSL_ASSERT(sUser.getLength());

		// This depends on the fact that Options are sorted (by struct ltOptions)
		// so that all options belonging to one user are together
		// (and that options with only a user set, sort first)

		// find the lower_bound of all options for the user
		TreeList::iterator const aFirst = m_aTreeList.lower_bound(_xUserOptions);

		// find the upper_bound of all options for the user (using the lower one)
		TreeList::iterator aLast = aFirst;
		while (aLast != m_aTreeList.end() && aLast->first->getUser() == sUser)
			++aLast;

		if (aFirst != aLast)
		{
			aDisposeList.reserve( std::distance(aFirst, aLast) );

			bool bHasPendingChanges = false;

			for (TreeList::iterator it = aFirst; it != aLast; ++it)
			{
				CFG_TRACE_INFO_NI("- Found TreeInfo for locale '%s'", OUSTRING2ASCII(it->first->getLocale()) );
				m_pDisposer->clearTasks(it->first);

				if (m_pCacheWriter->clearTasks(it->first))
					bHasPendingChanges = true;

				OSL_ASSERT(it->second != NULL);
				if (it->second != NULL)
				{
					//aDisposeList.push_back( *it ); 
					aDisposeList.push_back( std::make_pair(it->first,it->second) );
				}
			}

			m_aTreeList.erase(aFirst, aLast);

			if (!bHasPendingChanges) 
				bFlushUpdates = false;

			else if (!bFlushUpdates)
				CFG_TRACE_WARNING_NI("Found orphaned Changes in the cache - Discarding.");
		}
		else
			CFG_TRACE_INFO_NI("- No affected TreeInfo found" );
	}

	// got all out of external reach - now dispose/notify without lock
	aGuard.clear();

	for (DisposeList::iterator i = aDisposeList.begin(); i != aDisposeList.end(); ++i)
	{
		std::auto_ptr<TreeInfo> pDisposeInfo(i->second );
		i->second = 0;

		OSL_ASSERT(pDisposeInfo.get() != NULL);
		implDisposeOne(pDisposeInfo, i->first, bFlushUpdates);
	}
}

// -------------------------------------------------------------------------
void TreeManager::implDisposeOne(std::auto_ptr<TreeInfo> pDisposeInfo, vos::ORef<OOptions> const& _xOptions, bool bFlushUpdates)
{
	CFG_TRACE_INFO("Now removing TreeInfo (user '%s' with locale '%s') and broadcaster", 
					OUSTRING2ASCII(_xOptions->getUser()), OUSTRING2ASCII(_xOptions->getLocale()) );

	if (bFlushUpdates) try
	{
		CFG_TRACE_INFO_NI("- Flushing pending changes" );

		pDisposeInfo->syncPending(_xOptions,*this);
	}
    catch (uno::Exception& e)
    {
		CFG_TRACE_ERROR_NI("- Failed with exception (ignoring here)" );
    }
	ConfigChangeBroadcaster::disposeBroadcastHelper(pDisposeInfo->pBroadcastHelper);
	
	TreeInfo::DisposeList aDisposedList;
	pDisposeInfo->clearTree(aDisposedList);

	CFG_TRACE_INFO_NI("- Got %d module trees to dispose",int(aDisposedList.size())); 
		
	Sequence< OUString > aCloseList = TreeInfo::collectNodeIds(aDisposedList);

	if (aCloseList.getLength() > 0)
	{
		CFG_TRACE_INFO_NI("- Closing %d NodeIds at the session",int(aCloseList.getLength())); 
		this->closeNodes(aCloseList,_xOptions);
	}
}

// ConfigChangeBroadcaster
// -------------------------------------------------------------------------
ConfigChangeBroadcastHelper* TreeManager::getBroadcastHelper(vos::ORef<OOptions> const& _xOptions, bool bCreate)
{
	TreeInfo* pTreeInfo = requestTreeInfo(_xOptions, bCreate);
	return pTreeInfo ? pTreeInfo->pBroadcastHelper : 0;	
}


// -------------------------------------------------------------------------
TreeManager::TreeManager(IConfigSession* _pSession, const vos::ORef<OOptions>& _xOptions) 
		:m_refCount(0),
		 m_pSession(_pSession),
		 m_xDataReqCb(),
		 m_xDefaultOptions(_xOptions),
		 m_pLock(new OTreeAccessor),
		 m_pDisposer(NULL),
		 m_pCacheWriter(NULL),
		 m_bLazyWritePossible(true),
		 m_bDisposeMode(false)
{
}

// -------------------------------------------------------------------------
TreeManager::~TreeManager()
{
	OSL_ASSERT(m_refCount == 0);
	OSL_ENSURE(m_bDisposeMode == true, "TreeManager::dispose() wasn't called, something could went wrong.");

	delete m_pDisposer;
	delete m_pCacheWriter;
	delete m_pLock;
}

// -------------------------------------------------------------------------
std::auto_ptr<ISubtree> TreeManager::loadRemoteTemplate(const AbsolutePath& _rNodePath, 
													  const vos::ORef < OOptions >& _xOptions,
													  StatusInfo& _rStatus)
{
	OSL_ASSERT(m_pSession && m_pSession->asIUpdateProvider());

	OSL_ENSURE(0 == _rNodePath.getModuleName().toString().compareToAscii(REQUEST_TEMPLATE_DETECTION),
                "Template path must specify the template pseudo-module");

	::vos::ORef< TreeBuilderCallback > xBuilder = new TreeBuilderCallback( TreeBuilderCallback::NoModule(),_xOptions);

	// set the level parameter so it says "all levels", cause this is represented by different
	// numbers inside and outside the session
	sal_Int16 const	c_nLevels = IConfigSession::COMPLETE_SUBTREE;
	
	m_pSession->asIUpdateProvider()->openNode(_rNodePath, _xOptions, c_nLevels, xBuilder.getBodyPtr());

	std::auto_ptr<ISubtree> pReturn = xBuilder->waitToResponse(m_pSession->getMasterTimeout());
	_rStatus = xBuilder->getCallResult();

	// goLocalize(_rNodePath, _nLevel, _rStatus, *pReturn);

	return pReturn;
}

// -------------------------------------------------------------------------
void TreeManager::closeNodes(const Sequence< OUString >& _rNodeIds, 
							 const vos::ORef < OOptions >& /*_xOptions*/)
{
	if (IUpdateProvider* pNodeProvider = m_pSession->asIUpdateProvider())
	try
	{
		for (sal_Int32 i = 0; i < _rNodeIds.getLength(); ++i)
			pNodeProvider->closeNode( _rNodeIds[i], 0 );
	}
	catch (uno::Exception&)
	{
		// these exceptions are ignored
	}
}

// -------------------------------------------------------------------------
void SAL_CALL TreeManager::release(  ) throw () 
{
	if (! osl_decrementInterlockedCount( &m_refCount ))
	{
		delete this;
	}
}

// -------------------------------------------------------------------------
static 
std::auto_ptr<ISubtree> reduceSubtreeForLocale(std::auto_ptr<ISubtree> _pSubtree, const vos::ORef < OOptions >& _xOptions)
{
    OSL_ENSURE(!_pSubtree.get() || !isLocalizedValueSet(*_pSubtree), "Unexpected node. Expecting a subtree, Found a single localized value.");

    std::auto_ptr<ISubtree> aRet;

    std::auto_ptr<INode> aReduced = reduceExpandedForLocale(_pSubtree, _xOptions->getLocale());

    if (aReduced.get())
    {
        if (ISubtree* pReduced =aReduced->asISubtree())
        {
            aRet.reset(pReduced);
            aReduced.release();
        }
        else
        {
            OSL_ENSURE(false, "Tree unexpectedly reduced to non-tree");
        }
    }
    else
        OSL_ENSURE(!_pSubtree.get(), "Tree unexpectedly reduced to nothing");

    return aRet;
}
// -------------------------------------------------------------------------
ISubtree* TreeManager::requestSubtree(AbsolutePath const& aSubtreePath, 
									  const vos::ORef < OOptions >& _xOptions,
									  sal_Int16 nMinLevels)  CFG_UNO_THROW_ALL(  )
{
	OReadSynchronized aReadGuard(this);
	
	TreeInfo* pInfo = requestTreeInfo(_xOptions,true /*create TreeInfo*/);
	OSL_ENSURE(pInfo, "Could not create TreeInfo");

	
	CFG_TRACE_INFO("cache manager: checking the cache");
	ISubtree* pSubTree = pInfo->acquireSubtreeWithDepth(aSubtreePath, nMinLevels, 0);

//! LLA TEST
//	if (pSubTree)
//	{
//		invalidateTree(pSubTree, m_pSession, aAbsoluteSubtreePath, _xOptions);
//	}
	
	// cache miss ...
	if (!pSubTree)
	{
		CFG_TRACE_INFO_NI("cache manager: cache miss. going to load the node");
		vos::ORef< OTreeLoader > xLoader = pInfo->getLoader(aSubtreePath, nMinLevels, _xOptions, m_pSession);

		OSL_ENSURE(xLoader.getBodyPtr(), "Did not receive a loader for retrieving the node");
		CFG_TRACE_INFO_NI("cache manager: cache miss. going to load the node");
		if (!xLoader.getBodyPtr())
			throw container::NoSuchElementException((::rtl::OUString::createFromAscii("Error while retrieving the node")), NULL);
		
		// start loading
		xLoader->start(this);

		// now block for reading
		try
		{
			std::auto_ptr<ISubtree> pResponse = reduceSubtreeForLocale(xLoader->waitToResponse(),_xOptions);
			if (pResponse.get())
			{
                if (_xOptions->isForcingWritable())
                    forceWritable(*pResponse);

                sal_Int16 nLevels = xLoader->getLoadedLevels();
                sal_Int16 nDefaultLevels = m_pSession->isAutoLoadingDefaults() ? nLevels : 0;

                OSL_ENSURE(nLevels == ALL_LEVELS || (nLevels > nMinLevels && nMinLevels != ALL_LEVELS),
                            "Could not load all required levels");

				aReadGuard.clear();
				OWriteSynchronized aWriteGuard(this);
				pSubTree = pInfo->addSubtree(xLoader->getLoadedPath(), aSubtreePath, pResponse, 
                                             nLevels,nDefaultLevels);
			}
		}
		catch (uno::Exception& )
		{
			pInfo->releaseLoader(xLoader);
			throw;
		}

		pInfo->releaseLoader(xLoader);	
	}
	else
	{
		CFG_TRACE_INFO_NI("cache manager: found node in cache");
		OSL_ENSURE(_xOptions->canUseCache(),"TreeManager: Found node in cache for non-cachable request");
	}

	return pSubTree;
}

// -------------------------------------------------------------------------
void TreeManager::fetchSubtree(AbsolutePath const& aSubtreePath, const vos::ORef < OOptions >& _xOptions, sal_Int16 nMinLevels) CFG_NOTHROW()
{
	TreeInfo* pInfo = requestTreeInfo(_xOptions, true);
	OSL_ENSURE(pInfo, "Could not create TreeInfo");
	
	OReadSynchronized aReadGuard(this);
	CFG_TRACE_INFO("cache manager: checking the cache");
	ISubtree* pSubTree = pInfo->acquireSubtreeWithDepth(aSubtreePath, nMinLevels,0);

	// cache miss ...
	if (!pSubTree)
	{
		CFG_TRACE_INFO_NI("cache manager: cache miss. going to load the node");
		vos::ORef< OTreeLoader > xLoader = pInfo->getLoader(aSubtreePath, nMinLevels, _xOptions, m_pSession, sal_True);

		OSL_ENSURE(xLoader.getBodyPtr(), "Did not receive a loader for retrieving the node");
		CFG_TRACE_INFO_NI("cache manager: cache miss. going to load the node");
		if (xLoader.getBodyPtr())		
			xLoader->start(this);
	}
}

// -------------------------------------------------------------------------
sal_Bool TreeManager::fetchDefaultData(AbsolutePath const& aSubtreePath, 
									    const vos::ORef < OOptions >& _xOptions,
									    sal_Int16 nMinLevels) CFG_UNO_THROW_ALL(  )
{
	TreeInfo* pInfo = requestTreeInfo(_xOptions,false);

    if (!pInfo)
    {
	    OSL_ENSURE(pInfo, "Could not find TreeInfo for fetchDefaultRequest");
        return false;
    }

	CFG_TRACE_INFO("cache manager: checking the cache for defaults");

	ISubtree* pSubTree = pInfo->acquireSubtreeWithDepth(aSubtreePath, 0, nMinLevels);

	// cache miss ...
	if (!pSubTree)
	{
        if (IDefaultProvider * pDefaultLoader = m_pSession->asIDefaultProvider())
        {
            std::auto_ptr<ISubtree> aDefaultData = reduceSubtreeForLocale(
                                                        pDefaultLoader->requestDefaultData(aSubtreePath,_xOptions,nMinLevels),
                                                        _xOptions);

			if (aDefaultData.get())
			{
                // do we need synchronization here ??
				// OWriteSynchronized aWriteGuard(this);
				pSubTree = pInfo->addSubtree(aSubtreePath, aSubtreePath, aDefaultData, 0,nMinLevels);
			}
		}
        else
		    CFG_TRACE_WARNING_NI("cache manager: cannot load defaults: not supported by session");
	}
	else
	{
		CFG_TRACE_INFO_NI("cache manager: found default data in cache");
	}

	if (pSubTree)
    {
        OSL_VERIFY(pInfo->releaseSubtree(aSubtreePath) > 0);
        return true;
    }
    else    
        return false;
}

// -------------------------------------------------------------------------
std::auto_ptr<ISubtree> TreeManager::requestDefaultData(AbsolutePath const& aSubtreePath, 
										                const vos::ORef < OOptions >& _xOptions,
										                sal_Int16 nMinLevels) CFG_UNO_THROW_ALL(  )
{
    std::auto_ptr<ISubtree> aRet;

    // TODO: Insert check here, if the data is in the cache already
    if (IDefaultProvider * pDefaultLoader = m_pSession->asIDefaultProvider())
    {
        aRet = reduceSubtreeForLocale( 
                    pDefaultLoader->requestDefaultData(aSubtreePath,_xOptions,nMinLevels),
                    _xOptions );
    }
    else
        OSL_ENSURE(false,"The session does not support fetching defaults.");

    return aRet;
}

// -------------------------------------------------------------------------
AbsolutePath TreeManager::completeTemplateLocation(const Name& _rLogicalTemplateName, const Name &_rModule)
{
    using namespace configuration::Path;

	static const 
    Component aTemplateRoot = wrapSimpleName(OUString::createFromAscii("org.openoffice.Templates"));

    Component aTemplateModule = wrapSimpleName(_rModule);
    Component aTemplateName   = wrapSimpleName(_rLogicalTemplateName);

    Path::Rep aResult(aTemplateName);
    aResult.prepend(aTemplateModule);
    aResult.prepend(aTemplateRoot);

	return AbsolutePath(aResult);
}
		
inline Tree& TreeManager::getTemplatesTree()
{
	return m_aTemplates.tree();
}

// -------------------------------------------------------------------------
AbsolutePath TreeManager::ensureTemplate(const Name& _rName, Name const& _rModule) CFG_UNO_THROW_ALL(  )
{
	OSL_ENSURE(!_rName.isEmpty(), "TreeManager::ensureTemplate : invalid template name !");

	CFG_TRACE_INFO("cache manager: going to get a template named %s", OUSTRING2ASCII(_rName.toString()));

//	OReadSynchronized aReadGuard(this);
	osl::MutexGuard aGuard(m_aTemplates.mutex);

	AbsolutePath aTemplateLocation = completeTemplateLocation(_rName, _rModule);

	Tree& aTemplates = getTemplatesTree();
	const INode* pNode = aTemplates.getNode(aTemplateLocation);
	if (!pNode)
	{
		CFG_TRACE_INFO("cache manager: cache miss for that template");
		// at the moment, the XML tree builder is not able to supply leafs, it only handles sub trees.
		// as we don't know if the given path describes a inner node or a leaf, we have to ensure that only 
		// a tree is loaded
		OSL_ENSURE(aTemplateLocation.getDepth() > 1, "TreeManager::ensureTemplate : invalid template location !");
		AbsolutePath aTemplateParent(aTemplateLocation.getParentPath());
		
		vos::ORef<OOptions> xTemplateOptions = new OOptions(*m_xDefaultOptions);
		xTemplateOptions->setMultiLocaleMode();

		if (ITemplateProvider* pTreeProvider = m_pSession->asITemplateProvider())
		{
			::std::auto_ptr<INode> pINode = pTreeProvider->requestTemplateInstance(Name(), _rModule, xTemplateOptions);
			if (pINode.get())
			{
				if (ISubtree* pModuleTree = pINode->asISubtree())
				{
					auto_ptr<ISubtree> aSubtree( pModuleTree );
					pINode.release();
					
					CFG_TRACE_INFO("cache manager: adding the template to the cache");
					
					aTemplates.addSubtree(aTemplateParent, aSubtree, ALL_LEVELS, ALL_LEVELS);
				}
				else
					OSL_ENSURE(false,"Request for template module returned an INode that is no subtree");
			}
		}
		else if (ITreeProvider* pTreeProvider = m_pSession->asITreeProvider())
		{
			std::auto_ptr<ISubtree> pSubTree = pTreeProvider->loadSubtree(aTemplateParent, xTemplateOptions, ALL_LEVELS);
			if (pSubTree.get())
			{
				//	aReadGuard.clear();
				//	OWriteSynchronized aWriteGuard(this);
				// TODO : this is not really safe : between freeing the read guard an acquiring
				// the write guard, some other thread may have inserted the new subtree, too.
				// To avoid this, we would need a write access which could be nested within a
				// read access. 
				// which is a big TODO
				CFG_TRACE_INFO("cache manager: adding the template to the cache");
				aTemplates.addSubtree(aTemplateParent, pSubTree, ALL_LEVELS, ALL_LEVELS);
			}
		}		
		else if (m_pSession->asIUpdateProvider())	// indicates the remote session
		{
			StatusInfo aResult;
			std::auto_ptr<ISubtree> pNewSubTree = loadRemoteTemplate(aTemplateParent, xTemplateOptions, aResult);
			if (pNewSubTree.get())
			{
				//	aReadGuard.clear();
				//	OWriteSynchronized aWriteGuard(this);
				// TODO : this is not really safe : between freeing the read guard an acquiring the write guard, some other
				// thread may have inserted the new subtree, too.
				// To avoid this, we would need a write access which could be nested within a read access.
				// which is a big TODO
				CFG_TRACE_INFO("cache manager: adding the template to the cache");
				aTemplates.addSubtree(aTemplateParent, pNewSubTree, ALL_LEVELS, ALL_LEVELS);
			}			
			if (aResult.nCode)
				throw uno::Exception((::rtl::OUString::createFromAscii("The template description could not be loaded. Error message as got from the session : ")) += aResult.sMessage, NULL);
		}
		else
		{
			OSL_ENSURE(false, "TreeManager::requestSubtree : no implementations for requestSubtree available !");
		}		
	}

    return aTemplateLocation;
}

// -------------------------------------------------------------------------
::std::auto_ptr<INode> TreeManager::requestTemplateInstance(Name const& _rTemplateName, Name const& _rModule, 
															const vos::ORef < OOptions >& _xOptions) CFG_UNO_THROW_ALL(  )
{
//	OReadSynchronized aReadGuard(this);
	osl::MutexGuard aGuard(m_aTemplates.mutex);

	AbsolutePath aTemplateLocation = ensureTemplate(_rTemplateName, _rModule);

	const INode* pTemplate = getTemplatesTree().getNode(aTemplateLocation);
	if (!pTemplate)
		throw uno::Exception(::rtl::OUString::createFromAscii("The template description could not be loaded. An unknown error occured."), NULL);
	
	// #86095# we sometimes wrongly are passed NULL options - using default instead
	//OSL_ENSURE( _xOptions.isValid(), "ERROR: Requesting template instance without options" );
	vos::ORef< OOptions > xOptions( _xOptions.isValid() ? _xOptions : m_xDefaultOptions );

	std::auto_ptr<INode> pResult = cloneExpandedForLocale(pTemplate, xOptions->getLocale());

    if (pResult.get() && xOptions->isForcingWritable())
    {
        forceWritable(*pResult);
    }

    return pResult;
}

// -------------------------------------------------------------------------
namespace
{
    struct ForceWritable : NodeModification
    {
        void handle(ValueNode& _rValue)     { implForceWritable(_rValue); }
        void handle(ISubtree& _rSubtree)    { implForceWritable(_rSubtree); applyToChildren(_rSubtree); }

        void implForceWritable(INode& _rNode) { _rNode.forceWritableToFinalized(); }
	};
}

// -------------------------------------------------------------------------
void TreeManager::forceWritable(INode& _rNode)
{
    ForceWritable aVisitor;

    aVisitor.applyToNode(_rNode);
}
// -------------------------------------------------------------------------
void TreeManager::notifyUpdate(TreeChangeList const& aChangeTree) CFG_UNO_THROW_RTE(  )
{
	ConfigChangeBroadcaster::fireChanges(aChangeTree, false);
}

// -------------------------------------------------------------------------
void TreeManager::fireChanges(TreeChangeList const& aChangeTree, sal_Bool _bError)
{
	ConfigChangeBroadcaster::fireChanges(aChangeTree, _bError);
}

// -------------------------------------------------------------------------
class RemoveEmptySubtrees : public ChangeTreeModification
{
protected:
	typedef ::std::map< ::rtl::OUString, sal_Int32 > LeafNumbers;

	LeafNumbers m_aNumbers;
	sal_Int32	m_nCurrentNum;

public:
	RemoveEmptySubtrees() : m_nCurrentNum(0) { }

	virtual void handle(ValueChange& aValueNode);
	virtual void handle(AddNode& aAddNode);
	virtual void handle(RemoveNode& aRemoveNode);
	virtual void handle(SubtreeChange& aSubtree);
};

// .........................................................................
void RemoveEmptySubtrees::handle(ValueChange& aValueNode)	{ ++m_nCurrentNum; }
void RemoveEmptySubtrees::handle(AddNode& aAddNode)			{ ++m_nCurrentNum; }
void RemoveEmptySubtrees::handle(RemoveNode& aRemoveNode)	{ ++m_nCurrentNum; }

// .........................................................................
void RemoveEmptySubtrees::handle(SubtreeChange& aSubtree)
{
	// push the current state
	LeafNumbers aOldNumbers = m_aNumbers;
	sal_Int32 nOldNum = m_nCurrentNum;

	// initialize the state
	m_aNumbers.clear();
	m_nCurrentNum = 0;

	// handle the children
	aSubtree.forEachChange(*this);

	// remove all subtrees without any leafs
	for (	LeafNumbers::const_iterator aErase = m_aNumbers.begin();
			aErase != m_aNumbers.end();
			++aErase
		)
	{
		if (!aErase->second)
			aSubtree.removeChange(aErase->first);
	}

	// remember the state
	sal_Int32 nMyLeafs = m_nCurrentNum;

	// pop the old state
	m_aNumbers = aOldNumbers;
	m_nCurrentNum = nOldNum;

	// and add my own leaf number to the map of the old state
	m_aNumbers[aSubtree.getNodeName()] += nMyLeafs;
	m_nCurrentNum += nMyLeafs;
}

// -----------------------------------------------------------------------------
void TreeManager::updateTree(TreeChangeList& aChanges) CFG_UNO_THROW_ALL(  )
{
	// no write synchron guarding because the client is responsible	
	try
    {
	    // ---------- preworking on the changes ----------
	    // caller must own a write lock on this
	    CFG_TRACE_INFO("cache manager: updating the tree from a changes list");
	    
	    // normalize the update tree. This means that we want a tree with one root which has either more than one child
	    // or exactly one non-SubtreeChange-child
	    // This prevents us from giving unneccessary items to the session
	    AbsolutePath aRootPath = aChanges.getRootNodePath();
	    SubtreeChange const * pRootChanges = &aChanges.root;

	    // first, kick off all tree parts which are just empty SubtreeChanges (or SubtreeChanges containing only
	    // SubtreeChanges)
	    // As the changes tree we got is constant, we have to build our own hierarchy
	    SubtreeChangeReferrer aReferrer(*pRootChanges);
		    // this is a SubtreeChange, which, in it's dtor, does not delete all it's children, but only the ones which
		    // are SubtreeChanges itself. As it replaces all SubtreeChanges with SubtreeChangeReferrers, this means
		    // that upon destruction of the object, no Changes are deleted.
	    RemoveEmptySubtrees aRemoveDispensables;
	    aReferrer.dispatch(aRemoveDispensables);

	    // now the root of the changes tree is the "referrer" object
	    pRootChanges = &aReferrer;

	    // do the normalizing
	    SubtreeChange::ChildIterator aChildren = pRootChanges->begin();
	    while ((pRootChanges->size() == 1) && (aChildren->ISA(SubtreeChange)))
	    {
            aRootPath = aRootPath.compose( ONameCreator::createName(*aChildren, pRootChanges) );
		    pRootChanges = static_cast<const SubtreeChange*>(&*aChildren);
		    aChildren = pRootChanges->begin();
	    }
	    
	    OChangeActionCounter aChangeCounter;
	    // now count if there are any changes
	    aChangeCounter.handle(*pRootChanges);
	    CFG_TRACE_INFO_NI("cache manager: counted changes for update : additions: %i , removes: %i, value changes: %i", aChangeCounter.nAdds, aChangeCounter.nRemoves, aChangeCounter.nValues);
	    
	    if (!aChangeCounter.hasChanges())
		    return;	
	    
	    TreeInfo* pInfo = requestTreeInfo(aChanges.getOptions(),false);
	    OSL_ENSURE(pInfo, "No local data to update in updateTree");

	    if (!pInfo) throw lang::DisposedException(OUString::createFromAscii("Tree to be updated was already disposed"), NULL);

	    // the mutex prevent calls to insert Pending when on the other hand an other thread
	    // write down the pendings.
	    osl::MutexGuard aGuard(m_aUpdateMutex);

	    pInfo->addPending(aChanges);
	    
	    // merge the changes into the tree
	    pInfo->updateTree(aChanges);

	    // lasy writing
	    bool bLazyWrite = m_bLazyWritePossible && aChanges.getOptions()->getLazyWrite();

        // disable lazy writing, if the node is not cached !
        if (bLazyWrite && !aChanges.getOptions()->canUseCache())
        {
	        CFG_TRACE_WARNING_NI("cache manager: Lazy writing is incompatible with non-cached semantics: Executing synchronously");
            bLazyWrite = false;
        }

	    m_pCacheWriter->scheduleWrite(aChanges.getOptions(), bLazyWrite);
    }
	catch(configuration::Exception& ex)
	{
		configapi::ExceptionMapper e(ex);
		e.unhandled();
	}

}

// -----------------------------------------------------------------------------
void TreeManager::disableAsync()
{
	osl::MutexGuard aShotGuard(m_pCacheWriter->getShotMutex());
	osl::MutexGuard aGuard(m_aUpdateMutex);
	m_bLazyWritePossible = false;
	
	if (m_pCacheWriter)
		m_pCacheWriter->stopAndWriteCache();
}

// -----------------------------------------------------------------------------
void TreeManager::sessionUpdate(vos::ORef< OOptions > const& _xOptions, const AbsolutePath &_aRootPath, SubtreeChange *_pPendingChange) CFG_UNO_THROW_ALL(  )
{
	// this memberfunction is called by an other thread than updateTree(), it's called
	// by an timer, with lazy writing to speed up the update.

	// done in runDisposer() osl::MutexGuard aGuard(m_aUpdateMutex);
	try
	{
		CFG_TRACE_INFO_NI("cache manager: received updates for tree: '%s'", OUSTRING2ASCII(_aRootPath.toString()));

		TreeInfo* pInfo = requestTreeInfo(_xOptions,false);

        TreeChangeList aT(_xOptions, _aRootPath, *_pPendingChange, SubtreeChange::DeepChildCopy());
	
		// does the session implement a local tree provider? 
		ITreeProvider* pTreeProvider = m_pSession->asITreeProvider();
		if (m_pSession->asITreeProvider())
		{
			m_pSession->asITreeProvider()->updateTree(aT);
		}
		else if (m_pSession->asIUpdateProvider())
		{
            AbsolutePath    aRootPath = _aRootPath;
	        SubtreeChange const * pRootChanges = &aT.root;
		    // first, kick off all tree parts which are just empty SubtreeChanges (or SubtreeChanges containing only
		    // SubtreeChanges)
		    // As the changes tree we got is constant, we have to build our own hierarchy
		    SubtreeChangeReferrer aReferrer(*pRootChanges);
		    // this is a SubtreeChange, which, in it's dtor, does not delete all it's children, but only the ones which
		    // are SubtreeChanges itself. As it replaces all SubtreeChanges with SubtreeChangeReferrers, this means
		    // that upon destruction of the object, no Changes are deleted.
		    RemoveEmptySubtrees aRemoveDispensables;
		    aReferrer.dispatch(aRemoveDispensables);
	    
		    // now the root of the changes tree is the "referrer" object
		    pRootChanges = &aReferrer;

		    // do the normalizing
	        SubtreeChange::ChildIterator aChildren = pRootChanges->begin();
	        while ((pRootChanges->size() == 1) && (aChildren->ISA(SubtreeChange)))
	        {
                aRootPath = aRootPath.compose( ONameCreator::createName(*aChildren, pRootChanges) );
		        pRootChanges = static_cast<const SubtreeChange*>(&*aChildren);
		        aChildren = pRootChanges->begin();
	        }

			OSL_ENSURE(m_xDataReqCb.isEmpty(), "TreeManager::updateTree : already doing an update !");
			// this restriction has to be relaxed if this method is asynchronous		
			CFG_TRACE_INFO_NI("cache manager: going to update the node %s", OUSTRING2ASCII(aRootPath.toString()));

			m_xDataReqCb = new OUpdateCallback(aRootPath, _xOptions, *this);

			ORemoteUpdateXMLAttributeHandler aAttributeHandler;
			aAttributeHandler.setCurrentModule(aRootPath.getModuleName().toString());

			XMLFormater aInterpreter(pRootChanges, _xOptions, aAttributeHandler);

			ISubtree const* pSubTree = pInfo->getSubtree(aRootPath);
			OSL_ENSURE(pSubTree, "TreeManager::updateTree : getSubtree should always succeed in this situation !");
			if (pSubTree)
			{
				CFG_TRACE_INFO("cache manager: going to send the changes to the session");

				OSL_ENSURE(pSubTree->hasId(), "Node being updated has no Node ID set");

				m_pSession->asIUpdateProvider()->updateNode(pSubTree->getId(), 
															aRootPath, 
															_xOptions,
															&aInterpreter, 
															m_xDataReqCb.getBodyPtr());


				// TODO : this should be asynchronous : after doing the update on the session, we should return from here.
				// In case of success, we don't have to do anything else.
				// In case 	of an error, our callback (and thus we) will be notified with the real tree state, which
				// we would have to merge in our tree, then.

				CFG_TRACE_INFO_NI("cache manager: synchron update: waiting for the server response");
				// While we are synchronous we try to emulate that behavior,...
				if (m_xDataReqCb->waitForUpdate(m_pSession->getMasterTimeout()))
				{
					m_xDataReqCb = NULL;
				}
				else
				{
					CFG_TRACE_ERROR("cache manager: an error occured ...");
					// TODO : change the error informations transported in an IConfigSession, so we don't have to do
					// this, hmm, complex check
					StatusInfo aStatus = m_xDataReqCb->getStatus();
					if (0 == aStatus.nCode)
					{
						aStatus.sMessage = ::rtl::OUString::createFromAscii("an error occured while connecting with the server. code(");
						aStatus.sMessage += ::rtl::OUString::valueOf(m_xDataReqCb->getConnError());
						aStatus.sMessage += ::rtl::OUString::createFromAscii(").");
					}

					CFG_TRACE_ERROR_NI("cache manager: (%s, %i)", OUSTRING2ASCII(aStatus.sMessage), aStatus.nCode);

					m_xDataReqCb = NULL;
					throw lang::WrappedTargetException(aStatus.sMessage, NULL, Any());
				}
			}
		}
		else
		{
			OSL_ENSURE(false, "TreeManager::updateTree : no implementations for updates available !");
			throw uno::RuntimeException(OUString::createFromAscii("no implementations for updates available"), NULL);
		}
	}
	catch(configuration::Exception& ex)
	{
		configapi::ExceptionMapper e(ex);
		e.unhandled();
	}
}

// -----------------------------------------------------------------------------
// -----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
void TreeManager::releaseSubtree( AbsolutePath const& aSubtreePath, const vos::ORef < OOptions >& _xOptions ) CFG_NOTHROW()
{
	CFG_TRACE_INFO("TreeManager: releasing subtree '%s' for user '%s' with locale '%s'", OUSTRING2ASCII(aSubtreePath.toString()), OUSTRING2ASCII(_xOptions->getUser()), OUSTRING2ASCII(_xOptions->getLocale()) );
	if (TreeInfo* pInfo = requestTreeInfo(_xOptions,false))
	{
		CFG_TRACE_INFO_NI("TreeManager: decrementing refcount for subtree '%s'", OUSTRING2ASCII(aSubtreePath.toString()) );
		if (pInfo->releaseSubtree(aSubtreePath) == 0)
		{
			// start the cleanup thread
			if (_xOptions->canUseCache())
				m_pDisposer->scheduleCleanup(_xOptions);
			else
				disposeOne(_xOptions,true);
		}
	}
}

//-----------------------------------------------------------------------------
void TreeManager::disposeData(const vos::ORef < OOptions >& _xOptions) CFG_NOTHROW()
{
	OWriteSynchronized aWriteGuard(this);

	OSL_ENSURE(!_xOptions.isEmpty(), "TreeManager: Cannot dispose: NULL options are not permitted");
	if (_xOptions.isEmpty()) return;

	if (_xOptions->getLocale().getLength() != 0)
	{
		OSL_ENSURE(_xOptions->getUser().getLength() != 0, "TreeManager: Cannot dispose locale without user");
		CFG_TRACE_INFO( "TreeManager: Disposing data for options: USER='%s' and LOCALE = '%s'",
						OUSTRING2ASCII(_xOptions->getUser()), OUSTRING2ASCII(_xOptions->getLocale()) );

		this->disposeOne( _xOptions, false );
	}
	else if (_xOptions->getUser().getLength() != 0)
	{
		CFG_TRACE_INFO( "TreeManager: Disposing data for user: '%s'", OUSTRING2ASCII(_xOptions->getUser()) );

		this->disposeUser(_xOptions, false);
	}
	else
	{
		OSL_ENSURE(false, "TreeManager: Cannot dispose: neither user nor locale specified in options");
	}
}

// INotifyListener
// ----------------------------------------------------------------------------
void TreeManager::nodeUpdated(TreeChangeList& _rChanges)
{
	CFG_TRACE_INFO("cache manager: updating the tree from a notification");	
	try
	{		
		OClearableWriteSynchronized aWriteGuard(this);
		
		if (TreeInfo* pInfo = requestTreeInfo(_rChanges.getOptions(),false))
		{
			// first approve the changes and merge them with the current tree
			AbsolutePath aSubtreeName = _rChanges.getRootNodePath();

			const ISubtree* pTree = pInfo->getSubtree(aSubtreeName);
			OSL_ENSURE(pTree, "TreeManager::nodeUpdated : node not found in cache!");

			if (pTree)
			{
				if (adjustUpdateToTree(_rChanges.root,*pTree))
                {
					pInfo->updateTree(_rChanges);

		            aWriteGuard.downgrade(); // keep a read lock during notification

			        notifyUpdate(_rChanges);
                }
			}
		}	
	}
	catch (uno::RuntimeException&)
	{
		CFG_TRACE_ERROR("TreeManager::nodeUpdated : could not insert notifications, data may be inconsistent !");
	}
}

//==========================================================================
void TreeManager::acquireReadAccess() const
{
	m_pLock->acquireReadAccess();
}

void TreeManager::releaseReadAccess() const
{
	m_pLock->releaseReadAccess();
}

void TreeManager::acquireWriteAccess()
{
	m_pLock->acquireWriteAccess();
}

void TreeManager::releaseWriteAccess()
{
	m_pLock->releaseWriteAccess();
}


//==========================================================================
//= OUpdateCallback
//==========================================================================
// -------------------------------------------------------------------------
OUpdateCallback::OUpdateCallback(const AbsolutePath& _rRootPath,
                                 const vos::ORef<OOptions>& _rOptions, 
                                 TreeManager& _rCM)
		:m_pCacheManager(&_rCM)
		,m_aChanges(_rOptions,_rRootPath)
		,m_xBuilder()
	,m_nConnError(0)
	,m_refCount(0)
{
	m_aServerStatus.nCode = 0;
	m_aDoneCond.reset();
}

// -------------------------------------------------------------------------

OUpdateCallback::OUpdateCallback(const AbsolutePath& _rRootPath, 								 
								 const vos::ORef<OOptions>& _rOptions)
				:m_pCacheManager(NULL)
	            ,m_aChanges(_rOptions, _rRootPath)
	            ,m_xBuilder()
	            ,m_nConnError(0)
	            ,m_refCount(0)
{
	m_aServerStatus.nCode = 0;
	m_aDoneCond.reset();
}

// -------------------------------------------------------------------------
uno::Reference< xmlsax::XDocumentHandler > OUpdateCallback::getDataReader()
{
    if (m_xBuilder.isEmpty()) m_xBuilder.bind(new XMLTreeChangeListBuilder(m_aChanges));
	return static_cast< xmlsax::XDocumentHandler* >(m_xBuilder.getBodyPtr());
}
// -------------------------------------------------------------------------
sal_Bool OUpdateCallback::waitForUpdate(TimeValue* _pTimeout)
{
	// TODO : if the update on the tree is asnychronous, we don't need this method anymore ...
	TimeValue aTimeout;
	aTimeout.Seconds = 300;
	aTimeout.Nanosec = 0;
	m_aDoneCond.wait(_pTimeout ? _pTimeout : &aTimeout);

	return (0 == m_nConnError) && (0 == m_aServerStatus.nCode);
		// no connection error and no server error
}
// -------------------------------------------------------------------------
void OUpdateCallback::failed(sal_Int32 _nErrorCode)
{	
	m_nConnError = _nErrorCode;
	CFG_TRACE_ERROR("update callback: failed(%i)", _nErrorCode);
	m_aDoneCond.set();
}
// -------------------------------------------------------------------------
void OUpdateCallback::done(const StatusInfo& _rStatus)
{
#ifdef CFG_ENABLE_TRACING
	if (_rStatus.nCode)
	{
		CFG_TRACE_ERROR("update callback: done(%i)", _rStatus.nCode);
	}
#endif
	m_aServerStatus = _rStatus;
	m_aDoneCond.set();
}



} // namespace
