/*************************************************************************
 *
 *  $RCSfile: confname.cxx,v $
 *
 *  $Revision: 1.8 $
 *
 *  last change: $Author: jl $ $Date: 2001/03/21 12:15:40 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include "confname.hxx"
#include <osl/diagnose.h>

namespace configmgr
{
// -------------------------------------------------------------------------------------
	// Constants
// -------------------------------------------------------------------------------------
	const sal_Unicode NAME_DELIMITER = sal_Unicode('/');
	const sal_Char VALID_NAME_CHARACTERS[] = "_abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";

// -------------------------------------------------------------------------------------
	const ::sal_Unicode ConfigurationName::delimiter = NAME_DELIMITER;

	inline static 
	OUString const& makeStringDelimiter() 
	{ 
		static const OUString aStringDelimiter( &ConfigurationName::delimiter,1); 
		return aStringDelimiter;
	}
	inline static 
	OUString const& makeRootname() 
	{ 
		return makeStringDelimiter();
	}
// -------------------------------------------------------------------------------------
	OUString ConfigurationName::stringDelimiter() 
	{ 
		return makeStringDelimiter() ; 
	}
	OUString ConfigurationName::rootname() 
	{
		return makeRootname() ; 
	}

	ConfigurationName ConfigurationName::root() 
	{
		return ConfigurationName(makeRootname()); 
	}

// -------------------------------------------------------------------------------------
// Exceptions switched off:

// -------------------------------------------------------------------------------------
// validation and initialisation
// -------------------------------------------------------------------------------------
	sal_Int32 ConfigurationName::validate(OUString& sName, bool bMaybeAbsolute)  throw()
	{
		sal_Int32 nPos = 0;
		sal_Unicode const * const pName = sName.getStr();

		// no real validation yet
		//OUString const sNameChars( RTL_CONSTASCII_USTRINGPARAM(VALID_NAME_CHARACTERS) );

		if (bMaybeAbsolute)
		{
			if (pName[nPos] == NAME_DELIMITER) 
				++nPos;
		}

		sal_Int32 nNext = sName.indexOf(NAME_DELIMITER,nPos);
		while (nNext > nPos)
		{
			nPos = nNext + 1;
			nNext = sName.indexOf(NAME_DELIMITER,nPos);
		}

		OSL_ENSURE( nNext != 0, "Invalid Configuration Name: Name must be relative" );
		OSL_ENSURE( nNext != nPos, "Invalid Configuration Name: Empty name component" );

		if (nNext == nPos)
		{
			sName = OUString();
			return 0;
		}

		OSL_ASSERT( sName.lastIndexOf(NAME_DELIMITER)+1 == nPos );

		if (nPos == sName.getLength() && nPos > 1)	// remove delimiter at end if not "/" (root)
		{
			OSL_ASSERT( sName.lastIndexOf(NAME_DELIMITER)+1 == nPos );

			sName = sName.copy(0, nPos-1);
			nPos = sName.lastIndexOf(NAME_DELIMITER) + 1;
		}
		OSL_ASSERT( 0 <= nPos && (nPos < sName.getLength() || nPos <= 1));

		return nPos; // OK here
	}

// -------------------------------------------------------------------------------------
	void ConfigurationName::init(OUString const& aPath) throw()
	{
		m_sPath = aPath;
		m_nNamePos = validate(m_sPath, true);
	}

// -------------------------------------------------------------------------------------
	void ConfigurationName::init(ConfigurationName const& aBasePath, OUString const& aRelPath) throw()
	{
		if (aRelPath.getLength() > 0)
		{
			OUString sPath( aRelPath );
			sal_Int32 nPos = validate(sPath,false);

			if (sPath.getLength() != 0)
				init_concat( aBasePath.m_sPath, sPath, nPos );
			else // validation failed
				OSL_ASSERT( isEmpty() );
		}
		else // empty relative path to append
		{
			m_sPath = aBasePath.m_sPath;
			m_nNamePos = aBasePath.m_nNamePos;
		}

	}

// -------------------------------------------------------------------------------------
	void ConfigurationName::init_concat(OUString const& aBasePath, OUString const& aRelPath, sal_Int32 nRelPos) throw()
	{
		// PRE: aBasePath and aRelPath are prevalidated and aRelPath is non empty and non relative
		// PRE: nRelPos is the name position within aRelPath
		OSL_ASSERT( aRelPath.getLength() > 0);
		OSL_ASSERT( aRelPath.getStr()[0] != NAME_DELIMITER );
		OSL_ASSERT( aRelPath.lastIndexOf( NAME_DELIMITER )+1 == nRelPos );

		if (aBasePath.getLength() > 0)
		{
			if (aBasePath == makeRootname())
			{
				m_sPath = aBasePath + aRelPath;
				m_nNamePos = aBasePath.getLength() + nRelPos;
			}
			else
			{
				m_sPath = aBasePath +  makeStringDelimiter() + aRelPath;
				m_nNamePos = aBasePath.getLength() + 1 + nRelPos;
			}
		}
		else 
		{
			m_sPath = aRelPath;
			m_nNamePos = nRelPos;
		}
	}

// -------------------------------------------------------------------------------------
// Construction
// -------------------------------------------------------------------------------------
	ConfigurationName::ConfigurationName(OUString const& sPath, sal_Int32 nPos) throw()
		: m_sPath(sPath), m_nNamePos(nPos)
	{
	}

// -------------------------------------------------------------------------------------
	ConfigurationName::ConfigurationName()
		: m_sPath(), m_nNamePos(0)
	{
	}
	
// -------------------------------------------------------------------------------------
	ConfigurationName::ConfigurationName(OUString const& sFullName) throw()
		: m_sPath(), m_nNamePos(0)
	{
		init(sFullName);
	}
	
// -------------------------------------------------------------------------------------
	ConfigurationName::ConfigurationName(OUString const& sFullName, ConfigurationName::Absolute) throw()
		: m_sPath(), m_nNamePos(0)
	{
		if (sFullName.getLength() == 0)
			init( makeRootname() );
		else
		{
			init(sFullName);
			if (isRelative() && !isEmpty())
			{
				m_sPath = makeStringDelimiter() + m_sPath;
				++m_nNamePos;
			}
		}
	}
	
// -------------------------------------------------------------------------------------
	ConfigurationName::ConfigurationName(ConfigurationName const& sBaseName, OUString const& sRelativeName) throw()
		: m_sPath(), m_nNamePos(0)
	{
		init(sBaseName, sRelativeName);
	}
	
// -------------------------------------------------------------------------------------
	ConfigurationName::ConfigurationName(ConfigurationName const& sBaseName, ConfigurationName const& aRelativeName) throw()
		: m_sPath(), m_nNamePos(0)
	{
		if (!aRelativeName.isEmpty() && aRelativeName.isRelative())
			init_concat(sBaseName.m_sPath, aRelativeName.m_sPath, aRelativeName.m_nNamePos);
		else
			OSL_ASSERT(isEmpty());
	}

// -------------------------------------------------------------------------------------
// Properties
// -------------------------------------------------------------------------------------
	sal_Int32 ConfigurationName::depth() const
	{
		sal_Int32 n = 0;
		Iterator const stop = end();
		for(Iterator it = begin(); it != stop; ++it)
			++n;
		return n;
		// return std::distance(begin(),end());
	}
	
// -------------------------------------------------------------------------------------
	bool ConfigurationName::isEmpty() const
	{
		OSL_ASSERT(0 <= m_nNamePos && m_nNamePos <= m_sPath.getLength());
		return m_sPath.getLength() == 0;
	}
	
// -------------------------------------------------------------------------------------
	OUString ConfigurationName::localName()	const
	{
		OSL_ASSERT(0 <= m_nNamePos && m_nNamePos <= m_sPath.getLength());
		return m_sPath.copy(m_nNamePos);
	}
	
// -------------------------------------------------------------------------------------
	OUString ConfigurationName::fullName()	const 
	{
		OSL_ASSERT(0 <= m_nNamePos && m_nNamePos <= m_sPath.getLength());
		return m_sPath;
	}

	
// -------------------------------------------------------------------------------------
	bool ConfigurationName::isRelative() const
	{
		return m_sPath.getStr()[0] != NAME_DELIMITER;
	}
	
// -------------------------------------------------------------------------------------
	OUString ConfigurationName::moduleName() const
	{
		if (isRelative())
			return OUString();

		sal_Int32 nModuleEnd = m_sPath.indexOf(delimiter,1);
		if (nModuleEnd < 0)
			nModuleEnd = m_sPath.getLength();

		return m_sPath.copy(1,nModuleEnd-1);
	}

	
// -------------------------------------------------------------------------------------
// structure navigation operations
// -------------------------------------------------------------------------------------
	bool ConfigurationName::isNestedIn(ConfigurationName const& aBaseName) const
	{ 
		sal_Int32 nBase = aBaseName.m_sPath.getLength();
		return	(	(m_sPath.compareTo(aBaseName.m_sPath, nBase) == 0)
				&&	(	(m_sPath.getStr()[nBase] == NAME_DELIMITER)
					||	(	(1 == nBase)
						&&	(m_sPath.getStr()[0] == NAME_DELIMITER)
						)
					)
				);
	}
	
// -------------------------------------------------------------------------------------
	ConfigurationName ConfigurationName::relativeTo(ConfigurationName const& aBaseName) const throw()
	{
		if (!isNestedIn(aBaseName))
		{
			OSL_ENSURE(false, "Name is not nested in given base - cannot retrieve relative path");

			return ConfigurationName(); // just an empty one ?
		}

		sal_Int32 nBase = aBaseName.m_sPath.getLength();
		if ((1 == nBase) && (!aBaseName.isRelative()))
		{
			OSL_ENSURE(!isRelative(), "ConfigurationName: inconsistent !");
			return ConfigurationName(m_sPath.copy(1));
		}

		OSL_ENSURE(nBase < m_nNamePos, "ConfigurationName: Inconsistent name position");
		return ConfigurationName(m_sPath.copy(nBase+1), m_nNamePos - nBase -1);
	}

		
// -------------------------------------------------------------------------------------
	ConfigurationName ConfigurationName::getParentName() const
	{
		if (m_nNamePos <= 0)
		{
			OSL_ENSURE(false, "Cannot get parent of empty or root path");

			return ConfigurationName(); // just an empty one ?
		}

		if (1 == m_nNamePos)
			// m_nNamePos points to the first character _after_ the slash, so we're a absolute level-1 node,
			// so our parent is the empty root
			return ConfigurationName(OUString(), Absolute());

		OUString sParentPath = m_sPath.copy(0,m_nNamePos-1);
		sal_Int32 nParentIndex = sParentPath.lastIndexOf(NAME_DELIMITER);

		return ConfigurationName(sParentPath, nParentIndex + 1);
	}

// -------------------------------------------------------------------------------------
	bool operator==(ConfigurationName const& lhs, ConfigurationName const& rhs) throw()
	{
		OSL_ENSURE(lhs.m_nNamePos == rhs.m_nNamePos || lhs.m_sPath != rhs.m_sPath, "ConfigurationName: Inconsistent name positions");
		return sal_False != (lhs.m_sPath == rhs.m_sPath);
	}

// -------------------------------------------------------------------------------------
// Iteration
// -------------------------------------------------------------------------------------
	// iterator invariant
	static bool check_cfgname_iter(OUString const& aPath, int nPos, int nEnd)
	{
		int const nLen = aPath.getLength();
		if (nEnd >= 0)
		{
			OSL_ENSURE(0 <= nPos && nPos < nEnd, "ConfigurationName::Iterator: Invalid range !");
			OSL_ENSURE(nEnd < nLen, "ConfigurationName::Iterator: Invalid end position !");

			OSL_ENSURE(aPath[nPos] != NAME_DELIMITER, "ConfigurationName::Iterator: Invalid start position value !");
			OSL_ENSURE(aPath[nEnd] == NAME_DELIMITER, "ConfigurationName::Iterator: Invalid end position value !");
		}
		else // must be end value
		{
			OSL_ENSURE(nPos == nLen,	"ConfigurationName::Iterator: Invalid end value !");
			OSL_ENSURE(nEnd == -1,		"ConfigurationName::Iterator: Invalid end value !");
		}
		OSL_ENSURE(aPath.indexOf(NAME_DELIMITER,nPos) == nEnd, "ConfigurationName::Iterator: Invalid range !");
		return nEnd >= 0;
	}

#define CHECK_ITER_STATE(it,maybeend) \
	OSL_ENSURE(check_cfgname_iter((it)->aPath,(it)->nPos,(it)->nEnd) || (maybeend), "ConfigurationName::Iterator: End iterator illegal here !")

#define CHECK_ANY_ITER_STATE()		CHECK_ITER_STATE(this,true)
#define CHECK_VALID_ITER_STATE() 	CHECK_ITER_STATE(this,false)

	ConfigurationName::Iterator::Iterator(OUString const& aPath_, bool atEnd) 
	: aPath(aPath_ + makeStringDelimiter()) 
	{
		if (aPath.getStr()[0] == NAME_DELIMITER)
		{
			aPath = aPath.copy(1);
			if (aPath == makeStringDelimiter())
				atEnd = true;
		}

		if (atEnd)
		{
			nPos = aPath.getLength();
			nEnd = -1;
		}
		else
		{
			nPos = 0;
			nEnd = aPath.indexOf(NAME_DELIMITER);
		}
		CHECK_ANY_ITER_STATE(); // INVARIANT 
	}

	OUString const ConfigurationName::Iterator::operator*() const
	{
		CHECK_VALID_ITER_STATE(); // PRE
		return aPath.copy(nPos,nEnd-nPos);
	}

	ConfigurationName::Iterator& ConfigurationName::Iterator::operator++()
	{
		CHECK_VALID_ITER_STATE(); // PRE
		nPos = nEnd +1;
		nEnd = aPath.indexOf(NAME_DELIMITER, nPos);
		CHECK_ANY_ITER_STATE(); // POST

		return *this;
	}

	ConfigurationName::Iterator& ConfigurationName::Iterator::operator--()
	{
		CHECK_ANY_ITER_STATE(); // PRE
		nEnd = nPos-1;
		nPos = aPath.lastIndexOf(NAME_DELIMITER, nEnd-1) + 1;
		CHECK_VALID_ITER_STATE(); // POST

		return *this;
	}

	bool operator==(ConfigurationName::Iterator const& lhs, ConfigurationName::Iterator const& rhs)
	{
		return lhs.nPos == rhs.nPos && lhs.nEnd == rhs.nEnd && lhs.aPath == rhs.aPath;
	}

	ConfigurationName::ConfigurationName(Iterator const& first, Iterator const& last)
		: m_sPath(), m_nNamePos(0)
	{
		CHECK_ITER_STATE(&first,true);
		CHECK_ITER_STATE(&last, true);

		OSL_ENSURE(first.aPath == last.aPath,"Range must hold Iterators into same ConfigurationName");
		OSL_ENSURE(first.nPos <=  last.nPos,"Range must hold Iterators in proper order");

		if (first.nPos < last.nPos)
		{
			OSL_ASSERT(first.nPos >= 0);

			init( first.aPath.copy(first.nPos, last.nPos-1 -first.nPos) );
		}
		else
		{
			OSL_ASSERT(m_sPath.getLength()==0);
			OSL_ASSERT(m_nNamePos==0);
		}
		
	}
} // namespace config


