/*************************************************************************
 *
 *  $RCSfile: synchronize.hxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: jb $ $Date: 2000/11/10 12:14:48 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _CONFIGMGR_SYNCHRONIZE_HXX_
#define _CONFIGMGR_SYNCHRONIZE_HXX_

//..........................................................................
namespace configmgr
{
//..........................................................................

//==========================================================================
//= ISynchronizedData
//==========================================================================
/** interface for objects with multiple-read-single-write access
	<BR>
	The usual behaviour for such data is that unlimited concurrent read access is allowed, but
	while writing the data, all other access (no matter if reading or writing) is blocked.
	<BR>
	The behaviour of read access on a pending write access it not defined here, but implementation
	dependent. The concret implementation may specify that each read access blocks if a write
	access is pending (which seems reasonable), or it may serve all read accesses as soon as they
	are made by the client, no matter if there is a pending write access (which seems stupid :) for most cases).
*/
class ISynchronizedData
{
public:
	/// enter a section where the data is accessed for reading
	virtual void acquireReadAccess() const = 0;
	/// leave a section where the data is accessed for reading
	virtual void releaseReadAccess() const = 0;
	/// enter a section where the data is accessed for writing
	virtual void acquireWriteAccess() = 0;
	/// leave a section where the data is accessed for writing
	virtual void releaseWriteAccess() = 0;
};

namespace internal {	// should be moved to a more public place
	class Noncopyable { 
		Noncopyable(Noncopyable&);
		void operator=(Noncopyable&);
	public:
		Noncopyable() {}
	};
}
//==========================================================================
//= OReadSynchronized
//==========================================================================
/** is able to guard sections where data protected by an <type>ISynchronizedData</type> interface
	is accessed for reading
	<BR>
	this class is not thread-safe (which shouldn't be neccessary as long as the <type>ISynchronizedData</type>
	which is manipulated is.)
*/
class OReadSynchronized : internal::Noncopyable
{
protected:
	const ISynchronizedData*		m_pData;

public:
	/// ctor. will lock the given synchronization object for read access
	OReadSynchronized(const ISynchronizedData* _pData);

	/// dtor. will free the read access lock on the given data
	~OReadSynchronized();

	/// free the read access lock on the given data
	void clear();

	/** attach to a new data object. The read access for the old one (if any) will be released before
		acquiring the new one.
	*/
	void attach(const ISynchronizedData* _pData);
};

//==========================================================================
//= OWriteSynchronized
//==========================================================================
/** is able to guard sections where data protected by an <type>ISynchronizedData</type> interface
	is accessed for reading
*/
class OWriteSynchronized : internal::Noncopyable
{
protected:
	ISynchronizedData*		m_pData;

public:
	/// ctor. will lock the given synchronization object for write access
	OWriteSynchronized(ISynchronizedData* _pData);

	/// dtor. will free the write access lock on the given data
	~OWriteSynchronized();
};

//==========================================================================
//= OClearableWriteSynchronized
//==========================================================================
/** is able to guard sections where data protected by an <type>ISynchronizedData</type> interface
	is accessed for reading
	Can be cleared, or downgraded to a read access
*/
class OClearableWriteSynchronized : internal::Noncopyable
{
protected:
	ISynchronizedData*		m_pData;
	bool					m_bIsWrite;
public:
	/// ctor. will lock the given synchronization object for write access
	OClearableWriteSynchronized(ISynchronizedData* _pData);

	/// dtor. will free the currently active access lock on the given data
	~OClearableWriteSynchronized();

	bool isWriteSynchronized() const { return m_bIsWrite; }

	/// free the current write lock on the given data, but keep a read access lock
	void downgrade();

	/// free the current access lock on the given data
	void clear();

};

//..........................................................................
}	// namespace configmgr
//..........................................................................

#endif // _CONFIGMGR_SYNCHRONIZE_HXX_


