/*************************************************************************
 *
 *  $RCSfile: imapurl.hxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 16:13:03 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef CHAOS_IMAPURL_HXX
#define CHAOS_IMAPURL_HXX

#ifndef _ERRCODE_HXX
#include <tools/errcode.hxx>
#endif
#ifndef _STRING_HXX
#include <tools/string.hxx>
#endif

//============================================================================
/** Various parts of pseudo IMAP URLs.
 */
#define CNT_IMAP_URL_PREFIX "imap://"
#define CNT_IMAP_URL_UIDVALIDITY_PREFIX "UIDValidity="
#define CNT_IMAP_URL_UID_PREFIX "UID="

//============================================================================
namespace chaos {

namespace CntIMAPURL {

/** Translate some URL into a canonic pseudo IMAP URL.

	@param rURL  Some string.

	@param rCanonicURL  Returns the canonic form of rURL, if rURL is a pseudo
	IMAP URL; otherwise left unchanged.

	@return  True if rURL is a pseudo IMAP URL.
 */
bool makeCanonicURL(String const & rURL, String & rCanonicURL);

/** Translate some mailbox path into a canonic mailbox path.

	@param rMboxPath  Some string.

	@param rCanonicMboxPath  Returns the canonic form of rMboxPath, if
	rMboxPath is valid; otherwise left unchanged.

	@return  True if rMboxPath is valid.
 */
bool makeCanonicMboxPath(String const & rMboxPath, String & rCanonicMboxPath);

/** Return the literal name of the root mailbox in some pseudo IMAP mailbox
	URL.

	@param rMboxURL  A valid canonic pseudo IMAP mailbox URL.

	@return  A root mailbox literal name.
 */
ByteString getRootLiteralName(String const & rMboxURL);

/** Return the literal full name of the mailbox in some pseudo IMAP mailbox
	URL.

	@param rMboxURL  A valid canonic pseudo IMAP mailbox URL.

	@param cHierarchySeparator  Either 0 (mailbox has no hierarchy separator),
	or the hierarchy separator character.

	@param rMboxLiteralFullName  Returns a mailbox literal full name.

	@return  False if rMboxURL contains mailbox segment separators, but
	cHierarchy separator is 0 (in that case, rMboxLiteralFullName will not be
	modified).
 */
bool getMboxLiteralFullName(String const & rMboxURL,
							sal_Char cHierarchySeparator,
							ByteString & rMboxLiteralFullName);

/** Return the literal last name of the mailbox in some pseudo IMAP mailbox
	URL.

	@param rMboxURL  A valid canonic pseudo IMAP mailbox URL.

	@return  A mailbox literal last name.
 */
ByteString getMboxLiteralLastName(String const & rMboxURL);

/** Check whether some pseudo IMAP URL is an account URL.

	@param rURL  Some pseudo IMAP URL.

	@return  True if rURL is a valid account URL.
 */
bool isAcntURL(String const & rURL);

/** Check whether some text is a pseudo IMAP message URL that is an (indirect)
	child of some account URL.

	@param rAcntURL  Some canonic pseudo IMAP account URL.

	@param rURL  Some text.

	@param rCanonicMesgURL  Returns the canonic version of rURL, if rURL is a
	valid message URL; otherwise left unchanged.

	@return  True if rURL is a valid message URL.
 */
bool isMesgURL(String const & rAcntURL, String const & rURL,
			   String & rCanonicMesgURL);

/** Check whether some pseudo IMAP mailbox URL is a root mailbox URL.

	@param rAcntURL  A valid canonic pseudo IMAP account URL.

	@param rRootMboxURL  Some text.

	@return  True if rRootMboxURL is a valid canonic root mailbox URL.
 */
bool isRootMboxURL(String const & rAcntURL, String const & rRootMboxURL);

/** Check whether some pseudo IMAP mailbox URL is a direct child of some other
	pseudo IMAP mailbox URL.

	@param rParentMboxURL  A valid canonic pseudo IMAP mailbox URL.

	@param rSubMboxURL  Some text.

	@return  True if rSubMboxURL is a valid canonic mailbox URL that is a
	direct child of rParentMboxURL.
 */
bool isDirectSubMboxURL(String const & rParentMboxURL,
						String const & rSubMboxURL);

/** Check whether some canonic pseudo IMAP mailbox URL denotes an IMAP INBOX.

	@param rMboxURL  A canonic pseudo IMAP mailbox URL.

	@return  True if rMboxURL denotes an IMAP INBOX.
 */
bool isInboxURL(String const & rMboxURL);

/** Decompose a pseudo IMAP message URL.

	@param rMesgURL  A valid canonic pseudo IMAP message URL.

	@param rMboxURL  Returns a canonic pseudo IMAP mailbox URL.

	@param rUIDValidity  Returns either 0 (no UID validity present in
	rMesgURL), or a UID validity value.

	@param rMesgUID  Returns a message UID.
 */
void decomposeMesgURL(String const & rMesgURL, String & rMboxURL,
					  sal_uInt32 & rUIDValidity, sal_uInt32 & rMesgUID);

ErrCode makeCanonicHostPort(String const & rHostPort,
							String & rCanonicHostPort);

/** Create a canonic pseudo IMAP 'host and port only' URL.

	@param rHost  The canonic host.

	@param nPort  The port (0--65535).

	@return  A canonic pseudo IMAP URL.
 */
String createHostAndPortURL(String const & rHost, sal_uInt16 nPort);

/** Create a canonic pseudo IMAP account URL.

	@param bHasIUserAuth  True if rUserID holds a <userid> part (and rAuthType
	holds an <auth_type>).

	@param rUserID  If bHasIUserAuth is true, the unencoded canonic IMAP
	<userid> (otherwise ignored).

	@param rAuthType  If bHasIUserAuth is true, the unencoded IMAP
	<auth_type>, or "*", or an empty string (otherwise ignored).

	@param rHostPort  The canonic host and port.

	@return  A canonic pseudo IMAP URL.
 */
String createAcntURL(bool bHasIUserAuth, String const & rUserID,
					 String const & rAuthType, String const & rHostPort);

/** Create a canonic pseudo IMAP mailbox URL.

	@param rAcntURL  A canonic pseudo IMAP account URL.

	@param rMboxLiteralFullName  A mailbox canonic literal full name.

	@param cHierarchySeparator  Either 0 (mailbox has no hierarchy separator),
	or the hierarchy separator character.

	@return  A canonic pseudo IMAP URL.
 */
String createMboxURL(String const & rAcntURL,
					 ByteString const & rMboxCanonicLiteralFullName,
					 sal_Char cHierarchySeparator);

/** Create a canonic pseudo IMAP mailbox URL.

	@param rAcntURL  A canonic pseudo IMAP account URL.

	@param rMboxPath  A canonic mailbox path (non-empty, escpaped, no slashes
	at start or end).

	@return  A canonic pseudo IMAP URL.
 */
String createMboxURL(String const & rAcntURL, String const & rMboxPath);

/** Create a canonic pseudo IMAP mailbox URL.

	@param rParentURL  A canonic pseudo IMAP account or mailbox URL.

	@param rSubMboxLiteralLastName  A mailbox literal last name.

	@return  A canonic pseudo IMAP URL.
 */
String createSubMboxURL(String const & rParentURL,
						ByteString const & rSubMboxLiteralLastName);

/** Create a canonic pseudo IMAP message URL.

	@param rMboxURL  A canonic pseudo IMAP mailbox URL.

	@param nUIDValidity  Either 0 (no UID validity value given), or a UID
	validity value.

	@param nMesgUID  A message UID.

	@return  A canonic pseudo IMAP URL.
 */
String createMesgURL(String const & rMboxURL, sal_uInt32 nUIDValidity,
					 sal_uInt32 nMesgUID);

/** Create a canonic pseudo IMAP message URL.

	@param rAcntURL  A canonic pseudo IMAP account URL.

	@param rMboxPath  Some text.

	@param nMesgUID  Either a message UID or 0.

	@param rMesgURL  Returns the assembled message URL, if rMboxPath and
	nMesgUID are valid; otherwise left unchanged.

	@return  True if rMboxPath and nMesgUID are valid.
 */
bool createMesgURL(String const & rAcntURL, String const & rMboxPath,
				   sal_uInt32 nMesgUID, String & rMesgURL);

}

//============================================================================
namespace CntIMAPUTF7 {

ByteString translateToUTF7(String const & rSource);

void translateFromUTF7(sal_Char const * pBegin, sal_Char const * pEnd,
					   bool bToURI, String & rTarget);

inline String translateFromUTF7(ByteString const & rSource, bool bToURI)
{
	String aTarget;
	translateFromUTF7(rSource.GetBuffer(),
					  rSource.GetBuffer() + rSource.Len(), false, aTarget);
	return aTarget;
}

}

}

#endif // CHAOS_IMAPURL_HXX

