// Copyright 2000 by Kevin Atkinson under the terms of the LGPL

#include <iomanip>

#include <pspell/manager_impl.hh>

#include "as_config.h"
#include "dirs.h"

#include "config.hh"
#include "file_exceps.hh"
#include "bad_value.hh"

#ifdef ENABLE_WIN32_RELOCATABLE
#include "windows.h"
#endif

#define MODES "none,url,email,sgml,tex"

namespace aspell {
  // 
  // GlobalConf
  //
  void Config::init() 
  {
#ifdef ENABLE_WIN32_RELOCATABLE
    if (retrieve_bool("set-prefix")) {
      char filename[MAX_PATH];
      GetModuleFileName(0,filename,MAX_PATH);
      char * last_slash = 0;
      for (char * i = filename; *i != '\0'; ++i) {
	if (*i == '\\') {
	  *i = '/';
	  last_slash = i;
	}
      }
      *last_slash = '\0';
      replace("prefix", filename);
    }
#endif

    PspellConfig * c = new_pspell_config();
    c->replace("language-tag", retrieve("language-tag").c_str());
    c->replace("rem-all-module-search-order", "");
    c->replace("add-module-search-order", "aspell");
    PspellCanHaveError * possible_err = find_word_list(c);
    delete c;
    if (possible_err->error_number() == 0) {
      c = (PspellConfig *)possible_err;
      if (retrieve("master").size() == 0)
	replace("master", c->retrieve("master"));
      // FIXME: Some how also get the language name so that the 
      //   master dict. does not have to be read in to set the lang
      //   variable.
    } else {
      if (retrieve("master").size() == 0)
	replace("master", retrieve("lang").c_str());
    }
  }

  void Config::read_in(const Config * override) 
  {
    if (override != 0)
      read_in(&const_cast<Config *>(override)->real_config());
    else
      read_in();
  }
  
  void Config::read_in(PspellConfigImpl * override) 
  {

    if (override != 0)
      real_config().merge(*override);


    const char * env = getenv("ASPELL_CONF");
    if (env != 0)       
      read_in_string(env);
    
    Config per;
    try {
      per.read_in_file(retrieve("per-conf-path"));
    } catch (CantReadFile) {}
    merge(per);

    
    try {
      read_in_file(retrieve("conf-path"));
    } catch (CantReadFile) {}
    

    merge(per);
    if (env != 0) 
      read_in_string(env);
    if (override != 0)
      real_config().merge(*override);

    init();
  }

  inline string get_mode_data(Config & c, const string & mode) {
    try {
      string temp = "fm-" + mode;
      return c.retrieve(temp.c_str());
    } catch (UnknownKey) {
      throw BadValue("mode", mode, "one of " MODES);
    }
  }

  class Config::NotifierImpl : public Config::Notifier
  {
  private:
    NotifierImpl();
    NotifierImpl(const NotifierImpl &);
    NotifierImpl & operator= (const NotifierImpl &);
  public:
    Config * config;
    NotifierImpl(Config * c) : config(c) {}

    void item_updated(const KeyInfo * ki, const char * value) {
      if (strcmp(ki->name, "mode") == 0) {

	string m = get_mode_data(*config, value);
	config->replace("rem-all-filter","");
	unsigned int begin = 0, end = 0;

	do {
	  
	  while (end < m.size() && m[end] != ',') ++end;
	  string fil = m.substr(begin, end - begin);
	  config->replace("add-filter", fil.c_str());
	  begin = end + 1;
	  end   = begin;
	  
	} while (end < m.size());

      }
    }
  };

#define CANT_CHANGE 1

#ifdef ENABLE_WIN32_RELOCATABLE
#  define HOME_DIR "<prefix>"
#  define PERSONAL "<lang>.pws"
#  define REPL     "<lang>.prepl"
#else
#  define HOME_DIR "<$HOME|./>"
#  define PERSONAL ".aspell.<lang>.pws"
#  define REPL     ".aspell.<lang>.prepl"
#endif

  static const KeyInfo config_keys[] = {
    {"charset",  KeyInfoString, "iso8859-1",0}
    , {"conf",     KeyInfoString, "aspell.conf",  "main configuration file"             , {0, CANT_CHANGE}}
    , {"conf-dir", KeyInfoString, CONF_DIR,      "location of main configuration file" ,{0, CANT_CHANGE}}
    , {"conf-path",     KeyInfoString, "<conf-dir/conf>",     0}
    , {"data-dir", KeyInfoString, DATA_DIR,        "location of language data files"     }
    , {"dict-dir", KeyInfoString, DICT_DIR,        "location of the main word list"      }
    , {"extra-dicts", KeyInfoList, "", "extra dictionaries to use"}
    , {"filter",   KeyInfoList  , "url",             "add or removes a filter"}
    , {"fm-email", KeyInfoList  , "url,email",                         0}
    , {"fm-none",  KeyInfoList  , "",                                  0}
    , {"fm-sgml",  KeyInfoList  , "url,sgml,SGML&<charset>/<charset>", 0}
    , {"fm-tex",   KeyInfoList  , "url,tex",                           0}
    , {"fm-url",   KeyInfoList  , "url",                               0}
    , {"home-dir", KeyInfoString, HOME_DIR,   "location for personal files" }
    , {"ignore",   KeyInfoInt   , "1",            "ignore words <= n chars"             }
    , {"ignore-accents" , KeyInfoBool, "false", "ignore accents when checking words"}
    , {"ignore-case", KeyInfoBool  , "false",     "ignore case when checking words"}
    , {"ignore-repl", KeyInfoBool  , "false",     "ignore commands to store replacement pairs"}
    , {"keyboard", KeyInfoString, "standard", "keyboard definition to use for typo analysis"}
    , {"lang",     KeyInfoString, "english",      "default language to use when all else fails", {0, CANT_CHANGE}}
    , {"language-tag", KeyInfoString, "<$LANG|en>", "language code to use when selecting a dictionary", {0, CANT_CHANGE}}
    , {"local-data-dir", KeyInfoString, DATA_DIR,        "location of local language data files"     }
    , {"master",   KeyInfoString, "",             "base name of the main dictionary to use",           }
    , {"master-path",   KeyInfoString, "<dict-dir/master>",   0}
    , {"minimal-specified-component", KeyInfoInt, "255", 0}
    , {"mode",     KeyInfoString, "url",             "filter mode = " MODES }
    , {"per-conf", KeyInfoString, ".aspell.conf", "personal configuration file",{0, CANT_CHANGE}}
    , {"per-conf-path", KeyInfoString, "<home-dir/per-conf>", 0}
    , {"personal", KeyInfoString, PERSONAL,   "personal word list file name"}
    , {"personal-path", KeyInfoString, "<home-dir/personal>", 0}
    , {"prefix",   KeyInfoString, PREFIX, "prefix directory", {0, CANT_CHANGE}}
    , {"set-prefix", KeyInfoBool, "true", "set the prefix based on executable location", {0, CANT_CHANGE}} 
    , {"repl",     KeyInfoString, REPL, "replacements list file name" }
    , {"repl-path",     KeyInfoString, "<home-dir/repl>",     0}
    , {"run-together",        KeyInfoBool,  "false", "consider run-together words legal"}
    , {"run-together-limit",  KeyInfoInt,   "8", "maxium numbers that can be strung together"}
    , {"run-together-min",    KeyInfoInt,   "3", "minimal length of interior words"}
    , {"run-together-specified", KeyInfoBool, "false", 0}
    , {"save-repl", KeyInfoBool  , "true", "save replacement pairs on save all"}
    , {"strip-accents" , KeyInfoBool, "false", "strip accents from word lists"}
    , {"sug-mode",    KeyInfoString, "normal",    "suggestion mode = ultra|fast|normal|bad-spellers"}
  };
}


namespace afilter {
  typedef const autil::KeyInfo * ConstKeyInfoPtr;
  extern ConstKeyInfoPtr 
    email_options_begin, email_options_end,
    sgml_options_begin, sgml_options_end,
    tex_options_begin, tex_options_end;
}

namespace aspell {
  static const ConfigData::Module modules[] = {
    {"email", afilter::email_options_begin, afilter::email_options_end}
    , {"sgml",  afilter::sgml_options_begin,  afilter::sgml_options_end}
    , {"tex",   afilter::tex_options_begin,   afilter::tex_options_end}
  };
  
  Config::Config() 
    : ConfigData ("aspell",
		  config_keys, 
		  config_keys + sizeof(config_keys)/sizeof(KeyInfo))
    , notifier(new NotifierImpl(this))
  {
    set_modules(modules, modules+ sizeof(modules)/sizeof(ConfigData::Module));
    add_notifier(notifier);
  }

  Config::~Config() {
    delete notifier;
  }

  Config::Config(const Config & other)
    : ConfigData(other),
      notifier(new NotifierImpl(this))
  {
    replace_notifier(other.notifier, notifier);
  }

  Config & Config::operator= (const Config & other) 
  {
    ConfigData::operator=(other);
    replace_notifier(other.notifier, notifier);
    return *this;
  }
  
}
