/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.cluster.service;

import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.Version;
import org.opensearch.cluster.ClusterStateTaskExecutor;
import org.opensearch.cluster.service.ClusterManagerTaskThrottlerListener;
import org.opensearch.cluster.service.ClusterManagerThrottlingException;
import org.opensearch.cluster.service.TaskBatcher;
import org.opensearch.cluster.service.TaskBatcherListener;
import org.opensearch.common.annotation.PublicApi;
import org.opensearch.common.settings.ClusterSettings;
import org.opensearch.common.settings.Setting;
import org.opensearch.common.settings.Settings;
import org.opensearch.common.unit.TimeValue;

public class ClusterManagerTaskThrottler
implements TaskBatcherListener {
    private static final Logger logger = LogManager.getLogger(ClusterManagerTaskThrottler.class);
    public static final ThrottlingKey DEFAULT_THROTTLING_KEY = new ThrottlingKey("default-task-key", false);
    static volatile TimeValue baseDelay = TimeValue.timeValueSeconds((long)5L);
    static volatile TimeValue maxDelay = TimeValue.timeValueSeconds((long)30L);
    public static final Setting<Settings> THRESHOLD_SETTINGS = Setting.groupSetting("cluster_manager.throttling.thresholds.", Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<TimeValue> BASE_DELAY_SETTINGS = Setting.timeSetting("cluster_manager.throttling.retry.base.delay", baseDelay, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<TimeValue> MAX_DELAY_SETTINGS = Setting.timeSetting("cluster_manager.throttling.retry.max.delay", maxDelay, Setting.Property.Dynamic, Setting.Property.NodeScope);
    protected Map<String, ThrottlingKey> THROTTLING_TASK_KEYS = new ConcurrentHashMap<String, ThrottlingKey>();
    private final int MIN_THRESHOLD_VALUE = -1;
    private final ClusterManagerTaskThrottlerListener clusterManagerTaskThrottlerListener;
    final ConcurrentMap<String, Long> tasksCount;
    private final ConcurrentMap<String, Long> tasksThreshold;
    private final Supplier<Version> minNodeVersionSupplier;
    private AtomicBoolean startThrottling = new AtomicBoolean();

    public ClusterManagerTaskThrottler(Settings settings, ClusterSettings clusterSettings, Supplier<Version> minNodeVersionSupplier, ClusterManagerTaskThrottlerListener clusterManagerTaskThrottlerListener) {
        this.tasksCount = new ConcurrentHashMap<String, Long>(128);
        this.tasksThreshold = new ConcurrentHashMap<String, Long>(128);
        this.minNodeVersionSupplier = minNodeVersionSupplier;
        this.clusterManagerTaskThrottlerListener = clusterManagerTaskThrottlerListener;
        clusterSettings.addSettingsUpdateConsumer(THRESHOLD_SETTINGS, this::updateSetting, this::validateSetting);
        clusterSettings.addSettingsUpdateConsumer(BASE_DELAY_SETTINGS, this::updateBaseDelay);
        clusterSettings.addSettingsUpdateConsumer(MAX_DELAY_SETTINGS, this::updateMaxDelay);
        this.updateSetting(THRESHOLD_SETTINGS.get(settings));
        this.updateBaseDelay(BASE_DELAY_SETTINGS.get(settings));
        this.updateMaxDelay(MAX_DELAY_SETTINGS.get(settings));
    }

    void updateBaseDelay(TimeValue newBaseValue) {
        baseDelay = newBaseValue;
    }

    void updateMaxDelay(TimeValue newMaxValue) {
        maxDelay = newMaxValue;
    }

    public static TimeValue getBaseDelayForRetry() {
        return baseDelay;
    }

    public static TimeValue getMaxDelayForRetry() {
        return maxDelay;
    }

    protected ThrottlingKey registerClusterManagerTask(String taskKey, boolean throttlingEnabled) {
        ThrottlingKey throttlingKey = new ThrottlingKey(taskKey, throttlingEnabled);
        if (this.THROTTLING_TASK_KEYS.containsKey(taskKey)) {
            throw new IllegalArgumentException("There is already a Throttling key registered with same name: " + taskKey);
        }
        this.THROTTLING_TASK_KEYS.put(taskKey, throttlingKey);
        return throttlingKey;
    }

    void validateSetting(Settings settings) {
        Map<String, Settings> groups = settings.getAsGroups();
        if (groups.size() > 0 && this.minNodeVersionSupplier.get().compareTo(Version.V_2_5_0) < 0) {
            throw new IllegalArgumentException("All the nodes in cluster should be on version later than or equal to 2.5.0");
        }
        for (String key : groups.keySet()) {
            if (!this.THROTTLING_TASK_KEYS.containsKey(key)) {
                throw new IllegalArgumentException("Cluster manager task throttling is not configured for given task type: " + key);
            }
            if (!this.THROTTLING_TASK_KEYS.get(key).isThrottlingEnabled()) {
                throw new IllegalArgumentException("Throttling is not enabled for given task type: " + key);
            }
            int threshold = groups.get(key).getAsInt("value", -1);
            if (threshold >= -1) continue;
            throw new IllegalArgumentException("Provide positive integer for limit or -1 for disabling throttling");
        }
    }

    void updateSetting(Settings newSettings) {
        Map<String, Settings> groups = newSettings.getAsGroups();
        HashSet<String> settingKeys = new HashSet<String>();
        settingKeys.addAll(groups.keySet());
        settingKeys.addAll(this.tasksThreshold.keySet());
        Iterator iterator = settingKeys.iterator();
        while (iterator.hasNext()) {
            String key;
            Settings setting = groups.get(key = (String)iterator.next());
            this.updateLimit(key, setting == null ? -1 : setting.getAsInt("value", -1));
        }
    }

    void updateLimit(String taskKey, int limit) {
        assert (limit >= -1);
        if (limit == -1) {
            this.tasksThreshold.remove(taskKey);
        } else {
            this.tasksThreshold.put(taskKey, Long.valueOf(limit));
        }
    }

    Long getThrottlingLimit(String taskKey) {
        return (Long)this.tasksThreshold.get(taskKey);
    }

    private void failFastWhenThrottlingThresholdsAreAlreadyBreached(boolean throttlingEnabledWithThreshold, Long threshold, long existingTaskCount, int incomingTaskCount, String taskThrottlingKey) {
        if (throttlingEnabledWithThreshold && this.shouldThrottle(threshold, existingTaskCount, incomingTaskCount)) {
            throw new ClusterManagerThrottlingException("Throttling Exception : Limit exceeded for " + taskThrottlingKey, new Object[0]);
        }
    }

    @Override
    public void onBeginSubmit(List<? extends TaskBatcher.BatchedTask> tasks) {
        ThrottlingKey clusterManagerThrottlingKey = ((ClusterStateTaskExecutor)tasks.get((int)0).batchingKey).getClusterManagerThrottlingKey();
        String taskThrottlingKey = clusterManagerThrottlingKey.getTaskThrottlingKey();
        Long threshold = this.getThrottlingLimit(taskThrottlingKey);
        boolean isThrottlingEnabledWithThreshold = clusterManagerThrottlingKey.isThrottlingEnabled() && threshold != null;
        int incomingTaskCount = tasks.size();
        try {
            this.tasksCount.putIfAbsent(taskThrottlingKey, 0L);
            this.failFastWhenThrottlingThresholdsAreAlreadyBreached(isThrottlingEnabledWithThreshold, threshold, (Long)this.tasksCount.get(taskThrottlingKey), incomingTaskCount, taskThrottlingKey);
            this.tasksCount.computeIfPresent(taskThrottlingKey, (key, existingTaskCount) -> {
                this.failFastWhenThrottlingThresholdsAreAlreadyBreached(isThrottlingEnabledWithThreshold, threshold, (long)existingTaskCount, incomingTaskCount, taskThrottlingKey);
                return existingTaskCount + (long)incomingTaskCount;
            });
        }
        catch (ClusterManagerThrottlingException e) {
            this.clusterManagerTaskThrottlerListener.onThrottle(taskThrottlingKey, incomingTaskCount);
            logger.trace("Throwing Throttling Exception for [{}]. Trying to add [{}] tasks to queue, limit is set to [{}]", (Object)taskThrottlingKey, (Object)incomingTaskCount, (Object)threshold);
            throw e;
        }
    }

    private boolean shouldThrottle(Long threshold, Long count, int size) {
        if (!this.startThrottling.get()) {
            if (this.minNodeVersionSupplier.get().compareTo(Version.V_2_5_0) >= 0) {
                this.startThrottling.compareAndSet(false, true);
                logger.info("Starting cluster manager throttling as all nodes are higher than or equal to 2.5.0");
            } else {
                logger.info("Skipping cluster manager throttling as at least one node < 2.5.0 is present in cluster");
                return false;
            }
        }
        return count + (long)size > threshold;
    }

    @Override
    public void onSubmitFailure(List<? extends TaskBatcher.BatchedTask> tasks) {
        this.reduceTaskCount(tasks);
    }

    @Override
    public void onBeginProcessing(List<? extends TaskBatcher.BatchedTask> tasks) {
        this.reduceTaskCount(tasks);
    }

    @Override
    public void onTimeout(List<? extends TaskBatcher.BatchedTask> tasks) {
        this.reduceTaskCount(tasks);
    }

    private void reduceTaskCount(List<? extends TaskBatcher.BatchedTask> tasks) {
        String masterTaskKey = ((ClusterStateTaskExecutor)tasks.get((int)0).batchingKey).getClusterManagerThrottlingKey().getTaskThrottlingKey();
        this.tasksCount.computeIfPresent(masterTaskKey, (key, count) -> count - (long)tasks.size());
    }

    @PublicApi(since="1.0.0")
    public static class ThrottlingKey {
        private String taskThrottlingKey;
        private boolean throttlingEnabled;

        private ThrottlingKey(String taskThrottlingKey, boolean throttlingEnabled) {
            this.taskThrottlingKey = taskThrottlingKey;
            this.throttlingEnabled = throttlingEnabled;
        }

        public String getTaskThrottlingKey() {
            return this.taskThrottlingKey;
        }

        public boolean isThrottlingEnabled() {
            return this.throttlingEnabled;
        }
    }
}

