"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerData2SummaryRoutes = registerData2SummaryRoutes;
exports.registerSummaryAssistantRoutes = registerSummaryAssistantRoutes;
var _configSchema = require("@osd/config-schema");
var _llm = require("../../common/constants/llm");
var _get_opensearch_client_transport = require("../utils/get_opensearch_client_transport");
var _get_agent = require("./get_agent");
var _error_handler = require("./error_handler");
var _errors = require("./errors");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const SUMMARY_AGENT_CONFIG_ID = 'os_summary';
const LOG_PATTERN_SUMMARY_AGENT_CONFIG_ID = 'os_summary_with_log_pattern';
const OS_INSIGHT_AGENT_CONFIG_ID = 'os_insight';
const DATA2SUMMARY_AGENT_CONFIG_ID = 'os_data2summary';
function registerSummaryAssistantRoutes(router, assistantService) {
  router.post({
    path: _llm.SUMMARY_ASSISTANT_API.SUMMARIZE,
    validate: {
      body: _configSchema.schema.object({
        summaryType: _configSchema.schema.string(),
        insightType: _configSchema.schema.maybe(_configSchema.schema.string()),
        question: _configSchema.schema.string(),
        context: _configSchema.schema.maybe(_configSchema.schema.string()),
        index: _configSchema.schema.maybe(_configSchema.schema.string()),
        dsl: _configSchema.schema.maybe(_configSchema.schema.string()),
        topNLogPatternData: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const client = await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
      context,
      dataSourceId: req.query.dataSourceId
    });
    const assistantClient = assistantService.getScopedClient(req, context);
    const agentConfigId = req.body.index && req.body.dsl && req.body.topNLogPatternData ? LOG_PATTERN_SUMMARY_AGENT_CONFIG_ID : SUMMARY_AGENT_CONFIG_ID;
    try {
      var _response$body$infere;
      const response = await assistantClient.executeAgentByConfigName(agentConfigId, {
        context: req.body.context,
        question: req.body.question,
        index: req.body.index,
        input: req.body.dsl,
        topNLogPatternData: req.body.topNLogPatternData
      });
      let insightAgentIdExists = false;
      if (req.body.insightType) {
        insightAgentIdExists = !!(await detectInsightAgentId(req.body.insightType, req.body.summaryType, client));
      }
      const summary = (_response$body$infere = response.body.inference_results[0]) === null || _response$body$infere === void 0 || (_response$body$infere = _response$body$infere.output[0]) === null || _response$body$infere === void 0 ? void 0 : _response$body$infere.result;
      if (!summary) {
        return res.customError({
          body: 'Execute agent failed with empty response!',
          statusCode: 500
        });
      }
      return res.ok({
        body: {
          summary,
          insightAgentIdExists
        }
      });
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
  router.post({
    path: _llm.SUMMARY_ASSISTANT_API.INSIGHT,
    validate: {
      body: _configSchema.schema.object({
        summaryType: _configSchema.schema.string(),
        insightType: _configSchema.schema.string(),
        summary: _configSchema.schema.string(),
        context: _configSchema.schema.string(),
        question: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const client = await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
      context,
      dataSourceId: req.query.dataSourceId
    });
    const assistantClient = assistantService.getScopedClient(req, context);
    try {
      var _response$body$infere2;
      const insightAgentId = await detectInsightAgentId(req.body.insightType, req.body.summaryType, client);
      if (!insightAgentId) {
        return res.notFound({
          body: 'Agent not found'
        });
      }
      const response = await assistantClient.executeAgent(insightAgentId, {
        context: req.body.context,
        summary: req.body.summary,
        question: req.body.question
      });
      const insight = (_response$body$infere2 = response.body.inference_results[0]) === null || _response$body$infere2 === void 0 || (_response$body$infere2 = _response$body$infere2.output[0]) === null || _response$body$infere2 === void 0 ? void 0 : _response$body$infere2.result;
      if (!insight) {
        return res.customError({
          body: 'Execute agent failed with empty response!',
          statusCode: 500
        });
      }
      return res.ok({
        body: {
          insight
        }
      });
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
}
function detectInsightAgentId(insightType, summaryType, client) {
  // We have separate agent for os_insight and user_insight. And for user_insight, we can
  // only get it by searching on name since it is not stored in agent config.
  try {
    if (insightType === 'os_insight') {
      return (0, _get_agent.getAgentIdByConfigName)(OS_INSIGHT_AGENT_CONFIG_ID, client);
    } else if (insightType === 'user_insight' && summaryType === 'alerts') {
      return (0, _get_agent.searchAgent)({
        name: 'KB_For_Alert_Insight'
      }, client);
    }
  } catch (e) {
    // It only detects if the agent exists, we don't want to throw the error when not found the agent
    // So we return `undefined` to indicate the insight agent id not found
    if (e instanceof _errors.AgentNotFoundError) {
      return undefined;
    }
    throw e;
  }
}
function registerData2SummaryRoutes(router, assistantService) {
  router.post({
    path: _llm.SUMMARY_ASSISTANT_API.DATA2SUMMARY,
    validate: {
      body: _configSchema.schema.object({
        sample_data: _configSchema.schema.string(),
        sample_count: _configSchema.schema.maybe(_configSchema.schema.number()),
        total_count: _configSchema.schema.maybe(_configSchema.schema.number()),
        question: _configSchema.schema.maybe(_configSchema.schema.string()),
        ppl: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const assistantClient = assistantService.getScopedClient(req, context);
    try {
      const response = await assistantClient.executeAgentByConfigName(DATA2SUMMARY_AGENT_CONFIG_ID, {
        sample_data: req.body.sample_data,
        total_count: req.body.total_count,
        sample_count: req.body.sample_count,
        ppl: req.body.ppl,
        question: req.body.question
      });
      const result = response.body.inference_results[0].output[0].result;
      if (result) {
        return res.ok({
          body: result
        });
      } else {
        return res.customError({
          body: 'Execute agent failed with empty response!',
          statusCode: 500
        });
      }
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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