"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _constants = require("../utils/constants");
var _helpers = require("../utils/helpers");
var _DataStreamService = require("./DataStreamService");
var _MDSEnabledClientService = require("./MDSEnabledClientService");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class IndexService extends _MDSEnabledClientService.MDSEnabledClientService {
  constructor(...args) {
    super(...args);
    _defineProperty(this, "getIndices", async (context, request, response) => {
      try {
        // @ts-ignore
        const {
          from,
          size,
          sortField,
          sortDirection,
          terms,
          indices,
          dataStreams,
          showDataStreams,
          expandWildcards,
          exactSearch
        } = request.query;
        const params = {
          index: (0, _helpers.getSearchString)(terms, indices, dataStreams),
          format: "json"
        };
        if (sortField !== "managed" && sortField !== "data_stream") {
          params.s = `${sortField}:${sortDirection}`;
        }
        if (expandWildcards) {
          params.expand_wildcards = expandWildcards;
        }
        if (exactSearch) {
          params.index = exactSearch;
        }
        const callWithRequest = this.getClientBasedOnDataSource(context, request);
        const [recoverys, tasks, indicesResponse, indexToDataStreamMapping] = await Promise.all([callWithRequest("cat.recovery", {
          format: "json",
          detailed: true
        }).catch(() => []), callWithRequest("cat.tasks", {
          format: "json",
          detailed: true,
          actions: "indices:data/write/reindex"
        }).catch(() => []), callWithRequest("cat.indices", params), (0, _DataStreamService.getIndexToDataStreamMapping)(callWithRequest)]);
        const formattedTasks = tasks.map(item => {
          const {
            description
          } = item;
          const regexp = /reindex from \[([^\]]+)\] to \[([^\]]+)\]/i;
          const matchResult = description.match(regexp);
          if (matchResult) {
            const [, fromIndex, toIndex] = matchResult;
            return {
              ...item,
              fromIndex,
              toIndex
            };
          } else {
            return {
              ...item,
              fromIndex: "",
              toIndex: ""
            };
          }
        });
        const onGoingRecovery = recoverys.filter(item => item.stage !== "done");

        // Augment the indices with their parent data stream name.
        indicesResponse.forEach(index => {
          index.data_stream = indexToDataStreamMapping[index.index] || null;
          let extraStatus = index.status;
          if (index.health === "green") {
            if (formattedTasks.find(item => item.toIndex === index.index)) {
              extraStatus = "reindex";
            }
          } else {
            if (onGoingRecovery.find(item => item.index === index.index)) {
              extraStatus = "recovery";
            }
          }
          if (extraStatus) {
            index.extraStatus = extraStatus;
          }
        });
        function customSort(array, key, sortDirection) {
          return array.sort((a, b) => {
            let flag;
            const aValue = a[key];
            const bValue = b[key];
            if (sortDirection === "asc") {
              flag = aValue < bValue;
            } else {
              flag = aValue > bValue;
            }
            return flag ? -1 : 1;
          });
        }
        if (sortField === "status") {
          // add new more status to status field so we need to sort
          customSort(indicesResponse, "extraStatus", sortDirection);
        }

        // Filtering out indices that belong to a data stream. This must be done before pagination.
        const filteredIndices = showDataStreams ? indicesResponse : indicesResponse.filter(index => index.data_stream === null);

        // _cat doesn't support pagination, do our own in server pagination to at least reduce network bandwidth
        const fromNumber = parseInt(from, 10);
        const sizeNumber = parseInt(size, 10);
        const paginatedIndices = filteredIndices.slice(fromNumber, fromNumber + sizeNumber);
        const indexNames = paginatedIndices.map(value => value.index);
        const managedStatus = await this._getManagedStatus(callWithRequest, indexNames);
        const allIndices = paginatedIndices.map(catIndex => ({
          ...catIndex,
          managed: managedStatus[catIndex.index] ? "Yes" : "No",
          managedPolicy: managedStatus[catIndex.index]
        }));

        // NOTE: Cannot use response.ok due to typescript type checking
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              indices: sortField === "managed" ? customSort(allIndices, "managed", sortDirection) : allIndices,
              totalIndices: filteredIndices.length
            }
          }
        });
      } catch (err) {
        // Throws an error if there is no index matching pattern
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                indices: [],
                totalIndices: 0
              }
            }
          });
        }
        console.error("Index Management - IndexService - getIndices:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    _defineProperty(this, "_getManagedStatus", async (callWithRequest, indexNames) => {
      try {
        const explainParamas = {
          index: indexNames.toString()
        };
        const explainResponse = await callWithRequest("ism.explain", explainParamas);
        const managed = {};
        for (const indexName in explainResponse) {
          if (indexName === "total_managed_indices") continue;
          const explain = explainResponse[indexName];
          managed[indexName] = explain["index.plugins.index_state_management.policy_id"] === null ? "" : explain["index.plugins.index_state_management.policy_id"];
        }
        return managed;
      } catch (err) {
        // otherwise it could be an unauthorized access error to config index or some other error
        // in which case we will return managed status N/A
        console.error("Index Management - IndexService - _getManagedStatus:", err);
        return indexNames.reduce((accu, value) => ({
          ...accu,
          [value]: "N/A"
        }), {});
      }
    });
    _defineProperty(this, "applyPolicy", async (context, request, response) => {
      try {
        const {
          indices,
          policyId
        } = request.body;
        const callWithRequest = this.getClientBasedOnDataSource(context, request);
        const params = {
          index: indices.join(","),
          body: {
            policy_id: policyId
          }
        };
        const addResponse = await callWithRequest("ism.add", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              failures: addResponse.failures,
              updatedIndices: addResponse.updated_indices,
              failedIndices: addResponse.failed_indices.map(failedIndex => ({
                indexName: failedIndex.index_name,
                indexUuid: failedIndex.index_uuid,
                reason: failedIndex.reason
              }))
            }
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - applyPolicy:", err);
        // return { ok: false, error: err.message };
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    _defineProperty(this, "editRolloverAlias", async (context, request, response) => {
      try {
        const {
          alias,
          index
        } = request.body;
        const callWithRequest = this.getClientBasedOnDataSource(context, request);
        const params = {
          index,
          body: {
            [_constants.Setting.RolloverAlias]: alias
          }
        };
        const rollOverResponse = await callWithRequest("indices.putSettings", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: rollOverResponse
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - editRolloverAlias", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
  }
}
exports.default = IndexService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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