"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthentication = void 0;

var fs = _interopRequireWildcard(require("fs"));

var _wreck = _interopRequireDefault(require("@hapi/wreck"));

var _http = _interopRequireDefault(require("http"));

var _https = _interopRequireDefault(require("https"));

var _routes = require("./routes");

var _authentication_type = require("../authentication_type");

var _helper = require("./helper");

var _next_url = require("../../../utils/next_url");

var _common = require("../../../../common");

var _cookie_splitter = require("../../../session/cookie_splitter");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   *   Copyright OpenSearch Contributors
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   *   Licensed under the Apache License, Version 2.0 (the "License").
                                                                                                                                                                                                                   *   You may not use this file except in compliance with the License.
                                                                                                                                                                                                                   *   A copy of the License is located at
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   *       http://www.apache.org/licenses/LICENSE-2.0
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   *   or in the "license" file accompanying this file. This file is distributed
                                                                                                                                                                                                                   *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
                                                                                                                                                                                                                   *   express or implied. See the License for the specific language governing
                                                                                                                                                                                                                   *   permissions and limitations under the License.
                                                                                                                                                                                                                   */

class OpenIdAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, core, logger) {
    var _this$config$openid, _this$config$openid2;

    super(config, sessionStorageFactory, router, esClient, core, logger);

    _defineProperty(this, "type", _common.AuthType.OPEN_ID);

    _defineProperty(this, "openIdAuthConfig", void 0);

    _defineProperty(this, "authHeaderName", void 0);

    _defineProperty(this, "openIdConnectUrl", void 0);

    _defineProperty(this, "wreckClient", void 0);

    this.wreckClient = this.createWreckClient();
    this.openIdAuthConfig = {};
    this.authHeaderName = ((_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.header) || '';
    this.openIdAuthConfig.authHeaderName = this.authHeaderName;
    this.openIdConnectUrl = ((_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.connect_url) || '';
    let scope = this.config.openid.scope;

    if (scope.indexOf('openid') < 0) {
      scope = `openid ${scope}`;
    }

    this.openIdAuthConfig.scope = scope;
  }

  async init() {
    try {
      const response = await this.wreckClient.get(this.openIdConnectUrl);
      const payload = JSON.parse(response.payload);
      this.openIdAuthConfig.authorizationEndpoint = payload.authorization_endpoint;
      this.openIdAuthConfig.tokenEndpoint = payload.token_endpoint;
      this.openIdAuthConfig.endSessionEndpoint = payload.end_session_endpoint || undefined;
      this.createExtraStorage();
      const routes = new _routes.OpenIdAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.openIdAuthConfig, this.securityClient, this.coreSetup, this.wreckClient);
      routes.setupRoutes();
    } catch (error) {
      this.logger.error(error); // TODO: log more info

      throw new Error('Failed when trying to obtain the endpoints from your IdP');
    }
  }

  createWreckClient() {
    var _this$config$openid3, _this$config$openid4;

    const wreckHttpsOption = {};

    if ((_this$config$openid3 = this.config.openid) !== null && _this$config$openid3 !== void 0 && _this$config$openid3.root_ca) {
      wreckHttpsOption.ca = [fs.readFileSync(this.config.openid.root_ca)];
    }

    if (((_this$config$openid4 = this.config.openid) === null || _this$config$openid4 === void 0 ? void 0 : _this$config$openid4.verify_hostnames) === false) {
      this.logger.debug(`openId auth 'verify_hostnames' option is off.`);

      wreckHttpsOption.checkServerIdentity = (host, cert) => {
        return undefined;
      };
    }

    if (Object.keys(wreckHttpsOption).length > 0) {
      return _wreck.default.defaults({
        agents: {
          http: new _http.default.Agent(),
          https: new _https.default.Agent(wreckHttpsOption),
          httpsAllowUnauthorized: new _https.default.Agent({
            rejectUnauthorized: false
          })
        }
      });
    } else {
      return _wreck.default;
    }
  }

  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const extraCookiePrefix = this.config.openid.extra_storage.cookie_prefix;
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie

    };

    for (let i = 1; i <= this.config.openid.extra_storage.additional_cookies; i++) {
      hapiServer.states.add(extraCookiePrefix + i, extraCookieSettings);
    }
  }

  getExtraAuthStorageOptions() {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.openid.extra_storage.cookie_prefix,
      additionalCookies: this.config.openid.extra_storage.additional_cookies,
      logger: this.logger
    };
  }

  requestIncludesAuthInfo(request) {
    return request.headers.authorization ? true : false;
  }

  async getAdditionalAuthHeader(request) {
    return {};
  }

  getCookie(request, authInfo) {
    (0, _cookie_splitter.setExtraAuthStorage)(request, request.headers.authorization, this.getExtraAuthStorageOptions());
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: this.type,
      expiryTime: Date.now() + this.config.session.ttl
    };
  } // TODO: Add token expiration check here


  async isValidCookie(cookie, request) {
    var _cookie$credentials, _cookie$credentials2, _cookie$credentials3;

    if (cookie.authType !== this.type || !cookie.username || !cookie.expiryTime || !((_cookie$credentials = cookie.credentials) !== null && _cookie$credentials !== void 0 && _cookie$credentials.authHeaderValue) && !this.getExtraAuthStorageValue(request, cookie) || !((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.expires_at)) {
      return false;
    }

    if (((_cookie$credentials3 = cookie.credentials) === null || _cookie$credentials3 === void 0 ? void 0 : _cookie$credentials3.expires_at) > Date.now()) {
      return true;
    } // need to renew id token


    if (cookie.credentials.refresh_token) {
      try {
        var _this$config$openid5, _this$config$openid6;

        const query = {
          grant_type: 'refresh_token',
          client_id: (_this$config$openid5 = this.config.openid) === null || _this$config$openid5 === void 0 ? void 0 : _this$config$openid5.client_id,
          client_secret: (_this$config$openid6 = this.config.openid) === null || _this$config$openid6 === void 0 ? void 0 : _this$config$openid6.client_secret,
          refresh_token: cookie.credentials.refresh_token
        };
        const refreshTokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient); // if no id_token from refresh token call, maybe the Idp doesn't allow refresh id_token

        if (refreshTokenResponse.idToken) {
          cookie.credentials = {
            authHeaderValueExtra: true,
            refresh_token: refreshTokenResponse.refreshToken,
            expires_at: (0, _helper.getExpirationDate)(refreshTokenResponse) // expiresIn is in second

          };
          (0, _cookie_splitter.setExtraAuthStorage)(request, `Bearer ${refreshTokenResponse.idToken}`, this.getExtraAuthStorageOptions());
          return true;
        } else {
          return false;
        }
      } catch (error) {
        this.logger.error(error);
        return false;
      }
    } else {
      // no refresh token, and current token is expired
      return false;
    }
  }

  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      // nextUrl is a key value pair
      const nextUrl = (0, _next_url.composeNextUrlQueryParam)(request, this.coreSetup.http.basePath.serverBasePath);
      return response.redirected({
        headers: {
          location: `${this.coreSetup.http.basePath.serverBasePath}${_common.OPENID_AUTH_LOGIN}?${nextUrl}`
        }
      });
    } else {
      return response.unauthorized();
    }
  }

  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials4;

    let extraValue = '';

    if (!((_cookie$credentials4 = cookie.credentials) !== null && _cookie$credentials4 !== void 0 && _cookie$credentials4.authHeaderValueExtra)) {
      return extraValue;
    }

    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions());
    } catch (error) {
      this.logger.info(error);
    }

    return extraValue;
  }

  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials5;

    const header = {};

    if (cookie.credentials.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        header.authorization = extraAuthStorageValue;
        return header;
      } catch (error) {
        this.logger.error(error); // TODO Re-throw?
        // throw error;
      }
    }

    const authHeaderValue = (_cookie$credentials5 = cookie.credentials) === null || _cookie$credentials5 === void 0 ? void 0 : _cookie$credentials5.authHeaderValue;

    if (authHeaderValue) {
      header.authorization = authHeaderValue;
    }

    return header;
  }

}

exports.OpenIdAuthentication = OpenIdAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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