"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.time2CountWin = exports.getMetrics = exports.addToMetric = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _constants = require("./constants");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const time2CountWin = new Map();
exports.time2CountWin = time2CountWin;

const addToMetric = (entity, action, counter, reportMetadata) => {
  const count = 1; // remove outdated key-value pairs

  trim();
  const timeKey = getKey(Date.now());
  const rollingCounters = time2CountWin.get(timeKey);
  time2CountWin.set(timeKey, updateCounters(entity, action, counter, rollingCounters || _lodash.default.cloneDeep(_constants.DEFAULT_ROLLING_COUNTER), count, reportMetadata));
};

exports.addToMetric = addToMetric;

const getMetrics = () => {
  const preTimeKey = getPreKey(Date.now());
  const rollingCounters = time2CountWin.get(preTimeKey);
  const metrics = buildMetrics(rollingCounters);
  return metrics;
};

exports.getMetrics = getMetrics;

const trim = () => {
  if (time2CountWin.size > _constants.CAPACITY) {
    const currentKey = getKey(Date.now() - _constants.WINDOW * 1000);
    time2CountWin.forEach((_value, key, map) => {
      if (key < currentKey) {
        map.delete(key);
      }
    });
  }
};

const getKey = milliseconds => {
  return Math.floor(milliseconds / 1000 / _constants.INTERVAL);
};

const getPreKey = milliseconds => {
  return getKey(milliseconds) - 1;
};

const isEntity = arg => {
  return arg === 'report' || arg === 'report_definition' || arg === 'report_source';
};

const buildMetrics = rollingCounters => {
  if (!rollingCounters) {
    rollingCounters = _constants.DEFAULT_ROLLING_COUNTER;
  }

  const basicMetrics = _lodash.default.merge(rollingCounters, _constants.GLOBAL_BASIC_COUNTER);

  const overallActionMetrics = {
    request_total: 0,
    request_count: 0,
    success_count: 0,
    failed_request_count_system_error: 0,
    failed_request_count_user_error: 0
  };
  Object.keys(basicMetrics).forEach(keys => {
    if (isEntity(keys)) {
      for (const [action, counters] of Object.entries(basicMetrics[keys])) {
        overallActionMetrics.request_count += (counters === null || counters === void 0 ? void 0 : counters.count) || 0;
        overallActionMetrics.request_total += (counters === null || counters === void 0 ? void 0 : counters.total) || 0;
        overallActionMetrics.failed_request_count_system_error += (counters === null || counters === void 0 ? void 0 : counters.system_error) || 0;
        overallActionMetrics.failed_request_count_user_error += (counters === null || counters === void 0 ? void 0 : counters.user_error) || 0;
      }
    }
  });
  overallActionMetrics.success_count = overallActionMetrics.request_count - (overallActionMetrics.failed_request_count_system_error + overallActionMetrics.failed_request_count_user_error);
  return { ...basicMetrics,
    ...overallActionMetrics
  };
};

const updateCounters = (entity, action, counter, rollingCounter, count, reportMetadata) => {
  // update usage metrics
  if (reportMetadata) {
    const {
      report_definition: {
        report_params: {
          report_source: source,
          core_params: {
            report_format: format
          }
        }
      }
    } = reportMetadata; // @ts-ignore

    rollingCounter[source.toLowerCase().replace(' ', '_')][format]['download'][counter] += count; // update basic counter for total request count

    if (counter === 'count') {
      //@ts-ignore
      _constants.GLOBAL_BASIC_COUNTER[source.toLowerCase().replace(' ', '_')][format]['download']['total']++;
    }
  } else {
    // update action metric, per API
    // @ts-ignore
    rollingCounter[entity][action][counter] += count;

    if (counter === 'count') {
      // @ts-ignore
      _constants.GLOBAL_BASIC_COUNTER[entity][action]['total']++;
    }
  }

  return rollingCounter;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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