"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IndexPatternsService = void 0;

var _ = require(".");

var _index_pattern = require("./index_pattern");

var _ensure_default_index_pattern = require("./ensure_default_index_pattern");

var _fields = require("../fields");

var _common = require("../../../common");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const indexPatternCache = (0, _.createIndexPatternCache)();

class IndexPatternsService {
  constructor({
    uiSettings,
    savedObjectsClient,
    apiClient,
    fieldFormats,
    onNotification,
    onError,
    onUnsupportedTimePattern,
    onRedirectNoIndexPattern = () => {}
  }) {
    _defineProperty(this, "config", void 0);

    _defineProperty(this, "savedObjectsClient", void 0);

    _defineProperty(this, "savedObjectsCache", void 0);

    _defineProperty(this, "apiClient", void 0);

    _defineProperty(this, "fieldFormats", void 0);

    _defineProperty(this, "onNotification", void 0);

    _defineProperty(this, "onError", void 0);

    _defineProperty(this, "onUnsupportedTimePattern", void 0);

    _defineProperty(this, "ensureDefaultIndexPattern", void 0);

    _defineProperty(this, "createFieldList", void 0);

    _defineProperty(this, "createField", void 0);

    _defineProperty(this, "getIds", async (refresh = false) => {
      if (!this.savedObjectsCache || refresh) {
        await this.refreshSavedObjectsCache();
      }

      if (!this.savedObjectsCache) {
        return [];
      }

      return this.savedObjectsCache.map(obj => obj === null || obj === void 0 ? void 0 : obj.id);
    });

    _defineProperty(this, "getTitles", async (refresh = false) => {
      if (!this.savedObjectsCache || refresh) {
        await this.refreshSavedObjectsCache();
      }

      if (!this.savedObjectsCache) {
        return [];
      }

      return this.savedObjectsCache.map(obj => {
        var _obj$attributes;

        return obj === null || obj === void 0 ? void 0 : (_obj$attributes = obj.attributes) === null || _obj$attributes === void 0 ? void 0 : _obj$attributes.title;
      });
    });

    _defineProperty(this, "getFields", async (fields, refresh = false) => {
      if (!this.savedObjectsCache || refresh) {
        await this.refreshSavedObjectsCache();
      }

      if (!this.savedObjectsCache) {
        return [];
      }

      return this.savedObjectsCache.map(obj => {
        const result = {};
        fields.forEach(f => {
          var _obj$attributes2;

          return result[f] = obj[f] || (obj === null || obj === void 0 ? void 0 : (_obj$attributes2 = obj.attributes) === null || _obj$attributes2 === void 0 ? void 0 : _obj$attributes2[f]);
        });
        return result;
      });
    });

    _defineProperty(this, "getFieldsForTimePattern", (options = {}) => {
      return this.apiClient.getFieldsForTimePattern(options);
    });

    _defineProperty(this, "getFieldsForWildcard", (options = {}) => {
      return this.apiClient.getFieldsForWildcard(options);
    });

    _defineProperty(this, "clearCache", id => {
      this.savedObjectsCache = null;

      if (id) {
        indexPatternCache.clear(id);
      } else {
        indexPatternCache.clearAll();
      }
    });

    _defineProperty(this, "getCache", async () => {
      if (!this.savedObjectsCache) {
        await this.refreshSavedObjectsCache();
      }

      return this.savedObjectsCache;
    });

    _defineProperty(this, "getDefault", async () => {
      const defaultIndexPatternId = await this.config.get('defaultIndex');

      if (defaultIndexPatternId) {
        return await this.get(defaultIndexPatternId);
      }

      return null;
    });

    _defineProperty(this, "get", async id => {
      const cache = indexPatternCache.get(id);

      if (cache) {
        return cache;
      }

      const indexPattern = await this.make(id);
      return indexPatternCache.set(id, indexPattern);
    });

    this.apiClient = apiClient;
    this.config = uiSettings;
    this.savedObjectsClient = savedObjectsClient;
    this.fieldFormats = fieldFormats;
    this.onNotification = onNotification;
    this.onError = onError;
    this.onUnsupportedTimePattern = onUnsupportedTimePattern;
    this.ensureDefaultIndexPattern = (0, _ensure_default_index_pattern.createEnsureDefaultIndexPattern)(uiSettings, onRedirectNoIndexPattern);
    this.createFieldList = (0, _fields.getIndexPatternFieldListCreator)({
      fieldFormats,
      onNotification
    });

    this.createField = (indexPattern, spec, shortDotsEnable) => {
      return new _fields.Field(indexPattern, spec, shortDotsEnable, {
        fieldFormats,
        onNotification
      });
    };
  }

  async refreshSavedObjectsCache() {
    this.savedObjectsCache = await this.savedObjectsClient.find({
      type: 'index-pattern',
      fields: ['title'],
      perPage: 10000
    });
  }

  async specToIndexPattern(spec) {
    const shortDotsEnable = await this.config.get(_common.UI_SETTINGS.SHORT_DOTS_ENABLE);
    const metaFields = await this.config.get(_common.UI_SETTINGS.META_FIELDS);
    const uiSettingsValues = await this.config.getAll();
    const indexPattern = new _index_pattern.IndexPattern(spec.id, {
      getConfig: cfg => this.config.get(cfg),
      savedObjectsClient: this.savedObjectsClient,
      apiClient: this.apiClient,
      patternCache: indexPatternCache,
      fieldFormats: this.fieldFormats,
      onNotification: this.onNotification,
      onError: this.onError,
      onUnsupportedTimePattern: this.onUnsupportedTimePattern,
      uiSettingsValues: { ...uiSettingsValues,
        shortDotsEnable,
        metaFields
      }
    });
    indexPattern.initFromSpec(spec);
    return indexPattern;
  }

  async make(id) {
    const shortDotsEnable = await this.config.get(_common.UI_SETTINGS.SHORT_DOTS_ENABLE);
    const metaFields = await this.config.get(_common.UI_SETTINGS.META_FIELDS);
    const uiSettingsValues = await this.config.getAll();
    const indexPattern = new _index_pattern.IndexPattern(id, {
      getConfig: cfg => this.config.get(cfg),
      savedObjectsClient: this.savedObjectsClient,
      apiClient: this.apiClient,
      patternCache: indexPatternCache,
      fieldFormats: this.fieldFormats,
      onNotification: this.onNotification,
      onError: this.onError,
      onUnsupportedTimePattern: this.onUnsupportedTimePattern,
      uiSettingsValues: { ...uiSettingsValues,
        shortDotsEnable,
        metaFields
      }
    });
    return indexPattern.init();
  }
  /**
   * Deletes an index pattern from .kibana index
   * @param indexPatternId: Id of kibana Index Pattern to delete
   */


  async delete(indexPatternId) {
    indexPatternCache.clear(indexPatternId);
    return this.savedObjectsClient.delete('index-pattern', indexPatternId);
  }

}

exports.IndexPatternsService = IndexPatternsService;