"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OllyChatService = void 0;
var _constants = require("../../utils/constants");
var _get_agent = require("../../routes/get_agent");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const MEMORY_ID_FIELD = 'memory_id';
const INTERACTION_ID_FIELD = 'parent_interaction_id';
class OllyChatService {
  constructor(opensearchClientTransport) {
    this.opensearchClientTransport = opensearchClientTransport;
  }
  async getRootAgent() {
    return await (0, _get_agent.getAgent)(_constants.ROOT_AGENT_CONFIG_ID, this.opensearchClientTransport);
  }
  async requestAgentRun(payload) {
    if (payload.memory_id) {
      OllyChatService.abortControllers.set(payload.memory_id, new AbortController());
    }
    const rootAgentId = await this.getRootAgent();
    return await this.callExecuteAgentAPI(payload, rootAgentId);
  }
  async callExecuteAgentAPI(payload, rootAgentId) {
    try {
      var _agentFrameworkRespon;
      const agentFrameworkResponse = await this.opensearchClientTransport.request({
        method: 'POST',
        path: `${_constants.ML_COMMONS_BASE_API}/agents/${rootAgentId}/_execute`,
        body: {
          parameters: payload
        }
      }, {
        /**
         * It is time-consuming for LLM to generate final answer
         * Give it a large timeout window
         */
        requestTimeout: 5 * 60 * 1000,
        /**
         * Do not retry
         */
        maxRetries: 0
      });
      const outputBody = (_agentFrameworkRespon = agentFrameworkResponse.body.inference_results) === null || _agentFrameworkRespon === void 0 || (_agentFrameworkRespon = _agentFrameworkRespon[0]) === null || _agentFrameworkRespon === void 0 ? void 0 : _agentFrameworkRespon.output;
      const conversationIdItem = outputBody === null || outputBody === void 0 ? void 0 : outputBody.find(item => item.name === MEMORY_ID_FIELD);
      const interactionIdItem = outputBody === null || outputBody === void 0 ? void 0 : outputBody.find(item => item.name === INTERACTION_ID_FIELD);
      return {
        /**
         * Interactions will be stored in Agent framework,
         * thus we do not need to return the latest message back.
         */
        messages: [],
        conversationId: (conversationIdItem === null || conversationIdItem === void 0 ? void 0 : conversationIdItem.result) || '',
        interactionId: (interactionIdItem === null || interactionIdItem === void 0 ? void 0 : interactionIdItem.result) || ''
      };
    } catch (error) {
      throw error;
    } finally {
      if (payload.memory_id) {
        OllyChatService.abortControllers.delete(payload.memory_id);
      }
    }
  }
  async requestLLM(payload) {
    const {
      input,
      conversationId
    } = payload;
    const parametersPayload = {
      question: input.content,
      verbose: false
    };
    if (conversationId) {
      parametersPayload.memory_id = conversationId;
    }
    return await this.requestAgentRun(parametersPayload);
  }
  async regenerate(payload) {
    const {
      conversationId,
      interactionId
    } = payload;
    const parametersPayload = {
      memory_id: conversationId,
      regenerate_interaction_id: interactionId,
      verbose: false
    };
    return await this.requestAgentRun(parametersPayload);
  }
  abortAgentExecution(conversationId) {
    if (OllyChatService.abortControllers.has(conversationId)) {
      var _OllyChatService$abor;
      (_OllyChatService$abor = OllyChatService.abortControllers.get(conversationId)) === null || _OllyChatService$abor === void 0 || _OllyChatService$abor.abort();
    }
  }
}
exports.OllyChatService = OllyChatService;
_defineProperty(OllyChatService, "abortControllers", new Map());
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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