##########################################################################
# This file is part of Vacuum Magic
# Copyright (C) 2008-2009 by UPi <upi at sourceforge.net>
##########################################################################

use strict;

##########################################################################
# Background object
##########################################################################
# SetBackground(number)
# FadeInBackground
# FadeOutBackground
# AdvanceBackground
# DrawBackground
##########################################################################

use vars qw($Background $BackgroundNumber $BackgroundFadeTime);
our ($ScreenWidth, $ScreenHeight);

$BackgroundNumber = -1;
$BackgroundFadeTime = 200;

sub SetBackground {
  my ($number) = @_;
  
  $Background = &MakeBackground($number)  if $BackgroundNumber != $number;
  $BackgroundNumber = $number;
  &FadeInBackground();
}

sub MakeBackground {
  my ($number) = @_;
  my (@result);
  
  &Texture::UnloadTransientTextures();
  $number = $number || 0;
  $number = $number % 10;
  
  @result = (
    'bgX' => 0,
    'speed' => 0,
    'fade' => $BackgroundFadeTime,
    'guyX' => 0,
  );
  
  if ($number == 0) {
    return &MakeForestBackground(@result);
  } elsif ($number == 1) {
    return &MakeHalloweenBackground(@result);
  } elsif ($number == 2) {
    return {
      'bgX1' => 0,
      'bgY1' => 0,
      'advance' => \&AdvanceMountainBackground,
      'draw' => \&DrawMountainBackground,
      @result,
    };
  } elsif ($number == 3) {
    return &MakeMarsBackground(@result);
  } elsif ($number == 4) {
    return &MakeDesertBackground(@result);
  } elsif ($number == 5) {
    return &MakeCavernBackground(@result);
  } elsif ($number == 6) {
    return &MakeStarfieldBackground(@result);
  } elsif ($number == 7) {
    return &MakeArticBackground(@result);
  } elsif ($number == 8) {
    return {
      'bgX1' => 0,
      'bgY1' => 0,
      'advance' => \&AdvanceHubbleBackground,
      'draw' => \&DrawHubbleBackground,
      @result,
    };
  } elsif ($number == 9) {
    return &MakeCloudyBackground(@result);
  } elsif ($number == 100) {
    return &MakeCircularBackground(@result);
  }
}

sub FadeInBackground {
  my $self = $Background;
  $self->{fadeIn} = 1;
  $self->{fadeOut} = 0;
}

sub FadeOutBackground {
  my $self = $Background;
  $self->{fadeOut} = 1;
  $self->{fadeIn} = 0;
}

sub DrawBackground {
  my $self = $Background;
  my ($fade);
  
  $fade = $self->{fade};
  return  if $fade >= $BackgroundFadeTime;
  $self->{draw}->();
  if ($fade) {
    $fade = $fade / $BackgroundFadeTime;
    glColor(0,0,0,$fade);
    glDisable(GL_TEXTURE_2D);
    glBegin(GL_QUADS);
      glVertex(0,0); glVertex($ScreenWidth, 0);
      glVertex($ScreenWidth, $ScreenHeight); glVertex(0, $ScreenHeight);
    glEnd();
    glEnable(GL_TEXTURE_2D);
    glColor(1,1,1,1);
  }
}


# ========================================================================
# BACKGROUND ADVANCE AND DRAW FOR EACH BACKGROUND
# ========================================================================


# _AdjustBgY1 is used by Advance methods.
# It sets $Background->{bgY1} based on the average height of the players.
# In the end bgY1 will be somewhere between 0 and -60;

sub _AdjustBgY1 {
  my ($speed) = @_;
  my ($self, $x, $y);
  
  return  unless @Guy::Guys;
  $self = $Background;
  $x = $y = 0;

  foreach (@Guy::Guys) {
    $y -= $_->{y};
    $x -= $_->{x};
  }
  $y = $y / scalar(@Guy::Guys) / 10;
  $x = $x / scalar(@Guy::Guys) / 10;
  $y = 0  if $y > 0;
  $y = -60  if $y < -60;
  if ( abs($self->{bgY1} - $y) <= $speed) {
    $self->{bgY1} = $y;
  } elsif ($self->{bgY1} < $y) {
    $self->{bgY1} += $speed;
  } else {
    $self->{bgY1} -= $speed;
  }
  if ( abs($self->{guyX} - $x) <= $speed) {
    $self->{guyX} = $x;
  } elsif ($self->{guyX} < $x) {
    $self->{guyX} += $speed;
  } else {
    $self->{guyX} -= $speed;
  }
}

sub AdvanceBackground {
  my $self = $Background;
  my ($move, $speed);
  
  if ($self->{fadeIn}) {
    --$self->{fade};
    $self->{fadeIn} = $self->{fade} = 0  if $self->{fade} <= 0;
  } elsif ($self->{fadeOut}) {
    if ( (++$self->{fade}) > $BackgroundFadeTime ) {
      $self->{fadeOut} = 0;
      $self->{fade} = $BackgroundFadeTime;
    }
  }
  
  $speed = $Difficulty ?  0.6 + $Difficulty / 3 : 0;
  $move = $self->{speed};
  if ($move < $speed) {
    $move += 0.00390625;
  } elsif ($move > $speed) {
    $move -= 0.00390625;
  }
  $self->{speed} = $move;
  $self->{advance}->();
}


# = Cloudy background ===================================================


sub MakeCloudyBackground {
  return {
   'clouds' => [],
   'cloudDelay' => 0,
   'advance' => \&AdvanceCloudyBackground,
   'draw' => \&DrawCloudyBackground,
    @_,
  };
}

sub AdvanceCloudyBackground {
  my $self = $Background;
  my $move = $self->{speed};
  my ($clouds, $cloud);
  
  $self->{bgX} += $move;
  
  $clouds = $self->{clouds};
  foreach my $cloud (@$clouds) {
    $cloud->{x} -= $move;
    $cloud->{y} -= $move / 20;
  }
  shift @$clouds  if @$clouds and $clouds->[0]->{x} < -1200;
  $self->{cloudDelay} -= $move;
  if ($self->{cloudDelay} < 0) {
    $self->{cloudDelay} = rand(300) + 100;
    $cloud = {
      'x' => 800,
      'y' => 0,
      'type' => int(rand(3)),
    };
    push @$clouds, $cloud;
  }
}

sub DrawCloudyBackground {
  my $self = $Background;
  my ($bgx, $bgy, $cloud);
  
  $bgx = - ($self->{bgX} / 4 % 1920);
  $bgy = 28 - $Cos[$self->{bgX} / 12 % 800] * 28;
  $bgy = -$bgy;
  $Textures{'background1'}->Blit($bgx,        0, 744, 768,   4, $bgy, 248, 256);
  $Textures{'background2'}->Blit($bgx +  744, 0, 744, 768,   4, $bgy, 248, 256);
  $Textures{'background3'}->Blit($bgx + 1488, 0, 432, 768,   4, $bgy, 144,  -1);
  $Textures{'background1'}->Blit($bgx + 1920, 0, 744, 768,   4, $bgy, 248, 256);
  $Textures{'background2'}->Blit($bgx + 2664, 0, 744, 768,   4, $bgy, 248, 256);
  
  foreach $cloud (@{$self->{clouds}}) {
    if ($cloud->{type} == 0) {
      &DrawCloudLarge($cloud->{x}, $cloud->{y});
    } elsif ($cloud->{type} == 1) {
      &DrawCloud($cloud->{x}, $cloud->{y});
    } elsif ($cloud->{type} == 2) {
      &DrawCloudSmall($cloud->{x}, $cloud->{y});
    }
  }
}

sub DrawCloud {
  my ($cloudX, $cloudY) = @_;
  my ($cloudTexture, $cloudW);
  
  $cloudTexture = $::Textures{'cloud'};
  $cloudW = 254 * 2;
  $cloudTexture->Blit($cloudX,           $cloudY,   $cloudW, 128,    0, 64, 254, 64);
  $cloudTexture->Blit($cloudX + 252,     $cloudY + 128, 256, 64,  126, 32, 128, 32);
  $cloudTexture->Blit($cloudX + $cloudW, $cloudY,       192, 96,    2,  0,  96, 48);
}

sub DrawCloudLarge {
  my ($cloudX, $cloudY) = @_;
  my ($cloudTexture, $cloudW);
  
  $cloudTexture = $::Textures{'cloud'};
  $cloudW = 254 * 2;
  $cloudTexture->Blit($cloudX,           $cloudY,   $cloudW, 128,   0, 64, 254, 64);
  $cloudTexture->Blit($cloudX + 252,     $cloudY + 128, 256, 64,  126, 32, 128, 32);
  $cloudTexture->Blit($cloudX + 504,     $cloudY,        96, 128,  64, 64,  48, 64);
  $cloudTexture->Blit($cloudX + 600,     $cloudY,       288, 192, 112, 32, 144, 96);
  $cloudTexture->Blit($cloudX + 888,     $cloudY,       192, 96,    2,  0,  96, 48);
}

sub DrawCloudSmall {
  my ($cloudX, $cloudY) = @_;
  my ($cloudTexture, $cloudW);
  
  $cloudTexture = $::Textures{'cloud'};
  $cloudW = 254 * 2;
  $cloudTexture->Blit($cloudX,       $cloudY, 124, 96,    0, 78, 64, 48);
  $cloudTexture->Blit($cloudX + 124, $cloudY, 192, 96,    2,  0, 96, 48);
}


# = Mountain background ==================================================

sub AdvanceMountainBackground {
  my $self = $Background;
  my $move = $self->{speed};
  
  $self->{bgX} += $move / 4;
  $self->{bgX1} += $move / 2;
  &_AdjustBgY1(0.1);
}

sub DrawMountainBackground {
  my $self = $Background;
  my ($bgx, $bgy, $bgx1);
  
  # Sky texture is zoomed 1024 x 256 -> 2400 x 600
  # Two textures side-by-side are -> 4800 x 600
  $bgx = ($self->{bgX} % 2400);
  $Textures{'bg3_sky'}->Bind();
  
  &glBegin(GL_QUADS);
  for (my $i = 0; $i < 6; ++$i) {
    &glTexCoord($i/4, 1);       &glVertex( (-$bgx + $i * 600), 0, 0);
    &glTexCoord($i/4, 0);       &glVertex( (-$bgx + $i * 600) * 1.5 - 200, $ScreenHeight, 0);
    &glTexCoord($i/4+0.25, 0);  &glVertex( (-$bgx + $i * 600) * 1.5 - 200 + 900, $ScreenHeight, 0);
    &glTexCoord($i/4+0.25, 1);  &glVertex( (-$bgx + $i * 600) + 600, 0, 0);
  }
  &glEnd();
  
  
  # Mountain texture is zoomed 1024 x 128 -> 3200 x 200; 
  $bgx = ($self->{bgX1} % 3200);
  $Textures{'bg3_mountain'}->Blit(-$bgx, $self->{bgY1}, 3200*2, 200,  0, 0, 1024*2, 128);
}


# = Hubble Space Background ==============================================

sub AdvanceHubbleBackground {
  my $self = $Background;
  my $move = $self->{speed};
  my $y = 0;
  
  $self->{bgX} += $move / 4;
}

sub DrawHubbleBackground {
  my $self = $Background;
  my ($bgx, $bgy, $bgx1);
  
  # Nebula texture is zoomed 1024 x 256 -> 3072 x 768
  
  $bgx = ($self->{bgX} % 3072);
  $bgy = 84 - $Cos[$self->{bgX} / 12 % 800] * 84;
  $bgx1 = ($self->{bgX1} % 1024);
  
  $Textures{'bg4_nebula'}->Blit(-$bgx, -$bgy, 3072*2, 768,      0, 0, 1024*2, 256);
}


# = Mars Lander Background ===============================================

sub MakeMarsBackground {
  $Background = {
    'bgX1' => 0,
    'bgY1' => 0,
    'advance' => \&AdvanceMarsBackground,
    'draw' => \&DrawMarsBackground,
    'tiles' => [ [], [], [] ],
    @_,
  };
  foreach (1..6) { &MarsMakeMoreTiles(); }
  return $Background;
}

sub MarsMakeMoreTiles {
  my $self = $Background;
  foreach (@{$self->{tiles}}) {
    unshift @$_, '';
  }
  &MarsPopTiles();
}

sub MarsPopTiles {
  my $self = $Background;
  foreach (@{$self->{tiles}}) {
    shift @$_;
    push @$_, [int(rand(4))+1, int(rand(4))];
  }
}

sub AdvanceMarsBackground {
  my $self = $Background;
  my $move = $self->{speed};
  
  $self->{bgX} += $move / 4;
  while ($self->{bgX} > 256) {
    $self->{bgX} -= 256;
    &MarsPopTiles();
  }
  $self->{bgX1} += $move;
  &_AdjustBgY1(0.1);
}

sub DrawMarsBackground {
  my $self = $Background;
  my ($bgx, $tilex, $tiley, $bgy, $bgx1, $row, $tileNum, $tileRot);
  
  glDisable(GL_TEXTURE_2D);
  &glBegin(GL_QUADS);
    &glColor( 0, 0, 0.2); &glVertex(0, 0, 0);
    &glColor( 0, 0, 0.0); &glVertex(0, $ScreenHeight, 0);
    &glColor( 0, 0, 0.0); &glVertex($ScreenWidth, $ScreenHeight, 0);
    &glColor( 0, 0, 0.2); &glVertex($ScreenWidth, 0, 0);
  &glEnd();
  glEnable(GL_TEXTURE_2D);
  glColor(1,1,1,1);

  glBlendFunc(GL_SRC_ALPHA,GL_ONE);
  $bgx = $self->{bgX};
  $tiley = $self->{bgY1} / 2;
  foreach $row (@{$self->{tiles}}) {
    $tilex = -$bgx;
    foreach (@$row) {
      ($tileNum, $tileRot) = @$_;
      $Textures{"stars$tileNum"}->RotoBlit($tilex, $tiley, 256, 256, $tileRot);
      $tilex += 255;
    }
    $tiley += 255;
  }
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  # Mountain texture is zoomed 1024 x 128 -> 2048 x 256; 
  $bgx = ($self->{bgX1} % 2048);
  $Textures{'bg5_ground'}->Blit(-$bgx, $self->{bgY1} * 2, 2048*2, 256,  0, 0, 1024*2, 128);
}


# = Artic Background (from Pingus) =======================================

sub MakeArticBackground {
  return {
    'bgX' => 0,
    'bgY1' => 0,
    'advance' => \&AdvanceArticBackground,
    'draw' => \&DrawArticBackground,
    @_,
  };
}

sub AdvanceArticBackground {
  my $self = $Background;
  my $move = $self->{speed};
  
  $self->{bgX} += $move;
  &_AdjustBgY1(1);
}

sub DrawArticBackground {
  my $self = $Background;
  my ($bgx, $bgy, $bgx1);
  
  # Sky texture is zoomed 1024 * 256 -> 2560 * 640;
  $bgx = ($self->{bgX} / 16 % 800);
  $Textures{'artic1'}->Blit(-$bgx, 430 + $self->{bgY1} / 4, 1600, 185, 0, 0, 1024, 128);
  $bgx = ($self->{bgX} / 8 % 800);
  $Textures{'artic2'}->Blit(-$bgx, 108 + $self->{bgY1} / 3, 1600, 362, 0, 0, 1024, 256);
  $bgx = ($self->{bgX} / 4 % 800);
  $Textures{'artic3'}->Blit(-$bgx, 94 + $self->{bgY1} / 2, 1600, 306, 0, 0, 1024, 256);
  $bgx = ($self->{bgX} / 2 % 800);
  $Textures{'artic4'}->Blit(-$bgx, $self->{bgY1}, 1600, 171, 0, 0, 1024, 128);
  $bgx = ($self->{bgX} % 1200);
  $Textures{'artic5'}->Blit(850-$bgx, $self->{bgY1} * 1.5, 302, 104, 0, 0, 256, 86);
}


# = Starfield Background (from Pingus) ===================================

sub MakeStarfieldBackground {
  return {
    'rotate' => 0,
    'bgY1' => 0,
    'advance' => \&AdvanceStarfieldBackground,
    'draw' => \&DrawStarfieldBackground,
    @_,
  };
}

sub AdvanceStarfieldBackground {
  my $self = $Background;
  my $move = $self->{speed};
  
  $self->{rotate} += $move * 3;
  &_AdjustBgY1(1);
}

sub DrawStarfieldBackground {
  my $self = $Background;
  my ($texture, $bgy);
  
  $texture = $::Textures{starfield};
  $bgy = $self->{bgY1};
  &glLoadIdentity();
  &glTranslate(900 + $self->{guyX}, -1000 + $bgy, 0);
  &glRotate($self->{rotate} / 256, 0, 0, 1);
  $texture->Bind();

  &glBegin(GL_QUADS);
  &glTexCoord(0,0);   &glVertex(-2048, -2048, 0);
  &glTexCoord(0,8);   &glVertex(-2048,  2048, 0);
  &glTexCoord(4,8);   &glVertex( 2048,  2048, 0);
  &glTexCoord(4,0);   &glVertex( 2048, -2048, 0);
  &glEnd();
  &glLoadIdentity();
}


# = Circular background (most annoying!) =================================

sub MakeCircularBackground {
  return {
    'advance' => \&AdvanceCircularBackground,
    'draw' => \&DrawCircularBackground,
    @_,
  };
}

sub AdvanceCircularBackground {
  my $self = $Background;
  my $move = $self->{speed};
  
  $self->{anim} ++;
  $self->{bgX} += $move / 4;
  $self->{bgX1} += $move;
  &_AdjustBgY1(0.1);
}

sub DrawCircularBackground {
  my $self = $Background;
  my ($anim, @size);
  
  $anim = $self->{anim};
  @size = (1024, 1024, 0, 0, -1, -1);
  glColor( 0, 0.2, 0.2);
  $Textures{'circles'}->Blit(400 - 512 + $::Cos[$self->{anim} % 800] * 200, 400 - 512 + $::Sin[$self->{anim} % 800] * 200, @size);
  $Textures{'circles'}->Blit(400 - 512 + $::Cos[$self->{anim} * 1.4 % 800] * 150, 400 - 512 + $::Sin[$self->{anim} * 1.6 % 800] * 150, @size);
  $Textures{'circles'}->Blit(400 - 512 + $::Cos[$self->{anim} * 2.3 % 800] * 250, 400 - 512 + $::Sin[$self->{anim} * 2.3 % 800] * 250, @size);
  glColor( 1, 1, 1);
}


# = Cavern Background ====================================================

sub MakeCavernBackground {
  return {
    'bgX' => 0,
    'bgY1' => 0,
    'advance' => \&AdvanceCavernBackground,
    'draw' => \&DrawCavernBackground,
    @_,
  };
}

sub AdvanceCavernBackground {
  my $self = $Background;
  my $move = $self->{speed};
  
  $self->{bgX} += $move;
  &_AdjustBgY1(1);
}

sub DrawCavernBackground {
  my $self = $Background;
  my ($bgx, $bgy, $bgx0);
  
  $bgx0 = $self->{bgX};
  ::glColor(0.4, 0.4, 0.4);
  $bgx = ($bgx0 / 16 % 800);
  $Textures{'cavern2'}->Blit(-$bgx, 0 + $self->{bgY1} / 4, 1600, 800, 0, 0, 1024, 512);
  
  ::glColor(0.0, 0.1, 0.2);
  $bgx = ($bgx0 /10 % 400);
  $Textures{'cavern1'}->Blit(-$bgx, 0 + $self->{bgY1} / 3  + 80, 400*5, 400, 0, 0, 512*5, 512);
  ::glColor(0.2, 0.4, 0.5);
  $bgx = ($bgx0 / 7 % 600);
  $Textures{'cavern1'}->Blit(-$bgx, 0 + $self->{bgY1}      + 50, 600*4, 520, 0, 0, 512*4, 512);
  ::glColor(0.8, 0.8, 0.8);
  $bgx = ($bgx0 / 4 % 800);
  $Textures{'cavern1'}->Blit(-$bgx, 0 + $self->{bgY1} *1.5,      800*3, 690, 0, 0, 512*3, 512);
  ::glColor(1.0, 1.0, 1.0);
}

# = Halloween Background =================================================

sub MakeHalloweenBackground {
  return {
    'bgX' => 0,
    'bgY1' => 0,
    'advance' => \&AdvanceHalloweenBackground,
    'draw' => \&DrawHalloweenBackground,
    @_,
  };
}

sub AdvanceHalloweenBackground {
  my $self = $Background;
  my $move = $self->{speed};
  
  $self->{bgX} += $move;
  &_AdjustBgY1(1);
}

sub DrawHalloweenBackground {
  my $self = $Background;
  my ($bgx, $bgy, $bgx0, $bgy1, $moonx, $moony, $red, $blue);
  
  $bgx0 = $self->{bgX};
  $bgx = -($bgx0 / 7 % 1600);
  $Textures{'halloween4'}->Bind();
  $red = ($Sin[$bgx0 / 20 % 800] + 1) / 5 + 0.1;  # 0.1 .. 0.5
  $blue = 0.6 - $red;
  glBegin(GL_QUADS);
    glColor($red, 0.2, $blue);
    glTexCoord(2,1); glVertex($bgx + 3200, 0, 0);
    glTexCoord(0,1); glVertex($bgx, 0, 0);
    glColor($red/2, 0.1, $blue/2);
    glTexCoord(0,0); glVertex($bgx, 600, 0);
    glTexCoord(2,0); glVertex($bgx + 3200, 600, 0);
  glEnd();
  glColor(1,1,1);

  $moonx = 800 - ($bgx0 / 4 % 1530);
  $moony = 150;
  $Textures{'halloween_moon1'}->Blit($moonx, $moony, 256, 256, 0, 0, 256, 256);
  
  # Castle
  $bgx = 1400 - ($bgx0 / 2.83 % 1950);
  $Textures{'halloween2'}->Blit($bgx, $self->{bgY1} / 2, 512, 512, 0, 0, 512, 512);
  
  # Tree
  $bgx = 800 - ($bgx0 / 2.21 % 1583);
  $bgy1 = 20 + $self->{bgY1};
  $Textures{'halloween3'}->Blit($bgx, $bgy1, 600, 600, 0, 0, 512, 512);
  
  # Fence
  $bgx = ($bgx0 / 2 % 1024);
  $Textures{'halloween1'}->Blit(-$bgx, $bgy1, 2048, 256, 0, 0, 2048, 256);
  
  if ($bgy1 > 0) {
    glDisable(GL_TEXTURE_2D);
    glColor(0,0,0);
    glBegin(GL_QUADS);
      glVertex(0,0,0); glVertex(800,0,0);
      glVertex(800,$bgy1,0); glVertex(0,$bgy1,0);
    glEnd();
    glColor(1,1,1);
    glEnable(GL_TEXTURE_2D);
  }
  
  glBlendFunc(GL_SRC_ALPHA,GL_ONE);
  $Textures{'halloween_moon2'}->Blit($moonx, $moony, 256, 256, 0, 0, 256, 256);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

}


# = Forest Background ====================================================

sub MakeForestBackground {
  return {
    'bgX' => 0,
    'bgY1' => 0,
    'advance' => \&AdvanceForestBackground,
    'draw' => \&DrawForestBackground,
    @_,
  };
}

sub AdvanceForestBackground {
  my $self = $Background;
  my $move = $self->{speed};
  
  $self->{bgX} += $move;
  &_AdjustBgY1(1);
}

sub DrawForestBackground {
  my $self = $Background;
  my ($bgx, $bgy, $bgx0);
  
  $bgx0 = $self->{bgX};
  $bgx = ($bgx0 / 4.8 % 2048);
  $Textures{'ancient_forest3'}->Blit(-$bgx, $self->{bgY1}/3, 4096, 650, 0, 0, 2048, 512);
  
  glColor(0.8,0.8,0.8);
  $bgx0 = $self->{bgX};
  $bgx = ($bgx0 / 2.2 % 650);
  $Textures{'ancient_forest2'}->Blit(-$bgx, $self->{bgY1}/1.4, 1950, 650, 0, 0, 1536, 512);
  
  glColor(0.5,0.5,0.5);
  $bgx0 = $self->{bgX};
  $bgx = ($bgx0 / 1.6 % 2383);
  $Textures{'ancient_forest'}->Blit(800-$bgx, $self->{bgY1}, 750, 670, 0, 0, 512, 512);
  glColor(1,1,1);
}


# = Desert Background ====================================================

sub MakeDesertBackground {
  return {
    'bgX' => 0,
    'bgY1' => 0,
    'advance' => \&AdvanceDesertBackground,
    'draw' => \&DrawDesertBackground,
    @_,
  };
}

sub AdvanceDesertBackground {
  my $self = $Background;
  my $move = $self->{speed};
  
  $self->{bgX} += $move;
  &_AdjustBgY1(1);
}

sub DrawDesertBackground {
  my $self = $Background;
  my ($bgx, $bgy, $bgx0);
  
  glDisable(GL_TEXTURE_2D);
  glBegin(GL_QUADS);
  glColor(0.6,0.6,1); glVertex(0,$ScreenHeight/2,0);  glVertex($ScreenWidth, $ScreenHeight/2, 0);
  glColor(0.9,0.9,1); glVertex($ScreenWidth,$ScreenHeight,0);  glVertex(0, $ScreenHeight, 0);
  glEnd;
  glEnable(GL_TEXTURE_2D);
  glColor(1,1,1);
  
  $bgx0 = $self->{bgX};
  $bgx = ($bgx0  % 800);
  $Textures{'desert'}->Blit(-$bgx, $self->{bgY1}, 1600, 400, 0, 0, 1024, 256);
}

1;

