"use strict";

exports.__esModule = true;
exports.default = void 0;

var _declaration = _interopRequireDefault(require("./declaration"));

var _tokenize = _interopRequireDefault(require("./tokenize"));

var _comment = _interopRequireDefault(require("./comment"));

var _atRule = _interopRequireDefault(require("./at-rule"));

var _root = _interopRequireDefault(require("./root"));

var _rule = _interopRequireDefault(require("./rule"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var Parser =
/*#__PURE__*/
function () {
  function Parser(input) {
    this.input = input;
    this.root = new _root.default();
    this.current = this.root;
    this.spaces = '';
    this.semicolon = false;
    this.createTokenizer();
    this.root.source = {
      input: input,
      start: {
        line: 1,
        column: 1
      }
    };
  }

  var _proto = Parser.prototype;

  _proto.createTokenizer = function createTokenizer() {
    this.tokenizer = (0, _tokenize.default)(this.input);
  };

  _proto.parse = function parse() {
    var token;

    while (!this.tokenizer.endOfFile()) {
      token = this.tokenizer.nextToken();

      switch (token[0]) {
        case 'space':
          this.spaces += token[1];
          break;

        case ';':
          this.freeSemicolon(token);
          break;

        case '}':
          this.end(token);
          break;

        case 'comment':
          this.comment(token);
          break;

        case 'at-word':
          this.atrule(token);
          break;

        case '{':
          this.emptyRule(token);
          break;

        default:
          this.other(token);
          break;
      }
    }

    this.endFile();
  };

  _proto.comment = function comment(token) {
    var node = new _comment.default();
    this.init(node, token[2], token[3]);
    node.source.end = {
      line: token[4],
      column: token[5]
    };
    var text = token[1].slice(2, -2);

    if (/^\s*$/.test(text)) {
      node.text = '';
      node.raws.left = text;
      node.raws.right = '';
    } else {
      var match = text.match(/^(\s*)([^]*[^\s])(\s*)$/);
      node.text = match[2];
      node.raws.left = match[1];
      node.raws.right = match[3];
    }
  };

  _proto.emptyRule = function emptyRule(token) {
    var node = new _rule.default();
    this.init(node, token[2], token[3]);
    node.selector = '';
    node.raws.between = '';
    this.current = node;
  };

  _proto.other = function other(start) {
    var end = false;
    var type = null;
    var colon = false;
    var bracket = null;
    var brackets = [];
    var tokens = [];
    var token = start;

    while (token) {
      type = token[0];
      tokens.push(token);

      if (type === '(' || type === '[') {
        if (!bracket) bracket = token;
        brackets.push(type === '(' ? ')' : ']');
      } else if (brackets.length === 0) {
        if (type === ';') {
          if (colon) {
            this.decl(tokens);
            return;
          } else {
            break;
          }
        } else if (type === '{') {
          this.rule(tokens);
          return;
        } else if (type === '}') {
          this.tokenizer.back(tokens.pop());
          end = true;
          break;
        } else if (type === ':') {
          colon = true;
        }
      } else if (type === brackets[brackets.length - 1]) {
        brackets.pop();
        if (brackets.length === 0) bracket = null;
      }

      token = this.tokenizer.nextToken();
    }

    if (this.tokenizer.endOfFile()) end = true;
    if (brackets.length > 0) this.unclosedBracket(bracket);

    if (end && colon) {
      while (tokens.length) {
        token = tokens[tokens.length - 1][0];
        if (token !== 'space' && token !== 'comment') break;
        this.tokenizer.back(tokens.pop());
      }

      this.decl(tokens);
    } else {
      this.unknownWord(tokens);
    }
  };

  _proto.rule = function rule(tokens) {
    tokens.pop();
    var node = new _rule.default();
    this.init(node, tokens[0][2], tokens[0][3]);
    node.raws.between = this.spacesAndCommentsFromEnd(tokens);
    this.raw(node, 'selector', tokens);
    this.current = node;
  };

  _proto.decl = function decl(tokens) {
    var node = new _declaration.default();
    this.init(node);
    var last = tokens[tokens.length - 1];

    if (last[0] === ';') {
      this.semicolon = true;
      tokens.pop();
    }

    if (last[4]) {
      node.source.end = {
        line: last[4],
        column: last[5]
      };
    } else {
      node.source.end = {
        line: last[2],
        column: last[3]
      };
    }

    while (tokens[0][0] !== 'word') {
      if (tokens.length === 1) this.unknownWord(tokens);
      node.raws.before += tokens.shift()[1];
    }

    node.source.start = {
      line: tokens[0][2],
      column: tokens[0][3]
    };
    node.prop = '';

    while (tokens.length) {
      var type = tokens[0][0];

      if (type === ':' || type === 'space' || type === 'comment') {
        break;
      }

      node.prop += tokens.shift()[1];
    }

    node.raws.between = '';
    var token;

    while (tokens.length) {
      token = tokens.shift();

      if (token[0] === ':') {
        node.raws.between += token[1];
        break;
      } else {
        node.raws.between += token[1];
      }
    }

    if (node.prop[0] === '_' || node.prop[0] === '*') {
      node.raws.before += node.prop[0];
      node.prop = node.prop.slice(1);
    }

    node.raws.between += this.spacesAndCommentsFromStart(tokens);
    this.precheckMissedSemicolon(tokens);

    for (var i = tokens.length - 1; i > 0; i--) {
      token = tokens[i];

      if (token[1].toLowerCase() === '!important') {
        node.important = true;
        var string = this.stringFrom(tokens, i);
        string = this.spacesFromEnd(tokens) + string;
        if (string !== ' !important') node.raws.important = string;
        break;
      } else if (token[1].toLowerCase() === 'important') {
        var cache = tokens.slice(0);
        var str = '';

        for (var j = i; j > 0; j--) {
          var _type = cache[j][0];

          if (str.trim().indexOf('!') === 0 && _type !== 'space') {
            break;
          }

          str = cache.pop()[1] + str;
        }

        if (str.trim().indexOf('!') === 0) {
          node.important = true;
          node.raws.important = str;
          tokens = cache;
        }
      }

      if (token[0] !== 'space' && token[0] !== 'comment') {
        break;
      }
    }

    this.raw(node, 'value', tokens);
    if (node.value.indexOf(':') !== -1) this.checkMissedSemicolon(tokens);
  };

  _proto.atrule = function atrule(token) {
    var node = new _atRule.default();
    node.name = token[1].slice(1);

    if (node.name === '') {
      this.unnamedAtrule(node, token);
    }

    this.init(node, token[2], token[3]);
    var prev;
    var shift;
    var last = false;
    var open = false;
    var params = [];

    while (!this.tokenizer.endOfFile()) {
      token = this.tokenizer.nextToken();

      if (token[0] === ';') {
        node.source.end = {
          line: token[2],
          column: token[3]
        };
        this.semicolon = true;
        break;
      } else if (token[0] === '{') {
        open = true;
        break;
      } else if (token[0] === '}') {
        if (params.length > 0) {
          shift = params.length - 1;
          prev = params[shift];

          while (prev && prev[0] === 'space') {
            prev = params[--shift];
          }

          if (prev) {
            node.source.end = {
              line: prev[4],
              column: prev[5]
            };
          }
        }

        this.end(token);
        break;
      } else {
        params.push(token);
      }

      if (this.tokenizer.endOfFile()) {
        last = true;
        break;
      }
    }

    node.raws.between = this.spacesAndCommentsFromEnd(params);

    if (params.length) {
      node.raws.afterName = this.spacesAndCommentsFromStart(params);
      this.raw(node, 'params', params);

      if (last) {
        token = params[params.length - 1];
        node.source.end = {
          line: token[4],
          column: token[5]
        };
        this.spaces = node.raws.between;
        node.raws.between = '';
      }
    } else {
      node.raws.afterName = '';
      node.params = '';
    }

    if (open) {
      node.nodes = [];
      this.current = node;
    }
  };

  _proto.end = function end(token) {
    if (this.current.nodes && this.current.nodes.length) {
      this.current.raws.semicolon = this.semicolon;
    }

    this.semicolon = false;
    this.current.raws.after = (this.current.raws.after || '') + this.spaces;
    this.spaces = '';

    if (this.current.parent) {
      this.current.source.end = {
        line: token[2],
        column: token[3]
      };
      this.current = this.current.parent;
    } else {
      this.unexpectedClose(token);
    }
  };

  _proto.endFile = function endFile() {
    if (this.current.parent) this.unclosedBlock();

    if (this.current.nodes && this.current.nodes.length) {
      this.current.raws.semicolon = this.semicolon;
    }

    this.current.raws.after = (this.current.raws.after || '') + this.spaces;
  };

  _proto.freeSemicolon = function freeSemicolon(token) {
    this.spaces += token[1];

    if (this.current.nodes) {
      var prev = this.current.nodes[this.current.nodes.length - 1];

      if (prev && prev.type === 'rule' && !prev.raws.ownSemicolon) {
        prev.raws.ownSemicolon = this.spaces;
        this.spaces = '';
      }
    }
  }; // Helpers


  _proto.init = function init(node, line, column) {
    this.current.push(node);
    node.source = {
      start: {
        line: line,
        column: column
      },
      input: this.input
    };
    node.raws.before = this.spaces;
    this.spaces = '';
    if (node.type !== 'comment') this.semicolon = false;
  };

  _proto.raw = function raw(node, prop, tokens) {
    var token, type;
    var length = tokens.length;
    var value = '';
    var clean = true;
    var next, prev;
    var pattern = /^([.|#])?([\w])+/i;

    for (var i = 0; i < length; i += 1) {
      token = tokens[i];
      type = token[0];

      if (type === 'comment' && node.type === 'rule') {
        prev = tokens[i - 1];
        next = tokens[i + 1];

        if (prev[0] !== 'space' && next[0] !== 'space' && pattern.test(prev[1]) && pattern.test(next[1])) {
          value += token[1];
        } else {
          clean = false;
        }

        continue;
      }

      if (type === 'comment' || type === 'space' && i === length - 1) {
        clean = false;
      } else {
        value += token[1];
      }
    }

    if (!clean) {
      var raw = tokens.reduce(function (all, i) {
        return all + i[1];
      }, '');
      node.raws[prop] = {
        value: value,
        raw: raw
      };
    }

    node[prop] = value;
  };

  _proto.spacesAndCommentsFromEnd = function spacesAndCommentsFromEnd(tokens) {
    var lastTokenType;
    var spaces = '';

    while (tokens.length) {
      lastTokenType = tokens[tokens.length - 1][0];
      if (lastTokenType !== 'space' && lastTokenType !== 'comment') break;
      spaces = tokens.pop()[1] + spaces;
    }

    return spaces;
  };

  _proto.spacesAndCommentsFromStart = function spacesAndCommentsFromStart(tokens) {
    var next;
    var spaces = '';

    while (tokens.length) {
      next = tokens[0][0];
      if (next !== 'space' && next !== 'comment') break;
      spaces += tokens.shift()[1];
    }

    return spaces;
  };

  _proto.spacesFromEnd = function spacesFromEnd(tokens) {
    var lastTokenType;
    var spaces = '';

    while (tokens.length) {
      lastTokenType = tokens[tokens.length - 1][0];
      if (lastTokenType !== 'space') break;
      spaces = tokens.pop()[1] + spaces;
    }

    return spaces;
  };

  _proto.stringFrom = function stringFrom(tokens, from) {
    var result = '';

    for (var i = from; i < tokens.length; i++) {
      result += tokens[i][1];
    }

    tokens.splice(from, tokens.length - from);
    return result;
  };

  _proto.colon = function colon(tokens) {
    var brackets = 0;
    var token, type, prev;

    for (var i = 0; i < tokens.length; i++) {
      token = tokens[i];
      type = token[0];

      if (type === '(') {
        brackets += 1;
      }

      if (type === ')') {
        brackets -= 1;
      }

      if (brackets === 0 && type === ':') {
        if (!prev) {
          this.doubleColon(token);
        } else if (prev[0] === 'word' && prev[1] === 'progid') {
          continue;
        } else {
          return i;
        }
      }

      prev = token;
    }

    return false;
  }; // Errors


  _proto.unclosedBracket = function unclosedBracket(bracket) {
    throw this.input.error('Unclosed bracket', bracket[2], bracket[3]);
  };

  _proto.unknownWord = function unknownWord(tokens) {
    throw this.input.error('Unknown word', tokens[0][2], tokens[0][3]);
  };

  _proto.unexpectedClose = function unexpectedClose(token) {
    throw this.input.error('Unexpected }', token[2], token[3]);
  };

  _proto.unclosedBlock = function unclosedBlock() {
    var pos = this.current.source.start;
    throw this.input.error('Unclosed block', pos.line, pos.column);
  };

  _proto.doubleColon = function doubleColon(token) {
    throw this.input.error('Double colon', token[2], token[3]);
  };

  _proto.unnamedAtrule = function unnamedAtrule(node, token) {
    throw this.input.error('At-rule without name', token[2], token[3]);
  };

  _proto.precheckMissedSemicolon = function precheckMissedSemicolon()
  /* tokens */
  {// Hook for Safe Parser
  };

  _proto.checkMissedSemicolon = function checkMissedSemicolon(tokens) {
    var colon = this.colon(tokens);
    if (colon === false) return;
    var founded = 0;
    var token;

    for (var j = colon - 1; j >= 0; j--) {
      token = tokens[j];

      if (token[0] !== 'space') {
        founded += 1;
        if (founded === 2) break;
      }
    }

    throw this.input.error('Missed semicolon', token[2], token[3]);
  };

  return Parser;
}();

exports.default = Parser;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInBhcnNlci5lczYiXSwibmFtZXMiOlsiUGFyc2VyIiwiaW5wdXQiLCJyb290IiwiUm9vdCIsImN1cnJlbnQiLCJzcGFjZXMiLCJzZW1pY29sb24iLCJjcmVhdGVUb2tlbml6ZXIiLCJzb3VyY2UiLCJzdGFydCIsImxpbmUiLCJjb2x1bW4iLCJ0b2tlbml6ZXIiLCJwYXJzZSIsInRva2VuIiwiZW5kT2ZGaWxlIiwibmV4dFRva2VuIiwiZnJlZVNlbWljb2xvbiIsImVuZCIsImNvbW1lbnQiLCJhdHJ1bGUiLCJlbXB0eVJ1bGUiLCJvdGhlciIsImVuZEZpbGUiLCJub2RlIiwiQ29tbWVudCIsImluaXQiLCJ0ZXh0Iiwic2xpY2UiLCJ0ZXN0IiwicmF3cyIsImxlZnQiLCJyaWdodCIsIm1hdGNoIiwiUnVsZSIsInNlbGVjdG9yIiwiYmV0d2VlbiIsInR5cGUiLCJjb2xvbiIsImJyYWNrZXQiLCJicmFja2V0cyIsInRva2VucyIsInB1c2giLCJsZW5ndGgiLCJkZWNsIiwicnVsZSIsImJhY2siLCJwb3AiLCJ1bmNsb3NlZEJyYWNrZXQiLCJ1bmtub3duV29yZCIsInNwYWNlc0FuZENvbW1lbnRzRnJvbUVuZCIsInJhdyIsIkRlY2xhcmF0aW9uIiwibGFzdCIsImJlZm9yZSIsInNoaWZ0IiwicHJvcCIsInNwYWNlc0FuZENvbW1lbnRzRnJvbVN0YXJ0IiwicHJlY2hlY2tNaXNzZWRTZW1pY29sb24iLCJpIiwidG9Mb3dlckNhc2UiLCJpbXBvcnRhbnQiLCJzdHJpbmciLCJzdHJpbmdGcm9tIiwic3BhY2VzRnJvbUVuZCIsImNhY2hlIiwic3RyIiwiaiIsInRyaW0iLCJpbmRleE9mIiwidmFsdWUiLCJjaGVja01pc3NlZFNlbWljb2xvbiIsIkF0UnVsZSIsIm5hbWUiLCJ1bm5hbWVkQXRydWxlIiwicHJldiIsIm9wZW4iLCJwYXJhbXMiLCJhZnRlck5hbWUiLCJub2RlcyIsImFmdGVyIiwicGFyZW50IiwidW5leHBlY3RlZENsb3NlIiwidW5jbG9zZWRCbG9jayIsIm93blNlbWljb2xvbiIsImNsZWFuIiwibmV4dCIsInBhdHRlcm4iLCJyZWR1Y2UiLCJhbGwiLCJsYXN0VG9rZW5UeXBlIiwiZnJvbSIsInJlc3VsdCIsInNwbGljZSIsImRvdWJsZUNvbG9uIiwiZXJyb3IiLCJwb3MiLCJmb3VuZGVkIl0sIm1hcHBpbmdzIjoiOzs7OztBQUFBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOzs7O0lBRXFCQSxNOzs7QUFDbkIsa0JBQWFDLEtBQWIsRUFBb0I7QUFDbEIsU0FBS0EsS0FBTCxHQUFhQSxLQUFiO0FBRUEsU0FBS0MsSUFBTCxHQUFZLElBQUlDLGFBQUosRUFBWjtBQUNBLFNBQUtDLE9BQUwsR0FBZSxLQUFLRixJQUFwQjtBQUNBLFNBQUtHLE1BQUwsR0FBYyxFQUFkO0FBQ0EsU0FBS0MsU0FBTCxHQUFpQixLQUFqQjtBQUVBLFNBQUtDLGVBQUw7QUFDQSxTQUFLTCxJQUFMLENBQVVNLE1BQVYsR0FBbUI7QUFBRVAsTUFBQUEsS0FBSyxFQUFMQSxLQUFGO0FBQVNRLE1BQUFBLEtBQUssRUFBRTtBQUFFQyxRQUFBQSxJQUFJLEVBQUUsQ0FBUjtBQUFXQyxRQUFBQSxNQUFNLEVBQUU7QUFBbkI7QUFBaEIsS0FBbkI7QUFDRDs7OztTQUVESixlLDhCQUFtQjtBQUNqQixTQUFLSyxTQUFMLEdBQWlCLHVCQUFVLEtBQUtYLEtBQWYsQ0FBakI7QUFDRCxHOztTQUVEWSxLLG9CQUFTO0FBQ1AsUUFBSUMsS0FBSjs7QUFDQSxXQUFPLENBQUMsS0FBS0YsU0FBTCxDQUFlRyxTQUFmLEVBQVIsRUFBb0M7QUFDbENELE1BQUFBLEtBQUssR0FBRyxLQUFLRixTQUFMLENBQWVJLFNBQWYsRUFBUjs7QUFFQSxjQUFRRixLQUFLLENBQUMsQ0FBRCxDQUFiO0FBQ0UsYUFBSyxPQUFMO0FBQ0UsZUFBS1QsTUFBTCxJQUFlUyxLQUFLLENBQUMsQ0FBRCxDQUFwQjtBQUNBOztBQUVGLGFBQUssR0FBTDtBQUNFLGVBQUtHLGFBQUwsQ0FBbUJILEtBQW5CO0FBQ0E7O0FBRUYsYUFBSyxHQUFMO0FBQ0UsZUFBS0ksR0FBTCxDQUFTSixLQUFUO0FBQ0E7O0FBRUYsYUFBSyxTQUFMO0FBQ0UsZUFBS0ssT0FBTCxDQUFhTCxLQUFiO0FBQ0E7O0FBRUYsYUFBSyxTQUFMO0FBQ0UsZUFBS00sTUFBTCxDQUFZTixLQUFaO0FBQ0E7O0FBRUYsYUFBSyxHQUFMO0FBQ0UsZUFBS08sU0FBTCxDQUFlUCxLQUFmO0FBQ0E7O0FBRUY7QUFDRSxlQUFLUSxLQUFMLENBQVdSLEtBQVg7QUFDQTtBQTNCSjtBQTZCRDs7QUFDRCxTQUFLUyxPQUFMO0FBQ0QsRzs7U0FFREosTyxvQkFBU0wsSyxFQUFPO0FBQ2QsUUFBSVUsSUFBSSxHQUFHLElBQUlDLGdCQUFKLEVBQVg7QUFDQSxTQUFLQyxJQUFMLENBQVVGLElBQVYsRUFBZ0JWLEtBQUssQ0FBQyxDQUFELENBQXJCLEVBQTBCQSxLQUFLLENBQUMsQ0FBRCxDQUEvQjtBQUNBVSxJQUFBQSxJQUFJLENBQUNoQixNQUFMLENBQVlVLEdBQVosR0FBa0I7QUFBRVIsTUFBQUEsSUFBSSxFQUFFSSxLQUFLLENBQUMsQ0FBRCxDQUFiO0FBQWtCSCxNQUFBQSxNQUFNLEVBQUVHLEtBQUssQ0FBQyxDQUFEO0FBQS9CLEtBQWxCO0FBRUEsUUFBSWEsSUFBSSxHQUFHYixLQUFLLENBQUMsQ0FBRCxDQUFMLENBQVNjLEtBQVQsQ0FBZSxDQUFmLEVBQWtCLENBQUMsQ0FBbkIsQ0FBWDs7QUFDQSxRQUFJLFFBQVFDLElBQVIsQ0FBYUYsSUFBYixDQUFKLEVBQXdCO0FBQ3RCSCxNQUFBQSxJQUFJLENBQUNHLElBQUwsR0FBWSxFQUFaO0FBQ0FILE1BQUFBLElBQUksQ0FBQ00sSUFBTCxDQUFVQyxJQUFWLEdBQWlCSixJQUFqQjtBQUNBSCxNQUFBQSxJQUFJLENBQUNNLElBQUwsQ0FBVUUsS0FBVixHQUFrQixFQUFsQjtBQUNELEtBSkQsTUFJTztBQUNMLFVBQUlDLEtBQUssR0FBR04sSUFBSSxDQUFDTSxLQUFMLENBQVcseUJBQVgsQ0FBWjtBQUNBVCxNQUFBQSxJQUFJLENBQUNHLElBQUwsR0FBWU0sS0FBSyxDQUFDLENBQUQsQ0FBakI7QUFDQVQsTUFBQUEsSUFBSSxDQUFDTSxJQUFMLENBQVVDLElBQVYsR0FBaUJFLEtBQUssQ0FBQyxDQUFELENBQXRCO0FBQ0FULE1BQUFBLElBQUksQ0FBQ00sSUFBTCxDQUFVRSxLQUFWLEdBQWtCQyxLQUFLLENBQUMsQ0FBRCxDQUF2QjtBQUNEO0FBQ0YsRzs7U0FFRFosUyxzQkFBV1AsSyxFQUFPO0FBQ2hCLFFBQUlVLElBQUksR0FBRyxJQUFJVSxhQUFKLEVBQVg7QUFDQSxTQUFLUixJQUFMLENBQVVGLElBQVYsRUFBZ0JWLEtBQUssQ0FBQyxDQUFELENBQXJCLEVBQTBCQSxLQUFLLENBQUMsQ0FBRCxDQUEvQjtBQUNBVSxJQUFBQSxJQUFJLENBQUNXLFFBQUwsR0FBZ0IsRUFBaEI7QUFDQVgsSUFBQUEsSUFBSSxDQUFDTSxJQUFMLENBQVVNLE9BQVYsR0FBb0IsRUFBcEI7QUFDQSxTQUFLaEMsT0FBTCxHQUFlb0IsSUFBZjtBQUNELEc7O1NBRURGLEssa0JBQU9iLEssRUFBTztBQUNaLFFBQUlTLEdBQUcsR0FBRyxLQUFWO0FBQ0EsUUFBSW1CLElBQUksR0FBRyxJQUFYO0FBQ0EsUUFBSUMsS0FBSyxHQUFHLEtBQVo7QUFDQSxRQUFJQyxPQUFPLEdBQUcsSUFBZDtBQUNBLFFBQUlDLFFBQVEsR0FBRyxFQUFmO0FBRUEsUUFBSUMsTUFBTSxHQUFHLEVBQWI7QUFDQSxRQUFJM0IsS0FBSyxHQUFHTCxLQUFaOztBQUNBLFdBQU9LLEtBQVAsRUFBYztBQUNadUIsTUFBQUEsSUFBSSxHQUFHdkIsS0FBSyxDQUFDLENBQUQsQ0FBWjtBQUNBMkIsTUFBQUEsTUFBTSxDQUFDQyxJQUFQLENBQVk1QixLQUFaOztBQUVBLFVBQUl1QixJQUFJLEtBQUssR0FBVCxJQUFnQkEsSUFBSSxLQUFLLEdBQTdCLEVBQWtDO0FBQ2hDLFlBQUksQ0FBQ0UsT0FBTCxFQUFjQSxPQUFPLEdBQUd6QixLQUFWO0FBQ2QwQixRQUFBQSxRQUFRLENBQUNFLElBQVQsQ0FBY0wsSUFBSSxLQUFLLEdBQVQsR0FBZSxHQUFmLEdBQXFCLEdBQW5DO0FBQ0QsT0FIRCxNQUdPLElBQUlHLFFBQVEsQ0FBQ0csTUFBVCxLQUFvQixDQUF4QixFQUEyQjtBQUNoQyxZQUFJTixJQUFJLEtBQUssR0FBYixFQUFrQjtBQUNoQixjQUFJQyxLQUFKLEVBQVc7QUFDVCxpQkFBS00sSUFBTCxDQUFVSCxNQUFWO0FBQ0E7QUFDRCxXQUhELE1BR087QUFDTDtBQUNEO0FBQ0YsU0FQRCxNQU9PLElBQUlKLElBQUksS0FBSyxHQUFiLEVBQWtCO0FBQ3ZCLGVBQUtRLElBQUwsQ0FBVUosTUFBVjtBQUNBO0FBQ0QsU0FITSxNQUdBLElBQUlKLElBQUksS0FBSyxHQUFiLEVBQWtCO0FBQ3ZCLGVBQUt6QixTQUFMLENBQWVrQyxJQUFmLENBQW9CTCxNQUFNLENBQUNNLEdBQVAsRUFBcEI7QUFDQTdCLFVBQUFBLEdBQUcsR0FBRyxJQUFOO0FBQ0E7QUFDRCxTQUpNLE1BSUEsSUFBSW1CLElBQUksS0FBSyxHQUFiLEVBQWtCO0FBQ3ZCQyxVQUFBQSxLQUFLLEdBQUcsSUFBUjtBQUNEO0FBQ0YsT0FsQk0sTUFrQkEsSUFBSUQsSUFBSSxLQUFLRyxRQUFRLENBQUNBLFFBQVEsQ0FBQ0csTUFBVCxHQUFrQixDQUFuQixDQUFyQixFQUE0QztBQUNqREgsUUFBQUEsUUFBUSxDQUFDTyxHQUFUO0FBQ0EsWUFBSVAsUUFBUSxDQUFDRyxNQUFULEtBQW9CLENBQXhCLEVBQTJCSixPQUFPLEdBQUcsSUFBVjtBQUM1Qjs7QUFFRHpCLE1BQUFBLEtBQUssR0FBRyxLQUFLRixTQUFMLENBQWVJLFNBQWYsRUFBUjtBQUNEOztBQUVELFFBQUksS0FBS0osU0FBTCxDQUFlRyxTQUFmLEVBQUosRUFBZ0NHLEdBQUcsR0FBRyxJQUFOO0FBQ2hDLFFBQUlzQixRQUFRLENBQUNHLE1BQVQsR0FBa0IsQ0FBdEIsRUFBeUIsS0FBS0ssZUFBTCxDQUFxQlQsT0FBckI7O0FBRXpCLFFBQUlyQixHQUFHLElBQUlvQixLQUFYLEVBQWtCO0FBQ2hCLGFBQU9HLE1BQU0sQ0FBQ0UsTUFBZCxFQUFzQjtBQUNwQjdCLFFBQUFBLEtBQUssR0FBRzJCLE1BQU0sQ0FBQ0EsTUFBTSxDQUFDRSxNQUFQLEdBQWdCLENBQWpCLENBQU4sQ0FBMEIsQ0FBMUIsQ0FBUjtBQUNBLFlBQUk3QixLQUFLLEtBQUssT0FBVixJQUFxQkEsS0FBSyxLQUFLLFNBQW5DLEVBQThDO0FBQzlDLGFBQUtGLFNBQUwsQ0FBZWtDLElBQWYsQ0FBb0JMLE1BQU0sQ0FBQ00sR0FBUCxFQUFwQjtBQUNEOztBQUNELFdBQUtILElBQUwsQ0FBVUgsTUFBVjtBQUNELEtBUEQsTUFPTztBQUNMLFdBQUtRLFdBQUwsQ0FBaUJSLE1BQWpCO0FBQ0Q7QUFDRixHOztTQUVESSxJLGlCQUFNSixNLEVBQVE7QUFDWkEsSUFBQUEsTUFBTSxDQUFDTSxHQUFQO0FBRUEsUUFBSXZCLElBQUksR0FBRyxJQUFJVSxhQUFKLEVBQVg7QUFDQSxTQUFLUixJQUFMLENBQVVGLElBQVYsRUFBZ0JpQixNQUFNLENBQUMsQ0FBRCxDQUFOLENBQVUsQ0FBVixDQUFoQixFQUE4QkEsTUFBTSxDQUFDLENBQUQsQ0FBTixDQUFVLENBQVYsQ0FBOUI7QUFFQWpCLElBQUFBLElBQUksQ0FBQ00sSUFBTCxDQUFVTSxPQUFWLEdBQW9CLEtBQUtjLHdCQUFMLENBQThCVCxNQUE5QixDQUFwQjtBQUNBLFNBQUtVLEdBQUwsQ0FBUzNCLElBQVQsRUFBZSxVQUFmLEVBQTJCaUIsTUFBM0I7QUFDQSxTQUFLckMsT0FBTCxHQUFlb0IsSUFBZjtBQUNELEc7O1NBRURvQixJLGlCQUFNSCxNLEVBQVE7QUFDWixRQUFJakIsSUFBSSxHQUFHLElBQUk0QixvQkFBSixFQUFYO0FBQ0EsU0FBSzFCLElBQUwsQ0FBVUYsSUFBVjtBQUVBLFFBQUk2QixJQUFJLEdBQUdaLE1BQU0sQ0FBQ0EsTUFBTSxDQUFDRSxNQUFQLEdBQWdCLENBQWpCLENBQWpCOztBQUNBLFFBQUlVLElBQUksQ0FBQyxDQUFELENBQUosS0FBWSxHQUFoQixFQUFxQjtBQUNuQixXQUFLL0MsU0FBTCxHQUFpQixJQUFqQjtBQUNBbUMsTUFBQUEsTUFBTSxDQUFDTSxHQUFQO0FBQ0Q7O0FBQ0QsUUFBSU0sSUFBSSxDQUFDLENBQUQsQ0FBUixFQUFhO0FBQ1g3QixNQUFBQSxJQUFJLENBQUNoQixNQUFMLENBQVlVLEdBQVosR0FBa0I7QUFBRVIsUUFBQUEsSUFBSSxFQUFFMkMsSUFBSSxDQUFDLENBQUQsQ0FBWjtBQUFpQjFDLFFBQUFBLE1BQU0sRUFBRTBDLElBQUksQ0FBQyxDQUFEO0FBQTdCLE9BQWxCO0FBQ0QsS0FGRCxNQUVPO0FBQ0w3QixNQUFBQSxJQUFJLENBQUNoQixNQUFMLENBQVlVLEdBQVosR0FBa0I7QUFBRVIsUUFBQUEsSUFBSSxFQUFFMkMsSUFBSSxDQUFDLENBQUQsQ0FBWjtBQUFpQjFDLFFBQUFBLE1BQU0sRUFBRTBDLElBQUksQ0FBQyxDQUFEO0FBQTdCLE9BQWxCO0FBQ0Q7O0FBRUQsV0FBT1osTUFBTSxDQUFDLENBQUQsQ0FBTixDQUFVLENBQVYsTUFBaUIsTUFBeEIsRUFBZ0M7QUFDOUIsVUFBSUEsTUFBTSxDQUFDRSxNQUFQLEtBQWtCLENBQXRCLEVBQXlCLEtBQUtNLFdBQUwsQ0FBaUJSLE1BQWpCO0FBQ3pCakIsTUFBQUEsSUFBSSxDQUFDTSxJQUFMLENBQVV3QixNQUFWLElBQW9CYixNQUFNLENBQUNjLEtBQVAsR0FBZSxDQUFmLENBQXBCO0FBQ0Q7O0FBQ0QvQixJQUFBQSxJQUFJLENBQUNoQixNQUFMLENBQVlDLEtBQVosR0FBb0I7QUFBRUMsTUFBQUEsSUFBSSxFQUFFK0IsTUFBTSxDQUFDLENBQUQsQ0FBTixDQUFVLENBQVYsQ0FBUjtBQUFzQjlCLE1BQUFBLE1BQU0sRUFBRThCLE1BQU0sQ0FBQyxDQUFELENBQU4sQ0FBVSxDQUFWO0FBQTlCLEtBQXBCO0FBRUFqQixJQUFBQSxJQUFJLENBQUNnQyxJQUFMLEdBQVksRUFBWjs7QUFDQSxXQUFPZixNQUFNLENBQUNFLE1BQWQsRUFBc0I7QUFDcEIsVUFBSU4sSUFBSSxHQUFHSSxNQUFNLENBQUMsQ0FBRCxDQUFOLENBQVUsQ0FBVixDQUFYOztBQUNBLFVBQUlKLElBQUksS0FBSyxHQUFULElBQWdCQSxJQUFJLEtBQUssT0FBekIsSUFBb0NBLElBQUksS0FBSyxTQUFqRCxFQUE0RDtBQUMxRDtBQUNEOztBQUNEYixNQUFBQSxJQUFJLENBQUNnQyxJQUFMLElBQWFmLE1BQU0sQ0FBQ2MsS0FBUCxHQUFlLENBQWYsQ0FBYjtBQUNEOztBQUVEL0IsSUFBQUEsSUFBSSxDQUFDTSxJQUFMLENBQVVNLE9BQVYsR0FBb0IsRUFBcEI7QUFFQSxRQUFJdEIsS0FBSjs7QUFDQSxXQUFPMkIsTUFBTSxDQUFDRSxNQUFkLEVBQXNCO0FBQ3BCN0IsTUFBQUEsS0FBSyxHQUFHMkIsTUFBTSxDQUFDYyxLQUFQLEVBQVI7O0FBRUEsVUFBSXpDLEtBQUssQ0FBQyxDQUFELENBQUwsS0FBYSxHQUFqQixFQUFzQjtBQUNwQlUsUUFBQUEsSUFBSSxDQUFDTSxJQUFMLENBQVVNLE9BQVYsSUFBcUJ0QixLQUFLLENBQUMsQ0FBRCxDQUExQjtBQUNBO0FBQ0QsT0FIRCxNQUdPO0FBQ0xVLFFBQUFBLElBQUksQ0FBQ00sSUFBTCxDQUFVTSxPQUFWLElBQXFCdEIsS0FBSyxDQUFDLENBQUQsQ0FBMUI7QUFDRDtBQUNGOztBQUVELFFBQUlVLElBQUksQ0FBQ2dDLElBQUwsQ0FBVSxDQUFWLE1BQWlCLEdBQWpCLElBQXdCaEMsSUFBSSxDQUFDZ0MsSUFBTCxDQUFVLENBQVYsTUFBaUIsR0FBN0MsRUFBa0Q7QUFDaERoQyxNQUFBQSxJQUFJLENBQUNNLElBQUwsQ0FBVXdCLE1BQVYsSUFBb0I5QixJQUFJLENBQUNnQyxJQUFMLENBQVUsQ0FBVixDQUFwQjtBQUNBaEMsTUFBQUEsSUFBSSxDQUFDZ0MsSUFBTCxHQUFZaEMsSUFBSSxDQUFDZ0MsSUFBTCxDQUFVNUIsS0FBVixDQUFnQixDQUFoQixDQUFaO0FBQ0Q7O0FBQ0RKLElBQUFBLElBQUksQ0FBQ00sSUFBTCxDQUFVTSxPQUFWLElBQXFCLEtBQUtxQiwwQkFBTCxDQUFnQ2hCLE1BQWhDLENBQXJCO0FBQ0EsU0FBS2lCLHVCQUFMLENBQTZCakIsTUFBN0I7O0FBRUEsU0FBSyxJQUFJa0IsQ0FBQyxHQUFHbEIsTUFBTSxDQUFDRSxNQUFQLEdBQWdCLENBQTdCLEVBQWdDZ0IsQ0FBQyxHQUFHLENBQXBDLEVBQXVDQSxDQUFDLEVBQXhDLEVBQTRDO0FBQzFDN0MsTUFBQUEsS0FBSyxHQUFHMkIsTUFBTSxDQUFDa0IsQ0FBRCxDQUFkOztBQUNBLFVBQUk3QyxLQUFLLENBQUMsQ0FBRCxDQUFMLENBQVM4QyxXQUFULE9BQTJCLFlBQS9CLEVBQTZDO0FBQzNDcEMsUUFBQUEsSUFBSSxDQUFDcUMsU0FBTCxHQUFpQixJQUFqQjtBQUNBLFlBQUlDLE1BQU0sR0FBRyxLQUFLQyxVQUFMLENBQWdCdEIsTUFBaEIsRUFBd0JrQixDQUF4QixDQUFiO0FBQ0FHLFFBQUFBLE1BQU0sR0FBRyxLQUFLRSxhQUFMLENBQW1CdkIsTUFBbkIsSUFBNkJxQixNQUF0QztBQUNBLFlBQUlBLE1BQU0sS0FBSyxhQUFmLEVBQThCdEMsSUFBSSxDQUFDTSxJQUFMLENBQVUrQixTQUFWLEdBQXNCQyxNQUF0QjtBQUM5QjtBQUNELE9BTkQsTUFNTyxJQUFJaEQsS0FBSyxDQUFDLENBQUQsQ0FBTCxDQUFTOEMsV0FBVCxPQUEyQixXQUEvQixFQUE0QztBQUNqRCxZQUFJSyxLQUFLLEdBQUd4QixNQUFNLENBQUNiLEtBQVAsQ0FBYSxDQUFiLENBQVo7QUFDQSxZQUFJc0MsR0FBRyxHQUFHLEVBQVY7O0FBQ0EsYUFBSyxJQUFJQyxDQUFDLEdBQUdSLENBQWIsRUFBZ0JRLENBQUMsR0FBRyxDQUFwQixFQUF1QkEsQ0FBQyxFQUF4QixFQUE0QjtBQUMxQixjQUFJOUIsS0FBSSxHQUFHNEIsS0FBSyxDQUFDRSxDQUFELENBQUwsQ0FBUyxDQUFULENBQVg7O0FBQ0EsY0FBSUQsR0FBRyxDQUFDRSxJQUFKLEdBQVdDLE9BQVgsQ0FBbUIsR0FBbkIsTUFBNEIsQ0FBNUIsSUFBaUNoQyxLQUFJLEtBQUssT0FBOUMsRUFBdUQ7QUFDckQ7QUFDRDs7QUFDRDZCLFVBQUFBLEdBQUcsR0FBR0QsS0FBSyxDQUFDbEIsR0FBTixHQUFZLENBQVosSUFBaUJtQixHQUF2QjtBQUNEOztBQUNELFlBQUlBLEdBQUcsQ0FBQ0UsSUFBSixHQUFXQyxPQUFYLENBQW1CLEdBQW5CLE1BQTRCLENBQWhDLEVBQW1DO0FBQ2pDN0MsVUFBQUEsSUFBSSxDQUFDcUMsU0FBTCxHQUFpQixJQUFqQjtBQUNBckMsVUFBQUEsSUFBSSxDQUFDTSxJQUFMLENBQVUrQixTQUFWLEdBQXNCSyxHQUF0QjtBQUNBekIsVUFBQUEsTUFBTSxHQUFHd0IsS0FBVDtBQUNEO0FBQ0Y7O0FBRUQsVUFBSW5ELEtBQUssQ0FBQyxDQUFELENBQUwsS0FBYSxPQUFiLElBQXdCQSxLQUFLLENBQUMsQ0FBRCxDQUFMLEtBQWEsU0FBekMsRUFBb0Q7QUFDbEQ7QUFDRDtBQUNGOztBQUVELFNBQUtxQyxHQUFMLENBQVMzQixJQUFULEVBQWUsT0FBZixFQUF3QmlCLE1BQXhCO0FBRUEsUUFBSWpCLElBQUksQ0FBQzhDLEtBQUwsQ0FBV0QsT0FBWCxDQUFtQixHQUFuQixNQUE0QixDQUFDLENBQWpDLEVBQW9DLEtBQUtFLG9CQUFMLENBQTBCOUIsTUFBMUI7QUFDckMsRzs7U0FFRHJCLE0sbUJBQVFOLEssRUFBTztBQUNiLFFBQUlVLElBQUksR0FBRyxJQUFJZ0QsZUFBSixFQUFYO0FBQ0FoRCxJQUFBQSxJQUFJLENBQUNpRCxJQUFMLEdBQVkzRCxLQUFLLENBQUMsQ0FBRCxDQUFMLENBQVNjLEtBQVQsQ0FBZSxDQUFmLENBQVo7O0FBQ0EsUUFBSUosSUFBSSxDQUFDaUQsSUFBTCxLQUFjLEVBQWxCLEVBQXNCO0FBQ3BCLFdBQUtDLGFBQUwsQ0FBbUJsRCxJQUFuQixFQUF5QlYsS0FBekI7QUFDRDs7QUFDRCxTQUFLWSxJQUFMLENBQVVGLElBQVYsRUFBZ0JWLEtBQUssQ0FBQyxDQUFELENBQXJCLEVBQTBCQSxLQUFLLENBQUMsQ0FBRCxDQUEvQjtBQUVBLFFBQUk2RCxJQUFKO0FBQ0EsUUFBSXBCLEtBQUo7QUFDQSxRQUFJRixJQUFJLEdBQUcsS0FBWDtBQUNBLFFBQUl1QixJQUFJLEdBQUcsS0FBWDtBQUNBLFFBQUlDLE1BQU0sR0FBRyxFQUFiOztBQUVBLFdBQU8sQ0FBQyxLQUFLakUsU0FBTCxDQUFlRyxTQUFmLEVBQVIsRUFBb0M7QUFDbENELE1BQUFBLEtBQUssR0FBRyxLQUFLRixTQUFMLENBQWVJLFNBQWYsRUFBUjs7QUFFQSxVQUFJRixLQUFLLENBQUMsQ0FBRCxDQUFMLEtBQWEsR0FBakIsRUFBc0I7QUFDcEJVLFFBQUFBLElBQUksQ0FBQ2hCLE1BQUwsQ0FBWVUsR0FBWixHQUFrQjtBQUFFUixVQUFBQSxJQUFJLEVBQUVJLEtBQUssQ0FBQyxDQUFELENBQWI7QUFBa0JILFVBQUFBLE1BQU0sRUFBRUcsS0FBSyxDQUFDLENBQUQ7QUFBL0IsU0FBbEI7QUFDQSxhQUFLUixTQUFMLEdBQWlCLElBQWpCO0FBQ0E7QUFDRCxPQUpELE1BSU8sSUFBSVEsS0FBSyxDQUFDLENBQUQsQ0FBTCxLQUFhLEdBQWpCLEVBQXNCO0FBQzNCOEQsUUFBQUEsSUFBSSxHQUFHLElBQVA7QUFDQTtBQUNELE9BSE0sTUFHQSxJQUFJOUQsS0FBSyxDQUFDLENBQUQsQ0FBTCxLQUFhLEdBQWpCLEVBQXNCO0FBQzNCLFlBQUkrRCxNQUFNLENBQUNsQyxNQUFQLEdBQWdCLENBQXBCLEVBQXVCO0FBQ3JCWSxVQUFBQSxLQUFLLEdBQUdzQixNQUFNLENBQUNsQyxNQUFQLEdBQWdCLENBQXhCO0FBQ0FnQyxVQUFBQSxJQUFJLEdBQUdFLE1BQU0sQ0FBQ3RCLEtBQUQsQ0FBYjs7QUFDQSxpQkFBT29CLElBQUksSUFBSUEsSUFBSSxDQUFDLENBQUQsQ0FBSixLQUFZLE9BQTNCLEVBQW9DO0FBQ2xDQSxZQUFBQSxJQUFJLEdBQUdFLE1BQU0sQ0FBQyxFQUFFdEIsS0FBSCxDQUFiO0FBQ0Q7O0FBQ0QsY0FBSW9CLElBQUosRUFBVTtBQUNSbkQsWUFBQUEsSUFBSSxDQUFDaEIsTUFBTCxDQUFZVSxHQUFaLEdBQWtCO0FBQUVSLGNBQUFBLElBQUksRUFBRWlFLElBQUksQ0FBQyxDQUFELENBQVo7QUFBaUJoRSxjQUFBQSxNQUFNLEVBQUVnRSxJQUFJLENBQUMsQ0FBRDtBQUE3QixhQUFsQjtBQUNEO0FBQ0Y7O0FBQ0QsYUFBS3pELEdBQUwsQ0FBU0osS0FBVDtBQUNBO0FBQ0QsT0FiTSxNQWFBO0FBQ0wrRCxRQUFBQSxNQUFNLENBQUNuQyxJQUFQLENBQVk1QixLQUFaO0FBQ0Q7O0FBRUQsVUFBSSxLQUFLRixTQUFMLENBQWVHLFNBQWYsRUFBSixFQUFnQztBQUM5QnNDLFFBQUFBLElBQUksR0FBRyxJQUFQO0FBQ0E7QUFDRDtBQUNGOztBQUVEN0IsSUFBQUEsSUFBSSxDQUFDTSxJQUFMLENBQVVNLE9BQVYsR0FBb0IsS0FBS2Msd0JBQUwsQ0FBOEIyQixNQUE5QixDQUFwQjs7QUFDQSxRQUFJQSxNQUFNLENBQUNsQyxNQUFYLEVBQW1CO0FBQ2pCbkIsTUFBQUEsSUFBSSxDQUFDTSxJQUFMLENBQVVnRCxTQUFWLEdBQXNCLEtBQUtyQiwwQkFBTCxDQUFnQ29CLE1BQWhDLENBQXRCO0FBQ0EsV0FBSzFCLEdBQUwsQ0FBUzNCLElBQVQsRUFBZSxRQUFmLEVBQXlCcUQsTUFBekI7O0FBQ0EsVUFBSXhCLElBQUosRUFBVTtBQUNSdkMsUUFBQUEsS0FBSyxHQUFHK0QsTUFBTSxDQUFDQSxNQUFNLENBQUNsQyxNQUFQLEdBQWdCLENBQWpCLENBQWQ7QUFDQW5CLFFBQUFBLElBQUksQ0FBQ2hCLE1BQUwsQ0FBWVUsR0FBWixHQUFrQjtBQUFFUixVQUFBQSxJQUFJLEVBQUVJLEtBQUssQ0FBQyxDQUFELENBQWI7QUFBa0JILFVBQUFBLE1BQU0sRUFBRUcsS0FBSyxDQUFDLENBQUQ7QUFBL0IsU0FBbEI7QUFDQSxhQUFLVCxNQUFMLEdBQWNtQixJQUFJLENBQUNNLElBQUwsQ0FBVU0sT0FBeEI7QUFDQVosUUFBQUEsSUFBSSxDQUFDTSxJQUFMLENBQVVNLE9BQVYsR0FBb0IsRUFBcEI7QUFDRDtBQUNGLEtBVEQsTUFTTztBQUNMWixNQUFBQSxJQUFJLENBQUNNLElBQUwsQ0FBVWdELFNBQVYsR0FBc0IsRUFBdEI7QUFDQXRELE1BQUFBLElBQUksQ0FBQ3FELE1BQUwsR0FBYyxFQUFkO0FBQ0Q7O0FBRUQsUUFBSUQsSUFBSixFQUFVO0FBQ1JwRCxNQUFBQSxJQUFJLENBQUN1RCxLQUFMLEdBQWEsRUFBYjtBQUNBLFdBQUszRSxPQUFMLEdBQWVvQixJQUFmO0FBQ0Q7QUFDRixHOztTQUVETixHLGdCQUFLSixLLEVBQU87QUFDVixRQUFJLEtBQUtWLE9BQUwsQ0FBYTJFLEtBQWIsSUFBc0IsS0FBSzNFLE9BQUwsQ0FBYTJFLEtBQWIsQ0FBbUJwQyxNQUE3QyxFQUFxRDtBQUNuRCxXQUFLdkMsT0FBTCxDQUFhMEIsSUFBYixDQUFrQnhCLFNBQWxCLEdBQThCLEtBQUtBLFNBQW5DO0FBQ0Q7O0FBQ0QsU0FBS0EsU0FBTCxHQUFpQixLQUFqQjtBQUVBLFNBQUtGLE9BQUwsQ0FBYTBCLElBQWIsQ0FBa0JrRCxLQUFsQixHQUEwQixDQUFDLEtBQUs1RSxPQUFMLENBQWEwQixJQUFiLENBQWtCa0QsS0FBbEIsSUFBMkIsRUFBNUIsSUFBa0MsS0FBSzNFLE1BQWpFO0FBQ0EsU0FBS0EsTUFBTCxHQUFjLEVBQWQ7O0FBRUEsUUFBSSxLQUFLRCxPQUFMLENBQWE2RSxNQUFqQixFQUF5QjtBQUN2QixXQUFLN0UsT0FBTCxDQUFhSSxNQUFiLENBQW9CVSxHQUFwQixHQUEwQjtBQUFFUixRQUFBQSxJQUFJLEVBQUVJLEtBQUssQ0FBQyxDQUFELENBQWI7QUFBa0JILFFBQUFBLE1BQU0sRUFBRUcsS0FBSyxDQUFDLENBQUQ7QUFBL0IsT0FBMUI7QUFDQSxXQUFLVixPQUFMLEdBQWUsS0FBS0EsT0FBTCxDQUFhNkUsTUFBNUI7QUFDRCxLQUhELE1BR087QUFDTCxXQUFLQyxlQUFMLENBQXFCcEUsS0FBckI7QUFDRDtBQUNGLEc7O1NBRURTLE8sc0JBQVc7QUFDVCxRQUFJLEtBQUtuQixPQUFMLENBQWE2RSxNQUFqQixFQUF5QixLQUFLRSxhQUFMOztBQUN6QixRQUFJLEtBQUsvRSxPQUFMLENBQWEyRSxLQUFiLElBQXNCLEtBQUszRSxPQUFMLENBQWEyRSxLQUFiLENBQW1CcEMsTUFBN0MsRUFBcUQ7QUFDbkQsV0FBS3ZDLE9BQUwsQ0FBYTBCLElBQWIsQ0FBa0J4QixTQUFsQixHQUE4QixLQUFLQSxTQUFuQztBQUNEOztBQUNELFNBQUtGLE9BQUwsQ0FBYTBCLElBQWIsQ0FBa0JrRCxLQUFsQixHQUEwQixDQUFDLEtBQUs1RSxPQUFMLENBQWEwQixJQUFiLENBQWtCa0QsS0FBbEIsSUFBMkIsRUFBNUIsSUFBa0MsS0FBSzNFLE1BQWpFO0FBQ0QsRzs7U0FFRFksYSwwQkFBZUgsSyxFQUFPO0FBQ3BCLFNBQUtULE1BQUwsSUFBZVMsS0FBSyxDQUFDLENBQUQsQ0FBcEI7O0FBQ0EsUUFBSSxLQUFLVixPQUFMLENBQWEyRSxLQUFqQixFQUF3QjtBQUN0QixVQUFJSixJQUFJLEdBQUcsS0FBS3ZFLE9BQUwsQ0FBYTJFLEtBQWIsQ0FBbUIsS0FBSzNFLE9BQUwsQ0FBYTJFLEtBQWIsQ0FBbUJwQyxNQUFuQixHQUE0QixDQUEvQyxDQUFYOztBQUNBLFVBQUlnQyxJQUFJLElBQUlBLElBQUksQ0FBQ3RDLElBQUwsS0FBYyxNQUF0QixJQUFnQyxDQUFDc0MsSUFBSSxDQUFDN0MsSUFBTCxDQUFVc0QsWUFBL0MsRUFBNkQ7QUFDM0RULFFBQUFBLElBQUksQ0FBQzdDLElBQUwsQ0FBVXNELFlBQVYsR0FBeUIsS0FBSy9FLE1BQTlCO0FBQ0EsYUFBS0EsTUFBTCxHQUFjLEVBQWQ7QUFDRDtBQUNGO0FBQ0YsRyxFQUVEOzs7U0FFQXFCLEksaUJBQU1GLEksRUFBTWQsSSxFQUFNQyxNLEVBQVE7QUFDeEIsU0FBS1AsT0FBTCxDQUFhc0MsSUFBYixDQUFrQmxCLElBQWxCO0FBRUFBLElBQUFBLElBQUksQ0FBQ2hCLE1BQUwsR0FBYztBQUFFQyxNQUFBQSxLQUFLLEVBQUU7QUFBRUMsUUFBQUEsSUFBSSxFQUFKQSxJQUFGO0FBQVFDLFFBQUFBLE1BQU0sRUFBTkE7QUFBUixPQUFUO0FBQTJCVixNQUFBQSxLQUFLLEVBQUUsS0FBS0E7QUFBdkMsS0FBZDtBQUNBdUIsSUFBQUEsSUFBSSxDQUFDTSxJQUFMLENBQVV3QixNQUFWLEdBQW1CLEtBQUtqRCxNQUF4QjtBQUNBLFNBQUtBLE1BQUwsR0FBYyxFQUFkO0FBQ0EsUUFBSW1CLElBQUksQ0FBQ2EsSUFBTCxLQUFjLFNBQWxCLEVBQTZCLEtBQUsvQixTQUFMLEdBQWlCLEtBQWpCO0FBQzlCLEc7O1NBRUQ2QyxHLGdCQUFLM0IsSSxFQUFNZ0MsSSxFQUFNZixNLEVBQVE7QUFDdkIsUUFBSTNCLEtBQUosRUFBV3VCLElBQVg7QUFDQSxRQUFJTSxNQUFNLEdBQUdGLE1BQU0sQ0FBQ0UsTUFBcEI7QUFDQSxRQUFJMkIsS0FBSyxHQUFHLEVBQVo7QUFDQSxRQUFJZSxLQUFLLEdBQUcsSUFBWjtBQUNBLFFBQUlDLElBQUosRUFBVVgsSUFBVjtBQUNBLFFBQUlZLE9BQU8sR0FBRyxtQkFBZDs7QUFFQSxTQUFLLElBQUk1QixDQUFDLEdBQUcsQ0FBYixFQUFnQkEsQ0FBQyxHQUFHaEIsTUFBcEIsRUFBNEJnQixDQUFDLElBQUksQ0FBakMsRUFBb0M7QUFDbEM3QyxNQUFBQSxLQUFLLEdBQUcyQixNQUFNLENBQUNrQixDQUFELENBQWQ7QUFDQXRCLE1BQUFBLElBQUksR0FBR3ZCLEtBQUssQ0FBQyxDQUFELENBQVo7O0FBRUEsVUFBSXVCLElBQUksS0FBSyxTQUFULElBQXNCYixJQUFJLENBQUNhLElBQUwsS0FBYyxNQUF4QyxFQUFnRDtBQUM5Q3NDLFFBQUFBLElBQUksR0FBR2xDLE1BQU0sQ0FBQ2tCLENBQUMsR0FBRyxDQUFMLENBQWI7QUFDQTJCLFFBQUFBLElBQUksR0FBRzdDLE1BQU0sQ0FBQ2tCLENBQUMsR0FBRyxDQUFMLENBQWI7O0FBRUEsWUFDRWdCLElBQUksQ0FBQyxDQUFELENBQUosS0FBWSxPQUFaLElBQ0FXLElBQUksQ0FBQyxDQUFELENBQUosS0FBWSxPQURaLElBRUFDLE9BQU8sQ0FBQzFELElBQVIsQ0FBYThDLElBQUksQ0FBQyxDQUFELENBQWpCLENBRkEsSUFHQVksT0FBTyxDQUFDMUQsSUFBUixDQUFheUQsSUFBSSxDQUFDLENBQUQsQ0FBakIsQ0FKRixFQUtFO0FBQ0FoQixVQUFBQSxLQUFLLElBQUl4RCxLQUFLLENBQUMsQ0FBRCxDQUFkO0FBQ0QsU0FQRCxNQU9PO0FBQ0x1RSxVQUFBQSxLQUFLLEdBQUcsS0FBUjtBQUNEOztBQUVEO0FBQ0Q7O0FBRUQsVUFBSWhELElBQUksS0FBSyxTQUFULElBQXVCQSxJQUFJLEtBQUssT0FBVCxJQUFvQnNCLENBQUMsS0FBS2hCLE1BQU0sR0FBRyxDQUE5RCxFQUFrRTtBQUNoRTBDLFFBQUFBLEtBQUssR0FBRyxLQUFSO0FBQ0QsT0FGRCxNQUVPO0FBQ0xmLFFBQUFBLEtBQUssSUFBSXhELEtBQUssQ0FBQyxDQUFELENBQWQ7QUFDRDtBQUNGOztBQUNELFFBQUksQ0FBQ3VFLEtBQUwsRUFBWTtBQUNWLFVBQUlsQyxHQUFHLEdBQUdWLE1BQU0sQ0FBQytDLE1BQVAsQ0FBYyxVQUFDQyxHQUFELEVBQU05QixDQUFOO0FBQUEsZUFBWThCLEdBQUcsR0FBRzlCLENBQUMsQ0FBQyxDQUFELENBQW5CO0FBQUEsT0FBZCxFQUFzQyxFQUF0QyxDQUFWO0FBQ0FuQyxNQUFBQSxJQUFJLENBQUNNLElBQUwsQ0FBVTBCLElBQVYsSUFBa0I7QUFBRWMsUUFBQUEsS0FBSyxFQUFMQSxLQUFGO0FBQVNuQixRQUFBQSxHQUFHLEVBQUhBO0FBQVQsT0FBbEI7QUFDRDs7QUFDRDNCLElBQUFBLElBQUksQ0FBQ2dDLElBQUQsQ0FBSixHQUFhYyxLQUFiO0FBQ0QsRzs7U0FFRHBCLHdCLHFDQUEwQlQsTSxFQUFRO0FBQ2hDLFFBQUlpRCxhQUFKO0FBQ0EsUUFBSXJGLE1BQU0sR0FBRyxFQUFiOztBQUNBLFdBQU9vQyxNQUFNLENBQUNFLE1BQWQsRUFBc0I7QUFDcEIrQyxNQUFBQSxhQUFhLEdBQUdqRCxNQUFNLENBQUNBLE1BQU0sQ0FBQ0UsTUFBUCxHQUFnQixDQUFqQixDQUFOLENBQTBCLENBQTFCLENBQWhCO0FBQ0EsVUFBSStDLGFBQWEsS0FBSyxPQUFsQixJQUE2QkEsYUFBYSxLQUFLLFNBQW5ELEVBQThEO0FBQzlEckYsTUFBQUEsTUFBTSxHQUFHb0MsTUFBTSxDQUFDTSxHQUFQLEdBQWEsQ0FBYixJQUFrQjFDLE1BQTNCO0FBQ0Q7O0FBQ0QsV0FBT0EsTUFBUDtBQUNELEc7O1NBRURvRCwwQix1Q0FBNEJoQixNLEVBQVE7QUFDbEMsUUFBSTZDLElBQUo7QUFDQSxRQUFJakYsTUFBTSxHQUFHLEVBQWI7O0FBQ0EsV0FBT29DLE1BQU0sQ0FBQ0UsTUFBZCxFQUFzQjtBQUNwQjJDLE1BQUFBLElBQUksR0FBRzdDLE1BQU0sQ0FBQyxDQUFELENBQU4sQ0FBVSxDQUFWLENBQVA7QUFDQSxVQUFJNkMsSUFBSSxLQUFLLE9BQVQsSUFBb0JBLElBQUksS0FBSyxTQUFqQyxFQUE0QztBQUM1Q2pGLE1BQUFBLE1BQU0sSUFBSW9DLE1BQU0sQ0FBQ2MsS0FBUCxHQUFlLENBQWYsQ0FBVjtBQUNEOztBQUNELFdBQU9sRCxNQUFQO0FBQ0QsRzs7U0FFRDJELGEsMEJBQWV2QixNLEVBQVE7QUFDckIsUUFBSWlELGFBQUo7QUFDQSxRQUFJckYsTUFBTSxHQUFHLEVBQWI7O0FBQ0EsV0FBT29DLE1BQU0sQ0FBQ0UsTUFBZCxFQUFzQjtBQUNwQitDLE1BQUFBLGFBQWEsR0FBR2pELE1BQU0sQ0FBQ0EsTUFBTSxDQUFDRSxNQUFQLEdBQWdCLENBQWpCLENBQU4sQ0FBMEIsQ0FBMUIsQ0FBaEI7QUFDQSxVQUFJK0MsYUFBYSxLQUFLLE9BQXRCLEVBQStCO0FBQy9CckYsTUFBQUEsTUFBTSxHQUFHb0MsTUFBTSxDQUFDTSxHQUFQLEdBQWEsQ0FBYixJQUFrQjFDLE1BQTNCO0FBQ0Q7O0FBQ0QsV0FBT0EsTUFBUDtBQUNELEc7O1NBRUQwRCxVLHVCQUFZdEIsTSxFQUFRa0QsSSxFQUFNO0FBQ3hCLFFBQUlDLE1BQU0sR0FBRyxFQUFiOztBQUNBLFNBQUssSUFBSWpDLENBQUMsR0FBR2dDLElBQWIsRUFBbUJoQyxDQUFDLEdBQUdsQixNQUFNLENBQUNFLE1BQTlCLEVBQXNDZ0IsQ0FBQyxFQUF2QyxFQUEyQztBQUN6Q2lDLE1BQUFBLE1BQU0sSUFBSW5ELE1BQU0sQ0FBQ2tCLENBQUQsQ0FBTixDQUFVLENBQVYsQ0FBVjtBQUNEOztBQUNEbEIsSUFBQUEsTUFBTSxDQUFDb0QsTUFBUCxDQUFjRixJQUFkLEVBQW9CbEQsTUFBTSxDQUFDRSxNQUFQLEdBQWdCZ0QsSUFBcEM7QUFDQSxXQUFPQyxNQUFQO0FBQ0QsRzs7U0FFRHRELEssa0JBQU9HLE0sRUFBUTtBQUNiLFFBQUlELFFBQVEsR0FBRyxDQUFmO0FBQ0EsUUFBSTFCLEtBQUosRUFBV3VCLElBQVgsRUFBaUJzQyxJQUFqQjs7QUFDQSxTQUFLLElBQUloQixDQUFDLEdBQUcsQ0FBYixFQUFnQkEsQ0FBQyxHQUFHbEIsTUFBTSxDQUFDRSxNQUEzQixFQUFtQ2dCLENBQUMsRUFBcEMsRUFBd0M7QUFDdEM3QyxNQUFBQSxLQUFLLEdBQUcyQixNQUFNLENBQUNrQixDQUFELENBQWQ7QUFDQXRCLE1BQUFBLElBQUksR0FBR3ZCLEtBQUssQ0FBQyxDQUFELENBQVo7O0FBRUEsVUFBSXVCLElBQUksS0FBSyxHQUFiLEVBQWtCO0FBQ2hCRyxRQUFBQSxRQUFRLElBQUksQ0FBWjtBQUNEOztBQUNELFVBQUlILElBQUksS0FBSyxHQUFiLEVBQWtCO0FBQ2hCRyxRQUFBQSxRQUFRLElBQUksQ0FBWjtBQUNEOztBQUNELFVBQUlBLFFBQVEsS0FBSyxDQUFiLElBQWtCSCxJQUFJLEtBQUssR0FBL0IsRUFBb0M7QUFDbEMsWUFBSSxDQUFDc0MsSUFBTCxFQUFXO0FBQ1QsZUFBS21CLFdBQUwsQ0FBaUJoRixLQUFqQjtBQUNELFNBRkQsTUFFTyxJQUFJNkQsSUFBSSxDQUFDLENBQUQsQ0FBSixLQUFZLE1BQVosSUFBc0JBLElBQUksQ0FBQyxDQUFELENBQUosS0FBWSxRQUF0QyxFQUFnRDtBQUNyRDtBQUNELFNBRk0sTUFFQTtBQUNMLGlCQUFPaEIsQ0FBUDtBQUNEO0FBQ0Y7O0FBRURnQixNQUFBQSxJQUFJLEdBQUc3RCxLQUFQO0FBQ0Q7O0FBQ0QsV0FBTyxLQUFQO0FBQ0QsRyxFQUVEOzs7U0FFQWtDLGUsNEJBQWlCVCxPLEVBQVM7QUFDeEIsVUFBTSxLQUFLdEMsS0FBTCxDQUFXOEYsS0FBWCxDQUFpQixrQkFBakIsRUFBcUN4RCxPQUFPLENBQUMsQ0FBRCxDQUE1QyxFQUFpREEsT0FBTyxDQUFDLENBQUQsQ0FBeEQsQ0FBTjtBQUNELEc7O1NBRURVLFcsd0JBQWFSLE0sRUFBUTtBQUNuQixVQUFNLEtBQUt4QyxLQUFMLENBQVc4RixLQUFYLENBQWlCLGNBQWpCLEVBQWlDdEQsTUFBTSxDQUFDLENBQUQsQ0FBTixDQUFVLENBQVYsQ0FBakMsRUFBK0NBLE1BQU0sQ0FBQyxDQUFELENBQU4sQ0FBVSxDQUFWLENBQS9DLENBQU47QUFDRCxHOztTQUVEeUMsZSw0QkFBaUJwRSxLLEVBQU87QUFDdEIsVUFBTSxLQUFLYixLQUFMLENBQVc4RixLQUFYLENBQWlCLGNBQWpCLEVBQWlDakYsS0FBSyxDQUFDLENBQUQsQ0FBdEMsRUFBMkNBLEtBQUssQ0FBQyxDQUFELENBQWhELENBQU47QUFDRCxHOztTQUVEcUUsYSw0QkFBaUI7QUFDZixRQUFJYSxHQUFHLEdBQUcsS0FBSzVGLE9BQUwsQ0FBYUksTUFBYixDQUFvQkMsS0FBOUI7QUFDQSxVQUFNLEtBQUtSLEtBQUwsQ0FBVzhGLEtBQVgsQ0FBaUIsZ0JBQWpCLEVBQW1DQyxHQUFHLENBQUN0RixJQUF2QyxFQUE2Q3NGLEdBQUcsQ0FBQ3JGLE1BQWpELENBQU47QUFDRCxHOztTQUVEbUYsVyx3QkFBYWhGLEssRUFBTztBQUNsQixVQUFNLEtBQUtiLEtBQUwsQ0FBVzhGLEtBQVgsQ0FBaUIsY0FBakIsRUFBaUNqRixLQUFLLENBQUMsQ0FBRCxDQUF0QyxFQUEyQ0EsS0FBSyxDQUFDLENBQUQsQ0FBaEQsQ0FBTjtBQUNELEc7O1NBRUQ0RCxhLDBCQUFlbEQsSSxFQUFNVixLLEVBQU87QUFDMUIsVUFBTSxLQUFLYixLQUFMLENBQVc4RixLQUFYLENBQWlCLHNCQUFqQixFQUF5Q2pGLEtBQUssQ0FBQyxDQUFELENBQTlDLEVBQW1EQSxLQUFLLENBQUMsQ0FBRCxDQUF4RCxDQUFOO0FBQ0QsRzs7U0FFRDRDLHVCO0FBQXlCO0FBQWMsR0FDckM7QUFDRCxHOztTQUVEYSxvQixpQ0FBc0I5QixNLEVBQVE7QUFDNUIsUUFBSUgsS0FBSyxHQUFHLEtBQUtBLEtBQUwsQ0FBV0csTUFBWCxDQUFaO0FBQ0EsUUFBSUgsS0FBSyxLQUFLLEtBQWQsRUFBcUI7QUFFckIsUUFBSTJELE9BQU8sR0FBRyxDQUFkO0FBQ0EsUUFBSW5GLEtBQUo7O0FBQ0EsU0FBSyxJQUFJcUQsQ0FBQyxHQUFHN0IsS0FBSyxHQUFHLENBQXJCLEVBQXdCNkIsQ0FBQyxJQUFJLENBQTdCLEVBQWdDQSxDQUFDLEVBQWpDLEVBQXFDO0FBQ25DckQsTUFBQUEsS0FBSyxHQUFHMkIsTUFBTSxDQUFDMEIsQ0FBRCxDQUFkOztBQUNBLFVBQUlyRCxLQUFLLENBQUMsQ0FBRCxDQUFMLEtBQWEsT0FBakIsRUFBMEI7QUFDeEJtRixRQUFBQSxPQUFPLElBQUksQ0FBWDtBQUNBLFlBQUlBLE9BQU8sS0FBSyxDQUFoQixFQUFtQjtBQUNwQjtBQUNGOztBQUNELFVBQU0sS0FBS2hHLEtBQUwsQ0FBVzhGLEtBQVgsQ0FBaUIsa0JBQWpCLEVBQXFDakYsS0FBSyxDQUFDLENBQUQsQ0FBMUMsRUFBK0NBLEtBQUssQ0FBQyxDQUFELENBQXBELENBQU47QUFDRCxHIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IERlY2xhcmF0aW9uIGZyb20gJy4vZGVjbGFyYXRpb24nXG5pbXBvcnQgdG9rZW5pemVyIGZyb20gJy4vdG9rZW5pemUnXG5pbXBvcnQgQ29tbWVudCBmcm9tICcuL2NvbW1lbnQnXG5pbXBvcnQgQXRSdWxlIGZyb20gJy4vYXQtcnVsZSdcbmltcG9ydCBSb290IGZyb20gJy4vcm9vdCdcbmltcG9ydCBSdWxlIGZyb20gJy4vcnVsZSdcblxuZXhwb3J0IGRlZmF1bHQgY2xhc3MgUGFyc2VyIHtcbiAgY29uc3RydWN0b3IgKGlucHV0KSB7XG4gICAgdGhpcy5pbnB1dCA9IGlucHV0XG5cbiAgICB0aGlzLnJvb3QgPSBuZXcgUm9vdCgpXG4gICAgdGhpcy5jdXJyZW50ID0gdGhpcy5yb290XG4gICAgdGhpcy5zcGFjZXMgPSAnJ1xuICAgIHRoaXMuc2VtaWNvbG9uID0gZmFsc2VcblxuICAgIHRoaXMuY3JlYXRlVG9rZW5pemVyKClcbiAgICB0aGlzLnJvb3Quc291cmNlID0geyBpbnB1dCwgc3RhcnQ6IHsgbGluZTogMSwgY29sdW1uOiAxIH0gfVxuICB9XG5cbiAgY3JlYXRlVG9rZW5pemVyICgpIHtcbiAgICB0aGlzLnRva2VuaXplciA9IHRva2VuaXplcih0aGlzLmlucHV0KVxuICB9XG5cbiAgcGFyc2UgKCkge1xuICAgIGxldCB0b2tlblxuICAgIHdoaWxlICghdGhpcy50b2tlbml6ZXIuZW5kT2ZGaWxlKCkpIHtcbiAgICAgIHRva2VuID0gdGhpcy50b2tlbml6ZXIubmV4dFRva2VuKClcblxuICAgICAgc3dpdGNoICh0b2tlblswXSkge1xuICAgICAgICBjYXNlICdzcGFjZSc6XG4gICAgICAgICAgdGhpcy5zcGFjZXMgKz0gdG9rZW5bMV1cbiAgICAgICAgICBicmVha1xuXG4gICAgICAgIGNhc2UgJzsnOlxuICAgICAgICAgIHRoaXMuZnJlZVNlbWljb2xvbih0b2tlbilcbiAgICAgICAgICBicmVha1xuXG4gICAgICAgIGNhc2UgJ30nOlxuICAgICAgICAgIHRoaXMuZW5kKHRva2VuKVxuICAgICAgICAgIGJyZWFrXG5cbiAgICAgICAgY2FzZSAnY29tbWVudCc6XG4gICAgICAgICAgdGhpcy5jb21tZW50KHRva2VuKVxuICAgICAgICAgIGJyZWFrXG5cbiAgICAgICAgY2FzZSAnYXQtd29yZCc6XG4gICAgICAgICAgdGhpcy5hdHJ1bGUodG9rZW4pXG4gICAgICAgICAgYnJlYWtcblxuICAgICAgICBjYXNlICd7JzpcbiAgICAgICAgICB0aGlzLmVtcHR5UnVsZSh0b2tlbilcbiAgICAgICAgICBicmVha1xuXG4gICAgICAgIGRlZmF1bHQ6XG4gICAgICAgICAgdGhpcy5vdGhlcih0b2tlbilcbiAgICAgICAgICBicmVha1xuICAgICAgfVxuICAgIH1cbiAgICB0aGlzLmVuZEZpbGUoKVxuICB9XG5cbiAgY29tbWVudCAodG9rZW4pIHtcbiAgICBsZXQgbm9kZSA9IG5ldyBDb21tZW50KClcbiAgICB0aGlzLmluaXQobm9kZSwgdG9rZW5bMl0sIHRva2VuWzNdKVxuICAgIG5vZGUuc291cmNlLmVuZCA9IHsgbGluZTogdG9rZW5bNF0sIGNvbHVtbjogdG9rZW5bNV0gfVxuXG4gICAgbGV0IHRleHQgPSB0b2tlblsxXS5zbGljZSgyLCAtMilcbiAgICBpZiAoL15cXHMqJC8udGVzdCh0ZXh0KSkge1xuICAgICAgbm9kZS50ZXh0ID0gJydcbiAgICAgIG5vZGUucmF3cy5sZWZ0ID0gdGV4dFxuICAgICAgbm9kZS5yYXdzLnJpZ2h0ID0gJydcbiAgICB9IGVsc2Uge1xuICAgICAgbGV0IG1hdGNoID0gdGV4dC5tYXRjaCgvXihcXHMqKShbXl0qW15cXHNdKShcXHMqKSQvKVxuICAgICAgbm9kZS50ZXh0ID0gbWF0Y2hbMl1cbiAgICAgIG5vZGUucmF3cy5sZWZ0ID0gbWF0Y2hbMV1cbiAgICAgIG5vZGUucmF3cy5yaWdodCA9IG1hdGNoWzNdXG4gICAgfVxuICB9XG5cbiAgZW1wdHlSdWxlICh0b2tlbikge1xuICAgIGxldCBub2RlID0gbmV3IFJ1bGUoKVxuICAgIHRoaXMuaW5pdChub2RlLCB0b2tlblsyXSwgdG9rZW5bM10pXG4gICAgbm9kZS5zZWxlY3RvciA9ICcnXG4gICAgbm9kZS5yYXdzLmJldHdlZW4gPSAnJ1xuICAgIHRoaXMuY3VycmVudCA9IG5vZGVcbiAgfVxuXG4gIG90aGVyIChzdGFydCkge1xuICAgIGxldCBlbmQgPSBmYWxzZVxuICAgIGxldCB0eXBlID0gbnVsbFxuICAgIGxldCBjb2xvbiA9IGZhbHNlXG4gICAgbGV0IGJyYWNrZXQgPSBudWxsXG4gICAgbGV0IGJyYWNrZXRzID0gW11cblxuICAgIGxldCB0b2tlbnMgPSBbXVxuICAgIGxldCB0b2tlbiA9IHN0YXJ0XG4gICAgd2hpbGUgKHRva2VuKSB7XG4gICAgICB0eXBlID0gdG9rZW5bMF1cbiAgICAgIHRva2Vucy5wdXNoKHRva2VuKVxuXG4gICAgICBpZiAodHlwZSA9PT0gJygnIHx8IHR5cGUgPT09ICdbJykge1xuICAgICAgICBpZiAoIWJyYWNrZXQpIGJyYWNrZXQgPSB0b2tlblxuICAgICAgICBicmFja2V0cy5wdXNoKHR5cGUgPT09ICcoJyA/ICcpJyA6ICddJylcbiAgICAgIH0gZWxzZSBpZiAoYnJhY2tldHMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgIGlmICh0eXBlID09PSAnOycpIHtcbiAgICAgICAgICBpZiAoY29sb24pIHtcbiAgICAgICAgICAgIHRoaXMuZGVjbCh0b2tlbnMpXG4gICAgICAgICAgICByZXR1cm5cbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgYnJlYWtcbiAgICAgICAgICB9XG4gICAgICAgIH0gZWxzZSBpZiAodHlwZSA9PT0gJ3snKSB7XG4gICAgICAgICAgdGhpcy5ydWxlKHRva2VucylcbiAgICAgICAgICByZXR1cm5cbiAgICAgICAgfSBlbHNlIGlmICh0eXBlID09PSAnfScpIHtcbiAgICAgICAgICB0aGlzLnRva2VuaXplci5iYWNrKHRva2Vucy5wb3AoKSlcbiAgICAgICAgICBlbmQgPSB0cnVlXG4gICAgICAgICAgYnJlYWtcbiAgICAgICAgfSBlbHNlIGlmICh0eXBlID09PSAnOicpIHtcbiAgICAgICAgICBjb2xvbiA9IHRydWVcbiAgICAgICAgfVxuICAgICAgfSBlbHNlIGlmICh0eXBlID09PSBicmFja2V0c1ticmFja2V0cy5sZW5ndGggLSAxXSkge1xuICAgICAgICBicmFja2V0cy5wb3AoKVxuICAgICAgICBpZiAoYnJhY2tldHMubGVuZ3RoID09PSAwKSBicmFja2V0ID0gbnVsbFxuICAgICAgfVxuXG4gICAgICB0b2tlbiA9IHRoaXMudG9rZW5pemVyLm5leHRUb2tlbigpXG4gICAgfVxuXG4gICAgaWYgKHRoaXMudG9rZW5pemVyLmVuZE9mRmlsZSgpKSBlbmQgPSB0cnVlXG4gICAgaWYgKGJyYWNrZXRzLmxlbmd0aCA+IDApIHRoaXMudW5jbG9zZWRCcmFja2V0KGJyYWNrZXQpXG5cbiAgICBpZiAoZW5kICYmIGNvbG9uKSB7XG4gICAgICB3aGlsZSAodG9rZW5zLmxlbmd0aCkge1xuICAgICAgICB0b2tlbiA9IHRva2Vuc1t0b2tlbnMubGVuZ3RoIC0gMV1bMF1cbiAgICAgICAgaWYgKHRva2VuICE9PSAnc3BhY2UnICYmIHRva2VuICE9PSAnY29tbWVudCcpIGJyZWFrXG4gICAgICAgIHRoaXMudG9rZW5pemVyLmJhY2sodG9rZW5zLnBvcCgpKVxuICAgICAgfVxuICAgICAgdGhpcy5kZWNsKHRva2VucylcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy51bmtub3duV29yZCh0b2tlbnMpXG4gICAgfVxuICB9XG5cbiAgcnVsZSAodG9rZW5zKSB7XG4gICAgdG9rZW5zLnBvcCgpXG5cbiAgICBsZXQgbm9kZSA9IG5ldyBSdWxlKClcbiAgICB0aGlzLmluaXQobm9kZSwgdG9rZW5zWzBdWzJdLCB0b2tlbnNbMF1bM10pXG5cbiAgICBub2RlLnJhd3MuYmV0d2VlbiA9IHRoaXMuc3BhY2VzQW5kQ29tbWVudHNGcm9tRW5kKHRva2VucylcbiAgICB0aGlzLnJhdyhub2RlLCAnc2VsZWN0b3InLCB0b2tlbnMpXG4gICAgdGhpcy5jdXJyZW50ID0gbm9kZVxuICB9XG5cbiAgZGVjbCAodG9rZW5zKSB7XG4gICAgbGV0IG5vZGUgPSBuZXcgRGVjbGFyYXRpb24oKVxuICAgIHRoaXMuaW5pdChub2RlKVxuXG4gICAgbGV0IGxhc3QgPSB0b2tlbnNbdG9rZW5zLmxlbmd0aCAtIDFdXG4gICAgaWYgKGxhc3RbMF0gPT09ICc7Jykge1xuICAgICAgdGhpcy5zZW1pY29sb24gPSB0cnVlXG4gICAgICB0b2tlbnMucG9wKClcbiAgICB9XG4gICAgaWYgKGxhc3RbNF0pIHtcbiAgICAgIG5vZGUuc291cmNlLmVuZCA9IHsgbGluZTogbGFzdFs0XSwgY29sdW1uOiBsYXN0WzVdIH1cbiAgICB9IGVsc2Uge1xuICAgICAgbm9kZS5zb3VyY2UuZW5kID0geyBsaW5lOiBsYXN0WzJdLCBjb2x1bW46IGxhc3RbM10gfVxuICAgIH1cblxuICAgIHdoaWxlICh0b2tlbnNbMF1bMF0gIT09ICd3b3JkJykge1xuICAgICAgaWYgKHRva2Vucy5sZW5ndGggPT09IDEpIHRoaXMudW5rbm93bldvcmQodG9rZW5zKVxuICAgICAgbm9kZS5yYXdzLmJlZm9yZSArPSB0b2tlbnMuc2hpZnQoKVsxXVxuICAgIH1cbiAgICBub2RlLnNvdXJjZS5zdGFydCA9IHsgbGluZTogdG9rZW5zWzBdWzJdLCBjb2x1bW46IHRva2Vuc1swXVszXSB9XG5cbiAgICBub2RlLnByb3AgPSAnJ1xuICAgIHdoaWxlICh0b2tlbnMubGVuZ3RoKSB7XG4gICAgICBsZXQgdHlwZSA9IHRva2Vuc1swXVswXVxuICAgICAgaWYgKHR5cGUgPT09ICc6JyB8fCB0eXBlID09PSAnc3BhY2UnIHx8IHR5cGUgPT09ICdjb21tZW50Jykge1xuICAgICAgICBicmVha1xuICAgICAgfVxuICAgICAgbm9kZS5wcm9wICs9IHRva2Vucy5zaGlmdCgpWzFdXG4gICAgfVxuXG4gICAgbm9kZS5yYXdzLmJldHdlZW4gPSAnJ1xuXG4gICAgbGV0IHRva2VuXG4gICAgd2hpbGUgKHRva2Vucy5sZW5ndGgpIHtcbiAgICAgIHRva2VuID0gdG9rZW5zLnNoaWZ0KClcblxuICAgICAgaWYgKHRva2VuWzBdID09PSAnOicpIHtcbiAgICAgICAgbm9kZS5yYXdzLmJldHdlZW4gKz0gdG9rZW5bMV1cbiAgICAgICAgYnJlYWtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIG5vZGUucmF3cy5iZXR3ZWVuICs9IHRva2VuWzFdXG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKG5vZGUucHJvcFswXSA9PT0gJ18nIHx8IG5vZGUucHJvcFswXSA9PT0gJyonKSB7XG4gICAgICBub2RlLnJhd3MuYmVmb3JlICs9IG5vZGUucHJvcFswXVxuICAgICAgbm9kZS5wcm9wID0gbm9kZS5wcm9wLnNsaWNlKDEpXG4gICAgfVxuICAgIG5vZGUucmF3cy5iZXR3ZWVuICs9IHRoaXMuc3BhY2VzQW5kQ29tbWVudHNGcm9tU3RhcnQodG9rZW5zKVxuICAgIHRoaXMucHJlY2hlY2tNaXNzZWRTZW1pY29sb24odG9rZW5zKVxuXG4gICAgZm9yIChsZXQgaSA9IHRva2Vucy5sZW5ndGggLSAxOyBpID4gMDsgaS0tKSB7XG4gICAgICB0b2tlbiA9IHRva2Vuc1tpXVxuICAgICAgaWYgKHRva2VuWzFdLnRvTG93ZXJDYXNlKCkgPT09ICchaW1wb3J0YW50Jykge1xuICAgICAgICBub2RlLmltcG9ydGFudCA9IHRydWVcbiAgICAgICAgbGV0IHN0cmluZyA9IHRoaXMuc3RyaW5nRnJvbSh0b2tlbnMsIGkpXG4gICAgICAgIHN0cmluZyA9IHRoaXMuc3BhY2VzRnJvbUVuZCh0b2tlbnMpICsgc3RyaW5nXG4gICAgICAgIGlmIChzdHJpbmcgIT09ICcgIWltcG9ydGFudCcpIG5vZGUucmF3cy5pbXBvcnRhbnQgPSBzdHJpbmdcbiAgICAgICAgYnJlYWtcbiAgICAgIH0gZWxzZSBpZiAodG9rZW5bMV0udG9Mb3dlckNhc2UoKSA9PT0gJ2ltcG9ydGFudCcpIHtcbiAgICAgICAgbGV0IGNhY2hlID0gdG9rZW5zLnNsaWNlKDApXG4gICAgICAgIGxldCBzdHIgPSAnJ1xuICAgICAgICBmb3IgKGxldCBqID0gaTsgaiA+IDA7IGotLSkge1xuICAgICAgICAgIGxldCB0eXBlID0gY2FjaGVbal1bMF1cbiAgICAgICAgICBpZiAoc3RyLnRyaW0oKS5pbmRleE9mKCchJykgPT09IDAgJiYgdHlwZSAhPT0gJ3NwYWNlJykge1xuICAgICAgICAgICAgYnJlYWtcbiAgICAgICAgICB9XG4gICAgICAgICAgc3RyID0gY2FjaGUucG9wKClbMV0gKyBzdHJcbiAgICAgICAgfVxuICAgICAgICBpZiAoc3RyLnRyaW0oKS5pbmRleE9mKCchJykgPT09IDApIHtcbiAgICAgICAgICBub2RlLmltcG9ydGFudCA9IHRydWVcbiAgICAgICAgICBub2RlLnJhd3MuaW1wb3J0YW50ID0gc3RyXG4gICAgICAgICAgdG9rZW5zID0gY2FjaGVcbiAgICAgICAgfVxuICAgICAgfVxuXG4gICAgICBpZiAodG9rZW5bMF0gIT09ICdzcGFjZScgJiYgdG9rZW5bMF0gIT09ICdjb21tZW50Jykge1xuICAgICAgICBicmVha1xuICAgICAgfVxuICAgIH1cblxuICAgIHRoaXMucmF3KG5vZGUsICd2YWx1ZScsIHRva2VucylcblxuICAgIGlmIChub2RlLnZhbHVlLmluZGV4T2YoJzonKSAhPT0gLTEpIHRoaXMuY2hlY2tNaXNzZWRTZW1pY29sb24odG9rZW5zKVxuICB9XG5cbiAgYXRydWxlICh0b2tlbikge1xuICAgIGxldCBub2RlID0gbmV3IEF0UnVsZSgpXG4gICAgbm9kZS5uYW1lID0gdG9rZW5bMV0uc2xpY2UoMSlcbiAgICBpZiAobm9kZS5uYW1lID09PSAnJykge1xuICAgICAgdGhpcy51bm5hbWVkQXRydWxlKG5vZGUsIHRva2VuKVxuICAgIH1cbiAgICB0aGlzLmluaXQobm9kZSwgdG9rZW5bMl0sIHRva2VuWzNdKVxuXG4gICAgbGV0IHByZXZcbiAgICBsZXQgc2hpZnRcbiAgICBsZXQgbGFzdCA9IGZhbHNlXG4gICAgbGV0IG9wZW4gPSBmYWxzZVxuICAgIGxldCBwYXJhbXMgPSBbXVxuXG4gICAgd2hpbGUgKCF0aGlzLnRva2VuaXplci5lbmRPZkZpbGUoKSkge1xuICAgICAgdG9rZW4gPSB0aGlzLnRva2VuaXplci5uZXh0VG9rZW4oKVxuXG4gICAgICBpZiAodG9rZW5bMF0gPT09ICc7Jykge1xuICAgICAgICBub2RlLnNvdXJjZS5lbmQgPSB7IGxpbmU6IHRva2VuWzJdLCBjb2x1bW46IHRva2VuWzNdIH1cbiAgICAgICAgdGhpcy5zZW1pY29sb24gPSB0cnVlXG4gICAgICAgIGJyZWFrXG4gICAgICB9IGVsc2UgaWYgKHRva2VuWzBdID09PSAneycpIHtcbiAgICAgICAgb3BlbiA9IHRydWVcbiAgICAgICAgYnJlYWtcbiAgICAgIH0gZWxzZSBpZiAodG9rZW5bMF0gPT09ICd9Jykge1xuICAgICAgICBpZiAocGFyYW1zLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICBzaGlmdCA9IHBhcmFtcy5sZW5ndGggLSAxXG4gICAgICAgICAgcHJldiA9IHBhcmFtc1tzaGlmdF1cbiAgICAgICAgICB3aGlsZSAocHJldiAmJiBwcmV2WzBdID09PSAnc3BhY2UnKSB7XG4gICAgICAgICAgICBwcmV2ID0gcGFyYW1zWy0tc2hpZnRdXG4gICAgICAgICAgfVxuICAgICAgICAgIGlmIChwcmV2KSB7XG4gICAgICAgICAgICBub2RlLnNvdXJjZS5lbmQgPSB7IGxpbmU6IHByZXZbNF0sIGNvbHVtbjogcHJldls1XSB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIHRoaXMuZW5kKHRva2VuKVxuICAgICAgICBicmVha1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcGFyYW1zLnB1c2godG9rZW4pXG4gICAgICB9XG5cbiAgICAgIGlmICh0aGlzLnRva2VuaXplci5lbmRPZkZpbGUoKSkge1xuICAgICAgICBsYXN0ID0gdHJ1ZVxuICAgICAgICBicmVha1xuICAgICAgfVxuICAgIH1cblxuICAgIG5vZGUucmF3cy5iZXR3ZWVuID0gdGhpcy5zcGFjZXNBbmRDb21tZW50c0Zyb21FbmQocGFyYW1zKVxuICAgIGlmIChwYXJhbXMubGVuZ3RoKSB7XG4gICAgICBub2RlLnJhd3MuYWZ0ZXJOYW1lID0gdGhpcy5zcGFjZXNBbmRDb21tZW50c0Zyb21TdGFydChwYXJhbXMpXG4gICAgICB0aGlzLnJhdyhub2RlLCAncGFyYW1zJywgcGFyYW1zKVxuICAgICAgaWYgKGxhc3QpIHtcbiAgICAgICAgdG9rZW4gPSBwYXJhbXNbcGFyYW1zLmxlbmd0aCAtIDFdXG4gICAgICAgIG5vZGUuc291cmNlLmVuZCA9IHsgbGluZTogdG9rZW5bNF0sIGNvbHVtbjogdG9rZW5bNV0gfVxuICAgICAgICB0aGlzLnNwYWNlcyA9IG5vZGUucmF3cy5iZXR3ZWVuXG4gICAgICAgIG5vZGUucmF3cy5iZXR3ZWVuID0gJydcbiAgICAgIH1cbiAgICB9IGVsc2Uge1xuICAgICAgbm9kZS5yYXdzLmFmdGVyTmFtZSA9ICcnXG4gICAgICBub2RlLnBhcmFtcyA9ICcnXG4gICAgfVxuXG4gICAgaWYgKG9wZW4pIHtcbiAgICAgIG5vZGUubm9kZXMgPSBbXVxuICAgICAgdGhpcy5jdXJyZW50ID0gbm9kZVxuICAgIH1cbiAgfVxuXG4gIGVuZCAodG9rZW4pIHtcbiAgICBpZiAodGhpcy5jdXJyZW50Lm5vZGVzICYmIHRoaXMuY3VycmVudC5ub2Rlcy5sZW5ndGgpIHtcbiAgICAgIHRoaXMuY3VycmVudC5yYXdzLnNlbWljb2xvbiA9IHRoaXMuc2VtaWNvbG9uXG4gICAgfVxuICAgIHRoaXMuc2VtaWNvbG9uID0gZmFsc2VcblxuICAgIHRoaXMuY3VycmVudC5yYXdzLmFmdGVyID0gKHRoaXMuY3VycmVudC5yYXdzLmFmdGVyIHx8ICcnKSArIHRoaXMuc3BhY2VzXG4gICAgdGhpcy5zcGFjZXMgPSAnJ1xuXG4gICAgaWYgKHRoaXMuY3VycmVudC5wYXJlbnQpIHtcbiAgICAgIHRoaXMuY3VycmVudC5zb3VyY2UuZW5kID0geyBsaW5lOiB0b2tlblsyXSwgY29sdW1uOiB0b2tlblszXSB9XG4gICAgICB0aGlzLmN1cnJlbnQgPSB0aGlzLmN1cnJlbnQucGFyZW50XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMudW5leHBlY3RlZENsb3NlKHRva2VuKVxuICAgIH1cbiAgfVxuXG4gIGVuZEZpbGUgKCkge1xuICAgIGlmICh0aGlzLmN1cnJlbnQucGFyZW50KSB0aGlzLnVuY2xvc2VkQmxvY2soKVxuICAgIGlmICh0aGlzLmN1cnJlbnQubm9kZXMgJiYgdGhpcy5jdXJyZW50Lm5vZGVzLmxlbmd0aCkge1xuICAgICAgdGhpcy5jdXJyZW50LnJhd3Muc2VtaWNvbG9uID0gdGhpcy5zZW1pY29sb25cbiAgICB9XG4gICAgdGhpcy5jdXJyZW50LnJhd3MuYWZ0ZXIgPSAodGhpcy5jdXJyZW50LnJhd3MuYWZ0ZXIgfHwgJycpICsgdGhpcy5zcGFjZXNcbiAgfVxuXG4gIGZyZWVTZW1pY29sb24gKHRva2VuKSB7XG4gICAgdGhpcy5zcGFjZXMgKz0gdG9rZW5bMV1cbiAgICBpZiAodGhpcy5jdXJyZW50Lm5vZGVzKSB7XG4gICAgICBsZXQgcHJldiA9IHRoaXMuY3VycmVudC5ub2Rlc1t0aGlzLmN1cnJlbnQubm9kZXMubGVuZ3RoIC0gMV1cbiAgICAgIGlmIChwcmV2ICYmIHByZXYudHlwZSA9PT0gJ3J1bGUnICYmICFwcmV2LnJhd3Mub3duU2VtaWNvbG9uKSB7XG4gICAgICAgIHByZXYucmF3cy5vd25TZW1pY29sb24gPSB0aGlzLnNwYWNlc1xuICAgICAgICB0aGlzLnNwYWNlcyA9ICcnXG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLy8gSGVscGVyc1xuXG4gIGluaXQgKG5vZGUsIGxpbmUsIGNvbHVtbikge1xuICAgIHRoaXMuY3VycmVudC5wdXNoKG5vZGUpXG5cbiAgICBub2RlLnNvdXJjZSA9IHsgc3RhcnQ6IHsgbGluZSwgY29sdW1uIH0sIGlucHV0OiB0aGlzLmlucHV0IH1cbiAgICBub2RlLnJhd3MuYmVmb3JlID0gdGhpcy5zcGFjZXNcbiAgICB0aGlzLnNwYWNlcyA9ICcnXG4gICAgaWYgKG5vZGUudHlwZSAhPT0gJ2NvbW1lbnQnKSB0aGlzLnNlbWljb2xvbiA9IGZhbHNlXG4gIH1cblxuICByYXcgKG5vZGUsIHByb3AsIHRva2Vucykge1xuICAgIGxldCB0b2tlbiwgdHlwZVxuICAgIGxldCBsZW5ndGggPSB0b2tlbnMubGVuZ3RoXG4gICAgbGV0IHZhbHVlID0gJydcbiAgICBsZXQgY2xlYW4gPSB0cnVlXG4gICAgbGV0IG5leHQsIHByZXZcbiAgICBsZXQgcGF0dGVybiA9IC9eKFsufCNdKT8oW1xcd10pKy9pXG5cbiAgICBmb3IgKGxldCBpID0gMDsgaSA8IGxlbmd0aDsgaSArPSAxKSB7XG4gICAgICB0b2tlbiA9IHRva2Vuc1tpXVxuICAgICAgdHlwZSA9IHRva2VuWzBdXG5cbiAgICAgIGlmICh0eXBlID09PSAnY29tbWVudCcgJiYgbm9kZS50eXBlID09PSAncnVsZScpIHtcbiAgICAgICAgcHJldiA9IHRva2Vuc1tpIC0gMV1cbiAgICAgICAgbmV4dCA9IHRva2Vuc1tpICsgMV1cblxuICAgICAgICBpZiAoXG4gICAgICAgICAgcHJldlswXSAhPT0gJ3NwYWNlJyAmJlxuICAgICAgICAgIG5leHRbMF0gIT09ICdzcGFjZScgJiZcbiAgICAgICAgICBwYXR0ZXJuLnRlc3QocHJldlsxXSkgJiZcbiAgICAgICAgICBwYXR0ZXJuLnRlc3QobmV4dFsxXSlcbiAgICAgICAgKSB7XG4gICAgICAgICAgdmFsdWUgKz0gdG9rZW5bMV1cbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBjbGVhbiA9IGZhbHNlXG4gICAgICAgIH1cblxuICAgICAgICBjb250aW51ZVxuICAgICAgfVxuXG4gICAgICBpZiAodHlwZSA9PT0gJ2NvbW1lbnQnIHx8ICh0eXBlID09PSAnc3BhY2UnICYmIGkgPT09IGxlbmd0aCAtIDEpKSB7XG4gICAgICAgIGNsZWFuID0gZmFsc2VcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHZhbHVlICs9IHRva2VuWzFdXG4gICAgICB9XG4gICAgfVxuICAgIGlmICghY2xlYW4pIHtcbiAgICAgIGxldCByYXcgPSB0b2tlbnMucmVkdWNlKChhbGwsIGkpID0+IGFsbCArIGlbMV0sICcnKVxuICAgICAgbm9kZS5yYXdzW3Byb3BdID0geyB2YWx1ZSwgcmF3IH1cbiAgICB9XG4gICAgbm9kZVtwcm9wXSA9IHZhbHVlXG4gIH1cblxuICBzcGFjZXNBbmRDb21tZW50c0Zyb21FbmQgKHRva2Vucykge1xuICAgIGxldCBsYXN0VG9rZW5UeXBlXG4gICAgbGV0IHNwYWNlcyA9ICcnXG4gICAgd2hpbGUgKHRva2Vucy5sZW5ndGgpIHtcbiAgICAgIGxhc3RUb2tlblR5cGUgPSB0b2tlbnNbdG9rZW5zLmxlbmd0aCAtIDFdWzBdXG4gICAgICBpZiAobGFzdFRva2VuVHlwZSAhPT0gJ3NwYWNlJyAmJiBsYXN0VG9rZW5UeXBlICE9PSAnY29tbWVudCcpIGJyZWFrXG4gICAgICBzcGFjZXMgPSB0b2tlbnMucG9wKClbMV0gKyBzcGFjZXNcbiAgICB9XG4gICAgcmV0dXJuIHNwYWNlc1xuICB9XG5cbiAgc3BhY2VzQW5kQ29tbWVudHNGcm9tU3RhcnQgKHRva2Vucykge1xuICAgIGxldCBuZXh0XG4gICAgbGV0IHNwYWNlcyA9ICcnXG4gICAgd2hpbGUgKHRva2Vucy5sZW5ndGgpIHtcbiAgICAgIG5leHQgPSB0b2tlbnNbMF1bMF1cbiAgICAgIGlmIChuZXh0ICE9PSAnc3BhY2UnICYmIG5leHQgIT09ICdjb21tZW50JykgYnJlYWtcbiAgICAgIHNwYWNlcyArPSB0b2tlbnMuc2hpZnQoKVsxXVxuICAgIH1cbiAgICByZXR1cm4gc3BhY2VzXG4gIH1cblxuICBzcGFjZXNGcm9tRW5kICh0b2tlbnMpIHtcbiAgICBsZXQgbGFzdFRva2VuVHlwZVxuICAgIGxldCBzcGFjZXMgPSAnJ1xuICAgIHdoaWxlICh0b2tlbnMubGVuZ3RoKSB7XG4gICAgICBsYXN0VG9rZW5UeXBlID0gdG9rZW5zW3Rva2Vucy5sZW5ndGggLSAxXVswXVxuICAgICAgaWYgKGxhc3RUb2tlblR5cGUgIT09ICdzcGFjZScpIGJyZWFrXG4gICAgICBzcGFjZXMgPSB0b2tlbnMucG9wKClbMV0gKyBzcGFjZXNcbiAgICB9XG4gICAgcmV0dXJuIHNwYWNlc1xuICB9XG5cbiAgc3RyaW5nRnJvbSAodG9rZW5zLCBmcm9tKSB7XG4gICAgbGV0IHJlc3VsdCA9ICcnXG4gICAgZm9yIChsZXQgaSA9IGZyb207IGkgPCB0b2tlbnMubGVuZ3RoOyBpKyspIHtcbiAgICAgIHJlc3VsdCArPSB0b2tlbnNbaV1bMV1cbiAgICB9XG4gICAgdG9rZW5zLnNwbGljZShmcm9tLCB0b2tlbnMubGVuZ3RoIC0gZnJvbSlcbiAgICByZXR1cm4gcmVzdWx0XG4gIH1cblxuICBjb2xvbiAodG9rZW5zKSB7XG4gICAgbGV0IGJyYWNrZXRzID0gMFxuICAgIGxldCB0b2tlbiwgdHlwZSwgcHJldlxuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgdG9rZW5zLmxlbmd0aDsgaSsrKSB7XG4gICAgICB0b2tlbiA9IHRva2Vuc1tpXVxuICAgICAgdHlwZSA9IHRva2VuWzBdXG5cbiAgICAgIGlmICh0eXBlID09PSAnKCcpIHtcbiAgICAgICAgYnJhY2tldHMgKz0gMVxuICAgICAgfVxuICAgICAgaWYgKHR5cGUgPT09ICcpJykge1xuICAgICAgICBicmFja2V0cyAtPSAxXG4gICAgICB9XG4gICAgICBpZiAoYnJhY2tldHMgPT09IDAgJiYgdHlwZSA9PT0gJzonKSB7XG4gICAgICAgIGlmICghcHJldikge1xuICAgICAgICAgIHRoaXMuZG91YmxlQ29sb24odG9rZW4pXG4gICAgICAgIH0gZWxzZSBpZiAocHJldlswXSA9PT0gJ3dvcmQnICYmIHByZXZbMV0gPT09ICdwcm9naWQnKSB7XG4gICAgICAgICAgY29udGludWVcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICByZXR1cm4gaVxuICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIHByZXYgPSB0b2tlblxuICAgIH1cbiAgICByZXR1cm4gZmFsc2VcbiAgfVxuXG4gIC8vIEVycm9yc1xuXG4gIHVuY2xvc2VkQnJhY2tldCAoYnJhY2tldCkge1xuICAgIHRocm93IHRoaXMuaW5wdXQuZXJyb3IoJ1VuY2xvc2VkIGJyYWNrZXQnLCBicmFja2V0WzJdLCBicmFja2V0WzNdKVxuICB9XG5cbiAgdW5rbm93bldvcmQgKHRva2Vucykge1xuICAgIHRocm93IHRoaXMuaW5wdXQuZXJyb3IoJ1Vua25vd24gd29yZCcsIHRva2Vuc1swXVsyXSwgdG9rZW5zWzBdWzNdKVxuICB9XG5cbiAgdW5leHBlY3RlZENsb3NlICh0b2tlbikge1xuICAgIHRocm93IHRoaXMuaW5wdXQuZXJyb3IoJ1VuZXhwZWN0ZWQgfScsIHRva2VuWzJdLCB0b2tlblszXSlcbiAgfVxuXG4gIHVuY2xvc2VkQmxvY2sgKCkge1xuICAgIGxldCBwb3MgPSB0aGlzLmN1cnJlbnQuc291cmNlLnN0YXJ0XG4gICAgdGhyb3cgdGhpcy5pbnB1dC5lcnJvcignVW5jbG9zZWQgYmxvY2snLCBwb3MubGluZSwgcG9zLmNvbHVtbilcbiAgfVxuXG4gIGRvdWJsZUNvbG9uICh0b2tlbikge1xuICAgIHRocm93IHRoaXMuaW5wdXQuZXJyb3IoJ0RvdWJsZSBjb2xvbicsIHRva2VuWzJdLCB0b2tlblszXSlcbiAgfVxuXG4gIHVubmFtZWRBdHJ1bGUgKG5vZGUsIHRva2VuKSB7XG4gICAgdGhyb3cgdGhpcy5pbnB1dC5lcnJvcignQXQtcnVsZSB3aXRob3V0IG5hbWUnLCB0b2tlblsyXSwgdG9rZW5bM10pXG4gIH1cblxuICBwcmVjaGVja01pc3NlZFNlbWljb2xvbiAoLyogdG9rZW5zICovKSB7XG4gICAgLy8gSG9vayBmb3IgU2FmZSBQYXJzZXJcbiAgfVxuXG4gIGNoZWNrTWlzc2VkU2VtaWNvbG9uICh0b2tlbnMpIHtcbiAgICBsZXQgY29sb24gPSB0aGlzLmNvbG9uKHRva2VucylcbiAgICBpZiAoY29sb24gPT09IGZhbHNlKSByZXR1cm5cblxuICAgIGxldCBmb3VuZGVkID0gMFxuICAgIGxldCB0b2tlblxuICAgIGZvciAobGV0IGogPSBjb2xvbiAtIDE7IGogPj0gMDsgai0tKSB7XG4gICAgICB0b2tlbiA9IHRva2Vuc1tqXVxuICAgICAgaWYgKHRva2VuWzBdICE9PSAnc3BhY2UnKSB7XG4gICAgICAgIGZvdW5kZWQgKz0gMVxuICAgICAgICBpZiAoZm91bmRlZCA9PT0gMikgYnJlYWtcbiAgICAgIH1cbiAgICB9XG4gICAgdGhyb3cgdGhpcy5pbnB1dC5lcnJvcignTWlzc2VkIHNlbWljb2xvbicsIHRva2VuWzJdLCB0b2tlblszXSlcbiAgfVxufVxuIl0sImZpbGUiOiJwYXJzZXIuanMifQ==
