//
// DynamicAnyHolder.h
//
// $Id: //poco/1.4/Foundation/include/Poco/DynamicAnyHolder.h#2 $
//
// Library: Foundation
// Package: Core
// Module:  DynamicAnyHolder
//
// Definition of the DynamicAnyHolder class.
//
// Copyright (c) 2007, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Foundation_DynamicAnyHolder_INCLUDED
#define Foundation_DynamicAnyHolder_INCLUDED


#include "Poco/Foundation.h"
#include "Poco/NumberFormatter.h"
#include "Poco/NumberParser.h"
#include "Poco/DateTime.h"
#include "Poco/Timestamp.h"
#include "Poco/LocalDateTime.h"
#include "Poco/DateTimeFormat.h"
#include "Poco/DateTimeFormatter.h"
#include "Poco/DateTimeParser.h"
#include "Poco/String.h"
#include "Poco/Exception.h"
#include <vector>
#include <typeinfo>
#undef min
#undef max
#include <limits>


namespace Poco {


class DynamicAny;


class Foundation_API DynamicAnyHolder
	/// Interface for a data holder used by the DynamicAny class. 
	/// Provides methods to convert between data types.
	/// Only data types for which a convert method exists are supported, which are
	/// all C++ built-in types with addition of std::string, DateTime, LocalDateTime, Timestamp,
	/// and std::vector<DynamicAny>.
{
public:
	DynamicAnyHolder();
		/// Creates the DynamicAnyHolder.

	virtual ~DynamicAnyHolder();
		/// Destroys the DynamicAnyHolder.

	virtual DynamicAnyHolder* clone() const = 0;
		/// Deep-copies the DynamicAnyHolder.
		
	virtual const std::type_info& type() const = 0;
		/// Returns the type information of the stored content.

	virtual void convert(Int8& val) const = 0;
	virtual void convert(Int16& val) const = 0;
	virtual void convert(Int32& val) const = 0;
	virtual void convert(Int64& val) const = 0;
	virtual void convert(UInt8& val) const = 0;
	virtual void convert(UInt16& val) const = 0;
	virtual void convert(UInt32& val) const = 0;
	virtual void convert(UInt64& val) const = 0;
	virtual void convert(DateTime& val) const = 0;
	virtual void convert(LocalDateTime& val) const = 0;
	virtual void convert(Timestamp& val) const = 0;
	virtual bool isArray() const = 0;
	virtual bool isInteger() const = 0;
	virtual bool isSigned() const = 0;
	virtual bool isNumeric() const = 0;
	virtual bool isString() const = 0;

#ifndef POCO_LONG_IS_64_BIT
	void convert(long& val) const;
	void convert(unsigned long& val) const;
#endif

	virtual void convert(bool& val) const = 0;
	virtual void convert(float& val) const = 0;
	virtual void convert(double& val) const = 0;
	virtual void convert(char& val) const = 0;
	virtual void convert(std::string& val) const = 0;

protected:
	template <typename F, typename T>
	void convertToSmaller(const F& from, T& to) const
		/// This function is meant to convert signed numeric values from
		/// larger to smaller type. It checks the upper and lower bound and
		/// if from value is within limits of type T (i.e. check calls do not throw), 
		/// it is converted.
	{
		poco_static_assert (std::numeric_limits<F>::is_specialized);
		poco_static_assert (std::numeric_limits<T>::is_specialized);
		poco_static_assert (std::numeric_limits<F>::is_signed);
		poco_static_assert (std::numeric_limits<T>::is_signed);

		if (std::numeric_limits<F>::is_integer)
		{
			checkUpperLimit(from, to); 
			checkLowerLimit(from, to);
		}
		else
		{
			checkUpperLimitFloat(from, to); 
			checkLowerLimitFloat(from, to);
		}
		to = static_cast<T>(from);
	}

	template <typename F, typename T>
	void convertToSmallerUnsigned(const F& from, T& to) const
		/// This function is meant for converting unsigned integral data types,
		/// from larger to smaller type. Since lower limit is always 0 for unigned types, 
		/// only the upper limit is checked, thus saving some cycles compared to the signed 
		/// version of the function. If the value to be converted is smaller than
		/// the maximum value for the target type, the conversion is performed.
	{
		poco_static_assert (std::numeric_limits<F>::is_specialized);
		poco_static_assert (std::numeric_limits<T>::is_specialized);
		poco_static_assert (!std::numeric_limits<F>::is_signed);
		poco_static_assert (!std::numeric_limits<T>::is_signed);

		checkUpperLimit(from, to); 
		to = static_cast<T>(from);
	}

	template <typename F, typename T>
	void convertSignedToUnsigned(const F& from, T& to) const
		/// This function is meant for converting signed integral data types to
		/// unsigned data types. Negative values can not be converted and if one is 
		/// encountered, RangeException is thrown. 
		/// If upper limit is within the target data type limits, the conversion is performed.
	{
		poco_static_assert (std::numeric_limits<F>::is_specialized);
		poco_static_assert (std::numeric_limits<T>::is_specialized);
		poco_static_assert (std::numeric_limits<F>::is_signed);
		poco_static_assert (!std::numeric_limits<T>::is_signed);

		if (from < 0)
			throw RangeException("Value too small.");
		checkUpperLimit(from, to); 
		to = static_cast<T>(from);
	}

	template <typename F, typename T>
	void convertSignedFloatToUnsigned(const F& from, T& to) const
		/// This function is meant for converting floating point data types to
		/// unsigned integral data types. Negative values can not be converted and if one is 
		/// encountered, RangeException is thrown. 
		/// If uper limit is within the target data type limits, the conversion is performed.
	{
		poco_static_assert (std::numeric_limits<F>::is_specialized);
		poco_static_assert (std::numeric_limits<T>::is_specialized);
		poco_static_assert (!std::numeric_limits<F>::is_integer);
		poco_static_assert (std::numeric_limits<T>::is_integer);
		poco_static_assert (!std::numeric_limits<T>::is_signed);

		if (from < 0)
			throw RangeException("Value too small.");
		checkUpperLimitFloat(from, to); 
		to = static_cast<T>(from);
	}

	template <typename F, typename T>
	void convertUnsignedToSigned(const F& from, T& to) const
		/// This function is meant for converting unsigned integral data types to
		/// unsigned data types. Negative values can not be converted and if one is 
		/// encountered, RangeException is thrown. 
		/// If upper limit is within the target data type limits, the converiosn is performed.
	{
		poco_static_assert (std::numeric_limits<F>::is_specialized);
		poco_static_assert (std::numeric_limits<T>::is_specialized);
		poco_static_assert (!std::numeric_limits<F>::is_signed);
		poco_static_assert (std::numeric_limits<T>::is_signed);

		checkUpperLimit(from, to); 
		to = static_cast<T>(from);
	}

private:
	template <typename F, typename T>
	void checkUpperLimit(const F& from, T&) const
	{
		if ((sizeof(T) < sizeof(F)) &&
			(from > static_cast<F>(std::numeric_limits<T>::max())))
		{
			throw RangeException("Value too large.");
		}
		else
		if (static_cast<T>(from) > std::numeric_limits<T>::max()) 
		{
			throw RangeException("Value too large.");
		}
	}

	template <typename F, typename T>
	void checkUpperLimitFloat(const F& from, T&) const
	{
		if (from > std::numeric_limits<T>::max())
			throw RangeException("Value too large.");
	}

	template <typename F, typename T>
	void checkLowerLimitFloat(const F& from, T&) const
	{
		if (from < -std::numeric_limits<T>::max())
			throw RangeException("Value too small.");
	}

	template <typename F, typename T>
	void checkLowerLimit(const F& from, T&) const
	{
		if (from < std::numeric_limits<T>::min()) 
			throw RangeException("Value too small.");
	}
};


//
// inlines
//
#ifndef POCO_LONG_IS_64_BIT
inline void DynamicAnyHolder::convert(long& val) const
{
	Int32 tmp;
	convert(tmp);
	val = tmp;
}


inline void DynamicAnyHolder::convert(unsigned long& val) const
{
	UInt32 tmp;
	convert(tmp);
	val = tmp;
}
#endif


template <typename T>
class DynamicAnyHolderImpl: public DynamicAnyHolder
	/// Template based implementation of a DynamicAnyHolder. 
	/// Conversion work happens in the template specializations of this class.
	///
	/// DynamicAny can be used for any type for which a specialization for
	/// DynamicAnyHolderImpl is available.
	///
	/// DynamicAnyHolderImpl throws following exceptions:
	///		NotImplementedException (if the specialization for a type does not exist)
	///		RangeException (if an attempt is made to assign a numeric value outside of the target min/max limits
	///		SyntaxException (if an attempt is made to convert a string containing non-numeric characters to number)
	///
	/// All specializations must additionally implement a public member function:
	///     const T& value() const
	/// returning a const reference to the actual stored value.
{
public:
	DynamicAnyHolderImpl()
	{
	}

	~DynamicAnyHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(T);
	}

	bool isInteger() const
	{
		return std::numeric_limits<T>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<T>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<T>::is_specialized;
	}

	bool isString() const
	{
		return type() == typeid(std::string);
	}

	void convert(Int8&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(Int16&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(Int32&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(Int64&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}
	
	void convert(UInt8&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(UInt16&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(UInt32&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(UInt64&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(bool&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(float&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(double&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(char&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(std::string&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(DateTime&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(LocalDateTime&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(Timestamp&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	DynamicAnyHolder* clone() const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	bool isArray() const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}
};


template <>
class DynamicAnyHolderImpl<Int8>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(Int8 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(Int8);
	}

	void convert(Int8& val) const
	{
		val = _val;
	}

	void convert(Int16& val) const
	{
		val = _val;
	}
	
	void convert(Int32& val) const
	{
		val = _val;
	}

	void convert(Int64& val) const
	{
		val = _val;
	}

	void convert(UInt8& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		val = static_cast<char>(_val);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	void convert(DateTime&) const
	{
		throw BadCastException("Int8 -> DateTime");
	}

	void convert(LocalDateTime&) const
	{
		throw BadCastException("Int8 -> LocalDateTime");
	}

	void convert(Timestamp&) const
	{
		throw BadCastException("Int8 -> Timestamp");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}
	
	const Int8& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<Int8>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<Int8>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<Int8>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	Int8 _val;
};


template <>
class DynamicAnyHolderImpl<Int16>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(Int16 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(Int16);
	}

	void convert(Int8& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int16& val) const
	{
		val = _val;
	}
	
	void convert(Int32& val) const
	{
		val = _val;
	}

	void convert(Int64& val) const
	{
		val = _val;
	}

	void convert(UInt8& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	void convert(DateTime&) const
	{
		throw BadCastException("Int16 -> DateTime");
	}

	void convert(LocalDateTime&) const
	{
		throw BadCastException("Int16 -> LocalDateTime");
	}

	void convert(Timestamp&) const
	{
		throw BadCastException("Int16 -> Timestamp");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const Int16& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<Int16>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<Int16>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<Int16>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	Int16 _val;
};


template <>
class DynamicAnyHolderImpl<Int32>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(Int32 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(Int32);
	}

	void convert(Int8& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int16& val) const
	{
		convertToSmaller(_val, val);
	}
	
	void convert(Int32& val) const
	{
		val = _val;
	}

	void convert(Int64& val) const
	{
		val = _val;
	}

	void convert(UInt8& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	void convert(DateTime&) const
	{
		throw BadCastException("Int32 -> DateTime");
	}

	void convert(LocalDateTime&) const
	{
		throw BadCastException("Int32 -> LocalDateTime");
	}

	void convert(Timestamp&) const
	{
		throw BadCastException("Int32 -> Timestamp");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const Int32& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<Int32>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<Int32>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<Int32>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	Int32 _val;
};


template <>
class DynamicAnyHolderImpl<Int64>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(Int64 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(Int64);
	}

	void convert(Int8& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int16& val) const
	{
		convertToSmaller(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int64& val) const
	{
		val = _val;
	}

	void convert(UInt8& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	void convert(DateTime& dt) const
	{
		dt = Timestamp(_val);
	}

	void convert(LocalDateTime& ldt) const
	{
		ldt = Timestamp(_val);
	}

	void convert(Timestamp& val) const
	{
		val = Timestamp(_val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const Int64& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<Int64>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<Int64>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<Int64>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	Int64 _val;
};


template <>
class DynamicAnyHolderImpl<UInt8>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(UInt8 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(UInt8);
	}

	void convert(Int8& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int16& val) const
	{
		convertUnsignedToSigned(_val, val);
	}
	
	void convert(Int32& val) const
	{
		val = static_cast<Int32>(_val);
	}

	void convert(Int64& val) const
	{
		val = static_cast<Int64>(_val);
	}

	void convert(UInt8& val) const
	{
		val = _val;
	}

	void convert(UInt16& val) const
	{
		val = _val;
	}
	
	void convert(UInt32& val) const
	{
		val = _val;
	}

	void convert(UInt64& val) const
	{
		val = _val;
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	void convert(DateTime&) const
	{
		throw BadCastException("UInt8 -> DateTime");
	}

	void convert(LocalDateTime&) const
	{
		throw BadCastException("Unt8 -> LocalDateTime");
	}

	void convert(Timestamp&) const
	{
		throw BadCastException("UInt8 -> Timestamp");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const UInt8& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<UInt8>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<UInt8>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<UInt8>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	UInt8 _val;
};


template <>
class DynamicAnyHolderImpl<UInt16>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(UInt16 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(UInt16);
	}

	void convert(Int8& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int16& val) const
	{
		convertUnsignedToSigned(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int64& val) const
	{
		val = static_cast<Int64>(_val);
	}

	void convert(UInt8& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		val = _val;
	}
	
	void convert(UInt32& val) const
	{
		val = _val;
	}

	void convert(UInt64& val) const
	{
		val = _val;
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	void convert(DateTime&) const
	{
		throw BadCastException("UInt16 -> DateTime");
	}

	void convert(LocalDateTime&) const
	{
		throw BadCastException("UInt16 -> LocalDateTime");
	}

	void convert(Timestamp&) const
	{
		throw BadCastException("UInt16 -> Timestamp");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const UInt16& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<UInt16>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<UInt16>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<UInt16>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	UInt16 _val;
};


template <>
class DynamicAnyHolderImpl<UInt32>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(UInt32 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(UInt32);
	}

	void convert(Int8& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int16& val) const
	{
		convertUnsignedToSigned(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int64& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(UInt8& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		val = _val;
	}

	void convert(UInt64& val) const
	{
		val = _val;
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	void convert(DateTime&) const
	{
		throw BadCastException("UInt32 -> DateTime");
	}

	void convert(LocalDateTime&) const
	{
		throw BadCastException("UInt32 -> LocalDateTime");
	}

	void convert(Timestamp&) const
	{
		throw BadCastException("UInt32 -> Timestamp");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const UInt32& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<UInt32>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<UInt32>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<UInt32>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	UInt32 _val;
};


template <>
class DynamicAnyHolderImpl<UInt64>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(UInt64 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(UInt64);
	}

	void convert(Int8& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int16& val) const
	{
		convertUnsignedToSigned(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int64& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(UInt8& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		val = _val;
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	void convert(DateTime& dt) const
	{
		Int64 val;
		convertUnsignedToSigned(_val, val);
		dt = Timestamp(val);
	}

	void convert(LocalDateTime& ldt) const
	{
		Int64 val;
		convertUnsignedToSigned(_val, val);
		ldt = Timestamp(val);
	}

	void convert(Timestamp& val) const
	{
		Int64 tmp;
		convertUnsignedToSigned(_val, tmp);
		val = Timestamp(tmp);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const UInt64& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<UInt64>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<UInt64>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<UInt64>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	UInt64 _val;
};


template <>
class DynamicAnyHolderImpl<bool>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(bool val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(bool);
	}

	void convert(Int8& val) const
	{
		val = static_cast<Int8>(_val ? 1 : 0);
	}

	void convert(Int16& val) const
	{
		val = static_cast<Int16>(_val ? 1 : 0);
	}
	
	void convert(Int32& val) const
	{
		val = static_cast<Int32>(_val ? 1 : 0);
	}

	void convert(Int64& val) const
	{
		val = static_cast<Int64>(_val ? 1 : 0);
	}

	void convert(UInt8& val) const
	{
		val = static_cast<UInt8>(_val ? 1 : 0);
	}

	void convert(UInt16& val) const
	{
		val = static_cast<UInt16>(_val ? 1 : 0);
	}
	
	void convert(UInt32& val) const
	{
		val = static_cast<UInt32>(_val ? 1 : 0);
	}

	void convert(UInt64& val) const
	{
		val = static_cast<UInt64>(_val ? 1 : 0);
	}

	void convert(bool& val) const
	{
		val = _val;
	}

	void convert(float& val) const
	{
		val = (_val ? 1.0f : 0.0f);
	}

	void convert(double& val) const
	{
		val = (_val ? 1.0 : 0.0);
	}

	void convert(char& val) const
	{
		val = static_cast<char>(_val ? 1 : 0);
	}

	void convert(std::string& val) const
	{
		val = (_val ? "true" : "false");
	}

	void convert(DateTime&) const
	{
		throw BadCastException("bool -> DateTime");
	}

	void convert(LocalDateTime&) const
	{
		throw BadCastException("bool -> LocalDateTime");
	}

	void convert(Timestamp&) const
	{
		throw BadCastException("bool -> Timestamp");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const bool& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<bool>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<bool>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<bool>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	bool _val;
};


template <>
class DynamicAnyHolderImpl<float>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(float val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(float);
	}

	void convert(Int8& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int16& val) const
	{
		convertToSmaller(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int64& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(UInt8& val) const
	{
		convertSignedFloatToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedFloatToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedFloatToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedFloatToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = !(_val <= std::numeric_limits<float>::min() && 
			_val >= -1 * std::numeric_limits<float>::min());
	}

	void convert(float& val) const
	{
		val = _val;
	}

	void convert(double& val) const
	{
		val = _val;
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	void convert(DateTime&) const
	{
		throw BadCastException("float -> DateTime");
	}

	void convert(LocalDateTime&) const
	{
		throw BadCastException("float -> LocalDateTime");
	}

	void convert(Timestamp&) const
	{
		throw BadCastException("float -> Timestamp");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const float& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<float>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<float>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<float>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	float _val;
};


template <>
class DynamicAnyHolderImpl<double>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(double val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(double);
	}

	void convert(Int8& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int16& val) const
	{
		convertToSmaller(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int64& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(UInt8& val) const
	{
		convertSignedFloatToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedFloatToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedFloatToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedFloatToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = !(_val <= std::numeric_limits<double>::min() && 
			_val >= -1 * std::numeric_limits<double>::min());
	}

	void convert(float& val) const
	{
		double fMin = -1 * std::numeric_limits<float>::max();
		double fMax = std::numeric_limits<float>::max();

		if (_val < fMin) throw RangeException("Value too small.");
		if (_val > fMax) throw RangeException("Value too large.");

		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = _val;
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	void convert(DateTime&) const
	{
		throw BadCastException("double -> DateTime");
	}

	void convert(LocalDateTime&) const
	{
		throw BadCastException("double -> LocalDateTime");
	}

	void convert(Timestamp&) const
	{
		throw BadCastException("double -> Timestamp");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const double& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<double>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<double>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<double>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	double _val;
};


template <>
class DynamicAnyHolderImpl<char>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(char val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(char);
	}

	void convert(Int8& val) const
	{
		val = static_cast<Int8>(_val);
	}

	void convert(Int16& val) const
	{
		val = static_cast<UInt8>(_val);
	}
	
	void convert(Int32& val) const
	{
		val = static_cast<UInt8>(_val);
	}

	void convert(Int64& val) const
	{
		val = static_cast<UInt8>(_val);
	}

	void convert(UInt8& val) const
	{
		val = static_cast<UInt8>(_val);
	}

	void convert(UInt16& val) const
	{
		val = static_cast<UInt8>(_val);
	}
	
	void convert(UInt32& val) const
	{
		val = static_cast<UInt8>(_val);
	}

	void convert(UInt64& val) const
	{
		val = static_cast<UInt8>(_val);
	}

	void convert(bool& val) const
	{
		val = (_val != '\0');
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		val = _val;
	}

	void convert(std::string& val) const
	{
		val = std::string(1, _val);
	}

	void convert(DateTime&) const
	{
		throw BadCastException("char -> DateTime");
	}

	void convert(LocalDateTime&) const
	{
		throw BadCastException("char -> LocalDateTime");
	}

	void convert(Timestamp&) const
	{
		throw BadCastException("char -> Timestamp");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const char& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<char>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<char>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<char>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	char _val;
};


template <>
class DynamicAnyHolderImpl<std::string>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(const char* pVal): _val(pVal)
	{
	}

	DynamicAnyHolderImpl(const std::string& val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(std::string);
	}

	void convert(Int8& val) const
	{
		int v = NumberParser::parse(_val);
		convertToSmaller(v, val);
	}

	void convert(Int16& val) const
	{
		int v = NumberParser::parse(_val);
		convertToSmaller(v, val);
	}
	
	void convert(Int32& val) const
	{
		val = NumberParser::parse(_val);
	}

	void convert(Int64& val) const
	{
		val = NumberParser::parse64(_val);
	}

	void convert(UInt8& val) const
	{
		unsigned int v = NumberParser::parseUnsigned(_val);
		convertToSmallerUnsigned(v, val);
	}

	void convert(UInt16& val) const
	{
		unsigned int v = NumberParser::parseUnsigned(_val);
		convertToSmallerUnsigned(v, val);
	}
	
	void convert(UInt32& val) const
	{
		val = NumberParser::parseUnsigned(_val);
	}

	void convert(UInt64& val) const
	{
		val = NumberParser::parseUnsigned64(_val);
	}

	void convert(bool& val) const
	{
		static const std::string VAL_FALSE("false");
		static const std::string VAL_INTFALSE("0");

		if (_val == VAL_INTFALSE || (icompare(_val, VAL_FALSE) == 0))
			val = false;
		else
			val = true;
	}

	void convert(float& val) const
	{
		double v = NumberParser::parseFloat(_val);
		convertToSmaller(v, val);
	}

	void convert(double& val) const
	{
		val = NumberParser::parseFloat(_val);
	}

	void convert(char& val) const
	{
		if (_val.empty())
			val = '\0';
		else
			val = _val[0];
	}

	void convert(std::string& val) const
	{
		val = _val;
	}

	void convert(DateTime& val) const
	{
		int tzd = 0;
		if (!DateTimeParser::tryParse(DateTimeFormat::ISO8601_FORMAT, _val, val, tzd))
			throw BadCastException("string -> DateTime");
	}

	void convert(LocalDateTime& ldt) const
	{
		int tzd = 0;
		DateTime tmp;
		if (!DateTimeParser::tryParse(DateTimeFormat::ISO8601_FORMAT, _val, tmp, tzd))
			throw BadCastException("string -> LocalDateTime");

		ldt = LocalDateTime(tzd, tmp, false);
	}

	void convert(Timestamp& ts) const
	{
		int tzd = 0;
		DateTime tmp;
		if (!DateTimeParser::tryParse(DateTimeFormat::ISO8601_FORMAT, _val, tmp, tzd))
			throw BadCastException("string -> Timestamp");

		ts = tmp.timestamp();
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const std::string& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return false;
	}

	bool isSigned() const
	{
		return false;
	}

	bool isNumeric() const
	{
		return false;
	}

	bool isString() const
	{
		return true;
	}

private:
	std::string _val;
};


#ifndef POCO_LONG_IS_64_BIT


template <>
class DynamicAnyHolderImpl<long>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(long val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(long);
	}

	void convert(Int8& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int16& val) const
	{
		convertToSmaller(_val, val);
	}
	
	void convert(Int32& val) const
	{
		val = static_cast<Int32>(_val);
	}

	void convert(Int64& val) const
	{
		val = static_cast<Int64>(_val);
	}

	void convert(UInt8& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	void convert(DateTime&) const
	{
		throw BadCastException("long -> DateTime");
	}

	void convert(LocalDateTime&) const
	{
		throw BadCastException("long -> LocalDateTime");
	}

	void convert(Timestamp&) const
	{
		throw BadCastException("long -> Timestamp");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const long& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<long>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<long>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<long>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	long _val;
};


template <>
class DynamicAnyHolderImpl<unsigned long>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(unsigned long val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(unsigned long);
	}

	void convert(Int8& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int16& val) const
	{
		convertUnsignedToSigned(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int64& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(UInt8& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		val = static_cast<UInt64>(_val);
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	void convert(DateTime&) const
	{
		throw BadCastException("unsigned long -> DateTime");
	}

	void convert(LocalDateTime&) const
	{
		throw BadCastException("unsigned long -> LocalDateTime");
	}

	void convert(Timestamp&) const
	{
		throw BadCastException("unsigned long -> Timestamp");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const unsigned long& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return std::numeric_limits<unsigned long>::is_integer;
	}

	bool isSigned() const
	{
		return std::numeric_limits<unsigned long>::is_signed;
	}

	bool isNumeric() const
	{
		return std::numeric_limits<unsigned long>::is_specialized;
	}

	bool isString() const
	{
		return false;
	}

private:
	unsigned long _val;
};


#endif // 64bit


template <typename T>
class DynamicAnyHolderImpl<std::vector<T> >: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(const std::vector<T>& val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(std::vector<T>);
	}

	void convert(Int8& val) const
	{
		throw BadCastException("Cannot cast collection type to non-collection type");
	}

	void convert(Int16& val) const
	{
		throw BadCastException("Cannot cast collection type to non-collection type");
	}
	
	void convert(Int32& val) const
	{
		throw BadCastException("Cannot cast collection type to non-collection type");
	}

	void convert(Int64& val) const
	{
		throw BadCastException("Cannot cast collection type to non-collection type");
	}

	void convert(UInt8& val) const
	{
		throw BadCastException("Cannot cast collection type to non-collection type");
	}

	void convert(UInt16& val) const
	{
		throw BadCastException("Cannot cast collection type to non-collection type");
	}
	
	void convert(UInt32& val) const
	{
		throw BadCastException("Cannot cast collection type to non-collection type");
	}

	void convert(UInt64& val) const
	{
		throw BadCastException("Cannot cast collection type to non-collection type");
	}

	void convert(bool& val) const
	{
		throw BadCastException("Cannot cast collection type to non-collection type");
	}

	void convert(float& val) const
	{
		throw BadCastException("Cannot cast collection type to non-collection type");
	}

	void convert(double& val) const
	{
		throw BadCastException("Cannot cast collection type to non-collection type");
	}

	void convert(char& val) const
	{
		throw BadCastException("Cannot cast collection type to non-collection type");
	}

	void convert(std::string& val) const
	{
		throw BadCastException("Cannot cast collection type to non-collection type");
	}

	void convert(DateTime&) const
	{
		throw BadCastException("vector -> DateTime");
	}

	void convert(LocalDateTime&) const
	{
		throw BadCastException("vector -> LocalDateTime");
	}

	void convert(Timestamp&) const
	{
		throw BadCastException("vector -> Timestamp");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}
	
	const std::vector<T>& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return true;
	}

	bool isInteger() const
	{
		return false;
	}

	bool isSigned() const
	{
		return false;
	}

	bool isNumeric() const
	{
		return false;
	}

	bool isString() const
	{
		return false;
	}

	T& operator[](typename std::vector<T>::size_type n)
	{
		return _val.operator[](n);
	}

	const T& operator[](typename std::vector<T>::size_type n) const
	{
		return _val.operator[](n);
	}

private:
	std::vector<T> _val;
};


template <>
class DynamicAnyHolderImpl<DateTime>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(const DateTime& val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(DateTime);
	}

	void convert(Int8&) const
	{
		throw BadCastException();
	}

	void convert(Int16&) const
	{
		throw BadCastException();
	}
	
	void convert(Int32&) const
	{
		throw BadCastException();
	}

	void convert(Int64& val) const
	{
		val = _val.timestamp().epochMicroseconds();
	}

	void convert(UInt8&) const
	{
		throw BadCastException();
	}

	void convert(UInt16&) const
	{
		throw BadCastException();
	}
	
	void convert(UInt32&) const
	{
		throw BadCastException();
	}

	void convert(UInt64& val) const
	{
		val = _val.timestamp().epochMicroseconds();
	}

	void convert(bool&) const
	{
		throw BadCastException();
	}

	void convert(float&) const
	{
		throw BadCastException();
	}

	void convert(double&) const
	{
		throw BadCastException();
	}

	void convert(char&) const
	{
		throw BadCastException();
	}

	void convert(std::string& val) const
	{
		val = DateTimeFormatter::format(_val, Poco::DateTimeFormat::ISO8601_FORMAT);
	}

	void convert(DateTime& val) const
	{
		val = _val;
	}

	void convert(LocalDateTime& ldt) const
	{
		ldt = _val.timestamp();
	}

	void convert(Timestamp& ts) const
	{
		ts = _val.timestamp();
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}
	
	const DateTime& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return false;
	}

	bool isSigned() const
	{
		return false;
	}

	bool isNumeric() const
	{
		return false;
	}

	bool isString() const
	{
		return false;
	}

private:
	DateTime _val;
};


template <>
class DynamicAnyHolderImpl<LocalDateTime>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(const LocalDateTime& val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(LocalDateTime);
	}

	void convert(Int8&) const
	{
		throw BadCastException();
	}

	void convert(Int16&) const
	{
		throw BadCastException();
	}
	
	void convert(Int32&) const
	{
		throw BadCastException();
	}

	void convert(Int64& val) const
	{
		val = _val.timestamp().epochMicroseconds();
	}

	void convert(UInt8&) const
	{
		throw BadCastException();
	}

	void convert(UInt16&) const
	{
		throw BadCastException();
	}
	
	void convert(UInt32&) const
	{
		throw BadCastException();
	}

	void convert(UInt64& val) const
	{
		val = _val.timestamp().epochMicroseconds();
	}

	void convert(bool&) const
	{
		throw BadCastException();
	}

	void convert(float&) const
	{
		throw BadCastException();
	}

	void convert(double&) const
	{
		throw BadCastException();
	}

	void convert(char&) const
	{
		throw BadCastException();
	}

	void convert(std::string& val) const
	{
		val = DateTimeFormatter::format(_val, Poco::DateTimeFormat::ISO8601_FORMAT);
	}

	void convert(DateTime& val) const
	{
		val = _val.timestamp();
	}

	void convert(LocalDateTime& ldt) const
	{
		ldt = _val;
	}

	void convert(Timestamp& ts) const
	{
		ts = _val.timestamp();
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}
	
	const LocalDateTime& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return false;
	}

	bool isSigned() const
	{
		return false;
	}

	bool isNumeric() const
	{
		return false;
	}

	bool isString() const
	{
		return false;
	}

private:
	LocalDateTime _val;
};


template <>
class DynamicAnyHolderImpl<Timestamp>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(const Timestamp& val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(Timestamp);
	}

	void convert(Int8&) const
	{
		throw BadCastException();
	}

	void convert(Int16&) const
	{
		throw BadCastException();
	}
	
	void convert(Int32&) const
	{
		throw BadCastException();
	}

	void convert(Int64& val) const
	{
		val = _val.epochMicroseconds();
	}

	void convert(UInt8&) const
	{
		throw BadCastException();
	}

	void convert(UInt16&) const
	{
		throw BadCastException();
	}
	
	void convert(UInt32&) const
	{
		throw BadCastException();
	}

	void convert(UInt64& val) const
	{
		val = _val.epochMicroseconds();
	}

	void convert(bool&) const
	{
		throw BadCastException();
	}

	void convert(float&) const
	{
		throw BadCastException();
	}

	void convert(double&) const
	{
		throw BadCastException();
	}

	void convert(char&) const
	{
		throw BadCastException();
	}

	void convert(std::string& val) const
	{
		val = DateTimeFormatter::format(_val, Poco::DateTimeFormat::ISO8601_FORMAT);
	}

	void convert(DateTime& val) const
	{
		val = _val;
	}

	void convert(LocalDateTime& ldt) const
	{
		ldt = _val;
	}

	void convert(Timestamp& ts) const
	{
		ts = _val;
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}
	
	const Timestamp& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isInteger() const
	{
		return false;
	}

	bool isSigned() const
	{
		return false;
	}

	bool isNumeric() const
	{
		return false;
	}

	bool isString() const
	{
		return false;
	}

private:
	Timestamp _val;
};


} // namespace Poco


#endif // Foundation_DynamicAnyHolder_INCLUDED
