"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var vega_util_1 = require("vega-util");
var channel_1 = require("./channel");
var log = require("./log");
var util_1 = require("./util");
var ScaleType;
(function (ScaleType) {
    // Continuous - Quantitative
    ScaleType.LINEAR = 'linear';
    ScaleType.BIN_LINEAR = 'bin-linear';
    ScaleType.LOG = 'log';
    ScaleType.POW = 'pow';
    ScaleType.SQRT = 'sqrt';
    // Continuous - Time
    ScaleType.TIME = 'time';
    ScaleType.UTC = 'utc';
    // sequential
    ScaleType.SEQUENTIAL = 'sequential';
    // Quantile, Quantize, threshold
    ScaleType.QUANTILE = 'quantile';
    ScaleType.QUANTIZE = 'quantize';
    ScaleType.THRESHOLD = 'threshold';
    ScaleType.ORDINAL = 'ordinal';
    ScaleType.BIN_ORDINAL = 'bin-ordinal';
    ScaleType.POINT = 'point';
    ScaleType.BAND = 'band';
})(ScaleType = exports.ScaleType || (exports.ScaleType = {}));
/**
 * Index for scale categories -- only scale of the same categories can be merged together.
 * Current implementation is trying to be conservative and avoid merging scale type that might not work together
 */
var SCALE_CATEGORY_INDEX = {
    linear: 'numeric',
    log: 'numeric',
    pow: 'numeric',
    sqrt: 'numeric',
    'bin-linear': 'bin-linear',
    time: 'time',
    utc: 'time',
    sequential: 'sequential',
    ordinal: 'ordinal',
    'bin-ordinal': 'bin-ordinal',
    point: 'ordinal-position',
    band: 'ordinal-position'
};
exports.SCALE_TYPES = util_1.keys(SCALE_CATEGORY_INDEX);
/**
 * Whether the two given scale types can be merged together.
 */
function scaleCompatible(scaleType1, scaleType2) {
    var scaleCategory1 = SCALE_CATEGORY_INDEX[scaleType1];
    var scaleCategory2 = SCALE_CATEGORY_INDEX[scaleType2];
    return scaleCategory1 === scaleCategory2 ||
        (scaleCategory1 === 'ordinal-position' && scaleCategory2 === 'time') ||
        (scaleCategory2 === 'ordinal-position' && scaleCategory1 === 'time');
}
exports.scaleCompatible = scaleCompatible;
/**
 * Index for scale precedence -- high score = higher priority for merging.
 */
var SCALE_PRECEDENCE_INDEX = {
    // numeric
    linear: 0,
    log: 1,
    pow: 1,
    sqrt: 1,
    // time
    time: 0,
    utc: 0,
    // ordinal-position -- these have higher precedence than continuous scales as they support more types of data
    point: 10,
    band: 11,
    // non grouped types
    'bin-linear': 0,
    sequential: 0,
    ordinal: 0,
    'bin-ordinal': 0,
};
/**
 * Return scale categories -- only scale of the same categories can be merged together.
 */
function scaleTypePrecedence(scaleType) {
    return SCALE_PRECEDENCE_INDEX[scaleType];
}
exports.scaleTypePrecedence = scaleTypePrecedence;
exports.CONTINUOUS_TO_CONTINUOUS_SCALES = ['linear', 'bin-linear', 'log', 'pow', 'sqrt', 'time', 'utc'];
var CONTINUOUS_TO_CONTINUOUS_INDEX = vega_util_1.toSet(exports.CONTINUOUS_TO_CONTINUOUS_SCALES);
exports.CONTINUOUS_DOMAIN_SCALES = exports.CONTINUOUS_TO_CONTINUOUS_SCALES.concat(['sequential' /* TODO add 'quantile', 'quantize', 'threshold'*/]);
var CONTINUOUS_DOMAIN_INDEX = vega_util_1.toSet(exports.CONTINUOUS_DOMAIN_SCALES);
exports.DISCRETE_DOMAIN_SCALES = ['ordinal', 'bin-ordinal', 'point', 'band'];
var DISCRETE_DOMAIN_INDEX = vega_util_1.toSet(exports.DISCRETE_DOMAIN_SCALES);
var BIN_SCALES_INDEX = vega_util_1.toSet(['bin-linear', 'bin-ordinal']);
exports.TIME_SCALE_TYPES = ['time', 'utc'];
function hasDiscreteDomain(type) {
    return type in DISCRETE_DOMAIN_INDEX;
}
exports.hasDiscreteDomain = hasDiscreteDomain;
function isBinScale(type) {
    return type in BIN_SCALES_INDEX;
}
exports.isBinScale = isBinScale;
function hasContinuousDomain(type) {
    return type in CONTINUOUS_DOMAIN_INDEX;
}
exports.hasContinuousDomain = hasContinuousDomain;
function isContinuousToContinuous(type) {
    return type in CONTINUOUS_TO_CONTINUOUS_INDEX;
}
exports.isContinuousToContinuous = isContinuousToContinuous;
exports.defaultScaleConfig = {
    textXRangeStep: 90,
    rangeStep: 21,
    pointPadding: 0.5,
    bandPaddingInner: 0.1,
    facetSpacing: 16,
    minBandSize: 2,
    minFontSize: 8,
    maxFontSize: 40,
    minOpacity: 0.3,
    maxOpacity: 0.8,
    // FIXME: revise if these *can* become ratios of rangeStep
    minSize: 9,
    minStrokeWidth: 1,
    maxStrokeWidth: 4
};
function isExtendedScheme(scheme) {
    return scheme && !!scheme['name'];
}
exports.isExtendedScheme = isExtendedScheme;
function isSelectionDomain(domain) {
    return domain && domain['selection'];
}
exports.isSelectionDomain = isSelectionDomain;
var SCALE_PROPERTY_INDEX = {
    type: 1,
    domain: 1,
    range: 1,
    rangeStep: 1,
    scheme: 1,
    // Other properties
    reverse: 1,
    round: 1,
    // quantitative / time
    clamp: 1,
    nice: 1,
    // quantitative
    base: 1,
    exponent: 1,
    interpolate: 1,
    zero: 1,
    // band/point
    padding: 1,
    paddingInner: 1,
    paddingOuter: 1
};
exports.SCALE_PROPERTIES = util_1.flagKeys(SCALE_PROPERTY_INDEX);
var type = SCALE_PROPERTY_INDEX.type, domain = SCALE_PROPERTY_INDEX.domain, range = SCALE_PROPERTY_INDEX.range, rangeStep = SCALE_PROPERTY_INDEX.rangeStep, scheme = SCALE_PROPERTY_INDEX.scheme, NON_TYPE_DOMAIN_RANGE_VEGA_SCALE_PROPERTY_INDEX = tslib_1.__rest(SCALE_PROPERTY_INDEX, ["type", "domain", "range", "rangeStep", "scheme"]);
exports.NON_TYPE_DOMAIN_RANGE_VEGA_SCALE_PROPERTIES = util_1.flagKeys(NON_TYPE_DOMAIN_RANGE_VEGA_SCALE_PROPERTY_INDEX);
function scaleTypeSupportProperty(scaleType, propName) {
    switch (propName) {
        case 'type':
        case 'domain':
        case 'reverse':
        case 'range':
            return true;
        case 'scheme':
            return util_1.contains(['sequential', 'ordinal', 'bin-ordinal', 'quantile', 'quantize'], scaleType);
        case 'interpolate':
            // FIXME(https://github.com/vega/vega-lite/issues/2902) how about ordinal?
            return util_1.contains(['linear', 'bin-linear', 'pow', 'log', 'sqrt', 'utc', 'time'], scaleType);
        case 'round':
            return isContinuousToContinuous(scaleType) || scaleType === 'band' || scaleType === 'point';
        case 'padding':
            return isContinuousToContinuous(scaleType) || util_1.contains(['point', 'band'], scaleType);
        case 'paddingOuter':
        case 'rangeStep':
            return util_1.contains(['point', 'band'], scaleType);
        case 'paddingInner':
            return scaleType === 'band';
        case 'clamp':
            return isContinuousToContinuous(scaleType) || scaleType === 'sequential';
        case 'nice':
            return isContinuousToContinuous(scaleType) || scaleType === 'sequential' || scaleType === 'quantize';
        case 'exponent':
            return scaleType === 'pow';
        case 'base':
            return scaleType === 'log';
        case 'zero':
            return hasContinuousDomain(scaleType) && !util_1.contains([
                'log',
                'time', 'utc',
                'bin-linear',
                'threshold',
                'quantile' // quantile depends on distribution so zero does not matter
            ], scaleType);
    }
    /* istanbul ignore next: should never reach here*/
    throw new Error("Invalid scale property " + propName + ".");
}
exports.scaleTypeSupportProperty = scaleTypeSupportProperty;
/**
 * Returns undefined if the input channel supports the input scale property name
 */
function channelScalePropertyIncompatability(channel, propName) {
    switch (propName) {
        case 'interpolate':
        case 'scheme':
            if (!channel_1.isColorChannel(channel)) {
                return log.message.cannotUseScalePropertyWithNonColor(channel);
            }
            return undefined;
        case 'type':
        case 'domain':
        case 'range':
        case 'base':
        case 'exponent':
        case 'nice':
        case 'padding':
        case 'paddingInner':
        case 'paddingOuter':
        case 'rangeStep':
        case 'reverse':
        case 'round':
        case 'clamp':
        case 'zero':
            return undefined; // GOOD!
    }
    /* istanbul ignore next: it should never reach here */
    throw new Error("Invalid scale property \"" + propName + "\".");
}
exports.channelScalePropertyIncompatability = channelScalePropertyIncompatability;
function channelSupportScaleType(channel, scaleType) {
    switch (channel) {
        case channel_1.Channel.X:
        case channel_1.Channel.Y:
        case channel_1.Channel.SIZE: // TODO: size and opacity can support ordinal with more modification
        case channel_1.Channel.OPACITY:
            // Although it generally doesn't make sense to use band with size and opacity,
            // it can also work since we use band: 0.5 to get midpoint.
            return isContinuousToContinuous(scaleType) || util_1.contains(['band', 'point'], scaleType);
        case channel_1.Channel.COLOR:
        case channel_1.Channel.FILL:
        case channel_1.Channel.STROKE:
            return scaleType !== 'band'; // band does not make sense with color
        case channel_1.Channel.SHAPE:
            return scaleType === 'ordinal'; // shape = lookup only
    }
    /* istanbul ignore next: it should never reach here */
    return false;
}
exports.channelSupportScaleType = channelSupportScaleType;
//# sourceMappingURL=data:application/json;base64,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