--[[
Defines which attributes are exported by our C++ classes. The required C functions (used as
callbacks from Lua) are automatically generated by src/lua/generate_bindings.lua

The classes table contains all classes that will be accessible in Lua. The key is the name
of the class in C++. Keep in mind that it has to be a valid Lua identifiers, you can add a
typedef in catalua.cpp, e.g. `using cppstring = std::string;`, than add a class with the name
`cppstring` here. It maps to a table with the following values:

Each class requires at least the attributes and the functions table (they can be empty).
Optional values are:
- by_value (boolean, default: false): if true, copy the C++ object into memory managed by Lua
  (the copy may outlive the source C++ object), otherwise only a pointer to the C++ object
  is stored in Lua and it *needs* to stay valid in C++ until the Lua object is gone.
- by_value_and_reference (boolean, default: false): if true, the class is can be exported to Lua
  as value (copy of the object, managed by Lua) *and* as reference (to an object managed by C++
  code). This flag implies "by_value", "by_value" should therefor not be specified explicitly.
- has_equal (boolean, default: false): If true, generate the __eq entry in the metatable which
  will map to the C++ using the operator==.
- new (an array of parameter lists): defines the constructor of the object. This is only useful for
  by_value objects, it allows to create an instance of it in Lua. The entry should be an array,
  each element of it represents one overload of the constructor. Each element should be a list of
  parameters to those overloads (same as the list of arguments to member functions).
- int_id (optional, a string): if the class has an associated int_id (e.g. ter_t has int_id<ter_t>,
  which is typedefed to ter_id), this can be used to define that int_id (for ter_t is should be
  "ter_id"). At the end of this file, this will be used to create an actual entry in the classes
  table for the type name given here.
  This is done because all the int_id objects have essentially the same functions.

The attributes table contains the members of the C++ class. Each key is the name of the member,
it maps to a map with the following values:
- cpp_name (defaults to the name of the member): an alternatively name for the member,
  used only in the C++ part of the wrapper.
- writable (boolean, default: false): if true, a wrapper to set the member is generated (otherwise
  only the getter is generated and the member is effectively read-only).
- type (required): the type of the member (see below).

The functions table contains the member functions of the C++ class. Each entry (no keys, it's an
array) should have the following values:
- name (required, string): the name of the function as it appears in Lua.
- cpp_name (defaults to the name of the function, string): an alternatively name for the function,
  used only in the C++ part of the wrapper.
- args (an array of strings): the types (see below) of the parameters to the function (in order).
- rval (can be nil when the function returns void): the type (see below) that the function returns.

The global_functions table is basically the same as the class functions table, but it has
(currently) the function name as key of the table (this disallows function overloading, and should
be considered a bug). TODO: fix it

Types can be built in types: int, float, bool, string (std::string in C++), cstring (const char* in C++)
Or any class defined in the classes table.

Regarding reference and the by_value setting:
Instances of the itype class are created by C++ (specifically by the Item_factory). They are
never created elsewhere and once created each instance will stay valid (and at the same address)
until the game has ended. Furthermore it's actually important that each item id maps to exactly
one instance of itype. Similar for mtype.
Therefor those objects are exported as pointer only (by_value is false).

Other objects (e.g. tripoint) can be constructed as values in Lua (by_value is true).

The return value of functions ("rval") behaves a bit special:
- If it ends with a '&', as in `rval = "item&"`, it is assumed the function returns a reference.
  The by_value setting will determine whether to copy the referred object (to Lua memory), or
  to store only a reference in Lua memory.
  Using '&' on native Lua types is not allowed.
- If the return type is a class type (has an entry in classes), it is copied to Lua memory. This
  will fail (when the generated bindings are compiled or when they get linked) if the type does not
  support copying. One should either add `by_value = true` or `by_value_and_reference = true` to the type.
- Otherwise it should be a native Lua type, which will be copied.

Example: (Creature::pos returns a const reference to tripoint, game::zombie returns a reference to monster)
`local m = game.zombie(0)` stores a pointer to the monster returned by `game::zombie(0)`. This reference
must not be used after the monster has been removed from the game (e.g. because it died). Calling
the function again will give the very same pointer:
`local n = game.zombie(0)`
`m.setpos(tripoint(10,0,0))` This also affects `n` because `m` and `n` are pointers to the very
same C++ object.
`local p = some_monster:pos()` stores a copy of the result of `some_monster.pos()` in `p`. This
variable can be used long after `some_monster` has been killed and removed from the game. It has
no connection at all to the monster.
--]]

classes = {
    effect_type = {
        string_id = "efftype_id",
        attributes = {
        },
        functions = {
        }
    },
    calendar = {
        new = {
            { "calendar" },
            { "int" },
            { "int", "int", "int", "season_type", "int" },
        },
        by_value_and_reference = true,
        attributes = {
        },
        functions = {
            { name = "day_of_week", rval = "string", args = { } },
            { name = "day_of_year", rval = "int", args = { } },
            { name = "get_turn", rval = "int", args = { } },
            { name = "increment", rval = nil, args = { } },
            { name = "print_time", rval = "string", args = { "bool" } },
            { name = "print_time", rval = "string", args = { } },
            { name = "is_night", rval = "bool", args = { } },
            { name = "sunlight", rval = "float", args = { } },
            { name = "turn_of_year", rval = "int", args = { } },
            { name = "years", rval = "int", args = { } },
            { name = "sunset", rval = "calendar", args = { } },
            { name = "sunrise", rval = "calendar", args = { } },
        }
    },
    mutation_branch = {
        string_id = "trait_id",
        attributes = {
            activated = { type = "bool", writable = true },
            bodytemp_max = { type = "int", writable = true },
            bodytemp_min = { type = "int", writable = true },
            bodytemp_sleep = { type = "int", writable = true },
            cooldown = { type = "int", writable = true },
            cost = { type = "int", writable = true },
            description = { type = "string", writable = true },
            fatigue = { type = "bool", writable = true },
            hunger = { type = "bool", writable = true },
            mixed_effect = { type = "bool", writable = true },
            name = { type = "string", writable = true },
            points = { type = "int", writable = true },
            profession = { type = "bool", writable = true },
            purifiable = { type = "bool", writable = true },
            startingtrait = { type = "bool", writable = true },
            thirst = { type = "bool", writable = true },
            threshold = { type = "bool", writable = true },
            ugliness = { type = "int", writable = true },
            valid = { type = "bool", writable = true },
            visibility = { type = "int", writable = true },
        },
        functions = {
            { name = "get_display_color", rval = "nc_color", args = { } },
        }
    },
    Character = {
        parent = "Creature",
        attributes = {
            dex_cur = { type = "int", writable = true },
            dex_max = { type = "int", writable = true },
            int_cur = { type = "int", writable = true },
            int_max = { type = "int", writable = true },
            per_cur = { type = "int", writable = true },
            per_max = { type = "int", writable = true },
            str_cur = { type = "int", writable = true },
            str_max = { type = "int", writable = true },
            male = { type = "bool", writable = true },
            name = { type = "string", writable = true },
            nv_cached = { type = "bool", writable = true },
            weapon = { type = "item", writable = true },
        },
        functions = {
            { name = "add_effect", rval = nil, args = { "efftype_id", "int" } },
            { name = "add_effect", rval = nil, args = { "efftype_id", "int", "body_part" } },
            { name = "add_effect", rval = nil, args = { "efftype_id", "int", "body_part", "bool" } },
            { name = "add_effect", rval = nil, args = { "efftype_id", "int", "body_part", "bool", "int" } },
            { name = "add_effect", rval = nil, args = { "efftype_id", "int", "body_part", "bool", "int", "bool" } },
            { name = "add_traits", rval = nil, args = { } },
            { name = "aim_per_move", rval = "float", args = { "item", "float" } },
            { name = "bloodType", rval = "field_id", args = { } },
            { name = "can_pickVolume", rval = "bool", args = { "item" } },
            { name = "can_pickVolume", rval = "bool", args = { "item", "bool" } },
            { name = "can_pickWeight", rval = "bool", args = { "item" } },
            { name = "can_pickWeight", rval = "bool", args = { "item", "bool" } },
            { name = "can_use", rval = "bool", args = { "item" } },
            { name = "can_use", rval = "bool", args = { "item", "item" } },
            { name = "die", rval = nil, args = { "Creature" } },
            { name = "empty_skills", rval = nil, args = { } },
            { name = "empty_traits", rval = nil, args = { } },
            { name = "get_dodge_base", rval = "float", args = { } },
            { name = "get_hit_base", rval = "float", args = { } },
            { name = "get_item_position", rval = "int", args = { "item" } },
            { name = "get_name", rval = "string", args = { } },
            { name = "get_time_died", rval = "time_point", args = { } },
            { name = "get_vision_threshold", rval = "float", args = { "int" } },
            { name = "gibType", rval = "field_id", args = { } },
            { name = "has_active_item", rval = "bool", args = { "string" } },
            { name = "has_active_mutation", rval = "bool", args = { "trait_id" } },
            { name = "has_base_trait", rval = "bool", args = { "trait_id" } },
            { name = "has_nv", rval = "bool", args = { } },
            { name = "has_trait", rval = "bool", args = { "trait_id" } },
            { name = "has_trait_flag", rval = "bool", args = { "string" } },
            { name = "i_add", rval = "item&", args = { "item" } },
            { name = "i_add_or_drop", rval = "bool", args = { "item" } },
            { name = "i_add_or_drop", rval = "bool", args = { "item", "int" } },
            { name = "i_at", rval = "item&", args = { "int" } },
            { name = "i_rem", rval = "item", args = { "int" } },
            { name = "i_rem", rval = "item", args = { "item" } },
            { name = "i_rem_keep_contents", rval = nil, args = { "int" } },
            { name = "is_warm", rval = "bool", args = { } },
            { name = "is_wearing", rval = "bool", args = { "string" } },
            { name = "is_wearing_on_bp", rval = "bool", args = { "string", "body_part" } },
            { name = "is_worn", rval = "bool", args = { "item" } },
            { name = "limb_color", rval = "nc_color", args = { "body_part", "bool", "bool", "bool" } },
            { name = "made_of", rval = "bool", args = { "material_id" } },
            { name = "mod_int_bonus", rval = nil, args = { "int" } },
            { name = "mod_stat", rval = nil, args = { "string", "int" } },
            { name = "move_effects", rval = "bool", args = { "bool" } },
            { name = "mutation_effect", rval = nil, args = { "trait_id" } },
            { name = "mutation_loss_effect", rval = nil, args = { "trait_id" } },
            { name = "normalize", rval = nil, args = { } },
            { name = "pick_name", rval = nil, args = { } },
            { name = "pick_name", rval = nil, args = { "bool" } },
            { name = "random_bad_trait", rval = "trait_id", args = { } },
            { name = "random_good_trait", rval = "trait_id", args = { } },
            { name = "ranged_dex_mod", rval = "int", args = { } },
            { name = "ranged_per_mod", rval = "int", args = { } },
            { name = "recalc_hp", rval = nil, args = { } },
            { name = "recalc_sight_limits", rval = nil, args = { } },
            { name = "remove_mission_items", rval = nil, args = { "int" } },
            { name = "remove_weapon", rval = "item", args = { } },
            { name = "reset", rval = nil, args = { } },
            { name = "reset_bonuses", rval = nil, args = { } },
            { name = "reset_stats", rval = nil, args = { } },
            { name = "set_time_died", rval = nil, args = { "time_point" } },
            { name = "symbol", rval = "string", args = { } },
            { name = "set_mutation", rval = nil, args = { "trait_id" } },
            { name = "symbol_color", rval = "nc_color", args = { } },
            { name = "unset_mutation", rval = nil, args = { "trait_id" } },
            { name = "toggle_trait", rval = nil, args = { "trait_id" } },
            { name = "trait_by_invlet", rval = "trait_id", args = { "int" } },
            { name = "volume_capacity", rval = "volume", args = { } },
            { name = "volume_carried", rval = "volume", args = { } },
            { name = "weight_capacity", rval = "mass", args = { } },
            { name = "weight_carried", rval = "mass", args = { } },
            { name = "worn_with_flag", rval = "bool", args = { "string" } },
            { name = "get_healthy", rval = "int", args = { } },
            { name = "get_healthy_mod", rval = "int", args = { } },
            { name = "get_str", rval = "int", args = { } },
            { name = "get_str_base", rval = "int", args = { } },
            { name = "get_str_bonus", rval = "int", args = { } },
            { name = "get_dex", rval = "int", args = { } },
            { name = "get_dex_base", rval = "int", args = { } },
            { name = "get_dex_bonus", rval = "int", args = { } },
            { name = "get_int", rval = "int", args = { } },
            { name = "get_int_base", rval = "int", args = { } },
            { name = "get_int_bonus", rval = "int", args = { } },
            { name = "get_per", rval = "int", args = { } },
            { name = "get_per_base", rval = "int", args = { } },
            { name = "get_per_bonus", rval = "int", args = { } },
            { name = "get_hunger", rval = "int", args = { } },
            { name = "get_thirst", rval = "int", args = { } },
            { name = "get_stomach_food", rval = "int", args = { } },
            { name = "get_stomach_water", rval = "int", args = { } },
            { name = "mod_fatigue", rval = nil, args = { "int" } },
            { name = "set_fatigue", rval = nil, args = { "int" } },
            { name = "get_fatigue", rval = "int", args = { } },
            { name = "set_dex_bonus", rval = nil, args = { "int" } },
            { name = "set_healthy", rval = nil, args = { "int" } },
            { name = "set_healthy_mod", rval = nil, args = { "int" } },
            { name = "set_int_bonus", rval = nil, args = { "int" } },
            { name = "set_per_bonus", rval = nil, args = { "int" } },
            { name = "set_str_bonus", rval = nil, args = { "int" } },
            { name = "set_hunger", rval = nil, args = { "int" } },
            { name = "set_thirst", rval = nil, args = { "int" } },
            { name = "set_stomach_food", rval = nil, args = { "int" } },
            { name = "set_stomach_water", rval = nil, args = { "int" } },
            { name = "mod_dex_bonus", rval = nil, args = { "int" } },
            { name = "mod_healthy", rval = nil, args = { "int" } },
            { name = "mod_healthy_mod", rval = nil, args = { "int", "int" } },
            { name = "mod_per_bonus", rval = nil, args = { "int" } },
            { name = "mod_str_bonus", rval = nil, args = { "int" } },
            { name = "mod_hunger", rval = nil, args = { "int" } },
            { name = "mod_thirst", rval = nil, args = { "int" } },
            { name = "mod_stomach_food", rval = nil, args = { "int" } },
            { name = "mod_stomach_water", rval = nil, args = { "int" } },
            { name = "update_health", rval = nil, args = { "int" } },
            { name = "update_health", rval = nil, args = { } },
            { name = "mod_stat", rval = nil, args = { "string", "int" } }
        }
    },
    item_stack_iterator = {
        by_value = true,
        has_equal = true,
        new = {
            { "item_stack_iterator" },
        },
        attributes = {
        },
        functions = {
            { name = "elem", rval = "item&", cpp_name = "operator*"; args = { } },
            { name = "inc", rval = nil, cpp_name = "operator++", args = { } },
        }
    },
    map_stack = {
        by_value = true,
        attributes = {
        },
        functions = {
            { name = "cppbegin", rval = "item_stack_iterator", cpp_name = "begin", args = { } },
            { name = "cppend", rval = "item_stack_iterator", cpp_name = "end", args = {} },
            { name = "empty", rval = "bool", args = { } },
            { name = "erase", rval = "item_stack_iterator", args = { "item_stack_iterator" } },
            { name = "push_back", rval = nil, args = { "item" } },
            { name = "size", rval = "int", args = { } },
        }
    },
    game = {
        attributes = {
            lightning_active = { type = "bool", writable = true },
        },
        functions = {
            { name = "add_zombie", rval = "bool", args = { "monster" } },
            { name = "add_zombie", rval = "bool", args = { "monster", "bool" } },
            { name = "assign_faction_id", rval = "int", args = { } },
            { name = "assign_mission_id", rval = "int", args = { } },
            { name = "assign_npc_id", rval = "int", args = { } },
            { name = "cancel_activity", rval = nil, args = { } },
            { name = "clear_zombies", rval = nil, args = { } },
            { name = "critter_at", rval = "Creature&", args = { "tripoint" } },
            { name = "critter_at", rval = "Creature&", args = { "tripoint", "bool" } },
            { name = "critter_by_id", rval = "Creature&", args = { "int" } },
            { name = "do_blast", rval = nil, args = { "tripoint", "float", "float", "bool" } },
            { name = "draw", rval = nil, args = { } },
            { name = "draw_critter", rval = nil, args = { "Creature", "tripoint" } },
            { name = "draw_explosion", rval = nil, args = { "tripoint", "int", "nc_color" } },
            { name = "draw_hit_mon", rval = nil, args = { "tripoint", "monster" } },
            { name = "draw_hit_mon", rval = nil, args = { "tripoint", "monster", "bool" } },
            { name = "draw_hit_player", rval = nil, args = { "player", "int" } },
            { name = "draw_sct", rval = nil, args = { } },
            { name = "draw_ter", rval = nil, args = { "bool" } },
            { name = "draw_ter", rval = nil, args = { "tripoint" } },
            { name = "draw_ter", rval = nil, args = { "tripoint", "bool" } },
            { name = "draw_ter", rval = nil, args = { "tripoint", "bool", "bool" } },
            { name = "draw_ter", rval = nil, args = { } },
            { name = "draw_trail_to_square", rval = nil, args = { "tripoint", "bool" } },
            { name = "emp_blast", rval = nil, args = { "tripoint" } },
            { name = "explosion", rval = nil, args = { "tripoint", "float" } },
            { name = "explosion", rval = nil, args = { "tripoint", "float", "float" } },
            { name = "explosion", rval = nil, args = { "tripoint", "float", "float", "int" } },
            { name = "explosion", rval = nil, args = { "tripoint", "float", "float", "int", "bool" } },
            { name = "flashbang", rval = nil, args = { "tripoint" } },
            { name = "flashbang", rval = nil, args = { "tripoint", "bool" } },
            { name = "fling_creature", rval = nil, args = { "Creature", "int", "float" } },
            { name = "fling_creature", rval = nil, args = { "Creature", "int", "float", "bool" } },
            { name = "get_cur_om", rval = "overmap&", args = { } },
            { name = "get_levx", rval = "int", args = { } },
            { name = "get_levy", rval = "int", args = { } },
            { name = "get_levz", rval = "int", args = { } },
            { name = "get_temperature", rval = "int", args = { } },
            { name = "handle_liquid", rval = "bool", args = { "item" } },
            { name = "increase_kill_count", rval = nil, args = { "mtype_id" } },
            { name = "inv_for_all", rval = "int", args = { "string" } },
            { name = "inv_for_flag", rval = "int", args = { "string", "string" } },
            { name = "inventory_item_menu", rval = "int", args = { "int" } },
            { name = "inventory_item_menu", rval = "int", args = { "int", "int" } },
            { name = "inventory_item_menu", rval = "int", args = { "int", "int", "int" } },
            { name = "is_empty", rval = "bool", args = { "tripoint" } },
            { name = "is_hostile_nearby", rval = "Creature&", args = { } },
            { name = "is_hostile_very_close", rval = "Creature&", args = { } },
            { name = "is_in_sunlight", rval = "bool", args = { "tripoint" } },
            { name = "is_sheltered", rval = "bool", args = { "tripoint" } },
            { name = "kill_count", rval = "int", args = { "mtype_id" } },
            { name = "knockback", rval = nil, args = { "tripoint", "tripoint", "int", "int", "int" } },
            { name = "light_level", rval = "int", args = { "int" } },
            { name = "look_around", rval = "tripoint", args = { } },
            { name = "look_debug", rval = "tripoint", args = { } },
            { name = "natural_light_level", rval = "float", args = { "int" } },
            { name = "nuke", rval = nil, args = { "tripoint" } },
            { name = "num_creatures", rval = "int", args = { } },
            { name = "open_gate", rval = nil, args = { "tripoint" } },
            { name = "peek", rval = nil, args = { "tripoint" } },
            { name = "peek", rval = nil, args = { } },
            { name = "plfire", rval = nil, args = { } },
            { name = "plswim", rval = nil, args = { "tripoint" } },
            { name = "refresh_all", rval = nil, args = { } },
            { name = "remove_zombie", rval = nil, args = { "monster" } },
            { name = "resonance_cascade", rval = nil, args = { "tripoint" } },
            { name = "revive_corpse", rval = "bool", args = { "tripoint", "item" } },
            { name = "save", rval = "bool", args = { } },
            { name = "scrambler_blast", rval = nil, args = { "tripoint" } },
            { name = "shockwave", rval = nil, args = { "tripoint", "int", "int", "int", "int", "bool" } },
            { name = "spawn_hallucination", rval = "bool", args = { } },
            { name = "summon_mon", rval = "monster&", args = { "mtype_id", "tripoint" } },
            { name = "swap_critters", rval = "bool", args = { "Creature", "Creature" } },
            { name = "teleport", rval = nil, args = { "player" } },
            { name = "teleport", rval = nil, args = { "player", "bool" } },
            { name = "teleport", rval = nil, args = { } },
            { name = "unload", rval = nil, args = { "int" } },
            { name = "unload", rval = nil, args = { "item" } },
            { name = "unload", rval = nil, args = { } },
            { name = "use_computer", rval = nil, args = { "tripoint" } },
        }
    },
    encumbrance_data = {
        by_value = true,
        new = {
            { "encumbrance_data" },
        },
        attributes = {
            encumbrance = { type = "int", writable = true },
            armor_encumbrance = { type = "int", writable = true },
            layer_penalty = { type = "int", writable = true },
        },
        functions = {
        }
    },
    stats = {
        attributes = {
            damage_healed = { type = "int", writable = true },
            damage_taken = { type = "int", writable = true },
            headshots = { type = "int", writable = true },
            squares_walked = { type = "int", writable = true },
        },
        functions = {
        }
    },
    player = {
        parent = "Character",
        attributes = {
            blocks_left = { type = "int", writable = true },
            cash = { type = "int", writable = true },
            controlling_vehicle = { type = "bool", writable = true },
            dodges_left = { type = "int", writable = true },
            focus_pool = { type = "int", writable = true },
            grab_point = { type = "tripoint", writable = true },
            in_vehicle = { type = "bool", writable = true },
            keep_hands_free = { type = "bool", writable = true },
            last_batch = { type = "int", writable = true },
            last_climate_control_ret = { type = "bool", writable = true },
            lastconsumed = { type = "string", writable = true },
            lastrecipe = { type = "recipe_id", writable = true },
            lifetime_stats = { type = "stats", writable = true },
            max_power_level = { type = "int", writable = true },
            move_mode = { type = "string", writable = true },
            movecounter = { type = "int", writable = true },
            next_climate_control_check = { type = "time_point", writable = true },
            oxygen = { type = "int", writable = true },
            power_level = { type = "int", writable = true },
            radiation = { type = "int", writable = true },
            reactor_plut = { type = "int", writable = true },
            recoil = { type = "int", writable = true },
            scent = { type = "int", writable = true },
            slow_rad = { type = "int", writable = true },
            stamina = { type = "int", writable = true },
            start_location = { type = "start_location_id", writable = true },
            stim = { type = "int", writable = true },
            style_selected = { type = "matype_id", writable = true },
            tank_plut = { type = "int", writable = true },
            view_offset = { type = "tripoint", writable = true },
            volume = { type = "int", writable = true },
        },
        functions = {
            { name = "action_taken", rval = nil, args = { } },
            { name = "activate_bionic", rval = "bool", args = { "int" } },
            { name = "activate_bionic", rval = "bool", args = { "int", "bool" } },
            { name = "active_light", rval = "float", args = { } },
            { name = "add_addiction", rval = nil, args = { "add_type", "int" } },
            { name = "add_known_trap", rval = nil, args = { "tripoint", "trap" } },
            { name = "add_martialart", rval = nil, args = { "matype_id" } },
            { name = "add_pain_msg", rval = nil, args = { "int", "body_part" } },
            { name = "addiction_level", rval = "int", args = { "add_type" } },
            { name = "adjacent_tile", rval = "tripoint", args = { } },
            { name = "adjust_for_focus", rval = "int", args = { "int" } },
            { name = "amount_of", rval = "int", args = { "string" } },
            { name = "amount_worn", rval = "int", args = { "string" } },
            { name = "apply_damage", rval = nil, args = { "Creature", "body_part", "int" } },
            { name = "apply_persistent_morale", rval = nil, args = { } },
            { name = "attack_speed", rval = "int", args = { "item" } },
            { name = "avoid_trap", rval = "bool", args = { "tripoint", "trap" } },
            { name = "best_shield", rval = "item&", args = { } },
            { name = "bonus_damage", rval = "float", args = { "bool" } },
            { name = "basic_symbol_color", rval = "nc_color", args = { } },
            { name = "blossoms", rval = nil, args = { } },
            { name = "bodytemp_color", rval = "nc_color", args = { "int" } },
            { name = "bonus_item_warmth", rval = "int", args = { "body_part" } },
            { name = "boost_skill_level", rval = nil, args = { "skill_id", "int" } },
            { name = "burn_move_stamina", rval = nil, args = { "int" } },
            { name = "calc_focus_equilibrium", rval = "int", args = { } },
            { name = "can_arm_block", rval = "bool", args = { } },
            { name = "can_hear", rval = "bool", args = { "tripoint", "int" } },
            { name = "can_leg_block", rval = "bool", args = { } },
            { name = "can_limb_block", rval = "bool", args = { } },
            { name = "can_melee", rval = "bool", args = { } },
            { name = "can_sleep", rval = "bool", args = { } },
            { name = "can_reload", rval = "bool", args = { "item", "string" } },
            { name = "can_reload", rval = "bool", args = { "item" } },
            { name = "cancel_activity", rval = nil, args = { } },
            { name = "change_side", rval = "bool", args = { "int" } },
            { name = "change_side", rval = "bool", args = { "int", "bool" } },
            { name = "change_side", rval = "bool", args = { "item" } },
            { name = "change_side", rval = "bool", args = { "item", "bool" } },
            { name = "charge_power", rval = nil, args = { "int" } },
            { name = "charges_of", rval = "int", args = { "string" } },
            { name = "clairvoyance", rval = "int", args = { } },
            { name = "clear_destination", rval = nil, args = { } },
            { name = "clear_miss_reasons", rval = nil, args = { } },
            { name = "climbing_cost", rval = "int", args = { "tripoint", "tripoint" } },
            { name = "complete_craft", rval = nil, args = { } },
            { name = "complete_disassemble", rval = nil, args = { } },
            { name = "consume", rval = "bool", args = { "int" } },
            { name = "consume_charges", rval = "bool", args = { "item", "int" } },
            { name = "consume_effects", rval = nil, args = { "item" } },
            { name = "consume_item", rval = "bool", args = { "item" } },
            { name = "cough", rval = nil, args = { "bool" } },
            { name = "cough", rval = nil, args = { "bool", "int" } },
            { name = "cough", rval = nil, args = { } },
            { name = "craft", rval = nil, args = { } },
            { name = "crit_chance", rval = "float", args = { "int", "int", "item" } },
            { name = "crossed_threshold", rval = "bool", args = { } },
            { name = "deactivate_bionic", rval = "bool", args = { "int" } },
            { name = "deactivate_bionic", rval = "bool", args = { "int", "bool" } },
            { name = "digging", rval = "bool", args = { } },
            { name = "disassemble", rval = "bool", args = { } },
            { name = "disassemble", rval = "bool", args = { "int" } },
            { name = "disassemble", rval = "bool", args = { "item", "int", "bool" } },
            { name = "disassemble", rval = "bool", args = { "item", "int", "bool", "bool" } },
            { name = "disp_info", rval = nil, args = { } },
            { name = "disp_morale", rval = nil, args = { } },
            { name = "disp_name", rval = "string", args = { "bool" } },
            { name = "disp_name", rval = "string", args = { } },
            { name = "do_read", rval = nil, args = { "item" } },
            { name = "dodge_roll", rval = "float", args = { } },
            { name = "drench", rval = nil, args = { "int", "int", "bool" } },
            { name = "drench_mut_calc", rval = nil, args = { } },
            { name = "apply_wetness_morale", rval = nil, args = { "int" } },
            { name = "drink_from_hands", rval = "int", args = { "item" } },
            { name = "dump_memorial", rval = "string", args = { } },
            { name = "eat", rval = "bool", args = { "item" } },
            { name = "eat", rval = "bool", args = { "item", "bool" } },
            { name = "environmental_revert_effect", rval = nil, args = { } },
            { name = "fall_asleep", rval = nil, args = { "int" } },
            { name = "fall_damage_mod", rval = "float", args = { } },
            { name = "fine_detail_vision_mod", rval = "float", args = { } },
            { name = "fire_gun", rval = "int", args = { "tripoint" } },
            { name = "fire_gun", rval = "int", args = { "tripoint", "int" } },
            { name = "fire_gun", rval = "int", args = { "tripoint", "int", "item" } },
            { name = "fire_gun", rval = nil, args = { "tripoint", "bool" } },
            { name = "fire_gun", rval = nil, args = { "tripoint", "bool", "item" } },
            { name = "footwear_factor", rval = "float", args = { } },
            { name = "getID", rval = "int", args = { } },
            { name = "get_active_mission_target", rval = "tripoint", args = { } },
            { name = "get_armor_acid", rval = "int", args = { "body_part" } },
            { name = "get_armor_bash", rval = "int", args = { "body_part" } },
            { name = "get_armor_bash_base", rval = "int", args = { "body_part" } },
            { name = "get_armor_cut", rval = "int", args = { "body_part" } },
            { name = "get_armor_cut_base", rval = "int", args = { "body_part" } },
            { name = "get_armor_fire", rval = "int", args = { "body_part" } },
            { name = "get_category_dream", rval = "string", args = { "string", "int" } },
            { name = "get_combat_style", rval = "martialart&", args = { } },
            { name = "get_dodge", rval = "float", args = { } },
            { name = "get_dodge_base", rval = "float", args = { } },
            { name = "get_env_resist", rval = "int", args = { "body_part" } },
            { name = "get_highest_category", rval = "string", args = { } },
            { name = "get_hit_base", rval = "float", args = { } },
            { name = "get_hit_weapon", rval = "float", args = { "item" } },
            { name = "get_hp", rval = "int", args = { "hp_part" } },
            { name = "get_hp_max", rval = "int", args = { "hp_part" } },
            { name = "get_item_position", rval = "int", args = { "item" } },
            { name = "get_melee", rval = "float", args = { } },
            { name = "get_random_body_part", rval = "body_part", args = { "bool" } },
            { name = "get_sick", rval = nil, args = { } },
            { name = "get_size", rval = "m_size", args = { } },
            { name = "get_stamina_max", rval = "int", args = { } },
            { name = "get_wind_resistance", rval = "int", args = { "body_part" } },
            { name = "global_omt_location", rval = "tripoint", args = { } },
            { name = "global_sm_location", rval = "tripoint", args = { } },
            { name = "global_square_location", rval = "tripoint", args = { } },
            { name = "gun_value", rval = "float", args = { "item" } },
            { name = "gun_value", rval = "float", args = { "item", "int" } },
            { name = "gunmod_add", rval = nil, args = { "item", "item" } },
            { name = "gunmod_remove", rval = "bool", args = { "item", "item" } },
            { name = "handle_melee_wear", rval = "bool", args = { "item" } },
            { name = "handle_melee_wear", rval = "bool", args = { "item", "float" } },
            { name = "has_active_optcloak", rval = "bool", args = { } },
            { name = "has_addiction", rval = "bool", args = { "add_type" } },
            { name = "has_alarm_clock", rval = "bool", args = { } },
            { name = "has_amount", rval = "bool", args = { "string", "int" } },
            { name = "has_charges", rval = "bool", args = { "string", "int" } },
            { name = "has_child_flag", rval = "bool", args = { "trait_id" } },
            { name = "has_conflicting_trait", rval = "bool", args = { "trait_id" } },
            { name = "has_destination", rval = "bool", args = { } },
            { name = "has_grab_break_tec", rval = "bool", args = { } },
            { name = "has_gun_for_ammo", rval = "bool", args = { "ammotype" } },
            { name = "has_higher_trait", rval = "bool", args = { "trait_id" } },
            { name = "has_identified", rval = "bool", args = { "string" } },
            { name = "has_item", rval = "bool", args = { "item" } },
            { name = "has_item_with_flag", rval = "bool", args = { "string" } },
            { name = "has_lower_trait", rval = "bool", args = { "trait_id" } },
            { name = "has_mabuff", rval = "bool", args = { "mabuff_id" } },
            { name = "has_martialart", rval = "bool", args = { "matype_id" } },
            { name = "has_miss_recovery_tec", rval = "bool", args = { "item" } },
            { name = "has_mission_item", rval = "bool", args = { "int" } },
            { name = "has_morale_to_craft", rval = "bool", args = { } },
            { name = "has_morale_to_read", rval = "bool", args = { } },
            { name = "has_opposite_trait", rval = "bool", args = { "trait_id" } },
            { name = "has_technique", rval = "bool", args = { "matec_id", "item" } },
            { name = "has_two_arms", rval = "bool", args = { } },
            { name = "has_watch", rval = "bool", args = { } },
            { name = "has_weapon", rval = "bool", args = { } },
            { name = "heal", rval = nil, args = { "body_part", "int" } },
            { name = "heal", rval = nil, args = { "hp_part", "int" } },
            { name = "healall", rval = nil, args = { "int" } },
            { name = "hearing_ability", rval = "float", args = { } },
            { name = "hit_roll", rval = "float", args = { } },
            { name = "hitall", rval = "int", args = { "int", "int", "Creature" } },
            { name = "hp_percentage", rval = "int", args = { } },
            { name = "hurtall", rval = nil, args = { "int", "Creature" } },
            { name = "i_at", rval = "item&", args = { "int" } },
            { name = "impact", rval = "int", args = { "int", "tripoint" } },
            { name = "in_climate_control", rval = "bool", args = { } },
            { name = "install_bionics", rval = "bool", args = { "itype" } },
            { name = "install_bionics", rval = "bool", args = { "itype", "int" } },
            { name = "intimidation", rval = "int", args = { } },
            { name = "invalidate_crafting_inventory", rval = nil, args = { } },
            { name = "invlet_to_position", rval = "int", args = { "int" } },
            { name = "invoke_item", rval = "bool", args = { "item" } },
            { name = "invoke_item", rval = "bool", args = { "item", "string" } },
            { name = "invoke_item", rval = "bool", args = { "item", "string", "tripoint" } },
            { name = "invoke_item", rval = "bool", args = { "item", "tripoint" } },            
            { name = "is_armed", rval = "bool", args = { } },
            { name = "is_dead_state", rval = "bool", args = { } },
            { name = "is_deaf", rval = "bool", args = { } },
            { name = "is_elec_immune", rval = "bool", args = { } },
            { name = "is_hallucination", rval = "bool", args = { } },
            { name = "is_hibernating", rval = "bool", args = { } },
            { name = "is_immune_effect", rval = "bool", args = { "efftype_id" } },
            { name = "is_invisible", rval = "bool", args = { } },
            { name = "is_npc", rval = "bool", args = { } },
            { name = "is_on_ground", rval = "bool", args = { } },
            { name = "is_player", rval = "bool", args = { } },
            { name = "is_quiet", rval = "bool", args = { } },
            { name = "is_snuggling", rval = "string", args = { } },
            { name = "is_throw_immune", rval = "bool", args = { } },
            { name = "is_underwater", rval = "bool", args = { } },
            { name = "is_wearing_active_power_armor", rval = "bool", args = { } },
            { name = "is_wearing_power_armor", rval = "bool", args = { } },
            { name = "is_wearing_shoes", rval = "bool", args = { "string" } },
            { name = "is_wearing_shoes", rval = "bool", args = { } },
            { name = "item_handling_cost", rval = "int", args = { "item" } },
            { name = "item_handling_cost", rval = "int", args = { "item", "bool" } },
            { name = "item_handling_cost", rval = "int", args = { "item", "bool", "int" } },
            { name = "item_reload_cost", rval = "int", args = { "item", "item", "int" } },
            { name = "knock_back_from", rval = nil, args = { "tripoint" } },
            { name = "knows_trap", rval = "bool", args = { "tripoint" } },
            { name = "leak_level", rval = "int", args = { "string" } },
            { name = "load_info", rval = nil, args = { "string" } },
            { name = "long_craft", rval = nil, args = { } },
            { name = "ma_onattack_effects", rval = nil, args = { } },
            { name = "ma_onblock_effects", rval = nil, args = { } },
            { name = "ma_ondodge_effects", rval = nil, args = { } },
            { name = "ma_ongethit_effects", rval = nil, args = { } },
            { name = "ma_onhit_effects", rval = nil, args = { } },
            { name = "ma_onmove_effects", rval = nil, args = { } },
            { name = "ma_static_effects", rval = nil, args = { } },
            { name = "mabuff_block_bonus", rval = "int", args = { } },
            { name = "mabuff_dodge_bonus", rval = "int", args = { } },
            { name = "mabuff_speed_bonus", rval = "int", args = { } },
            { name = "mabuff_tohit_bonus", rval = "int", args = { } },
            { name = "make_all_craft", rval = nil, args = { "recipe_id", "int" } },
            { name = "make_craft", rval = nil, args = { "recipe_id", "int" } },
            { name = "make_craft_with_command", rval = nil, args = { "recipe_id", "int" } },
            { name = "make_craft_with_command", rval = nil, args = { "recipe_id", "int", "bool" } },
            { name = "making_would_work", rval = "bool", args = { "recipe_id", "int" } },
            { name = "max_quality", rval = "int", args = { "quality_id" } },
            { name = "melee_attack", rval = nil, args = { "Creature", "bool", "matec_id" } },
            { name = "melee_value", rval = "float", args = { "item" } },
            { name = "mend", rval = nil, args = { "int" } },
            { name = "metabolic_rate", rval = "float", args = { } },
            { name = "metabolic_rate_base", rval = "float", args = { } },
            { name = "mod_pain", rval = nil, args = { "int" } },
            { name = "set_pain", rval = nil, args = { "int" } },
            { name = "get_perceived_pain", rval = "int", args = { } },
            { name = "mod_painkiller", rval = nil, args = { "int" } },
            { name = "set_painkiller", rval = nil, args = { "int" } },
            { name = "get_painkiller", rval = "int", args = { } },
            { name = "mod_stat", rval = nil, args = { "string", "int" } },
            { name = "get_morale_level", rval = "int", args = { } },
            { name = "mutate", rval = nil, args = { } },
            { name = "mutate_category", rval = nil, args = { "string" } },
            { name = "mutate_towards", rval = "bool", args = { "trait_id" } },
            { name = "mutate_towards", rval = nil, args = { "trait_id" } },
            { name = "mutation_ok", rval = "bool", args = { "trait_id", "bool", "bool" } },
            { name = "natural_attack_restricted_on", rval = "bool", args = { "body_part" } },
            { name = "normalize", rval = nil, args = { } },
            { name = "num_bionics", rval = "int", args = { } },
            { name = "nutrition_for", rval = "int", args = { "item" } },
            { name = "on_dodge", rval = nil, args = { "Creature", "float" } },
            { name = "on_hit", rval = nil, args = { "Creature" } },
            { name = "on_hit", rval = nil, args = { "Creature", "body_part" } },
            { name = "on_hit", rval = nil, args = { "Creature", "body_part", "float" } },
            { name = "on_hurt", rval = nil, args = { "Creature" } },
            { name = "on_hurt", rval = nil, args = { "Creature", "bool" } },
            { name = "overmap_los", rval = "bool", args = { "tripoint", "int" } },
            { name = "overmap_sight_range", rval = "int", args = { "int" } },
            { name = "pause", rval = nil, args = { } },
            { name = "perform_special_attacks", rval = nil, args = { "Creature" } },
            { name = "pick_style", rval = "bool", args = { } },
            { name = "pick_style", rval = nil, args = { } },
            { name = "pick_technique", rval = "matec_id", args = { "Creature", "item", "bool", "bool", "bool" } },
            { name = "place_corpse", rval = nil, args = { } },
            { name = "pos", rval = "tripoint", args = { } },
            { name = "posx", rval = "int", args = { } },
            { name = "posy", rval = "int", args = { } },
            { name = "posz", rval = "int", args = { } },
            { name = "power_bionics", rval = nil, args = { } },
            { name = "power_mutations", rval = nil, args = { } },
            { name = "power_rating", rval = "float", args = { } },
            { name = "practice", rval = nil, args = { "skill_id", "int" } },
            { name = "practice", rval = nil, args = { "skill_id", "int", "int" } },
            { name = "process_active_items", rval = nil, args = { } },
            { name = "process_bionic", rval = nil, args = { "int" } },
            { name = "process_effects", rval = nil, args = { } },
            { name = "process_turn", rval = nil, args = { } },
            { name = "purifiable", rval = "bool", args = { "trait_id" } },
            { name = "ranged_dex_mod", rval = "int", args = { } },
            { name = "ranged_per_mod", rval = "int", args = { } },
            { name = "reach_attack", rval = nil, args = { "tripoint" } },
            { name = "read", rval = "bool", args = { "int" } },
            { name = "read_speed", rval = "int", args = { "bool" } },
            { name = "read_speed", rval = "int", args = { } },
            { name = "recalc_speed_bonus", rval = nil, args = { } },
            { name = "recraft", rval = nil, args = { } },
            { name = "reduce_charges", rval = "item", args = { "int", "int" } },
            { name = "reduce_charges", rval = "item", args = { "item", "int" } },
            { name = "regen", rval = nil, args = { "int" } },
            { name = "rem_addiction", rval = nil, args = { "add_type" } },
            { name = "remove_child_flag", rval = nil, args = { "trait_id" } },
            { name = "remove_mutation", rval = nil, args = { "trait_id" } },
            { name = "remove_random_bionic", rval = "bool", args = { } },
            { name = "reset_stats", rval = nil, args = { } },
            { name = "rooted", rval = nil, args = { } },
            { name = "rooted_message", rval = nil, args = { } },
            { name = "run_cost", rval = "int", args = { "int" } },
            { name = "run_cost", rval = "int", args = { "int", "bool" } },
            { name = "rust_rate", rval = "int", args = { "bool" } },
            { name = "rust_rate", rval = "int", args = { } },
            { name = "save_info", rval = "string", args = { } },
            { name = "scored_crit", rval = "bool", args = { "float", "item" } },
            { name = "search_surroundings", rval = nil, args = { } },
            { name = "sees", rval = "bool", args = { "Creature" } },
            { name = "sees", rval = "bool", args = { "tripoint" } },
            { name = "sees", rval = "bool", args = { "tripoint", "bool" } },
            { name = "sees_with_infrared", rval = "bool", args = { "Creature" } },
            { name = "setID", rval = nil, args = { "int" } },
            { name = "set_highest_cat_level", rval = nil, args = { } },
            { name = "get_skill_level", rval = "int", args = { "skill_id" } },
            { name = "set_skill_level", rval = nil, args = { "skill_id", "int" } },
            { name = "set_underwater", rval = nil, args = { "bool" } },
            { name = "setpos", rval = nil, args = { "tripoint" } },
            { name = "setx", rval = nil, args = { "int" } },
            { name = "sety", rval = nil, args = { "int" } },
            { name = "setz", rval = nil, args = { "int" } },
            { name = "shift_destination", rval = nil, args = { "int", "int" } },
            { name = "shoe_type_count", rval = "int", args = { "string" } },
            { name = "shout", rval = nil, args = { } },
            { name = "shout", rval = nil, args = { "string" } },
            { name = "sight_impaired", rval = "bool", args = { } },
            { name = "sight_range", rval = "int", args = { "int" } },
            { name = "skin_name", rval = "string", args = { } },
            { name = "sleep_spot", rval = "int", args = { "tripoint" } },
            { name = "sort_armor", rval = nil, args = { } },
            { name = "speed_rating", rval = "float", args = { } },
            { name = "spores", rval = nil, args = { } },
            { name = "stability_roll", rval = "int", args = { } },
            { name = "stomach_capacity", rval = "int", args = { } },
            { name = "store", rval = nil, args = { "item", "item" } },
            { name = "store", rval = nil, args = { "item", "item", "int" } },
            { name = "store", rval = nil, args = { "item", "item", "int", "bool" } },
            { name = "studied_all_recipes", rval = "bool", args = { "itype" } },
            { name = "suffer", rval = nil, args = { } },
            { name = "swim_speed", rval = "int", args = { } },
            { name = "takeoff", rval = bool, args = { "item" } },
            { name = "talk_skill", rval = "int", args = { } },
            { name = "temp_equalizer", rval = nil, args = { "body_part", "body_part" } },
            { name = "throw_range", rval = "int", args = { "item" } },
            { name = "toggle_move_mode", rval = nil, args = { } },
            { name = "try_to_sleep", rval = nil, args = { } },
            { name = "unarmed_attack", rval = "bool", args = { } },
            { name = "unarmed_value", rval = "float", args = { } },
            { name = "uncanny_dodge", rval = "bool", args = { } },
            { name = "unimpaired_range", rval = "int", args = { } },
            { name = "update_body_wetness", rval = nil, args = { "w_point" } },
            { name = "update_bodytemp", rval = nil, args = { } },
            { name = "update_health", rval = nil, args = { "int" } },
            { name = "update_health", rval = nil, args = { } },
            { name = "update_mental_focus", rval = nil, args = { } },
            { name = "update_morale", rval = nil, args = { } },
            { name = "update_needs", rval = nil, args = { "int" } },
            { name = "update_stamina", rval = nil, args = { "int" } },
            { name = "use", rval = nil, args = { "int" } },
            { name = "use_charges_if_avail", rval = "bool", args = { "string", "int" } },
            { name = "use_wielded", rval = nil, args = { } },
            { name = "visibility", rval = "int", args = { "bool" } },
            { name = "visibility", rval = "int", args = { "bool", "int" } },
            { name = "visibility", rval = "int", args = { } },
            { name = "vomit", rval = nil, args = { } },
            { name = "vomit_mod", rval = "float", args = { } },
            { name = "wake_up", rval = nil, args = { } },
            { name = "warmth", rval = "int", args = { "body_part" } },
            { name = "weapname", rval = "string", args = { } },
            { name = "weapon_value", rval = "float", args = { "item" } },
            { name = "weapon_value", rval = "float", args = { "item", "int" } },
            { name = "wear", rval = "bool", args = { "int" } },
            { name = "wear", rval = "bool", args = { "int", "bool" } },
            { name = "wear_item", rval = "bool", args = { "item" } },
            { name = "wear_item", rval = "bool", args = { "item", "bool" } },
            { name = "wearing_something_on", rval = "bool", args = { "body_part" } },
            { name = "wield", rval = "bool", args = { "item" } },
            { name = "wield_contents", rval = "bool", args = { "item" } },
            { name = "wield_contents", rval = "bool", args = { "item", "int" } },
            { name = "wield_contents", rval = "bool", args = { "item", "int", "int" } },
            { name = "wield_contents", rval = "bool", args = { "item", "int", "int", "bool" } },
        }
    },
    item = {
        new = {
            { },
            { "item" },
            { "string", "int" },
            { "string", "int", "int" },
            { "itype", "int" },
            { "itype", "int", "int" },
            -- TODO: export constructor with default_charges_tag paremeter
        },
        by_value_and_reference = true,
        attributes = {
            active = { type = "bool", writable = true },
            burnt = { type = "int", writable = true },
            charges = { type = "int", writable = true },
            frequency = { type = "int", writable = true },
            fridge = { type = "int", writable = true },
            invlet = { type = "int", writable = true },
            irridation = { type = "int", writable = true },
            item_counter = { type = "int", writable = true },
            mission_id = { type = "int", writable = true },
            note = { type = "int", writable = true },
            player_id = { type = "int", writable = true },
            poison = { type = "int", writable = true },
            type = { type = "itype", writable = true },
        },
        functions = {
            { name = "age", rval = "time_duration", args = { } },
            { name = "set_age", rval = nil, args = { "time_duration" } },
            { name = "birthday", rval = "time_point", args = { } },
            { name = "set_birthday", rval = nil, args = { "time_point" } },
            { name = "acid_resist", rval = "int", args = { } },
            { name = "acid_resist", rval = "int", args = { "bool" } },
            { name = "add_rain_to_container", rval = nil, args = { "bool" } },
            { name = "add_rain_to_container", rval = nil, args = { "bool", "int" } },
            { name = "add_technique", rval = nil, args = { "matec_id" } },
            { name = "already_used_by_player", rval = "bool", args = { "player" } },
            { name = "ammo_capacity", rval = "int", args = { } },
            { name = "ammo_current", rval = "string", args = { } },
            { name = "ammo_data", rval = "itype&", args = { } },
            { name = "ammo_remaining", rval = "int", args = { } },
            { name = "ammo_required", rval = "int", args = { } },
            { name = "ammo_type", rval = "ammotype", args = { "bool" } },
            { name = "ammo_type", rval = "ammotype", args = { } },
            { name = "amount_of", rval = "int", args = { "string", "bool" } },
            { name = "attack_time", rval = "int", args = { } },
            { name = "bash_resist", rval = "int", args = { } },
            { name = "bash_resist", rval = "int", args = { "bool" } },
            { name = "brewing_time", rval = "time_duration", args = { } },
            { name = "calc_rot", rval = nil, args = { "tripoint" } },
            { name = "can_holster", rval = "bool", args = { "item" } },
            { name = "can_holster", rval = "bool", args = { "item", "bool" } },
            { name = "can_revive", rval = "bool", args = { } },
            { name = "charges_of", rval = "int", args = { "string" } },
            { name = "chip_resistance", rval = "int", args = { "bool" } },
            { name = "chip_resistance", rval = "int", args = { } },
            { name = "clear_vars", rval = nil, args = { } },
            { name = "color", rval = "nc_color", args = { } },
            { name = "color_in_inventory", rval = "nc_color", args = { } },
            { name = "components_to_string", rval = "string", args = { } },
            { name = "conductive", rval = "bool", args = { } },
            { name = "convert", rval = "item&", args = { "string" } },
            { name = "count_by_charges", rval = "bool", args = { } },
            { name = "covers", rval = "bool", args = { "body_part" } },
            { name = "craft_has_charges", rval = "bool", args = { } },
            { name = "cut_resist", rval = "int", args = { } },
            { name = "cut_resist", rval = "int", args = { "bool" } },
            { name = "damage", rval = "int", args = { } },
            { name = "destroyed_at_zero_charges", rval = "bool", args = { } },
            { name = "display_name", rval = "string", args = { "int" } },
            { name = "display_name", rval = "string", args = { } },
            { name = "erase_var", rval = nil, args = { "string" } },
            { name = "fill_with", rval = nil, args = { "item" } },
            { name = "fire_resist", rval = "int", args = { } },
            { name = "fire_resist", rval = "int", args = { "bool" } },
            { name = "flammable", rval = "bool", args = { } },
            { name = "get_chapters", rval = "int", args = { } },
            { name = "get_coverage", rval = "int", args = { } },
            { name = "get_encumber", rval = "int", args = { } },
            { name = "get_env_resist", rval = "int", args = { } },
            { name = "get_free_mod_locations", rval = "int", args = { "string" } },
            { name = "get_gun_ups_drain", rval = "int", args = { } },
            { name = "get_layer", rval = "int", args = { } },
            { name = "get_mtype", rval = "mtype&", args = { } },
            { name = "get_plant_epoch", rval = "time_duration", args = { } },
            { name = "get_plant_name", rval = "string", args = { } },
            { name = "get_property_long", rval = "int", args = { "string" } },
            { name = "get_property_long", rval = "int", args = { "string", "int" } },
            { name = "get_property_string", rval = "string", args = { "string" } },
            { name = "get_property_string", rval = "string", args = { "string", "string" } },
            { name = "get_quality", rval = "int", args = { "quality_id" } },
            { name = "get_relative_rot", rval = "float", args = { } },
            { name = "get_remaining_capacity_for_liquid", rval = "int", args = { "item", "bool" } },
            { name = "get_remaining_chapters", rval = "int", args = { "player" } },
            { name = "get_rot", rval = "int", args = { } },
            { name = "get_side", rval = "side", args = { } },
            { name = "get_storage", rval = "volume", args = { } },
            { name = "get_thickness", rval = "int", args = { } },
            { name = "get_usable_item", rval = "item&", args = { "string" } },
            { name = "get_var", rval = "float", args = { "string", "float" } },
            { name = "get_var", rval = "int", args = { "string", "int" } },
            { name = "get_var", rval = "int", args = { "string", "int" } },
            { name = "get_var", rval = "string", args = { "string" } },
            { name = "get_var", rval = "string", args = { "string", "string" } },
            { name = "get_warmth", rval = "int", args = { } },
            { name = "getlight_emit", rval = "int", args = { } },
            { name = "goes_bad", rval = "bool", args = { } },
            { name = "gun_cycle_mode", rval = nil, args = { } },
            { name = "gun_damage", rval = "int", args = { "bool" } },
            { name = "gun_damage", rval = "int", args = { } },
            { name = "gun_dispersion", rval = "int", args = { "bool" } },
            { name = "gun_dispersion", rval = "int", args = { } },
            { name = "gun_pierce", rval = "int", args = { "bool" } },
            { name = "gun_pierce", rval = "int", args = { } },
            { name = "gun_range", rval = "int", args = { "bool" } },
            { name = "gun_range", rval = "int", args = { "player" } },
            { name = "gun_range", rval = "int", args = { } },
            { name = "gun_recoil", rval = "int", args = { "player" } },
            { name = "gun_recoil", rval = "int", args = { "player", "bool" } },
            { name = "gun_set_mode", rval = nil, args = { "string" } },
            { name = "gun_skill", rval = "skill_id", args = { } },
            { name = "gunmod_find", rval = "item&", args = { "string" } },
            { name = "has_flag", rval = "bool", args = { "string" } },
            { name = "has_label", rval = "bool", args = { } },
            { name = "has_property", rval = "bool", args = { "string" } },
            { name = "has_quality", rval = "bool", args = { "quality_id" } },
            { name = "has_quality", rval = "bool", args = { "quality_id", "int" } },
            { name = "has_rotten_away", rval = "bool", args = { } },
            { name = "has_technique", rval = "bool", args = { "matec_id" } },
            { name = "has_var", rval = "bool", args = { "string" } },
            { name = "in_its_container", rval = "item", args = { } },
            { name = "inc_damage", rval = "bool", args = { } },
            { name = "info", rval = "string", args = { "bool" } },
            { name = "info", rval = "string", args = { } },
            { name = "is_ammo", rval = "bool", args = { } },
            { name = "is_ammo_container", rval = "bool", args = { } },
            { name = "is_armor", rval = "bool", args = { } },
            { name = "is_artifact", rval = "bool", args = { } },
            { name = "is_bionic", rval = "bool", args = { } },
            { name = "is_book", rval = "bool", args = { } },
            { name = "is_container", rval = "bool", args = { } },
            { name = "is_container_empty", rval = "bool", args = { } },
            { name = "is_container_full", rval = "bool", args = { } },
            { name = "is_corpse", rval = "bool", args = { } },
            { name = "is_dangerous", rval = "bool", args = { } },
            { name = "is_emissive", rval = "bool", args = { } },
            { name = "is_firearm", rval = "bool", args = { } },
            { name = "is_food", rval = "bool", args = { } },
            { name = "is_food_container", rval = "bool", args = { } },
            { name = "is_going_bad", rval = "bool", args = { } },
            { name = "is_gun", rval = "bool", args = { } },
            { name = "is_gunmod", rval = "bool", args = { } },
            { name = "is_magazine", rval = "bool", args = { } },
            { name = "is_null", rval = "bool", args = { } },
            { name = "is_power_armor", rval = "bool", args = { } },
            { name = "is_salvageable", rval = "bool", args = { } },
            { name = "is_non_resealable_container", rval = "bool", args = { } },
            { name = "is_seed", rval = "bool", args = { } },
            { name = "is_sided", rval = "bool", args = { } },
            { name = "is_silent", rval = "bool", args = { } },
            { name = "is_tool", rval = "bool", args = { } },
            { name = "is_tool_reversible", rval = "bool", args = { } },
            { name = "is_two_handed", rval = "bool", args = { "player" } },
            { name = "is_watertight_container", rval = "bool", args = { } },
            { name = "is_melee", rval = "bool", args = { } },
            { name = "label", rval = "string", args = { } },
            { name = "label", rval = "string", args = { "int" } },
            { name = "charges_per_volume", rval = "int", args = { "volume" } },
            { name = "load_info", rval = nil, args = { "string" } },
            { name = "made_of", rval = "bool", args = { "phase_id" } },
            { name = "made_of", rval = "bool", args = { "material_id" } },
            { name = "magazine_current", rval = "item&", args = { } },
            { name = "magazine_current", rval = "item&", args = { } },
            { name = "magazine_default", rval = "string", args = { "bool" } },
            { name = "magazine_default", rval = "string", args = { } },
            { name = "magazine_integral", rval = "bool", args = { } },
            { name = "make_corpse", rval = nil, args = { "mtype_id", "int" } },
            { name = "make_corpse", rval = nil, args = { "mtype_id", "int", "string" } },
            { name = "make_corpse", rval = nil, args = { } },
            { name = "mark_as_used_by_player", rval = nil, args = { "player" } },
            { name = "mark_chapter_as_read", rval = nil, args = { "player" } },
            { name = "max_damage", rval = "int", args = { } },
            { name = "merge_charges", rval = "bool", args = { "item" } },
            { name = "min_damage", rval = "int", args = { } },
            { name = "mod_damage", rval = "bool", args = { "int" } },
            { name = "needs_processing", rval = "bool", args = { } },
            { name = "on_pickup", rval = nil, args = { "Character" } },
            { name = "on_takeoff", rval = nil, args = { "player" } },
            { name = "on_wear", rval = nil, args = { "player" } },
            { name = "on_wield", rval = nil, args = { "player" } },
            { name = "price", rval = "int", args = { "bool" } },
            { name = "process", rval = "bool", args = { "player", "tripoint", "bool" } },
            { name = "process_artifact", rval = nil, args = { "player", "tripoint" } },
            { name = "processing_speed", rval = "int", args = { } },
            { name = "put_in", rval = nil, args = { "item" } },
            { name = "ready_to_revive", rval = "bool", args = { "tripoint" } },
            { name = "mod_charges", rval = nil, args = { "int" } },
            { name = "reset_cable", rval = nil, args = { "player" } },
            { name = "rotten", rval = "bool", args = { } },
            { name = "set_mtype", rval = nil, args = { "mtype" } },
            { name = "set_relative_rot", rval = nil, args = { "float" } },
            { name = "set_snippet", rval = nil, args = { "string" } },
            { name = "set_var", rval = nil, args = { "string", "float" } },
            { name = "set_var", rval = nil, args = { "string", "int" } },
            { name = "set_var", rval = nil, args = { "string", "int" } },
            { name = "set_var", rval = nil, args = { "string", "string" } },
            { name = "sight_dispersion", rval = "int", args = { } },
            { name = "stacks_with", rval = "bool", args = { "item" } },
            { name = "symbol", rval = "string", args = { } },
            { name = "tname", rval = "string", args = { "int" } },
            { name = "tname", rval = "string", args = { "int", "bool" } },
            { name = "tname", rval = "string", args = { } },
            { name = "typeId", rval = "string", args = { } },
            { name = "type_name", rval = "string", args = { "int" } },
            { name = "type_name", rval = "string", args = { } },
            { name = "volume", rval = "volume", args = { } },
            { name = "melee_skill", rval = "skill_id", args = { } },
            { name = "weight", rval = "mass", args = { } },
        }
    },
    item_location = {
        attributes = {
        },
        by_value = true,
        functions = {
            { name = "describe", rval = "string", args = { "Character" } },
            { name = "describe", rval = "string", args = { } },
            { name = "get_item", rval = "item&", args = { } },
            { name = "obtain", rval = "int", args = { "Character" } },
            { name = "obtain", rval = "int", args = { "Character", "int" } },
            { name = "obtain_cost", rval = "int", args = { "Character" } },
            { name = "obtain_cost", rval = "int", args = { "Character", "int" } },
            { name = "remove_item", rval = nil, args = { } },
        }
    },
    point = {
        by_value = true,
        has_equal = true,
        attributes = {
            x = {
                type = "int",
                writable = true
            },
            y = {
                type = "int",
                writable = true
            }
        },
        new = {
            { "point" },
            { "int", "int" },
        },
        functions = {
        }
    },
    tripoint = {
        by_value = true,
        has_equal = true,
        attributes = {
            x = {
                type = "int",
                writable = true
            },
            y = {
                type = "int",
                writable = true
            },
            z = {
                type = "int",
                writable = true
            }
        },
        new = {
            { "tripoint" },
            { "int", "int", "int" },
        },
        functions = {
        }
    },
    uimenu = {
        attributes = {
            title = {
                type = "string",
                writable = true
            },
            selected = {
                type = "int",
                writable = false
            }
        },
        functions = {
            { name = "show", rval = nil, args = { } },
            { name = "query", rval = nil, args = { "bool" } },
            { name = "addentry", rval = nil, args = { "string" } },
        }
    },
    field_entry = {
        attributes = {
        },
        functions = {
            { name = "getFieldAge", rval = "int", args = { } },
            { name = "getFieldDensity", rval = "int", args = { } },
            { name = "getFieldType", rval = "field_id", args = { } },
            { name = "isAlive", rval = "bool", args = { } },
            { name = "is_dangerous", rval = "bool", args = { } },
            { name = "move_cost", rval = "int", args = { } },
            { name = "name", rval = "string", args = { } },
            { name = "setFieldAge", rval = "int", args = { "int" } },
            { name = "setFieldDensity", rval = "int", args = { "int" } },
            { name = "setFieldType", rval = "field_id", args = { "field_id" } },
        }
    },
    field = {
        attributes = {
        },
        functions = {
            { name = "addField", rval = "bool", args = { "field_id" } },
            { name = "addField", rval = "bool", args = { "field_id", "int" } },
            { name = "addField", rval = "bool", args = { "field_id", "int", "int" } },
            { name = "fieldCount", rval = "int", args = { } },
            { name = "fieldSymbol", rval = "field_id", args = { } },
            { name = "findField", rval = "field_entry&", args = { "field_id" } },
            { name = "findField", rval = "field_entry&", args = { "field_id" } },
            { name = "move_cost", rval = "int", args = { } },
            { name = "removeField", rval = "bool", args = { "field_id" } },
        }
    },
    map = {
        attributes = {
        },
        functions = {
            { name = "accessible_furniture", rval = "bool", args = { "tripoint", "tripoint", "int" } },
            { name = "accessible_items", rval = "bool", args = { "tripoint", "tripoint", "int" } },
            { name = "add_camp", rval = nil, args = { "tripoint", "string" } },
            { name = "add_corpse", rval = nil, args = { "tripoint" } },
            { name = "add_field", rval = "bool", args = { "tripoint", "field_id", "int", "int" } },
            { name = "add_item", rval = "item&", args = { "tripoint", "item" } },
            { name = "add_item_or_charges", rval = "item&", args = { "tripoint", "item" } },
            { name = "add_item_or_charges", rval = "item&", args = { "tripoint", "item", "bool" } },
            { name = "add_spawn", rval = nil, args = { "mtype_id", "int", "int", "int" } },
            { name = "add_spawn", rval = nil, args = { "mtype_id", "int", "int", "int", "bool" } },
            { name = "add_spawn", rval = nil, args = { "mtype_id", "int", "int", "int", "bool", "int" } },
            { name = "add_spawn", rval = nil, args = { "mtype_id", "int", "int", "int", "bool", "int", "int" } },
            { name = "add_spawn", rval = nil, args = { "mtype_id", "int", "int", "int", "bool", "int", "int", "string" } },
            { name = "adjust_field_age", rval = "int", args = { "tripoint", "field_id", "int" } },
            { name = "adjust_field_strength", rval = "int", args = { "tripoint", "field_id", "int" } },
            { name = "adjust_radiation", rval = nil, args = { "tripoint", "int" } },
            { name = "allow_camp", rval = "bool", args = { "tripoint", "int" } },
            { name = "ambient_light_at", rval = "float", args = { "tripoint" } },
            { name = "bash_rating", rval = "int", args = { "int", "tripoint" } },
            { name = "bash_rating", rval = "int", args = { "int", "tripoint", "bool" } },
            { name = "bash_resistance", rval = "int", args = { "tripoint" } },
            { name = "bash_resistance", rval = "int", args = { "tripoint", "bool" } },
            { name = "bash_strength", rval = "int", args = { "tripoint" } },
            { name = "bash_strength", rval = "int", args = { "tripoint", "bool" } },
            { name = "board_vehicle", rval = nil, args = { "tripoint", "player" } },
            { name = "can_move_furniture", rval = "bool", args = { "tripoint" } },
            { name = "can_move_furniture", rval = "bool", args = { "tripoint", "player" } },
            { name = "can_put_items", rval = "bool", args = { "tripoint" } },
            { name = "clear_path", rval = "bool", args = { "tripoint", "tripoint", "int", "int", "int" } },
            { name = "clear_spawns", rval = nil, args = { } },
            { name = "clear_traps", rval = nil, args = { } },
            { name = "climb_difficulty", rval = "int", args = { "tripoint" } },
            { name = "close_door", rval = "bool", args = { "tripoint", "bool", "bool" } },
            { name = "collapse_at", rval = nil, args = { "tripoint", "bool" } },
            { name = "collapse_check", rval = "int", args = { "tripoint" } },
            { name = "combined_movecost", rval = "int", args = { "tripoint", "tripoint" } },
            { name = "coord_to_angle", rval = "int", args = { "int", "int", "int", "int" } },
            { name = "could_see_items", rval = "bool", args = { "tripoint", "Creature" } },
            { name = "creature_in_field", rval = nil, args = { "Creature" } },
            { name = "creature_on_trap", rval = nil, args = { "Creature" } },
            { name = "creature_on_trap", rval = nil, args = { "Creature", "bool" } },
            { name = "crush", rval = nil, args = { "tripoint" } },
            { name = "decay_fields_and_scent", rval = nil, args = { "int" } },
            { name = "delete_graffiti", rval = nil, args = { "tripoint" } },
            { name = "delete_signage", rval = nil, args = { "tripoint" } },
            { name = "destroy", rval = nil, args = { "tripoint" } },
            { name = "destroy", rval = nil, args = { "tripoint", "bool" } },
            { name = "destroy_furn", rval = nil, args = { "tripoint" } },
            { name = "destroy_furn", rval = nil, args = { "tripoint", "bool" } },
            { name = "disarm_trap", rval = nil, args = { "tripoint" } },
            { name = "disp_name", rval = "string", args = { "tripoint" } },
            { name = "displace_vehicle", rval = nil, args = { "tripoint", "tripoint" } },
            { name = "displace_water", rval = "bool", args = { "tripoint" } },
            { name = "draw_circle_furn", rval = nil, args = { "furn_id", "int", "int", "int" } },
            { name = "draw_circle_furn", rval = nil, args = { "furn_str_id", "int", "int", "int" } },
            { name = "draw_circle_ter", rval = nil, args = { "ter_str_id", "int", "int", "int" } },
            { name = "draw_circle_ter", rval = nil, args = { "ter_id", "float", "float", "float" } },
            { name = "draw_circle_ter", rval = nil, args = { "ter_id", "int", "int", "int" } },
            { name = "draw_fill_background", rval = nil, args = { "ter_id" } },
            { name = "draw_fill_background", rval = nil, args = { "ter_str_id" } },
            { name = "draw_line_furn", rval = nil, args = { "furn_id", "int", "int", "int", "int" } },
            { name = "draw_line_furn", rval = nil, args = { "furn_str_id", "int", "int", "int", "int" } },
            { name = "draw_line_ter", rval = nil, args = { "ter_id", "int", "int", "int", "int" } },
            { name = "draw_line_ter", rval = nil, args = { "ter_str_id", "int", "int", "int", "int" } },
            { name = "draw_rough_circle_ter", rval = nil, args = { "ter_id", "int", "int", "int" } },
            { name = "draw_rough_circle_ter", rval = nil, args = { "ter_str_id", "int", "int", "int" } },
            { name = "draw_rough_circle_furn", rval = nil, args = { "furn_id", "int", "int", "int" } },
            { name = "draw_rough_circle_furn", rval = nil, args = { "furn_str_id", "int", "int", "int" } },
            { name = "draw_square_furn", rval = nil, args = { "furn_id", "int", "int", "int", "int" } },
            { name = "draw_square_furn", rval = nil, args = { "furn_str_id", "int", "int", "int", "int" } },
            { name = "draw_square_ter", rval = nil, args = { "ter_id", "int", "int", "int", "int" } },
            { name = "draw_square_ter", rval = nil, args = { "ter_str_id", "int", "int", "int", "int" } },
            { name = "features", rval = "string", args = { "tripoint" } },
            { name = "field_at", rval = "field&", args = { "tripoint" } },
            { name = "flammable_items_at", rval = "bool", args = { "tripoint" } },
            { name = "free_volume", rval = "volume", args = { "tripoint" } },
            { name = "furn", rval = "furn_id", args = { "tripoint" } },
            { name = "furn_set", rval = nil, args = { "tripoint", "furn_id" } },
            { name = "furn_set", rval = nil, args = { "tripoint", "furn_str_id" } },
            { name = "furnname", rval = "string", args = { "tripoint" } },
            { name = "get_abs_sub", rval = "tripoint", args = { } },
            { name = "get_field", rval = "field_entry&", args = { "tripoint", "field_id" } },
            { name = "get_field_age", rval = "int", args = { "tripoint", "field_id" } },
            { name = "get_field_strength", rval = "int", args = { "tripoint", "field_id" } },
            { name = "get_radiation", rval = "int", args = { "tripoint" } },
            { name = "get_signage", rval = "string", args = { "tripoint" } },
            { name = "get_ter_transforms_into", rval = "ter_id", args = { "tripoint" } },
            { name = "getabs", rval = "tripoint", args = { "tripoint" } },
            { name = "getlocal", rval = "tripoint", args = { "tripoint" } },
            { name = "getmapsize", rval = "int", args = { } },
            { name = "graffiti_at", rval = "string", args = { "tripoint" } },
            { name = "has_adjacent_furniture", rval = "bool", args = { "tripoint" } },
            { name = "has_flag", rval = "bool", args = { "string", "tripoint" } },
            { name = "has_flag_furn", rval = "bool", args = { "string", "tripoint" } },
            { name = "has_flag_ter", rval = "bool", args = { "string", "tripoint" } },
            { name = "has_flag_ter_or_furn", rval = "bool", args = { "string", "tripoint" } },
            { name = "has_floor", rval = "bool", args = { "tripoint" } },
            { name = "has_floor_or_support", rval = "bool", args = { "tripoint" } },
            { name = "has_furn", rval = "bool", args = { "tripoint" } },
            { name = "has_graffiti_at", rval = "bool", args = { "tripoint" } },
            { name = "has_items", rval = "bool", args = { "tripoint" } },
            { name = "has_nearby_fire", rval = "bool", args = { "tripoint" } },
            { name = "has_nearby_fire", rval = "bool", args = { "tripoint", "int" } },
            { name = "has_zlevels", rval = "bool", args = { } },
            { name = "hit_with_acid", rval = "bool", args = { "tripoint" } },
            { name = "hit_with_fire", rval = "bool", args = { "tripoint" } },
            { name = "i_at", rval = "map_stack", args = { "tripoint" } },
            { name = "i_clear", rval = nil, args = { "tripoint" } },
            { name = "i_rem", rval = "int", args = { "tripoint", "int" } },
            { name = "i_rem", rval = "item_stack_iterator", args = { "tripoint", "item_stack_iterator" } },
            { name = "i_rem", rval = nil, args = { "tripoint", "item" } },
            { name = "impassable", rval = "bool", args = { "tripoint" } },
            { name = "impassable_ter_furn", rval = "bool", args = { "tripoint" } },
            { name = "inbounds", rval = "bool", args = { "int", "int", "int" } },
            { name = "inbounds", rval = "bool", args = { "tripoint" } },
            { name = "is_bashable", rval = "bool", args = { "tripoint" } },
            { name = "is_bashable", rval = "bool", args = { "tripoint", "bool" } },
            { name = "is_bashable_furn", rval = "bool", args = { "tripoint" } },
            { name = "is_bashable_ter", rval = "bool", args = { "tripoint" } },
            { name = "is_bashable_ter", rval = "bool", args = { "tripoint", "bool" } },
            { name = "is_bashable_ter_furn", rval = "bool", args = { "tripoint" } },
            { name = "is_bashable_ter_furn", rval = "bool", args = { "tripoint", "bool" } },
            { name = "is_divable", rval = "bool", args = { "tripoint" } },
            { name = "is_outside", rval = "bool", args = { "tripoint" } },
            { name = "item_from", rval = "item&", args = { "tripoint", "int" } },
            { name = "light_transparency", rval = "float", args = { "tripoint" } },
            { name = "load", rval = nil, args = { "int", "int", "int", "bool" } },
            { name = "make_rubble", rval = nil, args = { "tripoint" } },
            { name = "make_rubble", rval = nil, args = { "tripoint", "furn_id", "bool" } },
            { name = "make_rubble", rval = nil, args = { "tripoint", "furn_id", "bool", "ter_id" } },
            { name = "make_rubble", rval = nil, args = { "tripoint", "furn_id", "bool", "ter_id", "bool" } },
            { name = "max_volume", rval = "volume", args = { "tripoint" } },
            { name = "mop_spills", rval = nil, args = { "tripoint" } },
            { name = "move_cost", rval = "int", args = { "tripoint" } },
            { name = "move_cost_ter_furn", rval = "int", args = { "tripoint" } },
            { name = "name", rval = "string", args = { "tripoint" } },
            { name = "on_vehicle_moved", rval = nil, args = { "int" } },
            { name = "open_door", rval = "bool", args = { "tripoint", "bool" } },
            { name = "open_door", rval = "bool", args = { "tripoint", "bool", "bool" } },
            { name = "passable", rval = "bool", args = { "tripoint" } },
            { name = "passable_ter_furn", rval = "bool", args = { "tripoint" } },
            { name = "pl_sees", rval = "bool", args = { "tripoint", "int" } },
            { name = "place_gas_pump", rval = nil, args = { "int", "int", "int" } },
            { name = "place_gas_pump", rval = nil, args = { "int", "int", "int", "string" } },
            { name = "place_npc", rval = "int", args = { "int", "int", "npc_template_id" } },
            { name = "place_spawns", rval = nil, args = { "mongroup_id", "int", "int", "int", "int", "int", "float" } },
            { name = "place_toilet", rval = nil, args = { "int", "int" } },
            { name = "place_toilet", rval = nil, args = { "int", "int", "int" } },
            { name = "place_vending", rval = nil, args = { "int", "int", "string" } },
            { name = "put_items_from_loc", rval = nil, args = { "string", "tripoint" } },
            { name = "put_items_from_loc", rval = nil, args = { "string", "tripoint", "int" } },
            { name = "random_outdoor_tile", rval = "point", args = { } },
            { name = "remove_field", rval = nil, args = { "tripoint", "field_id" } },
            { name = "remove_trap", rval = nil, args = { "tripoint" } },
            { name = "save", rval = nil, args = { } },
            { name = "sees", rval = "bool", args = { "tripoint", "tripoint", "int" } },
            { name = "sees_some_items", rval = "bool", args = { "tripoint", "Creature" } },
            { name = "set_field_age", rval = "int", args = { "tripoint", "field_id", "int" } },
            { name = "set_field_age", rval = "int", args = { "tripoint", "field_id", "int", "bool" } },
            { name = "set_field_strength", rval = "int", args = { "tripoint", "field_id", "int" } },
            { name = "set_field_strength", rval = "int", args = { "tripoint", "field_id", "int", "bool" } },
            { name = "set_graffiti", rval = nil, args = { "tripoint", "string" } },
            { name = "set_radiation", rval = nil, args = { "tripoint", "int" } },
            { name = "set_signage", rval = nil, args = { "tripoint", "string" } },
            { name = "set_temperature", rval = nil, args = { "tripoint", "int" } },
            { name = "smash_items", rval = nil, args = { "tripoint", "int" } },
            { name = "spawn_an_item", rval = "item&", args = { "tripoint", "item", "int", "int" } },
            { name = "spawn_artifact", rval = nil, args = { "tripoint" } },
            { name = "spawn_item", rval = nil, args = { "tripoint", "string" } },
            { name = "spawn_item", rval = nil, args = { "tripoint", "string", "int" } },
            { name = "spawn_item", rval = nil, args = { "tripoint", "string", "int", "int" } },
            { name = "spawn_item", rval = nil, args = { "tripoint", "string", "int", "int", "int" } },
            { name = "spawn_item", rval = nil, args = { "tripoint", "string", "int", "int", "int", "int" } },
            { name = "spawn_monsters", rval = nil, args = { "bool" } },
            { name = "stored_volume", rval = "volume", args = { "tripoint" } },
            { name = "supports_above", rval = "bool", args = { "tripoint" } },
            { name = "ter", rval = "ter_id", args = { "tripoint" } },
            { name = "ter_set", rval = nil, args = { "tripoint", "ter_id" } },
            { name = "ter_set", rval = nil, args = { "tripoint", "ter_str_id" } },
            { name = "tername", rval = "string", args = { "tripoint" } },
            { name = "tr_at", rval = "trap&", args = { "tripoint" } },
            { name = "trans", rval = "bool", args = { "tripoint" } },
            { name = "translate", rval = nil, args = { "ter_id", "ter_id" } },
            { name = "trap_set", rval = nil, args = { "tripoint", "trap_id" } },
            { name = "translate_radius", rval = nil, args = { "ter_id", "ter_id", "float", "tripoint" } },
            { name = "trigger_rc_items", rval = nil, args = { "string" } },
            { name = "unboard_vehicle", rval = nil, args = { "tripoint" } },
            { name = "valid_move", rval = "bool", args = { "tripoint", "tripoint" } },
            { name = "valid_move", rval = "bool", args = { "tripoint", "tripoint", "bool" } },
            { name = "valid_move", rval = "bool", args = { "tripoint", "tripoint", "bool", "bool" } },
            { name = "water_from", rval = "item", args = { "tripoint" } },
        }
    },
    ter_t = {
        int_id = "ter_id",
        string_id = "ter_str_id",

        attributes = {
            close = { type = "ter_str_id", writable = true },
            open = { type = "ter_str_id", writable = true },
            roof = { type = "ter_str_id", writable = true },
            transforms_into = { type = "ter_str_id", writable = true },
            trap = { type = "trap_id", writable = true },
            id = { type = "ter_str_id" },
            movecost = { type = "int", writable = true },
        },
        functions = {
            { name = "color", rval = "nc_color", args = { } },
            { name = "name", rval = "string", args = { } },
            { name = "symbol", rval = "int", args = { } },
        }
    },
    furn_t = {
        int_id = "furn_id",
        string_id = "furn_str_id",

        attributes = {
            close = { type = "furn_str_id", writable = true },
            id = { type = "furn_str_id" },
            max_volume = { type = "volume", writable = true },
            movecost = { type = "int", writable = true },
            open = { type = "furn_str_id", writable = true },
            transparent = { type = "bool", writable = true },
            move_str_req = { type = "int", writable = true },
        },
        functions = {
            { name = "color", rval = "nc_color", args = { } },
            { name = "name", rval = "string", args = { } },
            { name = "symbol", rval = "int", args = { } },
        }
    },
    Creature = {
        attributes = {
        },
        functions = {
            { name = "add_effect", rval = nil, args = { "efftype_id", "int" } },
            { name = "add_effect", rval = nil, args = { "efftype_id", "int", "body_part" } },
            { name = "add_effect", rval = nil, args = { "efftype_id", "int", "body_part", "bool" } },
            { name = "add_effect", rval = nil, args = { "efftype_id", "int", "body_part", "bool", "int" } },
            { name = "add_effect", rval = nil, args = { "efftype_id", "int", "body_part", "bool", "int", "bool" } },
            { name = "add_env_effect", rval = "bool", args = { "efftype_id", "body_part", "int", "int" } },
            { name = "add_env_effect", rval = "bool", args = { "efftype_id", "body_part", "int", "int", "body_part" } },
            { name = "add_env_effect", rval = "bool", args = { "efftype_id", "body_part", "int", "int", "body_part", "bool" } },
            { name = "add_env_effect", rval = "bool", args = { "efftype_id", "body_part", "int", "int", "body_part", "bool", "int" } },
            { name = "add_env_effect", rval = "bool", args = { "efftype_id", "body_part", "int", "int", "body_part", "bool", "int", "bool" } },
            { name = "apply_damage", rval = nil, args = { "Creature", "body_part", "int" } },
            { name = "avoid_trap", rval = "bool", args = { "tripoint", "trap" } },
            { name = "basic_symbol_color", rval = "nc_color", args = { } },
            { name = "bloodType", rval = "field_id", args = { } },
            { name = "check_dead_state", rval = nil, args = { } },
            { name = "clear_effects", rval = nil, args = { } },
            { name = "deal_melee_attack", rval = "int", args = { "Creature", "int" } },
            { name = "die", rval = nil, args = { "Creature" } },
            { name = "digging", rval = "bool", args = { } },
            { name = "disp_name", rval = "string", args = { "bool" } },
            { name = "disp_name", rval = "string", args = { } },
            { name = "dodge_roll", rval = "float", args = { } },
            { name = "fall_damage_mod", rval = "float", args = { } },
            { name = "get_armor_bash", rval = "int", args = { "body_part" } },
            { name = "get_armor_bash_base", rval = "int", args = { "body_part" } },
            { name = "get_armor_bash_bonus", rval = "int", args = { } },
            { name = "get_armor_cut", rval = "int", args = { "body_part" } },
            { name = "get_armor_cut_base", rval = "int", args = { "body_part" } },
            { name = "get_armor_cut_bonus", rval = "int", args = { } },
            { name = "get_bash_bonus", rval = "int", args = { } },
            { name = "get_bash_mult", rval = "float", args = { } },
            { name = "get_block_bonus", rval = "int", args = { } },
            { name = "get_cut_bonus", rval = "int", args = { } },
            { name = "get_cut_mult", rval = "float", args = { } },
            { name = "get_dodge", rval = "float", args = { } },
            { name = "get_dodge_base", rval = "float", args = { } },
            { name = "get_dodge_bonus", rval = "float", args = { } },
            { name = "get_effect_dur", rval = "int", args = { "efftype_id" } },
            { name = "get_effect_dur", rval = "int", args = { "efftype_id", "body_part" } },
            { name = "get_effect_int", rval = "int", args = { "efftype_id" } },
            { name = "get_effect_int", rval = "int", args = { "efftype_id", "body_part" } },
            { name = "get_env_resist", rval = "int", args = { "body_part" } },
            { name = "get_grab_resist", rval = "int", args = { } },
            { name = "get_hit", rval = "float", args = { } },
            { name = "get_hit_base", rval = "float", args = { } },
            { name = "get_hit_bonus", rval = "float", args = { } },
            { name = "get_hp", rval = "int", args = { "hp_part" } },
            { name = "get_hp", rval = "int", args = { } },
            { name = "get_hp_max", rval = "int", args = { "hp_part" } },
            { name = "get_hp_max", rval = "int", args = { } },
            { name = "get_killer", rval = "Creature&", args = { } },
            { name = "get_melee", rval = "int", args = { } },
            { name = "get_melee_quiet", rval = "bool", args = { } },
            { name = "get_name", rval = "string", args = { } },
            { name = "get_num_blocks", rval = "int", args = { } },
            { name = "get_num_blocks_bonus", rval = "int", args = { } },
            { name = "get_num_dodges", rval = "int", args = { } },
            { name = "get_num_dodges_bonus", rval = "int", args = { } },
            { name = "get_random_body_part", rval = "body_part", args = { "bool" } },
            { name = "get_random_body_part", rval = "body_part", args = { } },
            { name = "get_size", rval = "m_size", args = { } },
            { name = "get_speed", rval = "int", args = { } },
            { name = "get_speed_base", rval = "int", args = { } },
            { name = "get_speed_bonus", rval = "int", args = { } },
            { name = "get_throw_resist", rval = "int", args = { } },
            { name = "get_value", rval = "string", args = { "string" } },
            { name = "get_weight", rval = "mass", args = { } },
            { name = "gibType", rval = "field_id", args = { } },
            { name = "has_effect", rval = "bool", args = { "efftype_id" } },
            { name = "has_effect", rval = "bool", args = { "efftype_id", "body_part" } },
            { name = "has_grab_break_tec", rval = "bool", args = { } },
            { name = "has_trait", rval = "bool", args = { "trait_id" } },
            { name = "has_weapon", rval = "bool", args = { } },
            { name = "hit_roll", rval = "float", args = { } },
            { name = "hp_percentage", rval = "int", args = { } },
            { name = "impact", rval = "int", args = { "int", "tripoint" } },
            { name = "in_sleep_state", rval = "bool", args = { } },
            { name = "is_dead_state", rval = "bool", args = { } },
            { name = "is_elec_immune", rval = "bool", args = { } },
            { name = "is_fake", rval = "bool", args = { } },
            { name = "is_hallucination", rval = "bool", args = { } },
            { name = "is_immune_effect", rval = "bool", args = { "efftype_id" } },
            { name = "is_monster", rval = "bool", args = { } },
            { name = "is_npc", rval = "bool", args = { } },
            { name = "is_on_ground", rval = "bool", args = { } },
            { name = "is_player", rval = "bool", args = { } },
            { name = "is_symbol_highlighted", rval = "bool", args = { } },
            { name = "is_underwater", rval = "bool", args = { } },
            { name = "is_warm", rval = "bool", args = { } },
            { name = "knock_back_from", rval = nil, args = { "tripoint" } },
            { name = "melee_attack", rval = nil, args = { "Creature", "bool" } },
            { name = "melee_attack", rval = nil, args = { "Creature", "bool", "matec_id" } },
            { name = "mod_bash_bonus", rval = nil, args = { "int" } },
            { name = "mod_block_bonus", rval = nil, args = { "int" } },
            { name = "mod_cut_bonus", rval = nil, args = { "int" } },
            { name = "mod_dodge_bonus", rval = nil, args = { "float" } },
            { name = "mod_hit_bonus", rval = nil, args = { "float" } },
            { name = "mod_moves", rval = nil, args = { "int" } },
            { name = "mod_pain", rval = nil, args = { "int" } },
            { name = "mod_pain_noresist", rval = nil, args = { "int" } },
            { name = "set_pain", rval = nil, args = { "int" } },
            { name = "get_pain", rval = "int", args = { } },
            { name = "get_perceived_pain", rval = "int", args = { } },
            { name = "mod_speed_bonus", rval = nil, args = { "int" } },
            { name = "mod_stat", rval = nil, args = { "string", "int" } },
            { name = "move_effects", rval = "bool", args = { "bool" } },
            { name = "normalize", rval = nil, args = { } },
            { name = "on_dodge", rval = nil, args = { "Creature", "float" } },
            { name = "on_hit", rval = nil, args = { "Creature" } },
            { name = "on_hit", rval = nil, args = { "Creature", "body_part" } },
            { name = "on_hit", rval = nil, args = { "Creature", "body_part", "float" } },
            { name = "pos", rval = "tripoint", args = { } },
            { name = "posx", rval = "int", args = { } },
            { name = "posy", rval = "int", args = { } },
            { name = "posz", rval = "int", args = { } },
            { name = "power_rating", rval = "float", args = { } },
            { name = "process_effects", rval = nil, args = { } },
            { name = "process_turn", rval = nil, args = { } },
            { name = "remove_effect", rval = "bool", args = { "efftype_id" } },
            { name = "remove_effect", rval = "bool", args = { "efftype_id", "body_part" } },
            { name = "remove_value", rval = nil, args = { "string" } },
            { name = "reset", rval = nil, args = { } },
            { name = "reset_bonuses", rval = nil, args = { } },
            { name = "reset_stats", rval = nil, args = { } },
            { name = "sees", rval = "bool", args = { "Creature" } },
            { name = "sees", rval = "bool", args = { "tripoint" } },
            { name = "sees", rval = "bool", args = { "tripoint", "bool" } },
            { name = "set_armor_bash_bonus", rval = nil, args = { "int" } },
            { name = "set_armor_cut_bonus", rval = nil, args = { "int" } },
            { name = "set_bash_bonus", rval = nil, args = { "int" } },
            { name = "set_bash_mult", rval = nil, args = { "float" } },
            { name = "set_block_bonus", rval = nil, args = { "int" } },
            { name = "set_cut_bonus", rval = nil, args = { "int" } },
            { name = "set_cut_mult", rval = nil, args = { "float" } },
            { name = "set_dodge_bonus", rval = nil, args = { "float" } },
            { name = "set_fake", rval = nil, args = { "bool" } },
            { name = "set_grab_resist", rval = nil, args = { "int" } },
            { name = "set_hit_bonus", rval = nil, args = { "float" } },
            { name = "set_melee_quiet", rval = nil, args = { "bool" } },
            { name = "set_moves", rval = nil, args = { "int" } },
            { name = "set_num_blocks_bonus", rval = nil, args = { "int" } },
            { name = "set_num_dodges_bonus", rval = nil, args = { "int" } },
            { name = "set_speed_base", rval = nil, args = { "int" } },
            { name = "set_speed_bonus", rval = nil, args = { "int" } },
            { name = "set_throw_resist", rval = nil, args = { "int" } },
            { name = "set_value", rval = nil, args = { "string", "string" } },
            { name = "setpos", rval = nil, args = { "tripoint" } },
            { name = "sight_range", rval = "int", args = { "int" } },
            { name = "skin_name", rval = "string", args = { } },
            { name = "speed_rating", rval = "float", args = { } },
            { name = "stability_roll", rval = "int", args = { } },
            { name = "symbol", rval = "string", args = { } },
            { name = "symbol_color", rval = "nc_color", args = { } },
            { name = "uncanny_dodge", rval = "bool", args = { } },
            { name = "weight_capacity", rval = "mass", args = { } },
        }
    },
    monster = {
        parent = "Creature",
        attributes = {
            anger = { type = "int", writable = true },
            friendly = { type = "int", writable = true },
            hallucination = { type = "bool", writable = true },
            ignoring = { type = "int", writable = true },
            last_updated = { type = "int", writable = true },
            made_footstep = { type = "bool", writable = true },
            mission_id = { type = "int", writable = true },
            morale = { type = "int", writable = true },
            no_corpse_quiet = { type = "bool", writable = true },
            no_extra_death_drops = { type = "bool", writable = true },
            staircount = { type = "int", writable = true },
            type = { type = "mtype" },
            unique_name = { type = "string", writable = true },
            wander_pos = { type = "tripoint", writable = true },
            wandf = { type = "int", writable = true },
        },
        functions = {
            { name = "add_effect", rval = nil, args = { "efftype_id", "int" } },
            { name = "add_effect", rval = nil, args = { "efftype_id", "int", "body_part" } },
            { name = "add_effect", rval = nil, args = { "efftype_id", "int", "body_part", "bool" } },
            { name = "add_effect", rval = nil, args = { "efftype_id", "int", "body_part", "bool", "int" } },
            { name = "add_effect", rval = nil, args = { "efftype_id", "int", "body_part", "bool", "int", "bool" } },
            { name = "add_item", rval = nil, args = { "item" } },
            { name = "apply_damage", rval = nil, args = { "Creature", "body_part", "int" } },
            { name = "attack_at", rval = "bool", args = { "tripoint" } },
            { name = "attack_target", rval = "Creature&", args = { } },
            { name = "avoid_trap", rval = "bool", args = { "tripoint", "trap" } },
            { name = "bash_at", rval = "bool", args = { "tripoint" } },
            { name = "bash_estimate", rval = "int", args = { } },
            { name = "bash_skill", rval = "int", args = { } },
            { name = "basic_symbol_color", rval = "nc_color", args = { } },
            { name = "bloodType", rval = "field_id", args = { } },
            { name = "calc_climb_cost", rval = "int", args = { "tripoint", "tripoint" } },
            { name = "calc_movecost", rval = "int", args = { "tripoint", "tripoint" } },
            { name = "can_act", rval = "bool", args = { } },
            { name = "can_drown", rval = "bool", args = { } },
            { name = "can_hear", rval = "bool", args = { } },
            { name = "can_move_to", rval = "bool", args = { "tripoint" } },
            { name = "can_see", rval = "bool", args = { } },
            { name = "can_submerge", rval = "bool", args = { } },
            { name = "can_upgrade", rval = "bool", args = { } },
            { name = "color_with_effects", rval = "nc_color", args = { } },
            { name = "deal_melee_attack", rval = "int", args = { "Creature", "int" } },
            { name = "die", rval = nil, args = { "Creature" } },
            { name = "die_in_explosion", rval = nil, args = { "Creature" } },
            { name = "digging", rval = "bool", args = { } },
            { name = "disp_name", rval = "string", args = { "bool" } },
            { name = "disp_name", rval = "string", args = { } },
            { name = "dodge_roll", rval = "float", args = { } },
            { name = "drop_items_on_death", rval = nil, args = { } },
            { name = "explode", rval = nil, args = { } },
            { name = "fall_damage_mod", rval = "float", args = { } },
            { name = "footsteps", rval = nil, args = { "tripoint" } },
            { name = "get_armor_bash", rval = "int", args = { "body_part" } },
            { name = "get_armor_cut", rval = "int", args = { "body_part" } },
            { name = "get_dodge", rval = "float", args = { } },
            { name = "get_hp", rval = "int", args = { "hp_part" } },
            { name = "get_hp", rval = "int", args = { } },
            { name = "get_hp_max", rval = "int", args = { "hp_part" } },
            { name = "get_hp_max", rval = "int", args = { } },
            { name = "get_melee", rval = "float", args = { } },
            { name = "get_random_body_part", rval = "body_part", args = { "bool" } },
            { name = "get_size", rval = "m_size", args = { } },
            { name = "gibType", rval = "field_id", args = { } },
            { name = "group_bash_skill", rval = "int", args = { "tripoint" } },
            { name = "has_grab_break_tec", rval = "bool", args = { } },
            { name = "has_weapon", rval = "bool", args = { } },
            { name = "heal", rval = "int", args = { "int" } },
            { name = "heal", rval = "int", args = { "int", "bool" } },
            { name = "hear_sound", rval = nil, args = { "tripoint", "int", "int" } },
            { name = "hit_roll", rval = "float", args = { } },
            { name = "hp_percentage", rval = "int", args = { } },
            { name = "impact", rval = "int", args = { "int", "tripoint" } },
            { name = "init_from_item", rval = nil, args = { "item" } },
            { name = "is_dead", rval = "bool", args = { } },
            { name = "is_dead_state", rval = "bool", args = { } },
            { name = "is_elec_immune", rval = "bool", args = { } },
            { name = "is_fleeing", rval = "bool", args = { "player" } },
            { name = "is_hallucination", rval = "bool", args = { } },
            { name = "is_immune_effect", rval = "bool", args = { "efftype_id" } },
            { name = "is_monster", rval = "bool", args = { } },
            { name = "is_on_ground", rval = "bool", args = { } },
            { name = "is_symbol_highlighted", rval = "bool", args = { } },
            { name = "is_underwater", rval = "bool", args = { } },
            { name = "is_warm", rval = "bool", args = { } },
            { name = "knock_back_from", rval = nil, args = { "tripoint" } },
            { name = "made_of", rval = "bool", args = { "phase_id" } },
            { name = "made_of", rval = "bool", args = { "material_id" } },
            { name = "make_ally", rval = nil, args = { "monster" } },
            { name = "make_friendly", rval = nil, args = { } },
            { name = "make_fungus", rval = "bool", args = { } },
            { name = "melee_attack", rval = nil, args = { "Creature", "bool", "matec_id" } },
            { name = "move", rval = nil, args = { } },
            { name = "move_effects", rval = "bool", args = { "bool" } },
            { name = "move_target", rval = "tripoint", args = { } },
            { name = "move_to", rval = "bool", args = { "tripoint" } },
            { name = "move_to", rval = "bool", args = { "tripoint", "bool" } },
            { name = "move_to", rval = "bool", args = { "tripoint", "bool", "float" } },
            { name = "name", rval = "string", args = { "int" } },
            { name = "name", rval = "string", args = { } },
            { name = "name_with_armor", rval = "string", args = { } },
            { name = "on_dodge", rval = nil, args = { "Creature", "float" } },
            { name = "on_hit", rval = nil, args = { "Creature" } },
            { name = "on_hit", rval = nil, args = { "Creature", "body_part" } },
            { name = "on_hit", rval = nil, args = { "Creature", "body_part", "float" } },
            { name = "on_load", rval = nil, args = { } },
            { name = "on_unload", rval = nil, args = { } },
            { name = "poly", rval = nil, args = { "mtype_id" } },
            { name = "pos", rval = "tripoint", args = { } },
            { name = "posx", rval = "int", args = { } },
            { name = "posy", rval = "int", args = { } },
            { name = "posz", rval = "int", args = { } },
            { name = "power_rating", rval = "float", args = { } },
            { name = "process_effects", rval = nil, args = { } },
            { name = "process_triggers", rval = nil, args = { } },
            { name = "push_to", rval = "bool", args = { "tripoint", "int", "int" } },
            { name = "rate_target", rval = "float", args = { "Creature", "float" } },
            { name = "rate_target", rval = "float", args = { "Creature", "float", "bool" } },
            { name = "reset_special", rval = nil, args = { "string" } },
            { name = "reset_special_rng", rval = nil, args = { "string" } },
            { name = "scent_move", rval = "tripoint", args = { } },
            { name = "set_dest", rval = nil, args = { "tripoint" } },
            { name = "set_hp", rval = nil, args = { "int" } },
            { name = "set_special", rval = nil, args = { "string", "int" } },
            { name = "disable_special", rval = nil, args = { "string" } },
            { name = "setpos", rval = nil, args = { "tripoint" } },
            { name = "shift", rval = nil, args = { "int", "int" } },
            { name = "sight_range", rval = "int", args = { "int" } },
            { name = "skin_name", rval = "string", args = { } },
            { name = "spawn", rval = nil, args = { "tripoint" } },
            { name = "speed_rating", rval = "float", args = { } },
            { name = "stability_roll", rval = "int", args = { } },
            { name = "stumble", rval = nil, args = { } },
            { name = "symbol", rval = "string", args = { } },
            { name = "symbol_color", rval = "nc_color", args = { } },
            { name = "to_item", rval = "item", args = { } },
            { name = "turns_to_reach", rval = "int", args = { "int", "int" } },
            { name = "try_upgrade", rval = nil, args = { "bool" } },
            { name = "hasten_upgrade", rval = nil, args = { } },
            { name = "unset_dest", rval = nil, args = { } },
            { name = "wander", rval = "bool", args = { } },
            { name = "wander_to", rval = nil, args = { "tripoint", "int" } },
            { name = "will_reach", rval = "bool", args = { "int", "int" } },
        }
    },
    recipe = {
        string_id = "recipe_id",
        attributes = {
        },
        functions = {
        }
    },
    martialart = {
        string_id = "matype_id",
        attributes = {
        },
        functions = {
        }
    },
    material_type = {
        string_id = "material_id",
        attributes = {
        },
        functions = {
        }
    },
    start_location = {
        string_id = "start_location_id",
        attributes = {
        },
        functions = {
        }
    },
    ma_buff = {
        string_id = "mabuff_id",
        attributes = {
        },
        functions = {
        }
    },
    ma_technique = {
        string_id = "matec_id",
        attributes = {
        },
        functions = {
        }
    },
    Skill = {
        string_id = "skill_id",
        attributes = {
        },
        functions = {
        }
    },
    quality = {
        string_id = "quality_id",
        attributes = {
            name = { type = "string", writable = false }
        },
        functions = {
        }
    },
    npc_template = {
        string_id = "npc_template_id",
        attributes = {
        },
        functions = {
        }
    },
    species_type = {
        string_id = "species_id",
        attributes = {
        },
        functions = {
        }
    },
    ammunition_type = {
        string_id = "ammotype",
        attributes = {
        },
        functions = {
        }
    },
    MonsterGroup = {
        string_id = "mongroup_id",
        attributes = {
            defaultMonster = { type = "mtype_id", writable = true },
            is_safe = { type = "bool", writable = true },
            monster_group_time = { type = "int", writable = true },
            name = { type = "mongroup_id", writable = true },
            new_monster_group = { type = "mongroup_id", writable = true },
            replace_monster_group = { type = "bool", writable = true },
        },
        functions = {
            { name = "IsMonsterInGroup", rval = "bool", args = { "mtype_id" } },
        }
    },
    mtype = {
        string_id = "mtype_id",
        attributes = {
            agro = { type = "int", writable = true },
            armor_acid = { type = "int", writable = true },
            armor_bash = { type = "int", writable = true },
            armor_cut = { type = "int", writable = true },
            armor_fire = { type = "int", writable = true },
            armor_stab = { type = "int", writable = true },
            attack_cost = { type = "int", writable = true },
            color = { type = "nc_color", writable = true },
            death_drops = { type = "string", writable = true },
            def_chance = { type = "int", writable = true },
            difficulty = { type = "int", writable = true },
            hp = { type = "int", writable = true },
            id = { type = "mtype_id" },
            luminance = { type = "float", writable = true },
            melee_dice = { type = "int", writable = true },
            melee_sides = { type = "int", writable = true },
            melee_skill = { type = "int", writable = true },
            morale = { type = "int", writable = true },
            phase = { type = "phase_id", writable = true },
            revert_to_itype = { type = "string", writable = true },
            size = { type = "m_size", writable = true },
            sk_dodge = { type = "int", writable = true },
            speed = { type = "int", writable = true },
            sym = { type = "string", writable = true },
            upgrade_group = { type = "mongroup_id", writable = true },
            upgrades = { type = "bool", writable = true },
            half_life = { type = "int", writable = true },
            upgrade_into = { type = "mtype_id", writable = true },
            vision_day = { type = "int", writable = true },
            vision_night = { type = "int", writable = true },
        },
        functions = {
            { name = "bloodType", rval = "field_id", args = { } },
            { name = "get_meat_itype", rval = "string", args = { } },
            { name = "gibType", rval = "field_id", args = { } },
            { name = "has_flag", rval = "bool", args = { "string" } },
            { name = "made_of", rval = "bool", args = { "material_id" } },
            { name = "has_special_attack", rval = "bool", args = { "string" } },
            { name = "in_category", rval = "bool", args = { "string" } },
            { name = "in_species", rval = "bool", args = { "species_id" } },
            { name = "nname", rval = "string", args = { "int" } },
            { name = "nname", rval = "string", args = { } },
            { name = "same_species", rval = "bool", args = { "mtype" } },
            { name = "set_flag", rval = nil, args = { "string", "bool" } },
        }
    },
    mongroup = {
        attributes = {
            pos = { type = "tripoint", writable = false },
            target = { type = "tripoint", writable = false },
            dying = { type = "bool", writable = true },
            horde = { type = "bool", writable = true },
            diffuse = { type = "bool", writable = true },
            radius = { type = "int", writable = true },
            population = { type = "int", writable = true }
        },
        functions = {
            { name = "set_target", rval = nil, args = { "int", "int" } },
            { name = "inc_interest", rval = nil, args = { "int" } },
            { name = "dec_interest", rval = nil, args = { "int" } },
            { name = "set_interest", rval = nil, args = { "int" } },
        }
    },
    overmap = {
        attributes = {
        },
        functions = {
            { name = "add_note", rval = nil, args = { "int", "int", "int", "string" } },
            { name = "clear_mon_groups", rval = nil, args = { } },
            { name = "delete_note", rval = nil, args = { "int", "int", "int" } },
            { name = "find_random_omt", rval = "tripoint", args = { "string" } },
            { name = "has_note", rval = "bool", args = { "int", "int", "int" } },
            { name = "is_explored", rval = "bool", args = { "int", "int", "int" } },
            { name = "note", rval = "string", args = { "int", "int", "int" } },
        }
    },
    volume = {
        by_value = true,
        attributes = {
        },
        functions = {
            { name = "value", rval = "int", args = { } },
        },
    },
    mass = {
        by_value = true,
        attributes = {
        },
        functions = {
            { name = "value", rval = "int", args = { } },
        },
    },
    nc_color = {
        by_value = true,
        attributes = {
        },
        functions = {
        },
    },
    time_duration = {
        by_value = true,
        attributes = {
        },
        functions = {
        },
    },
    time_point = {
        by_value = true,
        attributes = {
        },
        functions = {
        },
    },
    itype = {
        attributes = {
            color = { type = "nc_color", writable = true },
            default_container = { type = "string", writable = true },
            description = { type = "string", writable = true },
            explode_in_fire = { type = "bool", writable = true },
            integral_volume = { type = "volume", writable = true },
            light_emission = { type = "int", writable = true },
            m_to_hit = { type = "int", writable = true },
            magazine_well = { type = "volume", writable = true },
            min_dex = { type = "int", writable = true },
            min_int = { type = "int", writable = true },
            min_per = { type = "int", writable = true },
            min_str = { type = "int", writable = true },
            phase = { type = "phase_id", writable = true },
            price = { type = "int", writable = true },
            price_post = { type = "int", writable = true },
            rigid = { type = "bool", writable = true },
            snippet_category = { type = "string", writable = true },
            stack_size = { type = "int", writable = true },
            sym = { type = "string", writable = true },
            volume = { type = "volume", writable = true },
            weight = { type = "mass", writable = true },
        },
        functions = {
            { name = "can_use", args = { "string" } },
            { name = "charges_default", rval = "int", args = { } },
            { name = "charges_to_use", rval = "int", args = { } },
            { name = "count_by_charges", rval = "bool", args = { } },
            { name = "get_item_type_string", rval = "string", args = { } },
            { name = "has_use", rval = "bool", args = { } },
            { name = "invoke", rval = "int", args = { "player", "item", "tripoint" } },
            { name = "invoke", rval = "int", args = { "player", "item", "tripoint", "string" } },
            { name = "maximum_charges", rval = "int", args = { } },
            { name = "nname", rval = "string", args = { "int" } },
            { name = "tick", rval = "int", args = { "player", "item", "tripoint" } },
        }
    },
    trap = {
        int_id = "trap_id",
        string_id = "trap_str_id",
        attributes = {
            id = { type = "trap_str_id" },
            loadid = { type = "trap_id" },
            color = { type = "nc_color" },
            sym = { type = "int" },
        },
        functions = {
            { name = "can_see", rval = "bool", args = { "tripoint", "player" } },
            { name = "detect_trap", rval = "bool", args = { "tripoint", "player" } },
            { name = "funnel_turns_per_charge", rval = "float", args = { "float" } },
            { name = "get_avoidance", rval = "int", args = { } },
            { name = "get_difficulty", rval = "int", args = { } },
            { name = "get_visibility", rval = "int", args = { } },
            { name = "is_3x3_trap", rval = "bool", args = { } },
            { name = "is_benign", rval = "bool", args = { } },
            { name = "is_funnel", rval = "bool", args = { } },
            { name = "is_null", rval = "bool", args = { } },
            { name = "on_disarmed", rval = nil, args = { "map", "tripoint" } },
            { name = "trigger", rval = nil, args = { "tripoint", "Creature" } },
            { name = "triggered_by_item", rval = "bool", args = { "item" } },
        }
    },
    w_point = {
        attributes = {
            temperature = { type = "float" },
            humidity = { type = "float" },
            pressure = { type = "float" },
            windpower = { type = "float" },
            acidic = { type = "bool" }
        },
        functions = {
        }
    }
}

enums = {
    body_part = {
        "bp_torso",
        "bp_head",
        "bp_eyes",
        "bp_mouth",
        "bp_arm_l",
        "bp_arm_r",
        "bp_hand_l",
        "bp_hand_r",
        "bp_leg_l",
        "bp_leg_r",
        "bp_foot_l",
        "bp_foot_r",
        "num_bp",
    },
    hp_part = {
        "hp_head",
        "hp_torso",
        "hp_arm_l",
        "hp_arm_r",
        "hp_leg_l",
        "hp_leg_r",
        "num_hp_parts",
    },
    side = {
        "side::BOTH",
        "side::LEFT",
        "side::RIGHT"
    },
    phase_id = {
        "PNULL",
        "SOLID",
        "LIQUID",
        "GAS",
        "PLASMA",
    },
    m_size = {
        "MS_TINY",
        "MS_SMALL",
        "MS_MEDIUM",
        "MS_LARGE",
        "MS_HUGE",
    },
    game_message_type = {
        "m_good",
        "m_bad",
        "m_mixed",
        "m_warning",
        "m_info",
        "m_neutral",
        "m_debug",
        "m_headshot",
        "m_critical",
        "m_grazing",
    },
    season_type = {
        "SPRING",
        "SUMMER",
        "AUTUMN",
        "WINTER",
    },
    add_type = {
        "ADD_NULL",
        "ADD_CAFFEINE",
        "ADD_ALCOHOL",
        "ADD_SLEEP",
        "ADD_PKILLER",
        "ADD_SPEED",
        "ADD_CIG",
        "ADD_COKE",
        "ADD_CRACK",
        "ADD_MUTAGEN",
        "ADD_DIAZEPAM",
        "ADD_MARLOSS_R",
        "ADD_MARLOSS_B",
        "ADD_MARLOSS_Y",
    },
    field_id = {
        "fd_null",
        "fd_blood",
        "fd_bile",
        "fd_gibs_flesh",
        "fd_gibs_veggy",
        "fd_web",
        "fd_slime",
        "fd_acid",
        "fd_sap",
        "fd_sludge",
        "fd_fire",
        "fd_rubble",
        "fd_smoke",
        "fd_toxic_gas",
        "fd_tear_gas",
        "fd_nuke_gas",
        "fd_gas_vent",
        "fd_fire_vent",
        "fd_flame_burst",
        "fd_electricity",
        "fd_fatigue",
        "fd_push_items",
        "fd_shock_vent",
        "fd_acid_vent",
        "fd_plasma",
        "fd_laser",
        "fd_spotlight",
        "fd_dazzling",
        "fd_blood_veggy",
        "fd_blood_insect",
        "fd_blood_invertebrate",
        "fd_gibs_insect",
        "fd_gibs_invertebrate",
        "fd_cigsmoke",
        "fd_weedsmoke",
        "fd_cracksmoke",
        "fd_methsmoke",
        "fd_bees",
        "fd_incendiary",
        "fd_relax_gas",
        "fd_fungal_haze",
        "fd_hot_air1",
        "fd_hot_air2",
        "fd_hot_air3",
        "fd_hot_air4",
        "fd_fungicidal_gas",
        "num_fields",
    }
}

global_functions = {
    add_msg = {
        cpp_name = "add_msg_wrapper",
        args     = { "string" },
        argnames = { "message" },
        rval = nil,
        desc = "Write a message to the game's standard message window."
    },
    popup = {
        cpp_name = "popup_wrapper",
        args = { "string" },
        rval = nil
    },
    string_input_popup = {
        cpp_name = "string_input_popup_wrapper",
        args = { "string", "int", "string" },
        rval = "string"
    },
    create_uimenu = {
        cpp_name = "create_uimenu",
        args = {},
        rval = "uimenu&"
    },
    get_terrain_type = {
        cpp_name = "get_terrain_type",
        args = {"int"},
        rval = "ter_t&"
    },
    rng = {
        cpp_name = "rng",
        args = {"int", "int"},
        rval = "int"
    },
    one_in = {
        cpp_name = "one_in",
        args = {"int"},
        rval = "bool"
    },
    distance = {
        cpp_name = "rl_dist",
        args = {"int", "int", "int", "int"},
        rval = "int"
    },
    trig_dist = {
        cpp_name = "trig_dist",
        args = {"int", "int", "int", "int"},
        rval = "int"
    },
    add_item_to_group = {
        cpp_name = "item_controller->add_item_to_group",
        args = { "string", "string", "int" },
        rval = "bool"
    },
    create_monster = {
        cpp_name = "create_monster",
        args = { "mtype_id", "tripoint" },
        rval = "monster&",
        desc = "Creates and spawns a new monster of given type. Returns a refernce to it, *or* nil if it could not be spawned."
    },
    get_calendar_turn = {
        cpp_name = "get_calendar_turn_wrapper",
        args = {},
        rval = "calendar&"
    }
}

--[[
Currently, for each function in each class, the bindings generator creates a matching function in
each of the parent classes. For example is_player, which is defined in Creature and player.
Because it's in Creature and monster inherits it from there, the generator adds a wrapper to the
metatable of the monster class that calls is_player on a monster object.
However, it creates the same "wrapper" for the player class (same scenario: it inherits it from
Creature).
Now player::is_player is wrapped twice: once through the heritage from Creature and once on its own.

The following snippet tries to correct this. It simply removes all the redundantly declared
functions in the subclasses.
--]]

for class_name, value in pairs(classes) do
    -- Collect all defined functions of the *parent* classes in this table
    local existing = { };
    value = classes[value.parent]
    while value do
        for _, func in ipairs(value.functions) do
            local n = func.name .. "_" .. table.concat(func.args, "|")
            existing[n] = true
        end
        value = classes[value.parent]
    end
    -- Now back to the actual class, remove all the functions that are in the table
    -- and therefor exist in at least on of the parent classes.
    value = classes[class_name]
    local i = 1
    while i <= #value.functions do
        local func = value.functions[i]
        local n = func.name .. "_" .. table.concat(func.args, "|")
        if existing[n] then
            table.remove(value.functions, i)
        else
            i = i + 1
        end
    end
end

-- This adds the int_id wrappers from the class definition as real classes.
-- All int_id<T>s have the same interface, so we only need to add some mark to T, that this class
-- T has an int_id of some name.
-- In the class definition: add "int_id" = "XXX" (XXX is the typedef id that is used by C++).
new_classes = {}
for name, value in pairs(classes) do
    if value.int_id then
        -- This is the common int_id<T> interface:
        local t = {
            by_value = true,
            has_equal = true,
            -- IDs *could* be constructed from int, but where does the Lua script get the int from?
            -- The int is only exposed as int_id<T>, so Lua should never know about it.
            attributes = { },
            -- Copy and default constructor
            new = { { value.int_id }, { } },
            functions = {
                -- Use with care, only for displaying the value for debugging purpose!
                { name = "to_i", rval = "int", args = { } },
                { name = "obj", rval = name .. "&", args = { } },
            }
        }
        if value.string_id then
            -- Allow conversion from int_id to string_id
            t[#t.functions] = { name = "id", rval = value.string_id, args = { } }
            -- And creation of an int_id from a string_id
            t.new = { { value.string_id }, { } }
        end
        new_classes[value.int_id] = t
    end
    -- Very similar to int_id above
    if value.string_id then
        local t = {
            by_value = true,
            has_equal = true,
            -- Copy and default constructor and construct from plain string.
            new = { { value.string_id }, { }, { "string" } },
            attributes = { },
            functions = {
                { name = "str", rval = "string", args = { } },
                { name = "is_valid", rval = "bool", args = { } },
                { name = "obj", rval = name .. "&", args = { } },
            }
        }
        if value.int_id then
            t.functions[#t.functions] = { name = "id", rval = value.int_id, args = { } }
        end
        new_classes[value.string_id] = t
    end
end
for name, value in pairs(new_classes) do
    classes[name] = value
end
new_classes = nil
