// $Id: DBug.java,v 1.4 1997/11/08 16:12:33 xzhu Exp $
// DBug.java
//
// Debug running time support package
// 
// Author: Xiaokun Kelvin ZHU
// Address: kelvin@iname.com
// 1997, 10, 12
// Version 1.1

// Copyright (C) 1997, X.K.ZHU
//   The `DBug.java' is free software and comes with NO WARRANTY of any
//   kind; you can redistribute it and/or modify it under the terms of the 
//   GNU Library General Public License as published by the Free Software 
//   Foundation; either version 2 of the License, or (at your option) any 
//   later version.

/**
  DBug is a java bean.
  see testDBug.java and testDBugMT.java for usage
 */
/******************************************************************************
 *									      *
 *	                           N O T I C E				      *
 *									      *
 *	              Copyright Abandoned, 1987, Fred Fish		      *
 *									      *
 *									      *
 *	This previously copyrighted work has been placed into the  public     *
 *	domain  by  the  author  and  may be freely used for any purpose,     *
 *	private or commercial.						      *
 *									      *
 *	Because of the number of inquiries I was receiving about the  use     *
 *	of this product in commercially developed works I have decided to     *
 *	simply make it public domain to further its unrestricted use.   I     *
 *	specifically  would  be  most happy to see this material become a     *
 *	part of the standard Unix distributions by AT&T and the  Berkeley     *
 *	Computer  Science  Research Group, and a standard part of the GNU     *
 *	system from the Free Software Foundation.			      *
 *									      *
 *	I would appreciate it, as a courtesy, if this notice is  left  in     *
 *	all copies and derivative works.  Thank you.			      *
 *									      *
 *	The author makes no warranty of any kind  with  respect  to  this     *
 *	product  and  explicitly disclaims any implied warranties of mer-     *
 *	chantability or fitness for any particular purpose.		      *
 *									      *
 ******************************************************************************
 */

/*	dbug.c   runtime support routines for dbug package
 *
 *  SCCS
 *
 *	@(#)dbug.c	1.25	7/25/89
 *
 *  DESCRIPTION
 *
 *	These are the runtime support routines for the dbug package.
 *	The dbug package has two main components; the user include
 *	file containing various macro definitions, and the runtime
 *	support routines which are called from the macro expansions.
 *
 *	Externally visible functions in the runtime support module
 *	use the naming convention pattern "_db_xx...xx_", thus
 *	they are unlikely to collide with user defined function names.
 *
 *  AUTHOR(S)
 *
 *	Fred Fish		(base code)
 *	Enhanced Software Technologies, Tempe, AZ
 *	asuvax!mcdphx!estinc!fnf
 *
 *	Binayak Banerjee	(profiling enhancements)
 *	seismo!bpa!sjuvax!bbanerje
 *
 *	Michael Widenius:
 *      DBUG_DUMP		- To dump a pice of memory.
 *	PUSH_FLAG "O"	- To be used insted of "o" if we don't
 *			  want flushing (for slow systems)
 *      Check of malloc on entry/exit (option "S")
 */
package zxk.util;

import java.io.*;
import java.io.Serializable;
import java.util.*;

public class DBug implements Serializable
{
  class DBugState implements Serializable
  {
    /*
     *	Debugging states can be pushed or popped off of a
     *	stack which is implemented as a linked list.  Note
     *	that the head of the list is the current state and the
     *	stack is pushed by adding a new state to the head of the
     *	list or popped by removing the first link.
     */
  
    int flags;		/* Current state flags */
    int maxdepth;		/* Current maximum trace depth */
    long delay;		/* Delay after each output line */
    int sub_level;	/* Sub this from code_state->level */
    PrintStream out_file;	/* Current output stream */
    String name;	        /* Name of output file */
    Vector functions;	/* List of functions */
    Vector keywords;	/* List of debug keywords */
    Vector keyAttr;	/* List of debug keyword attribute */
    Vector classes;	/* List of class names */
    Vector threads;	/* List of thread names */
    Vector threadAttr;	/* List of thread names */
  
    /**
     *  Default Constuctor
     *
     *	PushState    push current state onto stack and set up new one
     *
     *  DESCRIPTION
     *
     *  Pushes the current state on the state stack, and initializes
     *	a new state.  The only parameter inherited from the previous
     *	state is the function nesting level.  This action can be
     *  inhibited if desired, via the "r" flag.
     *
     *	The state stack is a linked list of states, with the new
     *	state added at the head.  This allows the stack to grow
     *	to the limits of memory if necessary.
     */
    DBugState ()
    {
      flags = 0;
      delay = 0;
      maxdepth = MAXDEPTH;
      sub_level=0;
      out_file = null;
      functions = null;
      keywords = null;
      keyAttr = new Vector();
      classes = null;
      threads = null;
      threadAttr = new Vector();
    }

    void
    getAttribute(Vector list, Vector attr)
    {
      int size = list.size();
      for (int i=0; i<size; ++i)
      {
        String obj = (String)list.elementAt(i);
	int j = obj.indexOf('*');
	if (j == -1)
	  attr.addElement(new Boolean(false));
	else
	{
	  attr.addElement(new Boolean(true));
	  obj = obj.substring(0,j);
	  list.removeElementAt(i);
	  list.insertElementAt(obj, i);
	}
      }
    }
  }

  private static final int MAXDEPTH = 200; // Maximum trace depth default
  private static final int INDENT  = 2;   /* Indentation per trace level */
  /*	The following flags are used to determine which
   *	capabilities the user has enabled with the state
   *	push macro.
   */
  private static final int TRACE_ON=0x0001; // Trace enabled
  private static final int DEBUG_ON=0x0002; // Debug enabled
  private static final int FILE_ON =0x0004; // File name print enabled
  private static final int LINE_ON =0x0008; // Line number print enabled
  private static final int DEPTH_ON=0x0010; // Function nest level print enabled
  private static final int THREAD_ON=0x0020; // Process name print enabled
  private static final int NUMBER_ON=0x0040; // Number each line of output
  private static final int TIME_ON = 0x0080; // Current Time (ms)
  private static final int CLASS_ON= 0x0100; // Classes on
  private static final int FLUSH_ON_WRITE=0x0200; // Flush on every write

  private int lineno; /* Current debugger output line number */
  // private int level;  /* Current function nesting level */
  // private String func;/* Name of current user function */
  // private String dbClass;/* Name of current user class */

  private DBugState dbugState;
  private Stack stack; // DBugState object stack
  private Hashtable frameHash; // frame hash table for Thread--DBugFrame

  /*
   *	Variables which are available externally but should only
   *	be accessed via the macro package facilities.
   */
  private PrintStream dbFp; /* Output stream, default stderr */
  private long startTime;

  private DBugFrame dbFrame; // temp var

  public static DBug dbug; // singleton pattern

  // class init
  static 
  {
    String dbugStr = System.getProperty("DBug.debug");
    if (dbugStr == null)
      dbugStr = new String("");
    dbug = new DBug();
    dbug.dbPush(dbugStr);
  }

  /** 
   * Default constructor
   */
  private 
  DBug()
  {
    lineno = 0;
    // level = 0;
    //func = new String("?func");
    //dbClass = new String("?class");

    dbFp = System.err;	/* Output stream, default stderr */
    // dbThread = Thread.currentThread().getName(); 
    stack = new Stack();

    frameHash = new Hashtable();
    DBugFrame dbF = new DBugFrame(new String("?func"), new String("?class"), 0);
    frameHash.put(Thread.currentThread(), dbF);

    dbugState = null;
    startTime = System.currentTimeMillis();
  }

  /**
   *	dbPush	push current debugger state and set up new one
   *
   *  DESCRIPTION
   *
   *	Given pointer to a debug control string in "control", pushes
   *	the current debug state, parses the control string, and sets
   *	up a new debug state.
   *
   *	The only attribute of the new state inherited from the previous
   *	state is the current function nesting level.  This can be
   *	overridden by using the "r" flag in the control string.
   *
   *	The debug control string is a sequence of colon separated fields
   *	as follows:
   *
   *		<field_1>:<field_2>:...:<field_N>
   *
   *	Each field consists of a mandatory flag character followed by
   *	an optional "," and comma separated list of modifiers:
   *
   *		flag[,modifier,modifier,...,modifier]
   *
   *	The currently recognized flag characters are:
   *
   *		c	Limit debugger actions to specified classes.
   *			A class must be identified with the
   *			DBUG_CLASS macro and match one in the list
   *			for debugger actions to occur.
   *
   *		C	Identify the class name for each
   *			line of debug or trace output.
   *
   *		d	Enable output from DBUG_<N> macros for
   *			for the current state.  May be followed
   *			by a list of keywords which selects output
   *			only for the DBUG macros with that keyword.
   *			A null list of keywords implies output for
   *			all macros. "key*" will match all the keyword
   *			beginning with "key" such as "key1", "keya1", etc.
   *
   *		D	Delay after each debugger output line.
   *			The argument is the number of tenths of seconds
   *			to delay, subject to machine capabilities.
   *			I.E.  -#D,20 is delay two seconds.
   *
   *		f	Limit debugging and/or tracing to the
   *			list of named functions.  Note that a null list will
   *			disable all functions.  The appropriate "d" or "t"
   *			flags must still be given, this flag only limits their
   *			actions if they are enabled.
   *
   *		L	Identify the source file line number for
   *			each line of debug or trace output.
   *
   *		m	Print the current time (ms) for
   *			each line of debug or trace output.
   *
   *		n	Print the current function nesting depth for
   *			each line of debug or trace output.
   *
   *		N	Number each line of dbug output.
   *
   *		o	Redirect the debugger output stream to the
   *			specified file.  The default output is stderr.
   *
   *		O	As O but the file is really flushed between each
   *		        write. When neaded the file is closed and reopened
   *			between each write.
   *
   *		p	Limit debugger actions to specified threades.
   *			A thread must be identified with the
   *			DBUG_THREAD macro and match one int the list 
   *			for debugger actions to occur.
   *			"thread*" mactch "thread", "thread1", ...
   *
   *		P	Print the current thread name for each
   *			line of debug or trace output.
   *
   *		r	When pushing a new state, do not inherit
   *			the previous state's function nesting level.
   *			Useful when the output is to start at the
   *			left margin.
   *
   *		t	Enable function call/exit trace lines.
   *			May be followed by a list (containing only
   *			one modifier) giving a numeric maximum
   *			trace level, beyond which no output will
   *			occur for either debugging or tracing
   *			macros.  The default is a compile time
   *			option.
   *
   *	Some examples of debug control strings which might appear
   *	on a shell command line (the "-#" is typically used to
   *	introduce a control string to an application program) are:
   *
   *		-#                 disable all debug
   *		-#d:t
   *		-#d:f,main,subr1:F:L:t,20
   *		-#d,input,output,files:n
   *
   *	For convenience, any leading "-#" is stripped off.
   * @param control control string.
   *
   */

  public synchronized void 
  dbPush (String control)
  {
    if (control == null)
      return;

    if (control.startsWith("-#"))
      control = control.substring(2);

    dbugState = new DBugState();
    dbugState.out_file = dbFp; // add default fp
    stack.push(dbugState);
    dbFrame = (DBugFrame)frameHash.get(Thread.currentThread());
    
    StringTokenizer st = new StringTokenizer(control, ":");
    while (st.hasMoreTokens())
    {
      String scan = st.nextToken();
      switch (scan.charAt(0)) 
      {
        case 'c':
          if (scan.length() > 1 && scan.charAt(1) == ',')
	    dbugState.classes = listParse (scan);
          break;
        case 'C':
          dbugState.flags |= CLASS_ON;
          break;
        case 'd':
          dbugState.flags |= DEBUG_ON;
          if (scan.length() > 1 && scan.charAt(1) == ',')
	  {
	    dbugState.keywords = listParse (scan.substring(2));
	    dbugState.getAttribute(dbugState.keywords, dbugState.keyAttr);
	  }
          break;
        case 'D':
          dbugState.delay = 0;
          if (scan.length() > 1 && scan.charAt(1) == ',')
          {
	    int v = new Integer(scan.substring(2)).intValue();
	    dbugState.delay = v / 10;
          }
          break;
        case 'f':
          if (scan.length() > 1 && scan.charAt(1) == ',')
	    dbugState.functions = listParse (scan.substring(2));
          break;
        case 'L':
          dbugState.flags |= LINE_ON;
          break;
        case 'm':
          dbugState.flags |= TIME_ON;
          break;
        case 'n':
          dbugState.flags |= DEPTH_ON;
          break;
        case 'N':
          dbugState.flags |= NUMBER_ON;
          break;
        case 'O':
          dbugState.flags |= FLUSH_ON_WRITE;
        case 'o':
          if (scan.length() > 1 && scan.charAt(1) == ',')
	    dbugOpenFile (scan.substring(2));
          else
    	    dbugOpenFile ("-");
          break;
        case 'p':
          if (scan.length() > 1 && scan.charAt(1) == ',')
	  {
	    dbugState.threads = listParse (scan);
	    dbugState.getAttribute(dbugState.threads, dbugState.threadAttr);
	  }
          break;
        case 'P':
          dbugState.flags |= THREAD_ON;
          break;
        case 'r':
          dbugState.sub_level= dbFrame.getLevel();
          break;
        case 't':
          dbugState.flags |= TRACE_ON;
          if (scan.length() > 1 && scan.charAt(1) == ',')
	  {
	    int v = new Integer(scan.substring(2)).intValue();
	    dbugState.maxdepth = v;
            }
          break;
      }
    }
  }


  /**
   *	dbPop    pop the debug stack
   *
   *  DESCRIPTION
   *
   *	Pops the debug stack, returning the debug state to its
   *	condition prior to the most recent _db_push_ invocation.
   *	Note that the pop will fail if it would remove the last
   *	valid state from the stack.  This prevents user errors
   *	in the push/pop sequence from screwing up the debugger.
   *	Maybe there should be some kind of warning printed if the
   *	user tries to pop too many states.
   *
   */
  public synchronized void 
  dbPop ()
  {
    if (dbugState != null) {
      closeFile (dbugState.out_file);
      dbugState = (DBugState)stack.peek();
      dbFp = dbugState.out_file;
      stack.pop();
    }
  }


  /**
   *	dbEnter    process entry point to user function
   *
   *  DESCRIPTION
   *
   *	Called at the beginning of each user function to tell
   *	the debugger that a new function has been entered.
   *	Note that the pointers to the previous user function
   *	name and previous user file name are stored on the
   *	caller's stack (this is why the ENTER macro must be
   *	the first "executable" code in a function, since it
   *	allocates these storage locations).  The previous nesting
   *	level is also stored on the callers stack for internal
   *	self consistency checks.
   *
   *	Also prints a trace line if tracing is enabled and
   *	increments the current function nesting depth.
   *
   *	Note that this mechanism allows the debugger to know
   *	what the current user function is at all times, without
   *	maintaining an internal stack for the function names.
   *
   * @param _func_;		current function name
   * @param _class_;		current class name
   * @param _line_;		called from source line number
   * @return the old DBugFrame
   */

  public synchronized void 
  dbEnter (String _func_, String _class_, int _line_)
  {
    //++level;
    Thread thread = Thread.currentThread();
    dbFrame = (DBugFrame)frameHash.get(thread);
    if (dbFrame == null)
    {
      dbFrame = new DBugFrame(_func_, _class_, 1);
      frameHash.put(thread, dbFrame);
    }
    else
    {
      dbFrame.pushFuncName(_func_);
      dbFrame.pushClassName(_class_);
      dbFrame.pushLevel(dbFrame.getLevel()+1);
    }
    if (doTrace ())
    {
      doPrefix (_line_);
      indent (dbFrame.getLevel());
      dbFp.println(">" + _func_);
      dbugFlush ();				/* This does a unlock */
    }
  }

  /**
   *	dbReturn    process exit from user function
   *
   *  DESCRIPTION
   *
   *	Called just before user function executes an explicit or implicit
   *	return.  Prints a trace line if trace is enabled, decrements
   *	the current nesting level, and restores the current function and
   *	file names from the defunct function's stack.
   *
   * @param _line_ the trace line
   */
  public synchronized void
  dbReturn (int _line_)
  {
    if ((dbugState.flags & (TRACE_ON | DEBUG_ON)) > 0)
    {
      dbFrame = (DBugFrame)frameHash.get(Thread.currentThread());
      if (dbFrame == null)
      {
        System.out.println("no DBugFrame for Thread: " + Thread.currentThread().getName());
	return;
      }
      // if (level != dbFrame.getLevel())
// 	dbFp.print(Thread.currentThread().getName() + " missing DBUG_RETURN or DBUG_VOID_RETURN macro in function " + func);
//      else
      {
	if (doTrace ())
	{
	  doPrefix (_line_);
	  indent (dbFrame.getLevel());
	  dbFp.println("<" + dbFrame.getFuncName());
	}
      }
      dbugFlush();
    }
    dbFrame.popLevel();
    dbFrame.popFuncName();
    dbFrame.popClassName();
  }


  /**
   *	dbPrint    handle print of debug lines
   *
   *  DESCRIPTION
   *
   *	When invoked via one of the DBUG macros, tests the current keyword
   *	set by calling _db_pargs_() to see if that macro has been selected
   *	for processing via the debugger control string, and if so, handles
   *	printing of the arguments via the format string.  The line number
   *	of the DBUG macro in the source is found in u_line.
   *
   *	Note that the format string SHOULD NOT include a terminating
   *	newline, this is supplied automatically.
   *
   * @param _line_ 
   * @param keyword
   * @param message
   */
  public synchronized void 
  dbPrint (int _line_, String keyword, String message)
  {
    if (dbKeyword(keyword)) 
    {
      dbFrame = (DBugFrame)frameHash.get(Thread.currentThread());
      if (dbFrame == null)
      {
        System.out.println("no DBugFrame for Thread: " + Thread.currentThread().getName());
	return;
      }
      doPrefix (_line_);
      if ((dbugState.flags & TRACE_ON) > 0) {
        indent (dbFrame.getLevel() + 1);
      } 
      else 
        dbFp.print(dbFrame.getFuncName() + ": ");
      dbFp.print(keyword + ": ");
      dbFp.println(message);
      dbugFlush();
    }
  }

  /**
   *	doTrace    check to see if tracing is current enabled
   *
   *  SYNOPSIS
   *
   *	public boolean doTrace (stack)
   *
   *  DESCRIPTION
   *
   *	Checks to see if tracing is enabled based on whether the
   *	user has specified tracing, the maximum trace depth has
   *	not yet been reached, the current function is selected,
   *	and the current process is selected.  Returns TRUE if
   *	tracing is enabled, FALSE otherwise.
   *
   */

  private boolean 
  doTrace ()
  {
    boolean trace = false;

    if ((dbugState.flags & TRACE_ON) > 0 &&
        dbFrame.getLevel() <= dbugState.maxdepth &&
        inList (dbugState.functions, dbFrame.getFuncName()) &&
        inListAttr (dbugState.threads, dbugState.threadAttr, 
	            Thread.currentThread().getName()) &&
        inList (dbugState.classes, dbFrame.getClassName()))
      trace = true;
    return (trace);
  }


  /**
   *	dbKeyword    test keyword for member of keyword list
   *
   *  DESCRIPTION
   *
   *	Test a keyword to determine if it is in the currently active
   *	keyword list.  As with the function list, a keyword is accepted
   *	if the list is null, otherwise it must match one of the list
   *	members.  When debugging is not on, no keywords are accepted.
   *	After the maximum trace level is exceeded, no keywords are
   *	accepted (this behavior subject to change).  Additionally,
   *	the current function and process must be accepted based on
   *	their respective lists.
   *
   * @param keyword keyword list
   * @returns true if keyword accepted, FALSE otherwise.
   *
   */

  private boolean
  dbKeyword (String keyword)
  {
    if ((dbugState.flags & DEBUG_ON) > 0 &&
      dbFrame.getLevel() <= dbugState.maxdepth &&
      inListAttr (dbugState.keywords, dbugState.keyAttr, keyword) &&
      inList (dbugState.functions, dbFrame.getFuncName()) &&
      inListAttr (dbugState.threads, dbugState.threadAttr, 
                  Thread.currentThread().getName()) &&
      inList (dbugState.classes, dbFrame.getClassName()))
      {
      return true;
      }
    return false;
  }

  /**
   *	indent    indent a line to the given indentation level
   *
   *  DESCRIPTION
   *
   *	indent a line to the given level.  Note that this is
   *	a simple minded but portable implementation.
   *	There are better ways.
   *
   *	Also, the indent must be scaled by the compile time option
   *	of character positions per nesting level.
   *
   * @param indent the indent value
   */
  private void 
  indent (int indent)
  {
    indent -= (1 + dbugState.sub_level);
    if (indent > 0)
      indent *= INDENT;
    else 
      indent = 0;
    for (int count = 0; count<indent ; count++)
    {
      if ((count % INDENT) == 0)
        dbFp.print("|");
      else
        dbFp.print(" ");
    }
  }


  /*
   *	doPrefix    print debugger line prefix prior to indentation
   *
   *  DESCRIPTION
   *
   *	Print prefix common to all debugger output lines, prior to
   *	doing indentation if necessary.  Print such information as
   *	current process name, current source file name and line number,
   *	and current function nesting depth.
   *
   */
  private void 
  doPrefix (int _line_)
  {
    lineno++;
    if ((dbugState.flags & NUMBER_ON) > 0) 
      dbFp.print("" + lineno + ": ");
    if ((dbugState.flags & TIME_ON) > 0)
      dbFp.print("" + (System.currentTimeMillis()-startTime) + ": ");
    if ((dbugState.flags & CLASS_ON) > 0) 
      dbFp.print(dbFrame.getClassName() + ": ");
    if ((dbugState.flags & LINE_ON) > 0) 
      dbFp.print(_line_ + ": ");
    if ((dbugState.flags & DEPTH_ON) > 0) 
      dbFp.print(""+ dbFrame.getLevel() + ": ");
    if ((dbugState.flags & THREAD_ON) > 0)
      dbFp.print(Thread.currentThread().getName()+ ": ");
  }


  /**
   *	dbugOpenFile    open new output stream for debugger output
   *
   *  DESCRIPTION
   *
   *	Given name of a new file (or "-" for stdout) opens the file
   *	and sets the output stream to the new file.
   * @param name file name and "-" for System.out
   */
  private void 
  dbugOpenFile (String name)
  {
    if (name == null)
      return;

    dbugState.name = name;
    if (name.charAt(0) == '-')
    {
      dbFp = System.out;
      dbugState.out_file = dbFp;
      dbugState.flags |= FLUSH_ON_WRITE;
    }
    else
    {
      try
      {
        PrintStream out = new PrintStream( new BufferedOutputStream( 
	    new FileOutputStream(name)));
	dbFp = out;
	dbugState.out_file = out;
      }
      catch (IOException e)
      {
        System.err.println(dbFrame.getClassName()+ " can't open debug output stream: " + name);
	System.err.flush();
      }
    }
  }


  /**
   *	CloseFile    close the debug output stream
   *
   *  DESCRIPTION
   *
   *	Closes the debug output stream unless it is standard output
   *	or standard error.
   * @param fp PrintStream
   */
  private void 
  closeFile (PrintStream fp)
  {
    if (fp != System.err && fp != System.out) 
    {
      try
      {
        ((OutputStream)fp).close();
      }
      catch (IOException e)
      {
        dbFp.println(dbFrame.getClassName() + ": can't close debug file: ");
        dbugFlush();
      }
    }
  }


  /**
   * flush dbug-stream 
   */
  private void 
  dbugFlush()
  {
    if ((dbugState.flags & FLUSH_ON_WRITE) > 0)
    {
      dbFp.flush();
      try 
      {
        Thread.sleep(dbugState.delay);
      } catch (InterruptedException e) {}
    }
  }


  /**
   *	listParse    parse list of modifiers in debug control string
   *
   *  SYNOPSIS
   *
   *	public Vector listParse (String str)
   *
   *  DESCRIPTION
   *
   *	Given pointer to a comma separated list of strings in "str",
   *	parses the list, building a list and returning a pointer to it.
   *	The original comma separated list is destroyed in the process of
   *	building the linked list, thus it had better be a duplicate
   *	if it is important.
   *
   *	Note that since each link is added at the head of the list,
   *	the final list will be in "reverse order", which is not
   *	significant for our usage here.
   *
   * @param str input string
   * @return the list Vector
   */
  public Vector
  listParse(String str)
  {
    StringTokenizer st = new StringTokenizer(str, ",");
    Vector v = new Vector();
    while (st.hasMoreTokens())
    {
      String token = st.nextToken();
      v.addElement(token);
    }
    return v;
  }

  /**
   *	inList    test a given string for member of a given list
   *
   *  SYNOPSIS
   *
   *	private boolean inList (Vector link, String str)
   *
   *  DESCRIPTION
   *
   *	Tests the string pointed to by "cp" to determine if it is in
   *	the list pointed to by "linkp".  Linkp points to the first
   *	link in the list.  If linkp is NULL then the string is treated
   *	as if it is in the list (I.E all strings are in the null list).
   *	This may seem rather strange at first but leads to the desired
   *	operation if no list is given.  The net effect is that all
   *	strings will be accepted when there is no list, and when there
   *	is a list, only those strings in the list will be accepted.
   *
   * @param link the Vector
   * @param str the object
   * @return true if Vector contains the object
   */
  private boolean
  inList(Vector link, String str)
  {
    if (link == null || link.size() == 0 || str == null || str.length() == 0)
      return true;

    int size = link.size();
    for (int i=0; i<size; ++i)
    {
      String obj = (String)link.elementAt(i);
      if (obj.equals(str))
        return true;
    }
    return false;
  }

  private boolean
  inListAttr (Vector link, Vector linkAttr, String str)
  {
    if (link == null || link.size() == 0 || str == null || str.length() == 0)
      return true;

    int size = link.size();
    for (int i=0; i<size; ++i)
    {
      String obj = (String)link.elementAt(i);
      if ( ((Boolean)linkAttr.elementAt(i)).booleanValue() )
      {
        if (str.startsWith(obj))
          return true;
      }
      else
      {
        if (obj.equals(str))
          return true;
      }
    }
    return false;
  }
}
