/* Interface definition for Hexagon ISA support.
   Copyright 2012 Free Software Foundation, Inc.

   This file is part of BFD, the Binary File Descriptor library.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street - Fifth Floor, Boston, MA 02110-1301, 
   USA.  */

#include <assert.h>
#include <limits.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include "sysdep.h"
#include "ansidecl.h"
#include "bfd.h"
#include "safe-ctype.h"
#include "libiberty.h"

#include "hexagon-isa.h"

/* These opcodes are known to the assembler.  */
const hexagon_opcode *hexagon_nop_opcode, *hexagon_kext_opcode;

/* Various types of Hexagon operands.  */
static const hexagon_operand hexagon_operands[] = {
  {"Rs32", 0, 5, 's', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_READ),
   "r%u"},
  {"Rt32", 0, 5, 't', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_READ),
   "r%u"},
  {"Ru32", 0, 5, 'u', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_READ),
   "r%u"},
  {"Rd32", 0, 5, 'd', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_WRITE),
   "r%u"},
  {"Re32", 0, 5, 'e', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_WRITE),
   "r%u"},
  {"Rf32", 0, 5, 'f', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_MODIFIED),
   "r%u"},
  {"Rx32", 0, 5, 'x', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_READ
    | HEXAGON_OPERAND_IS_WRITE),
   "r%u"},
  { "Ry32", 0,       5, 'y', 0,
    BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
    (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_READ
     | HEXAGON_OPERAND_IS_WRITE), "r%u"},
  {"Rz32", 0, 5, 'z', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_READ
    | HEXAGON_OPERAND_IS_MODIFIED),
   "r%u"},

  {"Rss32", 0, 5, 's', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_READ),
   "r%u:%u"},
  {"Rtt32", 0, 5, 't', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_READ),
   "r%u:%u"},
  {"Rdd32", 0, 5, 'd', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_WRITE),
   "r%u:%u"},
  {"Rxx32", 0, 5, 'x', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_READ | HEXAGON_OPERAND_IS_WRITE),
   "r%u:%u"},
  {"Ryy32", 0, 5, 'y', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_READ | HEXAGON_OPERAND_IS_WRITE),
   "r%u:%u"},

  {"Rd16", 0, 4, 'd', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_WRITE),
   "r%u"},
  {"Re16", 0, 4, 'e', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_WRITE),
   "r%u"},
  {"Rs16", 0, 4, 's', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_READ),
   "r%u"},
  {"Rt16", 0, 4, 't', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_READ),
   "r%u"},
  {"Ru16", 0, 4, 'u', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_READ),
   "r%u"},
  {"Rv16", 0, 4, 'v', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_READ),
   "r%u"},
  {"Rx16", 0, 4, 'x', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_READ
    | HEXAGON_OPERAND_IS_WRITE),
   "r%u"},
  {"Ry16", 0, 4, 'y', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_READ
    | HEXAGON_OPERAND_IS_WRITE),
   "r%u"},

  {"Rdd16", 0, 4, 'd', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_WRITE),
   "r%u:%u"},

  {"Rd8", 0, 3, 'd', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_WRITE),
   "r%u"},

  {"Rdd8", 0, 3, 'd', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_WRITE),
   "r%u:%u"},
  {"Ree8", 0, 3, 'e', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_WRITE),
   "r%u:%u"},
  {"Rtt8", 0, 3, 't', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_READ),
   "r%u:%u"},
  {"Rvv8", 0, 3, 'v', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SUBSET | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_READ),
   "r%u:%u"},

  {"Ns8", 0, 3, 's', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_NEW | HEXAGON_OPERAND_IS_READ),
   "r%u"},
  {"Nt8", 0, 3, 't', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_NEW | HEXAGON_OPERAND_IS_READ),
   "r%u"},

  {"Mu2", 0, 1, 'u', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_MODIFIER | HEXAGON_OPERAND_IS_READ),
   "m%u"},

  {"Ps4", 0, 2, 's', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_PREDICATE | HEXAGON_OPERAND_IS_READ),
   "p%u"},
  {"Pt4", 0, 2, 't', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_PREDICATE | HEXAGON_OPERAND_IS_READ),
   "p%u"},
  {"Pu4", 0, 2, 'u', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_PREDICATE | HEXAGON_OPERAND_IS_READ),
   "p%u"},
  {"Pv4", 0, 2, 'v', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_PREDICATE | HEXAGON_OPERAND_IS_READ),
   "p%u"},
  {"Pd4", 0, 2, 'd', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_PREDICATE | HEXAGON_OPERAND_IS_WRITE),
   "p%u"},
  {"Pe4", 0, 2, 'e', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_PREDICATE | HEXAGON_OPERAND_IS_WRITE),
   "p%u"},
  {"Px4", 0, 2, 'x', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_PREDICATE | HEXAGON_OPERAND_IS_READ
    | HEXAGON_OPERAND_IS_WRITE),
   "p%u"},

  {"Cs32", 0, 5, 's', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_CONTROL | HEXAGON_OPERAND_IS_READ),
   ""},
  {"Cd32", 0, 5, 'd', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_CONTROL | HEXAGON_OPERAND_IS_WRITE),
   ""},

  {"Css32", 0, 5, 's', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_CONTROL | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_READ),
   "c%u:%u"},
  {"Cdd32", 0, 5, 'd', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_CONTROL | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_WRITE),
   "c%u:%u"},

  {"Gs32", 0, 5, 's', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_GUEST | HEXAGON_OPERAND_IS_READ),
   ""},
  {"Gd32", 0, 5, 'd', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_GUEST | HEXAGON_OPERAND_IS_WRITE),
   ""},

  {"Gss32", 0, 5, 's', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_GUEST | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_READ),
   "g%u:%u"},
  {"Gdd32", 0, 5, 'd', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_GUEST | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_WRITE),
   "g%u:%u"},

  {"Ss64", 0, 6, 's', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SYSTEM | HEXAGON_OPERAND_IS_READ),
   ""},
  {"Sd64", 0, 6, 'd', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SYSTEM | HEXAGON_OPERAND_IS_WRITE),
   ""},

  {"Sss64", 0, 6, 's', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SYSTEM | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_READ),
   "s%u:%u"},
  {"Sdd64", 0, 6, 'd', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_SYSTEM | HEXAGON_OPERAND_IS_PAIR
    | HEXAGON_OPERAND_IS_WRITE),
   "s%u:%u"},

  {"#g16:0", 0, 16, 'i', 0,
   BFD_RELOC_HEX_GPREL16_0, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_GPREL_OR_ABS),
   "#%u"},
  {"#g16:0ie", 6, 16, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_IE_32_6_X, BFD_RELOC_HEX_IE_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_GPREL_OR_ABS),
   "#%u"},
  {"#g16:1", 0, 16, 'i', 1,
   BFD_RELOC_HEX_GPREL16_1, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_GPREL_OR_ABS),
   "#%u"},
  {"#g16:1ie", 6, 16, 'i', 1,
   BFD_RELOC_NONE, BFD_RELOC_HEX_IE_32_6_X, BFD_RELOC_HEX_IE_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_GPREL_OR_ABS),
   "#%u"},
  {"#g16:2", 0, 16, 'i', 2,
   BFD_RELOC_HEX_GPREL16_2, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_GPREL_OR_ABS),
   "#%u"},
  {"#g16:2ie", 6, 16, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_HEX_IE_32_6_X, BFD_RELOC_HEX_IE_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_GPREL_OR_ABS),
   "#%u"},
  {"#g16:3", 0, 16, 'i', 3,
   BFD_RELOC_HEX_GPREL16_3, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_GPREL_OR_ABS),
   "#%u"},
  {"#g16:3ie", 6, 16, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_HEX_IE_32_6_X, BFD_RELOC_HEX_IE_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_GPREL_OR_ABS),
   "#%u"},

  {"#u26:6", 0, 26, 'i', 6,
   BFD_RELOC_HEX_32_6_X, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_KXER),
   "#%u"},

  {"#u16:0", 0, 16, 'i', 0,
   BFD_RELOC_HEX_GPREL16_0, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_16_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u16:0ie", 6, 16, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_IE_32_6_X, BFD_RELOC_HEX_IE_16_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u16:1", 0, 16, 'i', 1,
   BFD_RELOC_HEX_GPREL16_1, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_16_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u16:1ie", 6, 16, 'i', 1,
   BFD_RELOC_NONE, BFD_RELOC_HEX_IE_32_6_X, BFD_RELOC_HEX_IE_16_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u16:2", 0, 16, 'i', 2,
   BFD_RELOC_HEX_GPREL16_2, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_16_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u16:2ie", 6, 16, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_HEX_IE_32_6_X, BFD_RELOC_HEX_IE_16_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u16:3", 0, 16, 'i', 3,
   BFD_RELOC_HEX_GPREL16_3, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_16_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u16:3ie", 6, 16, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_HEX_IE_32_6_X, BFD_RELOC_HEX_IE_16_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},

  {"#u16", 0, 16, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_16_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u11:3", 0, 11, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u10", 0, 10, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_10_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u9", 0, 9, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_9_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u8", 0, 8, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_8_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u7", 0, 7, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u6:0", 0, 6, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_6_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u6:1", 0, 6, 'i', 1,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_6_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u6:2", 0, 6, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_6_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u6:3", 0, 6, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_6_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u6", 0, 6, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_6_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u6pcrel", 3, 6, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_B32_PCREL_X, BFD_RELOC_HEX_6_PCREL_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_PC_RELATIVE
    | HEXAGON_OPERAND_IS_SIGNED),
   "#%u"},
  {"#u5:2", 0, 5, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u5:3", 0, 5, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u5", 0, 5, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u4:0", 0, 4, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u4:2", 0, 4, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u4", 0, 4, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u3:0", 0, 3, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u3:1", 0, 3, 'i', 1,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u3", 0, 3, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#u2", 0, 2, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U10", 0, 10, 'I', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_10_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U6:2", 0, 6, 'I', 2,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U6", 0, 6, 'I', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_6_X,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U5:2", 0, 5, 'I', 2,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U5:3", 0, 5, 'I', 3,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U5", 0, 5, 'I', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U4:0", 0, 4, 'I', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U4:2", 0, 4, 'I', 2,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U4", 0, 4, 'I', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U3:0", 0, 3, 'I', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U3:1", 0, 3, 'I', 1,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U3:2", 0, 3, 'I', 2,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U3:3", 0, 3, 'I', 3,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U3", 0, 3, 'I', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},
  {"#U2", 0, 2, 'I', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_IMMEDIATE,
   "#%u"},

  {"#s16", 0, 16, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s16got", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GOT_16, BFD_RELOC_HEX_GOT_32_6_X, BFD_RELOC_HEX_GOT_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s16gotrel", 4, 16, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_GOTREL_32_6_X, BFD_RELOC_HEX_GOTREL_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s16gdgot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GD_GOT_16, BFD_RELOC_HEX_GD_GOT_32_6_X,
   BFD_RELOC_HEX_GD_GOT_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s16ldgot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_LD_GOT_16, BFD_RELOC_HEX_LD_GOT_32_6_X,
   BFD_RELOC_HEX_LD_GOT_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s16ie", 4, 16, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_IE_32_6_X, BFD_RELOC_HEX_IE_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s16iegot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_IE_GOT_16, BFD_RELOC_HEX_IE_GOT_32_6_X,
   BFD_RELOC_HEX_IE_GOT_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s16dtprel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_DTPREL_16, BFD_RELOC_HEX_DTPREL_32_6_X,
   BFD_RELOC_HEX_DTPREL_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s16tprel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_TPREL_16, BFD_RELOC_HEX_TPREL_32_6_X,
   BFD_RELOC_HEX_TPREL_16_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s12", 0, 12, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_12_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:0", 0, 11, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:0got", 6, 11, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_GOT_32_6_X, BFD_RELOC_HEX_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:0gotrel", 6, 11, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_GOTREL_32_6_X, BFD_RELOC_HEX_GOTREL_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:0gdgot", 6, 11, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_GD_GOT_32_6_X, BFD_RELOC_HEX_GD_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:0ldgot", 6, 11, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_LD_GOT_32_6_X, BFD_RELOC_HEX_LD_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:0iegot", 6, 11, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_IE_GOT_32_6_X, BFD_RELOC_HEX_IE_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:0dtprel", 6, 11, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_DTPREL_32_6_X, BFD_RELOC_HEX_DTPREL_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:0tprel", 6, 11, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_TPREL_32_6_X, BFD_RELOC_HEX_TPREL_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:1", 0, 11, 'i', 1,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:1got", 6, 11, 'i', 1,
   BFD_RELOC_NONE, BFD_RELOC_HEX_GOT_32_6_X, BFD_RELOC_HEX_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:1gotrel", 6, 11, 'i', 1,
   BFD_RELOC_NONE, BFD_RELOC_HEX_GOTREL_32_6_X, BFD_RELOC_HEX_GOTREL_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:1gdgot", 6, 11, 'i', 1,
   BFD_RELOC_NONE, BFD_RELOC_HEX_GD_GOT_32_6_X, BFD_RELOC_HEX_GD_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:1ldgot", 6, 11, 'i', 1,
   BFD_RELOC_NONE, BFD_RELOC_HEX_LD_GOT_32_6_X, BFD_RELOC_HEX_LD_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:1iegot", 6, 11, 'i', 1,
   BFD_RELOC_NONE, BFD_RELOC_HEX_IE_GOT_32_6_X, BFD_RELOC_HEX_IE_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:1dtprel", 6, 11, 'i', 1,
   BFD_RELOC_NONE, BFD_RELOC_HEX_DTPREL_32_6_X, BFD_RELOC_HEX_DTPREL_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:1tprel", 6, 11, 'i', 1,
   BFD_RELOC_NONE, BFD_RELOC_HEX_TPREL_32_6_X, BFD_RELOC_HEX_TPREL_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:2", 0, 11, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:2got", 6, 11, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_HEX_GOT_32_6_X, BFD_RELOC_HEX_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:2gotrel", 6, 11, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_HEX_GOTREL_32_6_X, BFD_RELOC_HEX_GOTREL_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:2gdgot", 6, 11, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_HEX_GD_GOT_32_6_X, BFD_RELOC_HEX_GD_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:2ldgot", 6, 11, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_HEX_LD_GOT_32_6_X, BFD_RELOC_HEX_LD_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:2iegot", 6, 11, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_HEX_IE_GOT_32_6_X, BFD_RELOC_HEX_IE_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:2dtprel", 6, 11, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_HEX_DTPREL_32_6_X, BFD_RELOC_HEX_DTPREL_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:2tprel", 6, 11, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_HEX_TPREL_32_6_X, BFD_RELOC_HEX_TPREL_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:3", 0, 11, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:3got", 6, 11, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_HEX_GOT_32_6_X, BFD_RELOC_HEX_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:3gotrel", 6, 11, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_HEX_GOTREL_32_6_X, BFD_RELOC_HEX_GOTREL_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:3gdgot", 6, 11, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_HEX_GD_GOT_32_6_X, BFD_RELOC_HEX_GD_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:3ldgot", 6, 11, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_HEX_LD_GOT_32_6_X, BFD_RELOC_HEX_LD_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:3iegot", 6, 11, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_HEX_IE_GOT_32_6_X, BFD_RELOC_HEX_IE_GOT_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:3dtprel", 6, 11, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_HEX_DTPREL_32_6_X, BFD_RELOC_HEX_DTPREL_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s11:3tprel", 6, 11, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_HEX_TPREL_32_6_X, BFD_RELOC_HEX_TPREL_11_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s10", 0, 10, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_10_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s8", 0, 8, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_8_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s7", 0, 7, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_7_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s6:3", 0, 6, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_6_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s6", 0, 6, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_6_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s4:0", 0, 4, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s4:1", 0, 4, 'i', 1,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s4:2", 0, 4, 'i', 2,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s4:3", 0, 4, 'i', 3,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#s4", 0, 4, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#S8", 0, 8, 'I', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_8_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#S7", 0, 7, 'I', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_7_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#S6:3", 0, 6, 'I', 3,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_6_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},
  {"#S6", 0, 6, 'I', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_6_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED),
   "#%d"},

  {"#m9", 0, 9, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_HEX_32_6_X, BFD_RELOC_HEX_9_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_NEGATIVE),
   "#%d"},

  {"#r22:2", 0, 22, 'i', 2,
   BFD_RELOC_HEX_B22_PCREL, BFD_RELOC_HEX_B32_PCREL_X,
   BFD_RELOC_HEX_B22_PCREL_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_PC_RELATIVE
    | HEXAGON_OPERAND_IS_SIGNED),
   "@%u"},
  {"#r22:2plt", 0, 22, 'i', 2,
   BFD_RELOC_HEX_PLT_B22_PCREL, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_PC_RELATIVE
    | HEXAGON_OPERAND_IS_SIGNED),
   "@%u"},
  {"#r22:2gdplt", 0, 22, 'i', 2,
   BFD_RELOC_HEX_GD_PLT_B22_PCREL, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_PC_RELATIVE
    | HEXAGON_OPERAND_IS_SIGNED),
   "@%u"},
  {"#r22:2ldplt", 0, 22, 'i', 2,
   BFD_RELOC_HEX_LD_PLT_B22_PCREL, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_PC_RELATIVE
    | HEXAGON_OPERAND_IS_SIGNED),
   "@%u"},
  {"#r15:2", 0, 15, 'i', 2,
   BFD_RELOC_HEX_B15_PCREL, BFD_RELOC_HEX_B32_PCREL_X,
   BFD_RELOC_HEX_B15_PCREL_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_PC_RELATIVE
    | HEXAGON_OPERAND_IS_SIGNED),
   "@%u"},
  {"#r13:2", 0, 13, 'i', 2,
   BFD_RELOC_HEX_B13_PCREL, BFD_RELOC_HEX_B32_PCREL_X,
   BFD_RELOC_HEX_B13_PCREL_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_PC_RELATIVE
    | HEXAGON_OPERAND_IS_SIGNED),
   "@%u"},
  {"#r9:2", 0, 9, 'i', 2,
   BFD_RELOC_HEX_B9_PCREL, BFD_RELOC_HEX_B32_PCREL_X,
   BFD_RELOC_HEX_B9_PCREL_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_PC_RELATIVE
    | HEXAGON_OPERAND_IS_SIGNED),
   "@%u"},
  {"#r7:2", 0, 7, 'i', 2,
   BFD_RELOC_HEX_B7_PCREL, BFD_RELOC_HEX_B32_PCREL_X,
   BFD_RELOC_HEX_B7_PCREL_X,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_PC_RELATIVE
    | HEXAGON_OPERAND_IS_SIGNED),
   "@%u"},

  /* These do appear in instructions, but are only matched by the
     relocation.  */
  {"#u26:6", 0, 26, 'i', 6,
   BFD_RELOC_HEX_B32_PCREL_X, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_PC_RELATIVE
    | HEXAGON_OPERAND_IS_SIGNED | HEXAGON_OPERAND_IS_KXER),
   "#%u"},
  {"#u26:6", 0, 26, 'i', 6,
   BFD_RELOC_HEX_GOTREL_32_6_X, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_KXER),
   "#%u"},
  {"#u26:6", 0, 26, 'i', 6,
   BFD_RELOC_HEX_GOT_32_6_X, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_KXER),
   "#%u"},
  {"#u26:6", 0, 26, 'i', 6,
   BFD_RELOC_HEX_DTPREL_32_6_X, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_KXER),
   "#%u"},
  {"#u26:6", 0, 26, 'i', 6,
   BFD_RELOC_HEX_GD_GOT_32_6_X, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_KXER),
   "#%u"},
  {"#u26:6", 0, 26, 'i', 6,
   BFD_RELOC_HEX_LD_GOT_32_6_X, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_KXER),
   "#%u"},
  {"#u26:6", 0, 26, 'i', 6,
   BFD_RELOC_HEX_IE_32_6_X, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_KXER),
   "#%u"},
  {"#u26:6", 0, 26, 'i', 6,
   BFD_RELOC_HEX_IE_GOT_32_6_X, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_KXER),
   "#%u"},
  {"#u26:6", 0, 26, 'i', 6,
   BFD_RELOC_HEX_TPREL_32_6_X, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_KXER),
   "#%u"},

  /* These don't appear in any instructions.  We get these by mapping
     from the previous operands when we see the lo/hi modifier in the
     assembly.  */
  {"#u16_lo16", 4, 16, 'i', 0,
   BFD_RELOC_HEX_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_LO16),
   "#%u"},
  {"#u16_lo16got", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GOT_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_LO16),
   "#%u"},
  {"#u16_lo16gotrel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GOTREL_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_LO16),
   "#%u"},
  {"#u16_lo16gdgot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GD_GOT_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_LO16),
   "#%u"},
  {"#u16_lo16ldgot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_LD_GOT_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_LO16),
   "#%u"},
  {"#u16_lo16ie", 4, 16, 'i', 0,
   BFD_RELOC_HEX_IE_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_LO16),
   "#%u"},
  {"#u16_lo16iegot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_IE_GOT_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_LO16),
   "#%u"},
  {"#u16_lo16dtprel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_DTPREL_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_LO16),
   "#%u"},
  {"#u16_lo16tprel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_TPREL_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_LO16),
   "#%u"},
  {"#u16_hi16", 4, 16, 'i', 0,
   BFD_RELOC_HEX_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_HI16),
   "#%u"},
  {"#u16_hi16got", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GOT_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_HI16),
   "#%u"},
  {"#u16_hi16gotrel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GOTREL_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_HI16),
   "#%u"},
  {"#u16_hi16gdgot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GD_GOT_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_HI16),
   "#%u"},
  {"#u16_hi16ldgot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_LD_GOT_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_HI16),
   "#%u"},
  {"#u16_hi16ie", 4, 16, 'i', 0,
   BFD_RELOC_HEX_IE_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_HI16),
   "#%u"},
  {"#u16_hi16iegot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_IE_GOT_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_HI16),
   "#%u"},
  {"#u16_hi16dtprel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_DTPREL_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_HI16),
   "#%u"},
  {"#u16_hi16tprel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_TPREL_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_HI16),
   "#%u"},

  {"#s16_lo16", 4, 16, 'i', 0,
   BFD_RELOC_HEX_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_LO16),
   "#%d"},
  {"#s16_lo16got", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GOT_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_LO16),
   "#%d"},
  {"#s16_lo16gotrel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GOTREL_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_LO16),
   "#%d"},
  {"#s16_lo16gdgot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GD_GOT_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_LO16),
   "#%d"},
  {"#s16_lo16ldgot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_LD_GOT_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_LO16),
   "#%d"},
  {"#s16_lo16ie", 4, 16, 'i', 0,
   BFD_RELOC_HEX_IE_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_LO16),
   "#%d"},
  {"#s16_lo16iegot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_IE_GOT_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_LO16),
   "#%d"},
  {"#s16_lo16dtprel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_DTPREL_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_LO16),
   "#%d"},
  {"#s16_lo16tprel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_TPREL_LO16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_LO16),
   "#%d"},
  {"#s16_hi16", 4, 16, 'i', 0,
   BFD_RELOC_HEX_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_HI16),
   "#%d"},
  {"#s16_hi16got", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GOT_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_HI16),
   "#%d"},
  {"#s16_hi16gotrel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GOTREL_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_HI16),
   "#%d"},
  {"#s16_hi16gdgot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_GD_GOT_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_HI16),
   "#%d"},
  {"#s16_hi16ldgot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_LD_GOT_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_HI16),
   "#%d"},
  {"#s16_hi16ie", 4, 16, 'i', 0,
   BFD_RELOC_HEX_IE_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_HI16),
   "#%d"},
  {"#s16_hi16iegot", 4, 16, 'i', 0,
   BFD_RELOC_HEX_IE_GOT_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_HI16),
   "#%d"},
  {"#s16_hi16dtprel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_DTPREL_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_HI16),
   "#%d"},
  {"#s16_hi16tprel", 4, 16, 'i', 0,
   BFD_RELOC_HEX_TPREL_HI16, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_IMMEDIATE | HEXAGON_OPERAND_IS_SIGNED
    | HEXAGON_OPERAND_IS_HI16),
   "#%d"},

  /* These don't appear in any instructions.  They are used by the packet
     header.  */
  {"Hs", 0, 4, 's', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   0,
   "0x%x"},
  {"Hi", 0, 1, 'i', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   0,
   "%d"},
  {"Ho", 0, 1, 'o', 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   0,
   "%d"},

  /* These are just place-holders for implied operands.  */
  {"#-1", 0, 0, 0, 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_CONSTANT,
   "#-1"},
  {"#0", 0, 0, 0, 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_CONSTANT,
   "#0"},
  {"#1", 0, 0, 0, 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_CONSTANT,
   "#1"},
  {"#2", 0, 0, 0, 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_CONSTANT,
   "#2"},
  {"#3", 0, 0, 0, 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_CONSTANT,
   "#3"},
  {"#255", 0, 0, 0, 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   HEXAGON_OPERAND_IS_CONSTANT,
   "#255"},
  {"p0", 0, 0, 0, 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_PREDICATE | HEXAGON_OPERAND_IS_READ
    | HEXAGON_OPERAND_IS_WRITE),
   "p0"},
  {"p1", 0, 0, 0, 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_PREDICATE | HEXAGON_OPERAND_IS_READ
    | HEXAGON_OPERAND_IS_WRITE),
   "p1"},
  {"Lr", 0, 0, 0, 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_READ),
   "lr"},
  {"Sp", 0, 0, 0, 0,
   BFD_RELOC_NONE, BFD_RELOC_NONE, BFD_RELOC_NONE,
   (HEXAGON_OPERAND_IS_REGISTER | HEXAGON_OPERAND_IS_READ),
   "sp"},
};

static const size_t hexagon_operand_count =
  sizeof (hexagon_operands) / sizeof (hexagon_operands[0]);

/* Suck in the opcodes.  */
#define HEXAGON_OPCODE(syntax, enc, slots, implicit, attributes) \
  { syntax, enc, 0, NULL, NULL, slots, implicit, attributes, NULL }, 
#define HEXAGON_MAPPING(syntax, enc, map) 

static hexagon_opcode hexagon_opcodes_v2[] = {
#include "opcode/hexagon_iset_v2.h"
};

static hexagon_opcode hexagon_opcodes_v3[] = {
#include "opcode/hexagon_iset_v3.h"
};

static hexagon_opcode hexagon_opcodes_v4[] = {
#include "opcode/hexagon_iset_v4.h"
};

static hexagon_opcode hexagon_opcodes_v5[] = {
#include "opcode/hexagon_iset_v5.h"
};

static hexagon_opcode hexagon_opcodes_v55[] = {
#include "opcode/hexagon_iset_v55.h"
#include "opcode/hexagon_iset_v5.h"
};

#undef HEXAGON_OPCODE
#undef HEXAGON_MAPPING

static hexagon_opcode *hexagon_opcodes[] =
  {
    hexagon_opcodes_v2,
    hexagon_opcodes_v3,
    hexagon_opcodes_v4,
    hexagon_opcodes_v5,
    hexagon_opcodes_v55
  };

static const size_t hexagon_opcodes_count[] =
  {
    sizeof (hexagon_opcodes_v2) / sizeof (hexagon_opcode),
    sizeof (hexagon_opcodes_v3) / sizeof (hexagon_opcode),
    sizeof (hexagon_opcodes_v4) / sizeof (hexagon_opcode),
    sizeof (hexagon_opcodes_v5) / sizeof (hexagon_opcode),
    sizeof (hexagon_opcodes_v55) / sizeof (hexagon_opcode)
  };

/* Support for opcode hashing.  */
#define HEXAGON_HASH_0    (0 * HEXAGON_HASH_1)
#define HEXAGON_HASH_1    (1 * ('z' - 'a' + 1))
#define HEXAGON_HASH_2    (2 * HEXAGON_HASH_1)
#define HEXAGON_HASH_3    (3 * HEXAGON_HASH_1)
#define HEXAGON_HASH_4    (4 * HEXAGON_HASH_1)
#define HEXAGON_HASH_5    (5 * HEXAGON_HASH_1)
#define HEXAGON_HASH_SIZE (5 * HEXAGON_HASH_1 + 17)
#define HEXAGON_HASH_A    (HEXAGON_HASH_SIZE -  1)
#define HEXAGON_HASH_B    (HEXAGON_HASH_SIZE -  2)
#define HEXAGON_HASH_C    (HEXAGON_HASH_SIZE -  3)
#define HEXAGON_HASH_D    (HEXAGON_HASH_SIZE -  4)
#define HEXAGON_HASH_E    (HEXAGON_HASH_SIZE -  5)
#define HEXAGON_HASH_F    (HEXAGON_HASH_SIZE -  6)
#define HEXAGON_HASH_G    (HEXAGON_HASH_SIZE -  7)
#define HEXAGON_HASH_H    (HEXAGON_HASH_SIZE -  8)
#define HEXAGON_HASH_I    (HEXAGON_HASH_SIZE -  9)
#define HEXAGON_HASH_J    (HEXAGON_HASH_SIZE - 10)
#define HEXAGON_HASH_K    (HEXAGON_HASH_SIZE - 11)
#define HEXAGON_HASH_L    (HEXAGON_HASH_SIZE - 12)
#define HEXAGON_HASH_M    (HEXAGON_HASH_SIZE - 13)
#define HEXAGON_HASH_N    (HEXAGON_HASH_SIZE - 14)
#define HEXAGON_HASH_O    (HEXAGON_HASH_SIZE - 15)
#define HEXAGON_HASH_P    (HEXAGON_HASH_SIZE - 16)
#define HEXAGON_HASH_Z    (HEXAGON_HASH_SIZE - 17)

/* Indexed by first letter of opcode.  Points to chain of opcodes with same
   first letter.  */
static hexagon_opcode *opcode_map[HEXAGON_HASH_SIZE];

static size_t hexagon_icode_hash_bits;

/* Indexed by insn code.  Points to chain of opcodes with same insn code.  */
static hexagon_opcode **icode_map;

/* Create a hash for an opcode given its syntax string SYNTAX.
   The generic scheme is to use the first letter of the operation as the
   hash, with some exceptions for register transfers and other specific
   cases.
   All in all, it could be improved to result in a more balanced depth.  */
static hexagon_hash
hexagon_hash_opcode (const char *syntax)
{
  const char *pair, *peq;
  size_t token;
  int is_if;

  pair = strchr (syntax, ';');
  peq = strchr (syntax, '=');

  /* BLANK || PUNCT */
  token = strcspn (syntax, " \t`~!@#$%^&*()_-=+[{]}\\|;:'\",<.>/?");
  is_if = !strncmp (syntax, "if", token);

#if 0
  if (strncmp (syntax, "mem", 3))
    {
      /* Loads and stores.  */
      if (peq)
	/* Conditional and unconditional stores.  */
	return pif ? (pnew ? HEXAGON_HASH_A : HEXAGON_HASH_B) : HEXAGON_HASH_C;
      else
	/* Conditional and unconditional loads.  */
	return pif ? (pnew ? HEXAGON_HASH_D : HEXAGON_HASH_E) :	HEXAGON_HASH_F;
    }
#endif

  if (pair)
    {
      const char *p1 = pair + 1;
      while (ISSPACE (*p1))
	p1++;

      if (!strncmp (p1, "jump", 4)
	  || !strncmp (syntax, "jump", 4)
	  || !strncmp (p1, "call", 4)
	  || !strncmp (syntax, "call", 4)
	  || ((hexagon_if_arch_v4 () || hexagon_if_arch_v5 ()
	       || hexagon_if_arch_v55 ())
	      && (!strncmp (p1, "return", token)
		  || !strncmp (syntax, "return", token))))
	/* Conditional and unconditional branches.  */
	return is_if ? HEXAGON_HASH_G : HEXAGON_HASH_H;
    }
  else
    {
      const char *p1 = syntax;

      if (!strncmp (p1, "jump", 4)
	  || !strncmp (p1, "call", 4)
	  || ((hexagon_if_arch_v4 () || hexagon_if_arch_v5 ()
	       || hexagon_if_arch_v55 ())
	      && !strncmp (p1, "return", token)))
	/* Conditional and unconditional branches.  */
	return is_if ? HEXAGON_HASH_I : HEXAGON_HASH_J;
    }

  if (peq)
    {
      /* Assignments.  */
      const char *p1 = peq + 1;
      while (ISSPACE (*p1))
	p1++;

      if (*p1 != '#' && strchr (p1, '('))
	{
	  char ch;

	  /* Operations.  */
	  while (ISSPACE (*p1) || ISPUNCT (*p1))
	    p1++;

	  ch = TOLOWER (*p1);
	  if (ISALPHA (ch))
	    {
	      if (ch == 'v')
		{
		  /* Vector insns use the operation initial instead of
		     the prefix.  */
		  ch = TOLOWER (p1[1]) - 'a' + (is_if ? HEXAGON_HASH_1 : 0);

		  if (pair) 	/* Paired insns.  */
		    return HEXAGON_HASH_2 + ch;
		  else		/* Vector insns.  */
		    return HEXAGON_HASH_4 + ch;
		}
	      else
		{
		  ch = ch - 'a' + (is_if ? HEXAGON_HASH_1 : 0);
		  if (pair)	/* Paired insns.  */
		    return HEXAGON_HASH_2 + ch;
		  else		/* Single insns.  */
		    return HEXAGON_HASH_0 + ch;
		}
	    }
	}
      else
	{
	  /* Conditional and unconditional transfers.  */
	  if (pair)
	    return is_if ? HEXAGON_HASH_K : HEXAGON_HASH_L;
	  else
	    return is_if ? HEXAGON_HASH_M : HEXAGON_HASH_N;
	}
    }

  if (!strncmp (syntax, "allocframe", 10)
      || !strncmp (syntax, "deallocframe", token)
      || !strncmp (syntax, "nop", token))
    return pair ? HEXAGON_HASH_O : HEXAGON_HASH_P;

  /* Miscellanea.  */
  return HEXAGON_HASH_Z;
}

/* Convert the encoding string ENC to an instruction (bit pattern).  */
static hexagon_insn
hexagon_encode_opcode (const char *enc)
{
  hexagon_insn insn;
  int num_bits;
  int pair;

  for (insn = 0, num_bits = 0, pair = 0; *enc; enc++)
    {
      while (ISSPACE (*enc))
	enc++;

      if (!*enc)
	break;

      insn = (insn << 1) + ((*enc == '1') ? 1 : 0);
      num_bits++;
      pair |= (enc[0] == 'E' && enc[1] == 'E');
    }

  if (num_bits != HEXAGON_INSN_LEN * 8)
    {
      printf ("invalid number of bits: %d\n", num_bits);
      assert (FALSE);
    }

  return HEXAGON_END_PACKET_SET (insn,
				 pair ? HEXAGON_END_PAIR : HEXAGON_END_NOT);
}

/* Produce a bit mask from the encoding string ENC such that
   the bits that must be 1 or 0 are set to 1.  Then
   insn & hexagon_encode_mask (enc) == hexagon_encode_opcode (enc)
   if and only if insn matches ENC.  */
static hexagon_insn
hexagon_encode_mask (const char *enc)
{
  hexagon_insn mask;
  int num_bits;

  for (mask = 0, num_bits = 0; *enc; enc++)
    {
      while (ISSPACE (*enc))
	enc++;

      if (!*enc)
	break;

      mask = (mask << 1) + ((*enc == '1' || *enc == '0') ? 1 : 0);

      num_bits++;
    }

  assert (num_bits == HEXAGON_INSN_LEN * 8);
  return mask;
}

/* Return a hash value for the instruction INSN.  */
static hexagon_hash
hexagon_hash_icode (hexagon_insn insn)
{
  const hexagon_hash xtra
    = ~(-(HEXAGON_END_PACKET_MASK >> HEXAGON_END_PACKET_POS));
  hexagon_hash hash;

  hash = insn >> (HEXAGON_INSN_LEN * 8 - (hexagon_icode_hash_bits - xtra));
  hash |= ((HEXAGON_END_PACKET_GET (insn) >> HEXAGON_END_PACKET_POS)
	   << (hexagon_icode_hash_bits - xtra));

  return hash;
}

/* Current machine value, using the bfd_mach_hexagon_v* encoding, and
   set by calling hexagon_isa_init.  */
static int current_mach;

/* Return current architecture.  */
int
hexagon_arch (void)
{
  return current_mach;
}

/* Return nonzero if we are targeting the specified architecture ARCH.  */
int
hexagon_if_arch (int arch)
{
  return current_mach == arch;
}

/* Return true if the current architecture supports immediate extenders.  */
int
hexagon_if_arch_kext (void)
{
  return hexagon_if_arch_v4 () || hexagon_if_arch_v5 ()
	 || hexagon_if_arch_v55 ();
}

/* Return true if the current architecture supports non-GPR paired regs.  */
int
hexagon_if_arch_pairs (void)
{
  return hexagon_if_arch_v4 () || hexagon_if_arch_v5 ()
	 || hexagon_if_arch_v55 ();
}

/* Return true if the current architecture supports auto-anding of m
   multiple predicate changes.  */
int
hexagon_if_arch_autoand (void)
{
  return hexagon_if_arch_v4 () || hexagon_if_arch_v5 ()
	 || hexagon_if_arch_v55 ();
}


/* Install OPCODE in the lookup tables.  */
static void
hexagon_install_opcode (hexagon_opcode *opcode)
{
  hexagon_insn insn = hexagon_encode_opcode (opcode->enc);
  hexagon_hash hcode = hexagon_hash_opcode (opcode->syntax);
  hexagon_hash icode = hexagon_hash_icode (insn);

  opcode->insn = insn;

  opcode->next_asm = opcode_map[hcode];
  opcode_map[hcode] = opcode;

  opcode->next_dis = icode_map[icode];
  icode_map[icode] = opcode;

  if (HEXAGON_OPCODE_IS_NOP (opcode))
    hexagon_nop_opcode = opcode;

  if (HEXAGON_OPCODE_IS_EXTENDER (opcode))
    hexagon_kext_opcode = opcode;
}

/* Initialize any tables that need it.
   Must be called once at start up (or when first needed).
   OPCODES is the array of initializers for the current archicture, and
   OPCODE_COUNT is the number of entries in this array.
   MAPPINGS and MAPPINGS_COUNT are likewise the instruction mappings to be
   installed.  Mappings are part of the assembler support and are not
   needed e.g. by the linker; in this case, MAPPINGS can be null and
   MAPPINGS_COUNT passed as 0.  */
static void 
hexagon_opcode_init_tables (hexagon_opcode *opcodes,
			    size_t opcodes_count,
			    hexagon_opcode *mappings,
			    size_t mappings_count)
{
  size_t i;

  /* We may be intentionally called more than once (for example gdb will call
     us each time the user switches cpu).  These tables only need to be init'd
     once though.  */

  /* Scan the opcodes table to determine the number of bits
     that can reliably be used to hash opcodes.  */
  hexagon_icode_hash_bits = HEXAGON_INSN_LEN * 8;
  for (i = 0; i < opcodes_count; i++)
    {
      unsigned int bits;
      const char *str;

      for (bits = 0, str = opcodes[i].enc; *str; str++)
	{
	  if (*str == '0' || *str == '1' || *str == 'P' || *str == 'E')
	    bits++;
	  else
	    break;
	}

      if (bits < hexagon_icode_hash_bits)
	hexagon_icode_hash_bits = bits;
    }
  hexagon_icode_hash_bits
    += ~(-(HEXAGON_END_PACKET_MASK >> HEXAGON_END_PACKET_POS));

  /* Initialize hash maps.  */
  memset (opcode_map, 0, sizeof (opcode_map));
  icode_map = xcalloc (1 << hexagon_icode_hash_bits, sizeof (*icode_map));
  hexagon_nop_opcode = NULL;
  hexagon_kext_opcode = NULL;

  /* Scan the table backwards so macros appear at the front.  */
  for (i = mappings_count; i > 0; --i)
    hexagon_install_opcode (mappings + i - 1);
  for (i = opcodes_count; i > 0; --i)
    hexagon_install_opcode (opcodes + i - 1);
  assert (hexagon_nop_opcode);
  if (hexagon_if_arch_kext ())
    assert (hexagon_kext_opcode);
  
#if 0
  puts ("Hash Table Depth:");
  for (i = 0; i < HEXAGON_HASH_SIZE; i++)
    {
      const hexagon_opcode *op;
      size_t j;

      for (j = 0, op = opcode_map[i]; op; j++)
	op = HEXAGON_CODE_NEXT_ASM (op);

      printf ("#%02ld ('%c'): %02ld\t", i, (i % HEXAGON_HASH_1) + 'a', j);
    }
  puts ("");

  puts ("Hash Table Syntax:");
  for (i = 0; i < HEXAGON_HASH_SIZE; i++)
    {
      const hexagon_opcode *op;
      size_t j;

      printf ("#%02ld ('%c')\n", i, (i % HEXAGON_HASH_1) + 'a');

      for (j = 0, op = opcode_map[i]; op;
	   j++, op = HEXAGON_CODE_NEXT_ASM (op))
	puts (op->syntax);

      puts ("");
    }
#endif
}

void
hexagon_isa_init (int mach,
		  hexagon_opcode **mappings,
		  size_t *mapping_count)
{
  int n;

  if (current_mach == mach)
    return;
  current_mach = mach;

  if (mach < bfd_mach_hexagon_v2 || mach > bfd_mach_hexagon_v55)
    return;

  n = mach - bfd_mach_hexagon_v2;

  hexagon_opcode_init_tables (hexagon_opcodes[n],
			      hexagon_opcodes_count[n],
			      (mappings ? mappings[n] : NULL),
			      (mapping_count ? mapping_count[n] : 0));
}


/* Encode the value AVALUE for operand OPERAND in INSN, which is an instance
   of OPCODE.  IS_X is true if this opcode is extended.
   Returns true if successful, otherwise sets *ERRMSG and returns false.  */
int
hexagon_encode_operand (const hexagon_operand *operand, hexagon_insn *insn,
			const hexagon_opcode *opcode, long avalue,
			int is_x, char **errmsg)
{
  char enc_letter;
  int shift_count = 0;
  int num_bits = 0;
  size_t len = strlen (opcode->enc);
  unsigned bits;
  union
  {
    long s;
    unsigned long u;
  } value;
  int is_s;
  ptrdiff_t i;
  static char buf[500];

  is_s = operand->flags & HEXAGON_OPERAND_IS_SIGNED;
  enc_letter = operand->enc_letter;
  bits = operand->bits + operand->shift_count;
  value.s = avalue;

  if (operand->shift_count)
    {
      /* Make sure the low bits are zero.  */
      if (value.s & (~(~0L << operand->shift_count)))
	{
	  if (errmsg)
	    {
	      sprintf (buf, "low %d bits of immediate %ld must be zero",
		       operand->shift_count, value.s);
	      *errmsg = buf;
	    }
	  return FALSE;
	}
    }

  if (operand->flags & HEXAGON_OPERAND_IS_LO16)
    value.s = HEXAGON_LO16 (value.s);
  else if (operand->flags & HEXAGON_OPERAND_IS_HI16)
    value.s = HEXAGON_HI16 (value.s);
  else if (operand->flags & HEXAGON_OPERAND_IS_SUBSET)
    value.s = HEXAGON_SUBREGS_TO (value.s,
				  operand->flags & HEXAGON_OPERAND_IS_PAIR);

  /* If this is an extended value, keep only the low 6 bits.  Otherwise
     verify that the operand fits in the specified number of bits.  */
  if (is_x && bits >= 6 && hexagon_if_arch_kext ())
    value.s = HEXAGON_KXED_MASK (value.s);
  else if (is_s)
    {
      long smax = ~(~0 << (bits - 1));
      long smin = ~0 << (bits - 1);
      if (operand->flags & HEXAGON_OPERAND_IS_NEGATIVE)
	smin++;
      if (value.s < smin || (smax > 0 && value.s > smax))
	{
	  if (errmsg)
	    {
	      sprintf (buf, "value %ld out of range: %ld-%ld", value.s, smin,
		       smax);
	      *errmsg = buf;
	    }
	  return FALSE;
	}
    }
  else
    {
      unsigned long umax = ~(~0U << bits);
      unsigned long umin = 0U;
      if (value.u < umin || (umax > 0 && value.u > umax))
	{
	  if (errmsg)
	    {
	      sprintf (buf, "value %lu out of range: %lu-%lu", value.u, umin,
		       umax);
	      *errmsg = buf;
	    }
	  return FALSE;
	}
    }

  /* In the presence of an extender, the value is not shifted.  */
  if (!is_x)
    value.s >>= operand->shift_count;

  /* Read the encoding string backwards and put a bit in each time.  */
  for (i = len - 1; i >= 0; i--)
    if (!ISSPACE (opcode->enc[i]))
      {
	if (opcode->enc[i] == enc_letter)
	  {
	    // Clear the bit
	    (*insn) &= ~(1 << shift_count);
	    // Insert the new bit
	    (*insn) |= (value.s & 1) << shift_count;
	    value.s >>= 1;
	    num_bits++;
	  }
	shift_count++;
      }

  /* Make sure we encode the expected number of bits */
  if (num_bits != operand->bits)
    {
      if (errmsg)
	{
	  sprintf (buf, "did not encode expected number of bits: %d != %d\n"
		   "\tenc = %s\n\toperand = %s",
		   num_bits, operand->bits, opcode->enc, operand->fmt);
	  *errmsg = buf;
	}
      return FALSE;
    }

  return TRUE;
}

/* Determine whether the operand with format FMT appears in the opcode
   syntax string SYNTAX.  If LEN is zero, match all characters of FMT,
   otherwise match only the first LEN chars.  */
static int
hexagon_match_op (const char *syntax, const char *fmt, size_t len)
{
  size_t haystack_len = strlen (syntax);
  size_t needle_len = len ? len : strlen (fmt);
  size_t i;

  if (needle_len > haystack_len)
    return FALSE;
  if (needle_len == 0)
    return TRUE;
  for (i = 0; i <= haystack_len - needle_len; i++)
    if (strncmp (syntax + i, fmt, needle_len) == 0)
      return TRUE;
  return FALSE;
}

/* Find the operand corresponding to RELOC_TYPE in OPCODE.  */
const hexagon_operand *
hexagon_lookup_reloc (bfd_reloc_code_real_type reloc_type, int flags,
		      const hexagon_opcode *opcode)
{
  size_t i;

  if (reloc_type == BFD_RELOC_NONE || !opcode)
    return NULL;

  if (flags & HEXAGON_OPERAND_IS_KXED)
    {
      for (i = 0; i < hexagon_operand_count; i++)
	if (hexagon_operands[i].reloc_kxed == reloc_type
	    && (!(hexagon_operands[i].flags & HEXAGON_OPERAND_IS_IMMEDIATE)
		|| (HEXAGON_OPERAND_IS_EXTENDABLE (&hexagon_operands[i], opcode)
		    && hexagon_match_op (opcode->syntax,
					 hexagon_operands[i].fmt,
					 hexagon_operands[i].len))))
	return hexagon_operands + i;
    }

  else
    for (i = 0; i < hexagon_operand_count; i++)
      if (hexagon_operands[i].reloc_type == reloc_type)
	return hexagon_operands + i;

  return NULL;
}

/* Find the operand matching NAME.  If NAMELEN is zero, do an exact match.
   Otherwise, look for an exact match on the first NAMELEN chars of NAME.  */
const hexagon_operand *
hexagon_lookup_operand_name (const char *name, size_t namelen)
{
  size_t i;

  for (i = 0; i < hexagon_operand_count; i++)
    if ((namelen
	 && strlen (hexagon_operands[i].fmt) == namelen
	 && strncmp (name, hexagon_operands[i].fmt, namelen) == 0)
	|| (!namelen
	    && strcmp (name, hexagon_operands[i].fmt) == 0))
      return hexagon_operands + i;
  return NULL;
}

/* Return the first insn in the chain for assembling INSN.  */
const hexagon_opcode *
hexagon_opcode_lookup_asm (const char *insn)
{
  return opcode_map[hexagon_hash_opcode (insn)];
}

/* Return the first insn in the chain for disassembling INSN.  */
const hexagon_opcode *
hexagon_opcode_lookup_dis (hexagon_insn insn)
{
  return icode_map[hexagon_hash_icode (insn)];
}

/* Return the opcode descriptor for INSN.  */
const hexagon_opcode *
hexagon_lookup_insn (hexagon_insn insn)
{
  const hexagon_opcode *opcode;
  hexagon_insn encode;
  hexagon_insn mask;

  /* Reset the packet bits.  */
  insn = HEXAGON_END_PACKET_RESET (insn);

  for (opcode = hexagon_opcode_lookup_dis (insn);
       opcode; opcode = HEXAGON_CODE_NEXT_DIS (opcode))
    {
      if (opcode->map)
	continue;

      encode = opcode->insn;
      mask = hexagon_encode_mask (opcode->enc);

      if ((insn & mask) == (encode & mask))
	return opcode;
    }

  return NULL;
}



