/*
 This file is part of GNU Taler
 (C) 2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { Logger, succeedOrThrow } from "@gnu-taler/taler-util";
import * as fs from "node:fs";
import { GlobalTestState } from "../harness/harness.js";
import { setupMeasuresTestEnvironment } from "../harness/tops.js";

export const logger = new Logger("test-tops-aml-measures.ts");

interface FileField {
  CONTENTS: string;
  ENCODING: "base64";
  FILENAME: string;
  MIME_TYPE: string;
}


/**
 * Helper function to create a mock PDF file attachment.
 */
function createPdfAttachment(name: string): FileField {
  // Escape parentheses and backslashes for PDF literal strings
  const contentEscaped = name.replace(/\\/g, "\\\\").replace(/\(/g, "\\(").replace(/\)/g, "\\)");

  const contentStream = `BT /F1 12 Tf 10 20 Td (${contentEscaped}) Tj ET`;

  // Minimal PDF with standard font
  const pdf = `%PDF-1.1
1 0 obj
<< /Type /Catalog /Pages 2 0 R>>
endobj
2 0 obj
<< /Type /Pages /Kids[3 0 R]/Count 1>>
endobj
3 0 obj
<< /Type /Page /Parent 2 0 R /MediaBox[0 0 200 50]
/Resources<< /Font<< /F1<< /Type/Font /Subtype/Type1 /BaseFont/Helvetica >> >> >>
/Contents 4 0 R >>
endobj
4 0 obj
<< /Length ${contentStream.length}>>
stream
${contentStream}
endstream
endobj
xref
0 5
0000000000 65535 f
0000000009 00000 n
0000000056 00000 n
0000000107 00000 n
0000000251 00000 n
trailer<< /Size 5 /Root 1 0 R>>
startxref
${330 + contentStream.length}
%%EOF`;

  return {
    CONTENTS: Buffer.from(pdf, "utf8").toString("base64"),
    ENCODING: "base64",
    FILENAME: name + `.pdf`,
    MIME_TYPE: "application/pdf",
  };
}

/**
 * Test that invokes all measures defined for the TOPS deployment.
 */
export async function runTopsAmlPdfTest(t: GlobalTestState) {
  // Setup is done, now the real testing can start!

  const {
    decideMeasure,
    submitForm,
    submitOfficerForm,
    expectInvestigate,
    expectNoInvestigate,
    officerAcc,
    exchangeClient,
    accountPaytoHash,
    fakeChallenger,
    challengerPostal,
    challengerSms,
  } = await setupMeasuresTestEnvironment(t);

  // Test vqf_902_1_customer form - LEGAL_ENTITY variant
  {
    await decideMeasure("kyx");
    await expectNoInvestigate();
    await submitForm("vqf_902_1_customer", {
      FORM_ID: "vqf_902_1_customer",
      FORM_VERSION: 1,
      SIGNATURE: "Bob Smith",
      CUSTOMER_TYPE: "LEGAL_ENTITY",
      COMPANY_NAME: "Acme Corporation",
      REGISTERED_OFFICE_ADDRESS: "Castle St. 1\nWondertown\n8000 Zurich",
      CONTACT_PERSON_NAME: "Bob Smith",
      CONTACT_PHONE: "+41123456789",
      CONTACT_EMAIL: "bob@acme.com",
      LEGAL_ENTITY_IDENTIFICATION_DOCUMENT_COPY:
        createPdfAttachment("company-id-doc"),
      ESTABLISHER_LIST: [
        {
          FULL_NAME: "Alice Johnson",
          DOMICILE_ADDRESS: "Main St. 10\n8001 Zurich",
          DATE_OF_BIRTH: "1980-05-15",
          NATIONALITY: "CH",
          PERSONAL_IDENTIFICATION_DOCUMENT_COPY:
            createPdfAttachment("alice-id"),
          SIGNING_AUTHORITY_TYPE: "SINGLE",
          SIGNING_AUTHORITY_EVIDENCE_TYPE: "CR",
          SIGNING_AUTHORITY_EVIDENCE_DOCUMENT_COPY:
            createPdfAttachment("alice-cr"),
        },
        {
          FULL_NAME: "Charlie Brown",
          DOMICILE_ADDRESS: "Park Ave. 20\n8002 Zurich",
          DATE_OF_BIRTH: "1975-08-22",
          NATIONALITY: "DE",
          PERSONAL_IDENTIFICATION_DOCUMENT_COPY:
            createPdfAttachment("charlie-id"),
          SIGNING_AUTHORITY_TYPE: "COLLECTIVE_TWO",
          SIGNING_AUTHORITY_EVIDENCE_TYPE: "MANDATE",
          SIGNING_AUTHORITY_EVIDENCE_DOCUMENT_COPY:
            createPdfAttachment("charlie-mandate"),
        },
        {
          FULL_NAME: "Diana Prince",
          DOMICILE_ADDRESS: "Lake Road 30\n8003 Zurich",
          DATE_OF_BIRTH: "1985-12-01",
          NATIONALITY: "US",
          PERSONAL_IDENTIFICATION_DOCUMENT_COPY:
            createPdfAttachment("diana-id"),
          SIGNING_AUTHORITY_TYPE: "OTHER",
          SIGNING_AUTHORITY_EVIDENCE_TYPE: "OTHER",
          SIGNING_AUTHORITY_EVIDENCE_OTHER: "Board Resolution",
          SIGNING_AUTHORITY_EVIDENCE_DOCUMENT_COPY:
            createPdfAttachment("diana-resolution"),
        },
      ],
      CORRESPONDENCE_LANGUAGE: "en",
      CUSTOMER_TYPE_VQF: "OPERATIONAL",
    });

  }

  // Test vqf_902_1_customer form - NATURAL_PERSON variant (without sole proprietor)
  {
    await decideMeasure("kyx");
    await expectNoInvestigate();
    await submitForm("vqf_902_1_customer", {
      FORM_ID: "vqf_902_1_customer",
      FORM_VERSION: 1,
      SIGNATURE: "Bob Smith",
      CUSTOMER_TYPE: "NATURAL_PERSON",
      FULL_NAME: "John Doe",
      DOMICILE_ADDRESS: "Elm Street 5\n8010 Zurich",
      CONTACT_PHONE: "+41987654321",
      CONTACT_EMAIL: "john.doe@example.com",
      DATE_OF_BIRTH: "1990-03-12",
      NATIONALITY: "CH",
      PERSONAL_IDENTIFICATION_DOCUMENT_COPY:
        createPdfAttachment("john-passport"),
      CUSTOMER_IS_SOLE_PROPRIETOR: false,
      CORRESPONDENCE_LANGUAGE: "de",
      CUSTOMER_TYPE_VQF: "NATURAL",
    });

  }

  // Test vqf_902_1_customer form - NATURAL_PERSON variant (with sole proprietor)
  {
    await decideMeasure("kyx");
    await submitForm("vqf_902_1_customer", {
      FORM_ID: "vqf_902_1_customer",
      FORM_VERSION: 1,
      SIGNATURE: "Bob Smith",
      CUSTOMER_TYPE: "NATURAL_PERSON",
      FULL_NAME: "Jane Smith",
      DOMICILE_ADDRESS: "Oak Avenue 12\n8020 Zurich",
      CONTACT_PHONE: "+41555123456",
      CONTACT_EMAIL: "jane@smithconsulting.ch",
      DATE_OF_BIRTH: "1985-07-20",
      NATIONALITY: "FR",
      PERSONAL_IDENTIFICATION_DOCUMENT_COPY: createPdfAttachment("jane-id"),
      CUSTOMER_IS_SOLE_PROPRIETOR: true,
      COMPANY_NAME: "Smith Consulting GmbH",
      REGISTERED_OFFICE_ADDRESS: "Business Park 3\n8021 Zurich",
      LEGAL_ENTITY_IDENTIFICATION_DOCUMENT_COPY:
        createPdfAttachment("smith-biz-reg"),
      CORRESPONDENCE_LANGUAGE: "fr",
      CUSTOMER_TYPE_VQF: "OTHER",
    });

  }

  {
    await submitOfficerForm("vqf_902_1_officer", {
      FORM_ID: "vqf_902_1_officer",
      FORM_VERSION: 1,
      ACCEPTANCE_DATE: "2025-12-01",
      ACCEPTANCE_METHOD: "FACE_TO_FACE",
      ACCEPTANCE_FURTHER_INFO: "Meeting at branch office with ID verification",
      EMBARGO_TERRORISM_CHECK_RESULT: "NOT_LISTED",
      EMBARGO_TERRORISM_CHECK_DATE: "2025-12-01",
      SUPPLEMENTAL_FILES_LIST: [
        {
          FILE: createPdfAttachment("meeting-notes"),
          DESCRIPTION: "Meeting notes from customer onboarding",
        },
        {
          FILE: createPdfAttachment("id-verification"),
          DESCRIPTION: "ID verification documentation",
        },
        {
          FILE: createPdfAttachment("background-check"),
          DESCRIPTION: "Background check results",
        },
      ],
    });

  }

  // Test vqf_902_1_officer form with LISTED terrorism check
  {
    await submitOfficerForm("vqf_902_1_officer", {
      FORM_ID: "vqf_902_1_officer",
      FORM_VERSION: 1,
      ACCEPTANCE_DATE: "2025-12-02",
      ACCEPTANCE_METHOD: "AUTHENTICATED_COPY",
      ACCEPTANCE_FURTHER_INFO: "Authenticated copy received via postal service",
      EMBARGO_TERRORISM_CHECK_RESULT: "LISTED",
      EMBARGO_TERRORISM_CHECK_DATE: "2025-12-02",
      EMBARGO_TERRORISM_INFO:
        "Entry 4252 might match",
      SUPPLEMENTAL_FILES_LIST: [
        {
          FILE: createPdfAttachment("sanctions-report"),
          DESCRIPTION: "Detailed sanctions screening report",
        },
        {
          FILE: createPdfAttachment("compliance-review"),
          DESCRIPTION: "Compliance team review",
        },
        {
          FILE: createPdfAttachment("escalation-doc"),
          DESCRIPTION: "Escalation to senior management",
        },
      ],
    });

  }

  // Test vqf_902_4 form - without PEP or high risk
  {
    await submitOfficerForm("vqf_902_4", {
      FORM_ID: "vqf_902_4",
      FORM_VERSION: 1,
      PEP_FOREIGN: false,
      PEP_DOMESTIC: false,
      PEP_INTERNATIONAL_ORGANIZATION: false,
      PEP_HIGH_RISK: false,
      HIGH_RISK_COUNTRY: false,
      COUNTRY_RISK_NATIONALITY_TYPE: [
        "NATIONALITY_CUSTOMER",
        "DOMICILE_CUSTOMER",
      ],
      COUNTRY_RISK_NATIONALITY_LEVEL: "LOW",
      COUNTRY_RISK_BUSINESS_TYPE: ["CUSTOMER"],
      COUNTRY_RISK_BUSINESS_LEVEL: "LOW",
      COUNTRY_RISK_PAYMENTS_LEVEL: "LOW",
      INDUSTRY_RISK_TYPE: "CUSTOMER",
      INDUSTRY_RISK_LEVEL: "TRANSPARENT",
      CONTACT_RISK_LEVEL: "LOW",
      PRODUCT_RISK_LEVEL: ["HIGH_OFFSHORE", "HIGH_FREQUENT_TMER"],
      RISK_RATIONALE:
        "Customer is well-known local business with transparent operations",
      RISK_CLASSIFICATION_LEVEL: "NO_HIGH_RISK",
    });

  }

  // Test vqf_902_4 form - with PEP and high risk
  {
    await submitOfficerForm("vqf_902_4", {
      FORM_ID: "vqf_902_4",
      FORM_VERSION: 1,
      CUSTOMER_NAME: "Bob",
      PEP_FOREIGN: true,
      PEP_DOMESTIC: true,
      PEP_INTERNATIONAL_ORGANIZATION: false,
      PEP_HIGH_RISK: true,
      PEP_ACCEPTANCE_DATE: "2025-11-15",
      HIGH_RISK_COUNTRY: true,
      HIGH_RISK_ACCEPTANCE_DATE: "2025-11-15",
      COUNTRY_RISK_NATIONALITY_TYPE: [
        "NATIONALITY_CUSTOMER",
        "NATIONALITY_OWNER",
        "DOMICILE_CUSTOMER",
      ],
      COUNTRY_RISK_NATIONALITY_LEVEL: "HIGH",
      COUNTRY_RISK_BUSINESS_TYPE: ["CUSTOMER", "OWNER"],
      COUNTRY_RISK_BUSINESS_LEVEL: "HIGH",
      COUNTRY_RISK_PAYMENTS_LEVEL: "MEDIUM",
      INDUSTRY_RISK_TYPE: "CUSTOMER",
      INDUSTRY_RISK_LEVEL: "HIGH_RISK_TRADE",
      PRODUCT_RISK_LEVEL: ["LOW"],
      CONTACT_RISK_LEVEL: "HIGH",
      RISK_RATIONALE:
        "Customer is foreign PEP with connections to high-risk jurisdictions. Enhanced due diligence applied.",
      RISK_CLASSIFICATION_LEVEL: "HIGH_RISK",
    });

  }

  // Test vqf_902_5 form - with SAVINGS origin
  {
    await submitOfficerForm("vqf_902_5", {
      FORM_ID: "vqf_902_5",
      FORM_VERSION: 1,
      CUSTOMER_NAME: "Bob",
      BIZREL_PROFESSION:
        "Software Engineer at Tech Company\nPreviously worked as IT Consultant\n15 years experience",
      BIZREL_FINANCIAL_CIRCUMSTANCES:
        "Annual income: CHF 120,000\nTotal assets: CHF 500,000\nNo significant liabilities",
      BIZREL_ORIGIN_NATURE:
        "Cash deposits and investment returns\nAmount: CHF 200,000\nCurrency: CHF",
      BIZREL_ORIGIN_CATEGORY: ["SAVINGS", "OWN_BUSINESS"],
      BIZREL_ORIGIN_DETAIL:
        "Accumulated savings from employment over 10 years. Additional income from freelance software consulting projects.",
      BIZREL_PURPOSE:
        "Investment account for long-term wealth management and retirement planning",
      BIZREL_DEVELOPMENT:
        "Expect regular monthly deposits of CHF 3,000-5,000. Occasional larger deposits from consulting projects.",
      BIZREL_FINANCIAL_VOLUME:
        "Average monthly volume: CHF 4,000. Expected annual growth of 10-15%.",
      BIZREL_FINANCIAL_BENEFICIARIES_FULL_NAME: "Self (account holder)",
      BIZREL_THIRDPARTY_RELATIONSHIP:
        "No third parties involved. Account holder is sole beneficial owner.",
      BIZREL_THIRDPARTY_AMLA_FILES: "None",
      BIZREL_THIRDPARTY_REFERENCES:
        "Introduced by existing client Marcus Weber, Account #CH-12345",
      BIZREL_FURTHER_INFO:
        "Customer has been transparent about income sources. All documentation verified.",
    });

  }

  // Test vqf_902_5 form - with INHERITANCE and OTHER origin
  {
    await submitOfficerForm("vqf_902_5", {
      FORM_ID: "vqf_902_5",
      FORM_VERSION: 1,
      CUSTOMER_NAME: "Bob",
      BIZREL_PROFESSION:
        "Retired business owner\nFormer CEO of manufacturing company\nRetired since 2020",
      BIZREL_FINANCIAL_CIRCUMSTANCES:
        "Pension income: CHF 80,000 per year\nTotal net worth: CHF 2,500,000\nNo debt",
      BIZREL_ORIGIN_NATURE:
        "Inheritance and business sale proceeds\nAmount: CHF 1,500,000\nCurrency: CHF and EUR",
      BIZREL_ORIGIN_CATEGORY: ["INHERITANCE", "OTHER"],
      BIZREL_ORIGIN_CATEGORY_OTHER: "Sale of family business in 2020",
      BIZREL_ORIGIN_DETAIL:
        "CHF 800,000 from inheritance after parents' estate settlement in 2019. CHF 700,000 from sale of 60% stake in family manufacturing business to private equity firm.",
      BIZREL_PURPOSE: "Wealth preservation and estate planning for children",
      BIZREL_DEVELOPMENT:
        "Minimal transaction activity expected. Quarterly dividend reinvestment.",
      BIZREL_FINANCIAL_VOLUME:
        "Low volume - primarily passive investment portfolio",
      BIZREL_FINANCIAL_BENEFICIARIES_FULL_NAME:
        "Account holder and designated heirs: Maria Schmidt, Thomas Schmidt",
      BIZREL_THIRDPARTY_RELATIONSHIP:
        "Trust arrangement for benefit of children listed above",
      BIZREL_THIRDPARTY_AMLA_FILES: "Linked to estate file #EST-2019-445",
      BIZREL_THIRDPARTY_REFERENCES:
        "Legal advisor: Dr. Hans Mueller, Zurich Law Firm",
      BIZREL_FURTHER_INFO:
        "Estate documentation and business sale agreement on file. All sources verified through notarized documents.",
    });

  }

  // Test vqf_902_9_customer form
  {
    await submitOfficerForm("vqf_902_9_customer", {
      FORM_ID: "vqf_902_9_customer",
      FORM_VERSION: 1,
      IDENTITY_CONTRACTING_PARTNER:
        "Global Trading Ltd.\nCommerce Street 45\n8050 Zurich",
      IDENTITY_LIST: [
        {
          FULL_NAME: "Michael Anderson",
          DATE_OF_BIRTH: "1970-04-10",
          DOMICILE_ADDRESS: "Riverside Drive 88\n8051 Zurich",
          NATIONALITY: "GB",
        },
        {
          FULL_NAME: "Sarah Johnson",
          DATE_OF_BIRTH: "1975-09-25",
          DOMICILE_ADDRESS: "Mountain View 22\n8052 Zurich",
          NATIONALITY: "US",
        },
        {
          FULL_NAME: "Roberto Martinez",
          DATE_OF_BIRTH: "1968-11-30",
          DOMICILE_ADDRESS: "Garden Plaza 7\n8053 Zurich",
          NATIONALITY: "ES",
        },
      ],
      SIGNATURE: "Michael Anderson",
      SIGN_DATE: "2025-12-05",
    });

  }

  // Test vqf_902_9_officer form
  {
    await submitOfficerForm("vqf_902_9_officer", {
      FORM_ID: "vqf_902_9_officer",
      FORM_VERSION: 1,
      IDENTITY_CONTRACTING_PARTNER:
        "Tech Innovations AG\nInnovation Park 12\n8060 Zurich",
      IDENTITY_LIST: [
        {
          FULL_NAME: "Emma Thompson",
          DATE_OF_BIRTH: "1982-06-18",
          DOMICILE_ADDRESS: "Sunset Boulevard 15\n8061 Zurich",
          NATIONALITY: "CH",
        },
        {
          FULL_NAME: "David Chen",
          DATE_OF_BIRTH: "1978-03-05",
          DOMICILE_ADDRESS: "Tech Valley 99\n8062 Zurich",
          NATIONALITY: "SG",
        },
        {
          FULL_NAME: "Isabella Rossi",
          DATE_OF_BIRTH: "1985-12-22",
          DOMICILE_ADDRESS: "Lake View 44\n8063 Zurich",
          NATIONALITY: "IT",
        },
      ],
      ATTACHMENT_SIGNED_DOCUMENT: createPdfAttachment("signed-form-902-9"),
    });

  }

  // Test vqf_902_11_customer form - HAS_25_MORE_RIGHTS variant, without third party
  {
    await decideMeasure("form-vqf-902.11");
    await submitForm("vqf_902_11_customer", {
      FORM_ID: "vqf_902_11_customer",
      FORM_VERSION: 1,
      IDENTITY_CONTRACTING_PARTNER:
        "Manufacturing Solutions Ltd.\nIndustrial Zone 78\n8070 Zurich",
      CONTROL_REASON: "HAS_25_MORE_RIGHTS",
      IDENTITY_LIST: [
        {
          FULL_NAME: "William Brown",
          DOMICILE_ADDRESS: "Highland Road 33\n8071 Zurich",
        },
        {
          FULL_NAME: "Olivia Davis",
          DOMICILE_ADDRESS: "Valley Street 56\n8072 Zurich",
        },
        {
          FULL_NAME: "James Wilson",
          DOMICILE_ADDRESS: "Forest Avenue 11\n8073 Zurich",
        },
      ],
      THIRD_PARTY_OWNERSHIP: false,
      SIGNATURE: "William Brown",
      SIGN_DATE: "2025-12-04",
    });

  }

  // Test vqf_902_11_customer form - OTHER_WAY variant, with third party
  {
    await decideMeasure("form-vqf-902.11");
    await submitForm("vqf_902_11_customer", {
      FORM_ID: "vqf_902_11_customer",
      FORM_VERSION: 1,
      IDENTITY_CONTRACTING_PARTNER:
        "Service Holdings AG\nService Center 90\n8080 Zurich",
      CONTROL_REASON: "OTHER_WAY",
      IDENTITY_LIST: [
        {
          FULL_NAME: "Sophia Martinez",
          DOMICILE_ADDRESS: "Central Plaza 77\n8081 Zurich",
        },
        {
          FULL_NAME: "Lucas Garcia",
          DOMICILE_ADDRESS: "North Street 24\n8082 Zurich",
        },
        {
          FULL_NAME: "Mia Rodriguez",
          DOMICILE_ADDRESS: "South Boulevard 13\n8083 Zurich",
        },
      ],
      THIRD_PARTY_OWNERSHIP: true,
      SIGNATURE: "Sophia Martinez",
      SIGN_DATE: "2025-12-03",
    });

  }

  // Test vqf_902_11_customer form - DIRECTOR variant
  {
    await decideMeasure("form-vqf-902.11");
    await submitForm("vqf_902_11_customer", {
      FORM_ID: "vqf_902_11_customer",
      FORM_VERSION: 1,
      IDENTITY_CONTRACTING_PARTNER:
        "Consulting Partners GmbH\nConsulting Tower 5\n8090 Zurich",
      CONTROL_REASON: "DIRECTOR",
      IDENTITY_LIST: [
        {
          FULL_NAME: "Alexander Mueller",
          DOMICILE_ADDRESS: "Executive Lane 66\n8091 Zurich",
        },
        {
          FULL_NAME: "Charlotte Weber",
          DOMICILE_ADDRESS: "Management Street 88\n8092 Zurich",
        },
        {
          FULL_NAME: "Benjamin Schneider",
          DOMICILE_ADDRESS: "Director Avenue 99\n8093 Zurich",
        },
      ],
      THIRD_PARTY_OWNERSHIP: false,
      SIGNATURE: "Alexander Mueller",
      SIGN_DATE: "2025-12-02",
    });

  }

  // Test vqf_902_11_officer form - HAS_25_MORE_RIGHTS variant
  {
    await submitOfficerForm("vqf_902_11_officer", {
      FORM_ID: "vqf_902_11_officer",
      FORM_VERSION: 1,
      IDENTITY_CONTRACTING_PARTNER:
        "Investment Group SA\nFinance District 101\n8100 Zurich",
      CONTROL_REASON: "HAS_25_MORE_RIGHTS",
      IDENTITY_LIST: [
        {
          FULL_NAME: "Victoria Fischer",
          DOMICILE: "Investor Road 55\n8101 Zurich",
        },
        {
          FULL_NAME: "Daniel Hoffmann",
          DOMICILE: "Capital Street 77\n8102 Zurich",
        },
        {
          FULL_NAME: "Sophie Zimmermann",
          DOMICILE: "Fund Avenue 22\n8103 Zurich",
        },
      ],
      THIRD_PARTY_OWNERSHIP: true,
      ATTACHMENT_SIGNED_DOCUMENT: createPdfAttachment("signed-form-902-11"),
    });

  }

  // Test vqf_902_14 form - NO_SUSPICION result
  {
    await submitOfficerForm("vqf_902_14", {
      FORM_ID: "vqf_902_14",
      FORM_VERSION: 1,
      CUSTOMER_NAME: "Bob",
      INCRISK_REASON:
        "Large cash deposit of CHF 50,000 triggered monitoring alert. Customer explanation requested.",
      INCRISK_MEANS: "GATHERING",
      INCRISK_SUMMARY:
        "Customer provided proof of property sale. Documentation includes notarized sales agreement and bank transfer records. All information is consistent and plausible.",
      INCRISK_DOCUMENTS:
        "Property sales agreement (notarized), Bank transfer confirmation, Tax declaration",
      INCRISK_RESULT: "NO_SUSPICION",
    });

  }

  // Test vqf_902_14 form - SUBSTANTIATED_SUSPICION result
  {
    await submitOfficerForm("vqf_902_14", {
      FORM_ID: "vqf_902_14",
      FORM_VERSION: 1,
      INCRISK_REASON:
        "Multiple transactions to high-risk jurisdictions. Inconsistent explanations provided. Transaction patterns suggest possible structuring.",
      INCRISK_MEANS: "CONSULTATION",
      INCRISK_SUMMARY:
        "Customer's stated business activity does not align with transaction patterns. Multiple red flags identified including: (1) transactions just below reporting threshold, (2) payments to shell companies in offshore jurisdictions, (3) inconsistent business documentation.",
      INCRISK_DOCUMENTS:
        "Transaction logs, Customer interviews, Public records search, Third-party database checks",
      INCRISK_RESULT: "REASONABLE_SUSPICION",
    });

  }

  // Test vqf_902_14 form - OTHER result with explanation
  {
    await submitOfficerForm("vqf_902_14", {
      FORM_ID: "vqf_902_14",
      FORM_VERSION: 1,
      INCRISK_REASON:
        "Unusual activity pattern detected. Customer requested clarification on enhanced monitoring procedures.",
      INCRISK_MEANS: "OTHER",
      INCRISK_MEANS_OTHER:
        "Video conference with customer and legal counsel, review of corporate governance documents, consultation with external compliance specialist",
      INCRISK_SUMMARY:
        "Customer is undergoing corporate restructuring. Increased transaction volume is temporary and related to merger activity. All parties have been identified and verified.",
      INCRISK_DOCUMENTS:
        "Merger agreement, Board resolutions, Legal opinions, Updated beneficial ownership structure",
      INCRISK_RESULT: "OTHER",
      INCRISK_RESULT_OTHER:
        "Enhanced monitoring period established - 6-month review cycle with quarterly reporting requirements",
    });

  }

  // Test generic_note form
  {
    await submitOfficerForm("generic_note", {
      FORM_ID: "generic_note",
      FORM_VERSION: 1,
      NOTE_TEXT:
        "Follow-up meeting held with customer regarding recent transaction patterns. Customer provided satisfactory explanations and supporting documentation. No further concerns at this time.",
      SUPPLEMENTAL_FILES_LIST: [
        {
          DESCRIPTION: "Meeting minutes from customer interview",
          FILE: createPdfAttachment("meeting-minutes-20251205"),
        },
        {
          DESCRIPTION: "Supporting bank statements",
          FILE: createPdfAttachment("bank-statements-q4-2025"),
        },
        {
          DESCRIPTION: "Customer signed acknowledgment",
          FILE: createPdfAttachment("customer-acknowledgment"),
        },
      ],
    });

  }

  // Test generic_upload form
  {
    await submitOfficerForm("generic_upload", {
      FORM_ID: "generic_upload",
      FORM_VERSION: 1,
      NOTE_TEXT:
        "Customer requested to upload current business license and updated company registry extract as part of periodic review.",
      FILE: createPdfAttachment("business-license-2025"),
    });

  }

  // Test accept-tos form
  {
    await decideMeasure("accept-tos");
    await submitForm("accept-tos", {
      FORM_ID: "accept-tos",
      FORM_VERSION: 1,
      ACCEPTED_TERMS_OF_SERVICE: "v1",
      DOWNLOADED_TERMS_OF_SERVICE: true,
    });

  }

  // Test challenger-sms form
  {
    await decideMeasure("sms-registration");
    await fakeChallenger(challengerSms, {
      CONTACT_PHONE: "+4123456789",
    });
  }

  // Test challenger-postal form
  {
    await decideMeasure("postal-registration");
    await fakeChallenger(challengerPostal, {
      CONTACT_NAME: "Richard Stallman",
      ADDRESS_LINES: "Bundesgasse 1\n1234 Bern",
      ADDRESS_COUNTRY: "CH",
    });
  }

  // Final PDF generation
  {
    const res = succeedOrThrow(
      await exchangeClient.getAmlAttributesForAccountAsPdf(
        officerAcc,
        accountPaytoHash,
        { limit: 100, order: "asc" }
      ),
    );
    const f = t.testDir + `/aml-file.pdf`;
    fs.writeFileSync(f, Buffer.from(res));
    console.log(`written to ${f}`);
  }

  // Export accounts in different formats
  {
    const res = succeedOrThrow(
      await exchangeClient.getAmlAccountsAsOtherFormat(officerAcc, "text/csv"),
    );
    const f = t.testDir + `/accounts.csv`;
    fs.writeFileSync(f, Buffer.from(res));
    console.log(`written to ${f}`);
  }

  {
    const res = succeedOrThrow(
      await exchangeClient.getAmlAccountsAsOtherFormat(
        officerAcc,
        "application/json",
      ),
    );
    const f = t.testDir + `/accounts.json`;
    fs.writeFileSync(f, Buffer.from(res));
    console.log(`written to ${f}`);
  }

  {
    const res = succeedOrThrow(
      await exchangeClient.getAmlAccountsAsOtherFormat(
        officerAcc,
        "application/vnd.ms-excel",
      ),
    );
    const f = t.testDir + `/accounts.xls`;
    fs.writeFileSync(f, Buffer.from(res));
    console.log(`written to ${f}`);
  }
}

runTopsAmlPdfTest.suites = ["exchange"];
