/*
 This file is part of GNU Taler
 (C) 2020-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  CreditDebitIndicator,
  MerchantAuthMethod,
  TalerCorebankApiClient,
  TalerWireGatewayHttpClient,
  createEddsaKeyPair,
  encodeCrock,
  setPrintHttpRequestAsCurl,
} from "@gnu-taler/taler-util";
import { defaultCoinConfig } from "../harness/denomStructures.js";
import {
  BankService,
  ExchangeService,
  GlobalTestState,
  MERCHANT_DEFAULT_AUTH,
  MerchantService,
  getTestHarnessPaytoForLabel,
  setupDb,
} from "../harness/harness.js";

/**
 * Run test for basic, bank-integrated withdrawal.
 */
export async function runBankApiTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    currency: "TESTKUDOS",
    httpPort: 8082,
    database: db.connStr,
    allowRegistrations: true,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    httpPort: 8083,
    database: db.connStr,
  });

  let receiverName = "Exchange";
  let exchangeBankUsername = "exchange";
  let exchangeBankPassword = "mypw-password";
  let exchangePaytoUri = getTestHarnessPaytoForLabel(exchangeBankUsername);
  let wireGatewayAuth = {
    username: exchangeBankUsername,
    password: exchangeBankPassword,
  };

  let wireGatewayApiBaseUrl = new URL(
    "accounts/exchange/taler-wire-gateway/",
    bank.baseUrl,
  ).href;
  await exchange.addBankAccount("1", {
    wireGatewayAuth,
    wireGatewayApiBaseUrl,
    accountPaytoUri: exchangePaytoUri,
  });

  bank.setSuggestedExchange(exchange, exchangePaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  exchange.addOfferedCoins(defaultCoinConfig);

  await exchange.start();
  await exchange.pingUntilAvailable();

  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  setPrintHttpRequestAsCurl(true)

  const { accessToken: adminAccessToken } = await merchant.addInstanceWithWireAccount({
    id: "admin",
    name: "Default Instance",
    paytoUris: [getTestHarnessPaytoForLabel("merchant-default")],
  });

  await merchant.addInstanceWithWireAccount({
    id: "minst1",
    name: "minst1",
    paytoUris: [getTestHarnessPaytoForLabel("minst1")],
  }, { adminAccessToken });

  console.log("setup done!");

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "admin-password",
    },
  });

  await bankClient.registerAccountExtended({
    name: receiverName,
    password: exchangeBankPassword,
    username: exchangeBankUsername,
    is_taler_exchange: true,
    payto_uri: exchangePaytoUri,
  });

  const bankUser = await bankClient.registerAccount("user1", "password1");

  // Make sure that registering twice results in a 409 Conflict
  {
    const e = await t.assertThrowsTalerErrorAsyncLegacy(
      bankClient.registerAccount("user1", "password2"),
    );
    t.assertTrue(e.errorDetail.httpStatusCode === 409);
  }

  let balResp = await bankClient.getAccountBalance(bankUser.username);

  console.log(balResp);

  // Check that we got the sign-up bonus.
  t.assertAmountEquals(balResp.balance.amount, "TESTKUDOS:100");
  t.assertTrue(
    balResp.balance.credit_debit_indicator === CreditDebitIndicator.Credit,
  );

  const res = createEddsaKeyPair();

  const wireGatewayApiClient = new TalerWireGatewayHttpClient(
    wireGatewayApiBaseUrl,
  );

  await wireGatewayApiClient.addIncoming({
    auth: bank.getAdminAuth(),
    body: {
      amount: "TESTKUDOS:115",
      debit_account: bankUser.accountPaytoUri,
      reserve_pub: encodeCrock(res.eddsaPub),
    },
  });

  balResp = await bankClient.getAccountBalance(bankUser.username);
  t.assertAmountEquals(balResp.balance.amount, "TESTKUDOS:15");
  t.assertTrue(
    balResp.balance.credit_debit_indicator === CreditDebitIndicator.Debit,
  );
}

runBankApiTest.suites = ["fakebank"];
