/*
   This file is part of TALER
   Copyright (C) 2023, 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file exchangedb/pg_get_withdraw.c
 * @brief Implementation of the get_withdraw function for Postgres
 * @author Özgür Kesim
 */
#include "platform.h"
#include "taler_error_codes.h"
#include "taler_dbevents.h"
#include "taler_pq_lib.h"
#include "pg_get_withdraw.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
TEH_PG_get_withdraw (
  void *cls,
  const struct TALER_HashBlindedPlanchetsP *wch,
  struct TALER_EXCHANGEDB_Withdraw *wd)
{
  enum GNUNET_DB_QueryStatus ret;
  struct PostgresClosure *pg = cls;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_auto_from_type (wch),
    GNUNET_PQ_query_param_end
  };
  struct TALER_BlindedDenominationSignature *my_denom_sigs = NULL;
  uint64_t *my_denom_serials = NULL;
  struct GNUNET_CRYPTO_CSPublicRPairP *my_cs_r_values = NULL;
  size_t num_sigs = 0;
  size_t num_coins = 0;
  size_t num_cs_r_values = 0;
  bool no_noreveal_index;
  bool no_max_age;
  bool no_selected_h;
  bool no_blinding_seed;
  bool no_cs_r_values;
  bool no_cs_r_choices;

  struct GNUNET_PQ_ResultSpec rs[] = {
    GNUNET_PQ_result_spec_auto_from_type ("planchets_h",
                                          &wd->planchets_h),
    GNUNET_PQ_result_spec_auto_from_type ("reserve_sig",
                                          &wd->reserve_sig),
    GNUNET_PQ_result_spec_auto_from_type ("reserve_pub",
                                          &wd->reserve_pub),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_uint16 ("max_age",
                                    &wd->max_age),
      &no_max_age),
    TALER_PQ_result_spec_amount ("amount_with_fee",
                                 pg->currency,
                                 &wd->amount_with_fee),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_uint16 ("noreveal_index",
                                    &wd->noreveal_index),
      &no_noreveal_index),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_auto_from_type ("selected_h",
                                            &wd->selected_h),
      &no_selected_h),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_auto_from_type ("blinding_seed",
                                            &wd->blinding_seed),
      &no_blinding_seed),
    GNUNET_PQ_result_spec_allow_null (
      TALER_PQ_result_spec_array_cs_r_pub (
        pg->conn,
        "cs_r_values",
        &num_cs_r_values,
        &my_cs_r_values),
      &no_cs_r_values),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_uint64 ("cs_r_choices",
                                    &wd->cs_r_choices),
      &no_cs_r_choices),
    TALER_PQ_result_spec_array_blinded_denom_sig (
      pg->conn,
      "denom_sigs",
      &num_sigs,
      &my_denom_sigs),
    GNUNET_PQ_result_spec_array_uint64 (
      pg->conn,
      "denom_serials",
      &num_coins,
      &my_denom_serials),
    GNUNET_PQ_result_spec_end
  };

  PREPARE (pg,
           "get_withdraw",
           "SELECT"
           " planchets_h"
           ",blinding_seed"
           ",reserve_sig"
           ",reserve_pub"
           ",max_age"
           ",amount_with_fee"
           ",noreveal_index"
           ",selected_h"
           ",blinding_seed"
           ",cs_r_values"
           ",cs_r_choices"
           ",denom_sigs"
           ",denom_serials"
           " FROM withdraw"
           " WHERE planchets_h=$1;");
  ret = GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                  "get_withdraw",
                                                  params,
                                                  rs);
  if (0 > ret)
  {
    GNUNET_break (0);
    GNUNET_PQ_cleanup_result (rs);
    return ret;
  }
  if (GNUNET_DB_STATUS_SUCCESS_NO_RESULTS == ret)
  {
    GNUNET_PQ_cleanup_result (rs);
    return ret;
  }

  if ((no_max_age != no_noreveal_index) ||
      (no_max_age != no_selected_h))
  {
    GNUNET_break (0);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "got inconsistent state for max_age, noreveal_index and planchets_h in DB: "
                "no_max_age=%s, no_noreveal_index=%s, no_selected_h=%s\n",
                no_max_age ? "true" : "false",
                no_noreveal_index ? "true" : "false",
                no_selected_h ? "true" : "false");
    GNUNET_PQ_cleanup_result (rs);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  if (no_blinding_seed != no_cs_r_values)
  {
    GNUNET_break (0);
    GNUNET_PQ_cleanup_result (rs);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  if (no_cs_r_choices != no_cs_r_values)
  {
    GNUNET_break (0);
    GNUNET_PQ_cleanup_result (rs);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  if (num_coins != num_sigs)
  {
    GNUNET_break (0);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "got inconsistent number of entries in withdraw from DB: "
                "num_coins=%ld, num_sigs=%ld\n",
                num_coins,
                num_sigs);
    GNUNET_PQ_cleanup_result (rs);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  wd->age_proof_required = ! no_max_age;
  if (no_cs_r_values)
  {
    wd->cs_r_values = NULL;
    wd->num_cs_r_values = 0;
    wd->cs_r_choices = 0;
  }
  wd->denom_sigs = my_denom_sigs;
  wd->num_coins = num_coins;
  wd->denom_serials = my_denom_serials;
  wd->cs_r_values = my_cs_r_values;
  wd->num_cs_r_values = num_cs_r_values;
  /* ensure cleanup_result does not trash data we care about */
  my_denom_sigs = NULL;
  my_denom_serials = NULL;
  my_cs_r_values = NULL;
  num_sigs = 0;
  num_coins = 0;
  num_cs_r_values = 0;
  GNUNET_PQ_cleanup_result (rs);
  return ret;
}
