#!/usr/bin/env python

from gnuradio import gr, eng_notation
from gnuradio import audio
from gnuradio import usrp
from gnuradio import blks
from gnuradio.eng_option import eng_option
from optparse import OptionParser
import sys
import math


#
# return a gr.flow_graph
#
def build_graph (IF_freq, offset_freq, dual_chan):
    adc_rate = 64e6

    usrp_decim = 10
    sw_decim = 25
    audio_decimation = 8

    sample_rate = adc_rate / usrp_decim
    quad_rate = sample_rate / sw_decim         # 256 kHz
    audio_rate = quad_rate / audio_decimation  # 32 kHz

    fg = gr.flow_graph ()
    
    # usrp is data source
    src = usrp.source_c (0, usrp_decim)
    src.set_rx_freq (0, IF_freq)
    
    (head, tail) = build_pipeline (fg, sample_rate, sw_decim, audio_decimation, offset_freq)

    # sound card as final sink
    audio_sink = audio.sink (int (audio_rate))

    # now wire it all together
    fg.connect (src, head)
    fg.connect (tail, (audio_sink, 0))

    # Build second one
    if dual_chan:
        (head2, tail2) = build_pipeline (fg, sample_rate, sw_decim, audio_decimation, -offset_freq)
        fg.connect (src, head2)
        fg.connect (tail2, (audio_sink, 1))
    
    return fg

def build_pipeline (fg, sample_rate, sw_decim, audio_decimation,offset_freq):
    '''Given a flow_graph, fg, construct a pipeline
    for demodulating a broadcast FM signal.  The
    input is the downconverteed complex baseband
    signal. The output is the demodulated audio.

    build_pipeline returns a two element tuple
    containing the input and output endpoints.
    '''
    fm_demod_gain = 2200.0/32768.0

    quad_rate = sample_rate / sw_decim
    audio_rate = quad_rate / audio_decimation
    volume = 1.0

    channel_coeffs = \
      gr.firdes.low_pass (1.0,           # gain
                          sample_rate,   # sampling rate
                          200e3,         # low pass cutoff freq
                          200e3,         # width of trans. band
                          gr.firdes.WIN_HAMMING)

    # input: complex; output: complex
    channel_filter = \
      gr.freq_xlating_fir_filter_ccf (sw_decim,
                                      channel_coeffs,
                                      offset_freq,
                                      sample_rate)
    print len(channel_coeffs)
    # input: complex; output: float
    fm_demod = gr.quadrature_demod_cf (volume*fm_demod_gain)

    # compute FIR filter taps for audio filter
    width_of_transition_band = audio_rate / 32
    audio_coeffs = gr.firdes.low_pass (1.0,            # gain
                                       quad_rate,      # sampling rate
                                       audio_rate/2 - width_of_transition_band,
                                       width_of_transition_band,
                                       gr.firdes.WIN_HAMMING)

    # input: float; output: float
    audio_filter = gr.fir_filter_fff (audio_decimation, audio_coeffs)

    fg.connect (channel_filter, fm_demod)
    fg.connect (fm_demod, audio_filter)
    return ((channel_filter, 0), (audio_filter, 0))
    

def main (args):
    nargs = len (args)
    if nargs == 1:
        freq1 = float (args[0]) * 1e6
        dual_chan = False
        center_freq = freq1
        offset_freq = 0
    elif nargs == 2:
        freq1 = float (args[0]) * 1e6
        freq2 = float (args[1]) * 1e6
        dual_chan = True
        center_freq = (freq1 + freq2)/2
        offset_freq = abs(freq1 - freq2)/2
        if offset_freq > 3e6:
            sys.stderr.write ('Error:  Frequencies must be within 5 MHz\n')
            sys.exit (1)
    else:
        sys.stderr.write ('usage: wfm_rcv freq1 [freq2]\n')
        sys.exit (1)

        

    # connect to RF front end
    rf_front_end = gr.microtune_4937_eval_board ()
    if not rf_front_end.board_present_p ():
        raise IOError, 'RF front end not found'

    # set front end gain
    rf_front_end.set_AGC (300)
    IF_freq = rf_front_end.get_output_freq ()
    IF_freq = -5.75e6

    rf_front_end.set_RF_freq (center_freq)
    fg = build_graph (IF_freq,offset_freq,dual_chan)

    fg.start ()        # fork thread(s) and return
    raw_input ('Press Enter to quit: ')
    fg.stop ()

if __name__ == '__main__':
    main (sys.argv[1:])


