/* -*- Mode: c++ -*-
 *
 *  Copyright 1999 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */

#ifndef _VRUDPSOURCE_H_
#define _VRUDPSOURCE_H_

#include <VrSource.h>
#include <string.h> 
#include <sys/types.h> 
#include <sys/time.h>
#include <unistd.h>
#include <netinet/in.h> 
#include <sys/socket.h> 
#include <sys/wait.h>
#include <arpa/inet.h>
#include <iostream.h>
#include <netinet/in.h>

extern rtp_db *masterdb;
#define MAX_UDP_SIZE	3000
#define SECS_BETWEEN_1900_1970 2208988800u
template<class oType> 
class VrUDPSource : public VrSource<oType> {
protected:
  int sockfd;
  int sockrtcpfd;
  struct sockaddr_in my_addr;    /* my address information */
  struct sockaddr_in their_addr; /* connector's address information */
  virtual void initialize() { }
  unsigned char tempbuf[MAX_UDP_SIZE];
  int tempbuf_index, tempbuf_size;
public: 
  virtual int work(VrSampleRange output, void *o[]);
  VrUDPSource(int);
  virtual ~VrUDPSource() {}
};
	int max_fd;
	int actually_process = 0;
    
template<class oType> int
VrUDPSource<oType>::work(VrSampleRange output, void *ao[])
{
  int len, xxdiff, xxneeded;
  unsigned char data_in[PACKET_LENGTH];
  struct timeval now;
  unsigned char *cp = (unsigned char *)ao[0];
  int size = output.size * sizeof(oType);
  unsigned int addr_len = sizeof(struct sockaddr);
  static fd_set rfd; 
  
  while (size > 0) {
#if 0
   if (tempbuf_size > 0) {
      *cp++ = tempbuf[tempbuf_index++];
      tempbuf_size--;
      size--;
      continue;
   }
   if (size < MAX_UDP_SIZE) {
      tempbuf_index = 0;
      tempbuf_size = recvfrom(sockfd, tempbuf, sizeof(tempbuf), 0,
          (struct sockaddr *)&their_addr, &addr_len);
      continue;
   }
   len = recvfrom(sockfd, cp, size, 0,
       (struct sockaddr *)&their_addr, &addr_len);
   if (len > 0){
     //jca printf ("received %d bytes from %s\n", len, inet_ntoa(their_addr.sin_addr) );
     size -= len;
     cp += len;
   }
   else {
     printf ("[%s:%d] error in recvfrom %d\n", __FILE__, __LINE__, errno);
     break;
   }
#else
usleep(1200000);
    gettimeofday(&now,NULL);
	xxdiff = ((int)((now.tv_sec  - last_read_time.tv_sec) * 1000
			+ (now.tv_usec - last_read_time.tv_usec)/1000))
    * (BITS_PER_SAMPLE / 8) * SAMPLE_RATE / 1000 ;
   	xxneeded = BYTES_PER_BLOCK - avail_bytes;
#define xxxmin(a, b)	(((a) < (b))? (a): (b))
    if (xxdiff <= xxneeded)
    	usleep(xxxmin((int)(xxneeded /BITS_PER_SAMPLE), 10/*delay_ms*/) *1000);
	FD_ZERO(&rfd);
	FD_SET(sockfd, &rfd);
	FD_SET(sockrtcpfd, &rfd);
    while (select(max_fd + 1, &rfd, NULL, NULL, &no_time_at_all) > 0) {
		if (FD_ISSET(sockfd, &rfd)) {
			if ((len = recvfrom(sockfd, (char *)data_in,
				sizeof(data_in), 0, 0, 0)) <= 0)
				socket_error("recvfrom");
			else if (actually_process)
				process_input(data_in, len);
		}
		if (FD_ISSET(sockrtcpfd, &rfd)) {
			if ((len = recvfrom(sockrtcpfd, (char *)data_in,
				sizeof(data_in), 0, 0, 0)) <= 0)
				socket_error("recvfrom");
			else if (actually_process) {
				if (rtcp_check_rtcp_pkt(data_in, len))
					rtcp_decode_rtcp_pkt(data_in, len);
			}
		}
    }
	actually_process = 1;
	size = 0;		/* until we get timing correct */
    last_read_time = now;
#endif
  }
  return (int)output.size;
}

template<class oType> 
VrUDPSource<oType>::VrUDPSource(int port)
{
  tempbuf_index = 0;
  tempbuf_size = 0;
  my_addr.sin_family = AF_INET;         /* host byte order */
  my_addr.sin_port = htons(port);     /* short, network byte order */
  my_addr.sin_addr.s_addr = INADDR_ANY; /* auto-fill with my IP */
  bzero(&(my_addr.sin_zero), 8);        /* zero the rest of the struct */

  if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
    perror("socket");
    exit(1);
  }
  if (bind(sockfd, (struct sockaddr *)&my_addr, sizeof(struct sockaddr))  == -1) {
    perror("bind");
    exit(1);
  }
  my_addr.sin_port = htons(port+1);     /* short, network byte order */
  if ((sockrtcpfd = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
    perror("socket");
    exit(1);
  }
  if (bind(sockrtcpfd, (struct sockaddr *)&my_addr, sizeof(struct sockaddr))  == -1) {
    perror("bind");
    exit(1);
  }
	max_fd = sockfd;
	if (sockrtcpfd > max_fd)
		max_fd = sockrtcpfd;
}
static int rtp_header_validation(rtp_hdr_t *hdr, long *len, int *extlen)
{
	/* We only accept RTPv2 packets... */
	if (hdr->type != 2) {
		printf("rtp _header_validation: version != 2\n");
		return 0;
	}
	/* Check for valid audio payload types... */
	if (((hdr->pt > 23) && (hdr->pt < 33)) || ((hdr->pt > 71) && (hdr->pt < 77)) || (hdr->pt > 127)) {
		printf("rtp _header_validation: payload-type out of audio range\n");
		return 0;
	}
	/* If padding or header-extension is set, we punt on this one... */
	/* We should really deal with it though...                       */
	if (hdr->p) {
        int pad = *((unsigned char *)hdr + *len - 1);
        if (pad < 1) {
            printf("rtp _header_validation: padding but 0 len\n");
            return 0;
        }
        *len -= pad;
    }
    *extlen = 0;
    if (hdr->x)
        *extlen = *((unsigned long*)((unsigned char*)hdr + 4*(3+hdr->cc)))&0x0000ffff;
    /* Convert from network byte-order */
    hdr->seq = ntohs(hdr->seq);
    hdr->ts = ntohl(hdr->ts);
    hdr->ssrc = ntohl(hdr->ssrc); 
	return 1;
} 
static void init_seq(rtcp_dbentry *s, unsigned short seq)
{
	s->cycles = 0;
	s->lastseqno = seq;
	s->bad_seq = RTP_SEQ_MOD + 1;
	s->pckts_recv = 0;
	s->received_prior = 0;
	s->expected_prior = 0;
}
static void xxrtcp_update_seq(rtcp_dbentry *s, rtp_hdr_t *hdr) 
{
	unsigned short seq = hdr->seq;
	unsigned short udelta = seq - s->lastseqno;

	if (s->pckts_recv == 0) {
		s->probation = 0;
		init_seq(s, seq);
	} 
	if (s->probation) {
		/* Source is not valid until MIN_SEQUENTIAL packets with sequential
		 * sequence numbers have been received.  */
		/* packet is in sequence */
		s->lastseqno = seq;
		if (udelta != 1)
			s->probation = MIN_SEQUENTIAL - 1;
		else if (--s->probation == 0)
			goto sync_lab;
		return;
	} else if (udelta < MAX_DROPOUT) {
		/* in order, with permissible gap */
		if (seq < s->lastseqno) /* Sequence number wrapped, another 64K cycle.*/
			s->cycles += RTP_SEQ_MOD;
		s->lastseqno = seq;
	} else if (udelta <= RTP_SEQ_MOD - MAX_MISORDER) {
		/* the sequence number made a very large jump */
		if (seq == s->bad_seq)
			goto sync_lab;
		s->bad_seq = (seq + 1) & (RTP_SEQ_MOD - 1);
		return;
		}
	else if (udelta == 0)
        s->duplicates ++;
    else
        s->misordered ++;
	goto recv_lab;
sync_lab:
	init_seq(s, seq);
recv_lab:
	s->pckts_recv++;
}
void process_input(unsigned char *packet, long arg_len)
{
	rtcp_dbentry	     *sender;
	rtp_hdr_t	     *hdr;
	u_char		     *data_ptr;
	int len, extlen;
	unsigned long diff;
	int first_len;

	extlen = 0;
    hdr = (rtp_hdr_t *) (packet); 
    if (!rtp_header_validation(hdr, &arg_len, (int*)&extlen)) {
        printf("RTP Packet failed header validation!\n");
        return;
    }
printf ("update %lu time %lu\n", (long)hdr->ssrc, hdr->ts);
	sender = get_source(hdr->ssrc);
	if (!sender) {
        printf("Packet from unknown participant discarded\n");
        return;
    }
	sender->is_sender = 1;
	masterdb->pckts_received++;
    xxrtcp_update_seq(sender, hdr);
    data_ptr =  (unsigned char *)packet + 4 * (3 + hdr->cc) + extlen;
    len = arg_len - 4 * (3 + hdr->cc) - extlen;
	if (hdr->seq > sender->last_seq && sender->age > 4) {
		diff = 0xffff & (hdr->ts - sender->last_ts +
			((long) sender->last_ts < 0 && (long) hdr->ts > 0 ? 0x10000 : 0) );
printf ("[%s:%d] last %lx hdr %lx diff %lx\n", __FILE__, __LINE__, sender->last_ts, hdr->ts, diff);
memdump(data_ptr, len, "Rx");
		sender->inputp += diff;
		if (sender->inputp >= &sender->buffer[BUFFER_LENGTH])
			sender->inputp -= BUFFER_LENGTH;
		first_len = &sender->buffer[BUFFER_LENGTH] - sender->inputp;
		if (first_len > len)
			first_len = len;
		memcpy(sender->inputp, data_ptr, first_len);
		len -= first_len;
		if (len)
			memcpy(sender->buffer, data_ptr+first_len, len);
printf ("offset %d\n", sender->inputp - sender->buffer);
	}
	sender->last_ts = hdr->ts;
	sender->last_seq = hdr->seq;
	if (sender->age < 100)
		sender->age++;
}
 
static void rtcp_ntp_format(unsigned long * sec, unsigned long * frac)
{
	struct timeval  tv;
	unsigned long usec;

	gettimeofday(&tv, 0);
	*sec = tv.tv_sec + SECS_BETWEEN_1900_1970;
	usec = tv.tv_usec;
	*frac = (usec << 12) + (usec << 8) - ((usec * 3650) >> 6);
}

static int rtcp_check_rtcp_pkt(unsigned char *packet, int len)
{
	rtcp_t	*pkt = (rtcp_t *) packet;
	rtcp_t	*end = (rtcp_t *) (((char *) pkt) + len);
	rtcp_t	*r = pkt;
	int	 l = 0;
	int	 last = 0;

	/* All RTCP packets must be compound packets (RFC1889, section 6.1) */
	if (((ntohs(pkt->common.length) + 1) * 4) == len) {
		printf("Bogus RTCP packet: not a compound packet\n");
		return 0;
	}

	/* Check the RTCP version, payload type and padding of the first in  */
	/* the compund RTCP packet...                                        */
	if (pkt->common.type != 2) {
		printf("Bogus RTCP packet: version number != 2 in the first sub-packet\n");
		return 0;
	}
	if (pkt->common.p != 0) {
		printf("Bogus RTCP packet: padding bit is set, and this is the first packet in the compound\n");
		return 0;
	}
	if ((pkt->common.pt != RTCP_SR) && (pkt->common.pt != RTCP_RR)) {
		printf("Bogus RTCP packet: compund packet does not start with SR or RR\n");
		return 0;
	}

	/* Check all following parts of the compund RTCP packet. The RTP version */
	/* number must be 2, and the padding bit must be zero on all apart from  */
	/* the last packet.                                                      */
	do {
		if (r->common.type != 2) {
			printf("Bogus RTCP packet: version number != 2\n");
			return 0;
		}
		if (last == 1) {
			printf("Bogus RTCP packet: padding bit was set before the last packet in the compound\n");
			return 0;
		}
		if (r->common.p == 1) last = 1;
		l += (ntohs(r->common.length) + 1) * 4;
		r = (rtcp_t *) (((unsigned long *) r) + ntohs(r->common.length) + 1);
	} while (r < end);

	/* Check that the length of the packets matches the length of the UDP */
	/* packet in which they were received...                              */
	if ((r != end) || (l != len))  {
		printf("Bogus RTCP packet: length of RTCP packet does not match length of UDP packet\n");
		return 0;
	}

	return 1;
}

static int rtcp_add_sdes_item(unsigned char *buf, int type, char *val)
{
	rtcp_sdes_item_t *shdr = (rtcp_sdes_item_t *) buf;
	int             namelen;

	if (val == NULL) {
		printf("Cannot format SDES item. type=%d val=%p\n", type, val);
		return 0;
	}
	shdr->type = type;
	namelen = strlen(val);
	shdr->length = namelen;
	strcpy(shdr->data, val);
	return namelen + 2;
}

static unsigned char * rtcp_packet_fmt_sdes(unsigned char * ptr)
{
	rtcp_common_t  *hdr = (rtcp_common_t *) ptr;
	int             i, len;

len = 0;
	hdr->type = 2;
	hdr->p = 0;
	hdr->count = 1;
	hdr->pt = RTCP_SDES;
	*((unsigned long *) ptr + 1) = htonl(masterdb->myssrc);
	len = 8 + rtcp_add_sdes_item(&ptr[len], RTCP_SDES_CNAME, masterdb->my_dbe->sentry->cname);
	if ((++masterdb->sdes_pri_count % 3) == 0) {
		if ((++masterdb->sdes_sec_count % 8) == 0) {
			switch (++masterdb->sdes_ter_count % 4) {
			case 0 : if (masterdb->my_dbe->sentry->email) {
			         	len += rtcp_add_sdes_item(&ptr[len], RTCP_SDES_EMAIL, masterdb->my_dbe->sentry->email);
			  	 	break;
				 }
			case 1 : if (masterdb->my_dbe->sentry->phone) {
			           	len += rtcp_add_sdes_item(&ptr[len], RTCP_SDES_PHONE, masterdb->my_dbe->sentry->phone);
			  	   	break;
			 	 }
			case 2 : if (masterdb->my_dbe->sentry->loc) {
			           	len += rtcp_add_sdes_item(&ptr[len], RTCP_SDES_LOC, masterdb->my_dbe->sentry->loc);
			  	   	break;
			 	 }
			case 3 : len += rtcp_add_sdes_item(&ptr[len], RTCP_SDES_TOOL, masterdb->my_dbe->sentry->tool);
			}
		} else if (masterdb->my_dbe->sentry->name)
				len += rtcp_add_sdes_item(&ptr[len], RTCP_SDES_NAME, masterdb->my_dbe->sentry->name);
	}
	hdr->length = htons((unsigned short)(len / 4));
	for (i = len; i < ((int)(len / 4) + 1) * 4; i++)
		ptr[i] = 0;
	return ptr + 4 * ((int)(len / 4) + 1);
} 
/* Format a sender report packet, from the information available in the database.  */
static unsigned char * rtcp_packet_fmt_sr(unsigned char * ptr)
{
	rtcp_common_t  *hdr = (rtcp_common_t *) ptr;
	unsigned long		sec;
	unsigned long		frac;
	hdr->type = 2;
	hdr->p = 0;
	hdr->count = 0;
	hdr->pt = RTCP_SR;
	*((unsigned long *) ptr + 1) = htonl(masterdb->myssrc);
	rtcp_ntp_format(&sec, &frac);
	*((unsigned long *) ptr + 2) = htonl(sec);
	*((unsigned long *) ptr + 3) = htonl(frac);

	masterdb->map_rtp_time = 0; /* get_time(sp->device_clock); */
	*((unsigned long *) ptr + 4) = htonl(masterdb->map_rtp_time);
	*((unsigned long *) ptr + 5) = htonl(masterdb->pkt_count);
	*((unsigned long *) ptr + 6) = htonl(masterdb->byte_count);
	return ptr + 28;
}

/* * Create a recipient report header.  */
static unsigned char * rtcp_packet_fmt_rrhdr(unsigned char * ptr)
{
	rtcp_common_t  *hdr = (rtcp_common_t *) ptr;
	masterdb->map_rtp_time = 0; /* get_time(sp->device_clock); */
	hdr->type = 2;
	hdr->p = 0;
	hdr->count = 0;
	hdr->pt = RTCP_RR;
	*((unsigned long *) ptr + 1) = htonl(masterdb->myssrc);
	return ptr + 8;
}
static unsigned long xxntp_time32(void)
{
	struct timeval  tv;
	unsigned long sec, usec, frac;

	gettimeofday(&tv, 0);
	sec = tv.tv_sec + SECS_BETWEEN_1900_1970;
	usec = tv.tv_usec;
	frac = (usec << 12) + (usec << 8) - ((usec * 3650) >> 6);
	return (sec & 0xffff) << 16 | frac >> 16;
}

/* * Format a recipient report item, given the database item that this should * refer to.  */
static unsigned char * rtcp_packet_fmt_addrr(unsigned char * ptr, rtcp_dbentry * dbe)
{
	rtcp_rr_t      *rptr = (rtcp_rr_t *) ptr;

	rptr->ssrc = htonl(dbe->ssrc);
	rptr->loss = htonl(dbe->lost_frac << 24 | (dbe->lost_tot & 0xffffff));
	rptr->last_seq = htons((unsigned short)(0 /* dbe->cycles + dbe->lastseqno*/));
	rptr->jitter = htonl((unsigned long) 0);
	rptr->lsr = htonl(dbe->last_sr);
	rptr->dlsr = htonl(xxntp_time32() - dbe->last_sr_rx);
	return ptr + 24;
} 
static unsigned char * rtcp_packet_fmt_srrr(unsigned char *ptr)
{
	unsigned char *packet 	= ptr;
	rtcp_common_t *hdr    	= (rtcp_common_t *) ptr;
	rtcp_dbentry *sptr 	= NULL; /* masterdb->ssrc_db; */
	rtcp_dbentry *sptmp	= NULL;
	u_short packlen, offset	= 0;

	if (1 /* sending */) {
		ptr = rtcp_packet_fmt_sr(ptr);
	} else
		ptr = rtcp_packet_fmt_rrhdr(ptr);
	while (sptr) {
		sptmp = sptr->next;	/* We may free things below */
		if (/*now - sptr->last_active > expiry && */masterdb->my_dbe != sptr) {
			if (masterdb->myssrc != sptr->ssrc) {/* don't want to time out ourselves...  */
				printf ("delete db entry %ld\n", sptr->ssrc);
			}
		} else if (sptr->is_sender) {
			sptr->is_sender = 0;	/* Reset every report time */
			ptr = rtcp_packet_fmt_addrr(ptr, sptr);
			hdr->count++;
			packlen = ptr - packet;
			hdr->length = htons((unsigned short)((packlen - offset) / 4 - 1));
		}
		sptr = sptmp;
	}
	packlen = ptr - (unsigned char *) packet;
	hdr->length = htons((unsigned short)((packlen - offset) / 4 - 1));
	return ptr;
} 
static void rtcp_decode_rtcp_pkt(unsigned char *packet, int len)
{
unsigned long real_time = xxntp_time32();
	rtcp_t			*pkt = (rtcp_t *) packet;
	rtcp_dbentry		*dbe, *other_source;
	rtcp_sdes_item_t	*sdes;
	unsigned long			ssrc;
	unsigned long			*alignptr;
	int			i, lenstr;
	rtcp_user_rr            *rr, *tmp_rr;

	len /= 4;
	while (len > 0) {
		len -= ntohs(pkt->common.length) + 1;
		if (len < 0 || pkt->common.length == 0) {
			printf("Ignoring RTCP packet with weird format...\n");
			return;
		}
		switch (pkt->common.pt) {
		case RTCP_SR:
			ssrc = ntohl(pkt->r.sr.ssrc);
			dbe = find_source(ssrc); 
			/* Take note of mapping to use in synchronisation */
			dbe->mapping_valid = 1;
			dbe->last_ntp_sec = ntohl(pkt->r.sr.ntp_sec);
			dbe->last_ntp_frac = ntohl(pkt->r.sr.ntp_frac);
			dbe->last_rtp_ts = ntohl(pkt->r.sr.rtp_ts); 
			/* Update local clock map, need it for the sync [dm] */
			masterdb->map_rtp_time = 0; /* get_time(sp->device_clock);  */
			dbe->last_sr = ((dbe->last_ntp_sec & 0xffff) << 16) | (((dbe->last_ntp_frac & 0xffff0000) >> 16) & 0xffffffff);
			dbe->last_sr_rx = real_time; 
			/* Store the reception statistics for that user... */
			/* Clear the old RR list... */
			rr = dbe->rr;
			while (rr != NULL) {
				tmp_rr = rr->next;
				free(rr);
				rr = tmp_rr;
			}
			dbe->rr = NULL;
			/* Fill in the new RR list... */
			for (i = 0; i < pkt->common.count; i++) {
				rr = (rtcp_user_rr *) malloc(sizeof(rtcp_user_rr));
				rr->next = dbe->rr;
				rr->ssrc = ntohl(pkt->r.sr.rr[i].ssrc);
				rr->fraction_lost = (unsigned char)(ntohl(pkt->r.sr.rr[i].loss) >> 24);
				rr->pckts_lost = ntohl(pkt->r.sr.rr[i].loss) & 0x00ffffff;
				rr->ext_seq_num = ntohl(pkt->r.sr.rr[i].last_seq);
				rr->jitter = ntohl(pkt->r.sr.rr[i].jitter);
				rr->lsr = ntohl(pkt->r.sr.rr[i].lsr);
				rr->dlsr = ntohl(pkt->r.sr.rr[i].dlsr);
				dbe->rr = rr;
				other_source = find_source(rr->ssrc);
				/* is it reporting on my traffic? */
				if ((rr->ssrc == masterdb->myssrc) && dbe->sentry->cname) {
					/* Work out the round-trip-time... */
					printf("rtt %lx %f\n", ntohl(pkt->r.sr.ssrc), ((double) (real_time - rr->lsr - rr->dlsr)) / 65536.0);
				}
			}
			break;
		case RTCP_RR:
			ssrc = ntohl(pkt->r.rr.ssrc);
			dbe = find_source(ntohl(pkt->r.rr.ssrc));
			/* Store the reception statistics for that user... */
			/* Clear the old RR list... */
			rr = dbe->rr;
			while (rr != NULL) {
				tmp_rr = rr->next;
				free(rr);
				rr = tmp_rr;
			}
			dbe->rr = NULL;
			/* Fill in the new RR list... */
			for (i = 0; i < pkt->common.count; i++) {
				rr = (rtcp_user_rr *) malloc(sizeof(rtcp_user_rr));
				rr->next = dbe->rr;
				rr->ssrc = ntohl(pkt->r.rr.rr[i].ssrc);
				rr->fraction_lost = (unsigned char)(ntohl(pkt->r.rr.rr[i].loss) >> 24);
				rr->pckts_lost = ntohl(pkt->r.rr.rr[i].loss) & 0x00ffffff;
				rr->ext_seq_num = ntohl(pkt->r.rr.rr[i].last_seq);
				rr->jitter = ntohl(pkt->r.rr.rr[i].jitter);
				rr->lsr = ntohl(pkt->r.rr.rr[i].lsr);
				rr->dlsr = ntohl(pkt->r.rr.rr[i].dlsr);
				dbe->rr = rr;
				other_source =  find_source(rr->ssrc);
				/* is it reporting on my traffic? */
				if ((rr->ssrc == masterdb->myssrc) && (dbe->sentry->cname != NULL)) {
					/* Work out the round-trip-time... */
					printf("rtt %lx %f\n", ntohl(pkt->r.rr.ssrc), ((double) (real_time - rr->lsr - rr->dlsr)) / 65536.0);
				}
			} 
			break;
		case RTCP_BYE:
			for (i = 0; i < pkt->common.count; i++)
				printf ("bye: delete db entry %ld\n", ntohl(pkt->r.bye.src[i]));
			break;
		case RTCP_SDES:
			alignptr = (unsigned long *) pkt->r.sdes.s;
			for (i = 0; i < pkt->common.count; i++) {
				ssrc = ntohl(*alignptr);
				dbe = find_source(ssrc);
				sdes = (rtcp_sdes_item_t *) (alignptr + 1);
				while (sdes->type) {
					lenstr = sdes->length;
					sdes->length += 2;
					switch (sdes->type) {
					case RTCP_SDES_CNAME:
						if (dbe->sentry->cname) {
							if (strncmp(dbe->sentry->cname, sdes->data, lenstr) != 0)
								printf("CNAME change %d : [%s] --> [%s]\n", lenstr, dbe->sentry->cname, sdes->data);
							break;
						}
						dbe->sentry->cname = (char *) malloc(lenstr + 1);
						memcpy(dbe->sentry->cname, sdes->data, lenstr);
						dbe->sentry->cname[lenstr] = '\0';
						if (ssrc == masterdb->myssrc && masterdb->my_dbe->sentry->cname &&
								strcmp(dbe->sentry->cname, masterdb->my_dbe->sentry->cname)) {
							masterdb->myssrc = lrand48();
							masterdb->pkt_count = 0;
							masterdb->byte_count = 0;
						}
						break;
					case RTCP_SDES_NAME:
						if (dbe->sentry->name) {
							if (!strncmp(dbe->sentry->name, sdes->data, lenstr))
								break;
							free(dbe->sentry->name);
						}
						dbe->sentry->name = (char *) malloc(lenstr + 1);
						memcpy(dbe->sentry->name, sdes->data, lenstr);
						dbe->sentry->name[lenstr] = '\0';
						break;
					case RTCP_SDES_EMAIL:
						if (dbe->sentry->email) {
							if (!strncmp(dbe->sentry->email,sdes->data, lenstr))
								break;
							free(dbe->sentry->email);
						}
						dbe->sentry->email = (char *) malloc(lenstr + 1);
						memcpy(dbe->sentry->email, sdes->data, lenstr);
						dbe->sentry->email[lenstr] = '\0';
						break;
					case RTCP_SDES_PHONE:
						if (dbe->sentry->phone) {
							if (!strncmp(dbe->sentry->phone, sdes->data, lenstr))
								break;
							free(dbe->sentry->phone);
						}
						dbe->sentry->phone = (char *) malloc(lenstr + 1);
						memcpy(dbe->sentry->phone, sdes->data, lenstr);
						dbe->sentry->phone[lenstr] = '\0';
						break;
					case RTCP_SDES_LOC:
						if (dbe->sentry->loc) {
							if (!strncmp(dbe->sentry->loc, sdes->data, lenstr))
								break;
							free(dbe->sentry->loc);
						}
						dbe->sentry->loc = (char *) malloc(lenstr + 1);
						memcpy(dbe->sentry->loc, sdes->data, lenstr);
						dbe->sentry->loc[lenstr] = '\0';
						break;
					case RTCP_SDES_TOOL:
						if (dbe->sentry->tool) {
							if (!strncmp(dbe->sentry->tool, sdes->data, lenstr))
								break;
							free(dbe->sentry->tool);
						}
						dbe->sentry->tool = (char *) malloc(lenstr + 1);
						memcpy(dbe->sentry->tool, sdes->data, lenstr);
						dbe->sentry->tool[lenstr] = '\0';
						break;
					case RTCP_SDES_NOTE:
						if (dbe->sentry->note) {
							if (!strncmp(dbe->sentry->note, sdes->data, lenstr))
								break;
							free(dbe->sentry->note);
						}
						dbe->sentry->note = (char *) malloc(lenstr + 1);
						memcpy(dbe->sentry->note, sdes->data, lenstr);
						dbe->sentry->note[lenstr] = '\0';
						break;
					default:
						printf("SDES packet type %d ignored\n", sdes->type);
						break;
					}
					sdes = (rtcp_sdes_item_t *) ((unsigned char *) sdes + sdes->length);
				}
				while ((unsigned char *) sdes >= (unsigned char *) alignptr)
					alignptr++;	/* Next 32bit boundary */
			}
			break;
		case RTCP_APP:	
			break;
		default:
			break;
		}
		pkt = (rtcp_t *) ((unsigned long *) pkt + ntohs(pkt->common.length) + 1);
	}
}

#endif
