/* -*- Mode: c++ -*-
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */

#include "guppi.h"
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>

#define PAGE_SIZE 4096
#define SAMPLE_RATE 33000000

inline void pbufferpb(unsigned char *b, int index) {
  if(index!=0) {
    printf("Info: page boundary %u/%u: %2x %2x %2x %2x %2x %2x %2x %2x  %2x %2x %2x %2x %2x %2x %2x %2x\n",
	 index-1,index,
	 b[index*PAGE_SIZE-8],
	 b[index*PAGE_SIZE-7],
	 b[index*PAGE_SIZE-6],
	 b[index*PAGE_SIZE-5],
	 b[index*PAGE_SIZE-4],
	 b[index*PAGE_SIZE-3],
	 b[index*PAGE_SIZE-2],
	 b[index*PAGE_SIZE-1],
	 b[index*PAGE_SIZE],
	 b[index*PAGE_SIZE+1],
	 b[index*PAGE_SIZE+2],
	 b[index*PAGE_SIZE+3],
	 b[index*PAGE_SIZE+4],
	 b[index*PAGE_SIZE+5],
	 b[index*PAGE_SIZE+6],
	 b[index*PAGE_SIZE+7]);
  }
}

inline void pbuffer(unsigned char *b, int index) {
  if(index>=8) {
    printf("Info: area surrounding index %4x: %2x %2x %2x %2x %2x %2x %2x %2x  %2x %2x %2x %2x %2x %2x %2x %2x\n",
	   index,
	   b[index-8],
	   b[index-7],
	   b[index-6],
	   b[index-5],
	   b[index-4],
	   b[index-3],
	   b[index-2],
	   b[index-1],
	   b[index],
	   b[index+1],
	   b[index+2],
	   b[index+3],
	   b[index+4],
	   b[index+5],
	   b[index+6],
	   b[index+7]);
  }
}

int main(int argc, char **argv) {

  int gupfd;
  unsigned int bufferSize = 8000; //8000 pages = 32 MBytes = 1 second
  unsigned char *readbuf;
  unsigned char temp=0;

  //double check page size setting
  if(PAGE_SIZE != getpagesize()) {
    fprintf(stderr, "Page size incorrect, recompile.\n");
    exit(-1);
  }

  /* Read tests */
  if ((gupfd = open("/dev/guppi0", (O_RDONLY | O_NONBLOCK))) < 0) {
    perror("Fail: Can't open GuPPI for read");
    exit(-1);
  } else fprintf(stderr,"Sucess: open\n");

  if((ioctl(gupfd, GIOCSETBUFSIZE, bufferSize)) < 0) {
    perror("Failed to set buffersize");
  } else fprintf(stderr,"Sucess: set buffer size\n");

  //Check things that _shouldn't_ work
  if(mmap(0,bufferSize*PAGE_SIZE, PROT_READ|PROT_WRITE, MAP_SHARED, 
	  gupfd, 0)!=MAP_FAILED) {
    perror("Failure: mmap permitted PROT_WRITE!");
    exit(-1);
  } else fprintf(stderr,"Sucess: mmap protection test\n");   

  //do the real mmap
  if((readbuf=(unsigned char *) mmap(0,bufferSize*PAGE_SIZE, PROT_READ, MAP_SHARED, 
		   gupfd, 0))==MAP_FAILED) {
    perror("Failed to mmap data");
    exit(-1);
  } else fprintf(stderr,"Sucess: mmap\n");

  struct guppi_status status;

  //check status before receive
  status.num=0;
  if((ioctl(gupfd, GIOCSETGETSTATUS, &status)) < 0) {
    perror("Failed to get status");
    exit(-1);
  } else if(status.index != 0 || status.num != 0 || status.lost != 0) {
    fprintf(stderr,"Fail: status returns valid blocks (or overruns) before receive!\n");
    fprintf(stderr,"      the driver thinks the first index is %u\n", status.index);
    fprintf(stderr,"      num_valid=%u and num_lost=%u\n",status.num, status.lost);
    exit(-1);
  } else fprintf(stderr,"Sucess: status before receive\n");

  //Fault in all the pages by reading from them
  for(uint i=0;i<bufferSize;i++) {
    if((temp=readbuf[i*PAGE_SIZE])!=0) {
      fprintf(stderr,"Fail: buffer does not contain zeros (%d).\n",(int) temp);
      exit(-1);
    }
  }
  if(temp==0)
    fprintf(stderr,"Sucess: buffer faulted into user space\n");

  //Start receive
  if((ioctl(gupfd, GIOCSTART)) < 0) {
    perror("Failed to start receive");
  } else fprintf(stderr,"Sucess: start receive\n");

  //Print message as to whether we are verifing the data coming
  //from the guppi (using special daughter card)
  if(argc==1) {
    fprintf(stderr,"Warning: verifing received data -- be sure test daughter card is installed.\n");
  } else {
    fprintf(stderr,"Warning: not verifing received data!\n");
  }

  usleep(100000);

  int num=0;
  int synced=0;
  int position=0;
  status.num=0;
  uint i=0;
  for(uint x=0;x<10;x++) {
    i=i%bufferSize;
    //run through 20 buffers
    for(;i<bufferSize*20;) {

      /*free some blocks if appropriate*/
      unsigned int reserve=10+750*(i/bufferSize);
      if(reserve > bufferSize - 2048) //don't let the buf get too full
	reserve = bufferSize - 2048; //2048 = saftey in driver + transferring
      if(status.num > reserve) {
	status.num-=reserve;      
	//      fprintf(stderr,"Info: freeing %u blocks\n",status.num);
	num-=status.num;
	i+=status.num;
      } else
	status.num = 0;

      unsigned int currentindex = i % bufferSize;

      //free the blocks and find out how many blocks are ready
      if((ioctl(gupfd, GIOCSETGETSTATUS, &status)) < 0) {
	perror("Failed to get status");
	exit(-1);
      }
    
      if(status.index != currentindex) {
	fprintf(stderr,"Fail: current index should be %u, instead the driver thinks it's %u\n",currentindex, status.index);
	fprintf(stderr,"      num_valid=%u and num_lost=%u\n",status.num, status.lost);
	exit(-1);
      } else if(status.lost) {
	fprintf(stderr,"Warning: lost %u pages\n", status.lost);
      }
      if (status.num-num) {
	fprintf(stderr,"Info: got %u new pages (now holding %u pages at index %u)\n", status.num-num, status.num, status.index);

	/*verify data*/
	if(argc==1) {
	  //code for test daugther card
	  //special moc counts from 1-255, repeating
	  //the number 4095 times before incrementing it to the next
	  for(uint ii=status.index+num;ii<status.index+status.num;ii++) {
	    if(!synced) {
	      unsigned char first = readbuf[(ii%bufferSize)*PAGE_SIZE];
	      for(uint j=0;j<PAGE_SIZE;j++) {
		unsigned char c = readbuf[(ii%bufferSize)*PAGE_SIZE+j];
		if(!synced) {
		  if(c!=first) {
		    position = j;
		    synced=c;
		  }
		}
		if((synced && ((first!=255 && c!=first+1) ||
			       (first==255 && c!=1))) ||
		   first==0 || c==0) {
		  synced=0;
		  fprintf(stderr,"Fail: page %u does not contain the correct data (could not sync).\n",ii%bufferSize);
		  if(ii<bufferSize-1)
		    pbuffer(readbuf,(ii%bufferSize)*PAGE_SIZE+j);
		  break;
		}
	      }
	      /*	    if(synced)
			    fprintf(stderr,"Info: synced on page %u w/ transition to value %2x at position %4x.\n",ii%bufferSize, synced, position);*/
	    } else {
	      unsigned char c = readbuf[(ii%bufferSize)*PAGE_SIZE+position];
	      int nextsynced;
	      if(synced==255) {
		nextsynced=1;
	      } else {
		nextsynced=synced+1;
	      }

	      //check the first and last bytes and the bytes before and after
	      //the transition
	      if(c!=nextsynced ||
		 (position!=0 &&
		  (readbuf[(ii%bufferSize)*PAGE_SIZE] != synced ||
		   readbuf[(ii%bufferSize)*PAGE_SIZE+position-1] != synced ||
		   readbuf[(ii%bufferSize)*PAGE_SIZE+(PAGE_SIZE-1)]!=nextsynced)) ||
		 (position==0 &&
		  (readbuf[(ii%bufferSize)*PAGE_SIZE] != nextsynced ||
		   readbuf[(ii%bufferSize)*PAGE_SIZE+(PAGE_SIZE-2)]!=nextsynced ||
		   readbuf[(ii%bufferSize)*PAGE_SIZE+(PAGE_SIZE-1)]!=
		   (nextsynced==255 ? 1 : nextsynced+1)))) {
		fprintf(stderr,"Fail: page %u does not contain the correct data (expected %2x to %2x at %3x).\n",ii,synced,nextsynced,position);
		if(ii<bufferSize-1)
		  pbuffer(readbuf,(ii%bufferSize)*PAGE_SIZE+position);
		synced=0;
	      } else {
		if(position==0) {
		  synced=(nextsynced+1)%256;
		  position=4095;
		} else {
		  synced=nextsynced;
		}
	      }
	    } 
	    position = position-1;
	    if(position < 0) position=4095;
	  }
	  synced=0;
	} else {
	  pbufferpb(readbuf,status.index);
	  /*	if(status.index!=0)
		printf("page boundary %u/%u: %2x %2x %2x %2x %2x %2x %2x %2x  %2x %2x %2x %2x %2x %2x %2x %2x\n",
		status.index-1,status.index,
		readbuf[status.index*PAGE_SIZE-8],
		readbuf[status.index*PAGE_SIZE-7],
		readbuf[status.index*PAGE_SIZE-6],
		readbuf[status.index*PAGE_SIZE-5],
		readbuf[status.index*PAGE_SIZE-4],
		readbuf[status.index*PAGE_SIZE-3],
		readbuf[status.index*PAGE_SIZE-2],
		readbuf[status.index*PAGE_SIZE-1],
		readbuf[status.index*PAGE_SIZE],
		readbuf[status.index*PAGE_SIZE+1],
		readbuf[status.index*PAGE_SIZE+2],
		readbuf[status.index*PAGE_SIZE+3],
		readbuf[status.index*PAGE_SIZE+4],
		readbuf[status.index*PAGE_SIZE+5],
		readbuf[status.index*PAGE_SIZE+6],
		readbuf[status.index*PAGE_SIZE+7]);*/
	}
      }
      num=status.num;

      //slow down a bit
      unsigned long delay = bufferSize*2/3-reserve;
      if(reserve < bufferSize*2/3) {
	delay *=1024;
	delay = (unsigned long) (delay*1000000.0/SAMPLE_RATE);
	usleep(delay);
      }
    }
  }

  //Free wrong blocks
  status.index=1;
  status.num=1;
  if((ioctl(gupfd, GIOCSETGETSTATUS, status)) == 0) {
    perror("Fail: invalid free block request accepted by driver");
  } else fprintf(stderr,"Sucess: bad free test\n");

  //Try an overrun
  sleep(3);
  status.num=0;
  if((ioctl(gupfd, GIOCSETGETSTATUS, &status)) < 0) {
    perror("Failed to get status");
  }
  if(status.lost == 0) {
    fprintf(stderr, "Fail: lost block count incorrect.\n");
  } else {
    fprintf(stderr, "Sucess: lost block test: %u pages lost in 3 seconds.\n",
	    status.lost);
  }

  //Stop receive
  if((ioctl(gupfd, GIOCSTOP)) < 0) {
    perror("Failed to stop receive");
  } else fprintf(stderr,"Sucess: stop receive\n");

  //Munmap and Close
  if(munmap(readbuf,bufferSize*PAGE_SIZE)<0) {
    perror("Failed to munmap buffer");
  } else fprintf(stderr,"Sucess: munmap\n");
  if(close(gupfd)<0) {
    perror("Failed to close fd");
  } else fprintf(stderr,"Sucess: close\n");

}

