/* -*- Mode: c++ -*- 
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VRPCVFOSINK_H_
#define _VRPCVFOSINK_H_

extern "C" {
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
}

#include <VrGuppiSink.h>
#include <guppi.h>

template<class iType> 
class VrPcvfoSink: public VrGuppiSink<iType>  {
private:
  int pcvfofd;
  float txFrequency;
public: 
  virtual void work(int units) {VrGuppiSink<iType>::work(units);}
  virtual void initialize();
  void setTxFrequency(float f);
  VrPcvfoSink(int b);  
  ~VrPcvfoSink();  
};

template<class iType> void
VrPcvfoSink<iType>::setTxFrequency(float f)
{
  txFrequency = f;

  int retval;
  unsigned char code;
  if ((retval = lseek(pcvfofd, 0x210, SEEK_SET)) != 0x210) {
    fprintf(stderr, "VrPcvfoSink: Error moving to proper IO port, errno = %d\n", errno);
    exit(1);
  }
  code = 0x0;
  if ((retval = write(pcvfofd, &code, 1)) < 0) {
    fprintf(stderr, "VrPcvfoSink: Error writing to PCVFO, errno = %d\n", errno);
    exit(1);
  }
  usleep(10);

  int i;
  unsigned long delta;
  delta = (unsigned long) (.5 + txFrequency*((float)(1<<31)*2)/125000000.0);
  for (i = 3; i >= 0; i--) {
    if ((retval = lseek(pcvfofd, 0x210, SEEK_SET)) != 0x210) {
      fprintf(stderr, "VrPcvfoSink: Error moving to proper IO port, errno = %d\n", errno);
      exit(1);
    }
    code = delta >> i * 8;
    if ((retval = write(pcvfofd, &code, 1)) < 0) {
      fprintf(stderr, "VrPcvfoSink: Error writing to PCVFO, errno = %d\n", errno);
      exit(1);
    }
    usleep(10);
  }
  code = 0x0;
  if ((retval = write(pcvfofd, &code, 1)) < 0) {
    fprintf(stderr, "VrPcvfoSink: Error writing to PCVFO, errno = %d\n", errno);
    exit(1);
  }
}
 
template<class iType> void
VrPcvfoSink<iType>::initialize()
{
  VrGuppiSink<iType>::initialize();
}

template<class iType> 
VrPcvfoSink<iType>::VrPcvfoSink(int b)
  :VrGuppiSink<iType>(b), txFrequency(10000000)
{
  if ((pcvfofd = open("/dev/port", O_WRONLY)) < 0) {
    fprintf(stderr, "VrPcvfoSink: Error opening VFO device, errno = %d\n", errno);
    exit(1);
  }
  int retval;
  unsigned char code;
  if ((retval = lseek(pcvfofd, 0x210, SEEK_SET)) != 0x210) {
    fprintf(stderr, "VrPcvfoSink: Error moving to proper IO port, errno = %d\n", errno);
    exit(1);
  }

  // Must write 0,0 to 0x210, 0x211 to turn on the PCVFO output.

  code = 0x0;
  if ((retval = write(pcvfofd, &code, 1)) < 0) {
    fprintf(stderr, "VrPcvfoSink: Error writing to PCVFO, errno = %d\n", errno);
    exit(1);
  }
  usleep(10);
  if ((retval = write(pcvfofd, &code, 1)) < 0) {
    fprintf(stderr, "VrPcvfoSink: Error writing to PCVFO, errno = %d\n", errno);
    exit(1);
  }
}

template<class iType> 
VrPcvfoSink<iType>::~VrPcvfoSink(int b)
{
  int retval;
  unsigned char code;
  if ((retval = lseek(pcvfofd, 0x210, SEEK_SET)) != 0x210) {
    fprintf(stderr, "VrPcvfoSink: Error moving to proper IO port, errno = %d\n", errno);
    exit(1);
  }

  // Must write 4,0 to 0x210, 0x211 to turn off the PCVFO output.

  code = 0x4;
  if ((retval = write(pcvfofd, &code, 1)) < 0) {
    fprintf(stderr, "VrPcvfoSink: Error writing to PCVFO, errno = %d\n", errno);
    exit(1);
  }
  code = 0x0;
  if ((retval = write(pcvfofd, &code, 1)) < 0) {
    fprintf(stderr, "VrPcvfoSink: Error writing to PCVFO, errno = %d\n", errno);
    exit(1);
  }
  close(pcvfofd);
}

#endif










