/* -*- Mode: c++ -*- */
/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/*
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <VrSigProc.h>
#include <VrBuffer.h>
#include <VrConnect.h>
#ifdef HAVE_SYS_IPC_H
#include <sys/ipc.h>
#endif
#ifdef HAVE_SYS_SHM_H
#include <sys/shm.h>
#endif
#ifndef SHMMAX
//#include <asm/shmparam.h> //for SHMMAX
//from asm/shmparam.h
#define SHMMAX 0x2000000		/* max shared seg size (bytes) */
#endif /* SHMMAX */
#include <gr_math.h>

double 
VrBuffer::buffer_getSamplingFrequency()
{
  return upstream->getSamplingFrequency();
}

VrSampleIndex
VrBuffer::updateMinRP()
{
  VrSampleIndex newminRP = connectors[0]->minRP();

  //jca printf ("updateMinRP[%s]: from %s=%lld\n",
  //	upstream->name(), connectors[0]->getUpstreamModule()->name(), newminRP);

  unsigned int i=0;
  while(++i<numberConnectors) {
    VrSampleIndex t=connectors[i]->minRP();
    
    //jca printf ("updateMinRP[%s]: from %s=%lld\n",
    //	upstream->name(), connectors[i]->getUpstreamModule()->name(), t);

    if(t < newminRP)
      newminRP = t;
  }
  
  //It's OK that this has no mutual exclusion, since minRP
  //  is just a hint and will always eventually become correct
  //  (i.e. this->minRP <= min(connectors->minRP))
  if(newminRP > minRP) {

    //jca printf ("updateMinRP[%s]: change from %lld=%lld\n",
    //	upstream->name(), minRP, newminRP);

    minRP = newminRP;
  }

  return minRP;
}

//called from downstream module
//assumes data is there...
/*Pointer could be anywhere in either section of shared memory*/

char* VrBuffer::getReadPointer(VrSampleRange r) {

  if (PARANOID){
    if(!upstream->dataReady(r)) {
      fprintf (stderr, "getReadPointer: trying to read data that hasn't been written.\n");
      abort ();
    }
    if(bufferDupStart==NULL) {
      //buffer never created
      fprintf(stderr,"Buffer never created -- make sure setup is run on sinks downstream from all outputs of %s\n",
	      upstream->name());
      abort ();
    }
  }

  long long offset = r.index - WPdupstart;

  if (PARANOID){
    if(offset<- (long long) bufferSize || r.index < minRP) {
      fprintf (stderr, "VrBuffer(%p): read VrSampleIndex too small =%lld\n", this, r.index);
      abort ();
    }
    if(offset+r.size > bufferSize || r.index+r.size > upstream->getMarkedWP()) {
      fprintf (stderr, "VrBuffer(%p): read VrSampleIndex+size too large =%lld\n", this, r.index+r.size);
      abort ();
    }
  }
  
  return bufferDupStart+offset * type_size; 
}

//called from upstream module
/* Always writes at least part of the block to 
   "duplicate" (or 2nd) shared memory section */
char*
VrBuffer::getWritePointer(VrSampleRange r) {

  MUTEX_LOCK (&dupstartlock);

  long long offset = r.index - WPdupstart;

  //Buffer Wrap
  while(offset+r.size > bufferSize ) {
    offset -= bufferSize;
    WPdupstart += bufferSize;
  }

  MUTEX_UNLOCK (&dupstartlock);

  if (PARANOID){
    if(offset < - (long long) bufferSize) {
      fprintf (stderr, "VrBuffer(%p): WP VrSampleIndex too small: %lld, offset: %lld, bufferSize: %d, dup: %lld\n",
	       this, r.index, offset, bufferSize, WPdupstart);
      abort ();
    }
  }

  if(writespace(r.index+r.size)<0) {
    //markData should catch this
    fprintf(stderr, "VrBuffer: Buffer full!\n");
    abort ();
  } 

  return bufferDupStart+offset * type_size; 
}


void
VrBuffer::setup(unsigned int arg_type_size)
{
  if(bufferDupStart==NULL) {
    //buffer has not been created yet.
    allocateBuffer(arg_type_size);
    fprintf(stderr, "%s output bufferSize = %d (%d KB, %f sec)\n",
	    upstream->name(), bufferSize,
	    bufferSize * arg_type_size / 1024,
	    bufferSize/upstream->getSamplingFrequency());
  }
  else {
    fprintf(stderr,"Topology cannot be changed after inital setup.\n");
    abort ();
  }
}

void
VrBuffer::connect_buffer(VrConnect *c)
{
  VrConnect ** cs = new (VrConnect *[numberConnectors+1]);
  cs[numberConnectors]=c;
  if(numberConnectors>0) {
    for(unsigned int i=0;i<numberConnectors;i++)
      cs[i]=connectors[i];
    delete [] connectors;
  }
  connectors=cs;
  numberConnectors++;
  if(bufferDupStart!=NULL){
    fprintf(stderr,"Topology cannot be changed after inital setup.\n");
    abort ();
  }
}      

/*
 * Compute the minimum buffer_size that works (i.e., address space wrap-around works)
 * To work is to satisfy this contraint for integer buffer_size and k:
 *
 *     type_size * buffer_size == k * page_size
 */
static long
minimum_buffer_size (long type_size, long page_size)
{
  return page_size / gr_gcd (type_size, page_size);
}

void
VrBuffer::allocateBuffer(unsigned int arg_type_size) 
{ 
#ifdef HAVE_SYS_IPC_H
  type_size = arg_type_size;

  long preferred_buffer_size;
  preferred_buffer_size = upstream->getMaxOutputSize();
  preferred_buffer_size *= numberConnectors; 	 // each connector might want its own data
  preferred_buffer_size *= 2; 			 // hack (FIX?)
  preferred_buffer_size *= BufferSizeMultiplier; // BufferSizeMultiplier usually = nthreads + 1

  // Any buffersize we come up with must be a multiple of min_buffer_size.
  // Round up the preferred_buffer_size to a multiple of min_buffer_size

  long min_buffer_size = minimum_buffer_size (type_size, PAGE_SIZE);

  if (preferred_buffer_size % min_buffer_size != 0)
    preferred_buffer_size = ((preferred_buffer_size / min_buffer_size) + 1) * min_buffer_size;

  bufferSize = preferred_buffer_size;
  unsigned int bufferSize_bytes = bufferSize * type_size;

  //Check max size
  if (2 * bufferSize_bytes > SHMMAX) {
    fprintf (stderr,"Warning: could not allocate more than %d bytes for shared memory buffer.\n",SHMMAX);
    // grab the biggest we can
    long n = SHMMAX/2 / (min_buffer_size * type_size);
    assert (n >= 1);
    bufferSize = n * min_buffer_size;
    bufferSize_bytes = bufferSize * type_size;
  }

  //Allocate a 2N block of shared memory, then deallocate it
  //  and allocate a N block mapped twice into the same location
  //  the 2N block was mapped into.
  int shmid2 = 0;
  int shmid = 0;

  char *start = NULL;
  if ((shmid2 = shmget (IPC_PRIVATE, 2*bufferSize_bytes, IPC_CREAT | 0700)) == -1||
      (shmid = shmget (IPC_PRIVATE, bufferSize_bytes, IPC_CREAT | 0700)) == -1 ||
      (start  = (char *) shmat (shmid2, 0, 0)) == (char *) -1) {
    // fprintf (stderr, "shmid2 = %d, shmid = %d, start = %p\n", shmid2, shmid, start);
    perror("Could not allocate shared memory (1)");
    exit(-1);
    return;
  }

  shmdt ((char *) start);
  shmctl (shmid2, IPC_RMID, 0);
  if(shmat (shmid, start, 0) == (char *) -1 ||
     shmat (shmid, start+bufferSize_bytes, 0) == (char *) -1) {
    perror("Could not allocate shared memory (2)");
    exit(-1);
    return;
  }

  shmctl(shmid, IPC_RMID, 0);
  bufferDupStart=((char *)start)+bufferSize * type_size;
  //  cerr << upstream->name() <<" bufferDupStart = "<<(void *) bufferDupStart<<endl;
#endif
}

VrBuffer:: VrBuffer(VrSigProc* m)   
  :bufferDupStart(NULL),bufferSize(0), WPdupstart(0), minRP(0),
   upstream(m),connectors(NULL),numberConnectors(0),
   setupCalled(0)
{
#ifdef HAVE_SYS_IPC_H
  if (PAGE_SIZE != getpagesize()) {
    fprintf(stderr,"The PAGE_SIZE #define is not set to the actual page size. Recompile.\n");
    exit(-1);
  }
#else
  fprintf (stderr, "VrBuffer requires HAVE_SYS_IPC_H\n");
  exit (1);
#endif

  MUTEX_INIT (&dupstartlock);
}
