/* -*- Mode: c++ -*- */
/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/*
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */

#ifndef _GRAUDIOSOURCE_H_
#define _GRAUDIOSOURCE_H_

#define AUDIOINCHUNKSIZE 200

extern "C" {
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
}

#include <VrSource.h>
#include <fstream>
#include <algorithm>
#include <string>

template<class oType>
class GrAudioSource : public VrSource<oType> {
protected:
  int         audiofd;
  long        format;
  std::string device_name;
  oType	      scale_factor;
  int 	      num_streams;
public: 
  virtual int work2(VrSampleRange output, void *o[]);
  virtual void initialize();

  GrAudioSource(double sampling_freq, int streams, double output_range = 32767,
		const char* dev = "/dev/dsp")
    : VrSource<oType>(streams),audiofd(-1), device_name(dev),num_streams(streams) {
    setSamplingFrequency (sampling_freq);
    setOutputSize (AUDIOINCHUNKSIZE);
    scale_factor = (oType) (32767 / output_range);
  }

  virtual const char *name() { return "GrAudioSource"; }

  virtual ~GrAudioSource() {close(audiofd);}
};


template<class oType> void
GrAudioSource<oType>::initialize() 
{
  int temp = 0x7fff0004;
  if(audiofd==-1) {
    if ((audiofd = open(device_name.c_str(), O_RDONLY)) < 0) {
      cerr << "GrAudioSource: ";
      perror (device_name.c_str ());
      exit(1);
    }
    if ((ioctl(audiofd,SNDCTL_DSP_SETFRAGMENT,&temp)) < 0) {
      fprintf (stderr, "GrAudioSource: set fragment returned %d\n", errno);
      exit(1);
    }
  }
  
  // ioctl(audiofd,SNDCTL_DSP_RESET);
  
  int format=AFMT_S16_NE;
  int origf=format;
  if((ioctl(audiofd,SNDCTL_DSP_SETFMT,&format)) < 0) {
    cerr << "GrAudioSource: " << device_name << " IOCTL failed with errno " << errno << "\n";
    exit(1);
  }
  if(origf!=format) {
    fprintf (stderr, "GrAudioSource: Warning: unable to support format %d\n", origf);
    fprintf (stderr, "  card requested %d instead.\n", format);
  }
  
  // set number of channels no matter what.  Some hardware only does stereo
  int channels = 2;
  if (ioctl (audiofd, SNDCTL_DSP_CHANNELS, &channels) < 0){
    perror ("GrAudioSink: SNDCTL_DSP_CHANNELS failed");
    exit (1);
  }
  
  if (channels != 2){
    fprintf(stderr, "GrAudioSource: could not set STEREO mode\n");
    exit(1);
  }
  
  //  if(getInputSamplingFrequencyN(0)!=getInputSamplingFrequencyN(1))
  //    fprintf(stderr, "GrAudioSink Warning: left and right at different freq\n");

  int sf = (int)getSamplingFrequency();
  cerr <<"GrAudioSource: Sampling frequency = "<<sf<<endl;

  if ((ioctl(audiofd,SNDCTL_DSP_SPEED,&sf)) < 0) {
    cerr << device_name << ": Invalid sampling frequency...defaulting to 8 Khz\n";
    sf = 8000;
    if ((ioctl(audiofd,SNDCTL_DSP_SPEED,&sf)) < 0) {
      fprintf (stderr, "Couldn't even manage that...aborting\n");
      exit(1);
    }
  }
  if(sf!=getSamplingFrequency())
    fprintf(stderr, "GrAudioSource Warning: soundcard defaulted to %d Hz\n", sf);
}


template<class oType> int 
GrAudioSource<oType>::work2(VrSampleRange output, void *ao[])
{
  sync (output.index);		// force in-order execution

  unsigned size = output.size;
  oType **o = (oType **)ao;
  
  while(size > 0) {
    const unsigned N = 2048;
    short tmp[N*2];
    int nbytes =  std::min (size, N) * (sizeof(short) * 2);
    
    int count = read(audiofd, tmp, nbytes);
    
    if(count < 0) {
      perror ("GrAudioSource");
      exit(1);
    }
    
    assert ((count & ((sizeof(short) * 2) - 1)) == 0);
    
    if(num_streams == 1)
      for (unsigned int i = 0; i < count / (sizeof (short) * 2); i++) {
	o[0][i] = (oType)tmp[2*i]/scale_factor;
      }
    else	
      for (unsigned int i = 0; i < count / (sizeof (short) * 2); i++) {
	o[0][i] = (oType)tmp[2*i]/scale_factor;
	o[1][i] = (oType)tmp[2*i+1]/scale_factor;
      }
    
    size -= count / (sizeof (short) * 2);
    o[0] += count / (sizeof (short) * 2);
    if(num_streams==2)
      o[1] += count / (sizeof (short) * 2);
  }
  return output.size;
}

template<> int 
GrAudioSource<VrComplex>::work2(VrSampleRange output, void *ao[])
{
  unsigned size = output.size;
  VrComplex **o = (VrComplex **)ao;
  
  while(size > 0) {
    const unsigned N = 2048;
    short tmp[N*2];
    int nbytes =  std::min (size, N) * (sizeof(short) * 2);
    
    int count = read(audiofd, tmp, nbytes);
    
    if(count < 0) {
      perror ("GrAudioSource");
      exit(1);
    }
    
    assert ((count & ((sizeof(short) * 2) - 1)) == 0);
    
    for (unsigned int i = 0; i < count / (sizeof (short) * 2); i++)
      o[0][i] = VrComplex (tmp[2*i],tmp[2*i+1]) / scale_factor;
    
    size -= count / (sizeof (short) * 2);
    o[0] += count / (sizeof (short) * 2);
  }
  return output.size;
}

#endif
