/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _GRCOSTASLOOP_H_
#define _GRCOSTASLOOP_H_

#include <VrSigProc.h>
#include <gr_fir.h>
#include <gr_iir.h>
#include <gr_firdes.h>
#include <gr_nco.h>

template<class iType,class oType> 
class GrCostasLoop : public VrSigProc
{
 protected:
  gr_fir<iType,iType,float> *ifilter;
  gr_fir<iType,iType,float> *qfilter;
  gr_iir<double,double,double> *loopfilter;
  vector<double>fftaps;
  vector<double>fbtaps;
  gr_nco<iType,oType> *nco;
  unsigned int decimate;
  double sensitivity;
  double freq;
  double Fc;
  double arg;
  double argInc;
  double phi;
  double phase;
  virtual void initialize();
 public: 
  virtual const char *name() { return "GrCostasLoop"; }
  virtual int work(VrSampleRange output, void *ao[],
		   VrSampleRange inputs[], void *ai[]);
  virtual ~GrCostasLoop()
  {
	delete ifilter;
	delete qfilter;
	delete loopfilter;
	delete nco;
  }
  GrCostasLoop(int d, double f,double s,double Fc) 
	: VrSigProc(1,sizeof(iType),sizeof(oType)),
  	decimate(d),sensitivity(s),freq(f),Fc(Fc),arg(0) { }
};

template<class iType,class oType> void
GrCostasLoop<iType,oType>::initialize()  
{
	// Create all three filters
  double Fs =  getInputSamplingFrequencyN (0);
  vector<float> lpf_coeffs = 
	  gr_firdes::low_pass (1.0, Fs, Fc, Fc/10,gr_firdes::WIN_HAMMING,0);
  ifilter = new gr_fir<iType,iType,float>(lpf_coeffs);
  qfilter = new gr_fir<iType,iType,float>(lpf_coeffs);
  loopfilter = new gr_iir<double,double,double>(fftaps,fbtaps);
  nco = new gr_nco<float,float>();
  argInc = 2*M_PI*freq*(1 / (double)getInputSamplingFrequencyN(0));
}

template<class iType,class oType> int
GrCostasLoop<iType,oType>::work(VrSampleRange output, void *ao[],
		VrSampleRange inputs[], void *ai[])
{
  iType **i = (iType**)ai;
  oType **o = (iType**)ao;
  int size = output.size/decimate;

  float alpha=0.01;
  float beta=0.01;
  iType i_path_fifo[size];
  iType q_path_fifo[size];
  iType i_filtered[size];
  iType q_filtered[size];
  iType phase_det_out[size];
  VrComplex phase_corr;

  for (int j = 0; j < size; j++)
  {
	//mix incoming signal with cos and -sin of NCO
	for(int k = 0; k<decimate;k++)
	{
		i_path_fifo[j*decimate+k] = i[0][j*decimate+k] * nco->get_phase().real();
		q_path_fifo[j*decimate+k] = i[0][j*decimate+k] * -nco->get_phase().imag();
		nco->step();
	}
	//  Call both FIR filters.
	i_filtered[j]=ifilter->filter (i_path_fifo);
	q_filtered[j]=qfilter->filter (q_path_fifo);
	//  Then do multiply or ATAN
	phase_det_out[j] = atan2(q_filtered[j],i_filtered[j]);
	//  Call IIR filter for Loop filter
	// loopfilter.filter(nco_control,phase_det_out,size,decimate);
	//  Adjust phase and frequency
	phase_corr.real(cos(alpha*phase_det_out[j]));
	phase_corr.imag(sin(alpha*phase_det_out[j]));
	nco->rotate_phase(phase_corr);
	nco->delta_freq(beta*phase_det_out[j]);
	o[0][j] = 0;
  }
  //  Write output info, store state if necessary
  return output.size;
}

#endif
