/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2025 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA.
 *
 * otk - the osd toolkit
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <errno.h>
#include <pthread.h>

#include "../common/utils.h"

#include "../common.h"
#include "../xine-toolkit/dlist.h"
#include "../xine-toolkit/xitk.h"
#include "oxine.h"
#include "odk.h"
#include "otk.h"
/* #include "list.h" */
#include "utils.h"

/* seems safe now but does not really update the contents yet. */
#undef ENABLE_UPDATE_JOB

/*
#define LOG
*/

#define OTK_WIDGET_WINDOW    (1)
#define OTK_WIDGET_BUTTON    (2)
#define OTK_WIDGET_LABEL     (3)
#define OTK_WIDGET_LIST      (4)
#define OTK_WIDGET_LISTENTRY (5)
#define OTK_WIDGET_SLIDER    (6)
#define OTK_WIDGET_SELECTOR  (7)
#define OTK_WIDGET_LAYOUT    (8)
#define OTK_WIDGET_SCROLLBAR (9)

#define OTK_BUTTON_TEXT   (1)
#define OTK_BUTTON_PIXMAP (2)

typedef struct otk_button_s otk_button_t;
typedef struct otk_label_s otk_label_t;
typedef struct otk_window_s otk_window_t;
typedef struct otk_list_s otk_list_t;
typedef struct otk_listentry_s otk_listentry_t;
#ifdef OTK_WITH_SLIDER
typedef struct otk_slider_s otk_slider_t;
#endif
typedef struct otk_scrollbar_s otk_scrollbar_t;
typedef struct otk_selector_s otk_selector_t;
typedef struct otk_layout_s otk_layout_t;

/** TJ. Yes I know my xine_sarray _is_ faster, but I wanted to see what plain xitk_dlist can do :-) */
#undef SIZED_LIST_DEBUG

static void _sized_list_update_samples (sized_list_t *list) {
  int u;
  list->last_n[0] = list->list.head.next;
  list->last_i[0] = 0;
  if (list->num > 1) {
    for (u = 0; u < (1<<SIZED_LIST_SAMPLE_LD); u++) {
      int v = (list->num * (2 * u + 1)) >> (SIZED_LIST_SAMPLE_LD + 1);
      if (list->last_i[1 + u] < v) {
        do {
          list->last_n[1 + u] = list->last_n[1 + u]->next;
          list->last_i[1 + u]++;
        } while (list->last_i[1 + u] < v);
      } else if (list->last_i[1 + u] > v) {
        do {
          list->last_n[1 + u] = list->last_n[1 + u]->prev;
          list->last_i[1 + u]--;
        } while (list->last_i[1 + u] > v);
      }
#ifdef SIZED_LIST_DEBUG
      printf ("--------- _sized_list_update_samples (%d) = (%d, %p).\n", u + 1, list->last_i[1 + u], (void *)list->last_n[1 + u]);
#endif
    }
  } else {
    for (u = 0; u < (1<<SIZED_LIST_SAMPLE_LD); u++) {
      list->last_n[1 + u] = list->list.head.next;
      list->last_i[1 + u] = 0;
    }
  }
  list->last_n[(1<<SIZED_LIST_SAMPLE_LD)+1] = &list->list.tail;
  list->last_i[(1<<SIZED_LIST_SAMPLE_LD)+1] = list->num;
}

void sized_list_init (sized_list_t *list) {
  int u;
  xitk_dlist_init (&list->list);
  for (u = 0; u < (1<<SIZED_LIST_SAMPLE_LD) + 2; u++) {
    list->last_n[u] = &list->list.tail;
    list->last_i[u] = 0;
  }
  list->num = 0;
}

static void sized_list_append (sized_list_t *list, xitk_dnode_t *node) {
  xitk_dlist_add_tail (&list->list, node);
  list->num++;
  _sized_list_update_samples (list);
}

xitk_dnode_t *sized_list_get (sized_list_t *list, int i) {
  xitk_dnode_t *node;
  int u;
  if (list->num <= 0)
    return &list->list.tail;
  if (i < 0)
    i = 0;
  else if (i > list->num)
    i = list->num;
  u = (((1<<SIZED_LIST_SAMPLE_LD) + 1) * i + (list->num >> 1)) / list->num;
#ifdef SIZED_LIST_DEBUG
  printf ("--------- sized_list_get (%d of %d): found sample [%d] = %d.\n", i, list->num, u, list->last_i[u]);
#endif
  node = list->last_n[u];
  u = list->last_i[u];
  if (u > i) {
    /* back from sample */
    do {
#ifdef SIZED_LIST_DEBUG
      printf ("--------- sized_list_get (%d of %d): back from %d.\n", i - 1, list->num, u);
#endif
      node = node->prev, u--;
    } while (u > i);
  } else if (u < i) {
    /* forward */
    do {
#ifdef SIZED_LIST_DEBUG
      printf ("--------- sized_list_get (%d of %d): forward from %d.\n", i - 1, list->num, u);
#endif
      node = node->next, u++;
    } while (u < i);
  }
  return node;
}

void sized_list_insert (sized_list_t *list, xitk_dnode_t *item, int index) {
  int u;
  xitk_dnode_t *f = sized_list_get (list, index)->prev;
  xitk_dnode_insert_after (f, item);
  list->num++;
  for (u = 0; u < (1<<SIZED_LIST_SAMPLE_LD) + 1; u++)
    if (list->last_i[u] >= index)
      list->last_i[u]++;
  _sized_list_update_samples (list);
}

void sized_list_free (sized_list_t *list, void (*node_free) (xitk_dnode_t *node)) {
  int u;
  if (node_free) {
    xitk_dnode_t *next = list->list.tail.prev;
    while (next->prev) {
      xitk_dnode_t *node = next;
      next = node->prev;
      xitk_dnode_remove (node);
      node_free (node);
    }
  } else {
    while (list->list.head.next->next)
      xitk_dnode_remove (list->list.head.next);
  }
  for (u = 0; u < (1<<SIZED_LIST_SAMPLE_LD) + 2; u++) {
    list->last_n[u] = &list->list.tail;
    list->last_i[u] = 0;
  }
  list->num = 0;
}

typedef enum {
  OTK_WG_EVENT_DRAW = 0,
  OTK_WG_EVENT_SELECT,
  OTK_WG_EVENT_DESTROY,
  OTK_WG_EVENT_LAST
} otk_widget_event_type_t;

struct otk_widget_s {
  xitk_dnode_t    win_node, layout_node;

  int             x, y, w, h;
  uint8_t         widget_type, focus, selectable, needdraw, needupdate;
  uint8_t         needcalc; /* used by layout container */
  uint8_t         major; /* if parent otk_window_t should draw/destroy this widget set this to 0 */

  otk_t          *otk;
  odk_t          *odk;
  otk_widget_t   *win;
  const char     *text;
  char           *ftext;

  otk_widget_cb_t user_cb;
  void           *user_data;

  void (*event) (otk_widget_t *this, otk_widget_event_type_t type);
};

const char *otk_widget_text_get (otk_widget_t *widget) {
  return widget ? widget->text : NULL;
}

struct otk_layout_s {
  otk_widget_t widget;
  xitk_dlist_t subs;
  int rows, columns;
};

#ifdef OTK_WITH_SLIDER
struct otk_slider_s {
  otk_widget_t widget;

  int  old_pos;
  otk_slider_cb_t get_value;
};
#endif

struct otk_scrollbar_s {
  otk_widget_t widget;

  int  pos_start;
  int  pos_end;
};

struct otk_button_s {

  otk_widget_t     widget;

  uint8_t         *pixmap;
  otk_button_cb_t  cb, uc;
  void            *cb_data, *uc_data;
  int              type;
};

typedef struct {
  xitk_dnode_t node;
  char s[1];
} otk_node_string_t;

struct otk_selector_s {

  otk_widget_t       widget;

  sized_list_t       items; /** << otk_node_string_t's. */
  int                num, pos;
};

struct otk_listentry_s {
  otk_widget_t widget;

  otk_list_t  *list;
  xitk_dnode_t list_node;
  int          i_vis, i_list;
  uint8_t      visible;
};

struct otk_list_s {
  otk_widget_t     widget;

  otk_get_ptr_t    get_ptr;
  sized_list_t    *list, ilist;   /* of otk_listentry_t */
  const char * const *texts;
  int             *num, inum, mark, position, last_pos;

  int              entry_height;
  int              entries_visible;
  int              last_selected;

  int              font_size;
  char            *font;

  otk_list_cb_t    cb;

  otk_widget_t    *scrollbar;
};

struct otk_label_s {

  otk_widget_t     widget;

  int              alignment;
  int              font_size;
};

struct otk_window_s {

  otk_widget_t     widget;

  int              fill_color;
  xitk_dlist_t     subs;
};

typedef enum {
  OTK_IDX_text_window = 0,
  OTK_IDX_text,
  OTK_IDX_text_border,
  OTK_IDX_focused_button,
  OTK_IDX_focused_text,
  OTK_IDX_focused_text_border,
  OTK_IDX_label_window,
  OTK_IDX_label,
  OTK_IDX_label_border,
  OTK_IDX_slider,
  OTK_IDX_slider_knob,
  OTK_IDX_focused_slider,
  OTK_IDX_focused_slider_knob,
  OTK_IDX_LAST
} otk_idx_t;

static const uint8_t _otk_idx[] = {
  OTK_IDX_text_window, OTK_IDX_focused_button,
  OTK_IDX_slider,      OTK_IDX_focused_slider,
  OTK_IDX_slider_knob, OTK_IDX_focused_slider_knob
};

struct otk_s {
  oxine_t      *oxine;
  odk_t        *odk;
  xitk_t       *xitk;

  int           size[2];

  odk_palette_t palette;

  xitk_dlist_t  windows;

  otk_widget_t *focus_ptr, *hit_ptr;

  int           verbosity;

  uint8_t       showing;
  uint8_t       needdraw; /** << 0 (no), 1 (some), 2-3 (all. */
  int           num_drawn;

  int           button_font_size;
  char         *button_font;

  char         *label_font;
  int           label_color;
  int           label_font_size;
  char         *title_font;
  int           title_font_size;

  int           update_job;

  int (*event_handler)(void *data, oxine_event_t *ev);
  void *event_handler_data;

  pthread_mutex_t draw_mutex;
};

/** private utilities */

/* truncate the text to a fixed width */
static void check_text_width(odk_t *odk, char *text, int width) {
  int textwidth, num;
  odk_get_text_size(odk, text, &textwidth, &num);

  if (textwidth > width) {
    int i=strlen(text)-4;

    if (i<0) return;

    text[i]=text[i+1]=text[i+2]='.';text[i+3]=0;

    odk_get_text_size(odk, text, &textwidth, &num);
    while ((textwidth > width) && (i>0)) {
      i--;
      text[i]='.';
      text[i+3]=0;
      odk_get_text_size(odk, text, &textwidth, &num);
    }
  }
}

int is_correct_widget (otk_widget_t *widget, int expected);
#define is_correct_widget(_widget,_expected) (_widget && (_widget->widget_type == _expected))

/** callback utilities */

static otk_widget_t *find_widget_xy (otk_t *otk, int x, int y) {
  otk_window_t *win;
  otk_widget_t *widget;

  xitk_container (win, otk->windows.head.next, widget.win_node);
  while (win->widget.win_node.next) {
    xitk_container (widget, win->subs.head.next, win_node);
    while (widget->win_node.next) {
      if  ((widget->x <= x) && (widget->y <= y)
        && ((widget->x + widget->w) >= x)
        && ((widget->y + widget->h) >= y)
        && (widget->selectable))
	return widget;
      xitk_container (widget, widget->win_node.next, win_node);
    }
    xitk_container (win, win->widget.win_node.next, widget.win_node);
  }
  return NULL;
}

#define ABS(x) ((x)<0?-(x):(x))

#define UP 1
#define DOWN 2
#define LEFT 3
#define RIGHT 4

static otk_widget_t *find_neighbour (otk_t *otk, int direction) {
  otk_widget_t *neighbour = NULL;
  int neighbour_ratio = 0;
  int x = otk->focus_ptr->x + (otk->focus_ptr->w >> 1);
  int y = otk->focus_ptr->y + (otk->focus_ptr->h >> 1);
  otk_window_t *win;
  otk_widget_t *widget;

  xitk_container (win, otk->windows.head.next, widget.win_node);
  while (win->widget.win_node.next) {
    xitk_container (widget, win->subs.head.next, win_node);
    while (widget->win_node.next) {

      if ((widget != otk->focus_ptr) && widget->selectable) {
        int ratio = 0;
        int nx = widget->x + (widget->w >> 1) - x;
	int ny = widget->y + (widget->h >> 1) - y;
        double t;

        switch (direction) {
	  case UP:
            ny = -ny;
            /* fall through */
	  case DOWN:
            if (ny <= 0)
              break;
            t = (double)ABS (nx) / (double)ny + 1;
            ratio = 1;
	    break;
	  case LEFT:
            nx = -nx;
            /* fall through */
          case RIGHT:
            if (nx <= 0)
              break;
            t = (double)ABS (ny) / (double)nx + 1;
            ratio = 1;
	    break;
          default: ;
        }
        if (ratio) {
          nx >>= 3; /* avoid overrun */
          ny >>= 3;
          ratio = atan (t) * sqrt (nx * nx + ny * ny);
#ifdef LOG
          {
            otk_button_t *but = (otk_button_t*)widget;
            printf("%s %i\n", but->widget.text, ratio);
          }
#endif
          if ((ratio > 0) && ((ratio < neighbour_ratio) || !neighbour_ratio)) {
	    neighbour_ratio = ratio;
	    neighbour = widget;
          }
        }
      }
      xitk_container (widget, widget->win_node.next, win_node);
    }
    xitk_container (win, win->widget.win_node.next, widget.win_node);
  }
#ifdef LOG
  printf("\n");
#endif
  return neighbour;
}

static void _otk_widget_needdraw (otk_widget_t *w) {
  w->needdraw |= 1;
  if (!w->otk->needdraw)
    w->otk->needdraw = 1;
}

static void _otk_redraw (otk_t *otk) {
  otk_window_t *win;

  if (!otk->needdraw)
    return;

  pthread_mutex_lock (&otk->draw_mutex);

  otk->num_drawn = 0;
  otk->needdraw &= 2;
  if (otk->needdraw)
    odk_clear (otk->odk);

  xitk_container (win, otk->windows.head.next, widget.win_node);
  if (!win->widget.win_node.next) {
    odk_hide (otk->odk);
    pthread_mutex_unlock(&otk->draw_mutex);
    return;
  }

  do {
    win->widget.event (&win->widget, OTK_WG_EVENT_DRAW);
    xitk_container (win, win->widget.win_node.next, widget.win_node);
  } while (win->widget.win_node.next);

  odk_show (otk->odk);
  otk->needdraw = 0;
  if (otk->num_drawn && (otk->verbosity >= 2))
    printf ("gui.oxine.otk.draw (%d).\n", otk->num_drawn);
  pthread_mutex_unlock (&otk->draw_mutex);
}

/** global callbacks */

static int motion_handler (otk_t *otk, oxine_event_t *ev) {
  otk_widget_t *b;

  /* printf ("gui.oxine.otk.event.move (hit=%p x=%d, y=%d).\n", otk->hit_ptr, ev->x, ev->y); */
  if ((b = otk->hit_ptr)) {
    if (b->user_cb)
      b->user_cb (b, b->user_data, OTK_ACT_VALUE, ev->y);
    return OXINE_EVENT_HANDLED;
  } else {
    b = find_widget_xy (otk, ev->x, ev->y);
    if (!b)
      return 0;
    if (b->focus)
      return 0;
    otk_set_focus (b);
  }
  return OXINE_EVENT_HANDLED;
}

static void list_adapt_entries(otk_list_t *list) {
  int i_vis, i_list, i_max = list->entries_visible < *list->num ? list->entries_visible : *list->num;
  xitk_dnode_t *node;
  otk_listentry_t *entry, *lastentry = NULL;

  i_vis = 0;
  i_list = list->position;
  if (i_list > *list->num - list->entries_visible)
    i_list = *list->num - list->entries_visible;
  if (i_list < 0)
    i_list = 0;
  list->position = i_list;
  if (list->get_ptr) { /** << user list mode. */
  /* printf ("gui.oxine.otk_listitem.adapt.list (%p, (%d, %d, %d, %d), num=%d, first=%d).\n",
       (void *)list, list->widget.x, list->widget.y, list->widget.w, list->widget.h,
       list->list->num, list->position); */
    for (node = list->ilist.list.head.next; node->next; node = node->next) {
      xitk_container (entry, node, list_node);
      if (i_vis >= i_max)
        break;
      lastentry = entry;
      xitk_dnode_t *unode = sized_list_get (list->list, i_list);
      entry->widget.text = list->get_ptr (unode, OTK_GET_PTR_TEXT);
      entry->widget.user_data = list->get_ptr (unode, OTK_GET_PTR_ITEM);
      entry->i_list = i_list;
      entry->i_vis = i_vis;
      entry->visible = 1;
      entry->widget.selectable = 1;
      i_vis++, i_list++;
  /* printf ("gui.oxine.otk_listitem.adapt (%p, %s, (%d, %d, %d, %d), i_list=%d, i_vis=%d).\n",
         (void *)entry, entry->widget.text, entry->widget.x, entry->widget.y, entry->widget.w, entry->widget.h,
         entry->i_list, entry->i_vis); */
    }
  } else { /** << user array mode. */
    for (node = list->ilist.list.head.next; node->next; node = node->next) {
      if (i_vis >= i_max)
        break;
      xitk_container (entry, node, list_node);
      lastentry = entry;
      entry->widget.text = list->texts[i_list];
      entry->widget.user_cb = list->widget.user_cb;
      entry->widget.user_data = list->widget.user_data;
      entry->i_list = i_list;
      entry->i_vis = i_vis;
      entry->visible = 1;
      entry->widget.selectable = 1;
      i_vis++, i_list++;
    }
  }
  for (; node->next; node = node->next) {
    xitk_container (entry, node, list_node);
    entry->widget.text = NULL;
    entry->widget.user_cb = NULL;
    entry->widget.user_data = NULL;
    entry->i_list = 0;
    entry->i_vis = i_vis;
    entry->visible = 0;
    entry->widget.selectable = 0;
    i_vis++, i_list++;
  }
  if (list->widget.otk->focus_ptr && (list->widget.otk->focus_ptr->widget_type == OTK_WIDGET_LISTENTRY)) {
    xitk_container (entry, list->widget.otk->focus_ptr, widget);
    if ((entry->list == list) && !entry->visible) {
      entry->widget.focus = 0;
      list->widget.otk->focus_ptr = &lastentry->widget;
      if (lastentry)
        lastentry->widget.focus = 1;
    }
  }
  if (list->widget.otk->hit_ptr && (list->widget.otk->hit_ptr->widget_type == OTK_WIDGET_LISTENTRY)) {
    xitk_container (entry, list->widget.otk->hit_ptr, widget);
    if (entry->list == list)
      list->widget.otk->hit_ptr = NULL;
  }
  list->last_pos = list->position;

  i_vis = *list->num > 0 ? *list->num : 1;
  otk_scrollbar_set (list->scrollbar,
    (list->position * OTK_SCROLLBAR_FULL_VAL + (i_vis >> 1)) / i_vis,
    ((list->position + list->entries_visible) * OTK_SCROLLBAR_FULL_VAL + (i_vis >> 1)) / i_vis);
  _otk_widget_needdraw (list->widget.win);
}

static int list_page (otk_widget_t *w, int dir) {
  otk_list_t *list;
  otk_listentry_t *entry;
  int n, d;

  if (w->widget_type == OTK_WIDGET_LISTENTRY) {
    xitk_container (entry, w, widget);
    list = entry->list;
  } else if (w->widget_type == OTK_WIDGET_LIST) {
    xitk_container (list, w, widget);
  } else {
    return 0;
  }
  n = list->position + (dir < 0 ? -(list->entries_visible >> 1) : (list->entries_visible >> 1));
  if (n > *list->num - list->entries_visible)
    n = *list->num - list->entries_visible;
  if (n < 0)
    n = 0;
  d = n - list->position;
  if (!d)
    return 1;
  list->position = n;
  list_adapt_entries (list);
  return 2;
}

static int button_handler (otk_t *otk, oxine_event_t *ev) {
  otk_widget_t *b = NULL;

  if (!(ev->key & OXINE_KEY_RELEASE) && !(b = find_widget_xy (otk, ev->x, ev->y)))
    return 0;

  if(ev->key == OXINE_KEY_NULL) {
    /*printf("event already processed\n");*/
    return 0;
  }

  switch (ev->key) {
    case OXINE_BUTTON1 + OXINE_KEY_RELEASE:
      if (!otk->hit_ptr)
        return 0;
      if (otk->hit_ptr->user_cb)
        otk->hit_ptr->user_cb (otk->hit_ptr, otk->hit_ptr->user_data, OTK_ACT_HIT, -ev->y);
      otk->hit_ptr = NULL;
      break;
    case OXINE_BUTTON1:
      otk->hit_ptr = b;
      if (b->user_cb)
	otk->hit_ptr->user_cb (b, b->user_data, OTK_ACT_HIT, ev->y);
      b->event (b, OTK_WG_EVENT_SELECT);
      break;
    case OXINE_BUTTON4:
      if (!otk->focus_ptr)
        return 0;
      list_page (b, -1);
      break;
    case OXINE_BUTTON5:
      if (!otk->focus_ptr)
        return 0;
      list_page (b, 1);
      break;
    default: return 0;
  }
  return OXINE_EVENT_HANDLED;
}

static int key_handler (otk_t *otk, oxine_event_t *ev) {
  otk_widget_t *new = NULL, *fw;

  /* we only handle key events if we have a focus */
  fw = otk->focus_ptr;
  if (!fw)
    return 0;

  switch (ev->key) {
    case OXINE_KEY_NULL:
      return OXINE_EVENT_HANDLED;
    case OXINE_KEY_UP:
      if (fw->widget_type == OTK_WIDGET_LISTENTRY) {
	otk_listentry_t *entry = (otk_listentry_t*)fw;

        if (entry->i_vis == 0) {
          if (entry->i_list > 0) {
            entry->list->position--;
            list_adapt_entries (entry->list);
          }
	}
      }
      if (fw->widget_type == OTK_WIDGET_SCROLLBAR) {
        if (fw->user_cb) {
          _otk_widget_needdraw (fw);
          fw->user_cb (fw, fw->user_data, OTK_ACT_STEP, 1);
	}
        break;
      }
      new = find_neighbour (otk, UP);
      break;
    case OXINE_KEY_DOWN:
      if (fw->widget_type == OTK_WIDGET_LISTENTRY) {
	otk_listentry_t *entry = (otk_listentry_t*)fw;

        if (entry->i_vis == entry->list->entries_visible - 1) {
          if (entry->i_list < *entry->list->num - 1) {
            entry->list->position++;
            list_adapt_entries (entry->list);
          }
        }
      } else if (fw->widget_type == OTK_WIDGET_SCROLLBAR) {
        if (fw->user_cb) {
          _otk_widget_needdraw (fw);
          fw->user_cb (fw, fw->user_data, OTK_ACT_STEP, -1);
        }
        break;
      }
      new = find_neighbour (otk, DOWN);
      break;
    case OXINE_KEY_LEFT:
      new = find_neighbour(otk, LEFT);
      break;
    case OXINE_KEY_RIGHT:
      new = find_neighbour(otk, RIGHT);
      break;
    case OXINE_KEY_SELECT:
      fw->event (fw, OTK_WG_EVENT_SELECT);
      break;
    case OXINE_KEY_PRIOR:
      list_page (fw, -1);
      break;
    case OXINE_KEY_NEXT:
      list_page (fw, 1);
      break;
    default: return 0;
  }
  if (new && (new != fw))
    otk_set_focus (new);
  return OXINE_EVENT_HANDLED;
}

static int otk_event_handler(void *this, oxine_event_t *ev) {
  otk_t *otk = (otk_t*) this;
  int ret = 0;

  switch (ev->type) {
    case OXINE_EVENT_KEY:
      ret = key_handler (otk, ev);
      break;
    case OXINE_EVENT_MOTION:
      ret = motion_handler (otk, ev);
      break;
    case OXINE_EVENT_BUTTON:
      ret = button_handler (otk, ev);
      break;
    case OXINE_EVENT_FORMAT_CHANGED:
      ret = OXINE_EVENT_HANDLED;
      if (odk_get_size (otk->odk, otk->size))
        otk->needdraw = 2;
      break;
    default: ;
  }
  if (!ret && otk->event_handler)
    ret = otk->event_handler(otk->event_handler_data, ev);

  _otk_redraw (otk);
  return ret | (otk->hit_ptr ? OXINE_EVENT_WIDGET_HELD : 0);;
}

int otk_send_event(otk_t *otk, oxine_event_t *ev) {
  return odk_send_event(otk->odk, ev);
}

/** button widget */

static void button_event (otk_widget_t *this, otk_widget_event_type_t type) {
  otk_button_t *button = (otk_button_t*) this;

  if (type == OTK_WG_EVENT_DRAW) {
    char displayed_text[800];

    if (this->focus)
      odk_draw_rect (this->odk, this->x, this->y, this->x+this->w, this->y+this->h, 1,
        this->otk->palette.index[OTK_IDX_focused_button]);

    switch (button->type) {
      case OTK_BUTTON_TEXT:
        odk_set_font (this->odk, this->otk->button_font, this->otk->button_font_size);
        strlcpy (displayed_text, button->widget.text, sizeof (displayed_text));
        check_text_width (this->odk, displayed_text, button->widget.w);
        odk_draw_text (this->odk, this->x + (this->w >> 1), this->y + (this->h >> 1),
          displayed_text, ODK_ALIGN_CENTER | ODK_ALIGN_VCENTER, this->otk->palette.index[_otk_idx[0 + (this->focus & 1)]]);
        break;
      case OTK_BUTTON_PIXMAP:
        odk_draw_bitmap (this->odk, button->pixmap, this->x - this->w*2, this->y - this->h*2, 10, 10, NULL);
        break;
      default:
        printf ("OTK_BUTTON_TYPE : %d not supported\n", button->type);
    }
  } else if (type == OTK_WG_EVENT_DESTROY) {
    if (button->type == OTK_BUTTON_PIXMAP)
      free (button->pixmap);
  } else if (type == OTK_WG_EVENT_SELECT) {
    if (button->cb)
      button->cb (button->cb_data);
  }
}

#ifdef OTK_WITH_SLIDER
static void slider_event (otk_widget_t *this, otk_widget_event_type_t type) {
  otk_slider_t *slider = (otk_slider_t*) this;
  int value;

  if (!is_correct_widget (this, OTK_WIDGET_SLIDER))
    return;

  if (type == OTK_WG_EVENT_DRAW) {
    value = slider->get_value (this->odk);
    if (value == -1)
      value = slider->old_pos;
    slider->old_pos = value;
    value = (value * this->w - 20) / 100;

    odk_draw_rect (this->odk, this->x, this->y, this->x+this->w, this->y+this->h, 1,
      this->otk->palette.index[_otk_idx[0 + (this->focus & 1)]]);
    odk_draw_rect (this->odk, this->x, this->y + (this->h >> 1) - 5, this->x+this->w, this->y + (this->h >> 1) + 5, 1,
      this->otk->palette.index[_otk_idx[2 + (this->focus & 1)]]);
    odk_draw_rect (this->odk, this->x + value + 5, this->y + (this->h >> 1) - 15,
      this->x + value + 15, this->y + (this->h >> 1) + 15, 1, this->otk->palette.index[_otk_idx[4 + (this->focus & 1)]]);
  }
}
#endif

static void _otk_widget_destroy (otk_widget_t *w) {
  if (w->otk->focus_ptr == w)
    w->otk->focus_ptr = NULL;
  if (w->otk->hit_ptr == w)
    w->otk->hit_ptr= NULL;
  if (w->win_node.next)
    xitk_dnode_remove (&w->win_node);
  if (w->layout_node.next)
    xitk_dnode_remove (&w->layout_node);
  w->event (w, OTK_WG_EVENT_DESTROY);
  ho_free (w->ftext);
  ho_free (w);
}

static void scrollbar_event (otk_widget_t *this, otk_widget_event_type_t type) {
  otk_scrollbar_t *scrollbar = (otk_scrollbar_t*) this;

  if (type == OTK_WG_EVENT_DRAW) {
    int value1, value2;

    value1 = (scrollbar->pos_start * (this->h - 10) + (OTK_SCROLLBAR_FULL_VAL >> 1)) >> OTK_SCROLLBAR_FULL_LD;
    value2 = (scrollbar->pos_end   * (this->h - 10) + (OTK_SCROLLBAR_FULL_VAL >> 1)) >> OTK_SCROLLBAR_FULL_LD;
    /* minimum pad height */
    if (value2 - value1 < 10) {
      value1 = (((scrollbar->pos_start + scrollbar->pos_end) >> 1) * (this->h - 15)
             + (OTK_SCROLLBAR_FULL_VAL >> 1)) >> OTK_SCROLLBAR_FULL_LD;
      value2 = value1 + 10;
    }

    odk_draw_rect (this->odk, this->x, this->y, this->x+this->w, this->y+this->h, 1,
      this->otk->palette.index[_otk_idx[0 + (this->focus & 1)]]);
    /* odk_draw_rect (this->odk, this->x, this->y, this->x+this->w, this->y+this->h, 0,
      this->otk->palette.index[_otk_idx[2 + (this->focus & 1)]]); */
    odk_draw_rect (this->odk, this->x+5, this->y+5+value1, this->x+this->w-5, this->y+5+value2, 1,
      this->otk->palette.index[_otk_idx[4 + (this->focus & 1)]]);
  }
}

/*void button_list_scroll_up (otk_list_t *list)*/

#if 0
/* I've added this function because i don't want to break existing code.             *
 * When we'll remage otk and odk it's better to select button type in otk_button_new *
 * and then pass text or pixmap                                                      */
static void otk_button_add_pixmap (otk_button_t *button, uint8_t *pixmap) {

  if (!is_correct_widget((otk_widget_t *)button, OTK_WIDGET_BUTTON)) return ;

  button->type               = OTK_BUTTON_PIXMAP;
  button->pixmap             = pixmap;
  return ;
}
#endif
/* slider widget */

#ifdef OTK_WITH_SLIDER
otk_widget_t *otk_slider_grid_new (otk_slider_cb_t cb) {
  otk_slider_t *slider;

  slider = ho_new(otk_slider_t);

  xitk_dnode_init (&slider->widget.win_node);
  xitk_dnode_init (&slider->widget.layout_node);
  slider->widget.widget_type = OTK_WIDGET_SLIDER;
  slider->widget.selectable  = 1;
  slider->widget.event       = slider_event;
  slider->widget.needupdate  = 0;
  slider->widget.needdraw    = 1;
  slider->widget.needcalc    = 1;
  slider->widget.major       = 1;
  slider->widget.text        =
  slider->widget.ftext       = NULL;
  slider->old_pos            = 0;
  slider->get_value          = cb;

  return (otk_widget_t*) slider;
}
#endif

otk_widget_t *otk_button_grid_new (const char *text,
			      otk_button_cb_t cb,
			      void *cb_data) {
  otk_button_t *button;

  button = ho_new(otk_button_t);

  xitk_dnode_init (&button->widget.win_node);
  xitk_dnode_init (&button->widget.layout_node);
  button->widget.widget_type = OTK_WIDGET_BUTTON;
  if (text && !memcmp (text, OTK_TEMP_STR, sizeof (OTK_TEMP_STR) - 1))
    button->widget.text = button->widget.ftext = ho_strdup (text + sizeof (OTK_TEMP_STR) - 1);
  else
    button->widget.text = text, button->widget.ftext = NULL;
  button->type               = OTK_BUTTON_TEXT;
  button->widget.selectable  = 1;
  button->widget.event       = button_event;
  button->widget.needupdate  = 0;
  button->widget.needdraw    = 1;
  button->widget.needcalc    = 1;
  button->widget.major       = 1;
  button->cb                 = cb;
  button->cb_data            = cb_data;
  button->uc                 = NULL;

  return (otk_widget_t*) button;
}

otk_widget_t *otk_button_new (otk_widget_t *win, int x, int y,
			      int w, int h, const char *text,
			      otk_button_cb_t cb,
			      void *cb_data) {
  otk_button_t *button;
  otk_window_t *window = (otk_window_t*) win;

  if (!is_correct_widget(win, OTK_WIDGET_WINDOW)) return NULL;

  button = ho_new(otk_button_t);

  xitk_dnode_init (&button->widget.win_node);
  xitk_dnode_init (&button->widget.layout_node);
  button->widget.widget_type = OTK_WIDGET_BUTTON;
  button->widget.x           = win->x+x;
  button->widget.y           = win->y+y;
  button->widget.w           = w;
  button->widget.h           = h;
  if (text && !memcmp (text, OTK_TEMP_STR, sizeof (OTK_TEMP_STR) - 1))
    button->widget.text = button->widget.ftext = ho_strdup (text + sizeof (OTK_TEMP_STR) - 1);
  else
    button->widget.text = text, button->widget.ftext = NULL;
  button->type               = OTK_BUTTON_TEXT;
  button->widget.win         = win;
  button->widget.otk         = win->otk;
  button->widget.odk         = win->otk->odk;
  button->widget.selectable  = 1;
  button->widget.event       = button_event;
  button->widget.needupdate  = 0;
  button->widget.needdraw    = 1;
  button->widget.major       = 0;
  button->cb                 = cb;
  button->cb_data            = cb_data;
  button->uc                 = NULL;

  xitk_dlist_add_tail (&window->subs, &button->widget.win_node);

  return &button->widget;
}

otk_widget_t *otk_scrollbar_new (otk_widget_t *win, int x, int y, int w, int h, otk_widget_cb_t user_cb, void *user_data) {
  otk_scrollbar_t *scrollbar;
  otk_window_t *window = (otk_window_t*) win;

  if (!is_correct_widget(win, OTK_WIDGET_WINDOW)) return NULL;

  scrollbar = ho_new(otk_scrollbar_t);

  xitk_dnode_init (&scrollbar->widget.win_node);
  xitk_dnode_init (&scrollbar->widget.layout_node);
  scrollbar->widget.widget_type = OTK_WIDGET_SCROLLBAR;
  scrollbar->widget.x           = win->x+x;
  scrollbar->widget.y           = win->y+y;
  scrollbar->widget.w           = w;
  scrollbar->widget.h           = h;
  scrollbar->widget.win         = win;
  scrollbar->widget.otk         = win->otk;
  scrollbar->widget.odk         = win->otk->odk;
  scrollbar->widget.selectable  = 1;
  scrollbar->widget.event       = scrollbar_event;
  scrollbar->widget.needupdate  = 0;
  scrollbar->widget.needdraw    = 1;
  scrollbar->widget.major       = 0;
  scrollbar->widget.text        =
  scrollbar->widget.ftext       = NULL;
  scrollbar->widget.user_cb     = user_cb;
  scrollbar->widget.user_data   = user_data;
  scrollbar->pos_start          = 0;
  scrollbar->pos_end            = OTK_SCROLLBAR_FULL_VAL;

  xitk_dlist_add_tail (&window->subs, &scrollbar->widget.win_node);

  return &scrollbar->widget;
}

void otk_scrollbar_set (otk_widget_t *this, int pos_start, int pos_end) {
  otk_scrollbar_t *scrollbar = (otk_scrollbar_t*) this;

  if (!is_correct_widget(this, OTK_WIDGET_SCROLLBAR))
    return;

  if (pos_start < 0)
    pos_start = 0;
  if (pos_start > OTK_SCROLLBAR_FULL_VAL)
    pos_start = OTK_SCROLLBAR_FULL_VAL;
  if (pos_end < pos_start)
    pos_end = pos_start;
  if (pos_end > OTK_SCROLLBAR_FULL_VAL)
    pos_end = OTK_SCROLLBAR_FULL_VAL;

  if ((scrollbar->pos_start != pos_start) || (scrollbar->pos_end != pos_end)) {
    scrollbar->pos_start = pos_start;
    scrollbar->pos_end = pos_end;
    _otk_widget_needdraw (this);
  }
}

void otk_button_uc_set(otk_widget_t *this, otk_button_cb_t uc, void *uc_data) {
  otk_button_t *button = (otk_button_t *)this;

  if (!is_correct_widget(this, OTK_WIDGET_BUTTON)) return;

  button->uc = uc;
  button->widget.needupdate = 1;
  button->uc_data = uc_data;

}

/** list widget */

static void listentry_event (otk_widget_t *this, otk_widget_event_type_t type) {
  otk_listentry_t *listentry = (otk_listentry_t*) this;

  if (type == OTK_WG_EVENT_DRAW) {
    char displayed_text[800];
#ifdef LOG
    printf ("gui.oxine.otk.list.entry.draw (%d, %d, %s). -----------------\n", this->x, this->y, this->text);
#endif
    if (!listentry->visible)
      return;
    odk_set_font (this->odk, listentry->list->font, listentry->list->font_size);
    if (listentry->i_list == listentry->list->mark)
      memcpy (displayed_text, "->", 2);
    else
      memcpy (displayed_text, "  ", 2);
    strlcpy (displayed_text + 2, listentry->widget.text, sizeof (displayed_text) - 2);
    check_text_width (listentry->widget.odk, displayed_text, listentry->widget.w);

    /* printf("%d %d\n", this->x, this->y+this->h/2); */
    if (this->focus)
    odk_draw_rect (this->odk, this->x, this->y, this->x+this->w, this->y+this->h, 1,
      this->otk->palette.index[OTK_IDX_focused_button]);
    odk_draw_text (this->odk, this->x, this->y + (this->h >> 1), displayed_text, ODK_ALIGN_LEFT | ODK_ALIGN_VCENTER,
      this->otk->palette.index[_otk_idx[0 + (this->focus & 1)]]);
  } else if (type == OTK_WG_EVENT_SELECT) {
    listentry->list->last_selected = listentry->i_list;
    if (listentry->list->cb)
      listentry->list->cb (listentry->list->widget.user_data, listentry->widget.user_data);
    else if (listentry->widget.user_cb)
      listentry->widget.user_cb (&listentry->list->widget, listentry->list->widget.user_data, OTK_ACT_VALUE, listentry->i_list);
  } else if (type == OTK_WG_EVENT_DESTROY) {
    if (listentry->list_node.next)
      xitk_dnode_remove (&listentry->list_node);
    listentry->list = NULL;
  }
}

static void _otk_node_widget_destroy (xitk_dnode_t *node) {
  otk_listentry_t *e;
  xitk_container (e, node, list_node);
  _otk_widget_destroy (&e->widget);
}

static void list_scroll (otk_widget_t *w, void *this, int action, int value) {
  otk_list_t *list = (otk_list_t *)this;
  int n, wy, wh;

  /* printf ("gui.oxine.otk.list_scroll (w=%p action=%d, y=%d).\n", w, action, value); */
  switch (action) {
    case OTK_ACT_HIT:
      if (value < 0)
        return;
      /* fall through */
    case OTK_ACT_VALUE:
      n = *list->num - list->entries_visible;
      if (n <= 0)
        return;
      if (w->h <= 0)
        return;
      wh = (w->h * n + (*list->num >> 1)) / *list->num;
      wy = w->y + ((w->h - wh) >> 1);
      value -= wy;
      if (value > wh)
        value = wh;
      if (value < 0)
        value = 0;
      n = (n * value + (wh >> 1)) / wh;
      if (n == list->position)
        return;
      list->position = n;
      list_adapt_entries (list);
      break;
    case OTK_ACT_STEP:
      if (list_page (&list->widget, -value) != 2)
        return;
      break;
    default: return;
  }
}

void otk_list_set_pos (otk_widget_t *this, int newpos) {
  otk_list_t *list = (otk_list_t *) this;

  if (!is_correct_widget(this, OTK_WIDGET_LIST)) return;

  newpos -= list->entries_visible >> 1;
  if (newpos > *list->num - list->entries_visible)
    newpos = *list->num - list->entries_visible;
  if (newpos < 0)
    newpos = 0;
  list->position = newpos;

  list_adapt_entries(list);
}

int otk_list_get_pos(otk_widget_t *this) {
  otk_list_t *list = (otk_list_t *) this;

  if (!is_correct_widget(this, OTK_WIDGET_LIST)) return -1;

  return list->position;
}

static void list_event (otk_widget_t *this, otk_widget_event_type_t type) {
  otk_list_t *list = (otk_list_t*) this;
  /* we have the visible list items in the window list for navigation slready.
   * let them draw there as well. */
  if (type == OTK_WG_EVENT_DESTROY) {
    if (this->widget_type == OTK_WIDGET_LIST)
      sized_list_free (&list->ilist, _otk_node_widget_destroy);
  }
}

otk_widget_t *otk_list_new (otk_widget_t *win, int x, int y, int w, int h, otk_list_cb_t cb, void *cb_data) {
  otk_list_t *list;
  otk_window_t *window = (otk_window_t*) win;
  int i;

  if (!is_correct_widget(win, OTK_WIDGET_WINDOW)) return NULL;

  list = ho_new(otk_list_t);

  xitk_dnode_init (&list->widget.win_node);
  xitk_dnode_init (&list->widget.layout_node);
  list->widget.widget_type = OTK_WIDGET_LIST;
  list->widget.x           = win->x+x;
  list->widget.y           = win->y+y;
  list->widget.w           = w;
  list->widget.h           = h;
  list->widget.win         = win;
  list->widget.otk         = win->otk;
  list->widget.odk         = win->otk->odk;
  list->widget.selectable  = 0;
  list->widget.event       = list_event;
  list->widget.needupdate  = 0;
  list->widget.needdraw    = 1;
  list->widget.major       = 0;
  list->widget.text        = NULL;
  list->widget.user_data   = cb_data;
  list->position           = 0;
  list->last_pos           = 0;
  list->cb                 = cb;
  list->font               = ho_strdup("sans");
  list->font_size          = 30;
  list->entry_height       = 32;
  list->texts              = NULL;
  list->mark               = -1;
  list->inum               = 0;
  list->num                = &list->inum;

  sized_list_init (&list->ilist);
  list->list = &list->ilist;
  list->get_ptr = NULL;
  list->entries_visible = list->widget.h / list->entry_height;
  list->widget.h = list->entries_visible * list->entry_height;

  /* widget the visible ones only. */
  for (i = 0; i < list->entries_visible; i++) {
    otk_listentry_t *entry = ho_new (otk_listentry_t);
    if (!entry)
      break;
    xitk_dnode_init (&entry->widget.win_node);
    xitk_dnode_init (&entry->widget.layout_node);
    xitk_dnode_init (&entry->list_node);
    entry->widget.widget_type = OTK_WIDGET_LISTENTRY;
    entry->widget.w           = list->widget.w - 25;
    entry->widget.h           = list->entry_height;
    entry->widget.x           = list->widget.x;
    entry->widget.y           = list->widget.y + list->entry_height * i;
    entry->widget.win         = list->widget.win;
    entry->widget.otk         = list->widget.otk;
    entry->widget.odk         = list->widget.odk;
    entry->widget.selectable  = 0;
    entry->widget.event       = listentry_event;
    entry->widget.needupdate  = 0;
    entry->widget.needdraw    = 1;
    entry->widget.major       = 0;
    entry->widget.text        =
    entry->widget.ftext       = NULL;
    entry->widget.user_data   = NULL;
    entry->visible            = 0;
    entry->list               = list;
    odk_set_font (list->widget.odk, list->font, list->font_size);
    xitk_dlist_add_tail (&window->subs, &entry->widget.win_node);
    sized_list_insert (&list->ilist, &entry->list_node, list->ilist.num);
  }
  list->entries_visible = i;

  list->scrollbar = otk_scrollbar_new (win, x + list->widget.w - 25, y,
    27, list->widget.h, list_scroll, list);

  xitk_dlist_add_tail (&window->subs, &list->widget.win_node);

  return &list->widget;
}

void otk_set_user_list (otk_widget_t *w, sized_list_t *user_list, otk_get_ptr_t get_ptr) {
  otk_list_t *list;

  if (!is_correct_widget (w, OTK_WIDGET_LIST))
    return;
  xitk_container (list, w, widget);
  if (user_list && get_ptr) {
    list->widget.user_cb = NULL;
    list->texts = NULL;
    list->list = user_list;
    list->num = &user_list->num;
    list->get_ptr = get_ptr;
    list_adapt_entries (list);
  } else {
    list->list = &list->ilist;
    list->get_ptr = NULL;
    list->num = &list->inum;
  }
}

void otk_set_user_array (otk_widget_t *w, const char * const *texts, int n, int mark, otk_widget_cb_t user_cb, void *user_data) {
  otk_list_t *list;
  int i;

  if (!is_correct_widget (w, OTK_WIDGET_LIST))
    return;
  xitk_container (list, w, widget);
  list->list = &list->ilist;
  list->get_ptr = NULL;
  list->num = &list->inum;
  list->texts = texts;
  if (texts) {
    list->widget.user_cb = user_cb;
    list->widget.user_data = user_data;
    list->inum = (n >= 0) ? n : 0;
    list->mark = (mark < 0) ? -1 : (mark < list->inum) ? mark : list->inum - 1;
  } else {
    list->inum = 0;
    list->widget.user_cb = NULL;
    list->widget.user_data = NULL;
    list->mark = -1;
  }
  i = list->mark - (list->entries_visible >> 1);
  if (i > list->inum - 1)
    i = list->inum - 1;
  if (i < 0)
    i = 0;
  list->position = i;
}

/** label widget */

static void label_event (otk_widget_t *this, otk_widget_event_type_t type) {
  otk_label_t *label = (otk_label_t*) this;

  if (type == OTK_WG_EVENT_DRAW) {
    char displayed_text[800];

    strlcpy (displayed_text, label->widget.text, sizeof (displayed_text));
    odk_set_font (this->odk, this->otk->label_font,
      (label->font_size) ? label->font_size : this->otk->label_font_size);
    check_text_width (this->odk, displayed_text, this->win->w );
    odk_draw_text (this->odk, this->x, this->y, displayed_text, label->alignment,
      this->otk->palette.index[OTK_IDX_label_window]);
  }
}

otk_widget_t *otk_label_new (otk_widget_t *win, int x, int y, int alignment, const char *text) {

  otk_label_t *label;
  otk_window_t *window = (otk_window_t*) win;

  if (!is_correct_widget(win, OTK_WIDGET_WINDOW)) return NULL;

  label = ho_new(otk_label_t);

  xitk_dnode_init (&label->widget.win_node);
  xitk_dnode_init (&label->widget.layout_node);
  label->widget.widget_type = OTK_WIDGET_LABEL;
  label->widget.x           = win->x+x;
  label->widget.y           = win->y+y;
  label->widget.win         = win;
  label->widget.otk         = win->otk;
  label->widget.odk         = win->otk->odk;
  label->widget.event       = label_event;
  label->widget.needupdate  = 0;
  label->widget.needdraw    = 1;
  label->widget.major       = 0;
  if (text && !memcmp (text, OTK_TEMP_STR, sizeof (OTK_TEMP_STR) - 1))
    label->widget.text = label->widget.ftext = ho_strdup (text + sizeof (OTK_TEMP_STR) - 1);
  else
    label->widget.text = text, label->widget.ftext = NULL;
  label->alignment          = alignment;
  label->font_size          = 0; /* use default */

  xitk_dlist_add_tail (&window->subs, &label->widget.win_node);

  return &label->widget;
}

void otk_label_set_font_size(otk_widget_t *this, int font_size)
{
  otk_label_t *label = (otk_label_t *) this;
  label->font_size = font_size;
}


/** layout widget */

static void layout_event (otk_widget_t *this, otk_widget_event_type_t type) {
  otk_layout_t *lay = (otk_layout_t*) this;

  if (type == OTK_WG_EVENT_DRAW) {
    xitk_dnode_t *node;
    unsigned int sizey, sizex;

    sizey = lay->widget.h / lay->rows;
    sizex = lay->widget.w / lay->columns;

    for (node = lay->subs.head.next; node->next; node = node->next) {
      otk_widget_t *widget;
      xitk_container (widget, node, layout_node);
      if (widget->needcalc) {
        widget->x = lay->widget.x + widget->x * sizex;
        widget->y = lay->widget.y + widget->y * sizey;
        widget->w = sizex * widget->w;
        widget->h = sizey * widget->h;
        widget->needcalc = 0;
      }
      widget->event (widget, OTK_WG_EVENT_DRAW);
    }
  } else if (type == OTK_WG_EVENT_DESTROY) {
    while (lay->subs.head.next->next) {
      otk_widget_t *w;
      xitk_container (w, lay->subs.head.next, layout_node);
      _otk_widget_destroy (w);
    }
  }
}

void otk_layout_add_widget(otk_widget_t *layout, otk_widget_t *widget, int x, int y, int w, int h) {
  otk_layout_t *lay = (otk_layout_t*) layout;
  otk_window_t *win = (otk_window_t*) layout;

  if(is_correct_widget(layout, OTK_WIDGET_LAYOUT)) {
    otk_window_t *parent = (otk_window_t *) lay->widget.win;
    widget->x = x;
    widget->y = y;
    widget->w = w;
    widget->h = h;
    widget->win = layout;
    widget->otk = lay->widget.otk;
    widget->odk = lay->widget.odk;

    xitk_dlist_add_tail (&lay->subs, &widget->layout_node);
    xitk_dlist_add_tail (&parent->subs, &widget->win_node);
  }
  else if(is_correct_widget(layout, OTK_WIDGET_WINDOW)) {
    widget->x = win->widget.x + x;
    widget->y = win->widget.y + y;
    widget->w = w;
    widget->h = h;
    widget->win = (otk_widget_t*) win;
    widget->otk = win->widget.otk;
    widget->odk = win->widget.odk;

    xitk_dlist_add_tail (&win->subs, &widget->win_node);
  }
  else return;
}

otk_widget_t *otk_layout_new (otk_widget_t *win, int x, int y, int w, int h, int rows, int columns) {
  otk_layout_t *layout;
  otk_window_t *window = (otk_window_t *) win;

  if (!is_correct_widget(win, OTK_WIDGET_WINDOW)) return NULL;

  layout = ho_new(otk_layout_t);

  xitk_dnode_init (&layout->widget.win_node);
  xitk_dnode_init (&layout->widget.layout_node);
  layout->widget.widget_type = OTK_WIDGET_LAYOUT;
  layout->widget.x           = window->widget.x+x;
  layout->widget.y           = window->widget.y+y;
  layout->widget.w           = w;
  layout->widget.h           = h;
  layout->widget.win         = win;
  layout->widget.otk         = win->otk;
  layout->widget.odk         = win->odk;
  layout->widget.event       = layout_event;
  layout->widget.selectable  = 0;
  layout->widget.needupdate  = 0;
  layout->widget.needdraw    = 1;
  layout->widget.major       = 0;
  layout->widget.text        = NULL;
  layout->rows               = rows;
  layout->columns            = columns;
  xitk_dlist_init (&layout->subs);

  xitk_dlist_add_tail (&window->subs, &layout->widget.win_node);

  return (otk_widget_t*) layout;
}

/** selector widget */

static void selector_event (otk_widget_t *this, otk_widget_event_type_t type) {
  otk_selector_t *selector = (otk_selector_t*) this;

  if (type == OTK_WG_EVENT_DRAW) {
    otk_node_string_t *s;

    xitk_container (s, sized_list_get (&selector->items, selector->pos - 1), node);
    if (!s)
      return;

    /* TJ. due to a temporary bug in this function, i had the glitch of drawing
     * a freshly changed selector in smaller font. i find that nice, and redo it here.
     * BTW. this revealed another bug in libxine "cetus" font who has the small letter "f"
     * shifted down a few pixels for some reason there. */
    odk_set_font (this->odk, this->otk->button_font,
      (this->needdraw >= 2) ? (3 * this->otk->button_font_size) >> 2 : this->otk->button_font_size);

    if (this->focus)
      odk_draw_rect (this->odk, this->x, this->y, this->x+this->w, this->y+this->h, 1,
        this->otk->palette.index[OTK_IDX_focused_button]);
    odk_draw_text (this->odk, this->x + (this->w >> 1), this->y + (this->h >> 1), s->s, ODK_ALIGN_CENTER | ODK_ALIGN_VCENTER,
      this->otk->palette.index[_otk_idx[0 + (this->focus & 1)]]);
  } else if (type == OTK_WG_EVENT_DESTROY) {
    sized_list_free (&selector->items, (void (*) (xitk_dnode_t *))free);
  } else if (type == OTK_WG_EVENT_SELECT) {
    selector->pos++;
    if ((selector->pos < 1) || (selector->pos > selector->items.num))
      selector->pos = 1;
    _otk_widget_needdraw (this);
    this->needdraw |= 2; /** << see selector_draw (). */
    if (selector->widget.user_cb)
      selector->widget.user_cb (&selector->widget, selector->widget.user_data, OTK_ACT_VALUE, selector->pos);
  }
}

otk_widget_t *otk_selector_grid_new (const char *const *items, int num,
  otk_widget_cb_t user_cb, void *user_data) {
  otk_selector_t *selector;
  int i;

  selector = ho_new(otk_selector_t);

  xitk_dnode_init (&selector->widget.win_node);
  xitk_dnode_init (&selector->widget.layout_node);
  selector->widget.widget_type = OTK_WIDGET_SELECTOR;
  selector->widget.selectable  = 1;
  selector->widget.event       = selector_event;
  selector->widget.needupdate  = 0;
  selector->widget.needdraw    = 1;
  selector->widget.needcalc    = 1;
  selector->widget.major       = 1;
  selector->widget.text        = NULL;
  selector->widget.user_cb     = user_cb;
  selector->widget.user_data   = user_data;

  sized_list_init (&selector->items);
  for (i = 0; i < num; i++) {
    size_t l = strlen (items[i]);
    otk_node_string_t *s = malloc (sizeof (*s) + l);
    if (!s)
      break;
    xitk_dnode_init (&s->node);
    memcpy (s->s, items[i], l + 1);
    sized_list_append (&selector->items, &s->node);
  }
  selector->num = i;
  selector->pos = 1;

  return &selector->widget;
}

otk_widget_t *otk_selector_new (otk_widget_t *win,int x, int y, int w, int h,
  const char *const *items, int num, otk_widget_cb_t user_cb, void *user_data) {
  otk_selector_t *selector;
  otk_window_t *window = (otk_window_t*) win;
  int i;

  if (!is_correct_widget(win, OTK_WIDGET_WINDOW)) return NULL;

  selector = ho_new(otk_selector_t);

  xitk_dnode_init (&selector->widget.win_node);
  xitk_dnode_init (&selector->widget.layout_node);
  selector->widget.widget_type = OTK_WIDGET_SELECTOR;
  selector->widget.x           = win->x+x;
  selector->widget.y           = win->y+y;
  selector->widget.w           = w;
  selector->widget.h           = h;
  selector->widget.win         = win;
  selector->widget.otk         = win->otk;
  selector->widget.odk         = win->otk->odk;
  selector->widget.selectable  = 1;
  selector->widget.event       = selector_event;
  selector->widget.needupdate  = 0;
  selector->widget.needdraw    = 1;
  selector->widget.major       = 0;
  selector->widget.text        = NULL;
  selector->widget.user_cb     = user_cb;
  selector->widget.user_data   = user_data;

  sized_list_init (&selector->items);
  for (i = 0; i < num; i++) {
    size_t l = strlen (items[i]);
    otk_node_string_t *s = malloc (sizeof (*s) + l);
    if (!s)
      break;
    xitk_dnode_init (&s->node);
    memcpy (s->s, items[i], l + 1);
    sized_list_append (&selector->items, &s->node);
  }
  selector->num = i;
  selector->pos = 1;

  xitk_dlist_add_tail (&window->subs, &selector->widget.win_node);

  return &selector->widget;
}

void otk_selector_set(otk_widget_t *this, int pos) {
  otk_selector_t *selector = (otk_selector_t*) this;

  if (!is_correct_widget(this, OTK_WIDGET_SELECTOR)) return;

  if (pos < 1)
    pos = 1;
  if (pos > selector->num - 1)
    pos = selector->num - 1;
  selector->pos = pos;
}

/** window widget */

static void window_event (otk_widget_t *this, otk_widget_event_type_t type) {
  otk_window_t *win = (otk_window_t*) this;

  if (type == OTK_WG_EVENT_DRAW) {
    otk_widget_t *widget;

    if (this->otk->needdraw | this->needdraw) {
      odk_draw_rect (this->odk, this->x, this->y, this->x + this->w, this->y + this->h, 1, win->fill_color);
      xitk_container (widget, win->subs.head.next, win_node);
      while (widget->win_node.next) {
        /*  printf ("gui.oxine.window.widget.draw (%p, %d, (%d, %d, %d, %d)).\n",
          (void *)widget, widget->widget_type, widget->x, widget->y, widget->w, widget->h); */
        if (!widget->major) {
          widget->event (widget, OTK_WG_EVENT_DRAW);
          widget->needdraw = 0;
          this->otk->num_drawn++;
        }
        xitk_container (widget, widget->win_node.next, win_node);
      }
      this->needdraw = 0;
    } else {
      xitk_container (widget, win->subs.head.next, win_node);
      while (widget->win_node.next) {
        /*  printf ("gui.oxine.window.widget.draw (%p, %d, (%d, %d, %d, %d)).\n",
          (void *)widget, widget->widget_type, widget->x, widget->y, widget->w, widget->h); */
        if (!widget->major && widget->needdraw) {
          odk_draw_rect (this->odk, widget->x, widget->y, widget->x + widget->w, widget->y + widget->h, 1, win->fill_color);
          widget->event (widget, OTK_WG_EVENT_DRAW);
          widget->needdraw = 0;
          this->otk->num_drawn++;
        }
        xitk_container (widget, widget->win_node.next, win_node);
      }
    }
  } else if (type == OTK_WG_EVENT_DESTROY) {
    while (1) {
      otk_widget_t *w;
      xitk_container (w, win->subs.head.next, win_node);
      if (!w->win_node.next)
        break;
      xitk_dnode_remove (&w->win_node);
      if (!w->major)
        _otk_widget_destroy (w);
    }
    xitk_dlist_init (&win->subs);
  }
}

otk_widget_t *otk_window_new (otk_t *otk, const char *title, int x, int y,
			      int w, int h) {

  otk_window_t *window;

  window = ho_new(otk_window_t);

  xitk_dnode_init (&window->widget.win_node);
  xitk_dnode_init (&window->widget.layout_node);
  window->widget.widget_type = OTK_WIDGET_WINDOW;
  window->widget.x           = x;
  window->widget.y           = y;
  window->widget.w           = w;
  window->widget.h           = h;
  window->widget.otk         = otk;
  window->widget.odk         = otk->odk;
  window->widget.event       = window_event;
  window->widget.needupdate  = 0;
  window->widget.needdraw    = 1;
  window->widget.major       = 0;
  window->widget.text        = ho_strdup (title);
  window->fill_color         = otk->palette.index[OTK_IDX_text_window]+1;

  xitk_dlist_init (&window->subs);

  xitk_dlist_add_tail (&otk->windows, &window->widget.win_node);

  return &window->widget;
}

/** other global functions */

void otk_draw_all (otk_t *otk) {
  otk->needdraw = 2;
  return _otk_redraw (otk);
}

void otk_destroy_widget (otk_widget_t *w) {
  if (w)
    _otk_widget_destroy (w);
}

void otk_set_focus (otk_widget_t *widget) {
  if (!widget)
    return;
  if (!widget->selectable || !widget->otk)
    return;
  if (widget->otk->focus_ptr && (widget != widget->otk->focus_ptr)) {
    widget->otk->focus_ptr->focus = 0;
    _otk_widget_needdraw (widget->otk->focus_ptr);
  }
  widget->focus = 1;
  _otk_widget_needdraw (widget);
  widget->otk->focus_ptr = widget;
}

void otk_set_event_handler(otk_t *otk, int (*cb)(void *data, oxine_event_t *ev), void *data) {

  otk->event_handler = cb;
  otk->event_handler_data = data;
}

#ifdef YET_UNUSED
void otk_set_update (otk_widget_t *this, int update) {
  if (this)
    this->needupdate = !!update;
}
#endif

/* FIXME: is this really thread safe?
 * check if all otk updates need lock_job_mutex()
 */
#ifdef ENABLE_UPDATE_JOB
static void otk_update_job (xitk_t *xitk, void *data) {
  otk_t *otk = (otk_t *) data;
  int changed = 0;
  otk_window_t *win;
  otk_widget_t *widget;

  xitk_lock (xitk, 1);
  xitk_container (win, otk->windows.head.next, widget.win_node);
  while (win->widget.win_node.next) {

    xitk_container (widget, win->subs.head.next, win_node);
    while (widget->win_node.next) {

      if(widget->needupdate) {
	changed = 1;
	switch (widget->widget_type) {
#ifdef OTK_WITH_SLIDER
	case OTK_WIDGET_SLIDER:
	  {
	    otk_slider_t *slider = (otk_slider_t *) widget;
	    if(slider->get_value) {
	      slider->old_pos = slider->get_value(widget->odk);
	    }
            widget->event (widget, OTK_WG_EVENT_DRAW);
	    break;
	  }
#endif
	case OTK_WIDGET_BUTTON:
	  {
	    otk_button_t *button = (otk_button_t *) widget;
	    if(button->uc) {
	      button->uc(button->uc_data);
              widget->event (widget, OTK_WG_EVENT_DRAW);
	    }
	    break;
	  }
	default:
          widget->event (widget, OTK_WG_EVENT_DRAW);
	}
      }
      xitk_container (widget, widget->win_node.next, win_node);
    }
    xitk_container (win, win->widget.win_node.next, widget.win_node);
  }
  if (changed)
    odk_show (otk->odk);
  otk->update_job = xitk_at_add (otk->xitk, otk_update_job, otk, 1000);
  xitk_lock (xitk, 0);
}
#endif

void otk_clear(otk_t *otk) {
  if (!otk)
    return;

  while (1) {
    otk_window_t *win;
    xitk_container (win, otk->windows.head.next, widget.win_node);
    if (!win->widget.win_node.next)
      break;
    _otk_widget_destroy (&win->widget);
  }

  xitk_dlist_init (&otk->windows);
  if (odk_palette_update (&otk->palette))
    odk_palette_set (otk->odk, &otk->palette);
  otk_draw_all (otk);
}

static const odk_user_color_t uc[] = {
  [OTK_IDX_text]                = { 0xffff00ff, "3text_foreground",         N_("OSD text color") },
  [OTK_IDX_text_window]         = { 0x0080c0aa, "2text_window",             N_("OSD text pad color") },
  [OTK_IDX_text_border]         = { 0x008000ff, "4text_border",             N_("OSD text outline color") },
  [OTK_IDX_focused_text]        = { 0xff8080ff, "3focused_text_foreground", N_("OSD focused text color") },
  [OTK_IDX_focused_button]      = { 0x8080c0aa, "2focused_button",          N_("OSD focused text pad color") },
  [OTK_IDX_focused_text_border] = { 0x808080ff, "4focused_text_border",     N_("OSD focused text outline color") },
  [OTK_IDX_label]               = { 0xc08080ff, "3label_foreground",        N_("OSD label color") },
  [OTK_IDX_label_window]        = { 0x0080c0ff, "2label_window",            N_("OSD label pad color") },
  [OTK_IDX_label_border]        = { 0x0080c0ff, "4label_border",            N_("OSD label outline color") },
  [OTK_IDX_slider]              = { 0x008000ff, "1slider",                  N_("OSD slider color") },
  [OTK_IDX_slider_knob]         = { 0xffff00ff, "1slider_knob",             N_("OSD slider knob color") },
  [OTK_IDX_focused_slider]      = { 0x808080ff, "1focused_slider",          N_("OSD focused slider color") },
  [OTK_IDX_focused_slider_knob] = { 0xff8080ff, "1focused_slider_knob",     N_("OSD focused slider knob color") }
};

otk_t *otk_init (oxine_t *oxine, int verbosity) {
  otk_t *otk;
  /* int err; */

  otk = ho_new_tagged(otk_t,"otk object");

  otk->oxine = oxine;
  otk->odk = oxine->odk;
  otk->xitk = oxine->xitk;
  otk->verbosity = verbosity;

  xitk_dlist_init (&otk->windows);
  odk_set_event_handler (otk->odk, otk_event_handler, otk);

  odk_palette_init (&otk->palette, otk->oxine->xine, "gui.osdmenu.color_", uc, sizeof (uc) / sizeof (uc[0]));
  odk_palette_set (otk->odk, &otk->palette);

  otk->title_font = strdup("sans");
  otk->title_font_size = 40;
  otk->label_font = otk->title_font;
  otk->label_font_size = 40;
  otk->button_font = strdup ("cetus");
  otk->button_font_size = 40;

  pthread_mutex_init(&otk->draw_mutex, NULL);
#ifdef ENABLE_UPDATE_JOB
  otk->update_job = xitk_at_add (otk->xitk, otk_update_job, otk, 1000);
#else
  otk->update_job = 0;
#endif

  return otk;
}

void otk_free (otk_t *otk) {
#ifdef ENABLE_UPDATE_JOB
  xitk_at_remove (otk->xitk, otk_update_job, NULL, 0 /* otk->update_job */);
#endif
#ifdef HAVE_XINE_CONFIG_UNREGISTER_CALLBACKS
  xine_config_unregister_callbacks (otk->oxine->xine, NULL, NULL, otk, sizeof (*otk));
#endif

  otk_clear(otk);

  if (otk->title_font) free(otk->title_font);
  if (otk->button_font) free(otk->button_font);

  odk_set_event_handler(otk->odk, NULL, NULL);

  pthread_mutex_destroy(&otk->draw_mutex);
  ho_free (otk);
}
