/* vim: set ts=2 et sw=2 : */
/** @file icmp.c */
/*
 *  T50 - Experimental Mixed Packet Injector
 *
 *  Copyright (C) 2010 - 2025 - T50 developers
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stddef.h>
#include <assert.h>
#include <linux/ip.h>
#include <linux/icmp.h>

#include <t50_config.h>
#include <t50_cksum.h>
#include <t50_memalloc.h>
#include <t50_modules.h>
#include <t50_randomizer.h>

/**
 * ICMP packet header configuration.
 *
 * This function configures and sends the ICMP packet header.
 *
 * @param co Pointer to T50 configuration structure.
 * @param size Pointer to packet size (updated by the function).
 */
void icmp ( const config_options_T *const restrict co, unsigned int *restrict size )
{
  unsigned int greoptlen;   /* GRE options size. */

  struct iphdr *ip;

  /* ICMP header. */
  struct icmphdr *icmp;

  assert ( co );

  greoptlen = gre_opt_len ( co );
  *size = sizeof ( struct iphdr )   +
          sizeof ( struct icmphdr ) +
          greoptlen;

  /* Try to reallocate packet, if necessary */
  alloc_packet ( *size );

  /* IP Header structure making a pointer to Packet. */
  ip = ip_header ( packet, *size, co );

  /* GRE Encapsulation takes place. */
  gre_encapsulation ( packet, co,
                      sizeof ( struct iphdr ) +
                      sizeof ( struct icmphdr ) );

  char *begin_, *end_;

  begin_ = ( char * )( ip + 1 ) + greoptlen;

  /* ICMP Header structure making a pointer to Packet. */
  icmp = ( struct icmphdr * ) begin_;
 
  *icmp = ( struct icmphdr ) {
    .type             = co->icmp.type,
    .code             = co->icmp.code,
    .un.echo.id       = __RND ( co->icmp.id ),
    .un.echo.sequence = __RND ( co->icmp.sequence )
  };

  if ( co->icmp.type == ICMP_REDIRECT )
    switch ( co->icmp.code )
    {
      case ICMP_REDIR_HOST:
      case ICMP_REDIR_NET:
        icmp->un.gateway = INADDR_RND ( co->icmp.gateway );
    }

  end_ = ( char * )( icmp + 1 );

  /* Computing the checksum. */
  icmp->checksum = co->bogus_csum ? RANDOM() : htons ( cksum ( begin_, end_ - begin_ ) );

  /* GRE Encapsulation takes place. */
  gre_checksum ( packet, co, *size );
}
