% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lda_tidiers.R
\name{lda_tidiers}
\alias{lda_tidiers}
\alias{tidy.LDA}
\alias{tidy.CTM}
\alias{augment.LDA}
\alias{augment.CTM}
\alias{glance.LDA}
\alias{glance.CTM}
\title{Tidiers for LDA and CTM objects from the topicmodels package}
\usage{
\method{tidy}{LDA}(x, matrix = c("beta", "gamma"), log = FALSE, ...)

\method{tidy}{CTM}(x, matrix = c("beta", "gamma"), log = FALSE, ...)

\method{augment}{LDA}(x, data, ...)

\method{augment}{CTM}(x, data, ...)

\method{glance}{LDA}(x, ...)

\method{glance}{CTM}(x, ...)
}
\arguments{
\item{x}{An LDA or CTM (or LDA_VEM/CTA_VEM) object from the topicmodels package}

\item{matrix}{Whether to tidy the beta (per-term-per-topic, default)
or gamma (per-document-per-topic) matrix}

\item{log}{Whether beta/gamma should be on a log scale, default FALSE}

\item{...}{Extra arguments, not used}

\item{data}{For \code{augment}, the data given to the LDA or CTM function, either
as a DocumentTermMatrix or as a tidied table with "document" and "term"
columns}
}
\value{
\code{tidy} returns a tidied version of either the beta or gamma matrix.

If \code{matrix == "beta"} (default), returns a table with one row per topic and term,
with columns
\describe{
\item{topic}{Topic, as an integer}
\item{term}{Term}
\item{beta}{Probability of a term generated from a topic according to
the multinomial model}
}

If \code{matrix == "gamma"}, returns a table with one row per topic and document,
with columns
\describe{
\item{topic}{Topic, as an integer}
\item{document}{Document name or ID}
\item{gamma}{Probability of topic given document}
}

\code{augment} returns a table with one row per original
document-term pair, such as is returned by \link{tdm_tidiers}:
\describe{
\item{document}{Name of document (if present), or index}
\item{term}{Term}
\item{.topic}{Topic assignment}
}

If the \code{data} argument is provided, any columns in the original
data are included, combined based on the \code{document} and \code{term}
columns.

\code{glance} always returns a one-row table, with columns
\describe{
\item{iter}{Number of iterations used}
\item{terms}{Number of terms in the model}
\item{alpha}{If an LDA_VEM, the parameter of the Dirichlet distribution
for topics over documents}
}
}
\description{
Tidy the results of a Latent Dirichlet Allocation or Correlated Topic Model.
}
\examples{

if (requireNamespace("topicmodels", quietly = TRUE)) {
  set.seed(2016)
  library(dplyr)
  library(topicmodels)

  data("AssociatedPress", package = "topicmodels")
  ap <- AssociatedPress[1:100, ]
  lda <- LDA(ap, control = list(alpha = 0.1), k = 4)

  # get term distribution within each topic
  td_lda <- tidy(lda)
  td_lda

  library(ggplot2)

  # visualize the top terms within each topic
  td_lda_filtered <- td_lda \%>\%
    filter(beta > .004) \%>\%
    mutate(term = reorder(term, beta))

  ggplot(td_lda_filtered, aes(term, beta)) +
    geom_bar(stat = "identity") +
    facet_wrap(~ topic, scales = "free") +
    theme(axis.text.x = element_text(angle = 90, size = 15))

  # get classification of each document
  td_lda_docs <- tidy(lda, matrix = "gamma")
  td_lda_docs

  doc_classes <- td_lda_docs \%>\%
    group_by(document) \%>\%
    top_n(1) \%>\%
    ungroup()

  doc_classes

  # which were we most uncertain about?
  doc_classes \%>\%
    arrange(gamma)
}

}
