"""Plugin for adding checks to run on the worker node."""

import os
import re
from arcnagios.ce.jobplugin import JobPlugin
from arcnagios.nagutils import OK, CRITICAL, UNKNOWN

_presence_script = """\
missing=
for prog in %(required_programs)s; do
    type -p $prog >/dev/null 2>&1 || missing="$missing $prog"
done
if test -n "$missing"; then
    echo "__missing$missing" >%(output_file)s
else
    %(script_line)s
    err=$?
    [ $err -eq 0 ] || echo "__exit $err" >>%(output_file)s
fi
"""
_plain_script = """\
%(script_line)s
err=$?
[ $err -eq 0 ] || echo "__exit $err" >>%(output_file)s
"""

class ScriptedJobPlugin(JobPlugin):
    _missing_re = re.compile(r'__missing\s+(.*)')
    _exit_re = re.compile(r'__exit\s+(\d+)')
    _status_re = re.compile(r'__status\s+(\d+)\s+(.*)')
    _log_re = re.compile(r'__log\s+(\d+)\s+(.*)')

    def write_script(self, fh):
        script_line = self.getconf('script_line')
        fh.write('# Scripted test %s\n'%self.service_description)
        output_file = self.getconf('output_file')
        env = {
            'script_line': script_line,
            'output_file': output_file
        }
        if self.hasconf('required_programs'):
            env['required_programs'] = self.getconf('required_programs')
            fh.write(_presence_script % env)
        else:
            fh.write(_plain_script % env)
        fh.write('\n')

    def staged_inputs(self):
        return self.getconf_strlist('staged_inputs', default = [])

    def staged_outputs(self):
        return [(self.getconf('output_file'), None, [])]

    def check(self, report, jobdir, stored_urls):
        output_file = os.path.join(jobdir, self.getconf('output_file'))
        if self.hasconf('output_pattern'):
            pattern_re = re.compile(self.getconf('output_pattern'))
        else:
            pattern_re = None
        try:
            fh = open(output_file)
        except IOError:
            report.update_status(UNKNOWN,
                    'Did not receive output file %s.'%output_file)
            return
        for ln in fh:
            if pattern_re:
                mo = re.match(pattern_re, ln)
                if mo:
                    msg = self.getconf('status_ok', vars = mo.groupdict())
                    report.update_status(OK, msg)
                    fh.close()
                    return
            mo = re.match(self._missing_re, ln)
            if mo:
                msg = 'Missing program(s) %s'%mo.group(1)
                report.update_status(CRITICAL, msg)
                break
            mo = re.match(self._exit_re, ln)
            if mo:
                code = int(mo.group(1))
                if code:
                    msg = 'Script exited with code %d.'%code
                    report.update_status(CRITICAL, msg)
                    continue
            mo = re.match(self._status_re, ln)
            if mo:
                report.update_status(int(mo.group(1)), mo.group(2))
                continue
            mo = re.match(self._log_re, ln)
            if mo:
                report.log.log(int(mo.group(1)), mo.group(2))
                continue
        fh.close()
        if pattern_re:
            report.update_status(CRITICAL,
                self.getconf('status_critical', default = 'Pattern not found.'))
