// @HEADER
// *****************************************************************************
//           Trilinos: An Object-Oriented Solver Framework
//
// Copyright 2001-2024 NTESS and the Trilinos contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER
                                                                                
// Finite Element Problem Class
/* Provides function (F) and Jacobian evaluations for the following equation 
 * via a 1D linear finite element discretization with Epetra objects.
 *
 * d2u 
 * --- - k * u**2 = 0
 * dx2
 *
 * subject to @ x=0, u=1
 */

#ifndef _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H
#define _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H

// ml objects
#include "ml_common.h"
#if defined(HAVE_ML_NOX) && defined(HAVE_ML_EPETRA) && defined(HAVE_ML_AZTECOO) && defined(HAVE_ML_TEUCHOS) && defined(HAVE_ML_IFPACK) && defined(HAVE_ML_AMESOS) && defined(HAVE_ML_EPETRAEXT)

// Forward Declarations
class Epetra_Comm;
class Epetra_Map; 
class Epetra_Vector;
class Epetra_Import;
class Epetra_CrsGraph;
class Epetra_CrsMatrix;

// Flag to tell the evaluate routine what objects to fill
enum FillType {F_ONLY, MATRIX_ONLY, ALL}; 

// Finite Element Problem Class
class FiniteElementProblem { 

public:

  // Constructor
  FiniteElementProblem(int NumGlobalElements, Epetra_Comm& Comm);
  
  // Destructor
  ~FiniteElementProblem();

  // Evaluates the function (F) and/or the Jacobian using the solution 
  // values in solnVector.
  bool evaluate(FillType f, const Epetra_Vector *solnVector, 
		Epetra_Vector *rhsVector, Epetra_RowMatrix *matrix);

  // Return a reference to the Epetra_Vector with the initial guess
  // that is generated by the FiniteElementProblem class.
  Epetra_Vector& getSolution();
  
  // Return a reference to the Epetra_Vector with the Jacobian
  // that is generated by the FiniteElementProblem class.
  Epetra_CrsMatrix& getJacobian()
  {return *A;}

  // return a reference to the StandardMap
  const Epetra_Map& getMap() {return *StandardMap;}

  // return reference to the Graph
  const Epetra_CrsGraph& getGraph() { return *AA; } 

  // return ref to Comm
  Epetra_Comm& getComm() { return *Comm;}

  // get nullspace
  double* Get_Nullspace(const int nummyrows, const int numpde, const int dim_nullsp)
  {
     return NULL;
  }
  
  // get block info
  bool getblockinfo(int* nblocks, int** blocks, int** block_pde)
  {
     *nblocks = 0;
     *blocks = 0;
     *block_pde = 0;
     return false;
  }
  
private:

  // inserts the global column indices into the Graph
  Epetra_CrsGraph& generateGraph(Epetra_CrsGraph& AA);

private:

  FillType flag;
  Epetra_Map *StandardMap; 
  Epetra_Map *OverlapMap;
  Epetra_Import *Importer;
  Epetra_Vector *initialSolution;
  Epetra_Vector *rhs;
  Epetra_CrsGraph *AA;
  Epetra_CrsMatrix *A;
  Epetra_Comm *Comm;

  int MyPID;              // Process number
  int NumProc;            // Total number of processes
  int NumMyElements;      // Number of elements owned by this process
  int NumGlobalElements;  // Total Number of elements

};
#endif // defined(HAVE_ML_NOX) && defined(HAVE_ML_EPETRA) && defined(HAVE_ML_AZTECOO)
#endif




