% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/deriv.R
\name{deriv}
\alias{deriv}
\alias{deriv.BSpline}
\alias{deriv.MSpline}
\alias{deriv.ISpline}
\alias{deriv.CSpline}
\alias{deriv.BernsteinPoly}
\alias{deriv.NaturalSpline}
\alias{deriv.NaturalSplineK}
\title{Derivatives of Spline Basis Functions}
\usage{
\method{deriv}{BSpline}(expr, derivs = 1L, ...)

\method{deriv}{MSpline}(expr, derivs = 1L, ...)

\method{deriv}{ISpline}(expr, derivs = 1L, ...)

\method{deriv}{CSpline}(expr, derivs = 1L, ...)

\method{deriv}{BernsteinPoly}(expr, derivs = 1L, ...)

\method{deriv}{NaturalSpline}(expr, derivs = 1L, ...)

\method{deriv}{NaturalSplineK}(expr, derivs = 1L, ...)
}
\arguments{
\item{expr}{Objects of class \code{bSpline2}, \code{ibs}, \code{mSpline},
\code{iSpline}, \code{cSpline}, \code{bernsteinPoly} or
\code{naturalSpline} with attributes describing \code{knots},
\code{degree}, etc.}

\item{derivs}{A positive integer specifying the order of derivatives. By
default, it is \code{1L} for the first derivatives.}

\item{...}{Optional arguments that are not used.}
}
\value{
A numeric matrix of the same dimension with the input \code{expr}.
}
\description{
Returns derivatives of given order for the given spline basis functions.
}
\details{
At knots, the derivative is defined to be the right derivative except at the
right boundary knot. By default, the function returns the first derivatives.
For derivatives of order greater than one, nested function calls such as
\code{deriv(deriv(expr))} are supported but not recommended.  For a better
performance, argument \code{derivs} should be specified instead.

This function is designed for objects produced by this package.  It
internally extracts necessary specification about the spline/polynomial
basis matrix from its attributes. Therefore, the function will not work if
the key attributes are not available after some operations.
}
\examples{
library(splines2)

x <- c(seq.int(0, 1, 0.1), NA)  # NA's will be kept.
knots <- c(0.3, 0.5, 0.6)

## helper function
stopifnot_equivalent <- function(...) {
    stopifnot(all.equal(..., check.attributes = FALSE))
}

## integal of B-splines and the corresponding B-splines integrated
ibsMat <- ibs(x, knots = knots)
bsMat <- bSpline(x, knots = knots)

## the first derivative
d1Mat <- deriv(ibsMat)
stopifnot_equivalent(bsMat, d1Mat)

## the second derivative
d2Mat1 <- deriv(bsMat)
d2Mat2 <- deriv(ibsMat, derivs = 2L)
stopifnot_equivalent(d2Mat1, d2Mat2)

## nested calls are supported
d2Mat3 <- deriv(deriv(ibsMat))
stopifnot_equivalent(d2Mat2, d2Mat3)

## C-splines, I-splines, M-splines and the derivatives
csMat <- cSpline(x, knots = knots, intercept = TRUE, scale = FALSE)
isMat <- iSpline(x, knots = knots, intercept = TRUE)
stopifnot_equivalent(isMat, deriv(csMat))

msMat <- mSpline(x, knots = knots, intercept = TRUE)
stopifnot_equivalent(msMat, deriv(isMat))
stopifnot_equivalent(msMat, deriv(csMat, 2))
stopifnot_equivalent(msMat, deriv(deriv(csMat)))

dmsMat <- mSpline(x, knots = knots, intercept = TRUE, derivs = 1)
stopifnot_equivalent(dmsMat, deriv(msMat))
stopifnot_equivalent(dmsMat, deriv(isMat, 2))
stopifnot_equivalent(dmsMat, deriv(deriv(isMat)))
stopifnot_equivalent(dmsMat, deriv(csMat, 3))
stopifnot_equivalent(dmsMat, deriv(deriv(deriv(csMat))))
}
