open StdLabels

let x : int option = _

module Yojson = struct
  module Safe = struct
    type t =
      [ `Null
      | `Bool of bool
      | `Int of int
      | `Intlit of string
      | `Float of float
      | `String of string
      | `Assoc of (string * t) list
      | `List of t list
      | `Tuple of t list
      | `Variant of string * t option ]

    let to_string : t -> string = fun _ -> ""
    let from_string : string -> t = fun s -> `String s
  end
end

module Yojsonable = struct end
module Uri0 = struct
  type t
  let t_of_yojson _ = assert false
  let yojson_of_t _ = assert false
end

module Yojson_conv = struct
  let of_yojson_error _ _ = assert false
  let yojson_of_list _ _ = `List []
  let list_of_yojson _ _ = []
  let yojson_of_array _ _ = `List []
  let array_of_yojson _ _ = [||]
  let int_of_yojson _ = 0
  let yojson_of_int x = `Int x
  let bool_of_yojson _ = false
  let yojson_of_bool b = `Bool b
  let string_of_yojson _ = ""
  let yojson_of_string x = `String x
  let option_of_yojson _ _ = None
  let yojson_of_option _ _ = `Null
end

include Yojson_conv

module Ppx_yojson_conv_lib = struct
  let ( ! ) x = !x
  let poly_equal = ( = )

  module Yojson = Yojson
  module Yojson_conv = Yojson_conv
  module Yojsonable = Yojsonable

  module Yojson_conv_error = struct
    let record_duplicate_fields _ _ _ = assert false
    let record_extra_fields _ _ _ = assert false
    let record_undefined_elements _ _ _ = assert false
    let record_list_instead_atom _ _ = assert false
  end
end

module Jsonrpc = struct
  module Structured = struct
    let yojson_of_t _ = assert false
  end

  module Id = struct
    type t
    let t_of_yojson _ = assert false
    let yojson_of_t _ = assert false
  end
end

let unit_of_yojson _ = ()
let yojson_of_unit _ = `Null

module Json = struct
  type t = Ppx_yojson_conv_lib.Yojson.Safe.t

  let to_string t = Yojson.Safe.to_string t
  let of_string s = Yojson.Safe.from_string s
  let yojson_of_t x = x
  let t_of_yojson x = x
  let error = Ppx_yojson_conv_lib.Yojson_conv.of_yojson_error
  let yojson_of_list = Ppx_yojson_conv_lib.Yojson_conv.yojson_of_list

  module Jsonable = Ppx_yojson_conv_lib.Yojsonable

  let bool b = `Bool b
  let field fields name conv = List.assoc_opt name fields |> Option.map conv

  let field_exn fields name conv =
    match field fields name conv with
    | Some f -> f
    | None -> error ("missing field: " ^ name) (`Assoc fields)

  module Conv = struct
    include Ppx_yojson_conv_lib.Yojson_conv
  end

  module E = struct
    let foobar = 10
    let foobaz = 11
    let foofoo = 12

    let x = fo
  end

  module O = struct
    let ( <|> ) c1 c2 json =
      match c1 json with
      | s -> s
      | exception _ -> c2 json
  end

  module Object = struct
    type json = t
    type nonrec t = (string * t) list

    let yojson_of_t t : json = `Assoc t

    let t_of_yojson (t : json) : t =
      match t with
      | `Assoc t -> t
      | json -> error "object expected" json
  end

  module Option = struct
    type 'a t = 'a option

    let yojson_of_t f = function
      | None -> `Null
      | Some x -> f x

    let t_of_yojson f = function
      | `Null -> None
      | json -> Some (f json)
  end

  module Of = struct
    let list = Ppx_yojson_conv_lib.Yojson_conv.list_of_yojson

    let pair f g json =
      match json with
      | `List [ x; y ] -> (f x, g y)
      | json -> error "pair" json

    let int_pair =
      let int = Ppx_yojson_conv_lib.Yojson_conv.int_of_yojson in
      pair int int

    let untagged_union (type a) name (xs : (t -> a) list) (json : t) : a =
      match
        List.find_map xs ~f:(fun conv -> try Some (conv json) with _ -> None)
      with
      | None -> error name json
      | Some x -> x

    let literal_field (type a) (name : string) (k : string) (v : string)
        (f : t -> a) (json : t) : a =
      match json with
      | `Assoc xs -> (
        let ks, xs =
          List.partition_map xs ~f:(fun (k', v') ->
              if k = k' then
                if `String v = v' then Left k
                else error (Format.sprintf "%s: incorrect key %s" name k) json
              else Right (k', v'))
        in
        match ks with
        | [] -> error (Format.sprintf "%s: key %s not found" name k) json
        | [ _ ] -> f (`Assoc xs)
        | _ :: _ -> error (Format.sprintf "%s: multiple keys %s" name k) json)
      | _ -> error (Format.sprintf "%s: not a record (key: %s)" name k) json
  end

  module To = struct
    let list f xs = `List (List.map ~f xs)

    let literal_field (type a) (k : string) (v : string) (f : a -> t) (t : a) :
        t =
      match f t with
      | `Assoc xs -> `Assoc ((k, `String v) :: xs)
      | _ -> invalid_arg "To.literal_field"

    let int_pair (x, y) = `List [ `Int x; `Int y ]
  end

  module Nullable_option = struct
    type 'a t = 'a option

    let t_of_yojson f = function
      | `Null -> None
      | json -> Some (f json)

    let yojson_of_t f = function
      | None -> assert false
      | Some s -> f s
  end

  module Assoc = struct
    type ('a, 'b) t = ('a * 'b) list

    let yojson_of_t f g xs =
      let f k =
        match f k with
        | `String s -> s
        | json -> error "Json.Assoc.yojson_of_t not a string key" json
      in
      `Assoc (List.map xs ~f:(fun (k, v) -> (f k, g v)))

    let t_of_yojson f g json =
      let f s = f (`String s) in
      match json with
      | `Assoc xs -> List.map xs ~f:(fun (k, v) -> (f k, g v))
      | _ -> error "Json.Assoc.t_of_yojson: not an object" json
  end

  module Void = struct
    type t

    let t_of_yojson = ()
    let yojson_of_t (_ : t) = assert false
  end

  let read_json_params f v =
    match f (Jsonrpc.Structured.yojson_of_t v) with
    | r -> Ok r
    | exception _ -> Error "msg"

  let require_params json =
    match json with
    | None -> Error "params are required"
    | Some params -> Ok params

  let message_params params f =
    match require_params params with
    | Error e -> Error e
    | Ok x -> read_json_params f x
end

module NotebookDocumentSyncOptions = struct
  type t = unit [@@deriving_inline yojson]

  let _ = fun (_ : t) -> ()
  let t_of_yojson = (unit_of_yojson : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)
  let _ = t_of_yojson
  let yojson_of_t = (yojson_of_unit : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)
  let _ = yojson_of_t

  [@@@end]
end

module NotebookDocumentSyncRegistrationOptions = struct
  type t = unit [@@deriving_inline yojson]

  let _ = fun (_ : t) -> ()
  let t_of_yojson = (unit_of_yojson : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)
  let _ = t_of_yojson
  let yojson_of_t = (yojson_of_unit : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)
  let _ = yojson_of_t

  [@@@end]
end

module NotebookDocumentFilter = struct
  type t = unit [@@deriving_inline yojson]

  let _ = fun (_ : t) -> ()
  let t_of_yojson = (unit_of_yojson : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)
  let _ = t_of_yojson
  let yojson_of_t = (yojson_of_unit : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)
  let _ = yojson_of_t

  [@@@end]
end

module MarkedString = struct
  type t = { value : string; language : string option }

  let yojson_of_t { value; language } =
    match language with
    | None -> `String value
    | Some language ->
      `Assoc [ ("value", `String value); ("language", `String language) ]

  let t_of_yojson json =
    match json with
    | `String value -> { value; language = None }
    | `Assoc fields ->
      let value = Json.field_exn fields "value" Json.Conv.string_of_yojson in
      let language =
        Json.field_exn fields "language" Json.Conv.string_of_yojson
      in
      { value; language = Some language }
    | _ -> Json.error "invalid MarkedString" json
end

module DocumentUri = Uri0

module ProgressToken = struct
  type t = [ `Int of int | `String of string ]

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String j -> `String j
    | `Int i -> `Int i
    | _ -> Json.error "invalid ProgressToken" json

  let yojson_of_t (t : t) : Json.t = (t :> Json.t)
end

module ProgressParams = struct
  type 'a t = { token : ProgressToken.t; value : 'a }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : 'a t) -> ()

  let t_of_yojson :
      'a.
      (Ppx_yojson_conv_lib.Yojson.Safe.t -> 'a) ->
      Ppx_yojson_conv_lib.Yojson.Safe.t ->
      'a t =
    let _tp_loc = "lsp/src/types.ml.ProgressParams.t" in
    fun _of_a -> function
      | `Assoc field_yojsons as yojson -> (
        let token_field = ref Option.None
        and value_field = ref Option.None
        and duplicates = ref []
        and extra = ref [] in
        let rec iter = function
          | (field_name, _field_yojson) :: tail ->
            (match field_name with
            | "token" -> (
              match Ppx_yojson_conv_lib.( ! ) token_field with
              | Option.None ->
                let fvalue = ProgressToken.t_of_yojson _field_yojson in
                token_field := Option.Some fvalue
              | Option.Some _ ->
                duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates
              )
            | "value" -> (
              match Ppx_yojson_conv_lib.( ! ) value_field with
              | Option.None ->
                let fvalue = _of_a _field_yojson in
                value_field := Option.Some fvalue
              | Option.Some _ ->
                duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates
              )
            | _ -> ());
            iter tail
          | [] -> ()
        in
        iter field_yojsons;
        match Ppx_yojson_conv_lib.( ! ) duplicates with
        | _ :: _ ->
          Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
            (Ppx_yojson_conv_lib.( ! ) duplicates)
            yojson
        | [] -> (
          match Ppx_yojson_conv_lib.( ! ) extra with
          | _ :: _ ->
            Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
              (Ppx_yojson_conv_lib.( ! ) extra)
              yojson
          | [] -> (
            match
              ( Ppx_yojson_conv_lib.( ! ) token_field,
                Ppx_yojson_conv_lib.( ! ) value_field )
            with
            | Option.Some token_value, Option.Some value_value ->
              { token = token_value; value = value_value }
            | _ ->
              Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
                _tp_loc yojson
                [ ( Ppx_yojson_conv_lib.poly_equal
                      (Ppx_yojson_conv_lib.( ! ) token_field)
                      Option.None,
                    "token" );
                  ( Ppx_yojson_conv_lib.poly_equal
                      (Ppx_yojson_conv_lib.( ! ) value_field)
                      Option.None,
                    "value" )
                ])))
      | _ as yojson ->
        Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
          yojson

  let _ = t_of_yojson

  let yojson_of_t :
      'a.
      ('a -> Ppx_yojson_conv_lib.Yojson.Safe.t) ->
      'a t ->
      Ppx_yojson_conv_lib.Yojson.Safe.t =
   fun _of_a -> function
    | { token = v_token; value = v_value } ->
      let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
      let bnds =
        let arg = _of_a v_value in
        ("value", arg) :: bnds
      in
      let bnds =
        let arg = ProgressToken.yojson_of_t v_token in
        ("token", arg) :: bnds
      in
      `Assoc bnds

  let _ = yojson_of_t

  [@@@end]

  let create ~(token : ProgressToken.t) ~value = { token; value }
end

module TextDocumentFilter = struct
  type t =
    { language : string option;
      scheme : string option;
      pattern : string option
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.TextDocumentFilter.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let language_field = ref Option.None
       and scheme_field = ref Option.None
       and pattern_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "language" -> (
             match Ppx_yojson_conv_lib.( ! ) language_field with
             | Option.None ->
               let fvalue = option_of_yojson string_of_yojson _field_yojson in
               language_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "scheme" -> (
             match Ppx_yojson_conv_lib.( ! ) scheme_field with
             | Option.None ->
               let fvalue = option_of_yojson string_of_yojson _field_yojson in
               scheme_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "pattern" -> (
             match Ppx_yojson_conv_lib.( ! ) pattern_field with
             | Option.None ->
               let fvalue = option_of_yojson string_of_yojson _field_yojson in
               pattern_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) language_field,
               Ppx_yojson_conv_lib.( ! ) scheme_field,
               Ppx_yojson_conv_lib.( ! ) pattern_field )
           with
           | ( Option.Some language_value,
               Option.Some scheme_value,
               Option.Some pattern_value ) ->
             { language = language_value;
               scheme = scheme_value;
               pattern = pattern_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) language_field)
                     Option.None,
                   "language" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) scheme_field)
                     Option.None,
                   "scheme" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) pattern_field)
                     Option.None,
                   "pattern" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { language = v_language; scheme = v_scheme; pattern = v_pattern } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_option yojson_of_string v_pattern in
         ("pattern", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_option yojson_of_string v_scheme in
         ("scheme", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_option yojson_of_string v_language in
         ("language", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?language ?scheme ?pattern () = { language; scheme; pattern }
end

(*$ Lsp_gen.print_ml () *)
module SymbolTag = struct
  type t = Deprecated

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Deprecated -> `Int 1

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Deprecated
    | _ -> Json.error "Invalid value. Expected one of:\n1" json
end

module SymbolKind = struct
  type t =
    | File
    | Module
    | Namespace
    | Package
    | Class
    | Method
    | Property
    | Field
    | Constructor
    | Enum
    | Interface
    | Function
    | Variable
    | Constant
    | String
    | Number
    | Boolean
    | Array
    | Object
    | Key
    | Null
    | EnumMember
    | Struct
    | Event
    | Operator
    | TypeParameter

  let yojson_of_t (t : t) : Json.t =
    match t with
    | File -> `Int 1
    | Module -> `Int 2
    | Namespace -> `Int 3
    | Package -> `Int 4
    | Class -> `Int 5
    | Method -> `Int 6
    | Property -> `Int 7
    | Field -> `Int 8
    | Constructor -> `Int 9
    | Enum -> `Int 10
    | Interface -> `Int 11
    | Function -> `Int 12
    | Variable -> `Int 13
    | Constant -> `Int 14
    | String -> `Int 15
    | Number -> `Int 16
    | Boolean -> `Int 17
    | Array -> `Int 18
    | Object -> `Int 19
    | Key -> `Int 20
    | Null -> `Int 21
    | EnumMember -> `Int 22
    | Struct -> `Int 23
    | Event -> `Int 24
    | Operator -> `Int 25
    | TypeParameter -> `Int 26

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> File
    | `Int 2 -> Module
    | `Int 3 -> Namespace
    | `Int 4 -> Package
    | `Int 5 -> Class
    | `Int 6 -> Method
    | `Int 7 -> Property
    | `Int 8 -> Field
    | `Int 9 -> Constructor
    | `Int 10 -> Enum
    | `Int 11 -> Interface
    | `Int 12 -> Function
    | `Int 13 -> Variable
    | `Int 14 -> Constant
    | `Int 15 -> String
    | `Int 16 -> Number
    | `Int 17 -> Boolean
    | `Int 18 -> Array
    | `Int 19 -> Object
    | `Int 20 -> Key
    | `Int 21 -> Null
    | `Int 22 -> EnumMember
    | `Int 23 -> Struct
    | `Int 24 -> Event
    | `Int 25 -> Operator
    | `Int 26 -> TypeParameter
    | _ ->
      Json.error
        "Invalid value.\n\
         Expected one of: 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, \
         16, 17,\n\
         18, 19, 20, 21, 22, 23, 24, 25, 26"
        json
end

module ResourceOperationKind = struct
  type t = Create | Rename | Delete

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Create -> `String "create"
    | Rename -> `String "rename"
    | Delete -> `String "delete"

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "create" -> Create
    | `String "rename" -> Rename
    | `String "delete" -> Delete
    | _ ->
      Json.error
        "Invalid value. Expected one of:\n\"create\", \"rename\", \"delete\""
        json
end

module FailureHandlingKind = struct
  type t = Abort | Transactional | TextOnlyTransactional | Undo

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Abort -> `String "abort"
    | Transactional -> `String "transactional"
    | TextOnlyTransactional -> `String "textOnlyTransactional"
    | Undo -> `String "undo"

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "abort" -> Abort
    | `String "transactional" -> Transactional
    | `String "textOnlyTransactional" -> TextOnlyTransactional
    | `String "undo" -> Undo
    | _ ->
      Json.error
        "Invalid value. Expected one of: \"abort\", \"transactional\",\n\
         \"textOnlyTransactional\", \"undo\""
        json
end

module MarkupKind = struct
  type t = PlainText | Markdown

  let yojson_of_t (t : t) : Json.t =
    match t with
    | PlainText -> `String "plaintext"
    | Markdown -> `String "markdown"

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "plaintext" -> PlainText
    | `String "markdown" -> Markdown
    | _ ->
      Json.error "Invalid value. Expected one of: \"plaintext\", \"markdown\""
        json
end

module TokenFormat = struct
  type t = Relative

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Relative -> `String "relative"

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "relative" -> Relative
    | _ -> Json.error "Invalid value.\nExpected one of: \"relative\"" json
end

module PrepareSupportDefaultBehavior = struct
  type t = Identifier

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Identifier -> `Int 1

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Identifier
    | _ -> Json.error "Invalid value. Expected one of:\n1" json
end

module DiagnosticTag = struct
  type t = Unnecessary | Deprecated

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Unnecessary -> `Int 1
    | Deprecated -> `Int 2

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Unnecessary
    | `Int 2 -> Deprecated
    | _ -> Json.error "Invalid value. Expected one of: 1, 2" json
end

module FoldingRangeKind = struct
  type t = Comment | Imports | Region | Other of string

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Comment -> `String "comment"
    | Imports -> `String "imports"
    | Region -> `String "region"
    | Other s -> `String s

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "comment" -> Comment
    | `String "imports" -> Imports
    | `String "region" -> Region
    | `String s -> Other s
    | _ ->
      Json.error
        "Invalid\nvalue. Expected one of: \"comment\", \"imports\", \"region\""
        json
end

module InsertTextMode = struct
  type t = AsIs | AdjustIndentation

  let yojson_of_t (t : t) : Json.t =
    match t with
    | AsIs -> `Int 1
    | AdjustIndentation -> `Int 2

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> AsIs
    | `Int 2 -> AdjustIndentation
    | _ -> Json.error "Invalid value. Expected one of: 1, 2" json
end

module CompletionItemKind = struct
  type t =
    | Text
    | Method
    | Function
    | Constructor
    | Field
    | Variable
    | Class
    | Interface
    | Module
    | Property
    | Unit
    | Value
    | Enum
    | Keyword
    | Snippet
    | Color
    | File
    | Reference
    | Folder
    | EnumMember
    | Constant
    | Struct
    | Event
    | Operator
    | TypeParameter

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Text -> `Int 1
    | Method -> `Int 2
    | Function -> `Int 3
    | Constructor -> `Int 4
    | Field -> `Int 5
    | Variable -> `Int 6
    | Class -> `Int 7
    | Interface -> `Int 8
    | Module -> `Int 9
    | Property -> `Int 10
    | Unit -> `Int 11
    | Value -> `Int 12
    | Enum -> `Int 13
    | Keyword -> `Int 14
    | Snippet -> `Int 15
    | Color -> `Int 16
    | File -> `Int 17
    | Reference -> `Int 18
    | Folder -> `Int 19
    | EnumMember -> `Int 20
    | Constant -> `Int 21
    | Struct -> `Int 22
    | Event -> `Int 23
    | Operator -> `Int 24
    | TypeParameter -> `Int 25

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Text
    | `Int 2 -> Method
    | `Int 3 -> Function
    | `Int 4 -> Constructor
    | `Int 5 -> Field
    | `Int 6 -> Variable
    | `Int 7 -> Class
    | `Int 8 -> Interface
    | `Int 9 -> Module
    | `Int 10 -> Property
    | `Int 11 -> Unit
    | `Int 12 -> Value
    | `Int 13 -> Enum
    | `Int 14 -> Keyword
    | `Int 15 -> Snippet
    | `Int 16 -> Color
    | `Int 17 -> File
    | `Int 18 -> Reference
    | `Int 19 -> Folder
    | `Int 20 -> EnumMember
    | `Int 21 -> Constant
    | `Int 22 -> Struct
    | `Int 23 -> Event
    | `Int 24 -> Operator
    | `Int 25 -> TypeParameter
    | _ ->
      Json.error
        "Invalid value. Expected one of: 1, 2, 3,\n\
         4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, \
         23, 24,\n\
         25"
        json
end

module CompletionItemTag = struct
  type t = Deprecated

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Deprecated -> `Int 1

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Deprecated
    | _ -> Json.error "Invalid value. Expected one of:\n1" json
end

module CodeActionKind = struct
  type t =
    | Empty
    | QuickFix
    | Refactor
    | RefactorExtract
    | RefactorInline
    | RefactorRewrite
    | Source
    | SourceOrganizeImports
    | SourceFixAll
    | Other of string

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Empty -> `String ""
    | QuickFix -> `String "quickfix"
    | Refactor -> `String "refactor"
    | RefactorExtract -> `String "refactor.extract"
    | RefactorInline -> `String "refactor.inline"
    | RefactorRewrite -> `String "refactor.rewrite"
    | Source -> `String "source"
    | SourceOrganizeImports -> `String "source.organizeImports"
    | SourceFixAll -> `String "source.fixAll"
    | Other s -> `String s

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "" -> Empty
    | `String "quickfix" -> QuickFix
    | `String "refactor" -> Refactor
    | `String "refactor.extract" -> RefactorExtract
    | `String "refactor.inline" -> RefactorInline
    | `String "refactor.rewrite" -> RefactorRewrite
    | `String "source" -> Source
    | `String "source.organizeImports" -> SourceOrganizeImports
    | `String "source.fixAll" -> SourceFixAll
    | `String s -> Other s
    | _ ->
      Json.error
        "Invalid value. Expected one of: \"\",\n\
         \"quickfix\", \"refactor\", \"refactor.extract\", \"refactor.inline\",\n\
         \"refactor.rewrite\", \"source\", \"source.organizeImports\",\n\
         \"source.fixAll\""
        json
end

module PositionEncodingKind = struct
  type t = UTF8 | UTF16 | UTF32 | Other of string

  let yojson_of_t (t : t) : Json.t =
    match t with
    | UTF8 -> `String "utf-8"
    | UTF16 -> `String "utf-16"
    | UTF32 -> `String "utf-32"
    | Other s -> `String s

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "utf-8" -> UTF8
    | `String "utf-16" -> UTF16
    | `String "utf-32" -> UTF32
    | `String s -> Other s
    | _ ->
      Json.error
        "Invalid\nvalue. Expected one of: \"utf-8\", \"utf-16\", \"utf-32\""
        json
end

module DiagnosticSeverity = struct
  type t = Error | Warning | Information | Hint

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Error -> `Int 1
    | Warning -> `Int 2
    | Information -> `Int 3
    | Hint -> `Int 4

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Error
    | `Int 2 -> Warning
    | `Int 3 -> Information
    | `Int 4 -> Hint
    | _ -> Json.error "Invalid value. Expected one of: 1, 2, 3, 4" json
end

module CodeActionTriggerKind = struct
  type t = Invoked | Automatic

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Invoked -> `Int 1
    | Automatic -> `Int 2

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Invoked
    | `Int 2 -> Automatic
    | _ -> Json.error "Invalid\nvalue. Expected one of: 1, 2" json
end

module CompletionTriggerKind = struct
  type t = Invoked | TriggerCharacter | TriggerForIncompleteCompletions

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Invoked -> `Int 1
    | TriggerCharacter -> `Int 2
    | TriggerForIncompleteCompletions -> `Int 3

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Invoked
    | `Int 2 -> TriggerCharacter
    | `Int 3 -> TriggerForIncompleteCompletions
    | _ -> Json.error "Invalid value. Expected\none of: 1, 2, 3" json
end

module InsertTextFormat = struct
  type t = PlainText | Snippet

  let yojson_of_t (t : t) : Json.t =
    match t with
    | PlainText -> `Int 1
    | Snippet -> `Int 2

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> PlainText
    | `Int 2 -> Snippet
    | _ -> Json.error "Invalid\nvalue. Expected one of: 1, 2" json
end

module NotebookCellKind = struct
  type t = Markup | Code

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Markup -> `Int 1
    | Code -> `Int 2

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Markup
    | `Int 2 -> Code
    | _ -> Json.error "Invalid value.\nExpected one of: 1, 2" json
end

module FileChangeType = struct
  type t = Created | Changed | Deleted

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Created -> `Int 1
    | Changed -> `Int 2
    | Deleted -> `Int 3

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Created
    | `Int 2 -> Changed
    | `Int 3 -> Deleted
    | _ -> Json.error "Invalid value. Expected one of: 1, 2, 3" json
end

module WatchKind = struct
  type t = Create | Change | Delete | Other of string

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Create -> `Int 1
    | Change -> `Int 2
    | Delete -> `Int 4
    | Other s -> `String s

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Create
    | `Int 2 -> Change
    | `Int 4 -> Delete
    | `String s -> Other s
    | _ -> Json.error "Invalid value. Expected one of: 1, 2,\n4" json
end

module DocumentDiagnosticReportKind = struct
  type t = Full | Unchanged

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Full -> `String "full"
    | Unchanged -> `String "unchanged"

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "full" -> Full
    | `String "unchanged" -> Unchanged
    | _ ->
      Json.error "Invalid value. Expected one of: \"full\", \"unchanged\"" json
end

module DocumentHighlightKind = struct
  type t = Text | Read | Write

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Text -> `Int 1
    | Read -> `Int 2
    | Write -> `Int 3

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Text
    | `Int 2 -> Read
    | `Int 3 -> Write
    | _ -> Json.error "Invalid value. Expected one of: 1, 2, 3" json
end

module FileOperationPatternKind = struct
  type t = File | Folder

  let yojson_of_t (t : t) : Json.t =
    match t with
    | File -> `String "file"
    | Folder -> `String "folder"

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "file" -> File
    | `String "folder" -> Folder
    | _ ->
      Json.error "Invalid value. Expected one of: \"file\", \"folder\"" json
end

module TraceValues = struct
  type t = Compact | Off | Messages | Verbose

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Compact -> `String "compact"
    | Off -> `String "off"
    | Messages -> `String "messages"
    | Verbose -> `String "verbose"

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "compact" -> Compact
    | `String "off" -> Off
    | `String "messages" -> Messages
    | `String "verbose" -> Verbose
    | _ ->
      Json.error
        "Invalid value. Expected one of: \"compact\", \"off\",\n\
         \"messages\", \"verbose\""
        json
end

module TextDocumentSyncKind = struct
  type t = None | Full | Incremental

  let yojson_of_t (t : t) : Json.t =
    match t with
    | None -> `Int 0
    | Full -> `Int 1
    | Incremental -> `Int 2

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 0 -> None
    | `Int 1 -> Full
    | `Int 2 -> Incremental
    | _ -> Json.error "Invalid value. Expected one of: 0, 1, 2" json
end

module InlayHintKind = struct
  type t = Type | Parameter

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Type -> `Int 1
    | Parameter -> `Int 2

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Type
    | `Int 2 -> Parameter
    | _ -> Json.error "Invalid\nvalue. Expected one of: 1, 2" json
end

module InlineCompletionTriggerKind = struct
  type t = Invoked | Automatic

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Invoked -> `Int 0
    | Automatic -> `Int 1

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 0 -> Invoked
    | `Int 1 -> Automatic
    | _ -> Json.error "Invalid\nvalue. Expected one of: 0, 1" json
end

module MessageType = struct
  type t = Error | Warning | Info | Log | Debug

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Error -> `Int 1
    | Warning -> `Int 2
    | Info -> `Int 3
    | Log -> `Int 4
    | Debug -> `Int 5

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Error
    | `Int 2 -> Warning
    | `Int 3 -> Info
    | `Int 4 -> Log
    | `Int 5 -> Debug
    | _ -> Json.error "Invalid value. Expected one of:\n1, 2, 3, 4, 5" json
end

module UniquenessLevel = struct
  type t = Document | Project | Group | Scheme | Global

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Document -> `String "document"
    | Project -> `String "project"
    | Group -> `String "group"
    | Scheme -> `String "scheme"
    | Global -> `String "global"

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "document" -> Document
    | `String "project" -> Project
    | `String "group" -> Group
    | `String "scheme" -> Scheme
    | `String "global" -> Global
    | _ ->
      Json.error
        "Invalid value.\n\
         Expected one of: \"document\", \"project\", \"group\", \"scheme\",\n\
         \"global\""
        json
end

module MonikerKind = struct
  type t = Import | Export | Local

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Import -> `String "import"
    | Export -> `String "export"
    | Local -> `String "local"

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "import" -> Import
    | `String "export" -> Export
    | `String "local" -> Local
    | _ ->
      Json.error
        "Invalid value. Expected one of:\n\"import\", \"export\", \"local\""
        json
end

module SemanticTokenModifiers = struct
  type t =
    | Declaration
    | Definition
    | Readonly
    | Static
    | Deprecated
    | Abstract
    | Async
    | Modification
    | Documentation
    | DefaultLibrary

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Declaration -> `String "declaration"
    | Definition -> `String "definition"
    | Readonly -> `String "readonly"
    | Static -> `String "static"
    | Deprecated -> `String "deprecated"
    | Abstract -> `String "abstract"
    | Async -> `String "async"
    | Modification -> `String "modification"
    | Documentation -> `String "documentation"
    | DefaultLibrary -> `String "defaultLibrary"

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "declaration" -> Declaration
    | `String "definition" -> Definition
    | `String "readonly" -> Readonly
    | `String "static" -> Static
    | `String "deprecated" -> Deprecated
    | `String "abstract" -> Abstract
    | `String "async" -> Async
    | `String "modification" -> Modification
    | `String "documentation" -> Documentation
    | `String "defaultLibrary" -> DefaultLibrary
    | _ ->
      Json.error
        "Invalid value.\n\
         Expected one of: \"declaration\", \"definition\", \"readonly\", \
         \"static\",\n\
         \"deprecated\", \"abstract\", \"async\", \"modification\", \
         \"documentation\",\n\
         \"defaultLibrary\""
        json
end

module SemanticTokenTypes = struct
  type t =
    | Namespace
    | Type
    | Class
    | Enum
    | Interface
    | Struct
    | TypeParameter
    | Parameter
    | Variable
    | Property
    | EnumMember
    | Event
    | Function
    | Method
    | Macro
    | Keyword
    | Modifier
    | Comment
    | String
    | Number
    | Regexp
    | Operator
    | Decorator

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Namespace -> `String "namespace"
    | Type -> `String "type"
    | Class -> `String "class"
    | Enum -> `String "enum"
    | Interface -> `String "interface"
    | Struct -> `String "struct"
    | TypeParameter -> `String "typeParameter"
    | Parameter -> `String "parameter"
    | Variable -> `String "variable"
    | Property -> `String "property"
    | EnumMember -> `String "enumMember"
    | Event -> `String "event"
    | Function -> `String "function"
    | Method -> `String "method"
    | Macro -> `String "macro"
    | Keyword -> `String "keyword"
    | Modifier -> `String "modifier"
    | Comment -> `String "comment"
    | String -> `String "string"
    | Number -> `String "number"
    | Regexp -> `String "regexp"
    | Operator -> `String "operator"
    | Decorator -> `String "decorator"

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `String "namespace" -> Namespace
    | `String "type" -> Type
    | `String "class" -> Class
    | `String "enum" -> Enum
    | `String "interface" -> Interface
    | `String "struct" -> Struct
    | `String "typeParameter" -> TypeParameter
    | `String "parameter" -> Parameter
    | `String "variable" -> Variable
    | `String "property" -> Property
    | `String "enumMember" -> EnumMember
    | `String "event" -> Event
    | `String "function" -> Function
    | `String "method" -> Method
    | `String "macro" -> Macro
    | `String "keyword" -> Keyword
    | `String "modifier" -> Modifier
    | `String "comment" -> Comment
    | `String "string" -> String
    | `String "number" -> Number
    | `String "regexp" -> Regexp
    | `String "operator" -> Operator
    | `String "decorator" -> Decorator
    | _ ->
      Json.error
        "Invalid value. Expected one of: \"namespace\",\n\
         \"type\", \"class\", \"enum\", \"interface\", \"struct\", \
         \"typeParameter\",\n\
         \"parameter\", \"variable\", \"property\", \"enumMember\", \"event\",\n\
         \"function\", \"method\", \"macro\", \"keyword\", \"modifier\", \
         \"comment\",\n\
         \"string\", \"number\", \"regexp\", \"operator\", \"decorator\""
        json
end

module SignatureHelpTriggerKind = struct
  type t = Invoked | TriggerCharacter | ContentChange

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Invoked -> `Int 1
    | TriggerCharacter -> `Int 2
    | ContentChange -> `Int 3

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Invoked
    | `Int 2 -> TriggerCharacter
    | `Int 3 -> ContentChange
    | _ -> Json.error "Invalid value. Expected one of: 1, 2, 3" json
end

module TextDocumentSaveReason = struct
  type t = Manual | AfterDelay | FocusOut

  let yojson_of_t (t : t) : Json.t =
    match t with
    | Manual -> `Int 1
    | AfterDelay -> `Int 2
    | FocusOut -> `Int 3

  let t_of_yojson (json : Json.t) : t =
    match json with
    | `Int 1 -> Manual
    | `Int 2 -> AfterDelay
    | `Int 3 -> FocusOut
    | _ -> Json.error "Invalid value. Expected one of: 1, 2, 3" json
end

module Position = struct
  type t = { character : int; line : int }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.Position.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let character_field = ref Option.None
       and line_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "character" -> (
             match Ppx_yojson_conv_lib.( ! ) character_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               character_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "line" -> (
             match Ppx_yojson_conv_lib.( ! ) line_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               line_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) character_field,
               Ppx_yojson_conv_lib.( ! ) line_field )
           with
           | Option.Some character_value, Option.Some line_value ->
             { character = character_value; line = line_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) character_field)
                     Option.None,
                   "character" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) line_field)
                     Option.None,
                   "line" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { character = v_character; line = v_line } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_int v_line in
         ("line", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_int v_character in
         ("character", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(character : int) ~(line : int) : t = { character; line }
end

module Range = struct
  type t = { end_ : Position.t; [@key "end"] start : Position.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.Range.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let end__field = ref Option.None
       and start_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "end" -> (
             match Ppx_yojson_conv_lib.( ! ) end__field with
             | Option.None ->
               let fvalue = Position.t_of_yojson _field_yojson in
               end__field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "start" -> (
             match Ppx_yojson_conv_lib.( ! ) start_field with
             | Option.None ->
               let fvalue = Position.t_of_yojson _field_yojson in
               start_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) end__field,
               Ppx_yojson_conv_lib.( ! ) start_field )
           with
           | Option.Some end__value, Option.Some start_value ->
             { end_ = end__value; start = start_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) end__field)
                     Option.None,
                   "end_" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) start_field)
                     Option.None,
                   "start" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { end_ = v_end_; start = v_start } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = Position.yojson_of_t v_start in
         ("start", arg) :: bnds
       in
       let bnds =
         let arg = Position.yojson_of_t v_end_ in
         ("end", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(end_ : Position.t) ~(start : Position.t) : t = { end_; start }
end

module ChangeAnnotationIdentifier = struct
  type t = string [@@deriving_inline yojson]

  let _ = fun (_ : t) -> ()
  let t_of_yojson = (string_of_yojson : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)
  let _ = t_of_yojson
  let yojson_of_t = (yojson_of_string : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)
  let _ = yojson_of_t

  [@@@end]
end

module AnnotatedTextEdit = struct
  type t =
    { annotationId : ChangeAnnotationIdentifier.t;
      newText : string;
      range : Range.t
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.AnnotatedTextEdit.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let annotationId_field = ref Option.None
       and newText_field = ref Option.None
       and range_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "annotationId" -> (
             match Ppx_yojson_conv_lib.( ! ) annotationId_field with
             | Option.None ->
               let fvalue =
                 ChangeAnnotationIdentifier.t_of_yojson _field_yojson
               in
               annotationId_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "newText" -> (
             match Ppx_yojson_conv_lib.( ! ) newText_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               newText_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "range" -> (
             match Ppx_yojson_conv_lib.( ! ) range_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               range_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) annotationId_field,
               Ppx_yojson_conv_lib.( ! ) newText_field,
               Ppx_yojson_conv_lib.( ! ) range_field )
           with
           | ( Option.Some annotationId_value,
               Option.Some newText_value,
               Option.Some range_value ) ->
             { annotationId = annotationId_value;
               newText = newText_value;
               range = range_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) annotationId_field)
                     Option.None,
                   "annotationId" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) newText_field)
                     Option.None,
                   "newText" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) range_field)
                     Option.None,
                   "range" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { annotationId = v_annotationId; newText = v_newText; range = v_range }
       ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = Range.yojson_of_t v_range in
         ("range", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_newText in
         ("newText", arg) :: bnds
       in
       let bnds =
         let arg = ChangeAnnotationIdentifier.yojson_of_t v_annotationId in
         ("annotationId", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(annotationId : ChangeAnnotationIdentifier.t) ~(newText : string)
      ~(range : Range.t) : t =
    { annotationId; newText; range }
end

module DeleteFileOptions = struct
  type t =
    { ignoreIfNotExists : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      recursive : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DeleteFileOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let ignoreIfNotExists_field = ref Option.None
       and recursive_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "ignoreIfNotExists" -> (
             match Ppx_yojson_conv_lib.( ! ) ignoreIfNotExists_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               ignoreIfNotExists_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "recursive" -> (
             match Ppx_yojson_conv_lib.( ! ) recursive_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               recursive_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ignoreIfNotExists_value, recursive_value =
             ( Ppx_yojson_conv_lib.( ! ) ignoreIfNotExists_field,
               Ppx_yojson_conv_lib.( ! ) recursive_field )
           in
           { ignoreIfNotExists =
               (match ignoreIfNotExists_value with
               | Option.None -> None
               | Option.Some v -> v);
             recursive =
               (match recursive_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { ignoreIfNotExists = v_ignoreIfNotExists; recursive = v_recursive } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_recursive then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_recursive
           in
           let bnd = ("recursive", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_ignoreIfNotExists then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_ignoreIfNotExists
           in
           let bnd = ("ignoreIfNotExists", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(ignoreIfNotExists : bool option) ?(recursive : bool option)
      (() : unit) : t =
    { ignoreIfNotExists; recursive }
end

module DeleteFile = struct
  type t =
    { annotationId : ChangeAnnotationIdentifier.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      options : DeleteFileOptions.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      uri : DocumentUri.t
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DeleteFile.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let annotationId_field = ref Option.None
       and options_field = ref Option.None
       and uri_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "annotationId" -> (
             match Ppx_yojson_conv_lib.( ! ) annotationId_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   ChangeAnnotationIdentifier.t_of_yojson _field_yojson
               in
               annotationId_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "options" -> (
             match Ppx_yojson_conv_lib.( ! ) options_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson DeleteFileOptions.t_of_yojson
                   _field_yojson
               in
               options_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) annotationId_field,
               Ppx_yojson_conv_lib.( ! ) options_field,
               Ppx_yojson_conv_lib.( ! ) uri_field )
           with
           | annotationId_value, options_value, Option.Some uri_value ->
             { annotationId =
                 (match annotationId_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               options =
                 (match options_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               uri = uri_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { annotationId = v_annotationId; options = v_options; uri = v_uri } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_uri in
         ("uri", arg) :: bnds
       in
       let bnds =
         if None = v_options then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t DeleteFileOptions.yojson_of_t)
               v_options
           in
           let bnd = ("options", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_annotationId then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                ChangeAnnotationIdentifier.yojson_of_t)
               v_annotationId
           in
           let bnd = ("annotationId", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(annotationId : ChangeAnnotationIdentifier.t option)
      ?(options : DeleteFileOptions.t option) ~(uri : DocumentUri.t) (() : unit)
      : t =
    { annotationId; options; uri }

  let yojson_of_t (t : t) : Json.t =
    Json.To.literal_field "kind" "delete" yojson_of_t t

  let t_of_yojson (json : Json.t) : t =
    Json.Of.literal_field "t" "kind" "delete" t_of_yojson json
end

module RenameFileOptions = struct
  type t =
    { ignoreIfExists : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      overwrite : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.RenameFileOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let ignoreIfExists_field = ref Option.None
       and overwrite_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "ignoreIfExists" -> (
             match Ppx_yojson_conv_lib.( ! ) ignoreIfExists_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               ignoreIfExists_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "overwrite" -> (
             match Ppx_yojson_conv_lib.( ! ) overwrite_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               overwrite_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ignoreIfExists_value, overwrite_value =
             ( Ppx_yojson_conv_lib.( ! ) ignoreIfExists_field,
               Ppx_yojson_conv_lib.( ! ) overwrite_field )
           in
           { ignoreIfExists =
               (match ignoreIfExists_value with
               | Option.None -> None
               | Option.Some v -> v);
             overwrite =
               (match overwrite_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { ignoreIfExists = v_ignoreIfExists; overwrite = v_overwrite } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_overwrite then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_overwrite
           in
           let bnd = ("overwrite", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_ignoreIfExists then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_ignoreIfExists
           in
           let bnd = ("ignoreIfExists", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(ignoreIfExists : bool option) ?(overwrite : bool option)
      (() : unit) : t =
    { ignoreIfExists; overwrite }
end

module RenameFile = struct
  type t =
    { annotationId : ChangeAnnotationIdentifier.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      newUri : DocumentUri.t;
      oldUri : DocumentUri.t;
      options : RenameFileOptions.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.RenameFile.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let annotationId_field = ref Option.None
       and newUri_field = ref Option.None
       and oldUri_field = ref Option.None
       and options_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "annotationId" -> (
             match Ppx_yojson_conv_lib.( ! ) annotationId_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   ChangeAnnotationIdentifier.t_of_yojson _field_yojson
               in
               annotationId_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "newUri" -> (
             match Ppx_yojson_conv_lib.( ! ) newUri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               newUri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "oldUri" -> (
             match Ppx_yojson_conv_lib.( ! ) oldUri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               oldUri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "options" -> (
             match Ppx_yojson_conv_lib.( ! ) options_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson RenameFileOptions.t_of_yojson
                   _field_yojson
               in
               options_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) annotationId_field,
               Ppx_yojson_conv_lib.( ! ) newUri_field,
               Ppx_yojson_conv_lib.( ! ) oldUri_field,
               Ppx_yojson_conv_lib.( ! ) options_field )
           with
           | ( annotationId_value,
               Option.Some newUri_value,
               Option.Some oldUri_value,
               options_value ) ->
             { annotationId =
                 (match annotationId_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               newUri = newUri_value;
               oldUri = oldUri_value;
               options =
                 (match options_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) newUri_field)
                     Option.None,
                   "newUri" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) oldUri_field)
                     Option.None,
                   "oldUri" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { annotationId = v_annotationId;
         newUri = v_newUri;
         oldUri = v_oldUri;
         options = v_options
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_options then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t RenameFileOptions.yojson_of_t)
               v_options
           in
           let bnd = ("options", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_oldUri in
         ("oldUri", arg) :: bnds
       in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_newUri in
         ("newUri", arg) :: bnds
       in
       let bnds =
         if None = v_annotationId then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                ChangeAnnotationIdentifier.yojson_of_t)
               v_annotationId
           in
           let bnd = ("annotationId", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(annotationId : ChangeAnnotationIdentifier.t option)
      ~(newUri : DocumentUri.t) ~(oldUri : DocumentUri.t)
      ?(options : RenameFileOptions.t option) (() : unit) : t =
    { annotationId; newUri; oldUri; options }

  let yojson_of_t (t : t) : Json.t =
    Json.To.literal_field "kind" "rename" yojson_of_t t

  let t_of_yojson (json : Json.t) : t =
    Json.Of.literal_field "t" "kind" "rename" t_of_yojson json
end

module CreateFileOptions = struct
  type t =
    { ignoreIfExists : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      overwrite : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CreateFileOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let ignoreIfExists_field = ref Option.None
       and overwrite_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "ignoreIfExists" -> (
             match Ppx_yojson_conv_lib.( ! ) ignoreIfExists_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               ignoreIfExists_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "overwrite" -> (
             match Ppx_yojson_conv_lib.( ! ) overwrite_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               overwrite_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ignoreIfExists_value, overwrite_value =
             ( Ppx_yojson_conv_lib.( ! ) ignoreIfExists_field,
               Ppx_yojson_conv_lib.( ! ) overwrite_field )
           in
           { ignoreIfExists =
               (match ignoreIfExists_value with
               | Option.None -> None
               | Option.Some v -> v);
             overwrite =
               (match overwrite_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { ignoreIfExists = v_ignoreIfExists; overwrite = v_overwrite } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_overwrite then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_overwrite
           in
           let bnd = ("overwrite", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_ignoreIfExists then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_ignoreIfExists
           in
           let bnd = ("ignoreIfExists", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(ignoreIfExists : bool option) ?(overwrite : bool option)
      (() : unit) : t =
    { ignoreIfExists; overwrite }
end

module CreateFile = struct
  type t =
    { annotationId : ChangeAnnotationIdentifier.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      options : CreateFileOptions.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      uri : DocumentUri.t
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CreateFile.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let annotationId_field = ref Option.None
       and options_field = ref Option.None
       and uri_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "annotationId" -> (
             match Ppx_yojson_conv_lib.( ! ) annotationId_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   ChangeAnnotationIdentifier.t_of_yojson _field_yojson
               in
               annotationId_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "options" -> (
             match Ppx_yojson_conv_lib.( ! ) options_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson CreateFileOptions.t_of_yojson
                   _field_yojson
               in
               options_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) annotationId_field,
               Ppx_yojson_conv_lib.( ! ) options_field,
               Ppx_yojson_conv_lib.( ! ) uri_field )
           with
           | annotationId_value, options_value, Option.Some uri_value ->
             { annotationId =
                 (match annotationId_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               options =
                 (match options_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               uri = uri_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { annotationId = v_annotationId; options = v_options; uri = v_uri } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_uri in
         ("uri", arg) :: bnds
       in
       let bnds =
         if None = v_options then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t CreateFileOptions.yojson_of_t)
               v_options
           in
           let bnd = ("options", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_annotationId then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                ChangeAnnotationIdentifier.yojson_of_t)
               v_annotationId
           in
           let bnd = ("annotationId", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(annotationId : ChangeAnnotationIdentifier.t option)
      ?(options : CreateFileOptions.t option) ~(uri : DocumentUri.t) (() : unit)
      : t =
    { annotationId; options; uri }

  let yojson_of_t (t : t) : Json.t =
    Json.To.literal_field "kind" "create" yojson_of_t t

  let t_of_yojson (json : Json.t) : t =
    Json.Of.literal_field "t" "kind" "create" t_of_yojson json
end

module OptionalVersionedTextDocumentIdentifier = struct
  type t =
    { uri : DocumentUri.t;
      version : int Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.OptionalVersionedTextDocumentIdentifier.t"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let uri_field = ref Option.None
       and version_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "version" -> (
             match Ppx_yojson_conv_lib.( ! ) version_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson int_of_yojson _field_yojson
               in
               version_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) uri_field,
               Ppx_yojson_conv_lib.( ! ) version_field )
           with
           | Option.Some uri_value, version_value ->
             { uri = uri_value;
               version =
                 (match version_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { uri = v_uri; version = v_version } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_version then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_int) v_version
           in
           let bnd = ("version", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_uri in
         ("uri", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(uri : DocumentUri.t) ?(version : int option) (() : unit) : t =
    { uri; version }
end

module TextEdit = struct
  type t = { newText : string; range : Range.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.TextEdit.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let newText_field = ref Option.None
       and range_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "newText" -> (
             match Ppx_yojson_conv_lib.( ! ) newText_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               newText_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "range" -> (
             match Ppx_yojson_conv_lib.( ! ) range_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               range_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) newText_field,
               Ppx_yojson_conv_lib.( ! ) range_field )
           with
           | Option.Some newText_value, Option.Some range_value ->
             { newText = newText_value; range = range_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) newText_field)
                     Option.None,
                   "newText" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) range_field)
                     Option.None,
                   "range" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { newText = v_newText; range = v_range } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = Range.yojson_of_t v_range in
         ("range", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_newText in
         ("newText", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(newText : string) ~(range : Range.t) : t = { newText; range }
end

module TextDocumentEdit = struct
  type edits_pvar =
    [ `TextEdit of TextEdit.t | `AnnotatedTextEdit of AnnotatedTextEdit.t ]

  let edits_pvar_of_yojson (json : Json.t) : edits_pvar =
    Json.Of.untagged_union "edits_pvar"
      [ (fun json -> `TextEdit (TextEdit.t_of_yojson json));
        (fun json -> `AnnotatedTextEdit (AnnotatedTextEdit.t_of_yojson json))
      ]
      json

  let yojson_of_edits_pvar (edits_pvar : edits_pvar) : Json.t =
    match edits_pvar with
    | `TextEdit s -> TextEdit.yojson_of_t s
    | `AnnotatedTextEdit s -> AnnotatedTextEdit.yojson_of_t s

  type t =
    { edits : edits_pvar list;
      textDocument : OptionalVersionedTextDocumentIdentifier.t
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.TextDocumentEdit.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let edits_field = ref Option.None
       and textDocument_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "edits" -> (
             match Ppx_yojson_conv_lib.( ! ) edits_field with
             | Option.None ->
               let fvalue = list_of_yojson edits_pvar_of_yojson _field_yojson in
               edits_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue =
                 OptionalVersionedTextDocumentIdentifier.t_of_yojson
                   _field_yojson
               in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) edits_field,
               Ppx_yojson_conv_lib.( ! ) textDocument_field )
           with
           | Option.Some edits_value, Option.Some textDocument_value ->
             { edits = edits_value; textDocument = textDocument_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) edits_field)
                     Option.None,
                   "edits" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { edits = v_edits; textDocument = v_textDocument } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg =
           OptionalVersionedTextDocumentIdentifier.yojson_of_t v_textDocument
         in
         ("textDocument", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_list yojson_of_edits_pvar v_edits in
         ("edits", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(edits : edits_pvar list)
      ~(textDocument : OptionalVersionedTextDocumentIdentifier.t) : t =
    { edits; textDocument }
end

module ChangeAnnotation = struct
  type t =
    { description : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      label : string;
      needsConfirmation : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ChangeAnnotation.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let description_field = ref Option.None
       and label_field = ref Option.None
       and needsConfirmation_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "description" -> (
             match Ppx_yojson_conv_lib.( ! ) description_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               description_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "label" -> (
             match Ppx_yojson_conv_lib.( ! ) label_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               label_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "needsConfirmation" -> (
             match Ppx_yojson_conv_lib.( ! ) needsConfirmation_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               needsConfirmation_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) description_field,
               Ppx_yojson_conv_lib.( ! ) label_field,
               Ppx_yojson_conv_lib.( ! ) needsConfirmation_field )
           with
           | description_value, Option.Some label_value, needsConfirmation_value
             ->
             { description =
                 (match description_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               label = label_value;
               needsConfirmation =
                 (match needsConfirmation_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) label_field)
                     Option.None,
                   "label" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { description = v_description;
         label = v_label;
         needsConfirmation = v_needsConfirmation
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_needsConfirmation then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_needsConfirmation
           in
           let bnd = ("needsConfirmation", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_label in
         ("label", arg) :: bnds
       in
       let bnds =
         if None = v_description then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_description
           in
           let bnd = ("description", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(description : string option) ~(label : string)
      ?(needsConfirmation : bool option) (() : unit) : t =
    { description; label; needsConfirmation }
end

module WorkspaceEdit = struct
  type documentChanges_pvar =
    [ `TextDocumentEdit of TextDocumentEdit.t
    | `CreateFile of CreateFile.t
    | `RenameFile of RenameFile.t
    | `DeleteFile of DeleteFile.t ]

  let documentChanges_pvar_of_yojson (json : Json.t) : documentChanges_pvar =
    Json.Of.untagged_union "documentChanges_pvar"
      [ (fun json -> `TextDocumentEdit (TextDocumentEdit.t_of_yojson json));
        (fun json -> `CreateFile (CreateFile.t_of_yojson json));
        (fun json -> `RenameFile (RenameFile.t_of_yojson json));
        (fun json -> `DeleteFile (DeleteFile.t_of_yojson json))
      ]
      json

  let yojson_of_documentChanges_pvar
      (documentChanges_pvar : documentChanges_pvar) : Json.t =
    match documentChanges_pvar with
    | `TextDocumentEdit s -> TextDocumentEdit.yojson_of_t s
    | `CreateFile s -> CreateFile.yojson_of_t s
    | `RenameFile s -> RenameFile.yojson_of_t s
    | `DeleteFile s -> DeleteFile.yojson_of_t s

  type t =
    { changeAnnotations :
        (ChangeAnnotationIdentifier.t, ChangeAnnotation.t) Json.Assoc.t
        Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      changes :
        (DocumentUri.t, TextEdit.t list) Json.Assoc.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      documentChanges : documentChanges_pvar list Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.WorkspaceEdit.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let changeAnnotations_field = ref Option.None
       and changes_field = ref Option.None
       and documentChanges_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "changeAnnotations" -> (
             match Ppx_yojson_conv_lib.( ! ) changeAnnotations_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (Json.Assoc.t_of_yojson
                      ChangeAnnotationIdentifier.t_of_yojson
                      ChangeAnnotation.t_of_yojson)
                   _field_yojson
               in
               changeAnnotations_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "changes" -> (
             match Ppx_yojson_conv_lib.( ! ) changes_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (Json.Assoc.t_of_yojson DocumentUri.t_of_yojson
                      (list_of_yojson TextEdit.t_of_yojson))
                   _field_yojson
               in
               changes_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "documentChanges" -> (
             match Ppx_yojson_conv_lib.( ! ) documentChanges_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson documentChanges_pvar_of_yojson)
                   _field_yojson
               in
               documentChanges_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let changeAnnotations_value, changes_value, documentChanges_value =
             ( Ppx_yojson_conv_lib.( ! ) changeAnnotations_field,
               Ppx_yojson_conv_lib.( ! ) changes_field,
               Ppx_yojson_conv_lib.( ! ) documentChanges_field )
           in
           { changeAnnotations =
               (match changeAnnotations_value with
               | Option.None -> None
               | Option.Some v -> v);
             changes =
               (match changes_value with
               | Option.None -> None
               | Option.Some v -> v);
             documentChanges =
               (match documentChanges_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { changeAnnotations = v_changeAnnotations;
         changes = v_changes;
         documentChanges = v_documentChanges
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_documentChanges then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list yojson_of_documentChanges_pvar))
               v_documentChanges
           in
           let bnd = ("documentChanges", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_changes then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (Json.Assoc.yojson_of_t DocumentUri.yojson_of_t
                   (yojson_of_list TextEdit.yojson_of_t)))
               v_changes
           in
           let bnd = ("changes", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_changeAnnotations then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (Json.Assoc.yojson_of_t ChangeAnnotationIdentifier.yojson_of_t
                   ChangeAnnotation.yojson_of_t))
               v_changeAnnotations
           in
           let bnd = ("changeAnnotations", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create
      ?(changeAnnotations :
         (ChangeAnnotationIdentifier.t, ChangeAnnotation.t) Json.Assoc.t option)
      ?(changes : (DocumentUri.t, TextEdit.t list) Json.Assoc.t option)
      ?(documentChanges : documentChanges_pvar list option) (() : unit) : t =
    { changeAnnotations; changes; documentChanges }
end

module ApplyWorkspaceEditParams = struct
  type t =
    { edit : WorkspaceEdit.t;
      label : string Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ApplyWorkspaceEditParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let edit_field = ref Option.None
       and label_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "edit" -> (
             match Ppx_yojson_conv_lib.( ! ) edit_field with
             | Option.None ->
               let fvalue = WorkspaceEdit.t_of_yojson _field_yojson in
               edit_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "label" -> (
             match Ppx_yojson_conv_lib.( ! ) label_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               label_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) edit_field,
               Ppx_yojson_conv_lib.( ! ) label_field )
           with
           | Option.Some edit_value, label_value ->
             { edit = edit_value;
               label =
                 (match label_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) edit_field)
                     Option.None,
                   "edit" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { edit = v_edit; label = v_label } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_label then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_label
           in
           let bnd = ("label", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = WorkspaceEdit.yojson_of_t v_edit in
         ("edit", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(edit : WorkspaceEdit.t) ?(label : string option) (() : unit) : t
      =
    { edit; label }
end

module ApplyWorkspaceEditResult = struct
  type t =
    { applied : bool;
      failedChange : int Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      failureReason : string Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ApplyWorkspaceEditResult.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let applied_field = ref Option.None
       and failedChange_field = ref Option.None
       and failureReason_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "applied" -> (
             match Ppx_yojson_conv_lib.( ! ) applied_field with
             | Option.None ->
               let fvalue = bool_of_yojson _field_yojson in
               applied_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "failedChange" -> (
             match Ppx_yojson_conv_lib.( ! ) failedChange_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson int_of_yojson _field_yojson
               in
               failedChange_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "failureReason" -> (
             match Ppx_yojson_conv_lib.( ! ) failureReason_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               failureReason_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) applied_field,
               Ppx_yojson_conv_lib.( ! ) failedChange_field,
               Ppx_yojson_conv_lib.( ! ) failureReason_field )
           with
           | Option.Some applied_value, failedChange_value, failureReason_value
             ->
             { applied = applied_value;
               failedChange =
                 (match failedChange_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               failureReason =
                 (match failureReason_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) applied_field)
                     Option.None,
                   "applied" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { applied = v_applied;
         failedChange = v_failedChange;
         failureReason = v_failureReason
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_failureReason then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_failureReason
           in
           let bnd = ("failureReason", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_failedChange then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_int) v_failedChange
           in
           let bnd = ("failedChange", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_bool v_applied in
         ("applied", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(applied : bool) ?(failedChange : int option)
      ?(failureReason : string option) (() : unit) : t =
    { applied; failedChange; failureReason }
end

module BaseSymbolInformation = struct
  type t =
    { containerName : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      kind : SymbolKind.t;
      name : string;
      tags : SymbolTag.t list Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.BaseSymbolInformation.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let containerName_field = ref Option.None
       and kind_field = ref Option.None
       and name_field = ref Option.None
       and tags_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "containerName" -> (
             match Ppx_yojson_conv_lib.( ! ) containerName_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               containerName_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "kind" -> (
             match Ppx_yojson_conv_lib.( ! ) kind_field with
             | Option.None ->
               let fvalue = SymbolKind.t_of_yojson _field_yojson in
               kind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "name" -> (
             match Ppx_yojson_conv_lib.( ! ) name_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               name_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "tags" -> (
             match Ppx_yojson_conv_lib.( ! ) tags_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson SymbolTag.t_of_yojson)
                   _field_yojson
               in
               tags_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) containerName_field,
               Ppx_yojson_conv_lib.( ! ) kind_field,
               Ppx_yojson_conv_lib.( ! ) name_field,
               Ppx_yojson_conv_lib.( ! ) tags_field )
           with
           | ( containerName_value,
               Option.Some kind_value,
               Option.Some name_value,
               tags_value ) ->
             { containerName =
                 (match containerName_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               kind = kind_value;
               name = name_value;
               tags =
                 (match tags_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) kind_field)
                     Option.None,
                   "kind" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) name_field)
                     Option.None,
                   "name" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { containerName = v_containerName;
         kind = v_kind;
         name = v_name;
         tags = v_tags
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_tags then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list SymbolTag.yojson_of_t))
               v_tags
           in
           let bnd = ("tags", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_name in
         ("name", arg) :: bnds
       in
       let bnds =
         let arg = SymbolKind.yojson_of_t v_kind in
         ("kind", arg) :: bnds
       in
       let bnds =
         if None = v_containerName then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_containerName
           in
           let bnd = ("containerName", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(containerName : string option) ~(kind : SymbolKind.t)
      ~(name : string) ?(tags : SymbolTag.t list option) (() : unit) : t =
    { containerName; kind; name; tags }
end

module CallHierarchyClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CallHierarchyClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module CallHierarchyItem = struct
  type t =
    { data : Json.t option; [@yojson.option]
      detail : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      kind : SymbolKind.t;
      name : string;
      range : Range.t;
      selectionRange : Range.t;
      tags : SymbolTag.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      uri : DocumentUri.t
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CallHierarchyItem.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let data_field = ref Option.None
       and detail_field = ref Option.None
       and kind_field = ref Option.None
       and name_field = ref Option.None
       and range_field = ref Option.None
       and selectionRange_field = ref Option.None
       and tags_field = ref Option.None
       and uri_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "data" -> (
             match Ppx_yojson_conv_lib.( ! ) data_field with
             | Option.None ->
               let fvalue = Json.t_of_yojson _field_yojson in
               data_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "detail" -> (
             match Ppx_yojson_conv_lib.( ! ) detail_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               detail_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "kind" -> (
             match Ppx_yojson_conv_lib.( ! ) kind_field with
             | Option.None ->
               let fvalue = SymbolKind.t_of_yojson _field_yojson in
               kind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "name" -> (
             match Ppx_yojson_conv_lib.( ! ) name_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               name_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "range" -> (
             match Ppx_yojson_conv_lib.( ! ) range_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               range_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "selectionRange" -> (
             match Ppx_yojson_conv_lib.( ! ) selectionRange_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               selectionRange_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "tags" -> (
             match Ppx_yojson_conv_lib.( ! ) tags_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson SymbolTag.t_of_yojson)
                   _field_yojson
               in
               tags_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) data_field,
               Ppx_yojson_conv_lib.( ! ) detail_field,
               Ppx_yojson_conv_lib.( ! ) kind_field,
               Ppx_yojson_conv_lib.( ! ) name_field,
               Ppx_yojson_conv_lib.( ! ) range_field,
               Ppx_yojson_conv_lib.( ! ) selectionRange_field,
               Ppx_yojson_conv_lib.( ! ) tags_field,
               Ppx_yojson_conv_lib.( ! ) uri_field )
           with
           | ( data_value,
               detail_value,
               Option.Some kind_value,
               Option.Some name_value,
               Option.Some range_value,
               Option.Some selectionRange_value,
               tags_value,
               Option.Some uri_value ) ->
             { data = data_value;
               detail =
                 (match detail_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               kind = kind_value;
               name = name_value;
               range = range_value;
               selectionRange = selectionRange_value;
               tags =
                 (match tags_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               uri = uri_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) kind_field)
                     Option.None,
                   "kind" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) name_field)
                     Option.None,
                   "name" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) range_field)
                     Option.None,
                   "range" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) selectionRange_field)
                     Option.None,
                   "selectionRange" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { data = v_data;
         detail = v_detail;
         kind = v_kind;
         name = v_name;
         range = v_range;
         selectionRange = v_selectionRange;
         tags = v_tags;
         uri = v_uri
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_uri in
         ("uri", arg) :: bnds
       in
       let bnds =
         if None = v_tags then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list SymbolTag.yojson_of_t))
               v_tags
           in
           let bnd = ("tags", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = Range.yojson_of_t v_selectionRange in
         ("selectionRange", arg) :: bnds
       in
       let bnds =
         let arg = Range.yojson_of_t v_range in
         ("range", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_name in
         ("name", arg) :: bnds
       in
       let bnds =
         let arg = SymbolKind.yojson_of_t v_kind in
         ("kind", arg) :: bnds
       in
       let bnds =
         if None = v_detail then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_detail
           in
           let bnd = ("detail", arg) in
           bnd :: bnds
       in
       let bnds =
         match v_data with
         | Option.None -> bnds
         | Option.Some v ->
           let arg = Json.yojson_of_t v in
           let bnd = ("data", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(data : Json.t option) ?(detail : string option)
      ~(kind : SymbolKind.t) ~(name : string) ~(range : Range.t)
      ~(selectionRange : Range.t) ?(tags : SymbolTag.t list option)
      ~(uri : DocumentUri.t) (() : unit) : t =
    { data; detail; kind; name; range; selectionRange; tags; uri }
end

module CallHierarchyIncomingCall = struct
  type t = { from : CallHierarchyItem.t; fromRanges : Range.t list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CallHierarchyIncomingCall.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let from_field = ref Option.None
       and fromRanges_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "from" -> (
             match Ppx_yojson_conv_lib.( ! ) from_field with
             | Option.None ->
               let fvalue = CallHierarchyItem.t_of_yojson _field_yojson in
               from_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "fromRanges" -> (
             match Ppx_yojson_conv_lib.( ! ) fromRanges_field with
             | Option.None ->
               let fvalue = list_of_yojson Range.t_of_yojson _field_yojson in
               fromRanges_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) from_field,
               Ppx_yojson_conv_lib.( ! ) fromRanges_field )
           with
           | Option.Some from_value, Option.Some fromRanges_value ->
             { from = from_value; fromRanges = fromRanges_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) from_field)
                     Option.None,
                   "from" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) fromRanges_field)
                     Option.None,
                   "fromRanges" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { from = v_from; fromRanges = v_fromRanges } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list Range.yojson_of_t v_fromRanges in
         ("fromRanges", arg) :: bnds
       in
       let bnds =
         let arg = CallHierarchyItem.yojson_of_t v_from in
         ("from", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(from : CallHierarchyItem.t) ~(fromRanges : Range.t list) : t =
    { from; fromRanges }
end

module CallHierarchyIncomingCallsParams = struct
  type t =
    { item : CallHierarchyItem.t;
      partialResultToken : ProgressToken.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workDoneToken : ProgressToken.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CallHierarchyIncomingCallsParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let item_field = ref Option.None
       and partialResultToken_field = ref Option.None
       and workDoneToken_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "item" -> (
             match Ppx_yojson_conv_lib.( ! ) item_field with
             | Option.None ->
               let fvalue = CallHierarchyItem.t_of_yojson _field_yojson in
               item_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "partialResultToken" -> (
             match Ppx_yojson_conv_lib.( ! ) partialResultToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               partialResultToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneToken" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               workDoneToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) item_field,
               Ppx_yojson_conv_lib.( ! ) partialResultToken_field,
               Ppx_yojson_conv_lib.( ! ) workDoneToken_field )
           with
           | ( Option.Some item_value,
               partialResultToken_value,
               workDoneToken_value ) ->
             { item = item_value;
               partialResultToken =
                 (match partialResultToken_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               workDoneToken =
                 (match workDoneToken_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) item_field)
                     Option.None,
                   "item" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { item = v_item;
         partialResultToken = v_partialResultToken;
         workDoneToken = v_workDoneToken
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_workDoneToken
           in
           let bnd = ("workDoneToken", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_partialResultToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_partialResultToken
           in
           let bnd = ("partialResultToken", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = CallHierarchyItem.yojson_of_t v_item in
         ("item", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(item : CallHierarchyItem.t)
      ?(partialResultToken : ProgressToken.t option)
      ?(workDoneToken : ProgressToken.t option) (() : unit) : t =
    { item; partialResultToken; workDoneToken }
end

module CallHierarchyOptions = struct
  type t =
    { workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CallHierarchyOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let workDoneProgress_value =
             Ppx_yojson_conv_lib.( ! ) workDoneProgress_field
           in
           { workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { workDoneProgress = v_workDoneProgress } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(workDoneProgress : bool option) (() : unit) : t =
    { workDoneProgress }
end

module CallHierarchyOutgoingCall = struct
  type t = { fromRanges : Range.t list; to_ : CallHierarchyItem.t [@key "to"] }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CallHierarchyOutgoingCall.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let fromRanges_field = ref Option.None
       and to__field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "fromRanges" -> (
             match Ppx_yojson_conv_lib.( ! ) fromRanges_field with
             | Option.None ->
               let fvalue = list_of_yojson Range.t_of_yojson _field_yojson in
               fromRanges_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "to" -> (
             match Ppx_yojson_conv_lib.( ! ) to__field with
             | Option.None ->
               let fvalue = CallHierarchyItem.t_of_yojson _field_yojson in
               to__field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) fromRanges_field,
               Ppx_yojson_conv_lib.( ! ) to__field )
           with
           | Option.Some fromRanges_value, Option.Some to__value ->
             { fromRanges = fromRanges_value; to_ = to__value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) fromRanges_field)
                     Option.None,
                   "fromRanges" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) to__field)
                     Option.None,
                   "to_" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { fromRanges = v_fromRanges; to_ = v_to_ } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = CallHierarchyItem.yojson_of_t v_to_ in
         ("to", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_list Range.yojson_of_t v_fromRanges in
         ("fromRanges", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(fromRanges : Range.t list) ~(to_ : CallHierarchyItem.t) : t =
    { fromRanges; to_ }
end

module CallHierarchyOutgoingCallsParams = struct
  type t =
    { item : CallHierarchyItem.t;
      partialResultToken : ProgressToken.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workDoneToken : ProgressToken.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CallHierarchyOutgoingCallsParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let item_field = ref Option.None
       and partialResultToken_field = ref Option.None
       and workDoneToken_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "item" -> (
             match Ppx_yojson_conv_lib.( ! ) item_field with
             | Option.None ->
               let fvalue = CallHierarchyItem.t_of_yojson _field_yojson in
               item_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "partialResultToken" -> (
             match Ppx_yojson_conv_lib.( ! ) partialResultToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               partialResultToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneToken" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               workDoneToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) item_field,
               Ppx_yojson_conv_lib.( ! ) partialResultToken_field,
               Ppx_yojson_conv_lib.( ! ) workDoneToken_field )
           with
           | ( Option.Some item_value,
               partialResultToken_value,
               workDoneToken_value ) ->
             { item = item_value;
               partialResultToken =
                 (match partialResultToken_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               workDoneToken =
                 (match workDoneToken_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) item_field)
                     Option.None,
                   "item" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { item = v_item;
         partialResultToken = v_partialResultToken;
         workDoneToken = v_workDoneToken
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_workDoneToken
           in
           let bnd = ("workDoneToken", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_partialResultToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_partialResultToken
           in
           let bnd = ("partialResultToken", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = CallHierarchyItem.yojson_of_t v_item in
         ("item", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(item : CallHierarchyItem.t)
      ?(partialResultToken : ProgressToken.t option)
      ?(workDoneToken : ProgressToken.t option) (() : unit) : t =
    { item; partialResultToken; workDoneToken }
end

module TextDocumentIdentifier = struct
  type t = { uri : DocumentUri.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.TextDocumentIdentifier.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let uri_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) uri_field with
           | Option.Some uri_value -> { uri = uri_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { uri = v_uri } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_uri in
         ("uri", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(uri : DocumentUri.t) : t = { uri }
end

module CallHierarchyPrepareParams = struct
  type t =
    { position : Position.t;
      textDocument : TextDocumentIdentifier.t;
      workDoneToken : ProgressToken.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CallHierarchyPrepareParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let position_field = ref Option.None
       and textDocument_field = ref Option.None
       and workDoneToken_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "position" -> (
             match Ppx_yojson_conv_lib.( ! ) position_field with
             | Option.None ->
               let fvalue = Position.t_of_yojson _field_yojson in
               position_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue = TextDocumentIdentifier.t_of_yojson _field_yojson in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneToken" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               workDoneToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) position_field,
               Ppx_yojson_conv_lib.( ! ) textDocument_field,
               Ppx_yojson_conv_lib.( ! ) workDoneToken_field )
           with
           | ( Option.Some position_value,
               Option.Some textDocument_value,
               workDoneToken_value ) ->
             { position = position_value;
               textDocument = textDocument_value;
               workDoneToken =
                 (match workDoneToken_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) position_field)
                     Option.None,
                   "position" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { position = v_position;
         textDocument = v_textDocument;
         workDoneToken = v_workDoneToken
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_workDoneToken
           in
           let bnd = ("workDoneToken", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = TextDocumentIdentifier.yojson_of_t v_textDocument in
         ("textDocument", arg) :: bnds
       in
       let bnds =
         let arg = Position.yojson_of_t v_position in
         ("position", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(position : Position.t) ~(textDocument : TextDocumentIdentifier.t)
      ?(workDoneToken : ProgressToken.t option) (() : unit) : t =
    { position; textDocument; workDoneToken }
end

module NotebookCellTextDocumentFilter = struct
  type notebook_pvar =
    [ `String of string | `NotebookDocumentFilter of NotebookDocumentFilter.t ]

  let notebook_pvar_of_yojson (json : Json.t) : notebook_pvar =
    match json with
    | `String j -> `String j
    | _ ->
      Json.Of.untagged_union "notebook_pvar"
        [ (fun json ->
            `NotebookDocumentFilter (NotebookDocumentFilter.t_of_yojson json))
        ]
        json

  let yojson_of_notebook_pvar (notebook_pvar : notebook_pvar) : Json.t =
    match notebook_pvar with
    | `String j -> `String j
    | `NotebookDocumentFilter s -> NotebookDocumentFilter.yojson_of_t s

  type t =
    { language : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      notebook : notebook_pvar
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.NotebookCellTextDocumentFilter.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let language_field = ref Option.None
       and notebook_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "language" -> (
             match Ppx_yojson_conv_lib.( ! ) language_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               language_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "notebook" -> (
             match Ppx_yojson_conv_lib.( ! ) notebook_field with
             | Option.None ->
               let fvalue = notebook_pvar_of_yojson _field_yojson in
               notebook_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) language_field,
               Ppx_yojson_conv_lib.( ! ) notebook_field )
           with
           | language_value, Option.Some notebook_value ->
             { language =
                 (match language_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               notebook = notebook_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) notebook_field)
                     Option.None,
                   "notebook" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { language = v_language; notebook = v_notebook } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_notebook_pvar v_notebook in
         ("notebook", arg) :: bnds
       in
       let bnds =
         if None = v_language then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_language
           in
           let bnd = ("language", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(language : string option) ~(notebook : notebook_pvar) (() : unit)
      : t =
    { language; notebook }
end

module DocumentFilter = struct
  type t =
    [ `TextDocumentFilter of TextDocumentFilter.t
    | `NotebookCellTextDocumentFilter of NotebookCellTextDocumentFilter.t ]

  let t_of_yojson (json : Json.t) : t =
    Json.Of.untagged_union "t"
      [ (fun json -> `TextDocumentFilter (TextDocumentFilter.t_of_yojson json));
        (fun json ->
          `NotebookCellTextDocumentFilter
            (NotebookCellTextDocumentFilter.t_of_yojson json))
      ]
      json

  let yojson_of_t (t : t) : Json.t =
    match t with
    | `TextDocumentFilter s -> TextDocumentFilter.yojson_of_t s
    | `NotebookCellTextDocumentFilter s ->
      NotebookCellTextDocumentFilter.yojson_of_t s
end

module DocumentSelector = struct
  type t = DocumentFilter.t list [@@deriving_inline yojson]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DocumentSelector.t" in
     fun t -> list_of_yojson DocumentFilter.t_of_yojson t
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (fun v -> yojson_of_list DocumentFilter.yojson_of_t v
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]
end

module CallHierarchyRegistrationOptions = struct
  type t =
    { documentSelector : DocumentSelector.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      id : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CallHierarchyRegistrationOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let documentSelector_field = ref Option.None
       and id_field = ref Option.None
       and workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "documentSelector" -> (
             match Ppx_yojson_conv_lib.( ! ) documentSelector_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson DocumentSelector.t_of_yojson
                   _field_yojson
               in
               documentSelector_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "id" -> (
             match Ppx_yojson_conv_lib.( ! ) id_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               id_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let documentSelector_value, id_value, workDoneProgress_value =
             ( Ppx_yojson_conv_lib.( ! ) documentSelector_field,
               Ppx_yojson_conv_lib.( ! ) id_field,
               Ppx_yojson_conv_lib.( ! ) workDoneProgress_field )
           in
           { documentSelector =
               (match documentSelector_value with
               | Option.None -> None
               | Option.Some v -> v);
             id =
               (match id_value with
               | Option.None -> None
               | Option.Some v -> v);
             workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { documentSelector = v_documentSelector;
         id = v_id;
         workDoneProgress = v_workDoneProgress
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_id then bnds
         else
           let arg = (Json.Nullable_option.yojson_of_t yojson_of_string) v_id in
           let bnd = ("id", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentSelector then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t DocumentSelector.yojson_of_t)
               v_documentSelector
           in
           let bnd = ("documentSelector", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(documentSelector : DocumentSelector.t option)
      ?(id : string option) ?(workDoneProgress : bool option) (() : unit) : t =
    { documentSelector; id; workDoneProgress }
end

module CancelParams = struct
  type t = { id : Jsonrpc.Id.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CancelParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let id_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "id" -> (
             match Ppx_yojson_conv_lib.( ! ) id_field with
             | Option.None ->
               let fvalue = Jsonrpc.Id.t_of_yojson _field_yojson in
               id_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) id_field with
           | Option.Some id_value -> { id = id_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) id_field)
                     Option.None,
                   "id" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { id = v_id } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = Jsonrpc.Id.yojson_of_t v_id in
         ("id", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(id : Jsonrpc.Id.t) : t = { id }
end

module WorkspaceEditClientCapabilities = struct
  type changeAnnotationSupport =
    { groupsOnLabel : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : changeAnnotationSupport) -> ()

  let changeAnnotationSupport_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.WorkspaceEditClientCapabilities.changeAnnotationSupport"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let groupsOnLabel_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "groupsOnLabel" -> (
             match Ppx_yojson_conv_lib.( ! ) groupsOnLabel_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               groupsOnLabel_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let groupsOnLabel_value =
             Ppx_yojson_conv_lib.( ! ) groupsOnLabel_field
           in
           { groupsOnLabel =
               (match groupsOnLabel_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> changeAnnotationSupport)

  let _ = changeAnnotationSupport_of_yojson

  let yojson_of_changeAnnotationSupport =
    (function
     | { groupsOnLabel = v_groupsOnLabel } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_groupsOnLabel then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_groupsOnLabel
           in
           let bnd = ("groupsOnLabel", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : changeAnnotationSupport -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_changeAnnotationSupport

  [@@@end]

  let create_changeAnnotationSupport ?(groupsOnLabel : bool option) (() : unit)
      : changeAnnotationSupport =
    { groupsOnLabel }

  type t =
    { changeAnnotationSupport : changeAnnotationSupport Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      documentChanges : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      failureHandling : FailureHandlingKind.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      normalizesLineEndings : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      resourceOperations : ResourceOperationKind.t list Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.WorkspaceEditClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let changeAnnotationSupport_field = ref Option.None
       and documentChanges_field = ref Option.None
       and failureHandling_field = ref Option.None
       and normalizesLineEndings_field = ref Option.None
       and resourceOperations_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "changeAnnotationSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) changeAnnotationSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   changeAnnotationSupport_of_yojson _field_yojson
               in
               changeAnnotationSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "documentChanges" -> (
             match Ppx_yojson_conv_lib.( ! ) documentChanges_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               documentChanges_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "failureHandling" -> (
             match Ppx_yojson_conv_lib.( ! ) failureHandling_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   FailureHandlingKind.t_of_yojson _field_yojson
               in
               failureHandling_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "normalizesLineEndings" -> (
             match Ppx_yojson_conv_lib.( ! ) normalizesLineEndings_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               normalizesLineEndings_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "resourceOperations" -> (
             match Ppx_yojson_conv_lib.( ! ) resourceOperations_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson ResourceOperationKind.t_of_yojson)
                   _field_yojson
               in
               resourceOperations_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( changeAnnotationSupport_value,
                 documentChanges_value,
                 failureHandling_value,
                 normalizesLineEndings_value,
                 resourceOperations_value ) =
             ( Ppx_yojson_conv_lib.( ! ) changeAnnotationSupport_field,
               Ppx_yojson_conv_lib.( ! ) documentChanges_field,
               Ppx_yojson_conv_lib.( ! ) failureHandling_field,
               Ppx_yojson_conv_lib.( ! ) normalizesLineEndings_field,
               Ppx_yojson_conv_lib.( ! ) resourceOperations_field )
           in
           { changeAnnotationSupport =
               (match changeAnnotationSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             documentChanges =
               (match documentChanges_value with
               | Option.None -> None
               | Option.Some v -> v);
             failureHandling =
               (match failureHandling_value with
               | Option.None -> None
               | Option.Some v -> v);
             normalizesLineEndings =
               (match normalizesLineEndings_value with
               | Option.None -> None
               | Option.Some v -> v);
             resourceOperations =
               (match resourceOperations_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { changeAnnotationSupport = v_changeAnnotationSupport;
         documentChanges = v_documentChanges;
         failureHandling = v_failureHandling;
         normalizesLineEndings = v_normalizesLineEndings;
         resourceOperations = v_resourceOperations
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_resourceOperations then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list ResourceOperationKind.yojson_of_t))
               v_resourceOperations
           in
           let bnd = ("resourceOperations", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_normalizesLineEndings then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_normalizesLineEndings
           in
           let bnd = ("normalizesLineEndings", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_failureHandling then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t FailureHandlingKind.yojson_of_t)
               v_failureHandling
           in
           let bnd = ("failureHandling", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentChanges then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_documentChanges
           in
           let bnd = ("documentChanges", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_changeAnnotationSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_changeAnnotationSupport)
               v_changeAnnotationSupport
           in
           let bnd = ("changeAnnotationSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(changeAnnotationSupport : changeAnnotationSupport option)
      ?(documentChanges : bool option)
      ?(failureHandling : FailureHandlingKind.t option)
      ?(normalizesLineEndings : bool option)
      ?(resourceOperations : ResourceOperationKind.t list option) (() : unit) :
      t =
    { changeAnnotationSupport;
      documentChanges;
      failureHandling;
      normalizesLineEndings;
      resourceOperations
    }
end

module WorkspaceSymbolClientCapabilities = struct
  type tagSupport = { valueSet : SymbolTag.t list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : tagSupport) -> ()

  let tagSupport_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.WorkspaceSymbolClientCapabilities.tagSupport"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let valueSet_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "valueSet" -> (
             match Ppx_yojson_conv_lib.( ! ) valueSet_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson SymbolTag.t_of_yojson _field_yojson
               in
               valueSet_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) valueSet_field with
           | Option.Some valueSet_value -> { valueSet = valueSet_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) valueSet_field)
                     Option.None,
                   "valueSet" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> tagSupport)

  let _ = tagSupport_of_yojson

  let yojson_of_tagSupport =
    (function
     | { valueSet = v_valueSet } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list SymbolTag.yojson_of_t v_valueSet in
         ("valueSet", arg) :: bnds
       in
       `Assoc bnds
      : tagSupport -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_tagSupport

  [@@@end]

  let create_tagSupport ~(valueSet : SymbolTag.t list) : tagSupport =
    { valueSet }

  type symbolKind =
    { valueSet : SymbolKind.t list Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : symbolKind) -> ()

  let symbolKind_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.WorkspaceSymbolClientCapabilities.symbolKind"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let valueSet_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "valueSet" -> (
             match Ppx_yojson_conv_lib.( ! ) valueSet_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson SymbolKind.t_of_yojson)
                   _field_yojson
               in
               valueSet_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let valueSet_value = Ppx_yojson_conv_lib.( ! ) valueSet_field in
           { valueSet =
               (match valueSet_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> symbolKind)

  let _ = symbolKind_of_yojson

  let yojson_of_symbolKind =
    (function
     | { valueSet = v_valueSet } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_valueSet then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list SymbolKind.yojson_of_t))
               v_valueSet
           in
           let bnd = ("valueSet", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : symbolKind -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_symbolKind

  [@@@end]

  let create_symbolKind ?(valueSet : SymbolKind.t list option) (() : unit) :
      symbolKind =
    { valueSet }

  type resolveSupport = { properties : string list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : resolveSupport) -> ()

  let resolveSupport_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.WorkspaceSymbolClientCapabilities.resolveSupport"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let properties_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "properties" -> (
             match Ppx_yojson_conv_lib.( ! ) properties_field with
             | Option.None ->
               let fvalue = list_of_yojson string_of_yojson _field_yojson in
               properties_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) properties_field with
           | Option.Some properties_value -> { properties = properties_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) properties_field)
                     Option.None,
                   "properties" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> resolveSupport)

  let _ = resolveSupport_of_yojson

  let yojson_of_resolveSupport =
    (function
     | { properties = v_properties } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list yojson_of_string v_properties in
         ("properties", arg) :: bnds
       in
       `Assoc bnds
      : resolveSupport -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_resolveSupport

  [@@@end]

  let create_resolveSupport ~(properties : string list) : resolveSupport =
    { properties }

  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      resolveSupport : resolveSupport Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      symbolKind : symbolKind Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      tagSupport : tagSupport Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.WorkspaceSymbolClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and resolveSupport_field = ref Option.None
       and symbolKind_field = ref Option.None
       and tagSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "resolveSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) resolveSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson resolveSupport_of_yojson
                   _field_yojson
               in
               resolveSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "symbolKind" -> (
             match Ppx_yojson_conv_lib.( ! ) symbolKind_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson symbolKind_of_yojson
                   _field_yojson
               in
               symbolKind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "tagSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) tagSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson tagSupport_of_yojson
                   _field_yojson
               in
               tagSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( dynamicRegistration_value,
                 resolveSupport_value,
                 symbolKind_value,
                 tagSupport_value ) =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) resolveSupport_field,
               Ppx_yojson_conv_lib.( ! ) symbolKind_field,
               Ppx_yojson_conv_lib.( ! ) tagSupport_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             resolveSupport =
               (match resolveSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             symbolKind =
               (match symbolKind_value with
               | Option.None -> None
               | Option.Some v -> v);
             tagSupport =
               (match tagSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         resolveSupport = v_resolveSupport;
         symbolKind = v_symbolKind;
         tagSupport = v_tagSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_tagSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_tagSupport)
               v_tagSupport
           in
           let bnd = ("tagSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_symbolKind then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_symbolKind)
               v_symbolKind
           in
           let bnd = ("symbolKind", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_resolveSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_resolveSupport)
               v_resolveSupport
           in
           let bnd = ("resolveSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option)
      ?(resolveSupport : resolveSupport option)
      ?(symbolKind : symbolKind option) ?(tagSupport : tagSupport option)
      (() : unit) : t =
    { dynamicRegistration; resolveSupport; symbolKind; tagSupport }
end

module SemanticTokensWorkspaceClientCapabilities = struct
  type t =
    { refreshSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.SemanticTokensWorkspaceClientCapabilities.t"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let refreshSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "refreshSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) refreshSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               refreshSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let refreshSupport_value =
             Ppx_yojson_conv_lib.( ! ) refreshSupport_field
           in
           { refreshSupport =
               (match refreshSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { refreshSupport = v_refreshSupport } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_refreshSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_refreshSupport
           in
           let bnd = ("refreshSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(refreshSupport : bool option) (() : unit) : t =
    { refreshSupport }
end

module InlineValueWorkspaceClientCapabilities = struct
  type t =
    { refreshSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.InlineValueWorkspaceClientCapabilities.t"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let refreshSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "refreshSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) refreshSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               refreshSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let refreshSupport_value =
             Ppx_yojson_conv_lib.( ! ) refreshSupport_field
           in
           { refreshSupport =
               (match refreshSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { refreshSupport = v_refreshSupport } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_refreshSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_refreshSupport
           in
           let bnd = ("refreshSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(refreshSupport : bool option) (() : unit) : t =
    { refreshSupport }
end

module InlayHintWorkspaceClientCapabilities = struct
  type t =
    { refreshSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.InlayHintWorkspaceClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let refreshSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "refreshSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) refreshSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               refreshSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let refreshSupport_value =
             Ppx_yojson_conv_lib.( ! ) refreshSupport_field
           in
           { refreshSupport =
               (match refreshSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { refreshSupport = v_refreshSupport } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_refreshSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_refreshSupport
           in
           let bnd = ("refreshSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(refreshSupport : bool option) (() : unit) : t =
    { refreshSupport }
end

module FoldingRangeWorkspaceClientCapabilities = struct
  type t =
    { refreshSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.FoldingRangeWorkspaceClientCapabilities.t"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let refreshSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "refreshSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) refreshSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               refreshSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let refreshSupport_value =
             Ppx_yojson_conv_lib.( ! ) refreshSupport_field
           in
           { refreshSupport =
               (match refreshSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { refreshSupport = v_refreshSupport } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_refreshSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_refreshSupport
           in
           let bnd = ("refreshSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(refreshSupport : bool option) (() : unit) : t =
    { refreshSupport }
end

module FileOperationClientCapabilities = struct
  type t =
    { didCreate : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      didDelete : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      didRename : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      willCreate : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      willDelete : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      willRename : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.FileOperationClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let didCreate_field = ref Option.None
       and didDelete_field = ref Option.None
       and didRename_field = ref Option.None
       and dynamicRegistration_field = ref Option.None
       and willCreate_field = ref Option.None
       and willDelete_field = ref Option.None
       and willRename_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "didCreate" -> (
             match Ppx_yojson_conv_lib.( ! ) didCreate_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               didCreate_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "didDelete" -> (
             match Ppx_yojson_conv_lib.( ! ) didDelete_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               didDelete_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "didRename" -> (
             match Ppx_yojson_conv_lib.( ! ) didRename_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               didRename_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "willCreate" -> (
             match Ppx_yojson_conv_lib.( ! ) willCreate_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               willCreate_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "willDelete" -> (
             match Ppx_yojson_conv_lib.( ! ) willDelete_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               willDelete_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "willRename" -> (
             match Ppx_yojson_conv_lib.( ! ) willRename_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               willRename_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( didCreate_value,
                 didDelete_value,
                 didRename_value,
                 dynamicRegistration_value,
                 willCreate_value,
                 willDelete_value,
                 willRename_value ) =
             ( Ppx_yojson_conv_lib.( ! ) didCreate_field,
               Ppx_yojson_conv_lib.( ! ) didDelete_field,
               Ppx_yojson_conv_lib.( ! ) didRename_field,
               Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) willCreate_field,
               Ppx_yojson_conv_lib.( ! ) willDelete_field,
               Ppx_yojson_conv_lib.( ! ) willRename_field )
           in
           { didCreate =
               (match didCreate_value with
               | Option.None -> None
               | Option.Some v -> v);
             didDelete =
               (match didDelete_value with
               | Option.None -> None
               | Option.Some v -> v);
             didRename =
               (match didRename_value with
               | Option.None -> None
               | Option.Some v -> v);
             dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             willCreate =
               (match willCreate_value with
               | Option.None -> None
               | Option.Some v -> v);
             willDelete =
               (match willDelete_value with
               | Option.None -> None
               | Option.Some v -> v);
             willRename =
               (match willRename_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { didCreate = v_didCreate;
         didDelete = v_didDelete;
         didRename = v_didRename;
         dynamicRegistration = v_dynamicRegistration;
         willCreate = v_willCreate;
         willDelete = v_willDelete;
         willRename = v_willRename
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_willRename then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_willRename
           in
           let bnd = ("willRename", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_willDelete then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_willDelete
           in
           let bnd = ("willDelete", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_willCreate then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_willCreate
           in
           let bnd = ("willCreate", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_didRename then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_didRename
           in
           let bnd = ("didRename", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_didDelete then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_didDelete
           in
           let bnd = ("didDelete", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_didCreate then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_didCreate
           in
           let bnd = ("didCreate", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(didCreate : bool option) ?(didDelete : bool option)
      ?(didRename : bool option) ?(dynamicRegistration : bool option)
      ?(willCreate : bool option) ?(willDelete : bool option)
      ?(willRename : bool option) (() : unit) : t =
    { didCreate;
      didDelete;
      didRename;
      dynamicRegistration;
      willCreate;
      willDelete;
      willRename
    }
end

module ExecuteCommandClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ExecuteCommandClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module DidChangeWatchedFilesClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      relativePatternSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.DidChangeWatchedFilesClientCapabilities.t"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and relativePatternSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "relativePatternSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) relativePatternSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               relativePatternSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value, relativePatternSupport_value =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) relativePatternSupport_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             relativePatternSupport =
               (match relativePatternSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         relativePatternSupport = v_relativePatternSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_relativePatternSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_relativePatternSupport
           in
           let bnd = ("relativePatternSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option)
      ?(relativePatternSupport : bool option) (() : unit) : t =
    { dynamicRegistration; relativePatternSupport }
end

module DidChangeConfigurationClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.DidChangeConfigurationClientCapabilities.t"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module DiagnosticWorkspaceClientCapabilities = struct
  type t =
    { refreshSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DiagnosticWorkspaceClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let refreshSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "refreshSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) refreshSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               refreshSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let refreshSupport_value =
             Ppx_yojson_conv_lib.( ! ) refreshSupport_field
           in
           { refreshSupport =
               (match refreshSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { refreshSupport = v_refreshSupport } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_refreshSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_refreshSupport
           in
           let bnd = ("refreshSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(refreshSupport : bool option) (() : unit) : t =
    { refreshSupport }
end

module CodeLensWorkspaceClientCapabilities = struct
  type t =
    { refreshSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeLensWorkspaceClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let refreshSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "refreshSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) refreshSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               refreshSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let refreshSupport_value =
             Ppx_yojson_conv_lib.( ! ) refreshSupport_field
           in
           { refreshSupport =
               (match refreshSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { refreshSupport = v_refreshSupport } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_refreshSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_refreshSupport
           in
           let bnd = ("refreshSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(refreshSupport : bool option) (() : unit) : t =
    { refreshSupport }
end

module WorkspaceClientCapabilities = struct
  type t =
    { applyEdit : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      codeLens : CodeLensWorkspaceClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      configuration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      diagnostics :
        DiagnosticWorkspaceClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      didChangeConfiguration :
        DidChangeConfigurationClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      didChangeWatchedFiles :
        DidChangeWatchedFilesClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      executeCommand :
        ExecuteCommandClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      fileOperations : FileOperationClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      foldingRange :
        FoldingRangeWorkspaceClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      inlayHint : InlayHintWorkspaceClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      inlineValue :
        InlineValueWorkspaceClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      semanticTokens :
        SemanticTokensWorkspaceClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      symbol : WorkspaceSymbolClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workspaceEdit : WorkspaceEditClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workspaceFolders : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.WorkspaceClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let applyEdit_field = ref Option.None
       and codeLens_field = ref Option.None
       and configuration_field = ref Option.None
       and diagnostics_field = ref Option.None
       and didChangeConfiguration_field = ref Option.None
       and didChangeWatchedFiles_field = ref Option.None
       and executeCommand_field = ref Option.None
       and fileOperations_field = ref Option.None
       and foldingRange_field = ref Option.None
       and inlayHint_field = ref Option.None
       and inlineValue_field = ref Option.None
       and semanticTokens_field = ref Option.None
       and symbol_field = ref Option.None
       and workspaceEdit_field = ref Option.None
       and workspaceFolders_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "applyEdit" -> (
             match Ppx_yojson_conv_lib.( ! ) applyEdit_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               applyEdit_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "codeLens" -> (
             match Ppx_yojson_conv_lib.( ! ) codeLens_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   CodeLensWorkspaceClientCapabilities.t_of_yojson _field_yojson
               in
               codeLens_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "configuration" -> (
             match Ppx_yojson_conv_lib.( ! ) configuration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               configuration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "diagnostics" -> (
             match Ppx_yojson_conv_lib.( ! ) diagnostics_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   DiagnosticWorkspaceClientCapabilities.t_of_yojson
                   _field_yojson
               in
               diagnostics_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "didChangeConfiguration" -> (
             match Ppx_yojson_conv_lib.( ! ) didChangeConfiguration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   DidChangeConfigurationClientCapabilities.t_of_yojson
                   _field_yojson
               in
               didChangeConfiguration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "didChangeWatchedFiles" -> (
             match Ppx_yojson_conv_lib.( ! ) didChangeWatchedFiles_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   DidChangeWatchedFilesClientCapabilities.t_of_yojson
                   _field_yojson
               in
               didChangeWatchedFiles_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "executeCommand" -> (
             match Ppx_yojson_conv_lib.( ! ) executeCommand_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   ExecuteCommandClientCapabilities.t_of_yojson _field_yojson
               in
               executeCommand_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "fileOperations" -> (
             match Ppx_yojson_conv_lib.( ! ) fileOperations_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   FileOperationClientCapabilities.t_of_yojson _field_yojson
               in
               fileOperations_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "foldingRange" -> (
             match Ppx_yojson_conv_lib.( ! ) foldingRange_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   FoldingRangeWorkspaceClientCapabilities.t_of_yojson
                   _field_yojson
               in
               foldingRange_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "inlayHint" -> (
             match Ppx_yojson_conv_lib.( ! ) inlayHint_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   InlayHintWorkspaceClientCapabilities.t_of_yojson
                   _field_yojson
               in
               inlayHint_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "inlineValue" -> (
             match Ppx_yojson_conv_lib.( ! ) inlineValue_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   InlineValueWorkspaceClientCapabilities.t_of_yojson
                   _field_yojson
               in
               inlineValue_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "semanticTokens" -> (
             match Ppx_yojson_conv_lib.( ! ) semanticTokens_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   SemanticTokensWorkspaceClientCapabilities.t_of_yojson
                   _field_yojson
               in
               semanticTokens_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "symbol" -> (
             match Ppx_yojson_conv_lib.( ! ) symbol_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   WorkspaceSymbolClientCapabilities.t_of_yojson _field_yojson
               in
               symbol_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workspaceEdit" -> (
             match Ppx_yojson_conv_lib.( ! ) workspaceEdit_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   WorkspaceEditClientCapabilities.t_of_yojson _field_yojson
               in
               workspaceEdit_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workspaceFolders" -> (
             match Ppx_yojson_conv_lib.( ! ) workspaceFolders_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workspaceFolders_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( applyEdit_value,
                 codeLens_value,
                 configuration_value,
                 diagnostics_value,
                 didChangeConfiguration_value,
                 didChangeWatchedFiles_value,
                 executeCommand_value,
                 fileOperations_value,
                 foldingRange_value,
                 inlayHint_value,
                 inlineValue_value,
                 semanticTokens_value,
                 symbol_value,
                 workspaceEdit_value,
                 workspaceFolders_value ) =
             ( Ppx_yojson_conv_lib.( ! ) applyEdit_field,
               Ppx_yojson_conv_lib.( ! ) codeLens_field,
               Ppx_yojson_conv_lib.( ! ) configuration_field,
               Ppx_yojson_conv_lib.( ! ) diagnostics_field,
               Ppx_yojson_conv_lib.( ! ) didChangeConfiguration_field,
               Ppx_yojson_conv_lib.( ! ) didChangeWatchedFiles_field,
               Ppx_yojson_conv_lib.( ! ) executeCommand_field,
               Ppx_yojson_conv_lib.( ! ) fileOperations_field,
               Ppx_yojson_conv_lib.( ! ) foldingRange_field,
               Ppx_yojson_conv_lib.( ! ) inlayHint_field,
               Ppx_yojson_conv_lib.( ! ) inlineValue_field,
               Ppx_yojson_conv_lib.( ! ) semanticTokens_field,
               Ppx_yojson_conv_lib.( ! ) symbol_field,
               Ppx_yojson_conv_lib.( ! ) workspaceEdit_field,
               Ppx_yojson_conv_lib.( ! ) workspaceFolders_field )
           in
           { applyEdit =
               (match applyEdit_value with
               | Option.None -> None
               | Option.Some v -> v);
             codeLens =
               (match codeLens_value with
               | Option.None -> None
               | Option.Some v -> v);
             configuration =
               (match configuration_value with
               | Option.None -> None
               | Option.Some v -> v);
             diagnostics =
               (match diagnostics_value with
               | Option.None -> None
               | Option.Some v -> v);
             didChangeConfiguration =
               (match didChangeConfiguration_value with
               | Option.None -> None
               | Option.Some v -> v);
             didChangeWatchedFiles =
               (match didChangeWatchedFiles_value with
               | Option.None -> None
               | Option.Some v -> v);
             executeCommand =
               (match executeCommand_value with
               | Option.None -> None
               | Option.Some v -> v);
             fileOperations =
               (match fileOperations_value with
               | Option.None -> None
               | Option.Some v -> v);
             foldingRange =
               (match foldingRange_value with
               | Option.None -> None
               | Option.Some v -> v);
             inlayHint =
               (match inlayHint_value with
               | Option.None -> None
               | Option.Some v -> v);
             inlineValue =
               (match inlineValue_value with
               | Option.None -> None
               | Option.Some v -> v);
             semanticTokens =
               (match semanticTokens_value with
               | Option.None -> None
               | Option.Some v -> v);
             symbol =
               (match symbol_value with
               | Option.None -> None
               | Option.Some v -> v);
             workspaceEdit =
               (match workspaceEdit_value with
               | Option.None -> None
               | Option.Some v -> v);
             workspaceFolders =
               (match workspaceFolders_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { applyEdit = v_applyEdit;
         codeLens = v_codeLens;
         configuration = v_configuration;
         diagnostics = v_diagnostics;
         didChangeConfiguration = v_didChangeConfiguration;
         didChangeWatchedFiles = v_didChangeWatchedFiles;
         executeCommand = v_executeCommand;
         fileOperations = v_fileOperations;
         foldingRange = v_foldingRange;
         inlayHint = v_inlayHint;
         inlineValue = v_inlineValue;
         semanticTokens = v_semanticTokens;
         symbol = v_symbol;
         workspaceEdit = v_workspaceEdit;
         workspaceFolders = v_workspaceFolders
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workspaceFolders then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workspaceFolders
           in
           let bnd = ("workspaceFolders", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_workspaceEdit then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                WorkspaceEditClientCapabilities.yojson_of_t)
               v_workspaceEdit
           in
           let bnd = ("workspaceEdit", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_symbol then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                WorkspaceSymbolClientCapabilities.yojson_of_t)
               v_symbol
           in
           let bnd = ("symbol", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_semanticTokens then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                SemanticTokensWorkspaceClientCapabilities.yojson_of_t)
               v_semanticTokens
           in
           let bnd = ("semanticTokens", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_inlineValue then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                InlineValueWorkspaceClientCapabilities.yojson_of_t)
               v_inlineValue
           in
           let bnd = ("inlineValue", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_inlayHint then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                InlayHintWorkspaceClientCapabilities.yojson_of_t)
               v_inlayHint
           in
           let bnd = ("inlayHint", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_foldingRange then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                FoldingRangeWorkspaceClientCapabilities.yojson_of_t)
               v_foldingRange
           in
           let bnd = ("foldingRange", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_fileOperations then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                FileOperationClientCapabilities.yojson_of_t)
               v_fileOperations
           in
           let bnd = ("fileOperations", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_executeCommand then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                ExecuteCommandClientCapabilities.yojson_of_t)
               v_executeCommand
           in
           let bnd = ("executeCommand", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_didChangeWatchedFiles then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                DidChangeWatchedFilesClientCapabilities.yojson_of_t)
               v_didChangeWatchedFiles
           in
           let bnd = ("didChangeWatchedFiles", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_didChangeConfiguration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                DidChangeConfigurationClientCapabilities.yojson_of_t)
               v_didChangeConfiguration
           in
           let bnd = ("didChangeConfiguration", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_diagnostics then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                DiagnosticWorkspaceClientCapabilities.yojson_of_t)
               v_diagnostics
           in
           let bnd = ("diagnostics", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_configuration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_configuration
           in
           let bnd = ("configuration", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_codeLens then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                CodeLensWorkspaceClientCapabilities.yojson_of_t)
               v_codeLens
           in
           let bnd = ("codeLens", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_applyEdit then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_applyEdit
           in
           let bnd = ("applyEdit", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(applyEdit : bool option)
      ?(codeLens : CodeLensWorkspaceClientCapabilities.t option)
      ?(configuration : bool option)
      ?(diagnostics : DiagnosticWorkspaceClientCapabilities.t option)
      ?(didChangeConfiguration :
         DidChangeConfigurationClientCapabilities.t option)
      ?(didChangeWatchedFiles :
         DidChangeWatchedFilesClientCapabilities.t option)
      ?(executeCommand : ExecuteCommandClientCapabilities.t option)
      ?(fileOperations : FileOperationClientCapabilities.t option)
      ?(foldingRange : FoldingRangeWorkspaceClientCapabilities.t option)
      ?(inlayHint : InlayHintWorkspaceClientCapabilities.t option)
      ?(inlineValue : InlineValueWorkspaceClientCapabilities.t option)
      ?(semanticTokens : SemanticTokensWorkspaceClientCapabilities.t option)
      ?(symbol : WorkspaceSymbolClientCapabilities.t option)
      ?(workspaceEdit : WorkspaceEditClientCapabilities.t option)
      ?(workspaceFolders : bool option) (() : unit) : t =
    { applyEdit;
      codeLens;
      configuration;
      diagnostics;
      didChangeConfiguration;
      didChangeWatchedFiles;
      executeCommand;
      fileOperations;
      foldingRange;
      inlayHint;
      inlineValue;
      semanticTokens;
      symbol;
      workspaceEdit;
      workspaceFolders
    }
end

module ShowMessageRequestClientCapabilities = struct
  type messageActionItem =
    { additionalPropertiesSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : messageActionItem) -> ()

  let messageActionItem_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.ShowMessageRequestClientCapabilities.messageActionItem"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let additionalPropertiesSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "additionalPropertiesSupport" -> (
             match
               Ppx_yojson_conv_lib.( ! ) additionalPropertiesSupport_field
             with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               additionalPropertiesSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let additionalPropertiesSupport_value =
             Ppx_yojson_conv_lib.( ! ) additionalPropertiesSupport_field
           in
           { additionalPropertiesSupport =
               (match additionalPropertiesSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> messageActionItem)

  let _ = messageActionItem_of_yojson

  let yojson_of_messageActionItem =
    (function
     | { additionalPropertiesSupport = v_additionalPropertiesSupport } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_additionalPropertiesSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_additionalPropertiesSupport
           in
           let bnd = ("additionalPropertiesSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : messageActionItem -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_messageActionItem

  [@@@end]

  let create_messageActionItem ?(additionalPropertiesSupport : bool option)
      (() : unit) : messageActionItem =
    { additionalPropertiesSupport }

  type t =
    { messageActionItem : messageActionItem Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ShowMessageRequestClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let messageActionItem_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "messageActionItem" -> (
             match Ppx_yojson_conv_lib.( ! ) messageActionItem_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson messageActionItem_of_yojson
                   _field_yojson
               in
               messageActionItem_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let messageActionItem_value =
             Ppx_yojson_conv_lib.( ! ) messageActionItem_field
           in
           { messageActionItem =
               (match messageActionItem_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { messageActionItem = v_messageActionItem } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_messageActionItem then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_messageActionItem)
               v_messageActionItem
           in
           let bnd = ("messageActionItem", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(messageActionItem : messageActionItem option) (() : unit) : t =
    { messageActionItem }
end

module ShowDocumentClientCapabilities = struct
  type t = { support : bool }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ShowDocumentClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let support_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "support" -> (
             match Ppx_yojson_conv_lib.( ! ) support_field with
             | Option.None ->
               let fvalue = bool_of_yojson _field_yojson in
               support_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) support_field with
           | Option.Some support_value -> { support = support_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) support_field)
                     Option.None,
                   "support" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { support = v_support } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_bool v_support in
         ("support", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(support : bool) : t = { support }
end

module WindowClientCapabilities = struct
  type t =
    { showDocument : ShowDocumentClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      showMessage :
        ShowMessageRequestClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.WindowClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let showDocument_field = ref Option.None
       and showMessage_field = ref Option.None
       and workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "showDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) showDocument_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   ShowDocumentClientCapabilities.t_of_yojson _field_yojson
               in
               showDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "showMessage" -> (
             match Ppx_yojson_conv_lib.( ! ) showMessage_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   ShowMessageRequestClientCapabilities.t_of_yojson
                   _field_yojson
               in
               showMessage_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let showDocument_value, showMessage_value, workDoneProgress_value =
             ( Ppx_yojson_conv_lib.( ! ) showDocument_field,
               Ppx_yojson_conv_lib.( ! ) showMessage_field,
               Ppx_yojson_conv_lib.( ! ) workDoneProgress_field )
           in
           { showDocument =
               (match showDocument_value with
               | Option.None -> None
               | Option.Some v -> v);
             showMessage =
               (match showMessage_value with
               | Option.None -> None
               | Option.Some v -> v);
             workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { showDocument = v_showDocument;
         showMessage = v_showMessage;
         workDoneProgress = v_workDoneProgress
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_showMessage then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                ShowMessageRequestClientCapabilities.yojson_of_t)
               v_showMessage
           in
           let bnd = ("showMessage", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_showDocument then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                ShowDocumentClientCapabilities.yojson_of_t)
               v_showDocument
           in
           let bnd = ("showDocument", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(showDocument : ShowDocumentClientCapabilities.t option)
      ?(showMessage : ShowMessageRequestClientCapabilities.t option)
      ?(workDoneProgress : bool option) (() : unit) : t =
    { showDocument; showMessage; workDoneProgress }
end

module TypeHierarchyClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.TypeHierarchyClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module TypeDefinitionClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      linkSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.TypeDefinitionClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and linkSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "linkSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) linkSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               linkSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value, linkSupport_value =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) linkSupport_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             linkSupport =
               (match linkSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         linkSupport = v_linkSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_linkSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_linkSupport
           in
           let bnd = ("linkSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) ?(linkSupport : bool option)
      (() : unit) : t =
    { dynamicRegistration; linkSupport }
end

module TextDocumentSyncClientCapabilities = struct
  type t =
    { didSave : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      willSave : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      willSaveWaitUntil : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.TextDocumentSyncClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let didSave_field = ref Option.None
       and dynamicRegistration_field = ref Option.None
       and willSave_field = ref Option.None
       and willSaveWaitUntil_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "didSave" -> (
             match Ppx_yojson_conv_lib.( ! ) didSave_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               didSave_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "willSave" -> (
             match Ppx_yojson_conv_lib.( ! ) willSave_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               willSave_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "willSaveWaitUntil" -> (
             match Ppx_yojson_conv_lib.( ! ) willSaveWaitUntil_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               willSaveWaitUntil_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( didSave_value,
                 dynamicRegistration_value,
                 willSave_value,
                 willSaveWaitUntil_value ) =
             ( Ppx_yojson_conv_lib.( ! ) didSave_field,
               Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) willSave_field,
               Ppx_yojson_conv_lib.( ! ) willSaveWaitUntil_field )
           in
           { didSave =
               (match didSave_value with
               | Option.None -> None
               | Option.Some v -> v);
             dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             willSave =
               (match willSave_value with
               | Option.None -> None
               | Option.Some v -> v);
             willSaveWaitUntil =
               (match willSaveWaitUntil_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { didSave = v_didSave;
         dynamicRegistration = v_dynamicRegistration;
         willSave = v_willSave;
         willSaveWaitUntil = v_willSaveWaitUntil
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_willSaveWaitUntil then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_willSaveWaitUntil
           in
           let bnd = ("willSaveWaitUntil", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_willSave then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_willSave
           in
           let bnd = ("willSave", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_didSave then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_didSave
           in
           let bnd = ("didSave", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(didSave : bool option) ?(dynamicRegistration : bool option)
      ?(willSave : bool option) ?(willSaveWaitUntil : bool option) (() : unit) :
      t =
    { didSave; dynamicRegistration; willSave; willSaveWaitUntil }
end

module SignatureHelpClientCapabilities = struct
  type parameterInformation =
    { labelOffsetSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : parameterInformation) -> ()

  let parameterInformation_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.SignatureHelpClientCapabilities.parameterInformation"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let labelOffsetSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "labelOffsetSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) labelOffsetSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               labelOffsetSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let labelOffsetSupport_value =
             Ppx_yojson_conv_lib.( ! ) labelOffsetSupport_field
           in
           { labelOffsetSupport =
               (match labelOffsetSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> parameterInformation)

  let _ = parameterInformation_of_yojson

  let yojson_of_parameterInformation =
    (function
     | { labelOffsetSupport = v_labelOffsetSupport } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_labelOffsetSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_labelOffsetSupport
           in
           let bnd = ("labelOffsetSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : parameterInformation -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_parameterInformation

  [@@@end]

  let create_parameterInformation ?(labelOffsetSupport : bool option)
      (() : unit) : parameterInformation =
    { labelOffsetSupport }

  type signatureInformation =
    { documentationFormat : MarkupKind.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      parameterInformation : parameterInformation Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      activeParameterSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      noActiveParameterSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : signatureInformation) -> ()

  let signatureInformation_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.SignatureHelpClientCapabilities.signatureInformation"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let documentationFormat_field = ref Option.None
       and parameterInformation_field = ref Option.None
       and activeParameterSupport_field = ref Option.None
       and noActiveParameterSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "documentationFormat" -> (
             match Ppx_yojson_conv_lib.( ! ) documentationFormat_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson MarkupKind.t_of_yojson)
                   _field_yojson
               in
               documentationFormat_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "parameterInformation" -> (
             match Ppx_yojson_conv_lib.( ! ) parameterInformation_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson parameterInformation_of_yojson
                   _field_yojson
               in
               parameterInformation_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "activeParameterSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) activeParameterSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               activeParameterSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "noActiveParameterSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) noActiveParameterSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               noActiveParameterSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( documentationFormat_value,
                 parameterInformation_value,
                 activeParameterSupport_value,
                 noActiveParameterSupport_value ) =
             ( Ppx_yojson_conv_lib.( ! ) documentationFormat_field,
               Ppx_yojson_conv_lib.( ! ) parameterInformation_field,
               Ppx_yojson_conv_lib.( ! ) activeParameterSupport_field,
               Ppx_yojson_conv_lib.( ! ) noActiveParameterSupport_field )
           in
           { documentationFormat =
               (match documentationFormat_value with
               | Option.None -> None
               | Option.Some v -> v);
             parameterInformation =
               (match parameterInformation_value with
               | Option.None -> None
               | Option.Some v -> v);
             activeParameterSupport =
               (match activeParameterSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             noActiveParameterSupport =
               (match noActiveParameterSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> signatureInformation)

  let _ = signatureInformation_of_yojson

  let yojson_of_signatureInformation =
    (function
     | { documentationFormat = v_documentationFormat;
         parameterInformation = v_parameterInformation;
         activeParameterSupport = v_activeParameterSupport;
         noActiveParameterSupport = v_noActiveParameterSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_noActiveParameterSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_noActiveParameterSupport
           in
           let bnd = ("noActiveParameterSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_activeParameterSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_activeParameterSupport
           in
           let bnd = ("activeParameterSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_parameterInformation then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_parameterInformation)
               v_parameterInformation
           in
           let bnd = ("parameterInformation", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentationFormat then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list MarkupKind.yojson_of_t))
               v_documentationFormat
           in
           let bnd = ("documentationFormat", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : signatureInformation -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_signatureInformation

  [@@@end]

  let create_signatureInformation
      ?(documentationFormat : MarkupKind.t list option)
      ?(parameterInformation : parameterInformation option)
      ?(activeParameterSupport : bool option)
      ?(noActiveParameterSupport : bool option) (() : unit) :
      signatureInformation =
    { documentationFormat;
      parameterInformation;
      activeParameterSupport;
      noActiveParameterSupport
    }

  type t =
    { contextSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      signatureInformation : signatureInformation Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.SignatureHelpClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let contextSupport_field = ref Option.None
       and dynamicRegistration_field = ref Option.None
       and signatureInformation_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "contextSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) contextSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               contextSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "signatureInformation" -> (
             match Ppx_yojson_conv_lib.( ! ) signatureInformation_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson signatureInformation_of_yojson
                   _field_yojson
               in
               signatureInformation_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( contextSupport_value,
                 dynamicRegistration_value,
                 signatureInformation_value ) =
             ( Ppx_yojson_conv_lib.( ! ) contextSupport_field,
               Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) signatureInformation_field )
           in
           { contextSupport =
               (match contextSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             signatureInformation =
               (match signatureInformation_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { contextSupport = v_contextSupport;
         dynamicRegistration = v_dynamicRegistration;
         signatureInformation = v_signatureInformation
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_signatureInformation then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_signatureInformation)
               v_signatureInformation
           in
           let bnd = ("signatureInformation", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_contextSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_contextSupport
           in
           let bnd = ("contextSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(contextSupport : bool option)
      ?(dynamicRegistration : bool option)
      ?(signatureInformation : signatureInformation option) (() : unit) : t =
    { contextSupport; dynamicRegistration; signatureInformation }
end

module SemanticTokensClientCapabilities = struct
  type full =
    { delta : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : full) -> ()

  let full_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.SemanticTokensClientCapabilities.full" in
     function
     | `Assoc field_yojsons as yojson -> (
       let delta_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "delta" -> (
             match Ppx_yojson_conv_lib.( ! ) delta_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               delta_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let delta_value = Ppx_yojson_conv_lib.( ! ) delta_field in
           { delta =
               (match delta_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> full)

  let _ = full_of_yojson

  let yojson_of_full =
    (function
     | { delta = v_delta } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_delta then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_delta
           in
           let bnd = ("delta", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : full -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_full

  [@@@end]

  let create_full ?(delta : bool option) (() : unit) : full = { delta }

  type full_pvar = [ `Bool of bool | `Full of full ]

  let full_pvar_of_yojson (json : Json.t) : full_pvar =
    match json with
    | `Bool j -> `Bool j
    | _ ->
      Json.Of.untagged_union "full_pvar"
        [ (fun json -> `Full (full_of_yojson json)) ]
        json

  let yojson_of_full_pvar (full_pvar : full_pvar) : Json.t =
    match full_pvar with
    | `Bool j -> `Bool j
    | `Full s -> yojson_of_full s

  type requests =
    { range : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      full : full_pvar Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : requests) -> ()

  let requests_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.SemanticTokensClientCapabilities.requests"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let range_field = ref Option.None
       and full_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "range" -> (
             match Ppx_yojson_conv_lib.( ! ) range_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               range_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "full" -> (
             match Ppx_yojson_conv_lib.( ! ) full_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson full_pvar_of_yojson
                   _field_yojson
               in
               full_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let range_value, full_value =
             ( Ppx_yojson_conv_lib.( ! ) range_field,
               Ppx_yojson_conv_lib.( ! ) full_field )
           in
           { range =
               (match range_value with
               | Option.None -> None
               | Option.Some v -> v);
             full =
               (match full_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> requests)

  let _ = requests_of_yojson

  let yojson_of_requests =
    (function
     | { range = v_range; full = v_full } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_full then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_full_pvar) v_full
           in
           let bnd = ("full", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_range then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_range
           in
           let bnd = ("range", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : requests -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_requests

  [@@@end]

  let create_requests ?(range : bool option) ?(full : full_pvar option)
      (() : unit) : requests =
    { range; full }

  type t =
    { augmentsSyntaxTokens : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      formats : TokenFormat.t list;
      multilineTokenSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      overlappingTokenSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      requests : requests;
      serverCancelSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      tokenModifiers : string list;
      tokenTypes : string list
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.SemanticTokensClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let augmentsSyntaxTokens_field = ref Option.None
       and dynamicRegistration_field = ref Option.None
       and formats_field = ref Option.None
       and multilineTokenSupport_field = ref Option.None
       and overlappingTokenSupport_field = ref Option.None
       and requests_field = ref Option.None
       and serverCancelSupport_field = ref Option.None
       and tokenModifiers_field = ref Option.None
       and tokenTypes_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "augmentsSyntaxTokens" -> (
             match Ppx_yojson_conv_lib.( ! ) augmentsSyntaxTokens_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               augmentsSyntaxTokens_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "formats" -> (
             match Ppx_yojson_conv_lib.( ! ) formats_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson TokenFormat.t_of_yojson _field_yojson
               in
               formats_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "multilineTokenSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) multilineTokenSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               multilineTokenSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "overlappingTokenSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) overlappingTokenSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               overlappingTokenSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "requests" -> (
             match Ppx_yojson_conv_lib.( ! ) requests_field with
             | Option.None ->
               let fvalue = requests_of_yojson _field_yojson in
               requests_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "serverCancelSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) serverCancelSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               serverCancelSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "tokenModifiers" -> (
             match Ppx_yojson_conv_lib.( ! ) tokenModifiers_field with
             | Option.None ->
               let fvalue = list_of_yojson string_of_yojson _field_yojson in
               tokenModifiers_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "tokenTypes" -> (
             match Ppx_yojson_conv_lib.( ! ) tokenTypes_field with
             | Option.None ->
               let fvalue = list_of_yojson string_of_yojson _field_yojson in
               tokenTypes_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) augmentsSyntaxTokens_field,
               Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) formats_field,
               Ppx_yojson_conv_lib.( ! ) multilineTokenSupport_field,
               Ppx_yojson_conv_lib.( ! ) overlappingTokenSupport_field,
               Ppx_yojson_conv_lib.( ! ) requests_field,
               Ppx_yojson_conv_lib.( ! ) serverCancelSupport_field,
               Ppx_yojson_conv_lib.( ! ) tokenModifiers_field,
               Ppx_yojson_conv_lib.( ! ) tokenTypes_field )
           with
           | ( augmentsSyntaxTokens_value,
               dynamicRegistration_value,
               Option.Some formats_value,
               multilineTokenSupport_value,
               overlappingTokenSupport_value,
               Option.Some requests_value,
               serverCancelSupport_value,
               Option.Some tokenModifiers_value,
               Option.Some tokenTypes_value ) ->
             { augmentsSyntaxTokens =
                 (match augmentsSyntaxTokens_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               dynamicRegistration =
                 (match dynamicRegistration_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               formats = formats_value;
               multilineTokenSupport =
                 (match multilineTokenSupport_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               overlappingTokenSupport =
                 (match overlappingTokenSupport_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               requests = requests_value;
               serverCancelSupport =
                 (match serverCancelSupport_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               tokenModifiers = tokenModifiers_value;
               tokenTypes = tokenTypes_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) formats_field)
                     Option.None,
                   "formats" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) requests_field)
                     Option.None,
                   "requests" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) tokenModifiers_field)
                     Option.None,
                   "tokenModifiers" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) tokenTypes_field)
                     Option.None,
                   "tokenTypes" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { augmentsSyntaxTokens = v_augmentsSyntaxTokens;
         dynamicRegistration = v_dynamicRegistration;
         formats = v_formats;
         multilineTokenSupport = v_multilineTokenSupport;
         overlappingTokenSupport = v_overlappingTokenSupport;
         requests = v_requests;
         serverCancelSupport = v_serverCancelSupport;
         tokenModifiers = v_tokenModifiers;
         tokenTypes = v_tokenTypes
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list yojson_of_string v_tokenTypes in
         ("tokenTypes", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_list yojson_of_string v_tokenModifiers in
         ("tokenModifiers", arg) :: bnds
       in
       let bnds =
         if None = v_serverCancelSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_serverCancelSupport
           in
           let bnd = ("serverCancelSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_requests v_requests in
         ("requests", arg) :: bnds
       in
       let bnds =
         if None = v_overlappingTokenSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_overlappingTokenSupport
           in
           let bnd = ("overlappingTokenSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_multilineTokenSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_multilineTokenSupport
           in
           let bnd = ("multilineTokenSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_list TokenFormat.yojson_of_t v_formats in
         ("formats", arg) :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_augmentsSyntaxTokens then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_augmentsSyntaxTokens
           in
           let bnd = ("augmentsSyntaxTokens", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(augmentsSyntaxTokens : bool option)
      ?(dynamicRegistration : bool option) ~(formats : TokenFormat.t list)
      ?(multilineTokenSupport : bool option)
      ?(overlappingTokenSupport : bool option) ~(requests : requests)
      ?(serverCancelSupport : bool option) ~(tokenModifiers : string list)
      ~(tokenTypes : string list) (() : unit) : t =
    { augmentsSyntaxTokens;
      dynamicRegistration;
      formats;
      multilineTokenSupport;
      overlappingTokenSupport;
      requests;
      serverCancelSupport;
      tokenModifiers;
      tokenTypes
    }
end

module SelectionRangeClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.SelectionRangeClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module RenameClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      honorsChangeAnnotations : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      prepareSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      prepareSupportDefaultBehavior :
        PrepareSupportDefaultBehavior.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.RenameClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and honorsChangeAnnotations_field = ref Option.None
       and prepareSupport_field = ref Option.None
       and prepareSupportDefaultBehavior_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "honorsChangeAnnotations" -> (
             match Ppx_yojson_conv_lib.( ! ) honorsChangeAnnotations_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               honorsChangeAnnotations_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "prepareSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) prepareSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               prepareSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "prepareSupportDefaultBehavior" -> (
             match
               Ppx_yojson_conv_lib.( ! ) prepareSupportDefaultBehavior_field
             with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   PrepareSupportDefaultBehavior.t_of_yojson _field_yojson
               in
               prepareSupportDefaultBehavior_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( dynamicRegistration_value,
                 honorsChangeAnnotations_value,
                 prepareSupport_value,
                 prepareSupportDefaultBehavior_value ) =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) honorsChangeAnnotations_field,
               Ppx_yojson_conv_lib.( ! ) prepareSupport_field,
               Ppx_yojson_conv_lib.( ! ) prepareSupportDefaultBehavior_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             honorsChangeAnnotations =
               (match honorsChangeAnnotations_value with
               | Option.None -> None
               | Option.Some v -> v);
             prepareSupport =
               (match prepareSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             prepareSupportDefaultBehavior =
               (match prepareSupportDefaultBehavior_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         honorsChangeAnnotations = v_honorsChangeAnnotations;
         prepareSupport = v_prepareSupport;
         prepareSupportDefaultBehavior = v_prepareSupportDefaultBehavior
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_prepareSupportDefaultBehavior then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                PrepareSupportDefaultBehavior.yojson_of_t)
               v_prepareSupportDefaultBehavior
           in
           let bnd = ("prepareSupportDefaultBehavior", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_prepareSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_prepareSupport
           in
           let bnd = ("prepareSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_honorsChangeAnnotations then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_honorsChangeAnnotations
           in
           let bnd = ("honorsChangeAnnotations", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option)
      ?(honorsChangeAnnotations : bool option) ?(prepareSupport : bool option)
      ?(prepareSupportDefaultBehavior : PrepareSupportDefaultBehavior.t option)
      (() : unit) : t =
    { dynamicRegistration;
      honorsChangeAnnotations;
      prepareSupport;
      prepareSupportDefaultBehavior
    }
end

module ReferenceClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ReferenceClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module DocumentRangeFormattingClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      rangesSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.DocumentRangeFormattingClientCapabilities.t"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and rangesSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "rangesSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) rangesSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               rangesSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value, rangesSupport_value =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) rangesSupport_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             rangesSupport =
               (match rangesSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         rangesSupport = v_rangesSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_rangesSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_rangesSupport
           in
           let bnd = ("rangesSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) ?(rangesSupport : bool option)
      (() : unit) : t =
    { dynamicRegistration; rangesSupport }
end

module PublishDiagnosticsClientCapabilities = struct
  type tagSupport = { valueSet : DiagnosticTag.t list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : tagSupport) -> ()

  let tagSupport_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.PublishDiagnosticsClientCapabilities.tagSupport"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let valueSet_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "valueSet" -> (
             match Ppx_yojson_conv_lib.( ! ) valueSet_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson DiagnosticTag.t_of_yojson _field_yojson
               in
               valueSet_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) valueSet_field with
           | Option.Some valueSet_value -> { valueSet = valueSet_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) valueSet_field)
                     Option.None,
                   "valueSet" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> tagSupport)

  let _ = tagSupport_of_yojson

  let yojson_of_tagSupport =
    (function
     | { valueSet = v_valueSet } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list DiagnosticTag.yojson_of_t v_valueSet in
         ("valueSet", arg) :: bnds
       in
       `Assoc bnds
      : tagSupport -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_tagSupport

  [@@@end]

  let create_tagSupport ~(valueSet : DiagnosticTag.t list) : tagSupport =
    { valueSet }

  type t =
    { codeDescriptionSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      dataSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      relatedInformation : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      tagSupport : tagSupport Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      versionSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.PublishDiagnosticsClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let codeDescriptionSupport_field = ref Option.None
       and dataSupport_field = ref Option.None
       and relatedInformation_field = ref Option.None
       and tagSupport_field = ref Option.None
       and versionSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "codeDescriptionSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) codeDescriptionSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               codeDescriptionSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "dataSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) dataSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dataSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "relatedInformation" -> (
             match Ppx_yojson_conv_lib.( ! ) relatedInformation_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               relatedInformation_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "tagSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) tagSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson tagSupport_of_yojson
                   _field_yojson
               in
               tagSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "versionSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) versionSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               versionSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( codeDescriptionSupport_value,
                 dataSupport_value,
                 relatedInformation_value,
                 tagSupport_value,
                 versionSupport_value ) =
             ( Ppx_yojson_conv_lib.( ! ) codeDescriptionSupport_field,
               Ppx_yojson_conv_lib.( ! ) dataSupport_field,
               Ppx_yojson_conv_lib.( ! ) relatedInformation_field,
               Ppx_yojson_conv_lib.( ! ) tagSupport_field,
               Ppx_yojson_conv_lib.( ! ) versionSupport_field )
           in
           { codeDescriptionSupport =
               (match codeDescriptionSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             dataSupport =
               (match dataSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             relatedInformation =
               (match relatedInformation_value with
               | Option.None -> None
               | Option.Some v -> v);
             tagSupport =
               (match tagSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             versionSupport =
               (match versionSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { codeDescriptionSupport = v_codeDescriptionSupport;
         dataSupport = v_dataSupport;
         relatedInformation = v_relatedInformation;
         tagSupport = v_tagSupport;
         versionSupport = v_versionSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_versionSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_versionSupport
           in
           let bnd = ("versionSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_tagSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_tagSupport)
               v_tagSupport
           in
           let bnd = ("tagSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_relatedInformation then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_relatedInformation
           in
           let bnd = ("relatedInformation", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dataSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_dataSupport
           in
           let bnd = ("dataSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_codeDescriptionSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_codeDescriptionSupport
           in
           let bnd = ("codeDescriptionSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(codeDescriptionSupport : bool option)
      ?(dataSupport : bool option) ?(relatedInformation : bool option)
      ?(tagSupport : tagSupport option) ?(versionSupport : bool option)
      (() : unit) : t =
    { codeDescriptionSupport;
      dataSupport;
      relatedInformation;
      tagSupport;
      versionSupport
    }
end

module DocumentOnTypeFormattingClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.DocumentOnTypeFormattingClientCapabilities.t"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module MonikerClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.MonikerClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module LinkedEditingRangeClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.LinkedEditingRangeClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module InlineValueClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.InlineValueClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module InlineCompletionClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.InlineCompletionClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module InlayHintClientCapabilities = struct
  type resolveSupport = { properties : string list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : resolveSupport) -> ()

  let resolveSupport_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.InlayHintClientCapabilities.resolveSupport"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let properties_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "properties" -> (
             match Ppx_yojson_conv_lib.( ! ) properties_field with
             | Option.None ->
               let fvalue = list_of_yojson string_of_yojson _field_yojson in
               properties_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) properties_field with
           | Option.Some properties_value -> { properties = properties_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) properties_field)
                     Option.None,
                   "properties" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> resolveSupport)

  let _ = resolveSupport_of_yojson

  let yojson_of_resolveSupport =
    (function
     | { properties = v_properties } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list yojson_of_string v_properties in
         ("properties", arg) :: bnds
       in
       `Assoc bnds
      : resolveSupport -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_resolveSupport

  [@@@end]

  let create_resolveSupport ~(properties : string list) : resolveSupport =
    { properties }

  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      resolveSupport : resolveSupport Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.InlayHintClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and resolveSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "resolveSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) resolveSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson resolveSupport_of_yojson
                   _field_yojson
               in
               resolveSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value, resolveSupport_value =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) resolveSupport_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             resolveSupport =
               (match resolveSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         resolveSupport = v_resolveSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_resolveSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_resolveSupport)
               v_resolveSupport
           in
           let bnd = ("resolveSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option)
      ?(resolveSupport : resolveSupport option) (() : unit) : t =
    { dynamicRegistration; resolveSupport }
end

module ImplementationClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      linkSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ImplementationClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and linkSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "linkSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) linkSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               linkSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value, linkSupport_value =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) linkSupport_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             linkSupport =
               (match linkSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         linkSupport = v_linkSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_linkSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_linkSupport
           in
           let bnd = ("linkSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) ?(linkSupport : bool option)
      (() : unit) : t =
    { dynamicRegistration; linkSupport }
end

module HoverClientCapabilities = struct
  type t =
    { contentFormat : MarkupKind.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.HoverClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let contentFormat_field = ref Option.None
       and dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "contentFormat" -> (
             match Ppx_yojson_conv_lib.( ! ) contentFormat_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson MarkupKind.t_of_yojson)
                   _field_yojson
               in
               contentFormat_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let contentFormat_value, dynamicRegistration_value =
             ( Ppx_yojson_conv_lib.( ! ) contentFormat_field,
               Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field )
           in
           { contentFormat =
               (match contentFormat_value with
               | Option.None -> None
               | Option.Some v -> v);
             dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { contentFormat = v_contentFormat;
         dynamicRegistration = v_dynamicRegistration
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_contentFormat then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list MarkupKind.yojson_of_t))
               v_contentFormat
           in
           let bnd = ("contentFormat", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(contentFormat : MarkupKind.t list option)
      ?(dynamicRegistration : bool option) (() : unit) : t =
    { contentFormat; dynamicRegistration }
end

module DocumentFormattingClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DocumentFormattingClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module FoldingRangeClientCapabilities = struct
  type foldingRangeKind =
    { valueSet : FoldingRangeKind.t list Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : foldingRangeKind) -> ()

  let foldingRangeKind_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.FoldingRangeClientCapabilities.foldingRangeKind"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let valueSet_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "valueSet" -> (
             match Ppx_yojson_conv_lib.( ! ) valueSet_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson FoldingRangeKind.t_of_yojson)
                   _field_yojson
               in
               valueSet_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let valueSet_value = Ppx_yojson_conv_lib.( ! ) valueSet_field in
           { valueSet =
               (match valueSet_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> foldingRangeKind)

  let _ = foldingRangeKind_of_yojson

  let yojson_of_foldingRangeKind =
    (function
     | { valueSet = v_valueSet } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_valueSet then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list FoldingRangeKind.yojson_of_t))
               v_valueSet
           in
           let bnd = ("valueSet", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : foldingRangeKind -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_foldingRangeKind

  [@@@end]

  let create_foldingRangeKind ?(valueSet : FoldingRangeKind.t list option)
      (() : unit) : foldingRangeKind =
    { valueSet }

  type foldingRange =
    { collapsedText : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : foldingRange) -> ()

  let foldingRange_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.FoldingRangeClientCapabilities.foldingRange"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let collapsedText_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "collapsedText" -> (
             match Ppx_yojson_conv_lib.( ! ) collapsedText_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               collapsedText_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let collapsedText_value =
             Ppx_yojson_conv_lib.( ! ) collapsedText_field
           in
           { collapsedText =
               (match collapsedText_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> foldingRange)

  let _ = foldingRange_of_yojson

  let yojson_of_foldingRange =
    (function
     | { collapsedText = v_collapsedText } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_collapsedText then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_collapsedText
           in
           let bnd = ("collapsedText", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : foldingRange -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_foldingRange

  [@@@end]

  let create_foldingRange ?(collapsedText : bool option) (() : unit) :
      foldingRange =
    { collapsedText }

  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      foldingRange : foldingRange Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      foldingRangeKind : foldingRangeKind Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      lineFoldingOnly : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      rangeLimit : int Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.FoldingRangeClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and foldingRange_field = ref Option.None
       and foldingRangeKind_field = ref Option.None
       and lineFoldingOnly_field = ref Option.None
       and rangeLimit_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "foldingRange" -> (
             match Ppx_yojson_conv_lib.( ! ) foldingRange_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson foldingRange_of_yojson
                   _field_yojson
               in
               foldingRange_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "foldingRangeKind" -> (
             match Ppx_yojson_conv_lib.( ! ) foldingRangeKind_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson foldingRangeKind_of_yojson
                   _field_yojson
               in
               foldingRangeKind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "lineFoldingOnly" -> (
             match Ppx_yojson_conv_lib.( ! ) lineFoldingOnly_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               lineFoldingOnly_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "rangeLimit" -> (
             match Ppx_yojson_conv_lib.( ! ) rangeLimit_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson int_of_yojson _field_yojson
               in
               rangeLimit_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( dynamicRegistration_value,
                 foldingRange_value,
                 foldingRangeKind_value,
                 lineFoldingOnly_value,
                 rangeLimit_value ) =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) foldingRange_field,
               Ppx_yojson_conv_lib.( ! ) foldingRangeKind_field,
               Ppx_yojson_conv_lib.( ! ) lineFoldingOnly_field,
               Ppx_yojson_conv_lib.( ! ) rangeLimit_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             foldingRange =
               (match foldingRange_value with
               | Option.None -> None
               | Option.Some v -> v);
             foldingRangeKind =
               (match foldingRangeKind_value with
               | Option.None -> None
               | Option.Some v -> v);
             lineFoldingOnly =
               (match lineFoldingOnly_value with
               | Option.None -> None
               | Option.Some v -> v);
             rangeLimit =
               (match rangeLimit_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         foldingRange = v_foldingRange;
         foldingRangeKind = v_foldingRangeKind;
         lineFoldingOnly = v_lineFoldingOnly;
         rangeLimit = v_rangeLimit
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_rangeLimit then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_int) v_rangeLimit
           in
           let bnd = ("rangeLimit", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_lineFoldingOnly then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_lineFoldingOnly
           in
           let bnd = ("lineFoldingOnly", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_foldingRangeKind then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_foldingRangeKind)
               v_foldingRangeKind
           in
           let bnd = ("foldingRangeKind", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_foldingRange then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_foldingRange)
               v_foldingRange
           in
           let bnd = ("foldingRange", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option)
      ?(foldingRange : foldingRange option)
      ?(foldingRangeKind : foldingRangeKind option)
      ?(lineFoldingOnly : bool option) ?(rangeLimit : int option) (() : unit) :
      t =
    { dynamicRegistration;
      foldingRange;
      foldingRangeKind;
      lineFoldingOnly;
      rangeLimit
    }
end

module DocumentSymbolClientCapabilities = struct
  type tagSupport = { valueSet : SymbolTag.t list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : tagSupport) -> ()

  let tagSupport_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.DocumentSymbolClientCapabilities.tagSupport"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let valueSet_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "valueSet" -> (
             match Ppx_yojson_conv_lib.( ! ) valueSet_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson SymbolTag.t_of_yojson _field_yojson
               in
               valueSet_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) valueSet_field with
           | Option.Some valueSet_value -> { valueSet = valueSet_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) valueSet_field)
                     Option.None,
                   "valueSet" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> tagSupport)

  let _ = tagSupport_of_yojson

  let yojson_of_tagSupport =
    (function
     | { valueSet = v_valueSet } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list SymbolTag.yojson_of_t v_valueSet in
         ("valueSet", arg) :: bnds
       in
       `Assoc bnds
      : tagSupport -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_tagSupport

  [@@@end]

  let create_tagSupport ~(valueSet : SymbolTag.t list) : tagSupport =
    { valueSet }

  type symbolKind =
    { valueSet : SymbolKind.t list Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : symbolKind) -> ()

  let symbolKind_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.DocumentSymbolClientCapabilities.symbolKind"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let valueSet_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "valueSet" -> (
             match Ppx_yojson_conv_lib.( ! ) valueSet_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson SymbolKind.t_of_yojson)
                   _field_yojson
               in
               valueSet_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let valueSet_value = Ppx_yojson_conv_lib.( ! ) valueSet_field in
           { valueSet =
               (match valueSet_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> symbolKind)

  let _ = symbolKind_of_yojson

  let yojson_of_symbolKind =
    (function
     | { valueSet = v_valueSet } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_valueSet then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list SymbolKind.yojson_of_t))
               v_valueSet
           in
           let bnd = ("valueSet", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : symbolKind -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_symbolKind

  [@@@end]

  let create_symbolKind ?(valueSet : SymbolKind.t list option) (() : unit) :
      symbolKind =
    { valueSet }

  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      hierarchicalDocumentSymbolSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      labelSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      symbolKind : symbolKind Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      tagSupport : tagSupport Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DocumentSymbolClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and hierarchicalDocumentSymbolSupport_field = ref Option.None
       and labelSupport_field = ref Option.None
       and symbolKind_field = ref Option.None
       and tagSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "hierarchicalDocumentSymbolSupport" -> (
             match
               Ppx_yojson_conv_lib.( ! ) hierarchicalDocumentSymbolSupport_field
             with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               hierarchicalDocumentSymbolSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "labelSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) labelSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               labelSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "symbolKind" -> (
             match Ppx_yojson_conv_lib.( ! ) symbolKind_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson symbolKind_of_yojson
                   _field_yojson
               in
               symbolKind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "tagSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) tagSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson tagSupport_of_yojson
                   _field_yojson
               in
               tagSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( dynamicRegistration_value,
                 hierarchicalDocumentSymbolSupport_value,
                 labelSupport_value,
                 symbolKind_value,
                 tagSupport_value ) =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) hierarchicalDocumentSymbolSupport_field,
               Ppx_yojson_conv_lib.( ! ) labelSupport_field,
               Ppx_yojson_conv_lib.( ! ) symbolKind_field,
               Ppx_yojson_conv_lib.( ! ) tagSupport_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             hierarchicalDocumentSymbolSupport =
               (match hierarchicalDocumentSymbolSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             labelSupport =
               (match labelSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             symbolKind =
               (match symbolKind_value with
               | Option.None -> None
               | Option.Some v -> v);
             tagSupport =
               (match tagSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         hierarchicalDocumentSymbolSupport = v_hierarchicalDocumentSymbolSupport;
         labelSupport = v_labelSupport;
         symbolKind = v_symbolKind;
         tagSupport = v_tagSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_tagSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_tagSupport)
               v_tagSupport
           in
           let bnd = ("tagSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_symbolKind then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_symbolKind)
               v_symbolKind
           in
           let bnd = ("symbolKind", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_labelSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_labelSupport
           in
           let bnd = ("labelSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_hierarchicalDocumentSymbolSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_hierarchicalDocumentSymbolSupport
           in
           let bnd = ("hierarchicalDocumentSymbolSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option)
      ?(hierarchicalDocumentSymbolSupport : bool option)
      ?(labelSupport : bool option) ?(symbolKind : symbolKind option)
      ?(tagSupport : tagSupport option) (() : unit) : t =
    { dynamicRegistration;
      hierarchicalDocumentSymbolSupport;
      labelSupport;
      symbolKind;
      tagSupport
    }
end

module DocumentLinkClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      tooltipSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DocumentLinkClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and tooltipSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "tooltipSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) tooltipSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               tooltipSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value, tooltipSupport_value =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) tooltipSupport_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             tooltipSupport =
               (match tooltipSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         tooltipSupport = v_tooltipSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_tooltipSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_tooltipSupport
           in
           let bnd = ("tooltipSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option)
      ?(tooltipSupport : bool option) (() : unit) : t =
    { dynamicRegistration; tooltipSupport }
end

module DocumentHighlightClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DocumentHighlightClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module DiagnosticClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      markupMessageSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      relatedDocumentSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DiagnosticClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and markupMessageSupport_field = ref Option.None
       and relatedDocumentSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "markupMessageSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) markupMessageSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               markupMessageSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "relatedDocumentSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) relatedDocumentSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               relatedDocumentSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( dynamicRegistration_value,
                 markupMessageSupport_value,
                 relatedDocumentSupport_value ) =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) markupMessageSupport_field,
               Ppx_yojson_conv_lib.( ! ) relatedDocumentSupport_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             markupMessageSupport =
               (match markupMessageSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             relatedDocumentSupport =
               (match relatedDocumentSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         markupMessageSupport = v_markupMessageSupport;
         relatedDocumentSupport = v_relatedDocumentSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_relatedDocumentSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_relatedDocumentSupport
           in
           let bnd = ("relatedDocumentSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_markupMessageSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_markupMessageSupport
           in
           let bnd = ("markupMessageSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option)
      ?(markupMessageSupport : bool option)
      ?(relatedDocumentSupport : bool option) (() : unit) : t =
    { dynamicRegistration; markupMessageSupport; relatedDocumentSupport }
end

module DefinitionClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      linkSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DefinitionClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and linkSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "linkSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) linkSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               linkSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value, linkSupport_value =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) linkSupport_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             linkSupport =
               (match linkSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         linkSupport = v_linkSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_linkSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_linkSupport
           in
           let bnd = ("linkSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) ?(linkSupport : bool option)
      (() : unit) : t =
    { dynamicRegistration; linkSupport }
end

module DeclarationClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      linkSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DeclarationClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and linkSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "linkSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) linkSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               linkSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value, linkSupport_value =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) linkSupport_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             linkSupport =
               (match linkSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         linkSupport = v_linkSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_linkSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_linkSupport
           in
           let bnd = ("linkSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) ?(linkSupport : bool option)
      (() : unit) : t =
    { dynamicRegistration; linkSupport }
end

module CompletionClientCapabilities = struct
  type completionList =
    { itemDefaults : string list Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : completionList) -> ()

  let completionList_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.CompletionClientCapabilities.completionList"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let itemDefaults_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "itemDefaults" -> (
             match Ppx_yojson_conv_lib.( ! ) itemDefaults_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson string_of_yojson)
                   _field_yojson
               in
               itemDefaults_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let itemDefaults_value =
             Ppx_yojson_conv_lib.( ! ) itemDefaults_field
           in
           { itemDefaults =
               (match itemDefaults_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> completionList)

  let _ = completionList_of_yojson

  let yojson_of_completionList =
    (function
     | { itemDefaults = v_itemDefaults } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_itemDefaults then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list yojson_of_string))
               v_itemDefaults
           in
           let bnd = ("itemDefaults", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : completionList -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_completionList

  [@@@end]

  let create_completionList ?(itemDefaults : string list option) (() : unit) :
      completionList =
    { itemDefaults }

  type completionItemKind =
    { valueSet : CompletionItemKind.t list Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : completionItemKind) -> ()

  let completionItemKind_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.CompletionClientCapabilities.completionItemKind"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let valueSet_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "valueSet" -> (
             match Ppx_yojson_conv_lib.( ! ) valueSet_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson CompletionItemKind.t_of_yojson)
                   _field_yojson
               in
               valueSet_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let valueSet_value = Ppx_yojson_conv_lib.( ! ) valueSet_field in
           { valueSet =
               (match valueSet_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> completionItemKind)

  let _ = completionItemKind_of_yojson

  let yojson_of_completionItemKind =
    (function
     | { valueSet = v_valueSet } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_valueSet then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list CompletionItemKind.yojson_of_t))
               v_valueSet
           in
           let bnd = ("valueSet", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : completionItemKind -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_completionItemKind

  [@@@end]

  let create_completionItemKind ?(valueSet : CompletionItemKind.t list option)
      (() : unit) : completionItemKind =
    { valueSet }

  type insertTextModeSupport = { valueSet : InsertTextMode.t list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : insertTextModeSupport) -> ()

  let insertTextModeSupport_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.CompletionClientCapabilities.insertTextModeSupport"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let valueSet_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "valueSet" -> (
             match Ppx_yojson_conv_lib.( ! ) valueSet_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson InsertTextMode.t_of_yojson _field_yojson
               in
               valueSet_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) valueSet_field with
           | Option.Some valueSet_value -> { valueSet = valueSet_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) valueSet_field)
                     Option.None,
                   "valueSet" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> insertTextModeSupport)

  let _ = insertTextModeSupport_of_yojson

  let yojson_of_insertTextModeSupport =
    (function
     | { valueSet = v_valueSet } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list InsertTextMode.yojson_of_t v_valueSet in
         ("valueSet", arg) :: bnds
       in
       `Assoc bnds
      : insertTextModeSupport -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_insertTextModeSupport

  [@@@end]

  let create_insertTextModeSupport ~(valueSet : InsertTextMode.t list) :
      insertTextModeSupport =
    { valueSet }

  type resolveSupport = { properties : string list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : resolveSupport) -> ()

  let resolveSupport_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.CompletionClientCapabilities.resolveSupport"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let properties_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "properties" -> (
             match Ppx_yojson_conv_lib.( ! ) properties_field with
             | Option.None ->
               let fvalue = list_of_yojson string_of_yojson _field_yojson in
               properties_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) properties_field with
           | Option.Some properties_value -> { properties = properties_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) properties_field)
                     Option.None,
                   "properties" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> resolveSupport)

  let _ = resolveSupport_of_yojson

  let yojson_of_resolveSupport =
    (function
     | { properties = v_properties } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list yojson_of_string v_properties in
         ("properties", arg) :: bnds
       in
       `Assoc bnds
      : resolveSupport -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_resolveSupport

  [@@@end]

  let create_resolveSupport ~(properties : string list) : resolveSupport =
    { properties }

  type tagSupport = { valueSet : CompletionItemTag.t list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : tagSupport) -> ()

  let tagSupport_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CompletionClientCapabilities.tagSupport" in
     function
     | `Assoc field_yojsons as yojson -> (
       let valueSet_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "valueSet" -> (
             match Ppx_yojson_conv_lib.( ! ) valueSet_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson CompletionItemTag.t_of_yojson _field_yojson
               in
               valueSet_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) valueSet_field with
           | Option.Some valueSet_value -> { valueSet = valueSet_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) valueSet_field)
                     Option.None,
                   "valueSet" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> tagSupport)

  let _ = tagSupport_of_yojson

  let yojson_of_tagSupport =
    (function
     | { valueSet = v_valueSet } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list CompletionItemTag.yojson_of_t v_valueSet in
         ("valueSet", arg) :: bnds
       in
       `Assoc bnds
      : tagSupport -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_tagSupport

  [@@@end]

  let create_tagSupport ~(valueSet : CompletionItemTag.t list) : tagSupport =
    { valueSet }

  type completionItem =
    { snippetSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      commitCharactersSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      documentationFormat : MarkupKind.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      deprecatedSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      preselectSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      tagSupport : tagSupport Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      insertReplaceSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      resolveSupport : resolveSupport Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      insertTextModeSupport : insertTextModeSupport Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      labelDetailsSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : completionItem) -> ()

  let completionItem_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.CompletionClientCapabilities.completionItem"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let snippetSupport_field = ref Option.None
       and commitCharactersSupport_field = ref Option.None
       and documentationFormat_field = ref Option.None
       and deprecatedSupport_field = ref Option.None
       and preselectSupport_field = ref Option.None
       and tagSupport_field = ref Option.None
       and insertReplaceSupport_field = ref Option.None
       and resolveSupport_field = ref Option.None
       and insertTextModeSupport_field = ref Option.None
       and labelDetailsSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "snippetSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) snippetSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               snippetSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "commitCharactersSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) commitCharactersSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               commitCharactersSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "documentationFormat" -> (
             match Ppx_yojson_conv_lib.( ! ) documentationFormat_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson MarkupKind.t_of_yojson)
                   _field_yojson
               in
               documentationFormat_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "deprecatedSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) deprecatedSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               deprecatedSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "preselectSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) preselectSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               preselectSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "tagSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) tagSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson tagSupport_of_yojson
                   _field_yojson
               in
               tagSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "insertReplaceSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) insertReplaceSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               insertReplaceSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "resolveSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) resolveSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson resolveSupport_of_yojson
                   _field_yojson
               in
               resolveSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "insertTextModeSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) insertTextModeSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   insertTextModeSupport_of_yojson _field_yojson
               in
               insertTextModeSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "labelDetailsSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) labelDetailsSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               labelDetailsSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( snippetSupport_value,
                 commitCharactersSupport_value,
                 documentationFormat_value,
                 deprecatedSupport_value,
                 preselectSupport_value,
                 tagSupport_value,
                 insertReplaceSupport_value,
                 resolveSupport_value,
                 insertTextModeSupport_value,
                 labelDetailsSupport_value ) =
             ( Ppx_yojson_conv_lib.( ! ) snippetSupport_field,
               Ppx_yojson_conv_lib.( ! ) commitCharactersSupport_field,
               Ppx_yojson_conv_lib.( ! ) documentationFormat_field,
               Ppx_yojson_conv_lib.( ! ) deprecatedSupport_field,
               Ppx_yojson_conv_lib.( ! ) preselectSupport_field,
               Ppx_yojson_conv_lib.( ! ) tagSupport_field,
               Ppx_yojson_conv_lib.( ! ) insertReplaceSupport_field,
               Ppx_yojson_conv_lib.( ! ) resolveSupport_field,
               Ppx_yojson_conv_lib.( ! ) insertTextModeSupport_field,
               Ppx_yojson_conv_lib.( ! ) labelDetailsSupport_field )
           in
           { snippetSupport =
               (match snippetSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             commitCharactersSupport =
               (match commitCharactersSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             documentationFormat =
               (match documentationFormat_value with
               | Option.None -> None
               | Option.Some v -> v);
             deprecatedSupport =
               (match deprecatedSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             preselectSupport =
               (match preselectSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             tagSupport =
               (match tagSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             insertReplaceSupport =
               (match insertReplaceSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             resolveSupport =
               (match resolveSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             insertTextModeSupport =
               (match insertTextModeSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             labelDetailsSupport =
               (match labelDetailsSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> completionItem)

  let _ = completionItem_of_yojson

  let yojson_of_completionItem =
    (function
     | { snippetSupport = v_snippetSupport;
         commitCharactersSupport = v_commitCharactersSupport;
         documentationFormat = v_documentationFormat;
         deprecatedSupport = v_deprecatedSupport;
         preselectSupport = v_preselectSupport;
         tagSupport = v_tagSupport;
         insertReplaceSupport = v_insertReplaceSupport;
         resolveSupport = v_resolveSupport;
         insertTextModeSupport = v_insertTextModeSupport;
         labelDetailsSupport = v_labelDetailsSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_labelDetailsSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_labelDetailsSupport
           in
           let bnd = ("labelDetailsSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_insertTextModeSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_insertTextModeSupport)
               v_insertTextModeSupport
           in
           let bnd = ("insertTextModeSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_resolveSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_resolveSupport)
               v_resolveSupport
           in
           let bnd = ("resolveSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_insertReplaceSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_insertReplaceSupport
           in
           let bnd = ("insertReplaceSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_tagSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_tagSupport)
               v_tagSupport
           in
           let bnd = ("tagSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_preselectSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_preselectSupport
           in
           let bnd = ("preselectSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_deprecatedSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_deprecatedSupport
           in
           let bnd = ("deprecatedSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentationFormat then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list MarkupKind.yojson_of_t))
               v_documentationFormat
           in
           let bnd = ("documentationFormat", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_commitCharactersSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_commitCharactersSupport
           in
           let bnd = ("commitCharactersSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_snippetSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_snippetSupport
           in
           let bnd = ("snippetSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : completionItem -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_completionItem

  [@@@end]

  let create_completionItem ?(snippetSupport : bool option)
      ?(commitCharactersSupport : bool option)
      ?(documentationFormat : MarkupKind.t list option)
      ?(deprecatedSupport : bool option) ?(preselectSupport : bool option)
      ?(tagSupport : tagSupport option) ?(insertReplaceSupport : bool option)
      ?(resolveSupport : resolveSupport option)
      ?(insertTextModeSupport : insertTextModeSupport option)
      ?(labelDetailsSupport : bool option) (() : unit) : completionItem =
    { snippetSupport;
      commitCharactersSupport;
      documentationFormat;
      deprecatedSupport;
      preselectSupport;
      tagSupport;
      insertReplaceSupport;
      resolveSupport;
      insertTextModeSupport;
      labelDetailsSupport
    }

  type t =
    { completionItem : completionItem Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      completionItemKind : completionItemKind Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      completionList : completionList Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      contextSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      insertTextMode : InsertTextMode.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CompletionClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let completionItem_field = ref Option.None
       and completionItemKind_field = ref Option.None
       and completionList_field = ref Option.None
       and contextSupport_field = ref Option.None
       and dynamicRegistration_field = ref Option.None
       and insertTextMode_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "completionItem" -> (
             match Ppx_yojson_conv_lib.( ! ) completionItem_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson completionItem_of_yojson
                   _field_yojson
               in
               completionItem_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "completionItemKind" -> (
             match Ppx_yojson_conv_lib.( ! ) completionItemKind_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson completionItemKind_of_yojson
                   _field_yojson
               in
               completionItemKind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "completionList" -> (
             match Ppx_yojson_conv_lib.( ! ) completionList_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson completionList_of_yojson
                   _field_yojson
               in
               completionList_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "contextSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) contextSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               contextSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "insertTextMode" -> (
             match Ppx_yojson_conv_lib.( ! ) insertTextMode_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson InsertTextMode.t_of_yojson
                   _field_yojson
               in
               insertTextMode_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( completionItem_value,
                 completionItemKind_value,
                 completionList_value,
                 contextSupport_value,
                 dynamicRegistration_value,
                 insertTextMode_value ) =
             ( Ppx_yojson_conv_lib.( ! ) completionItem_field,
               Ppx_yojson_conv_lib.( ! ) completionItemKind_field,
               Ppx_yojson_conv_lib.( ! ) completionList_field,
               Ppx_yojson_conv_lib.( ! ) contextSupport_field,
               Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) insertTextMode_field )
           in
           { completionItem =
               (match completionItem_value with
               | Option.None -> None
               | Option.Some v -> v);
             completionItemKind =
               (match completionItemKind_value with
               | Option.None -> None
               | Option.Some v -> v);
             completionList =
               (match completionList_value with
               | Option.None -> None
               | Option.Some v -> v);
             contextSupport =
               (match contextSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             insertTextMode =
               (match insertTextMode_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { completionItem = v_completionItem;
         completionItemKind = v_completionItemKind;
         completionList = v_completionList;
         contextSupport = v_contextSupport;
         dynamicRegistration = v_dynamicRegistration;
         insertTextMode = v_insertTextMode
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_insertTextMode then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t InsertTextMode.yojson_of_t)
               v_insertTextMode
           in
           let bnd = ("insertTextMode", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_contextSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_contextSupport
           in
           let bnd = ("contextSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_completionList then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_completionList)
               v_completionList
           in
           let bnd = ("completionList", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_completionItemKind then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_completionItemKind)
               v_completionItemKind
           in
           let bnd = ("completionItemKind", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_completionItem then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_completionItem)
               v_completionItem
           in
           let bnd = ("completionItem", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(completionItem : completionItem option)
      ?(completionItemKind : completionItemKind option)
      ?(completionList : completionList option) ?(contextSupport : bool option)
      ?(dynamicRegistration : bool option)
      ?(insertTextMode : InsertTextMode.t option) (() : unit) : t =
    { completionItem;
      completionItemKind;
      completionList;
      contextSupport;
      dynamicRegistration;
      insertTextMode
    }
end

module DocumentColorClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DocumentColorClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module CodeLensClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeLensClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value =
             Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option) (() : unit) : t =
    { dynamicRegistration }
end

module CodeActionClientCapabilities = struct
  type resolveSupport = { properties : string list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : resolveSupport) -> ()

  let resolveSupport_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.CodeActionClientCapabilities.resolveSupport"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let properties_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "properties" -> (
             match Ppx_yojson_conv_lib.( ! ) properties_field with
             | Option.None ->
               let fvalue = list_of_yojson string_of_yojson _field_yojson in
               properties_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) properties_field with
           | Option.Some properties_value -> { properties = properties_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) properties_field)
                     Option.None,
                   "properties" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> resolveSupport)

  let _ = resolveSupport_of_yojson

  let yojson_of_resolveSupport =
    (function
     | { properties = v_properties } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list yojson_of_string v_properties in
         ("properties", arg) :: bnds
       in
       `Assoc bnds
      : resolveSupport -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_resolveSupport

  [@@@end]

  let create_resolveSupport ~(properties : string list) : resolveSupport =
    { properties }

  type codeActionKind = { valueSet : CodeActionKind.t list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : codeActionKind) -> ()

  let codeActionKind_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.CodeActionClientCapabilities.codeActionKind"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let valueSet_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "valueSet" -> (
             match Ppx_yojson_conv_lib.( ! ) valueSet_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson CodeActionKind.t_of_yojson _field_yojson
               in
               valueSet_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) valueSet_field with
           | Option.Some valueSet_value -> { valueSet = valueSet_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) valueSet_field)
                     Option.None,
                   "valueSet" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> codeActionKind)

  let _ = codeActionKind_of_yojson

  let yojson_of_codeActionKind =
    (function
     | { valueSet = v_valueSet } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list CodeActionKind.yojson_of_t v_valueSet in
         ("valueSet", arg) :: bnds
       in
       `Assoc bnds
      : codeActionKind -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_codeActionKind

  [@@@end]

  let create_codeActionKind ~(valueSet : CodeActionKind.t list) : codeActionKind
      =
    { valueSet }

  type codeActionLiteralSupport = { codeActionKind : codeActionKind }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : codeActionLiteralSupport) -> ()

  let codeActionLiteralSupport_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.CodeActionClientCapabilities.codeActionLiteralSupport"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let codeActionKind_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "codeActionKind" -> (
             match Ppx_yojson_conv_lib.( ! ) codeActionKind_field with
             | Option.None ->
               let fvalue = codeActionKind_of_yojson _field_yojson in
               codeActionKind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) codeActionKind_field with
           | Option.Some codeActionKind_value ->
             { codeActionKind = codeActionKind_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) codeActionKind_field)
                     Option.None,
                   "codeActionKind" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> codeActionLiteralSupport)

  let _ = codeActionLiteralSupport_of_yojson

  let yojson_of_codeActionLiteralSupport =
    (function
     | { codeActionKind = v_codeActionKind } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_codeActionKind v_codeActionKind in
         ("codeActionKind", arg) :: bnds
       in
       `Assoc bnds
      : codeActionLiteralSupport -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_codeActionLiteralSupport

  [@@@end]

  let create_codeActionLiteralSupport ~(codeActionKind : codeActionKind) :
      codeActionLiteralSupport =
    { codeActionKind }

  type t =
    { codeActionLiteralSupport :
        codeActionLiteralSupport Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      dataSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      disabledSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      honorsChangeAnnotations : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      isPreferredSupport : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      resolveSupport : resolveSupport Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeActionClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let codeActionLiteralSupport_field = ref Option.None
       and dataSupport_field = ref Option.None
       and disabledSupport_field = ref Option.None
       and dynamicRegistration_field = ref Option.None
       and honorsChangeAnnotations_field = ref Option.None
       and isPreferredSupport_field = ref Option.None
       and resolveSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "codeActionLiteralSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) codeActionLiteralSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   codeActionLiteralSupport_of_yojson _field_yojson
               in
               codeActionLiteralSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "dataSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) dataSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dataSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "disabledSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) disabledSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               disabledSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "honorsChangeAnnotations" -> (
             match Ppx_yojson_conv_lib.( ! ) honorsChangeAnnotations_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               honorsChangeAnnotations_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "isPreferredSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) isPreferredSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               isPreferredSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "resolveSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) resolveSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson resolveSupport_of_yojson
                   _field_yojson
               in
               resolveSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( codeActionLiteralSupport_value,
                 dataSupport_value,
                 disabledSupport_value,
                 dynamicRegistration_value,
                 honorsChangeAnnotations_value,
                 isPreferredSupport_value,
                 resolveSupport_value ) =
             ( Ppx_yojson_conv_lib.( ! ) codeActionLiteralSupport_field,
               Ppx_yojson_conv_lib.( ! ) dataSupport_field,
               Ppx_yojson_conv_lib.( ! ) disabledSupport_field,
               Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) honorsChangeAnnotations_field,
               Ppx_yojson_conv_lib.( ! ) isPreferredSupport_field,
               Ppx_yojson_conv_lib.( ! ) resolveSupport_field )
           in
           { codeActionLiteralSupport =
               (match codeActionLiteralSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             dataSupport =
               (match dataSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             disabledSupport =
               (match disabledSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             honorsChangeAnnotations =
               (match honorsChangeAnnotations_value with
               | Option.None -> None
               | Option.Some v -> v);
             isPreferredSupport =
               (match isPreferredSupport_value with
               | Option.None -> None
               | Option.Some v -> v);
             resolveSupport =
               (match resolveSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { codeActionLiteralSupport = v_codeActionLiteralSupport;
         dataSupport = v_dataSupport;
         disabledSupport = v_disabledSupport;
         dynamicRegistration = v_dynamicRegistration;
         honorsChangeAnnotations = v_honorsChangeAnnotations;
         isPreferredSupport = v_isPreferredSupport;
         resolveSupport = v_resolveSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_resolveSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_resolveSupport)
               v_resolveSupport
           in
           let bnd = ("resolveSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_isPreferredSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_isPreferredSupport
           in
           let bnd = ("isPreferredSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_honorsChangeAnnotations then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_honorsChangeAnnotations
           in
           let bnd = ("honorsChangeAnnotations", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_disabledSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_disabledSupport
           in
           let bnd = ("disabledSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dataSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_dataSupport
           in
           let bnd = ("dataSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_codeActionLiteralSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                yojson_of_codeActionLiteralSupport)
               v_codeActionLiteralSupport
           in
           let bnd = ("codeActionLiteralSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(codeActionLiteralSupport : codeActionLiteralSupport option)
      ?(dataSupport : bool option) ?(disabledSupport : bool option)
      ?(dynamicRegistration : bool option)
      ?(honorsChangeAnnotations : bool option)
      ?(isPreferredSupport : bool option)
      ?(resolveSupport : resolveSupport option) (() : unit) : t =
    { codeActionLiteralSupport;
      dataSupport;
      disabledSupport;
      dynamicRegistration;
      honorsChangeAnnotations;
      isPreferredSupport;
      resolveSupport
    }
end

module TextDocumentClientCapabilities = struct
  type t =
    { callHierarchy : CallHierarchyClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      codeAction : CodeActionClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      codeLens : CodeLensClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      colorProvider : DocumentColorClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      completion : CompletionClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      declaration : DeclarationClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      definition : DefinitionClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      diagnostic : DiagnosticClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      documentHighlight :
        DocumentHighlightClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      documentLink : DocumentLinkClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      documentSymbol :
        DocumentSymbolClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      foldingRange : FoldingRangeClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      formatting :
        DocumentFormattingClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      hover : HoverClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      implementation :
        ImplementationClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      inlayHint : InlayHintClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      inlineCompletion :
        InlineCompletionClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      inlineValue : InlineValueClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      linkedEditingRange :
        LinkedEditingRangeClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      moniker : MonikerClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      onTypeFormatting :
        DocumentOnTypeFormattingClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      publishDiagnostics :
        PublishDiagnosticsClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      rangeFormatting :
        DocumentRangeFormattingClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      references : ReferenceClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      rename : RenameClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      selectionRange :
        SelectionRangeClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      semanticTokens :
        SemanticTokensClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      signatureHelp : SignatureHelpClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      synchronization :
        TextDocumentSyncClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      typeDefinition :
        TypeDefinitionClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      typeHierarchy : TypeHierarchyClientCapabilities.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.TextDocumentClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let callHierarchy_field = ref Option.None
       and codeAction_field = ref Option.None
       and codeLens_field = ref Option.None
       and colorProvider_field = ref Option.None
       and completion_field = ref Option.None
       and declaration_field = ref Option.None
       and definition_field = ref Option.None
       and diagnostic_field = ref Option.None
       and documentHighlight_field = ref Option.None
       and documentLink_field = ref Option.None
       and documentSymbol_field = ref Option.None
       and foldingRange_field = ref Option.None
       and formatting_field = ref Option.None
       and hover_field = ref Option.None
       and implementation_field = ref Option.None
       and inlayHint_field = ref Option.None
       and inlineCompletion_field = ref Option.None
       and inlineValue_field = ref Option.None
       and linkedEditingRange_field = ref Option.None
       and moniker_field = ref Option.None
       and onTypeFormatting_field = ref Option.None
       and publishDiagnostics_field = ref Option.None
       and rangeFormatting_field = ref Option.None
       and references_field = ref Option.None
       and rename_field = ref Option.None
       and selectionRange_field = ref Option.None
       and semanticTokens_field = ref Option.None
       and signatureHelp_field = ref Option.None
       and synchronization_field = ref Option.None
       and typeDefinition_field = ref Option.None
       and typeHierarchy_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "callHierarchy" -> (
             match Ppx_yojson_conv_lib.( ! ) callHierarchy_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   CallHierarchyClientCapabilities.t_of_yojson _field_yojson
               in
               callHierarchy_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "codeAction" -> (
             match Ppx_yojson_conv_lib.( ! ) codeAction_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   CodeActionClientCapabilities.t_of_yojson _field_yojson
               in
               codeAction_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "codeLens" -> (
             match Ppx_yojson_conv_lib.( ! ) codeLens_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   CodeLensClientCapabilities.t_of_yojson _field_yojson
               in
               codeLens_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "colorProvider" -> (
             match Ppx_yojson_conv_lib.( ! ) colorProvider_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   DocumentColorClientCapabilities.t_of_yojson _field_yojson
               in
               colorProvider_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "completion" -> (
             match Ppx_yojson_conv_lib.( ! ) completion_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   CompletionClientCapabilities.t_of_yojson _field_yojson
               in
               completion_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "declaration" -> (
             match Ppx_yojson_conv_lib.( ! ) declaration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   DeclarationClientCapabilities.t_of_yojson _field_yojson
               in
               declaration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "definition" -> (
             match Ppx_yojson_conv_lib.( ! ) definition_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   DefinitionClientCapabilities.t_of_yojson _field_yojson
               in
               definition_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "diagnostic" -> (
             match Ppx_yojson_conv_lib.( ! ) diagnostic_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   DiagnosticClientCapabilities.t_of_yojson _field_yojson
               in
               diagnostic_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "documentHighlight" -> (
             match Ppx_yojson_conv_lib.( ! ) documentHighlight_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   DocumentHighlightClientCapabilities.t_of_yojson _field_yojson
               in
               documentHighlight_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "documentLink" -> (
             match Ppx_yojson_conv_lib.( ! ) documentLink_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   DocumentLinkClientCapabilities.t_of_yojson _field_yojson
               in
               documentLink_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "documentSymbol" -> (
             match Ppx_yojson_conv_lib.( ! ) documentSymbol_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   DocumentSymbolClientCapabilities.t_of_yojson _field_yojson
               in
               documentSymbol_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "foldingRange" -> (
             match Ppx_yojson_conv_lib.( ! ) foldingRange_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   FoldingRangeClientCapabilities.t_of_yojson _field_yojson
               in
               foldingRange_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "formatting" -> (
             match Ppx_yojson_conv_lib.( ! ) formatting_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   DocumentFormattingClientCapabilities.t_of_yojson
                   _field_yojson
               in
               formatting_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "hover" -> (
             match Ppx_yojson_conv_lib.( ! ) hover_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   HoverClientCapabilities.t_of_yojson _field_yojson
               in
               hover_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "implementation" -> (
             match Ppx_yojson_conv_lib.( ! ) implementation_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   ImplementationClientCapabilities.t_of_yojson _field_yojson
               in
               implementation_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "inlayHint" -> (
             match Ppx_yojson_conv_lib.( ! ) inlayHint_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   InlayHintClientCapabilities.t_of_yojson _field_yojson
               in
               inlayHint_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "inlineCompletion" -> (
             match Ppx_yojson_conv_lib.( ! ) inlineCompletion_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   InlineCompletionClientCapabilities.t_of_yojson _field_yojson
               in
               inlineCompletion_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "inlineValue" -> (
             match Ppx_yojson_conv_lib.( ! ) inlineValue_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   InlineValueClientCapabilities.t_of_yojson _field_yojson
               in
               inlineValue_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "linkedEditingRange" -> (
             match Ppx_yojson_conv_lib.( ! ) linkedEditingRange_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   LinkedEditingRangeClientCapabilities.t_of_yojson
                   _field_yojson
               in
               linkedEditingRange_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "moniker" -> (
             match Ppx_yojson_conv_lib.( ! ) moniker_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   MonikerClientCapabilities.t_of_yojson _field_yojson
               in
               moniker_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "onTypeFormatting" -> (
             match Ppx_yojson_conv_lib.( ! ) onTypeFormatting_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   DocumentOnTypeFormattingClientCapabilities.t_of_yojson
                   _field_yojson
               in
               onTypeFormatting_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "publishDiagnostics" -> (
             match Ppx_yojson_conv_lib.( ! ) publishDiagnostics_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   PublishDiagnosticsClientCapabilities.t_of_yojson
                   _field_yojson
               in
               publishDiagnostics_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "rangeFormatting" -> (
             match Ppx_yojson_conv_lib.( ! ) rangeFormatting_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   DocumentRangeFormattingClientCapabilities.t_of_yojson
                   _field_yojson
               in
               rangeFormatting_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "references" -> (
             match Ppx_yojson_conv_lib.( ! ) references_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   ReferenceClientCapabilities.t_of_yojson _field_yojson
               in
               references_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "rename" -> (
             match Ppx_yojson_conv_lib.( ! ) rename_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   RenameClientCapabilities.t_of_yojson _field_yojson
               in
               rename_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "selectionRange" -> (
             match Ppx_yojson_conv_lib.( ! ) selectionRange_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   SelectionRangeClientCapabilities.t_of_yojson _field_yojson
               in
               selectionRange_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "semanticTokens" -> (
             match Ppx_yojson_conv_lib.( ! ) semanticTokens_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   SemanticTokensClientCapabilities.t_of_yojson _field_yojson
               in
               semanticTokens_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "signatureHelp" -> (
             match Ppx_yojson_conv_lib.( ! ) signatureHelp_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   SignatureHelpClientCapabilities.t_of_yojson _field_yojson
               in
               signatureHelp_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "synchronization" -> (
             match Ppx_yojson_conv_lib.( ! ) synchronization_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   TextDocumentSyncClientCapabilities.t_of_yojson _field_yojson
               in
               synchronization_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "typeDefinition" -> (
             match Ppx_yojson_conv_lib.( ! ) typeDefinition_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   TypeDefinitionClientCapabilities.t_of_yojson _field_yojson
               in
               typeDefinition_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "typeHierarchy" -> (
             match Ppx_yojson_conv_lib.( ! ) typeHierarchy_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   TypeHierarchyClientCapabilities.t_of_yojson _field_yojson
               in
               typeHierarchy_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( callHierarchy_value,
                 codeAction_value,
                 codeLens_value,
                 colorProvider_value,
                 completion_value,
                 declaration_value,
                 definition_value,
                 diagnostic_value,
                 documentHighlight_value,
                 documentLink_value,
                 documentSymbol_value,
                 foldingRange_value,
                 formatting_value,
                 hover_value,
                 implementation_value,
                 inlayHint_value,
                 inlineCompletion_value,
                 inlineValue_value,
                 linkedEditingRange_value,
                 moniker_value,
                 onTypeFormatting_value,
                 publishDiagnostics_value,
                 rangeFormatting_value,
                 references_value,
                 rename_value,
                 selectionRange_value,
                 semanticTokens_value,
                 signatureHelp_value,
                 synchronization_value,
                 typeDefinition_value,
                 typeHierarchy_value ) =
             ( Ppx_yojson_conv_lib.( ! ) callHierarchy_field,
               Ppx_yojson_conv_lib.( ! ) codeAction_field,
               Ppx_yojson_conv_lib.( ! ) codeLens_field,
               Ppx_yojson_conv_lib.( ! ) colorProvider_field,
               Ppx_yojson_conv_lib.( ! ) completion_field,
               Ppx_yojson_conv_lib.( ! ) declaration_field,
               Ppx_yojson_conv_lib.( ! ) definition_field,
               Ppx_yojson_conv_lib.( ! ) diagnostic_field,
               Ppx_yojson_conv_lib.( ! ) documentHighlight_field,
               Ppx_yojson_conv_lib.( ! ) documentLink_field,
               Ppx_yojson_conv_lib.( ! ) documentSymbol_field,
               Ppx_yojson_conv_lib.( ! ) foldingRange_field,
               Ppx_yojson_conv_lib.( ! ) formatting_field,
               Ppx_yojson_conv_lib.( ! ) hover_field,
               Ppx_yojson_conv_lib.( ! ) implementation_field,
               Ppx_yojson_conv_lib.( ! ) inlayHint_field,
               Ppx_yojson_conv_lib.( ! ) inlineCompletion_field,
               Ppx_yojson_conv_lib.( ! ) inlineValue_field,
               Ppx_yojson_conv_lib.( ! ) linkedEditingRange_field,
               Ppx_yojson_conv_lib.( ! ) moniker_field,
               Ppx_yojson_conv_lib.( ! ) onTypeFormatting_field,
               Ppx_yojson_conv_lib.( ! ) publishDiagnostics_field,
               Ppx_yojson_conv_lib.( ! ) rangeFormatting_field,
               Ppx_yojson_conv_lib.( ! ) references_field,
               Ppx_yojson_conv_lib.( ! ) rename_field,
               Ppx_yojson_conv_lib.( ! ) selectionRange_field,
               Ppx_yojson_conv_lib.( ! ) semanticTokens_field,
               Ppx_yojson_conv_lib.( ! ) signatureHelp_field,
               Ppx_yojson_conv_lib.( ! ) synchronization_field,
               Ppx_yojson_conv_lib.( ! ) typeDefinition_field,
               Ppx_yojson_conv_lib.( ! ) typeHierarchy_field )
           in
           { callHierarchy =
               (match callHierarchy_value with
               | Option.None -> None
               | Option.Some v -> v);
             codeAction =
               (match codeAction_value with
               | Option.None -> None
               | Option.Some v -> v);
             codeLens =
               (match codeLens_value with
               | Option.None -> None
               | Option.Some v -> v);
             colorProvider =
               (match colorProvider_value with
               | Option.None -> None
               | Option.Some v -> v);
             completion =
               (match completion_value with
               | Option.None -> None
               | Option.Some v -> v);
             declaration =
               (match declaration_value with
               | Option.None -> None
               | Option.Some v -> v);
             definition =
               (match definition_value with
               | Option.None -> None
               | Option.Some v -> v);
             diagnostic =
               (match diagnostic_value with
               | Option.None -> None
               | Option.Some v -> v);
             documentHighlight =
               (match documentHighlight_value with
               | Option.None -> None
               | Option.Some v -> v);
             documentLink =
               (match documentLink_value with
               | Option.None -> None
               | Option.Some v -> v);
             documentSymbol =
               (match documentSymbol_value with
               | Option.None -> None
               | Option.Some v -> v);
             foldingRange =
               (match foldingRange_value with
               | Option.None -> None
               | Option.Some v -> v);
             formatting =
               (match formatting_value with
               | Option.None -> None
               | Option.Some v -> v);
             hover =
               (match hover_value with
               | Option.None -> None
               | Option.Some v -> v);
             implementation =
               (match implementation_value with
               | Option.None -> None
               | Option.Some v -> v);
             inlayHint =
               (match inlayHint_value with
               | Option.None -> None
               | Option.Some v -> v);
             inlineCompletion =
               (match inlineCompletion_value with
               | Option.None -> None
               | Option.Some v -> v);
             inlineValue =
               (match inlineValue_value with
               | Option.None -> None
               | Option.Some v -> v);
             linkedEditingRange =
               (match linkedEditingRange_value with
               | Option.None -> None
               | Option.Some v -> v);
             moniker =
               (match moniker_value with
               | Option.None -> None
               | Option.Some v -> v);
             onTypeFormatting =
               (match onTypeFormatting_value with
               | Option.None -> None
               | Option.Some v -> v);
             publishDiagnostics =
               (match publishDiagnostics_value with
               | Option.None -> None
               | Option.Some v -> v);
             rangeFormatting =
               (match rangeFormatting_value with
               | Option.None -> None
               | Option.Some v -> v);
             references =
               (match references_value with
               | Option.None -> None
               | Option.Some v -> v);
             rename =
               (match rename_value with
               | Option.None -> None
               | Option.Some v -> v);
             selectionRange =
               (match selectionRange_value with
               | Option.None -> None
               | Option.Some v -> v);
             semanticTokens =
               (match semanticTokens_value with
               | Option.None -> None
               | Option.Some v -> v);
             signatureHelp =
               (match signatureHelp_value with
               | Option.None -> None
               | Option.Some v -> v);
             synchronization =
               (match synchronization_value with
               | Option.None -> None
               | Option.Some v -> v);
             typeDefinition =
               (match typeDefinition_value with
               | Option.None -> None
               | Option.Some v -> v);
             typeHierarchy =
               (match typeHierarchy_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { callHierarchy = v_callHierarchy;
         codeAction = v_codeAction;
         codeLens = v_codeLens;
         colorProvider = v_colorProvider;
         completion = v_completion;
         declaration = v_declaration;
         definition = v_definition;
         diagnostic = v_diagnostic;
         documentHighlight = v_documentHighlight;
         documentLink = v_documentLink;
         documentSymbol = v_documentSymbol;
         foldingRange = v_foldingRange;
         formatting = v_formatting;
         hover = v_hover;
         implementation = v_implementation;
         inlayHint = v_inlayHint;
         inlineCompletion = v_inlineCompletion;
         inlineValue = v_inlineValue;
         linkedEditingRange = v_linkedEditingRange;
         moniker = v_moniker;
         onTypeFormatting = v_onTypeFormatting;
         publishDiagnostics = v_publishDiagnostics;
         rangeFormatting = v_rangeFormatting;
         references = v_references;
         rename = v_rename;
         selectionRange = v_selectionRange;
         semanticTokens = v_semanticTokens;
         signatureHelp = v_signatureHelp;
         synchronization = v_synchronization;
         typeDefinition = v_typeDefinition;
         typeHierarchy = v_typeHierarchy
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_typeHierarchy then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                TypeHierarchyClientCapabilities.yojson_of_t)
               v_typeHierarchy
           in
           let bnd = ("typeHierarchy", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_typeDefinition then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                TypeDefinitionClientCapabilities.yojson_of_t)
               v_typeDefinition
           in
           let bnd = ("typeDefinition", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_synchronization then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                TextDocumentSyncClientCapabilities.yojson_of_t)
               v_synchronization
           in
           let bnd = ("synchronization", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_signatureHelp then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                SignatureHelpClientCapabilities.yojson_of_t)
               v_signatureHelp
           in
           let bnd = ("signatureHelp", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_semanticTokens then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                SemanticTokensClientCapabilities.yojson_of_t)
               v_semanticTokens
           in
           let bnd = ("semanticTokens", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_selectionRange then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                SelectionRangeClientCapabilities.yojson_of_t)
               v_selectionRange
           in
           let bnd = ("selectionRange", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_rename then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                RenameClientCapabilities.yojson_of_t)
               v_rename
           in
           let bnd = ("rename", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_references then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                ReferenceClientCapabilities.yojson_of_t)
               v_references
           in
           let bnd = ("references", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_rangeFormatting then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                DocumentRangeFormattingClientCapabilities.yojson_of_t)
               v_rangeFormatting
           in
           let bnd = ("rangeFormatting", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_publishDiagnostics then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                PublishDiagnosticsClientCapabilities.yojson_of_t)
               v_publishDiagnostics
           in
           let bnd = ("publishDiagnostics", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_onTypeFormatting then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                DocumentOnTypeFormattingClientCapabilities.yojson_of_t)
               v_onTypeFormatting
           in
           let bnd = ("onTypeFormatting", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_moniker then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                MonikerClientCapabilities.yojson_of_t)
               v_moniker
           in
           let bnd = ("moniker", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_linkedEditingRange then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                LinkedEditingRangeClientCapabilities.yojson_of_t)
               v_linkedEditingRange
           in
           let bnd = ("linkedEditingRange", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_inlineValue then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                InlineValueClientCapabilities.yojson_of_t)
               v_inlineValue
           in
           let bnd = ("inlineValue", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_inlineCompletion then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                InlineCompletionClientCapabilities.yojson_of_t)
               v_inlineCompletion
           in
           let bnd = ("inlineCompletion", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_inlayHint then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                InlayHintClientCapabilities.yojson_of_t)
               v_inlayHint
           in
           let bnd = ("inlayHint", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_implementation then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                ImplementationClientCapabilities.yojson_of_t)
               v_implementation
           in
           let bnd = ("implementation", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_hover then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                HoverClientCapabilities.yojson_of_t)
               v_hover
           in
           let bnd = ("hover", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_formatting then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                DocumentFormattingClientCapabilities.yojson_of_t)
               v_formatting
           in
           let bnd = ("formatting", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_foldingRange then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                FoldingRangeClientCapabilities.yojson_of_t)
               v_foldingRange
           in
           let bnd = ("foldingRange", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentSymbol then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                DocumentSymbolClientCapabilities.yojson_of_t)
               v_documentSymbol
           in
           let bnd = ("documentSymbol", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentLink then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                DocumentLinkClientCapabilities.yojson_of_t)
               v_documentLink
           in
           let bnd = ("documentLink", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentHighlight then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                DocumentHighlightClientCapabilities.yojson_of_t)
               v_documentHighlight
           in
           let bnd = ("documentHighlight", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_diagnostic then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                DiagnosticClientCapabilities.yojson_of_t)
               v_diagnostic
           in
           let bnd = ("diagnostic", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_definition then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                DefinitionClientCapabilities.yojson_of_t)
               v_definition
           in
           let bnd = ("definition", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_declaration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                DeclarationClientCapabilities.yojson_of_t)
               v_declaration
           in
           let bnd = ("declaration", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_completion then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                CompletionClientCapabilities.yojson_of_t)
               v_completion
           in
           let bnd = ("completion", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_colorProvider then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                DocumentColorClientCapabilities.yojson_of_t)
               v_colorProvider
           in
           let bnd = ("colorProvider", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_codeLens then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                CodeLensClientCapabilities.yojson_of_t)
               v_codeLens
           in
           let bnd = ("codeLens", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_codeAction then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                CodeActionClientCapabilities.yojson_of_t)
               v_codeAction
           in
           let bnd = ("codeAction", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_callHierarchy then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                CallHierarchyClientCapabilities.yojson_of_t)
               v_callHierarchy
           in
           let bnd = ("callHierarchy", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(callHierarchy : CallHierarchyClientCapabilities.t option)
      ?(codeAction : CodeActionClientCapabilities.t option)
      ?(codeLens : CodeLensClientCapabilities.t option)
      ?(colorProvider : DocumentColorClientCapabilities.t option)
      ?(completion : CompletionClientCapabilities.t option)
      ?(declaration : DeclarationClientCapabilities.t option)
      ?(definition : DefinitionClientCapabilities.t option)
      ?(diagnostic : DiagnosticClientCapabilities.t option)
      ?(documentHighlight : DocumentHighlightClientCapabilities.t option)
      ?(documentLink : DocumentLinkClientCapabilities.t option)
      ?(documentSymbol : DocumentSymbolClientCapabilities.t option)
      ?(foldingRange : FoldingRangeClientCapabilities.t option)
      ?(formatting : DocumentFormattingClientCapabilities.t option)
      ?(hover : HoverClientCapabilities.t option)
      ?(implementation : ImplementationClientCapabilities.t option)
      ?(inlayHint : InlayHintClientCapabilities.t option)
      ?(inlineCompletion : InlineCompletionClientCapabilities.t option)
      ?(inlineValue : InlineValueClientCapabilities.t option)
      ?(linkedEditingRange : LinkedEditingRangeClientCapabilities.t option)
      ?(moniker : MonikerClientCapabilities.t option)
      ?(onTypeFormatting : DocumentOnTypeFormattingClientCapabilities.t option)
      ?(publishDiagnostics : PublishDiagnosticsClientCapabilities.t option)
      ?(rangeFormatting : DocumentRangeFormattingClientCapabilities.t option)
      ?(references : ReferenceClientCapabilities.t option)
      ?(rename : RenameClientCapabilities.t option)
      ?(selectionRange : SelectionRangeClientCapabilities.t option)
      ?(semanticTokens : SemanticTokensClientCapabilities.t option)
      ?(signatureHelp : SignatureHelpClientCapabilities.t option)
      ?(synchronization : TextDocumentSyncClientCapabilities.t option)
      ?(typeDefinition : TypeDefinitionClientCapabilities.t option)
      ?(typeHierarchy : TypeHierarchyClientCapabilities.t option) (() : unit) :
      t =
    { callHierarchy;
      codeAction;
      codeLens;
      colorProvider;
      completion;
      declaration;
      definition;
      diagnostic;
      documentHighlight;
      documentLink;
      documentSymbol;
      foldingRange;
      formatting;
      hover;
      implementation;
      inlayHint;
      inlineCompletion;
      inlineValue;
      linkedEditingRange;
      moniker;
      onTypeFormatting;
      publishDiagnostics;
      rangeFormatting;
      references;
      rename;
      selectionRange;
      semanticTokens;
      signatureHelp;
      synchronization;
      typeDefinition;
      typeHierarchy
    }
end

module NotebookDocumentSyncClientCapabilities = struct
  type t =
    { dynamicRegistration : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      executionSummarySupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.NotebookDocumentSyncClientCapabilities.t"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let dynamicRegistration_field = ref Option.None
       and executionSummarySupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "dynamicRegistration" -> (
             match Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               dynamicRegistration_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "executionSummarySupport" -> (
             match Ppx_yojson_conv_lib.( ! ) executionSummarySupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               executionSummarySupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let dynamicRegistration_value, executionSummarySupport_value =
             ( Ppx_yojson_conv_lib.( ! ) dynamicRegistration_field,
               Ppx_yojson_conv_lib.( ! ) executionSummarySupport_field )
           in
           { dynamicRegistration =
               (match dynamicRegistration_value with
               | Option.None -> None
               | Option.Some v -> v);
             executionSummarySupport =
               (match executionSummarySupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { dynamicRegistration = v_dynamicRegistration;
         executionSummarySupport = v_executionSummarySupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_executionSummarySupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_executionSummarySupport
           in
           let bnd = ("executionSummarySupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_dynamicRegistration then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_dynamicRegistration
           in
           let bnd = ("dynamicRegistration", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(dynamicRegistration : bool option)
      ?(executionSummarySupport : bool option) (() : unit) : t =
    { dynamicRegistration; executionSummarySupport }
end

module NotebookDocumentClientCapabilities = struct
  type t = { synchronization : NotebookDocumentSyncClientCapabilities.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.NotebookDocumentClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let synchronization_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "synchronization" -> (
             match Ppx_yojson_conv_lib.( ! ) synchronization_field with
             | Option.None ->
               let fvalue =
                 NotebookDocumentSyncClientCapabilities.t_of_yojson
                   _field_yojson
               in
               synchronization_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) synchronization_field with
           | Option.Some synchronization_value ->
             { synchronization = synchronization_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) synchronization_field)
                     Option.None,
                   "synchronization" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { synchronization = v_synchronization } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg =
           NotebookDocumentSyncClientCapabilities.yojson_of_t v_synchronization
         in
         ("synchronization", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(synchronization : NotebookDocumentSyncClientCapabilities.t) : t =
    { synchronization }
end

module RegularExpressionsClientCapabilities = struct
  type t =
    { engine : string;
      version : string Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.RegularExpressionsClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let engine_field = ref Option.None
       and version_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "engine" -> (
             match Ppx_yojson_conv_lib.( ! ) engine_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               engine_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "version" -> (
             match Ppx_yojson_conv_lib.( ! ) version_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               version_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) engine_field,
               Ppx_yojson_conv_lib.( ! ) version_field )
           with
           | Option.Some engine_value, version_value ->
             { engine = engine_value;
               version =
                 (match version_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) engine_field)
                     Option.None,
                   "engine" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { engine = v_engine; version = v_version } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_version then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_version
           in
           let bnd = ("version", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_engine in
         ("engine", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(engine : string) ?(version : string option) (() : unit) : t =
    { engine; version }
end

module MarkdownClientCapabilities = struct
  type t =
    { allowedTags : string list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      parser : string;
      version : string Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.MarkdownClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let allowedTags_field = ref Option.None
       and parser_field = ref Option.None
       and version_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "allowedTags" -> (
             match Ppx_yojson_conv_lib.( ! ) allowedTags_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson string_of_yojson)
                   _field_yojson
               in
               allowedTags_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "parser" -> (
             match Ppx_yojson_conv_lib.( ! ) parser_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               parser_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "version" -> (
             match Ppx_yojson_conv_lib.( ! ) version_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               version_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) allowedTags_field,
               Ppx_yojson_conv_lib.( ! ) parser_field,
               Ppx_yojson_conv_lib.( ! ) version_field )
           with
           | allowedTags_value, Option.Some parser_value, version_value ->
             { allowedTags =
                 (match allowedTags_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               parser = parser_value;
               version =
                 (match version_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) parser_field)
                     Option.None,
                   "parser" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { allowedTags = v_allowedTags; parser = v_parser; version = v_version }
       ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_version then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_version
           in
           let bnd = ("version", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_parser in
         ("parser", arg) :: bnds
       in
       let bnds =
         if None = v_allowedTags then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list yojson_of_string))
               v_allowedTags
           in
           let bnd = ("allowedTags", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(allowedTags : string list option) ~(parser : string)
      ?(version : string option) (() : unit) : t =
    { allowedTags; parser; version }
end

module GeneralClientCapabilities = struct
  type staleRequestSupport =
    { cancel : bool; retryOnContentModified : string list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : staleRequestSupport) -> ()

  let staleRequestSupport_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.GeneralClientCapabilities.staleRequestSupport"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let cancel_field = ref Option.None
       and retryOnContentModified_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "cancel" -> (
             match Ppx_yojson_conv_lib.( ! ) cancel_field with
             | Option.None ->
               let fvalue = bool_of_yojson _field_yojson in
               cancel_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "retryOnContentModified" -> (
             match Ppx_yojson_conv_lib.( ! ) retryOnContentModified_field with
             | Option.None ->
               let fvalue = list_of_yojson string_of_yojson _field_yojson in
               retryOnContentModified_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) cancel_field,
               Ppx_yojson_conv_lib.( ! ) retryOnContentModified_field )
           with
           | Option.Some cancel_value, Option.Some retryOnContentModified_value
             ->
             { cancel = cancel_value;
               retryOnContentModified = retryOnContentModified_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) cancel_field)
                     Option.None,
                   "cancel" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) retryOnContentModified_field)
                     Option.None,
                   "retryOnContentModified" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> staleRequestSupport)

  let _ = staleRequestSupport_of_yojson

  let yojson_of_staleRequestSupport =
    (function
     | { cancel = v_cancel; retryOnContentModified = v_retryOnContentModified }
       ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list yojson_of_string v_retryOnContentModified in
         ("retryOnContentModified", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_bool v_cancel in
         ("cancel", arg) :: bnds
       in
       `Assoc bnds
      : staleRequestSupport -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_staleRequestSupport

  [@@@end]

  let create_staleRequestSupport ~(cancel : bool)
      ~(retryOnContentModified : string list) : staleRequestSupport =
    { cancel; retryOnContentModified }

  type t =
    { markdown : MarkdownClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      positionEncodings : PositionEncodingKind.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      regularExpressions :
        RegularExpressionsClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      staleRequestSupport : staleRequestSupport Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.GeneralClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let markdown_field = ref Option.None
       and positionEncodings_field = ref Option.None
       and regularExpressions_field = ref Option.None
       and staleRequestSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "markdown" -> (
             match Ppx_yojson_conv_lib.( ! ) markdown_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   MarkdownClientCapabilities.t_of_yojson _field_yojson
               in
               markdown_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "positionEncodings" -> (
             match Ppx_yojson_conv_lib.( ! ) positionEncodings_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson PositionEncodingKind.t_of_yojson)
                   _field_yojson
               in
               positionEncodings_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "regularExpressions" -> (
             match Ppx_yojson_conv_lib.( ! ) regularExpressions_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   RegularExpressionsClientCapabilities.t_of_yojson
                   _field_yojson
               in
               regularExpressions_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "staleRequestSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) staleRequestSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson staleRequestSupport_of_yojson
                   _field_yojson
               in
               staleRequestSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( markdown_value,
                 positionEncodings_value,
                 regularExpressions_value,
                 staleRequestSupport_value ) =
             ( Ppx_yojson_conv_lib.( ! ) markdown_field,
               Ppx_yojson_conv_lib.( ! ) positionEncodings_field,
               Ppx_yojson_conv_lib.( ! ) regularExpressions_field,
               Ppx_yojson_conv_lib.( ! ) staleRequestSupport_field )
           in
           { markdown =
               (match markdown_value with
               | Option.None -> None
               | Option.Some v -> v);
             positionEncodings =
               (match positionEncodings_value with
               | Option.None -> None
               | Option.Some v -> v);
             regularExpressions =
               (match regularExpressions_value with
               | Option.None -> None
               | Option.Some v -> v);
             staleRequestSupport =
               (match staleRequestSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { markdown = v_markdown;
         positionEncodings = v_positionEncodings;
         regularExpressions = v_regularExpressions;
         staleRequestSupport = v_staleRequestSupport
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_staleRequestSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_staleRequestSupport)
               v_staleRequestSupport
           in
           let bnd = ("staleRequestSupport", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_regularExpressions then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                RegularExpressionsClientCapabilities.yojson_of_t)
               v_regularExpressions
           in
           let bnd = ("regularExpressions", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_positionEncodings then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list PositionEncodingKind.yojson_of_t))
               v_positionEncodings
           in
           let bnd = ("positionEncodings", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_markdown then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                MarkdownClientCapabilities.yojson_of_t)
               v_markdown
           in
           let bnd = ("markdown", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(markdown : MarkdownClientCapabilities.t option)
      ?(positionEncodings : PositionEncodingKind.t list option)
      ?(regularExpressions : RegularExpressionsClientCapabilities.t option)
      ?(staleRequestSupport : staleRequestSupport option) (() : unit) : t =
    { markdown; positionEncodings; regularExpressions; staleRequestSupport }
end

module ClientCapabilities = struct
  type t =
    { experimental : Json.t option; [@yojson.option]
      general : GeneralClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      notebookDocument :
        NotebookDocumentClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      textDocument : TextDocumentClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      window : WindowClientCapabilities.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workspace : WorkspaceClientCapabilities.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ClientCapabilities.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let experimental_field = ref Option.None
       and general_field = ref Option.None
       and notebookDocument_field = ref Option.None
       and textDocument_field = ref Option.None
       and window_field = ref Option.None
       and workspace_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "experimental" -> (
             match Ppx_yojson_conv_lib.( ! ) experimental_field with
             | Option.None ->
               let fvalue = Json.t_of_yojson _field_yojson in
               experimental_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "general" -> (
             match Ppx_yojson_conv_lib.( ! ) general_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   GeneralClientCapabilities.t_of_yojson _field_yojson
               in
               general_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "notebookDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) notebookDocument_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   NotebookDocumentClientCapabilities.t_of_yojson _field_yojson
               in
               notebookDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   TextDocumentClientCapabilities.t_of_yojson _field_yojson
               in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "window" -> (
             match Ppx_yojson_conv_lib.( ! ) window_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   WindowClientCapabilities.t_of_yojson _field_yojson
               in
               window_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workspace" -> (
             match Ppx_yojson_conv_lib.( ! ) workspace_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   WorkspaceClientCapabilities.t_of_yojson _field_yojson
               in
               workspace_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( experimental_value,
                 general_value,
                 notebookDocument_value,
                 textDocument_value,
                 window_value,
                 workspace_value ) =
             ( Ppx_yojson_conv_lib.( ! ) experimental_field,
               Ppx_yojson_conv_lib.( ! ) general_field,
               Ppx_yojson_conv_lib.( ! ) notebookDocument_field,
               Ppx_yojson_conv_lib.( ! ) textDocument_field,
               Ppx_yojson_conv_lib.( ! ) window_field,
               Ppx_yojson_conv_lib.( ! ) workspace_field )
           in
           { experimental = experimental_value;
             general =
               (match general_value with
               | Option.None -> None
               | Option.Some v -> v);
             notebookDocument =
               (match notebookDocument_value with
               | Option.None -> None
               | Option.Some v -> v);
             textDocument =
               (match textDocument_value with
               | Option.None -> None
               | Option.Some v -> v);
             window =
               (match window_value with
               | Option.None -> None
               | Option.Some v -> v);
             workspace =
               (match workspace_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { experimental = v_experimental;
         general = v_general;
         notebookDocument = v_notebookDocument;
         textDocument = v_textDocument;
         window = v_window;
         workspace = v_workspace
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workspace then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                WorkspaceClientCapabilities.yojson_of_t)
               v_workspace
           in
           let bnd = ("workspace", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_window then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                WindowClientCapabilities.yojson_of_t)
               v_window
           in
           let bnd = ("window", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_textDocument then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                TextDocumentClientCapabilities.yojson_of_t)
               v_textDocument
           in
           let bnd = ("textDocument", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_notebookDocument then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                NotebookDocumentClientCapabilities.yojson_of_t)
               v_notebookDocument
           in
           let bnd = ("notebookDocument", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_general then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                GeneralClientCapabilities.yojson_of_t)
               v_general
           in
           let bnd = ("general", arg) in
           bnd :: bnds
       in
       let bnds =
         match v_experimental with
         | Option.None -> bnds
         | Option.Some v ->
           let arg = Json.yojson_of_t v in
           let bnd = ("experimental", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(experimental : Json.t option)
      ?(general : GeneralClientCapabilities.t option)
      ?(notebookDocument : NotebookDocumentClientCapabilities.t option)
      ?(textDocument : TextDocumentClientCapabilities.t option)
      ?(window : WindowClientCapabilities.t option)
      ?(workspace : WorkspaceClientCapabilities.t option) (() : unit) : t =
    { experimental; general; notebookDocument; textDocument; window; workspace }
end

module Location = struct
  type t = { range : Range.t; uri : DocumentUri.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.Location.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let range_field = ref Option.None
       and uri_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "range" -> (
             match Ppx_yojson_conv_lib.( ! ) range_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               range_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) range_field,
               Ppx_yojson_conv_lib.( ! ) uri_field )
           with
           | Option.Some range_value, Option.Some uri_value ->
             { range = range_value; uri = uri_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) range_field)
                     Option.None,
                   "range" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { range = v_range; uri = v_uri } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_uri in
         ("uri", arg) :: bnds
       in
       let bnds =
         let arg = Range.yojson_of_t v_range in
         ("range", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(range : Range.t) ~(uri : DocumentUri.t) : t = { range; uri }
end

module DiagnosticRelatedInformation = struct
  type t = { location : Location.t; message : string }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DiagnosticRelatedInformation.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let location_field = ref Option.None
       and message_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "location" -> (
             match Ppx_yojson_conv_lib.( ! ) location_field with
             | Option.None ->
               let fvalue = Location.t_of_yojson _field_yojson in
               location_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "message" -> (
             match Ppx_yojson_conv_lib.( ! ) message_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               message_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) location_field,
               Ppx_yojson_conv_lib.( ! ) message_field )
           with
           | Option.Some location_value, Option.Some message_value ->
             { location = location_value; message = message_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) location_field)
                     Option.None,
                   "location" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) message_field)
                     Option.None,
                   "message" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { location = v_location; message = v_message } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_string v_message in
         ("message", arg) :: bnds
       in
       let bnds =
         let arg = Location.yojson_of_t v_location in
         ("location", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(location : Location.t) ~(message : string) : t =
    { location; message }
end

module MarkupContent = struct
  type t = { kind : MarkupKind.t; value : string }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.MarkupContent.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let kind_field = ref Option.None
       and value_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "kind" -> (
             match Ppx_yojson_conv_lib.( ! ) kind_field with
             | Option.None ->
               let fvalue = MarkupKind.t_of_yojson _field_yojson in
               kind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "value" -> (
             match Ppx_yojson_conv_lib.( ! ) value_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               value_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) kind_field,
               Ppx_yojson_conv_lib.( ! ) value_field )
           with
           | Option.Some kind_value, Option.Some value_value ->
             { kind = kind_value; value = value_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) kind_field)
                     Option.None,
                   "kind" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) value_field)
                     Option.None,
                   "value" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { kind = v_kind; value = v_value } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_string v_value in
         ("value", arg) :: bnds
       in
       let bnds =
         let arg = MarkupKind.yojson_of_t v_kind in
         ("kind", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(kind : MarkupKind.t) ~(value : string) : t = { kind; value }
end

module CodeDescription = struct
  type t = { href : DocumentUri.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeDescription.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let href_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "href" -> (
             match Ppx_yojson_conv_lib.( ! ) href_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               href_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) href_field with
           | Option.Some href_value -> { href = href_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) href_field)
                     Option.None,
                   "href" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { href = v_href } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_href in
         ("href", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(href : DocumentUri.t) : t = { href }
end

module Diagnostic = struct
  type message_pvar = [ `String of string | `MarkupContent of MarkupContent.t ]

  let message_pvar_of_yojson (json : Json.t) : message_pvar =
    match json with
    | `String j -> `String j
    | _ ->
      Json.Of.untagged_union "message_pvar"
        [ (fun json -> `MarkupContent (MarkupContent.t_of_yojson json)) ]
        json

  let yojson_of_message_pvar (message_pvar : message_pvar) : Json.t =
    match message_pvar with
    | `String j -> `String j
    | `MarkupContent s -> MarkupContent.yojson_of_t s

  type t =
    { code : Jsonrpc.Id.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      codeDescription : CodeDescription.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      data : Json.t option; [@yojson.option]
      message : message_pvar;
      range : Range.t;
      relatedInformation :
        DiagnosticRelatedInformation.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      severity : DiagnosticSeverity.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      source : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      tags : DiagnosticTag.t list Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.Diagnostic.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let code_field = ref Option.None
       and codeDescription_field = ref Option.None
       and data_field = ref Option.None
       and message_field = ref Option.None
       and range_field = ref Option.None
       and relatedInformation_field = ref Option.None
       and severity_field = ref Option.None
       and source_field = ref Option.None
       and tags_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "code" -> (
             match Ppx_yojson_conv_lib.( ! ) code_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson Jsonrpc.Id.t_of_yojson
                   _field_yojson
               in
               code_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "codeDescription" -> (
             match Ppx_yojson_conv_lib.( ! ) codeDescription_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson CodeDescription.t_of_yojson
                   _field_yojson
               in
               codeDescription_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "data" -> (
             match Ppx_yojson_conv_lib.( ! ) data_field with
             | Option.None ->
               let fvalue = Json.t_of_yojson _field_yojson in
               data_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "message" -> (
             match Ppx_yojson_conv_lib.( ! ) message_field with
             | Option.None ->
               let fvalue = message_pvar_of_yojson _field_yojson in
               message_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "range" -> (
             match Ppx_yojson_conv_lib.( ! ) range_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               range_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "relatedInformation" -> (
             match Ppx_yojson_conv_lib.( ! ) relatedInformation_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson DiagnosticRelatedInformation.t_of_yojson)
                   _field_yojson
               in
               relatedInformation_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "severity" -> (
             match Ppx_yojson_conv_lib.( ! ) severity_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson DiagnosticSeverity.t_of_yojson
                   _field_yojson
               in
               severity_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "source" -> (
             match Ppx_yojson_conv_lib.( ! ) source_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               source_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "tags" -> (
             match Ppx_yojson_conv_lib.( ! ) tags_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson DiagnosticTag.t_of_yojson)
                   _field_yojson
               in
               tags_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) code_field,
               Ppx_yojson_conv_lib.( ! ) codeDescription_field,
               Ppx_yojson_conv_lib.( ! ) data_field,
               Ppx_yojson_conv_lib.( ! ) message_field,
               Ppx_yojson_conv_lib.( ! ) range_field,
               Ppx_yojson_conv_lib.( ! ) relatedInformation_field,
               Ppx_yojson_conv_lib.( ! ) severity_field,
               Ppx_yojson_conv_lib.( ! ) source_field,
               Ppx_yojson_conv_lib.( ! ) tags_field )
           with
           | ( code_value,
               codeDescription_value,
               data_value,
               Option.Some message_value,
               Option.Some range_value,
               relatedInformation_value,
               severity_value,
               source_value,
               tags_value ) ->
             { code =
                 (match code_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               codeDescription =
                 (match codeDescription_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               data = data_value;
               message = message_value;
               range = range_value;
               relatedInformation =
                 (match relatedInformation_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               severity =
                 (match severity_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               source =
                 (match source_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               tags =
                 (match tags_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) message_field)
                     Option.None,
                   "message" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) range_field)
                     Option.None,
                   "range" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { code = v_code;
         codeDescription = v_codeDescription;
         data = v_data;
         message = v_message;
         range = v_range;
         relatedInformation = v_relatedInformation;
         severity = v_severity;
         source = v_source;
         tags = v_tags
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_tags then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list DiagnosticTag.yojson_of_t))
               v_tags
           in
           let bnd = ("tags", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_source then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_source
           in
           let bnd = ("source", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_severity then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t DiagnosticSeverity.yojson_of_t)
               v_severity
           in
           let bnd = ("severity", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_relatedInformation then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list DiagnosticRelatedInformation.yojson_of_t))
               v_relatedInformation
           in
           let bnd = ("relatedInformation", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = Range.yojson_of_t v_range in
         ("range", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_message_pvar v_message in
         ("message", arg) :: bnds
       in
       let bnds =
         match v_data with
         | Option.None -> bnds
         | Option.Some v ->
           let arg = Json.yojson_of_t v in
           let bnd = ("data", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_codeDescription then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t CodeDescription.yojson_of_t)
               v_codeDescription
           in
           let bnd = ("codeDescription", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_code then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t Jsonrpc.Id.yojson_of_t) v_code
           in
           let bnd = ("code", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(code : Jsonrpc.Id.t option)
      ?(codeDescription : CodeDescription.t option) ?(data : Json.t option)
      ~(message : message_pvar) ~(range : Range.t)
      ?(relatedInformation : DiagnosticRelatedInformation.t list option)
      ?(severity : DiagnosticSeverity.t option) ?(source : string option)
      ?(tags : DiagnosticTag.t list option) (() : unit) : t =
    { code;
      codeDescription;
      data;
      message;
      range;
      relatedInformation;
      severity;
      source;
      tags
    }
end

module Command = struct
  type t =
    { arguments : Json.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      command : string;
      title : string
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.Command.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let arguments_field = ref Option.None
       and command_field = ref Option.None
       and title_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "arguments" -> (
             match Ppx_yojson_conv_lib.( ! ) arguments_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson Json.t_of_yojson)
                   _field_yojson
               in
               arguments_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "command" -> (
             match Ppx_yojson_conv_lib.( ! ) command_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               command_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "title" -> (
             match Ppx_yojson_conv_lib.( ! ) title_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               title_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) arguments_field,
               Ppx_yojson_conv_lib.( ! ) command_field,
               Ppx_yojson_conv_lib.( ! ) title_field )
           with
           | arguments_value, Option.Some command_value, Option.Some title_value
             ->
             { arguments =
                 (match arguments_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               command = command_value;
               title = title_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) command_field)
                     Option.None,
                   "command" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) title_field)
                     Option.None,
                   "title" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { arguments = v_arguments; command = v_command; title = v_title } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_string v_title in
         ("title", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_command in
         ("command", arg) :: bnds
       in
       let bnds =
         if None = v_arguments then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list Json.yojson_of_t))
               v_arguments
           in
           let bnd = ("arguments", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(arguments : Json.t list option) ~(command : string)
      ~(title : string) (() : unit) : t =
    { arguments; command; title }
end

module CodeAction = struct
  type disabled = { reason : string }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : disabled) -> ()

  let disabled_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeAction.disabled" in
     function
     | `Assoc field_yojsons as yojson -> (
       let reason_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "reason" -> (
             match Ppx_yojson_conv_lib.( ! ) reason_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               reason_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) reason_field with
           | Option.Some reason_value -> { reason = reason_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) reason_field)
                     Option.None,
                   "reason" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> disabled)

  let _ = disabled_of_yojson

  let yojson_of_disabled =
    (function
     | { reason = v_reason } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_string v_reason in
         ("reason", arg) :: bnds
       in
       `Assoc bnds
      : disabled -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_disabled

  [@@@end]

  let create_disabled ~(reason : string) : disabled = { reason }

  type t =
    { command : Command.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      data : Json.t option; [@yojson.option]
      diagnostics : Diagnostic.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      disabled : disabled Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      edit : WorkspaceEdit.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      isPreferred : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      kind : CodeActionKind.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      title : string
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeAction.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let command_field = ref Option.None
       and data_field = ref Option.None
       and diagnostics_field = ref Option.None
       and disabled_field = ref Option.None
       and edit_field = ref Option.None
       and isPreferred_field = ref Option.None
       and kind_field = ref Option.None
       and title_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "command" -> (
             match Ppx_yojson_conv_lib.( ! ) command_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson Command.t_of_yojson
                   _field_yojson
               in
               command_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "data" -> (
             match Ppx_yojson_conv_lib.( ! ) data_field with
             | Option.None ->
               let fvalue = Json.t_of_yojson _field_yojson in
               data_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "diagnostics" -> (
             match Ppx_yojson_conv_lib.( ! ) diagnostics_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson Diagnostic.t_of_yojson)
                   _field_yojson
               in
               diagnostics_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "disabled" -> (
             match Ppx_yojson_conv_lib.( ! ) disabled_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson disabled_of_yojson
                   _field_yojson
               in
               disabled_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "edit" -> (
             match Ppx_yojson_conv_lib.( ! ) edit_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson WorkspaceEdit.t_of_yojson
                   _field_yojson
               in
               edit_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "isPreferred" -> (
             match Ppx_yojson_conv_lib.( ! ) isPreferred_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               isPreferred_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "kind" -> (
             match Ppx_yojson_conv_lib.( ! ) kind_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson CodeActionKind.t_of_yojson
                   _field_yojson
               in
               kind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "title" -> (
             match Ppx_yojson_conv_lib.( ! ) title_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               title_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) command_field,
               Ppx_yojson_conv_lib.( ! ) data_field,
               Ppx_yojson_conv_lib.( ! ) diagnostics_field,
               Ppx_yojson_conv_lib.( ! ) disabled_field,
               Ppx_yojson_conv_lib.( ! ) edit_field,
               Ppx_yojson_conv_lib.( ! ) isPreferred_field,
               Ppx_yojson_conv_lib.( ! ) kind_field,
               Ppx_yojson_conv_lib.( ! ) title_field )
           with
           | ( command_value,
               data_value,
               diagnostics_value,
               disabled_value,
               edit_value,
               isPreferred_value,
               kind_value,
               Option.Some title_value ) ->
             { command =
                 (match command_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               data = data_value;
               diagnostics =
                 (match diagnostics_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               disabled =
                 (match disabled_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               edit =
                 (match edit_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               isPreferred =
                 (match isPreferred_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               kind =
                 (match kind_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               title = title_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) title_field)
                     Option.None,
                   "title" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { command = v_command;
         data = v_data;
         diagnostics = v_diagnostics;
         disabled = v_disabled;
         edit = v_edit;
         isPreferred = v_isPreferred;
         kind = v_kind;
         title = v_title
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_string v_title in
         ("title", arg) :: bnds
       in
       let bnds =
         if None = v_kind then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t CodeActionKind.yojson_of_t)
               v_kind
           in
           let bnd = ("kind", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_isPreferred then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_isPreferred
           in
           let bnd = ("isPreferred", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_edit then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t WorkspaceEdit.yojson_of_t) v_edit
           in
           let bnd = ("edit", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_disabled then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_disabled) v_disabled
           in
           let bnd = ("disabled", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_diagnostics then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list Diagnostic.yojson_of_t))
               v_diagnostics
           in
           let bnd = ("diagnostics", arg) in
           bnd :: bnds
       in
       let bnds =
         match v_data with
         | Option.None -> bnds
         | Option.Some v ->
           let arg = Json.yojson_of_t v in
           let bnd = ("data", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_command then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t Command.yojson_of_t) v_command
           in
           let bnd = ("command", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(command : Command.t option) ?(data : Json.t option)
      ?(diagnostics : Diagnostic.t list option) ?(disabled : disabled option)
      ?(edit : WorkspaceEdit.t option) ?(isPreferred : bool option)
      ?(kind : CodeActionKind.t option) ~(title : string) (() : unit) : t =
    { command; data; diagnostics; disabled; edit; isPreferred; kind; title }
end

module CodeActionContext = struct
  type t =
    { diagnostics : Diagnostic.t list;
      only : CodeActionKind.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      triggerKind : CodeActionTriggerKind.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeActionContext.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let diagnostics_field = ref Option.None
       and only_field = ref Option.None
       and triggerKind_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "diagnostics" -> (
             match Ppx_yojson_conv_lib.( ! ) diagnostics_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson Diagnostic.t_of_yojson _field_yojson
               in
               diagnostics_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "only" -> (
             match Ppx_yojson_conv_lib.( ! ) only_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson CodeActionKind.t_of_yojson)
                   _field_yojson
               in
               only_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "triggerKind" -> (
             match Ppx_yojson_conv_lib.( ! ) triggerKind_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   CodeActionTriggerKind.t_of_yojson _field_yojson
               in
               triggerKind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) diagnostics_field,
               Ppx_yojson_conv_lib.( ! ) only_field,
               Ppx_yojson_conv_lib.( ! ) triggerKind_field )
           with
           | Option.Some diagnostics_value, only_value, triggerKind_value ->
             { diagnostics = diagnostics_value;
               only =
                 (match only_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               triggerKind =
                 (match triggerKind_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) diagnostics_field)
                     Option.None,
                   "diagnostics" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { diagnostics = v_diagnostics;
         only = v_only;
         triggerKind = v_triggerKind
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_triggerKind then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t CodeActionTriggerKind.yojson_of_t)
               v_triggerKind
           in
           let bnd = ("triggerKind", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_only then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list CodeActionKind.yojson_of_t))
               v_only
           in
           let bnd = ("only", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_list Diagnostic.yojson_of_t v_diagnostics in
         ("diagnostics", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(diagnostics : Diagnostic.t list)
      ?(only : CodeActionKind.t list option)
      ?(triggerKind : CodeActionTriggerKind.t option) (() : unit) : t =
    { diagnostics; only; triggerKind }
end

module CodeActionOptions = struct
  type t =
    { codeActionKinds : CodeActionKind.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      resolveProvider : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeActionOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let codeActionKinds_field = ref Option.None
       and resolveProvider_field = ref Option.None
       and workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "codeActionKinds" -> (
             match Ppx_yojson_conv_lib.( ! ) codeActionKinds_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson CodeActionKind.t_of_yojson)
                   _field_yojson
               in
               codeActionKinds_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "resolveProvider" -> (
             match Ppx_yojson_conv_lib.( ! ) resolveProvider_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               resolveProvider_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( codeActionKinds_value,
                 resolveProvider_value,
                 workDoneProgress_value ) =
             ( Ppx_yojson_conv_lib.( ! ) codeActionKinds_field,
               Ppx_yojson_conv_lib.( ! ) resolveProvider_field,
               Ppx_yojson_conv_lib.( ! ) workDoneProgress_field )
           in
           { codeActionKinds =
               (match codeActionKinds_value with
               | Option.None -> None
               | Option.Some v -> v);
             resolveProvider =
               (match resolveProvider_value with
               | Option.None -> None
               | Option.Some v -> v);
             workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { codeActionKinds = v_codeActionKinds;
         resolveProvider = v_resolveProvider;
         workDoneProgress = v_workDoneProgress
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_resolveProvider then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_resolveProvider
           in
           let bnd = ("resolveProvider", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_codeActionKinds then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list CodeActionKind.yojson_of_t))
               v_codeActionKinds
           in
           let bnd = ("codeActionKinds", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(codeActionKinds : CodeActionKind.t list option)
      ?(resolveProvider : bool option) ?(workDoneProgress : bool option)
      (() : unit) : t =
    { codeActionKinds; resolveProvider; workDoneProgress }
end

module CodeActionParams = struct
  type t =
    { context : CodeActionContext.t;
      partialResultToken : ProgressToken.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      range : Range.t;
      textDocument : TextDocumentIdentifier.t;
      workDoneToken : ProgressToken.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeActionParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let context_field = ref Option.None
       and partialResultToken_field = ref Option.None
       and range_field = ref Option.None
       and textDocument_field = ref Option.None
       and workDoneToken_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "context" -> (
             match Ppx_yojson_conv_lib.( ! ) context_field with
             | Option.None ->
               let fvalue = CodeActionContext.t_of_yojson _field_yojson in
               context_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "partialResultToken" -> (
             match Ppx_yojson_conv_lib.( ! ) partialResultToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               partialResultToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "range" -> (
             match Ppx_yojson_conv_lib.( ! ) range_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               range_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue = TextDocumentIdentifier.t_of_yojson _field_yojson in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneToken" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               workDoneToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) context_field,
               Ppx_yojson_conv_lib.( ! ) partialResultToken_field,
               Ppx_yojson_conv_lib.( ! ) range_field,
               Ppx_yojson_conv_lib.( ! ) textDocument_field,
               Ppx_yojson_conv_lib.( ! ) workDoneToken_field )
           with
           | ( Option.Some context_value,
               partialResultToken_value,
               Option.Some range_value,
               Option.Some textDocument_value,
               workDoneToken_value ) ->
             { context = context_value;
               partialResultToken =
                 (match partialResultToken_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               range = range_value;
               textDocument = textDocument_value;
               workDoneToken =
                 (match workDoneToken_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) context_field)
                     Option.None,
                   "context" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) range_field)
                     Option.None,
                   "range" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { context = v_context;
         partialResultToken = v_partialResultToken;
         range = v_range;
         textDocument = v_textDocument;
         workDoneToken = v_workDoneToken
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_workDoneToken
           in
           let bnd = ("workDoneToken", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = TextDocumentIdentifier.yojson_of_t v_textDocument in
         ("textDocument", arg) :: bnds
       in
       let bnds =
         let arg = Range.yojson_of_t v_range in
         ("range", arg) :: bnds
       in
       let bnds =
         if None = v_partialResultToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_partialResultToken
           in
           let bnd = ("partialResultToken", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = CodeActionContext.yojson_of_t v_context in
         ("context", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(context : CodeActionContext.t)
      ?(partialResultToken : ProgressToken.t option) ~(range : Range.t)
      ~(textDocument : TextDocumentIdentifier.t)
      ?(workDoneToken : ProgressToken.t option) (() : unit) : t =
    { context; partialResultToken; range; textDocument; workDoneToken }
end

module CodeActionRegistrationOptions = struct
  type t =
    { codeActionKinds : CodeActionKind.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      documentSelector : DocumentSelector.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      resolveProvider : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeActionRegistrationOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let codeActionKinds_field = ref Option.None
       and documentSelector_field = ref Option.None
       and resolveProvider_field = ref Option.None
       and workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "codeActionKinds" -> (
             match Ppx_yojson_conv_lib.( ! ) codeActionKinds_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson CodeActionKind.t_of_yojson)
                   _field_yojson
               in
               codeActionKinds_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "documentSelector" -> (
             match Ppx_yojson_conv_lib.( ! ) documentSelector_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson DocumentSelector.t_of_yojson
                   _field_yojson
               in
               documentSelector_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "resolveProvider" -> (
             match Ppx_yojson_conv_lib.( ! ) resolveProvider_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               resolveProvider_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( codeActionKinds_value,
                 documentSelector_value,
                 resolveProvider_value,
                 workDoneProgress_value ) =
             ( Ppx_yojson_conv_lib.( ! ) codeActionKinds_field,
               Ppx_yojson_conv_lib.( ! ) documentSelector_field,
               Ppx_yojson_conv_lib.( ! ) resolveProvider_field,
               Ppx_yojson_conv_lib.( ! ) workDoneProgress_field )
           in
           { codeActionKinds =
               (match codeActionKinds_value with
               | Option.None -> None
               | Option.Some v -> v);
             documentSelector =
               (match documentSelector_value with
               | Option.None -> None
               | Option.Some v -> v);
             resolveProvider =
               (match resolveProvider_value with
               | Option.None -> None
               | Option.Some v -> v);
             workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { codeActionKinds = v_codeActionKinds;
         documentSelector = v_documentSelector;
         resolveProvider = v_resolveProvider;
         workDoneProgress = v_workDoneProgress
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_resolveProvider then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_resolveProvider
           in
           let bnd = ("resolveProvider", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentSelector then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t DocumentSelector.yojson_of_t)
               v_documentSelector
           in
           let bnd = ("documentSelector", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_codeActionKinds then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list CodeActionKind.yojson_of_t))
               v_codeActionKinds
           in
           let bnd = ("codeActionKinds", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(codeActionKinds : CodeActionKind.t list option)
      ?(documentSelector : DocumentSelector.t option)
      ?(resolveProvider : bool option) ?(workDoneProgress : bool option)
      (() : unit) : t =
    { codeActionKinds; documentSelector; resolveProvider; workDoneProgress }
end

module CodeLens = struct
  type t =
    { command : Command.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      data : Json.t option; [@yojson.option]
      range : Range.t
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeLens.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let command_field = ref Option.None
       and data_field = ref Option.None
       and range_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "command" -> (
             match Ppx_yojson_conv_lib.( ! ) command_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson Command.t_of_yojson
                   _field_yojson
               in
               command_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "data" -> (
             match Ppx_yojson_conv_lib.( ! ) data_field with
             | Option.None ->
               let fvalue = Json.t_of_yojson _field_yojson in
               data_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "range" -> (
             match Ppx_yojson_conv_lib.( ! ) range_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               range_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) command_field,
               Ppx_yojson_conv_lib.( ! ) data_field,
               Ppx_yojson_conv_lib.( ! ) range_field )
           with
           | command_value, data_value, Option.Some range_value ->
             { command =
                 (match command_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               data = data_value;
               range = range_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) range_field)
                     Option.None,
                   "range" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { command = v_command; data = v_data; range = v_range } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = Range.yojson_of_t v_range in
         ("range", arg) :: bnds
       in
       let bnds =
         match v_data with
         | Option.None -> bnds
         | Option.Some v ->
           let arg = Json.yojson_of_t v in
           let bnd = ("data", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_command then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t Command.yojson_of_t) v_command
           in
           let bnd = ("command", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(command : Command.t option) ?(data : Json.t option)
      ~(range : Range.t) (() : unit) : t =
    { command; data; range }
end

module CodeLensOptions = struct
  type t =
    { resolveProvider : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeLensOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let resolveProvider_field = ref Option.None
       and workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "resolveProvider" -> (
             match Ppx_yojson_conv_lib.( ! ) resolveProvider_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               resolveProvider_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let resolveProvider_value, workDoneProgress_value =
             ( Ppx_yojson_conv_lib.( ! ) resolveProvider_field,
               Ppx_yojson_conv_lib.( ! ) workDoneProgress_field )
           in
           { resolveProvider =
               (match resolveProvider_value with
               | Option.None -> None
               | Option.Some v -> v);
             workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { resolveProvider = v_resolveProvider;
         workDoneProgress = v_workDoneProgress
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_resolveProvider then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_resolveProvider
           in
           let bnd = ("resolveProvider", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(resolveProvider : bool option) ?(workDoneProgress : bool option)
      (() : unit) : t =
    { resolveProvider; workDoneProgress }
end

module CodeLensParams = struct
  type t =
    { partialResultToken : ProgressToken.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      textDocument : TextDocumentIdentifier.t;
      workDoneToken : ProgressToken.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeLensParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let partialResultToken_field = ref Option.None
       and textDocument_field = ref Option.None
       and workDoneToken_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "partialResultToken" -> (
             match Ppx_yojson_conv_lib.( ! ) partialResultToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               partialResultToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue = TextDocumentIdentifier.t_of_yojson _field_yojson in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneToken" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               workDoneToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) partialResultToken_field,
               Ppx_yojson_conv_lib.( ! ) textDocument_field,
               Ppx_yojson_conv_lib.( ! ) workDoneToken_field )
           with
           | ( partialResultToken_value,
               Option.Some textDocument_value,
               workDoneToken_value ) ->
             { partialResultToken =
                 (match partialResultToken_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               textDocument = textDocument_value;
               workDoneToken =
                 (match workDoneToken_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { partialResultToken = v_partialResultToken;
         textDocument = v_textDocument;
         workDoneToken = v_workDoneToken
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_workDoneToken
           in
           let bnd = ("workDoneToken", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = TextDocumentIdentifier.yojson_of_t v_textDocument in
         ("textDocument", arg) :: bnds
       in
       let bnds =
         if None = v_partialResultToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_partialResultToken
           in
           let bnd = ("partialResultToken", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(partialResultToken : ProgressToken.t option)
      ~(textDocument : TextDocumentIdentifier.t)
      ?(workDoneToken : ProgressToken.t option) (() : unit) : t =
    { partialResultToken; textDocument; workDoneToken }
end

module CodeLensRegistrationOptions = struct
  type t =
    { documentSelector : DocumentSelector.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      resolveProvider : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CodeLensRegistrationOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let documentSelector_field = ref Option.None
       and resolveProvider_field = ref Option.None
       and workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "documentSelector" -> (
             match Ppx_yojson_conv_lib.( ! ) documentSelector_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson DocumentSelector.t_of_yojson
                   _field_yojson
               in
               documentSelector_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "resolveProvider" -> (
             match Ppx_yojson_conv_lib.( ! ) resolveProvider_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               resolveProvider_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( documentSelector_value,
                 resolveProvider_value,
                 workDoneProgress_value ) =
             ( Ppx_yojson_conv_lib.( ! ) documentSelector_field,
               Ppx_yojson_conv_lib.( ! ) resolveProvider_field,
               Ppx_yojson_conv_lib.( ! ) workDoneProgress_field )
           in
           { documentSelector =
               (match documentSelector_value with
               | Option.None -> None
               | Option.Some v -> v);
             resolveProvider =
               (match resolveProvider_value with
               | Option.None -> None
               | Option.Some v -> v);
             workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { documentSelector = v_documentSelector;
         resolveProvider = v_resolveProvider;
         workDoneProgress = v_workDoneProgress
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_resolveProvider then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_resolveProvider
           in
           let bnd = ("resolveProvider", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentSelector then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t DocumentSelector.yojson_of_t)
               v_documentSelector
           in
           let bnd = ("documentSelector", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(documentSelector : DocumentSelector.t option)
      ?(resolveProvider : bool option) ?(workDoneProgress : bool option)
      (() : unit) : t =
    { documentSelector; resolveProvider; workDoneProgress }
end

module Color = struct
  type t = { alpha : int; blue : int; green : int; red : int }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.Color.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let alpha_field = ref Option.None
       and blue_field = ref Option.None
       and green_field = ref Option.None
       and red_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "alpha" -> (
             match Ppx_yojson_conv_lib.( ! ) alpha_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               alpha_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "blue" -> (
             match Ppx_yojson_conv_lib.( ! ) blue_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               blue_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "green" -> (
             match Ppx_yojson_conv_lib.( ! ) green_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               green_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "red" -> (
             match Ppx_yojson_conv_lib.( ! ) red_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               red_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) alpha_field,
               Ppx_yojson_conv_lib.( ! ) blue_field,
               Ppx_yojson_conv_lib.( ! ) green_field,
               Ppx_yojson_conv_lib.( ! ) red_field )
           with
           | ( Option.Some alpha_value,
               Option.Some blue_value,
               Option.Some green_value,
               Option.Some red_value ) ->
             { alpha = alpha_value;
               blue = blue_value;
               green = green_value;
               red = red_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) alpha_field)
                     Option.None,
                   "alpha" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) blue_field)
                     Option.None,
                   "blue" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) green_field)
                     Option.None,
                   "green" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) red_field)
                     Option.None,
                   "red" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { alpha = v_alpha; blue = v_blue; green = v_green; red = v_red } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_int v_red in
         ("red", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_int v_green in
         ("green", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_int v_blue in
         ("blue", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_int v_alpha in
         ("alpha", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(alpha : int) ~(blue : int) ~(green : int) ~(red : int) : t =
    { alpha; blue; green; red }
end

module ColorInformation = struct
  type t = { color : Color.t; range : Range.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ColorInformation.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let color_field = ref Option.None
       and range_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "color" -> (
             match Ppx_yojson_conv_lib.( ! ) color_field with
             | Option.None ->
               let fvalue = Color.t_of_yojson _field_yojson in
               color_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "range" -> (
             match Ppx_yojson_conv_lib.( ! ) range_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               range_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) color_field,
               Ppx_yojson_conv_lib.( ! ) range_field )
           with
           | Option.Some color_value, Option.Some range_value ->
             { color = color_value; range = range_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) color_field)
                     Option.None,
                   "color" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) range_field)
                     Option.None,
                   "range" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { color = v_color; range = v_range } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = Range.yojson_of_t v_range in
         ("range", arg) :: bnds
       in
       let bnds =
         let arg = Color.yojson_of_t v_color in
         ("color", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(color : Color.t) ~(range : Range.t) : t = { color; range }
end

module ColorPresentation = struct
  type t =
    { additionalTextEdits : TextEdit.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      label : string;
      textEdit : TextEdit.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ColorPresentation.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let additionalTextEdits_field = ref Option.None
       and label_field = ref Option.None
       and textEdit_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "additionalTextEdits" -> (
             match Ppx_yojson_conv_lib.( ! ) additionalTextEdits_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson TextEdit.t_of_yojson)
                   _field_yojson
               in
               additionalTextEdits_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "label" -> (
             match Ppx_yojson_conv_lib.( ! ) label_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               label_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textEdit" -> (
             match Ppx_yojson_conv_lib.( ! ) textEdit_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson TextEdit.t_of_yojson
                   _field_yojson
               in
               textEdit_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) additionalTextEdits_field,
               Ppx_yojson_conv_lib.( ! ) label_field,
               Ppx_yojson_conv_lib.( ! ) textEdit_field )
           with
           | additionalTextEdits_value, Option.Some label_value, textEdit_value
             ->
             { additionalTextEdits =
                 (match additionalTextEdits_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               label = label_value;
               textEdit =
                 (match textEdit_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) label_field)
                     Option.None,
                   "label" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { additionalTextEdits = v_additionalTextEdits;
         label = v_label;
         textEdit = v_textEdit
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_textEdit then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t TextEdit.yojson_of_t) v_textEdit
           in
           let bnd = ("textEdit", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_label in
         ("label", arg) :: bnds
       in
       let bnds =
         if None = v_additionalTextEdits then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list TextEdit.yojson_of_t))
               v_additionalTextEdits
           in
           let bnd = ("additionalTextEdits", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(additionalTextEdits : TextEdit.t list option) ~(label : string)
      ?(textEdit : TextEdit.t option) (() : unit) : t =
    { additionalTextEdits; label; textEdit }
end

module ColorPresentationParams = struct
  type t =
    { color : Color.t;
      partialResultToken : ProgressToken.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      range : Range.t;
      textDocument : TextDocumentIdentifier.t;
      workDoneToken : ProgressToken.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ColorPresentationParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let color_field = ref Option.None
       and partialResultToken_field = ref Option.None
       and range_field = ref Option.None
       and textDocument_field = ref Option.None
       and workDoneToken_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "color" -> (
             match Ppx_yojson_conv_lib.( ! ) color_field with
             | Option.None ->
               let fvalue = Color.t_of_yojson _field_yojson in
               color_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "partialResultToken" -> (
             match Ppx_yojson_conv_lib.( ! ) partialResultToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               partialResultToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "range" -> (
             match Ppx_yojson_conv_lib.( ! ) range_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               range_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue = TextDocumentIdentifier.t_of_yojson _field_yojson in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneToken" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               workDoneToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) color_field,
               Ppx_yojson_conv_lib.( ! ) partialResultToken_field,
               Ppx_yojson_conv_lib.( ! ) range_field,
               Ppx_yojson_conv_lib.( ! ) textDocument_field,
               Ppx_yojson_conv_lib.( ! ) workDoneToken_field )
           with
           | ( Option.Some color_value,
               partialResultToken_value,
               Option.Some range_value,
               Option.Some textDocument_value,
               workDoneToken_value ) ->
             { color = color_value;
               partialResultToken =
                 (match partialResultToken_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               range = range_value;
               textDocument = textDocument_value;
               workDoneToken =
                 (match workDoneToken_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) color_field)
                     Option.None,
                   "color" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) range_field)
                     Option.None,
                   "range" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { color = v_color;
         partialResultToken = v_partialResultToken;
         range = v_range;
         textDocument = v_textDocument;
         workDoneToken = v_workDoneToken
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_workDoneToken
           in
           let bnd = ("workDoneToken", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = TextDocumentIdentifier.yojson_of_t v_textDocument in
         ("textDocument", arg) :: bnds
       in
       let bnds =
         let arg = Range.yojson_of_t v_range in
         ("range", arg) :: bnds
       in
       let bnds =
         if None = v_partialResultToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_partialResultToken
           in
           let bnd = ("partialResultToken", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = Color.yojson_of_t v_color in
         ("color", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(color : Color.t) ?(partialResultToken : ProgressToken.t option)
      ~(range : Range.t) ~(textDocument : TextDocumentIdentifier.t)
      ?(workDoneToken : ProgressToken.t option) (() : unit) : t =
    { color; partialResultToken; range; textDocument; workDoneToken }
end

module CompletionContext = struct
  type t =
    { triggerCharacter : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      triggerKind : CompletionTriggerKind.t
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CompletionContext.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let triggerCharacter_field = ref Option.None
       and triggerKind_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "triggerCharacter" -> (
             match Ppx_yojson_conv_lib.( ! ) triggerCharacter_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               triggerCharacter_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "triggerKind" -> (
             match Ppx_yojson_conv_lib.( ! ) triggerKind_field with
             | Option.None ->
               let fvalue = CompletionTriggerKind.t_of_yojson _field_yojson in
               triggerKind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) triggerCharacter_field,
               Ppx_yojson_conv_lib.( ! ) triggerKind_field )
           with
           | triggerCharacter_value, Option.Some triggerKind_value ->
             { triggerCharacter =
                 (match triggerCharacter_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               triggerKind = triggerKind_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) triggerKind_field)
                     Option.None,
                   "triggerKind" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { triggerCharacter = v_triggerCharacter; triggerKind = v_triggerKind } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = CompletionTriggerKind.yojson_of_t v_triggerKind in
         ("triggerKind", arg) :: bnds
       in
       let bnds =
         if None = v_triggerCharacter then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string)
               v_triggerCharacter
           in
           let bnd = ("triggerCharacter", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(triggerCharacter : string option)
      ~(triggerKind : CompletionTriggerKind.t) (() : unit) : t =
    { triggerCharacter; triggerKind }
end

module InsertReplaceEdit = struct
  type t = { insert : Range.t; newText : string; replace : Range.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.InsertReplaceEdit.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let insert_field = ref Option.None
       and newText_field = ref Option.None
       and replace_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "insert" -> (
             match Ppx_yojson_conv_lib.( ! ) insert_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               insert_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "newText" -> (
             match Ppx_yojson_conv_lib.( ! ) newText_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               newText_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "replace" -> (
             match Ppx_yojson_conv_lib.( ! ) replace_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               replace_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) insert_field,
               Ppx_yojson_conv_lib.( ! ) newText_field,
               Ppx_yojson_conv_lib.( ! ) replace_field )
           with
           | ( Option.Some insert_value,
               Option.Some newText_value,
               Option.Some replace_value ) ->
             { insert = insert_value;
               newText = newText_value;
               replace = replace_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) insert_field)
                     Option.None,
                   "insert" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) newText_field)
                     Option.None,
                   "newText" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) replace_field)
                     Option.None,
                   "replace" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { insert = v_insert; newText = v_newText; replace = v_replace } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = Range.yojson_of_t v_replace in
         ("replace", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_newText in
         ("newText", arg) :: bnds
       in
       let bnds =
         let arg = Range.yojson_of_t v_insert in
         ("insert", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(insert : Range.t) ~(newText : string) ~(replace : Range.t) : t =
    { insert; newText; replace }
end

module CompletionItemLabelDetails = struct
  type t =
    { description : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      detail : string Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CompletionItemLabelDetails.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let description_field = ref Option.None
       and detail_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "description" -> (
             match Ppx_yojson_conv_lib.( ! ) description_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               description_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "detail" -> (
             match Ppx_yojson_conv_lib.( ! ) detail_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               detail_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let description_value, detail_value =
             ( Ppx_yojson_conv_lib.( ! ) description_field,
               Ppx_yojson_conv_lib.( ! ) detail_field )
           in
           { description =
               (match description_value with
               | Option.None -> None
               | Option.Some v -> v);
             detail =
               (match detail_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { description = v_description; detail = v_detail } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_detail then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_detail
           in
           let bnd = ("detail", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_description then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_description
           in
           let bnd = ("description", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(description : string option) ?(detail : string option)
      (() : unit) : t =
    { description; detail }
end

module CompletionItem = struct
  type documentation_pvar =
    [ `String of string | `MarkupContent of MarkupContent.t ]

  let documentation_pvar_of_yojson (json : Json.t) : documentation_pvar =
    match json with
    | `String j -> `String j
    | _ ->
      Json.Of.untagged_union "documentation_pvar"
        [ (fun json -> `MarkupContent (MarkupContent.t_of_yojson json)) ]
        json

  let yojson_of_documentation_pvar (documentation_pvar : documentation_pvar) :
      Json.t =
    match documentation_pvar with
    | `String j -> `String j
    | `MarkupContent s -> MarkupContent.yojson_of_t s

  type textEdit_pvar =
    [ `TextEdit of TextEdit.t | `InsertReplaceEdit of InsertReplaceEdit.t ]

  let textEdit_pvar_of_yojson (json : Json.t) : textEdit_pvar =
    Json.Of.untagged_union "textEdit_pvar"
      [ (fun json -> `TextEdit (TextEdit.t_of_yojson json));
        (fun json -> `InsertReplaceEdit (InsertReplaceEdit.t_of_yojson json))
      ]
      json

  let yojson_of_textEdit_pvar (textEdit_pvar : textEdit_pvar) : Json.t =
    match textEdit_pvar with
    | `TextEdit s -> TextEdit.yojson_of_t s
    | `InsertReplaceEdit s -> InsertReplaceEdit.yojson_of_t s

  type t =
    { additionalTextEdits : TextEdit.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      command : Command.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      commitCharacters : string list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      data : Json.t option; [@yojson.option]
      deprecated : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      detail : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      documentation : documentation_pvar Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      filterText : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      insertText : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      insertTextFormat : InsertTextFormat.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      insertTextMode : InsertTextMode.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      kind : CompletionItemKind.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      label : string;
      labelDetails : CompletionItemLabelDetails.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      preselect : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      sortText : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      tags : CompletionItemTag.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      textEdit : textEdit_pvar Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      textEditText : string Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CompletionItem.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let additionalTextEdits_field = ref Option.None
       and command_field = ref Option.None
       and commitCharacters_field = ref Option.None
       and data_field = ref Option.None
       and deprecated_field = ref Option.None
       and detail_field = ref Option.None
       and documentation_field = ref Option.None
       and filterText_field = ref Option.None
       and insertText_field = ref Option.None
       and insertTextFormat_field = ref Option.None
       and insertTextMode_field = ref Option.None
       and kind_field = ref Option.None
       and label_field = ref Option.None
       and labelDetails_field = ref Option.None
       and preselect_field = ref Option.None
       and sortText_field = ref Option.None
       and tags_field = ref Option.None
       and textEdit_field = ref Option.None
       and textEditText_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "additionalTextEdits" -> (
             match Ppx_yojson_conv_lib.( ! ) additionalTextEdits_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson TextEdit.t_of_yojson)
                   _field_yojson
               in
               additionalTextEdits_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "command" -> (
             match Ppx_yojson_conv_lib.( ! ) command_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson Command.t_of_yojson
                   _field_yojson
               in
               command_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "commitCharacters" -> (
             match Ppx_yojson_conv_lib.( ! ) commitCharacters_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson string_of_yojson)
                   _field_yojson
               in
               commitCharacters_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "data" -> (
             match Ppx_yojson_conv_lib.( ! ) data_field with
             | Option.None ->
               let fvalue = Json.t_of_yojson _field_yojson in
               data_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "deprecated" -> (
             match Ppx_yojson_conv_lib.( ! ) deprecated_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               deprecated_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "detail" -> (
             match Ppx_yojson_conv_lib.( ! ) detail_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               detail_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "documentation" -> (
             match Ppx_yojson_conv_lib.( ! ) documentation_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson documentation_pvar_of_yojson
                   _field_yojson
               in
               documentation_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "filterText" -> (
             match Ppx_yojson_conv_lib.( ! ) filterText_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               filterText_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "insertText" -> (
             match Ppx_yojson_conv_lib.( ! ) insertText_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               insertText_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "insertTextFormat" -> (
             match Ppx_yojson_conv_lib.( ! ) insertTextFormat_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson InsertTextFormat.t_of_yojson
                   _field_yojson
               in
               insertTextFormat_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "insertTextMode" -> (
             match Ppx_yojson_conv_lib.( ! ) insertTextMode_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson InsertTextMode.t_of_yojson
                   _field_yojson
               in
               insertTextMode_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "kind" -> (
             match Ppx_yojson_conv_lib.( ! ) kind_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson CompletionItemKind.t_of_yojson
                   _field_yojson
               in
               kind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "label" -> (
             match Ppx_yojson_conv_lib.( ! ) label_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               label_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "labelDetails" -> (
             match Ppx_yojson_conv_lib.( ! ) labelDetails_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   CompletionItemLabelDetails.t_of_yojson _field_yojson
               in
               labelDetails_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "preselect" -> (
             match Ppx_yojson_conv_lib.( ! ) preselect_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               preselect_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "sortText" -> (
             match Ppx_yojson_conv_lib.( ! ) sortText_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               sortText_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "tags" -> (
             match Ppx_yojson_conv_lib.( ! ) tags_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson CompletionItemTag.t_of_yojson)
                   _field_yojson
               in
               tags_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textEdit" -> (
             match Ppx_yojson_conv_lib.( ! ) textEdit_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson textEdit_pvar_of_yojson
                   _field_yojson
               in
               textEdit_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textEditText" -> (
             match Ppx_yojson_conv_lib.( ! ) textEditText_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               textEditText_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) additionalTextEdits_field,
               Ppx_yojson_conv_lib.( ! ) command_field,
               Ppx_yojson_conv_lib.( ! ) commitCharacters_field,
               Ppx_yojson_conv_lib.( ! ) data_field,
               Ppx_yojson_conv_lib.( ! ) deprecated_field,
               Ppx_yojson_conv_lib.( ! ) detail_field,
               Ppx_yojson_conv_lib.( ! ) documentation_field,
               Ppx_yojson_conv_lib.( ! ) filterText_field,
               Ppx_yojson_conv_lib.( ! ) insertText_field,
               Ppx_yojson_conv_lib.( ! ) insertTextFormat_field,
               Ppx_yojson_conv_lib.( ! ) insertTextMode_field,
               Ppx_yojson_conv_lib.( ! ) kind_field,
               Ppx_yojson_conv_lib.( ! ) label_field,
               Ppx_yojson_conv_lib.( ! ) labelDetails_field,
               Ppx_yojson_conv_lib.( ! ) preselect_field,
               Ppx_yojson_conv_lib.( ! ) sortText_field,
               Ppx_yojson_conv_lib.( ! ) tags_field,
               Ppx_yojson_conv_lib.( ! ) textEdit_field,
               Ppx_yojson_conv_lib.( ! ) textEditText_field )
           with
           | ( additionalTextEdits_value,
               command_value,
               commitCharacters_value,
               data_value,
               deprecated_value,
               detail_value,
               documentation_value,
               filterText_value,
               insertText_value,
               insertTextFormat_value,
               insertTextMode_value,
               kind_value,
               Option.Some label_value,
               labelDetails_value,
               preselect_value,
               sortText_value,
               tags_value,
               textEdit_value,
               textEditText_value ) ->
             { additionalTextEdits =
                 (match additionalTextEdits_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               command =
                 (match command_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               commitCharacters =
                 (match commitCharacters_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               data = data_value;
               deprecated =
                 (match deprecated_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               detail =
                 (match detail_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               documentation =
                 (match documentation_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               filterText =
                 (match filterText_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               insertText =
                 (match insertText_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               insertTextFormat =
                 (match insertTextFormat_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               insertTextMode =
                 (match insertTextMode_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               kind =
                 (match kind_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               label = label_value;
               labelDetails =
                 (match labelDetails_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               preselect =
                 (match preselect_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               sortText =
                 (match sortText_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               tags =
                 (match tags_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               textEdit =
                 (match textEdit_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               textEditText =
                 (match textEditText_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) label_field)
                     Option.None,
                   "label" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { additionalTextEdits = v_additionalTextEdits;
         command = v_command;
         commitCharacters = v_commitCharacters;
         data = v_data;
         deprecated = v_deprecated;
         detail = v_detail;
         documentation = v_documentation;
         filterText = v_filterText;
         insertText = v_insertText;
         insertTextFormat = v_insertTextFormat;
         insertTextMode = v_insertTextMode;
         kind = v_kind;
         label = v_label;
         labelDetails = v_labelDetails;
         preselect = v_preselect;
         sortText = v_sortText;
         tags = v_tags;
         textEdit = v_textEdit;
         textEditText = v_textEditText
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_textEditText then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_textEditText
           in
           let bnd = ("textEditText", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_textEdit then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_textEdit_pvar)
               v_textEdit
           in
           let bnd = ("textEdit", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_tags then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list CompletionItemTag.yojson_of_t))
               v_tags
           in
           let bnd = ("tags", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_sortText then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_sortText
           in
           let bnd = ("sortText", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_preselect then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_preselect
           in
           let bnd = ("preselect", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_labelDetails then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                CompletionItemLabelDetails.yojson_of_t)
               v_labelDetails
           in
           let bnd = ("labelDetails", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_label in
         ("label", arg) :: bnds
       in
       let bnds =
         if None = v_kind then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t CompletionItemKind.yojson_of_t)
               v_kind
           in
           let bnd = ("kind", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_insertTextMode then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t InsertTextMode.yojson_of_t)
               v_insertTextMode
           in
           let bnd = ("insertTextMode", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_insertTextFormat then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t InsertTextFormat.yojson_of_t)
               v_insertTextFormat
           in
           let bnd = ("insertTextFormat", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_insertText then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_insertText
           in
           let bnd = ("insertText", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_filterText then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_filterText
           in
           let bnd = ("filterText", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentation then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_documentation_pvar)
               v_documentation
           in
           let bnd = ("documentation", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_detail then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_detail
           in
           let bnd = ("detail", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_deprecated then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_deprecated
           in
           let bnd = ("deprecated", arg) in
           bnd :: bnds
       in
       let bnds =
         match v_data with
         | Option.None -> bnds
         | Option.Some v ->
           let arg = Json.yojson_of_t v in
           let bnd = ("data", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_commitCharacters then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list yojson_of_string))
               v_commitCharacters
           in
           let bnd = ("commitCharacters", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_command then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t Command.yojson_of_t) v_command
           in
           let bnd = ("command", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_additionalTextEdits then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list TextEdit.yojson_of_t))
               v_additionalTextEdits
           in
           let bnd = ("additionalTextEdits", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(additionalTextEdits : TextEdit.t list option)
      ?(command : Command.t option) ?(commitCharacters : string list option)
      ?(data : Json.t option) ?(deprecated : bool option)
      ?(detail : string option) ?(documentation : documentation_pvar option)
      ?(filterText : string option) ?(insertText : string option)
      ?(insertTextFormat : InsertTextFormat.t option)
      ?(insertTextMode : InsertTextMode.t option)
      ?(kind : CompletionItemKind.t option) ~(label : string)
      ?(labelDetails : CompletionItemLabelDetails.t option)
      ?(preselect : bool option) ?(sortText : string option)
      ?(tags : CompletionItemTag.t list option)
      ?(textEdit : textEdit_pvar option) ?(textEditText : string option)
      (() : unit) : t =
    { additionalTextEdits;
      command;
      commitCharacters;
      data;
      deprecated;
      detail;
      documentation;
      filterText;
      insertText;
      insertTextFormat;
      insertTextMode;
      kind;
      label;
      labelDetails;
      preselect;
      sortText;
      tags;
      textEdit;
      textEditText
    }
end

module CompletionList = struct
  type editRange = { insert : Range.t; replace : Range.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : editRange) -> ()

  let editRange_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CompletionList.editRange" in
     function
     | `Assoc field_yojsons as yojson -> (
       let insert_field = ref Option.None
       and replace_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "insert" -> (
             match Ppx_yojson_conv_lib.( ! ) insert_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               insert_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "replace" -> (
             match Ppx_yojson_conv_lib.( ! ) replace_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               replace_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) insert_field,
               Ppx_yojson_conv_lib.( ! ) replace_field )
           with
           | Option.Some insert_value, Option.Some replace_value ->
             { insert = insert_value; replace = replace_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) insert_field)
                     Option.None,
                   "insert" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) replace_field)
                     Option.None,
                   "replace" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> editRange)

  let _ = editRange_of_yojson

  let yojson_of_editRange =
    (function
     | { insert = v_insert; replace = v_replace } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = Range.yojson_of_t v_replace in
         ("replace", arg) :: bnds
       in
       let bnds =
         let arg = Range.yojson_of_t v_insert in
         ("insert", arg) :: bnds
       in
       `Assoc bnds
      : editRange -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_editRange

  [@@@end]

  let create_editRange ~(insert : Range.t) ~(replace : Range.t) : editRange =
    { insert; replace }

  type editRange_pvar = [ `Range of Range.t | `EditRange of editRange ]

  let editRange_pvar_of_yojson (json : Json.t) : editRange_pvar =
    Json.Of.untagged_union "editRange_pvar"
      [ (fun json -> `Range (Range.t_of_yojson json));
        (fun json -> `EditRange (editRange_of_yojson json))
      ]
      json

  let yojson_of_editRange_pvar (editRange_pvar : editRange_pvar) : Json.t =
    match editRange_pvar with
    | `Range s -> Range.yojson_of_t s
    | `EditRange s -> yojson_of_editRange s

  type itemDefaults =
    { commitCharacters : string list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      editRange : editRange_pvar Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      insertTextFormat : InsertTextFormat.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      insertTextMode : InsertTextMode.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      data : Json.t option [@yojson.option]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : itemDefaults) -> ()

  let itemDefaults_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CompletionList.itemDefaults" in
     function
     | `Assoc field_yojsons as yojson -> (
       let commitCharacters_field = ref Option.None
       and editRange_field = ref Option.None
       and insertTextFormat_field = ref Option.None
       and insertTextMode_field = ref Option.None
       and data_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "commitCharacters" -> (
             match Ppx_yojson_conv_lib.( ! ) commitCharacters_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson string_of_yojson)
                   _field_yojson
               in
               commitCharacters_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "editRange" -> (
             match Ppx_yojson_conv_lib.( ! ) editRange_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson editRange_pvar_of_yojson
                   _field_yojson
               in
               editRange_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "insertTextFormat" -> (
             match Ppx_yojson_conv_lib.( ! ) insertTextFormat_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson InsertTextFormat.t_of_yojson
                   _field_yojson
               in
               insertTextFormat_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "insertTextMode" -> (
             match Ppx_yojson_conv_lib.( ! ) insertTextMode_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson InsertTextMode.t_of_yojson
                   _field_yojson
               in
               insertTextMode_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "data" -> (
             match Ppx_yojson_conv_lib.( ! ) data_field with
             | Option.None ->
               let fvalue = Json.t_of_yojson _field_yojson in
               data_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( commitCharacters_value,
                 editRange_value,
                 insertTextFormat_value,
                 insertTextMode_value,
                 data_value ) =
             ( Ppx_yojson_conv_lib.( ! ) commitCharacters_field,
               Ppx_yojson_conv_lib.( ! ) editRange_field,
               Ppx_yojson_conv_lib.( ! ) insertTextFormat_field,
               Ppx_yojson_conv_lib.( ! ) insertTextMode_field,
               Ppx_yojson_conv_lib.( ! ) data_field )
           in
           { commitCharacters =
               (match commitCharacters_value with
               | Option.None -> None
               | Option.Some v -> v);
             editRange =
               (match editRange_value with
               | Option.None -> None
               | Option.Some v -> v);
             insertTextFormat =
               (match insertTextFormat_value with
               | Option.None -> None
               | Option.Some v -> v);
             insertTextMode =
               (match insertTextMode_value with
               | Option.None -> None
               | Option.Some v -> v);
             data = data_value
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> itemDefaults)

  let _ = itemDefaults_of_yojson

  let yojson_of_itemDefaults =
    (function
     | { commitCharacters = v_commitCharacters;
         editRange = v_editRange;
         insertTextFormat = v_insertTextFormat;
         insertTextMode = v_insertTextMode;
         data = v_data
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         match v_data with
         | Option.None -> bnds
         | Option.Some v ->
           let arg = Json.yojson_of_t v in
           let bnd = ("data", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_insertTextMode then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t InsertTextMode.yojson_of_t)
               v_insertTextMode
           in
           let bnd = ("insertTextMode", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_insertTextFormat then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t InsertTextFormat.yojson_of_t)
               v_insertTextFormat
           in
           let bnd = ("insertTextFormat", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_editRange then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_editRange_pvar)
               v_editRange
           in
           let bnd = ("editRange", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_commitCharacters then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list yojson_of_string))
               v_commitCharacters
           in
           let bnd = ("commitCharacters", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : itemDefaults -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_itemDefaults

  [@@@end]

  let create_itemDefaults ?(commitCharacters : string list option)
      ?(editRange : editRange_pvar option)
      ?(insertTextFormat : InsertTextFormat.t option)
      ?(insertTextMode : InsertTextMode.t option) ?(data : Json.t option)
      (() : unit) : itemDefaults =
    { commitCharacters; editRange; insertTextFormat; insertTextMode; data }

  type t =
    { isIncomplete : bool;
      itemDefaults : itemDefaults Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      items : CompletionItem.t list
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CompletionList.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let isIncomplete_field = ref Option.None
       and itemDefaults_field = ref Option.None
       and items_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "isIncomplete" -> (
             match Ppx_yojson_conv_lib.( ! ) isIncomplete_field with
             | Option.None ->
               let fvalue = bool_of_yojson _field_yojson in
               isIncomplete_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "itemDefaults" -> (
             match Ppx_yojson_conv_lib.( ! ) itemDefaults_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson itemDefaults_of_yojson
                   _field_yojson
               in
               itemDefaults_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "items" -> (
             match Ppx_yojson_conv_lib.( ! ) items_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson CompletionItem.t_of_yojson _field_yojson
               in
               items_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) isIncomplete_field,
               Ppx_yojson_conv_lib.( ! ) itemDefaults_field,
               Ppx_yojson_conv_lib.( ! ) items_field )
           with
           | ( Option.Some isIncomplete_value,
               itemDefaults_value,
               Option.Some items_value ) ->
             { isIncomplete = isIncomplete_value;
               itemDefaults =
                 (match itemDefaults_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               items = items_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) isIncomplete_field)
                     Option.None,
                   "isIncomplete" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) items_field)
                     Option.None,
                   "items" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { isIncomplete = v_isIncomplete;
         itemDefaults = v_itemDefaults;
         items = v_items
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list CompletionItem.yojson_of_t v_items in
         ("items", arg) :: bnds
       in
       let bnds =
         if None = v_itemDefaults then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_itemDefaults)
               v_itemDefaults
           in
           let bnd = ("itemDefaults", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_bool v_isIncomplete in
         ("isIncomplete", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(isIncomplete : bool) ?(itemDefaults : itemDefaults option)
      ~(items : CompletionItem.t list) (() : unit) : t =
    { isIncomplete; itemDefaults; items }
end

module CompletionOptions = struct
  type completionItem =
    { labelDetailsSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : completionItem) -> ()

  let completionItem_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CompletionOptions.completionItem" in
     function
     | `Assoc field_yojsons as yojson -> (
       let labelDetailsSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "labelDetailsSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) labelDetailsSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               labelDetailsSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let labelDetailsSupport_value =
             Ppx_yojson_conv_lib.( ! ) labelDetailsSupport_field
           in
           { labelDetailsSupport =
               (match labelDetailsSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> completionItem)

  let _ = completionItem_of_yojson

  let yojson_of_completionItem =
    (function
     | { labelDetailsSupport = v_labelDetailsSupport } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_labelDetailsSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_labelDetailsSupport
           in
           let bnd = ("labelDetailsSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : completionItem -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_completionItem

  [@@@end]

  let create_completionItem ?(labelDetailsSupport : bool option) (() : unit) :
      completionItem =
    { labelDetailsSupport }

  type t =
    { allCommitCharacters : string list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      completionItem : completionItem Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      resolveProvider : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      triggerCharacters : string list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CompletionOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let allCommitCharacters_field = ref Option.None
       and completionItem_field = ref Option.None
       and resolveProvider_field = ref Option.None
       and triggerCharacters_field = ref Option.None
       and workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "allCommitCharacters" -> (
             match Ppx_yojson_conv_lib.( ! ) allCommitCharacters_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson string_of_yojson)
                   _field_yojson
               in
               allCommitCharacters_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "completionItem" -> (
             match Ppx_yojson_conv_lib.( ! ) completionItem_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson completionItem_of_yojson
                   _field_yojson
               in
               completionItem_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "resolveProvider" -> (
             match Ppx_yojson_conv_lib.( ! ) resolveProvider_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               resolveProvider_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "triggerCharacters" -> (
             match Ppx_yojson_conv_lib.( ! ) triggerCharacters_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson string_of_yojson)
                   _field_yojson
               in
               triggerCharacters_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( allCommitCharacters_value,
                 completionItem_value,
                 resolveProvider_value,
                 triggerCharacters_value,
                 workDoneProgress_value ) =
             ( Ppx_yojson_conv_lib.( ! ) allCommitCharacters_field,
               Ppx_yojson_conv_lib.( ! ) completionItem_field,
               Ppx_yojson_conv_lib.( ! ) resolveProvider_field,
               Ppx_yojson_conv_lib.( ! ) triggerCharacters_field,
               Ppx_yojson_conv_lib.( ! ) workDoneProgress_field )
           in
           { allCommitCharacters =
               (match allCommitCharacters_value with
               | Option.None -> None
               | Option.Some v -> v);
             completionItem =
               (match completionItem_value with
               | Option.None -> None
               | Option.Some v -> v);
             resolveProvider =
               (match resolveProvider_value with
               | Option.None -> None
               | Option.Some v -> v);
             triggerCharacters =
               (match triggerCharacters_value with
               | Option.None -> None
               | Option.Some v -> v);
             workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { allCommitCharacters = v_allCommitCharacters;
         completionItem = v_completionItem;
         resolveProvider = v_resolveProvider;
         triggerCharacters = v_triggerCharacters;
         workDoneProgress = v_workDoneProgress
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_triggerCharacters then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list yojson_of_string))
               v_triggerCharacters
           in
           let bnd = ("triggerCharacters", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_resolveProvider then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_resolveProvider
           in
           let bnd = ("resolveProvider", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_completionItem then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_completionItem)
               v_completionItem
           in
           let bnd = ("completionItem", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_allCommitCharacters then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list yojson_of_string))
               v_allCommitCharacters
           in
           let bnd = ("allCommitCharacters", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(allCommitCharacters : string list option)
      ?(completionItem : completionItem option) ?(resolveProvider : bool option)
      ?(triggerCharacters : string list option)
      ?(workDoneProgress : bool option) (() : unit) : t =
    { allCommitCharacters;
      completionItem;
      resolveProvider;
      triggerCharacters;
      workDoneProgress
    }
end

module CompletionParams = struct
  type t =
    { context : CompletionContext.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      partialResultToken : ProgressToken.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      position : Position.t;
      textDocument : TextDocumentIdentifier.t;
      workDoneToken : ProgressToken.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CompletionParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let context_field = ref Option.None
       and partialResultToken_field = ref Option.None
       and position_field = ref Option.None
       and textDocument_field = ref Option.None
       and workDoneToken_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "context" -> (
             match Ppx_yojson_conv_lib.( ! ) context_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson CompletionContext.t_of_yojson
                   _field_yojson
               in
               context_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "partialResultToken" -> (
             match Ppx_yojson_conv_lib.( ! ) partialResultToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               partialResultToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "position" -> (
             match Ppx_yojson_conv_lib.( ! ) position_field with
             | Option.None ->
               let fvalue = Position.t_of_yojson _field_yojson in
               position_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue = TextDocumentIdentifier.t_of_yojson _field_yojson in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneToken" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               workDoneToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) context_field,
               Ppx_yojson_conv_lib.( ! ) partialResultToken_field,
               Ppx_yojson_conv_lib.( ! ) position_field,
               Ppx_yojson_conv_lib.( ! ) textDocument_field,
               Ppx_yojson_conv_lib.( ! ) workDoneToken_field )
           with
           | ( context_value,
               partialResultToken_value,
               Option.Some position_value,
               Option.Some textDocument_value,
               workDoneToken_value ) ->
             { context =
                 (match context_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               partialResultToken =
                 (match partialResultToken_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               position = position_value;
               textDocument = textDocument_value;
               workDoneToken =
                 (match workDoneToken_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) position_field)
                     Option.None,
                   "position" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { context = v_context;
         partialResultToken = v_partialResultToken;
         position = v_position;
         textDocument = v_textDocument;
         workDoneToken = v_workDoneToken
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_workDoneToken
           in
           let bnd = ("workDoneToken", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = TextDocumentIdentifier.yojson_of_t v_textDocument in
         ("textDocument", arg) :: bnds
       in
       let bnds =
         let arg = Position.yojson_of_t v_position in
         ("position", arg) :: bnds
       in
       let bnds =
         if None = v_partialResultToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_partialResultToken
           in
           let bnd = ("partialResultToken", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_context then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t CompletionContext.yojson_of_t)
               v_context
           in
           let bnd = ("context", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(context : CompletionContext.t option)
      ?(partialResultToken : ProgressToken.t option) ~(position : Position.t)
      ~(textDocument : TextDocumentIdentifier.t)
      ?(workDoneToken : ProgressToken.t option) (() : unit) : t =
    { context; partialResultToken; position; textDocument; workDoneToken }
end

module CompletionRegistrationOptions = struct
  type completionItem =
    { labelDetailsSupport : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : completionItem) -> ()

  let completionItem_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.CompletionRegistrationOptions.completionItem"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let labelDetailsSupport_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "labelDetailsSupport" -> (
             match Ppx_yojson_conv_lib.( ! ) labelDetailsSupport_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               labelDetailsSupport_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let labelDetailsSupport_value =
             Ppx_yojson_conv_lib.( ! ) labelDetailsSupport_field
           in
           { labelDetailsSupport =
               (match labelDetailsSupport_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> completionItem)

  let _ = completionItem_of_yojson

  let yojson_of_completionItem =
    (function
     | { labelDetailsSupport = v_labelDetailsSupport } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_labelDetailsSupport then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_labelDetailsSupport
           in
           let bnd = ("labelDetailsSupport", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : completionItem -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_completionItem

  [@@@end]

  let create_completionItem ?(labelDetailsSupport : bool option) (() : unit) :
      completionItem =
    { labelDetailsSupport }

  type t =
    { allCommitCharacters : string list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      completionItem : completionItem Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      documentSelector : DocumentSelector.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      resolveProvider : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      triggerCharacters : string list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CompletionRegistrationOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let allCommitCharacters_field = ref Option.None
       and completionItem_field = ref Option.None
       and documentSelector_field = ref Option.None
       and resolveProvider_field = ref Option.None
       and triggerCharacters_field = ref Option.None
       and workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "allCommitCharacters" -> (
             match Ppx_yojson_conv_lib.( ! ) allCommitCharacters_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson string_of_yojson)
                   _field_yojson
               in
               allCommitCharacters_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "completionItem" -> (
             match Ppx_yojson_conv_lib.( ! ) completionItem_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson completionItem_of_yojson
                   _field_yojson
               in
               completionItem_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "documentSelector" -> (
             match Ppx_yojson_conv_lib.( ! ) documentSelector_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson DocumentSelector.t_of_yojson
                   _field_yojson
               in
               documentSelector_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "resolveProvider" -> (
             match Ppx_yojson_conv_lib.( ! ) resolveProvider_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               resolveProvider_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "triggerCharacters" -> (
             match Ppx_yojson_conv_lib.( ! ) triggerCharacters_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson string_of_yojson)
                   _field_yojson
               in
               triggerCharacters_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let ( allCommitCharacters_value,
                 completionItem_value,
                 documentSelector_value,
                 resolveProvider_value,
                 triggerCharacters_value,
                 workDoneProgress_value ) =
             ( Ppx_yojson_conv_lib.( ! ) allCommitCharacters_field,
               Ppx_yojson_conv_lib.( ! ) completionItem_field,
               Ppx_yojson_conv_lib.( ! ) documentSelector_field,
               Ppx_yojson_conv_lib.( ! ) resolveProvider_field,
               Ppx_yojson_conv_lib.( ! ) triggerCharacters_field,
               Ppx_yojson_conv_lib.( ! ) workDoneProgress_field )
           in
           { allCommitCharacters =
               (match allCommitCharacters_value with
               | Option.None -> None
               | Option.Some v -> v);
             completionItem =
               (match completionItem_value with
               | Option.None -> None
               | Option.Some v -> v);
             documentSelector =
               (match documentSelector_value with
               | Option.None -> None
               | Option.Some v -> v);
             resolveProvider =
               (match resolveProvider_value with
               | Option.None -> None
               | Option.Some v -> v);
             triggerCharacters =
               (match triggerCharacters_value with
               | Option.None -> None
               | Option.Some v -> v);
             workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { allCommitCharacters = v_allCommitCharacters;
         completionItem = v_completionItem;
         documentSelector = v_documentSelector;
         resolveProvider = v_resolveProvider;
         triggerCharacters = v_triggerCharacters;
         workDoneProgress = v_workDoneProgress
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_triggerCharacters then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list yojson_of_string))
               v_triggerCharacters
           in
           let bnd = ("triggerCharacters", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_resolveProvider then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_resolveProvider
           in
           let bnd = ("resolveProvider", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentSelector then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t DocumentSelector.yojson_of_t)
               v_documentSelector
           in
           let bnd = ("documentSelector", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_completionItem then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_completionItem)
               v_completionItem
           in
           let bnd = ("completionItem", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_allCommitCharacters then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list yojson_of_string))
               v_allCommitCharacters
           in
           let bnd = ("allCommitCharacters", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(allCommitCharacters : string list option)
      ?(completionItem : completionItem option)
      ?(documentSelector : DocumentSelector.t option)
      ?(resolveProvider : bool option) ?(triggerCharacters : string list option)
      ?(workDoneProgress : bool option) (() : unit) : t =
    { allCommitCharacters;
      completionItem;
      documentSelector;
      resolveProvider;
      triggerCharacters;
      workDoneProgress
    }
end

module ConfigurationItem = struct
  type t =
    { scopeUri : DocumentUri.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      section : string Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ConfigurationItem.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let scopeUri_field = ref Option.None
       and section_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "scopeUri" -> (
             match Ppx_yojson_conv_lib.( ! ) scopeUri_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson DocumentUri.t_of_yojson
                   _field_yojson
               in
               scopeUri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "section" -> (
             match Ppx_yojson_conv_lib.( ! ) section_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               section_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let scopeUri_value, section_value =
             ( Ppx_yojson_conv_lib.( ! ) scopeUri_field,
               Ppx_yojson_conv_lib.( ! ) section_field )
           in
           { scopeUri =
               (match scopeUri_value with
               | Option.None -> None
               | Option.Some v -> v);
             section =
               (match section_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { scopeUri = v_scopeUri; section = v_section } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_section then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_section
           in
           let bnd = ("section", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_scopeUri then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t DocumentUri.yojson_of_t)
               v_scopeUri
           in
           let bnd = ("scopeUri", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(scopeUri : DocumentUri.t option) ?(section : string option)
      (() : unit) : t =
    { scopeUri; section }
end

module ConfigurationParams = struct
  type t = { items : ConfigurationItem.t list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ConfigurationParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let items_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "items" -> (
             match Ppx_yojson_conv_lib.( ! ) items_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson ConfigurationItem.t_of_yojson _field_yojson
               in
               items_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) items_field with
           | Option.Some items_value -> { items = items_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) items_field)
                     Option.None,
                   "items" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { items = v_items } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list ConfigurationItem.yojson_of_t v_items in
         ("items", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(items : ConfigurationItem.t list) : t = { items }
end

module FileCreate = struct
  type t = { uri : string }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.FileCreate.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let uri_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) uri_field with
           | Option.Some uri_value -> { uri = uri_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { uri = v_uri } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_string v_uri in
         ("uri", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(uri : string) : t = { uri }
end

module CreateFilesParams = struct
  type t = { files : FileCreate.t list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.CreateFilesParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let files_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "files" -> (
             match Ppx_yojson_conv_lib.( ! ) files_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson FileCreate.t_of_yojson _field_yojson
               in
               files_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) files_field with
           | Option.Some files_value -> { files = files_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) files_field)
                     Option.None,
                   "files" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { files = v_files } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list FileCreate.yojson_of_t v_files in
         ("files", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(files : FileCreate.t list) : t = { files }
end

module Declaration = struct
  type t = [ `Location of Location.t | `List of Location.t list ]

  let t_of_yojson (json : Json.t) : t =
    Json.Of.untagged_union "t"
      [ (fun json -> `Location (Location.t_of_yojson json));
        (fun json -> `List (Json.Of.list Location.t_of_yojson json))
      ]
      json

  let yojson_of_t (t : t) : Json.t =
    match t with
    | `Location s -> Location.yojson_of_t s
    | `List s -> Json.To.list Location.yojson_of_t s
end

module LocationLink = struct
  type t =
    { originSelectionRange : Range.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      targetRange : Range.t;
      targetSelectionRange : Range.t;
      targetUri : DocumentUri.t
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.LocationLink.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let originSelectionRange_field = ref Option.None
       and targetRange_field = ref Option.None
       and targetSelectionRange_field = ref Option.None
       and targetUri_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "originSelectionRange" -> (
             match Ppx_yojson_conv_lib.( ! ) originSelectionRange_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson Range.t_of_yojson
                   _field_yojson
               in
               originSelectionRange_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "targetRange" -> (
             match Ppx_yojson_conv_lib.( ! ) targetRange_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               targetRange_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "targetSelectionRange" -> (
             match Ppx_yojson_conv_lib.( ! ) targetSelectionRange_field with
             | Option.None ->
               let fvalue = Range.t_of_yojson _field_yojson in
               targetSelectionRange_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "targetUri" -> (
             match Ppx_yojson_conv_lib.( ! ) targetUri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               targetUri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) originSelectionRange_field,
               Ppx_yojson_conv_lib.( ! ) targetRange_field,
               Ppx_yojson_conv_lib.( ! ) targetSelectionRange_field,
               Ppx_yojson_conv_lib.( ! ) targetUri_field )
           with
           | ( originSelectionRange_value,
               Option.Some targetRange_value,
               Option.Some targetSelectionRange_value,
               Option.Some targetUri_value ) ->
             { originSelectionRange =
                 (match originSelectionRange_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               targetRange = targetRange_value;
               targetSelectionRange = targetSelectionRange_value;
               targetUri = targetUri_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) targetRange_field)
                     Option.None,
                   "targetRange" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) targetSelectionRange_field)
                     Option.None,
                   "targetSelectionRange" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) targetUri_field)
                     Option.None,
                   "targetUri" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { originSelectionRange = v_originSelectionRange;
         targetRange = v_targetRange;
         targetSelectionRange = v_targetSelectionRange;
         targetUri = v_targetUri
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_targetUri in
         ("targetUri", arg) :: bnds
       in
       let bnds =
         let arg = Range.yojson_of_t v_targetSelectionRange in
         ("targetSelectionRange", arg) :: bnds
       in
       let bnds =
         let arg = Range.yojson_of_t v_targetRange in
         ("targetRange", arg) :: bnds
       in
       let bnds =
         if None = v_originSelectionRange then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t Range.yojson_of_t)
               v_originSelectionRange
           in
           let bnd = ("originSelectionRange", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(originSelectionRange : Range.t option) ~(targetRange : Range.t)
      ~(targetSelectionRange : Range.t) ~(targetUri : DocumentUri.t) (() : unit)
      : t =
    { originSelectionRange; targetRange; targetSelectionRange; targetUri }
end

module DeclarationLink = struct
  type t = LocationLink.t [@@deriving_inline yojson]

  let _ = fun (_ : t) -> ()
  let t_of_yojson =
    (LocationLink.t_of_yojson : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)
  let _ = t_of_yojson
  let yojson_of_t =
    (LocationLink.yojson_of_t : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)
  let _ = yojson_of_t

  [@@@end]
end

module DeclarationOptions = struct
  type t =
    { workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DeclarationOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let workDoneProgress_value =
             Ppx_yojson_conv_lib.( ! ) workDoneProgress_field
           in
           { workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { workDoneProgress = v_workDoneProgress } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(workDoneProgress : bool option) (() : unit) : t =
    { workDoneProgress }
end

module DeclarationParams = struct
  type t =
    { partialResultToken : ProgressToken.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      position : Position.t;
      textDocument : TextDocumentIdentifier.t;
      workDoneToken : ProgressToken.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DeclarationParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let partialResultToken_field = ref Option.None
       and position_field = ref Option.None
       and textDocument_field = ref Option.None
       and workDoneToken_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "partialResultToken" -> (
             match Ppx_yojson_conv_lib.( ! ) partialResultToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               partialResultToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "position" -> (
             match Ppx_yojson_conv_lib.( ! ) position_field with
             | Option.None ->
               let fvalue = Position.t_of_yojson _field_yojson in
               position_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue = TextDocumentIdentifier.t_of_yojson _field_yojson in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneToken" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               workDoneToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) partialResultToken_field,
               Ppx_yojson_conv_lib.( ! ) position_field,
               Ppx_yojson_conv_lib.( ! ) textDocument_field,
               Ppx_yojson_conv_lib.( ! ) workDoneToken_field )
           with
           | ( partialResultToken_value,
               Option.Some position_value,
               Option.Some textDocument_value,
               workDoneToken_value ) ->
             { partialResultToken =
                 (match partialResultToken_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               position = position_value;
               textDocument = textDocument_value;
               workDoneToken =
                 (match workDoneToken_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) position_field)
                     Option.None,
                   "position" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { partialResultToken = v_partialResultToken;
         position = v_position;
         textDocument = v_textDocument;
         workDoneToken = v_workDoneToken
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_workDoneToken
           in
           let bnd = ("workDoneToken", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = TextDocumentIdentifier.yojson_of_t v_textDocument in
         ("textDocument", arg) :: bnds
       in
       let bnds =
         let arg = Position.yojson_of_t v_position in
         ("position", arg) :: bnds
       in
       let bnds =
         if None = v_partialResultToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_partialResultToken
           in
           let bnd = ("partialResultToken", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(partialResultToken : ProgressToken.t option)
      ~(position : Position.t) ~(textDocument : TextDocumentIdentifier.t)
      ?(workDoneToken : ProgressToken.t option) (() : unit) : t =
    { partialResultToken; position; textDocument; workDoneToken }
end

module DeclarationRegistrationOptions = struct
  type t =
    { documentSelector : DocumentSelector.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      id : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DeclarationRegistrationOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let documentSelector_field = ref Option.None
       and id_field = ref Option.None
       and workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "documentSelector" -> (
             match Ppx_yojson_conv_lib.( ! ) documentSelector_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson DocumentSelector.t_of_yojson
                   _field_yojson
               in
               documentSelector_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "id" -> (
             match Ppx_yojson_conv_lib.( ! ) id_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               id_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let documentSelector_value, id_value, workDoneProgress_value =
             ( Ppx_yojson_conv_lib.( ! ) documentSelector_field,
               Ppx_yojson_conv_lib.( ! ) id_field,
               Ppx_yojson_conv_lib.( ! ) workDoneProgress_field )
           in
           { documentSelector =
               (match documentSelector_value with
               | Option.None -> None
               | Option.Some v -> v);
             id =
               (match id_value with
               | Option.None -> None
               | Option.Some v -> v);
             workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { documentSelector = v_documentSelector;
         id = v_id;
         workDoneProgress = v_workDoneProgress
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_id then bnds
         else
           let arg = (Json.Nullable_option.yojson_of_t yojson_of_string) v_id in
           let bnd = ("id", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentSelector then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t DocumentSelector.yojson_of_t)
               v_documentSelector
           in
           let bnd = ("documentSelector", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(documentSelector : DocumentSelector.t option)
      ?(id : string option) ?(workDoneProgress : bool option) (() : unit) : t =
    { documentSelector; id; workDoneProgress }
end

module Definition = struct
  type t = [ `Location of Location.t | `List of Location.t list ]

  let t_of_yojson (json : Json.t) : t =
    Json.Of.untagged_union "t"
      [ (fun json -> `Location (Location.t_of_yojson json));
        (fun json -> `List (Json.Of.list Location.t_of_yojson json))
      ]
      json

  let yojson_of_t (t : t) : Json.t =
    match t with
    | `Location s -> Location.yojson_of_t s
    | `List s -> Json.To.list Location.yojson_of_t s
end

module DefinitionLink = struct
  type t = LocationLink.t [@@deriving_inline yojson]

  let _ = fun (_ : t) -> ()
  let t_of_yojson =
    (LocationLink.t_of_yojson : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)
  let _ = t_of_yojson
  let yojson_of_t =
    (LocationLink.yojson_of_t : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)
  let _ = yojson_of_t

  [@@@end]
end

module DefinitionOptions = struct
  type t =
    { workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DefinitionOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let workDoneProgress_value =
             Ppx_yojson_conv_lib.( ! ) workDoneProgress_field
           in
           { workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { workDoneProgress = v_workDoneProgress } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(workDoneProgress : bool option) (() : unit) : t =
    { workDoneProgress }
end

module DefinitionParams = struct
  type t =
    { partialResultToken : ProgressToken.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      position : Position.t;
      textDocument : TextDocumentIdentifier.t;
      workDoneToken : ProgressToken.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DefinitionParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let partialResultToken_field = ref Option.None
       and position_field = ref Option.None
       and textDocument_field = ref Option.None
       and workDoneToken_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "partialResultToken" -> (
             match Ppx_yojson_conv_lib.( ! ) partialResultToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               partialResultToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "position" -> (
             match Ppx_yojson_conv_lib.( ! ) position_field with
             | Option.None ->
               let fvalue = Position.t_of_yojson _field_yojson in
               position_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue = TextDocumentIdentifier.t_of_yojson _field_yojson in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneToken" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               workDoneToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) partialResultToken_field,
               Ppx_yojson_conv_lib.( ! ) position_field,
               Ppx_yojson_conv_lib.( ! ) textDocument_field,
               Ppx_yojson_conv_lib.( ! ) workDoneToken_field )
           with
           | ( partialResultToken_value,
               Option.Some position_value,
               Option.Some textDocument_value,
               workDoneToken_value ) ->
             { partialResultToken =
                 (match partialResultToken_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               position = position_value;
               textDocument = textDocument_value;
               workDoneToken =
                 (match workDoneToken_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) position_field)
                     Option.None,
                   "position" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { partialResultToken = v_partialResultToken;
         position = v_position;
         textDocument = v_textDocument;
         workDoneToken = v_workDoneToken
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_workDoneToken
           in
           let bnd = ("workDoneToken", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = TextDocumentIdentifier.yojson_of_t v_textDocument in
         ("textDocument", arg) :: bnds
       in
       let bnds =
         let arg = Position.yojson_of_t v_position in
         ("position", arg) :: bnds
       in
       let bnds =
         if None = v_partialResultToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_partialResultToken
           in
           let bnd = ("partialResultToken", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(partialResultToken : ProgressToken.t option)
      ~(position : Position.t) ~(textDocument : TextDocumentIdentifier.t)
      ?(workDoneToken : ProgressToken.t option) (() : unit) : t =
    { partialResultToken; position; textDocument; workDoneToken }
end

module DefinitionRegistrationOptions = struct
  type t =
    { documentSelector : DocumentSelector.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DefinitionRegistrationOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let documentSelector_field = ref Option.None
       and workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "documentSelector" -> (
             match Ppx_yojson_conv_lib.( ! ) documentSelector_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson DocumentSelector.t_of_yojson
                   _field_yojson
               in
               documentSelector_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let documentSelector_value, workDoneProgress_value =
             ( Ppx_yojson_conv_lib.( ! ) documentSelector_field,
               Ppx_yojson_conv_lib.( ! ) workDoneProgress_field )
           in
           { documentSelector =
               (match documentSelector_value with
               | Option.None -> None
               | Option.Some v -> v);
             workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { documentSelector = v_documentSelector;
         workDoneProgress = v_workDoneProgress
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentSelector then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t DocumentSelector.yojson_of_t)
               v_documentSelector
           in
           let bnd = ("documentSelector", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(documentSelector : DocumentSelector.t option)
      ?(workDoneProgress : bool option) (() : unit) : t =
    { documentSelector; workDoneProgress }
end

module FileDelete = struct
  type t = { uri : string }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.FileDelete.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let uri_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) uri_field with
           | Option.Some uri_value -> { uri = uri_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { uri = v_uri } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_string v_uri in
         ("uri", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(uri : string) : t = { uri }
end

module DeleteFilesParams = struct
  type t = { files : FileDelete.t list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DeleteFilesParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let files_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "files" -> (
             match Ppx_yojson_conv_lib.( ! ) files_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson FileDelete.t_of_yojson _field_yojson
               in
               files_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) files_field with
           | Option.Some files_value -> { files = files_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) files_field)
                     Option.None,
                   "files" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { files = v_files } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list FileDelete.yojson_of_t v_files in
         ("files", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(files : FileDelete.t list) : t = { files }
end

module DiagnosticOptions = struct
  type t =
    { identifier : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      interFileDependencies : bool;
      workDoneProgress : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workspaceDiagnostics : bool
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DiagnosticOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let identifier_field = ref Option.None
       and interFileDependencies_field = ref Option.None
       and workDoneProgress_field = ref Option.None
       and workspaceDiagnostics_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "identifier" -> (
             match Ppx_yojson_conv_lib.( ! ) identifier_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               identifier_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "interFileDependencies" -> (
             match Ppx_yojson_conv_lib.( ! ) interFileDependencies_field with
             | Option.None ->
               let fvalue = bool_of_yojson _field_yojson in
               interFileDependencies_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workspaceDiagnostics" -> (
             match Ppx_yojson_conv_lib.( ! ) workspaceDiagnostics_field with
             | Option.None ->
               let fvalue = bool_of_yojson _field_yojson in
               workspaceDiagnostics_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) identifier_field,
               Ppx_yojson_conv_lib.( ! ) interFileDependencies_field,
               Ppx_yojson_conv_lib.( ! ) workDoneProgress_field,
               Ppx_yojson_conv_lib.( ! ) workspaceDiagnostics_field )
           with
           | ( identifier_value,
               Option.Some interFileDependencies_value,
               workDoneProgress_value,
               Option.Some workspaceDiagnostics_value ) ->
             { identifier =
                 (match identifier_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               interFileDependencies = interFileDependencies_value;
               workDoneProgress =
                 (match workDoneProgress_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               workspaceDiagnostics = workspaceDiagnostics_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) interFileDependencies_field)
                     Option.None,
                   "interFileDependencies" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) workspaceDiagnostics_field)
                     Option.None,
                   "workspaceDiagnostics" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { identifier = v_identifier;
         interFileDependencies = v_interFileDependencies;
         workDoneProgress = v_workDoneProgress;
         workspaceDiagnostics = v_workspaceDiagnostics
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_bool v_workspaceDiagnostics in
         ("workspaceDiagnostics", arg) :: bnds
       in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_bool v_interFileDependencies in
         ("interFileDependencies", arg) :: bnds
       in
       let bnds =
         if None = v_identifier then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_identifier
           in
           let bnd = ("identifier", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(identifier : string option) ~(interFileDependencies : bool)
      ?(workDoneProgress : bool option) ~(workspaceDiagnostics : bool)
      (() : unit) : t =
    { identifier;
      interFileDependencies;
      workDoneProgress;
      workspaceDiagnostics
    }
end

module DiagnosticRegistrationOptions = struct
  type t =
    { documentSelector : DocumentSelector.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      id : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      identifier : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      interFileDependencies : bool;
      workDoneProgress : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workspaceDiagnostics : bool
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DiagnosticRegistrationOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let documentSelector_field = ref Option.None
       and id_field = ref Option.None
       and identifier_field = ref Option.None
       and interFileDependencies_field = ref Option.None
       and workDoneProgress_field = ref Option.None
       and workspaceDiagnostics_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "documentSelector" -> (
             match Ppx_yojson_conv_lib.( ! ) documentSelector_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson DocumentSelector.t_of_yojson
                   _field_yojson
               in
               documentSelector_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "id" -> (
             match Ppx_yojson_conv_lib.( ! ) id_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               id_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "identifier" -> (
             match Ppx_yojson_conv_lib.( ! ) identifier_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               identifier_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "interFileDependencies" -> (
             match Ppx_yojson_conv_lib.( ! ) interFileDependencies_field with
             | Option.None ->
               let fvalue = bool_of_yojson _field_yojson in
               interFileDependencies_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workspaceDiagnostics" -> (
             match Ppx_yojson_conv_lib.( ! ) workspaceDiagnostics_field with
             | Option.None ->
               let fvalue = bool_of_yojson _field_yojson in
               workspaceDiagnostics_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) documentSelector_field,
               Ppx_yojson_conv_lib.( ! ) id_field,
               Ppx_yojson_conv_lib.( ! ) identifier_field,
               Ppx_yojson_conv_lib.( ! ) interFileDependencies_field,
               Ppx_yojson_conv_lib.( ! ) workDoneProgress_field,
               Ppx_yojson_conv_lib.( ! ) workspaceDiagnostics_field )
           with
           | ( documentSelector_value,
               id_value,
               identifier_value,
               Option.Some interFileDependencies_value,
               workDoneProgress_value,
               Option.Some workspaceDiagnostics_value ) ->
             { documentSelector =
                 (match documentSelector_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               id =
                 (match id_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               identifier =
                 (match identifier_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               interFileDependencies = interFileDependencies_value;
               workDoneProgress =
                 (match workDoneProgress_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               workspaceDiagnostics = workspaceDiagnostics_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) interFileDependencies_field)
                     Option.None,
                   "interFileDependencies" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) workspaceDiagnostics_field)
                     Option.None,
                   "workspaceDiagnostics" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { documentSelector = v_documentSelector;
         id = v_id;
         identifier = v_identifier;
         interFileDependencies = v_interFileDependencies;
         workDoneProgress = v_workDoneProgress;
         workspaceDiagnostics = v_workspaceDiagnostics
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_bool v_workspaceDiagnostics in
         ("workspaceDiagnostics", arg) :: bnds
       in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_bool v_interFileDependencies in
         ("interFileDependencies", arg) :: bnds
       in
       let bnds =
         if None = v_identifier then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_identifier
           in
           let bnd = ("identifier", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_id then bnds
         else
           let arg = (Json.Nullable_option.yojson_of_t yojson_of_string) v_id in
           let bnd = ("id", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentSelector then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t DocumentSelector.yojson_of_t)
               v_documentSelector
           in
           let bnd = ("documentSelector", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(documentSelector : DocumentSelector.t option)
      ?(id : string option) ?(identifier : string option)
      ~(interFileDependencies : bool) ?(workDoneProgress : bool option)
      ~(workspaceDiagnostics : bool) (() : unit) : t =
    { documentSelector;
      id;
      identifier;
      interFileDependencies;
      workDoneProgress;
      workspaceDiagnostics
    }
end

module DiagnosticServerCancellationData = struct
  type t = { retriggerRequest : bool }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DiagnosticServerCancellationData.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let retriggerRequest_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "retriggerRequest" -> (
             match Ppx_yojson_conv_lib.( ! ) retriggerRequest_field with
             | Option.None ->
               let fvalue = bool_of_yojson _field_yojson in
               retriggerRequest_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) retriggerRequest_field with
           | Option.Some retriggerRequest_value ->
             { retriggerRequest = retriggerRequest_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) retriggerRequest_field)
                     Option.None,
                   "retriggerRequest" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { retriggerRequest = v_retriggerRequest } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_bool v_retriggerRequest in
         ("retriggerRequest", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(retriggerRequest : bool) : t = { retriggerRequest }
end

module DidChangeConfigurationParams = struct
  type t = { settings : Json.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DidChangeConfigurationParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let settings_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "settings" -> (
             match Ppx_yojson_conv_lib.( ! ) settings_field with
             | Option.None ->
               let fvalue = Json.t_of_yojson _field_yojson in
               settings_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) settings_field with
           | Option.Some settings_value -> { settings = settings_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) settings_field)
                     Option.None,
                   "settings" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { settings = v_settings } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = Json.yojson_of_t v_settings in
         ("settings", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(settings : Json.t) : t = { settings }
end

module DidChangeConfigurationRegistrationOptions = struct
  type section_pvar = [ `String of string | `List of string list ]

  let section_pvar_of_yojson (json : Json.t) : section_pvar =
    match json with
    | `String j -> `String j
    | _ ->
      Json.Of.untagged_union "section_pvar"
        [ (fun json -> `List (Json.Of.list string_of_yojson json)) ]
        json

  let yojson_of_section_pvar (section_pvar : section_pvar) : Json.t =
    match section_pvar with
    | `String j -> `String j
    | `List s -> Json.To.list yojson_of_string s

  type t =
    { section : section_pvar Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.DidChangeConfigurationRegistrationOptions.t"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let section_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "section" -> (
             match Ppx_yojson_conv_lib.( ! ) section_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson section_pvar_of_yojson
                   _field_yojson
               in
               section_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let section_value = Ppx_yojson_conv_lib.( ! ) section_field in
           { section =
               (match section_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { section = v_section } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_section then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_section_pvar) v_section
           in
           let bnd = ("section", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(section : section_pvar option) (() : unit) : t = { section }
end

module VersionedNotebookDocumentIdentifier = struct
  type t = { uri : DocumentUri.t; version : int }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.VersionedNotebookDocumentIdentifier.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let uri_field = ref Option.None
       and version_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "version" -> (
             match Ppx_yojson_conv_lib.( ! ) version_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               version_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) uri_field,
               Ppx_yojson_conv_lib.( ! ) version_field )
           with
           | Option.Some uri_value, Option.Some version_value ->
             { uri = uri_value; version = version_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) version_field)
                     Option.None,
                   "version" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { uri = v_uri; version = v_version } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_int v_version in
         ("version", arg) :: bnds
       in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_uri in
         ("uri", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(uri : DocumentUri.t) ~(version : int) : t = { uri; version }
end

module TextDocumentContentChangeEvent = struct
  type t =
    { range : Range.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      rangeLength : int Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      text : string
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.TextDocumentContentChangeEvent.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let range_field = ref Option.None
       and rangeLength_field = ref Option.None
       and text_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "range" -> (
             match Ppx_yojson_conv_lib.( ! ) range_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson Range.t_of_yojson
                   _field_yojson
               in
               range_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "rangeLength" -> (
             match Ppx_yojson_conv_lib.( ! ) rangeLength_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson int_of_yojson _field_yojson
               in
               rangeLength_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "text" -> (
             match Ppx_yojson_conv_lib.( ! ) text_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               text_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) range_field,
               Ppx_yojson_conv_lib.( ! ) rangeLength_field,
               Ppx_yojson_conv_lib.( ! ) text_field )
           with
           | range_value, rangeLength_value, Option.Some text_value ->
             { range =
                 (match range_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               rangeLength =
                 (match rangeLength_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               text = text_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) text_field)
                     Option.None,
                   "text" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { range = v_range; rangeLength = v_rangeLength; text = v_text } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_string v_text in
         ("text", arg) :: bnds
       in
       let bnds =
         if None = v_rangeLength then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_int) v_rangeLength
           in
           let bnd = ("rangeLength", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_range then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t Range.yojson_of_t) v_range
           in
           let bnd = ("range", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(range : Range.t option) ?(rangeLength : int option)
      ~(text : string) (() : unit) : t =
    { range; rangeLength; text }
end

module VersionedTextDocumentIdentifier = struct
  type t = { uri : DocumentUri.t; version : int }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.VersionedTextDocumentIdentifier.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let uri_field = ref Option.None
       and version_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "version" -> (
             match Ppx_yojson_conv_lib.( ! ) version_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               version_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) uri_field,
               Ppx_yojson_conv_lib.( ! ) version_field )
           with
           | Option.Some uri_value, Option.Some version_value ->
             { uri = uri_value; version = version_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) version_field)
                     Option.None,
                   "version" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { uri = v_uri; version = v_version } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_int v_version in
         ("version", arg) :: bnds
       in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_uri in
         ("uri", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(uri : DocumentUri.t) ~(version : int) : t = { uri; version }
end

module ExecutionSummary = struct
  type t =
    { executionOrder : int;
      success : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.ExecutionSummary.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let executionOrder_field = ref Option.None
       and success_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "executionOrder" -> (
             match Ppx_yojson_conv_lib.( ! ) executionOrder_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               executionOrder_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "success" -> (
             match Ppx_yojson_conv_lib.( ! ) success_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               success_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) executionOrder_field,
               Ppx_yojson_conv_lib.( ! ) success_field )
           with
           | Option.Some executionOrder_value, success_value ->
             { executionOrder = executionOrder_value;
               success =
                 (match success_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) executionOrder_field)
                     Option.None,
                   "executionOrder" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { executionOrder = v_executionOrder; success = v_success } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_success then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool) v_success
           in
           let bnd = ("success", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_int v_executionOrder in
         ("executionOrder", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(executionOrder : int) ?(success : bool option) (() : unit) : t =
    { executionOrder; success }
end

module NotebookCell = struct
  type t =
    { document : DocumentUri.t;
      executionSummary : ExecutionSummary.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      kind : NotebookCellKind.t;
      metadata : Json.Object.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.NotebookCell.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let document_field = ref Option.None
       and executionSummary_field = ref Option.None
       and kind_field = ref Option.None
       and metadata_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "document" -> (
             match Ppx_yojson_conv_lib.( ! ) document_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               document_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "executionSummary" -> (
             match Ppx_yojson_conv_lib.( ! ) executionSummary_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ExecutionSummary.t_of_yojson
                   _field_yojson
               in
               executionSummary_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "kind" -> (
             match Ppx_yojson_conv_lib.( ! ) kind_field with
             | Option.None ->
               let fvalue = NotebookCellKind.t_of_yojson _field_yojson in
               kind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "metadata" -> (
             match Ppx_yojson_conv_lib.( ! ) metadata_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson Json.Object.t_of_yojson
                   _field_yojson
               in
               metadata_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) document_field,
               Ppx_yojson_conv_lib.( ! ) executionSummary_field,
               Ppx_yojson_conv_lib.( ! ) kind_field,
               Ppx_yojson_conv_lib.( ! ) metadata_field )
           with
           | ( Option.Some document_value,
               executionSummary_value,
               Option.Some kind_value,
               metadata_value ) ->
             { document = document_value;
               executionSummary =
                 (match executionSummary_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               kind = kind_value;
               metadata =
                 (match metadata_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) document_field)
                     Option.None,
                   "document" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) kind_field)
                     Option.None,
                   "kind" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { document = v_document;
         executionSummary = v_executionSummary;
         kind = v_kind;
         metadata = v_metadata
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_metadata then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t Json.Object.yojson_of_t)
               v_metadata
           in
           let bnd = ("metadata", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = NotebookCellKind.yojson_of_t v_kind in
         ("kind", arg) :: bnds
       in
       let bnds =
         if None = v_executionSummary then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ExecutionSummary.yojson_of_t)
               v_executionSummary
           in
           let bnd = ("executionSummary", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_document in
         ("document", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(document : DocumentUri.t)
      ?(executionSummary : ExecutionSummary.t option)
      ~(kind : NotebookCellKind.t) ?(metadata : Json.Object.t option)
      (() : unit) : t =
    { document; executionSummary; kind; metadata }
end

module TextDocumentItem = struct
  type t =
    { languageId : string; text : string; uri : DocumentUri.t; version : int }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.TextDocumentItem.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let languageId_field = ref Option.None
       and text_field = ref Option.None
       and uri_field = ref Option.None
       and version_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "languageId" -> (
             match Ppx_yojson_conv_lib.( ! ) languageId_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               languageId_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "text" -> (
             match Ppx_yojson_conv_lib.( ! ) text_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               text_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "version" -> (
             match Ppx_yojson_conv_lib.( ! ) version_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               version_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) languageId_field,
               Ppx_yojson_conv_lib.( ! ) text_field,
               Ppx_yojson_conv_lib.( ! ) uri_field,
               Ppx_yojson_conv_lib.( ! ) version_field )
           with
           | ( Option.Some languageId_value,
               Option.Some text_value,
               Option.Some uri_value,
               Option.Some version_value ) ->
             { languageId = languageId_value;
               text = text_value;
               uri = uri_value;
               version = version_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) languageId_field)
                     Option.None,
                   "languageId" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) text_field)
                     Option.None,
                   "text" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) version_field)
                     Option.None,
                   "version" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { languageId = v_languageId;
         text = v_text;
         uri = v_uri;
         version = v_version
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_int v_version in
         ("version", arg) :: bnds
       in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_uri in
         ("uri", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_text in
         ("text", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_languageId in
         ("languageId", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(languageId : string) ~(text : string) ~(uri : DocumentUri.t)
      ~(version : int) : t =
    { languageId; text; uri; version }
end

module NotebookCellArrayChange = struct
  type t =
    { cells : NotebookCell.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      deleteCount : int;
      start : int
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.NotebookCellArrayChange.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let cells_field = ref Option.None
       and deleteCount_field = ref Option.None
       and start_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "cells" -> (
             match Ppx_yojson_conv_lib.( ! ) cells_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson NotebookCell.t_of_yojson)
                   _field_yojson
               in
               cells_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "deleteCount" -> (
             match Ppx_yojson_conv_lib.( ! ) deleteCount_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               deleteCount_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "start" -> (
             match Ppx_yojson_conv_lib.( ! ) start_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               start_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) cells_field,
               Ppx_yojson_conv_lib.( ! ) deleteCount_field,
               Ppx_yojson_conv_lib.( ! ) start_field )
           with
           | cells_value, Option.Some deleteCount_value, Option.Some start_value
             ->
             { cells =
                 (match cells_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               deleteCount = deleteCount_value;
               start = start_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) deleteCount_field)
                     Option.None,
                   "deleteCount" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) start_field)
                     Option.None,
                   "start" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { cells = v_cells; deleteCount = v_deleteCount; start = v_start } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_int v_start in
         ("start", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_int v_deleteCount in
         ("deleteCount", arg) :: bnds
       in
       let bnds =
         if None = v_cells then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list NotebookCell.yojson_of_t))
               v_cells
           in
           let bnd = ("cells", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(cells : NotebookCell.t list option) ~(deleteCount : int)
      ~(start : int) (() : unit) : t =
    { cells; deleteCount; start }
end

module NotebookDocumentChangeEvent = struct
  type textContent =
    { document : VersionedTextDocumentIdentifier.t;
      changes : TextDocumentContentChangeEvent.t list
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : textContent) -> ()

  let textContent_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.NotebookDocumentChangeEvent.textContent" in
     function
     | `Assoc field_yojsons as yojson -> (
       let document_field = ref Option.None
       and changes_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "document" -> (
             match Ppx_yojson_conv_lib.( ! ) document_field with
             | Option.None ->
               let fvalue =
                 VersionedTextDocumentIdentifier.t_of_yojson _field_yojson
               in
               document_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "changes" -> (
             match Ppx_yojson_conv_lib.( ! ) changes_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson TextDocumentContentChangeEvent.t_of_yojson
                   _field_yojson
               in
               changes_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) document_field,
               Ppx_yojson_conv_lib.( ! ) changes_field )
           with
           | Option.Some document_value, Option.Some changes_value ->
             { document = document_value; changes = changes_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) document_field)
                     Option.None,
                   "document" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) changes_field)
                     Option.None,
                   "changes" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> textContent)

  let _ = textContent_of_yojson

  let yojson_of_textContent =
    (function
     | { document = v_document; changes = v_changes } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg =
           yojson_of_list TextDocumentContentChangeEvent.yojson_of_t v_changes
         in
         ("changes", arg) :: bnds
       in
       let bnds =
         let arg = VersionedTextDocumentIdentifier.yojson_of_t v_document in
         ("document", arg) :: bnds
       in
       `Assoc bnds
      : textContent -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_textContent

  [@@@end]

  let create_textContent ~(document : VersionedTextDocumentIdentifier.t)
      ~(changes : TextDocumentContentChangeEvent.t list) : textContent =
    { document; changes }

  type structure =
    { array : NotebookCellArrayChange.t;
      didOpen : TextDocumentItem.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      didClose : TextDocumentIdentifier.t list Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : structure) -> ()

  let structure_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.NotebookDocumentChangeEvent.structure" in
     function
     | `Assoc field_yojsons as yojson -> (
       let array_field = ref Option.None
       and didOpen_field = ref Option.None
       and didClose_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "array" -> (
             match Ppx_yojson_conv_lib.( ! ) array_field with
             | Option.None ->
               let fvalue = NotebookCellArrayChange.t_of_yojson _field_yojson in
               array_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "didOpen" -> (
             match Ppx_yojson_conv_lib.( ! ) didOpen_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson TextDocumentItem.t_of_yojson)
                   _field_yojson
               in
               didOpen_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "didClose" -> (
             match Ppx_yojson_conv_lib.( ! ) didClose_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson TextDocumentIdentifier.t_of_yojson)
                   _field_yojson
               in
               didClose_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) array_field,
               Ppx_yojson_conv_lib.( ! ) didOpen_field,
               Ppx_yojson_conv_lib.( ! ) didClose_field )
           with
           | Option.Some array_value, didOpen_value, didClose_value ->
             { array = array_value;
               didOpen =
                 (match didOpen_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               didClose =
                 (match didClose_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) array_field)
                     Option.None,
                   "array" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> structure)

  let _ = structure_of_yojson

  let yojson_of_structure =
    (function
     | { array = v_array; didOpen = v_didOpen; didClose = v_didClose } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_didClose then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list TextDocumentIdentifier.yojson_of_t))
               v_didClose
           in
           let bnd = ("didClose", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_didOpen then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list TextDocumentItem.yojson_of_t))
               v_didOpen
           in
           let bnd = ("didOpen", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = NotebookCellArrayChange.yojson_of_t v_array in
         ("array", arg) :: bnds
       in
       `Assoc bnds
      : structure -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_structure

  [@@@end]

  let create_structure ~(array : NotebookCellArrayChange.t)
      ?(didOpen : TextDocumentItem.t list option)
      ?(didClose : TextDocumentIdentifier.t list option) (() : unit) : structure
      =
    { array; didOpen; didClose }

  type cells =
    { structure : structure Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      data : NotebookCell.t list Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      textContent : textContent list Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : cells) -> ()

  let cells_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.NotebookDocumentChangeEvent.cells" in
     function
     | `Assoc field_yojsons as yojson -> (
       let structure_field = ref Option.None
       and data_field = ref Option.None
       and textContent_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "structure" -> (
             match Ppx_yojson_conv_lib.( ! ) structure_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson structure_of_yojson
                   _field_yojson
               in
               structure_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "data" -> (
             match Ppx_yojson_conv_lib.( ! ) data_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson NotebookCell.t_of_yojson)
                   _field_yojson
               in
               data_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textContent" -> (
             match Ppx_yojson_conv_lib.( ! ) textContent_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (list_of_yojson textContent_of_yojson)
                   _field_yojson
               in
               textContent_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let structure_value, data_value, textContent_value =
             ( Ppx_yojson_conv_lib.( ! ) structure_field,
               Ppx_yojson_conv_lib.( ! ) data_field,
               Ppx_yojson_conv_lib.( ! ) textContent_field )
           in
           { structure =
               (match structure_value with
               | Option.None -> None
               | Option.Some v -> v);
             data =
               (match data_value with
               | Option.None -> None
               | Option.Some v -> v);
             textContent =
               (match textContent_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> cells)

  let _ = cells_of_yojson

  let yojson_of_cells =
    (function
     | { structure = v_structure; data = v_data; textContent = v_textContent }
       ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_textContent then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list yojson_of_textContent))
               v_textContent
           in
           let bnd = ("textContent", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_data then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (yojson_of_list NotebookCell.yojson_of_t))
               v_data
           in
           let bnd = ("data", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_structure then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_structure) v_structure
           in
           let bnd = ("structure", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : cells -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_cells

  [@@@end]

  let create_cells ?(structure : structure option)
      ?(data : NotebookCell.t list option)
      ?(textContent : textContent list option) (() : unit) : cells =
    { structure; data; textContent }

  type t =
    { cells : cells Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      metadata : Json.Object.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.NotebookDocumentChangeEvent.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let cells_field = ref Option.None
       and metadata_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "cells" -> (
             match Ppx_yojson_conv_lib.( ! ) cells_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson cells_of_yojson _field_yojson
               in
               cells_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "metadata" -> (
             match Ppx_yojson_conv_lib.( ! ) metadata_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson Json.Object.t_of_yojson
                   _field_yojson
               in
               metadata_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let cells_value, metadata_value =
             ( Ppx_yojson_conv_lib.( ! ) cells_field,
               Ppx_yojson_conv_lib.( ! ) metadata_field )
           in
           { cells =
               (match cells_value with
               | Option.None -> None
               | Option.Some v -> v);
             metadata =
               (match metadata_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { cells = v_cells; metadata = v_metadata } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_metadata then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t Json.Object.yojson_of_t)
               v_metadata
           in
           let bnd = ("metadata", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_cells then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_cells) v_cells
           in
           let bnd = ("cells", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(cells : cells option) ?(metadata : Json.Object.t option)
      (() : unit) : t =
    { cells; metadata }
end

module DidChangeNotebookDocumentParams = struct
  type t =
    { change : NotebookDocumentChangeEvent.t;
      notebookDocument : VersionedNotebookDocumentIdentifier.t
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DidChangeNotebookDocumentParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let change_field = ref Option.None
       and notebookDocument_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "change" -> (
             match Ppx_yojson_conv_lib.( ! ) change_field with
             | Option.None ->
               let fvalue =
                 NotebookDocumentChangeEvent.t_of_yojson _field_yojson
               in
               change_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "notebookDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) notebookDocument_field with
             | Option.None ->
               let fvalue =
                 VersionedNotebookDocumentIdentifier.t_of_yojson _field_yojson
               in
               notebookDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) change_field,
               Ppx_yojson_conv_lib.( ! ) notebookDocument_field )
           with
           | Option.Some change_value, Option.Some notebookDocument_value ->
             { change = change_value;
               notebookDocument = notebookDocument_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) change_field)
                     Option.None,
                   "change" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) notebookDocument_field)
                     Option.None,
                   "notebookDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { change = v_change; notebookDocument = v_notebookDocument } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg =
           VersionedNotebookDocumentIdentifier.yojson_of_t v_notebookDocument
         in
         ("notebookDocument", arg) :: bnds
       in
       let bnds =
         let arg = NotebookDocumentChangeEvent.yojson_of_t v_change in
         ("change", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(change : NotebookDocumentChangeEvent.t)
      ~(notebookDocument : VersionedNotebookDocumentIdentifier.t) : t =
    { change; notebookDocument }
end

module DidChangeTextDocumentParams = struct
  type t =
    { contentChanges : TextDocumentContentChangeEvent.t list;
      textDocument : VersionedTextDocumentIdentifier.t
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DidChangeTextDocumentParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let contentChanges_field = ref Option.None
       and textDocument_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "contentChanges" -> (
             match Ppx_yojson_conv_lib.( ! ) contentChanges_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson TextDocumentContentChangeEvent.t_of_yojson
                   _field_yojson
               in
               contentChanges_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue =
                 VersionedTextDocumentIdentifier.t_of_yojson _field_yojson
               in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) contentChanges_field,
               Ppx_yojson_conv_lib.( ! ) textDocument_field )
           with
           | Option.Some contentChanges_value, Option.Some textDocument_value ->
             { contentChanges = contentChanges_value;
               textDocument = textDocument_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) contentChanges_field)
                     Option.None,
                   "contentChanges" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { contentChanges = v_contentChanges; textDocument = v_textDocument } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = VersionedTextDocumentIdentifier.yojson_of_t v_textDocument in
         ("textDocument", arg) :: bnds
       in
       let bnds =
         let arg =
           yojson_of_list TextDocumentContentChangeEvent.yojson_of_t
             v_contentChanges
         in
         ("contentChanges", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(contentChanges : TextDocumentContentChangeEvent.t list)
      ~(textDocument : VersionedTextDocumentIdentifier.t) : t =
    { contentChanges; textDocument }
end

module FileEvent = struct
  type t = { type_ : FileChangeType.t; [@key "type"] uri : DocumentUri.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.FileEvent.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let type__field = ref Option.None
       and uri_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "type" -> (
             match Ppx_yojson_conv_lib.( ! ) type__field with
             | Option.None ->
               let fvalue = FileChangeType.t_of_yojson _field_yojson in
               type__field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) type__field,
               Ppx_yojson_conv_lib.( ! ) uri_field )
           with
           | Option.Some type__value, Option.Some uri_value ->
             { type_ = type__value; uri = uri_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) type__field)
                     Option.None,
                   "type_" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { type_ = v_type_; uri = v_uri } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_uri in
         ("uri", arg) :: bnds
       in
       let bnds =
         let arg = FileChangeType.yojson_of_t v_type_ in
         ("type", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(type_ : FileChangeType.t) ~(uri : DocumentUri.t) : t =
    { type_; uri }
end

module DidChangeWatchedFilesParams = struct
  type t = { changes : FileEvent.t list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DidChangeWatchedFilesParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let changes_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "changes" -> (
             match Ppx_yojson_conv_lib.( ! ) changes_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson FileEvent.t_of_yojson _field_yojson
               in
               changes_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) changes_field with
           | Option.Some changes_value -> { changes = changes_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) changes_field)
                     Option.None,
                   "changes" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { changes = v_changes } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list FileEvent.yojson_of_t v_changes in
         ("changes", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(changes : FileEvent.t list) : t = { changes }
end

module Pattern = struct
  type t = string [@@deriving_inline yojson]

  let _ = fun (_ : t) -> ()
  let t_of_yojson = (string_of_yojson : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)
  let _ = t_of_yojson
  let yojson_of_t = (yojson_of_string : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)
  let _ = yojson_of_t

  [@@@end]
end

module WorkspaceFolder = struct
  type t = { name : string; uri : DocumentUri.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.WorkspaceFolder.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let name_field = ref Option.None
       and uri_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "name" -> (
             match Ppx_yojson_conv_lib.( ! ) name_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               name_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) name_field,
               Ppx_yojson_conv_lib.( ! ) uri_field )
           with
           | Option.Some name_value, Option.Some uri_value ->
             { name = name_value; uri = uri_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) name_field)
                     Option.None,
                   "name" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { name = v_name; uri = v_uri } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_uri in
         ("uri", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_name in
         ("name", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(name : string) ~(uri : DocumentUri.t) : t = { name; uri }
end

module RelativePattern = struct
  type t = { baseUri : unit; pattern : Pattern.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.RelativePattern.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let baseUri_field = ref Option.None
       and pattern_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "baseUri" -> (
             match Ppx_yojson_conv_lib.( ! ) baseUri_field with
             | Option.None ->
               let fvalue = unit_of_yojson _field_yojson in
               baseUri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "pattern" -> (
             match Ppx_yojson_conv_lib.( ! ) pattern_field with
             | Option.None ->
               let fvalue = Pattern.t_of_yojson _field_yojson in
               pattern_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) baseUri_field,
               Ppx_yojson_conv_lib.( ! ) pattern_field )
           with
           | Option.Some baseUri_value, Option.Some pattern_value ->
             { baseUri = baseUri_value; pattern = pattern_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) baseUri_field)
                     Option.None,
                   "baseUri" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) pattern_field)
                     Option.None,
                   "pattern" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { baseUri = v_baseUri; pattern = v_pattern } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = Pattern.yojson_of_t v_pattern in
         ("pattern", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_unit v_baseUri in
         ("baseUri", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(baseUri : unit) ~(pattern : Pattern.t) : t = { baseUri; pattern }
end

module GlobPattern = struct
  type t = [ `Pattern of Pattern.t | `RelativePattern of RelativePattern.t ]

  let t_of_yojson (json : Json.t) : t =
    Json.Of.untagged_union "t"
      [ (fun json -> `Pattern (Pattern.t_of_yojson json));
        (fun json -> `RelativePattern (RelativePattern.t_of_yojson json))
      ]
      json

  let yojson_of_t (t : t) : Json.t =
    match t with
    | `Pattern s -> Pattern.yojson_of_t s
    | `RelativePattern s -> RelativePattern.yojson_of_t s
end

module FileSystemWatcher = struct
  type t =
    { globPattern : GlobPattern.t;
      kind : WatchKind.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.FileSystemWatcher.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let globPattern_field = ref Option.None
       and kind_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "globPattern" -> (
             match Ppx_yojson_conv_lib.( ! ) globPattern_field with
             | Option.None ->
               let fvalue = GlobPattern.t_of_yojson _field_yojson in
               globPattern_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "kind" -> (
             match Ppx_yojson_conv_lib.( ! ) kind_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson WatchKind.t_of_yojson
                   _field_yojson
               in
               kind_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) globPattern_field,
               Ppx_yojson_conv_lib.( ! ) kind_field )
           with
           | Option.Some globPattern_value, kind_value ->
             { globPattern = globPattern_value;
               kind =
                 (match kind_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) globPattern_field)
                     Option.None,
                   "globPattern" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { globPattern = v_globPattern; kind = v_kind } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_kind then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t WatchKind.yojson_of_t) v_kind
           in
           let bnd = ("kind", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = GlobPattern.yojson_of_t v_globPattern in
         ("globPattern", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(globPattern : GlobPattern.t) ?(kind : WatchKind.t option)
      (() : unit) : t =
    { globPattern; kind }
end

module DidChangeWatchedFilesRegistrationOptions = struct
  type t = { watchers : FileSystemWatcher.t list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.DidChangeWatchedFilesRegistrationOptions.t"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let watchers_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "watchers" -> (
             match Ppx_yojson_conv_lib.( ! ) watchers_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson FileSystemWatcher.t_of_yojson _field_yojson
               in
               watchers_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) watchers_field with
           | Option.Some watchers_value -> { watchers = watchers_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) watchers_field)
                     Option.None,
                   "watchers" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { watchers = v_watchers } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list FileSystemWatcher.yojson_of_t v_watchers in
         ("watchers", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(watchers : FileSystemWatcher.t list) : t = { watchers }
end

module WorkspaceFoldersChangeEvent = struct
  type t = { added : WorkspaceFolder.t list; removed : WorkspaceFolder.t list }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.WorkspaceFoldersChangeEvent.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let added_field = ref Option.None
       and removed_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "added" -> (
             match Ppx_yojson_conv_lib.( ! ) added_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson WorkspaceFolder.t_of_yojson _field_yojson
               in
               added_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "removed" -> (
             match Ppx_yojson_conv_lib.( ! ) removed_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson WorkspaceFolder.t_of_yojson _field_yojson
               in
               removed_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) added_field,
               Ppx_yojson_conv_lib.( ! ) removed_field )
           with
           | Option.Some added_value, Option.Some removed_value ->
             { added = added_value; removed = removed_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) added_field)
                     Option.None,
                   "added" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) removed_field)
                     Option.None,
                   "removed" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { added = v_added; removed = v_removed } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_list WorkspaceFolder.yojson_of_t v_removed in
         ("removed", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_list WorkspaceFolder.yojson_of_t v_added in
         ("added", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(added : WorkspaceFolder.t list)
      ~(removed : WorkspaceFolder.t list) : t =
    { added; removed }
end

module DidChangeWorkspaceFoldersParams = struct
  type t = { event : WorkspaceFoldersChangeEvent.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DidChangeWorkspaceFoldersParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let event_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "event" -> (
             match Ppx_yojson_conv_lib.( ! ) event_field with
             | Option.None ->
               let fvalue =
                 WorkspaceFoldersChangeEvent.t_of_yojson _field_yojson
               in
               event_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) event_field with
           | Option.Some event_value -> { event = event_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) event_field)
                     Option.None,
                   "event" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { event = v_event } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = WorkspaceFoldersChangeEvent.yojson_of_t v_event in
         ("event", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(event : WorkspaceFoldersChangeEvent.t) : t = { event }
end

module NotebookDocumentIdentifier = struct
  type t = { uri : DocumentUri.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.NotebookDocumentIdentifier.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let uri_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) uri_field with
           | Option.Some uri_value -> { uri = uri_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { uri = v_uri } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_uri in
         ("uri", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(uri : DocumentUri.t) : t = { uri }
end

module DidCloseNotebookDocumentParams = struct
  type t =
    { cellTextDocuments : TextDocumentIdentifier.t list;
      notebookDocument : NotebookDocumentIdentifier.t
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DidCloseNotebookDocumentParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let cellTextDocuments_field = ref Option.None
       and notebookDocument_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "cellTextDocuments" -> (
             match Ppx_yojson_conv_lib.( ! ) cellTextDocuments_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson TextDocumentIdentifier.t_of_yojson _field_yojson
               in
               cellTextDocuments_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "notebookDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) notebookDocument_field with
             | Option.None ->
               let fvalue =
                 NotebookDocumentIdentifier.t_of_yojson _field_yojson
               in
               notebookDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) cellTextDocuments_field,
               Ppx_yojson_conv_lib.( ! ) notebookDocument_field )
           with
           | ( Option.Some cellTextDocuments_value,
               Option.Some notebookDocument_value ) ->
             { cellTextDocuments = cellTextDocuments_value;
               notebookDocument = notebookDocument_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) cellTextDocuments_field)
                     Option.None,
                   "cellTextDocuments" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) notebookDocument_field)
                     Option.None,
                   "notebookDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { cellTextDocuments = v_cellTextDocuments;
         notebookDocument = v_notebookDocument
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = NotebookDocumentIdentifier.yojson_of_t v_notebookDocument in
         ("notebookDocument", arg) :: bnds
       in
       let bnds =
         let arg =
           yojson_of_list TextDocumentIdentifier.yojson_of_t v_cellTextDocuments
         in
         ("cellTextDocuments", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(cellTextDocuments : TextDocumentIdentifier.t list)
      ~(notebookDocument : NotebookDocumentIdentifier.t) : t =
    { cellTextDocuments; notebookDocument }
end

module DidCloseTextDocumentParams = struct
  type t = { textDocument : TextDocumentIdentifier.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DidCloseTextDocumentParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let textDocument_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue = TextDocumentIdentifier.t_of_yojson _field_yojson in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) textDocument_field with
           | Option.Some textDocument_value ->
             { textDocument = textDocument_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { textDocument = v_textDocument } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = TextDocumentIdentifier.yojson_of_t v_textDocument in
         ("textDocument", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(textDocument : TextDocumentIdentifier.t) : t = { textDocument }
end

module NotebookDocument = struct
  type t =
    { cells : NotebookCell.t list;
      metadata : Json.Object.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      notebookType : string;
      uri : DocumentUri.t;
      version : int
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.NotebookDocument.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let cells_field = ref Option.None
       and metadata_field = ref Option.None
       and notebookType_field = ref Option.None
       and uri_field = ref Option.None
       and version_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "cells" -> (
             match Ppx_yojson_conv_lib.( ! ) cells_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson NotebookCell.t_of_yojson _field_yojson
               in
               cells_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "metadata" -> (
             match Ppx_yojson_conv_lib.( ! ) metadata_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson Json.Object.t_of_yojson
                   _field_yojson
               in
               metadata_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "notebookType" -> (
             match Ppx_yojson_conv_lib.( ! ) notebookType_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               notebookType_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "uri" -> (
             match Ppx_yojson_conv_lib.( ! ) uri_field with
             | Option.None ->
               let fvalue = DocumentUri.t_of_yojson _field_yojson in
               uri_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "version" -> (
             match Ppx_yojson_conv_lib.( ! ) version_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               version_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) cells_field,
               Ppx_yojson_conv_lib.( ! ) metadata_field,
               Ppx_yojson_conv_lib.( ! ) notebookType_field,
               Ppx_yojson_conv_lib.( ! ) uri_field,
               Ppx_yojson_conv_lib.( ! ) version_field )
           with
           | ( Option.Some cells_value,
               metadata_value,
               Option.Some notebookType_value,
               Option.Some uri_value,
               Option.Some version_value ) ->
             { cells = cells_value;
               metadata =
                 (match metadata_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               notebookType = notebookType_value;
               uri = uri_value;
               version = version_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) cells_field)
                     Option.None,
                   "cells" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) notebookType_field)
                     Option.None,
                   "notebookType" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) uri_field)
                     Option.None,
                   "uri" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) version_field)
                     Option.None,
                   "version" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { cells = v_cells;
         metadata = v_metadata;
         notebookType = v_notebookType;
         uri = v_uri;
         version = v_version
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_int v_version in
         ("version", arg) :: bnds
       in
       let bnds =
         let arg = DocumentUri.yojson_of_t v_uri in
         ("uri", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_string v_notebookType in
         ("notebookType", arg) :: bnds
       in
       let bnds =
         if None = v_metadata then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t Json.Object.yojson_of_t)
               v_metadata
           in
           let bnd = ("metadata", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_list NotebookCell.yojson_of_t v_cells in
         ("cells", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(cells : NotebookCell.t list) ?(metadata : Json.Object.t option)
      ~(notebookType : string) ~(uri : DocumentUri.t) ~(version : int)
      (() : unit) : t =
    { cells; metadata; notebookType; uri; version }
end

module DidOpenNotebookDocumentParams = struct
  type t =
    { cellTextDocuments : TextDocumentItem.t list;
      notebookDocument : NotebookDocument.t
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DidOpenNotebookDocumentParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let cellTextDocuments_field = ref Option.None
       and notebookDocument_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "cellTextDocuments" -> (
             match Ppx_yojson_conv_lib.( ! ) cellTextDocuments_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson TextDocumentItem.t_of_yojson _field_yojson
               in
               cellTextDocuments_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "notebookDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) notebookDocument_field with
             | Option.None ->
               let fvalue = NotebookDocument.t_of_yojson _field_yojson in
               notebookDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) cellTextDocuments_field,
               Ppx_yojson_conv_lib.( ! ) notebookDocument_field )
           with
           | ( Option.Some cellTextDocuments_value,
               Option.Some notebookDocument_value ) ->
             { cellTextDocuments = cellTextDocuments_value;
               notebookDocument = notebookDocument_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) cellTextDocuments_field)
                     Option.None,
                   "cellTextDocuments" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) notebookDocument_field)
                     Option.None,
                   "notebookDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { cellTextDocuments = v_cellTextDocuments;
         notebookDocument = v_notebookDocument
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = NotebookDocument.yojson_of_t v_notebookDocument in
         ("notebookDocument", arg) :: bnds
       in
       let bnds =
         let arg =
           yojson_of_list TextDocumentItem.yojson_of_t v_cellTextDocuments
         in
         ("cellTextDocuments", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(cellTextDocuments : TextDocumentItem.t list)
      ~(notebookDocument : NotebookDocument.t) : t =
    { cellTextDocuments; notebookDocument }
end

module DidOpenTextDocumentParams = struct
  type t = { textDocument : TextDocumentItem.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DidOpenTextDocumentParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let textDocument_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue = TextDocumentItem.t_of_yojson _field_yojson in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) textDocument_field with
           | Option.Some textDocument_value ->
             { textDocument = textDocument_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { textDocument = v_textDocument } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = TextDocumentItem.yojson_of_t v_textDocument in
         ("textDocument", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(textDocument : TextDocumentItem.t) : t = { textDocument }
end

module DidSaveNotebookDocumentParams = struct
  type t = { notebookDocument : NotebookDocumentIdentifier.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DidSaveNotebookDocumentParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let notebookDocument_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "notebookDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) notebookDocument_field with
             | Option.None ->
               let fvalue =
                 NotebookDocumentIdentifier.t_of_yojson _field_yojson
               in
               notebookDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) notebookDocument_field with
           | Option.Some notebookDocument_value ->
             { notebookDocument = notebookDocument_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) notebookDocument_field)
                     Option.None,
                   "notebookDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { notebookDocument = v_notebookDocument } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = NotebookDocumentIdentifier.yojson_of_t v_notebookDocument in
         ("notebookDocument", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(notebookDocument : NotebookDocumentIdentifier.t) : t =
    { notebookDocument }
end

module DidSaveTextDocumentParams = struct
  type t =
    { text : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      textDocument : TextDocumentIdentifier.t
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DidSaveTextDocumentParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let text_field = ref Option.None
       and textDocument_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "text" -> (
             match Ppx_yojson_conv_lib.( ! ) text_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               text_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue = TextDocumentIdentifier.t_of_yojson _field_yojson in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) text_field,
               Ppx_yojson_conv_lib.( ! ) textDocument_field )
           with
           | text_value, Option.Some textDocument_value ->
             { text =
                 (match text_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               textDocument = textDocument_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { text = v_text; textDocument = v_textDocument } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = TextDocumentIdentifier.yojson_of_t v_textDocument in
         ("textDocument", arg) :: bnds
       in
       let bnds =
         if None = v_text then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_text
           in
           let bnd = ("text", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(text : string option) ~(textDocument : TextDocumentIdentifier.t)
      (() : unit) : t =
    { text; textDocument }
end

module DocumentColorOptions = struct
  type t =
    { workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DocumentColorOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let workDoneProgress_value =
             Ppx_yojson_conv_lib.( ! ) workDoneProgress_field
           in
           { workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { workDoneProgress = v_workDoneProgress } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(workDoneProgress : bool option) (() : unit) : t =
    { workDoneProgress }
end

module DocumentColorParams = struct
  type t =
    { partialResultToken : ProgressToken.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      textDocument : TextDocumentIdentifier.t;
      workDoneToken : ProgressToken.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DocumentColorParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let partialResultToken_field = ref Option.None
       and textDocument_field = ref Option.None
       and workDoneToken_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "partialResultToken" -> (
             match Ppx_yojson_conv_lib.( ! ) partialResultToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               partialResultToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue = TextDocumentIdentifier.t_of_yojson _field_yojson in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneToken" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               workDoneToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) partialResultToken_field,
               Ppx_yojson_conv_lib.( ! ) textDocument_field,
               Ppx_yojson_conv_lib.( ! ) workDoneToken_field )
           with
           | ( partialResultToken_value,
               Option.Some textDocument_value,
               workDoneToken_value ) ->
             { partialResultToken =
                 (match partialResultToken_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               textDocument = textDocument_value;
               workDoneToken =
                 (match workDoneToken_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { partialResultToken = v_partialResultToken;
         textDocument = v_textDocument;
         workDoneToken = v_workDoneToken
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_workDoneToken
           in
           let bnd = ("workDoneToken", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = TextDocumentIdentifier.yojson_of_t v_textDocument in
         ("textDocument", arg) :: bnds
       in
       let bnds =
         if None = v_partialResultToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_partialResultToken
           in
           let bnd = ("partialResultToken", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(partialResultToken : ProgressToken.t option)
      ~(textDocument : TextDocumentIdentifier.t)
      ?(workDoneToken : ProgressToken.t option) (() : unit) : t =
    { partialResultToken; textDocument; workDoneToken }
end

module DocumentColorRegistrationOptions = struct
  type t =
    { documentSelector : DocumentSelector.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      id : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DocumentColorRegistrationOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let documentSelector_field = ref Option.None
       and id_field = ref Option.None
       and workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "documentSelector" -> (
             match Ppx_yojson_conv_lib.( ! ) documentSelector_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson DocumentSelector.t_of_yojson
                   _field_yojson
               in
               documentSelector_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "id" -> (
             match Ppx_yojson_conv_lib.( ! ) id_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               id_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let documentSelector_value, id_value, workDoneProgress_value =
             ( Ppx_yojson_conv_lib.( ! ) documentSelector_field,
               Ppx_yojson_conv_lib.( ! ) id_field,
               Ppx_yojson_conv_lib.( ! ) workDoneProgress_field )
           in
           { documentSelector =
               (match documentSelector_value with
               | Option.None -> None
               | Option.Some v -> v);
             id =
               (match id_value with
               | Option.None -> None
               | Option.Some v -> v);
             workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { documentSelector = v_documentSelector;
         id = v_id;
         workDoneProgress = v_workDoneProgress
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_id then bnds
         else
           let arg = (Json.Nullable_option.yojson_of_t yojson_of_string) v_id in
           let bnd = ("id", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_documentSelector then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t DocumentSelector.yojson_of_t)
               v_documentSelector
           in
           let bnd = ("documentSelector", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(documentSelector : DocumentSelector.t option)
      ?(id : string option) ?(workDoneProgress : bool option) (() : unit) : t =
    { documentSelector; id; workDoneProgress }
end

module DocumentDiagnosticParams = struct
  type t =
    { identifier : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      partialResultToken : ProgressToken.t Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      previousResultId : string Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      textDocument : TextDocumentIdentifier.t;
      workDoneToken : ProgressToken.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DocumentDiagnosticParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let identifier_field = ref Option.None
       and partialResultToken_field = ref Option.None
       and previousResultId_field = ref Option.None
       and textDocument_field = ref Option.None
       and workDoneToken_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "identifier" -> (
             match Ppx_yojson_conv_lib.( ! ) identifier_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               identifier_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "partialResultToken" -> (
             match Ppx_yojson_conv_lib.( ! ) partialResultToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               partialResultToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "previousResultId" -> (
             match Ppx_yojson_conv_lib.( ! ) previousResultId_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               previousResultId_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue = TextDocumentIdentifier.t_of_yojson _field_yojson in
               textDocument_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "workDoneToken" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneToken_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson ProgressToken.t_of_yojson
                   _field_yojson
               in
               workDoneToken_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) identifier_field,
               Ppx_yojson_conv_lib.( ! ) partialResultToken_field,
               Ppx_yojson_conv_lib.( ! ) previousResultId_field,
               Ppx_yojson_conv_lib.( ! ) textDocument_field,
               Ppx_yojson_conv_lib.( ! ) workDoneToken_field )
           with
           | ( identifier_value,
               partialResultToken_value,
               previousResultId_value,
               Option.Some textDocument_value,
               workDoneToken_value ) ->
             { identifier =
                 (match identifier_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               partialResultToken =
                 (match partialResultToken_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               previousResultId =
                 (match previousResultId_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               textDocument = textDocument_value;
               workDoneToken =
                 (match workDoneToken_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) textDocument_field)
                     Option.None,
                   "textDocument" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { identifier = v_identifier;
         partialResultToken = v_partialResultToken;
         previousResultId = v_previousResultId;
         textDocument = v_textDocument;
         workDoneToken = v_workDoneToken
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_workDoneToken
           in
           let bnd = ("workDoneToken", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = TextDocumentIdentifier.yojson_of_t v_textDocument in
         ("textDocument", arg) :: bnds
       in
       let bnds =
         if None = v_previousResultId then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string)
               v_previousResultId
           in
           let bnd = ("previousResultId", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_partialResultToken then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t ProgressToken.yojson_of_t)
               v_partialResultToken
           in
           let bnd = ("partialResultToken", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_identifier then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_identifier
           in
           let bnd = ("identifier", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(identifier : string option)
      ?(partialResultToken : ProgressToken.t option)
      ?(previousResultId : string option)
      ~(textDocument : TextDocumentIdentifier.t)
      ?(workDoneToken : ProgressToken.t option) (() : unit) : t =
    { identifier;
      partialResultToken;
      previousResultId;
      textDocument;
      workDoneToken
    }
end

module UnchangedDocumentDiagnosticReport = struct
  type t = { resultId : string }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.UnchangedDocumentDiagnosticReport.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let resultId_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "resultId" -> (
             match Ppx_yojson_conv_lib.( ! ) resultId_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               resultId_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) resultId_field with
           | Option.Some resultId_value -> { resultId = resultId_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) resultId_field)
                     Option.None,
                   "resultId" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { resultId = v_resultId } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_string v_resultId in
         ("resultId", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(resultId : string) : t = { resultId }

  let yojson_of_t (t : t) : Json.t =
    Json.To.literal_field "kind" "unchanged" yojson_of_t t

  let t_of_yojson (json : Json.t) : t =
    Json.Of.literal_field "t" "kind" "unchanged" t_of_yojson json
end

module FullDocumentDiagnosticReport = struct
  type t =
    { items : Diagnostic.t list;
      resultId : string Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.FullDocumentDiagnosticReport.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let items_field = ref Option.None
       and resultId_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "items" -> (
             match Ppx_yojson_conv_lib.( ! ) items_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson Diagnostic.t_of_yojson _field_yojson
               in
               items_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "resultId" -> (
             match Ppx_yojson_conv_lib.( ! ) resultId_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               resultId_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) items_field,
               Ppx_yojson_conv_lib.( ! ) resultId_field )
           with
           | Option.Some items_value, resultId_value ->
             { items = items_value;
               resultId =
                 (match resultId_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) items_field)
                     Option.None,
                   "items" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { items = v_items; resultId = v_resultId } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_resultId then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_resultId
           in
           let bnd = ("resultId", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_list Diagnostic.yojson_of_t v_items in
         ("items", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(items : Diagnostic.t list) ?(resultId : string option)
      (() : unit) : t =
    { items; resultId }

  let yojson_of_t (t : t) : Json.t =
    Json.To.literal_field "kind" "full" yojson_of_t t

  let t_of_yojson (json : Json.t) : t =
    Json.Of.literal_field "t" "kind" "full" t_of_yojson json
end

module RelatedUnchangedDocumentDiagnosticReport = struct
  type relatedDocuments_pvar =
    [ `FullDocumentDiagnosticReport of FullDocumentDiagnosticReport.t
    | `UnchangedDocumentDiagnosticReport of UnchangedDocumentDiagnosticReport.t
    ]

  let relatedDocuments_pvar_of_yojson (json : Json.t) : relatedDocuments_pvar =
    Json.Of.untagged_union "relatedDocuments_pvar"
      [ (fun json ->
          `FullDocumentDiagnosticReport
            (FullDocumentDiagnosticReport.t_of_yojson json));
        (fun json ->
          `UnchangedDocumentDiagnosticReport
            (UnchangedDocumentDiagnosticReport.t_of_yojson json))
      ]
      json

  let yojson_of_relatedDocuments_pvar
      (relatedDocuments_pvar : relatedDocuments_pvar) : Json.t =
    match relatedDocuments_pvar with
    | `FullDocumentDiagnosticReport s ->
      FullDocumentDiagnosticReport.yojson_of_t s
    | `UnchangedDocumentDiagnosticReport s ->
      UnchangedDocumentDiagnosticReport.yojson_of_t s

  type t =
    { relatedDocuments :
        (DocumentUri.t, relatedDocuments_pvar) Json.Assoc.t
        Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      resultId : string
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc =
       "lsp/src/types.ml.RelatedUnchangedDocumentDiagnosticReport.t"
     in
     function
     | `Assoc field_yojsons as yojson -> (
       let relatedDocuments_field = ref Option.None
       and resultId_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "relatedDocuments" -> (
             match Ppx_yojson_conv_lib.( ! ) relatedDocuments_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (Json.Assoc.t_of_yojson DocumentUri.t_of_yojson
                      relatedDocuments_pvar_of_yojson)
                   _field_yojson
               in
               relatedDocuments_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "resultId" -> (
             match Ppx_yojson_conv_lib.( ! ) resultId_field with
             | Option.None ->
               let fvalue = string_of_yojson _field_yojson in
               resultId_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) relatedDocuments_field,
               Ppx_yojson_conv_lib.( ! ) resultId_field )
           with
           | relatedDocuments_value, Option.Some resultId_value ->
             { relatedDocuments =
                 (match relatedDocuments_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               resultId = resultId_value
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) resultId_field)
                     Option.None,
                   "resultId" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { relatedDocuments = v_relatedDocuments; resultId = v_resultId } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg = yojson_of_string v_resultId in
         ("resultId", arg) :: bnds
       in
       let bnds =
         if None = v_relatedDocuments then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (Json.Assoc.yojson_of_t DocumentUri.yojson_of_t
                   yojson_of_relatedDocuments_pvar))
               v_relatedDocuments
           in
           let bnd = ("relatedDocuments", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create
      ?(relatedDocuments :
         (DocumentUri.t, relatedDocuments_pvar) Json.Assoc.t option)
      ~(resultId : string) (() : unit) : t =
    { relatedDocuments; resultId }

  let yojson_of_t (t : t) : Json.t =
    Json.To.literal_field "kind" "unchanged" yojson_of_t t

  let t_of_yojson (json : Json.t) : t =
    Json.Of.literal_field "t" "kind" "unchanged" t_of_yojson json
end

module RelatedFullDocumentDiagnosticReport = struct
  type relatedDocuments_pvar =
    [ `FullDocumentDiagnosticReport of FullDocumentDiagnosticReport.t
    | `UnchangedDocumentDiagnosticReport of UnchangedDocumentDiagnosticReport.t
    ]

  let relatedDocuments_pvar_of_yojson (json : Json.t) : relatedDocuments_pvar =
    Json.Of.untagged_union "relatedDocuments_pvar"
      [ (fun json ->
          `FullDocumentDiagnosticReport
            (FullDocumentDiagnosticReport.t_of_yojson json));
        (fun json ->
          `UnchangedDocumentDiagnosticReport
            (UnchangedDocumentDiagnosticReport.t_of_yojson json))
      ]
      json

  let yojson_of_relatedDocuments_pvar
      (relatedDocuments_pvar : relatedDocuments_pvar) : Json.t =
    match relatedDocuments_pvar with
    | `FullDocumentDiagnosticReport s ->
      FullDocumentDiagnosticReport.yojson_of_t s
    | `UnchangedDocumentDiagnosticReport s ->
      UnchangedDocumentDiagnosticReport.yojson_of_t s

  type t =
    { items : Diagnostic.t list;
      relatedDocuments :
        (DocumentUri.t, relatedDocuments_pvar) Json.Assoc.t
        Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      resultId : string Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.RelatedFullDocumentDiagnosticReport.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let items_field = ref Option.None
       and relatedDocuments_field = ref Option.None
       and resultId_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "items" -> (
             match Ppx_yojson_conv_lib.( ! ) items_field with
             | Option.None ->
               let fvalue =
                 list_of_yojson Diagnostic.t_of_yojson _field_yojson
               in
               items_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "relatedDocuments" -> (
             match Ppx_yojson_conv_lib.( ! ) relatedDocuments_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson
                   (Json.Assoc.t_of_yojson DocumentUri.t_of_yojson
                      relatedDocuments_pvar_of_yojson)
                   _field_yojson
               in
               relatedDocuments_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "resultId" -> (
             match Ppx_yojson_conv_lib.( ! ) resultId_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson string_of_yojson _field_yojson
               in
               resultId_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) items_field,
               Ppx_yojson_conv_lib.( ! ) relatedDocuments_field,
               Ppx_yojson_conv_lib.( ! ) resultId_field )
           with
           | Option.Some items_value, relatedDocuments_value, resultId_value ->
             { items = items_value;
               relatedDocuments =
                 (match relatedDocuments_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               resultId =
                 (match resultId_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) items_field)
                     Option.None,
                   "items" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { items = v_items;
         relatedDocuments = v_relatedDocuments;
         resultId = v_resultId
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_resultId then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_string) v_resultId
           in
           let bnd = ("resultId", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_relatedDocuments then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t
                (Json.Assoc.yojson_of_t DocumentUri.yojson_of_t
                   yojson_of_relatedDocuments_pvar))
               v_relatedDocuments
           in
           let bnd = ("relatedDocuments", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_list Diagnostic.yojson_of_t v_items in
         ("items", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ~(items : Diagnostic.t list)
      ?(relatedDocuments :
         (DocumentUri.t, relatedDocuments_pvar) Json.Assoc.t option)
      ?(resultId : string option) (() : unit) : t =
    { items; relatedDocuments; resultId }

  let yojson_of_t (t : t) : Json.t =
    Json.To.literal_field "kind" "full" yojson_of_t t

  let t_of_yojson (json : Json.t) : t =
    Json.Of.literal_field "t" "kind" "full" t_of_yojson json
end

module DocumentDiagnosticReport = struct
  type t =
    [ `RelatedFullDocumentDiagnosticReport of
      RelatedFullDocumentDiagnosticReport.t
    | `RelatedUnchangedDocumentDiagnosticReport of
      RelatedUnchangedDocumentDiagnosticReport.t ]

  let t_of_yojson (json : Json.t) : t =
    Json.Of.untagged_union "t"
      [ (fun json ->
          `RelatedFullDocumentDiagnosticReport
            (RelatedFullDocumentDiagnosticReport.t_of_yojson json));
        (fun json ->
          `RelatedUnchangedDocumentDiagnosticReport
            (RelatedUnchangedDocumentDiagnosticReport.t_of_yojson json))
      ]
      json

  let yojson_of_t (t : t) : Json.t =
    match t with
    | `RelatedFullDocumentDiagnosticReport s ->
      RelatedFullDocumentDiagnosticReport.yojson_of_t s
    | `RelatedUnchangedDocumentDiagnosticReport s ->
      RelatedUnchangedDocumentDiagnosticReport.yojson_of_t s
end

module DocumentDiagnosticReportPartialResult = struct
  type relatedDocuments_pvar =
    [ `FullDocumentDiagnosticReport of FullDocumentDiagnosticReport.t
    | `UnchangedDocumentDiagnosticReport of UnchangedDocumentDiagnosticReport.t
    ]

  let relatedDocuments_pvar_of_yojson (json : Json.t) : relatedDocuments_pvar =
    Json.Of.untagged_union "relatedDocuments_pvar"
      [ (fun json ->
          `FullDocumentDiagnosticReport
            (FullDocumentDiagnosticReport.t_of_yojson json));
        (fun json ->
          `UnchangedDocumentDiagnosticReport
            (UnchangedDocumentDiagnosticReport.t_of_yojson json))
      ]
      json

  let yojson_of_relatedDocuments_pvar
      (relatedDocuments_pvar : relatedDocuments_pvar) : Json.t =
    match relatedDocuments_pvar with
    | `FullDocumentDiagnosticReport s ->
      FullDocumentDiagnosticReport.yojson_of_t s
    | `UnchangedDocumentDiagnosticReport s ->
      UnchangedDocumentDiagnosticReport.yojson_of_t s

  type t =
    { relatedDocuments : (DocumentUri.t, relatedDocuments_pvar) Json.Assoc.t }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DocumentDiagnosticReportPartialResult.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let relatedDocuments_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "relatedDocuments" -> (
             match Ppx_yojson_conv_lib.( ! ) relatedDocuments_field with
             | Option.None ->
               let fvalue =
                 Json.Assoc.t_of_yojson DocumentUri.t_of_yojson
                   relatedDocuments_pvar_of_yojson _field_yojson
               in
               relatedDocuments_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match Ppx_yojson_conv_lib.( ! ) relatedDocuments_field with
           | Option.Some relatedDocuments_value ->
             { relatedDocuments = relatedDocuments_value }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) relatedDocuments_field)
                     Option.None,
                   "relatedDocuments" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { relatedDocuments = v_relatedDocuments } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         let arg =
           Json.Assoc.yojson_of_t DocumentUri.yojson_of_t
             yojson_of_relatedDocuments_pvar v_relatedDocuments
         in
         ("relatedDocuments", arg) :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create
      ~(relatedDocuments : (DocumentUri.t, relatedDocuments_pvar) Json.Assoc.t)
      : t =
    { relatedDocuments }
end

module DocumentFormattingOptions = struct
  type t =
    { workDoneProgress : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DocumentFormattingOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let workDoneProgress_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "workDoneProgress" -> (
             match Ppx_yojson_conv_lib.( ! ) workDoneProgress_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               workDoneProgress_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] ->
           let workDoneProgress_value =
             Ppx_yojson_conv_lib.( ! ) workDoneProgress_field
           in
           { workDoneProgress =
               (match workDoneProgress_value with
               | Option.None -> None
               | Option.Some v -> v)
           }))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { workDoneProgress = v_workDoneProgress } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_workDoneProgress then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_workDoneProgress
           in
           let bnd = ("workDoneProgress", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(workDoneProgress : bool option) (() : unit) : t =
    { workDoneProgress }
end

module FormattingOptions = struct
  type t =
    { insertFinalNewline : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      insertSpaces : bool;
      tabSize : int;
      trimFinalNewlines : bool Json.Nullable_option.t;
          [@default None] [@yojson_drop_default ( = )]
      trimTrailingWhitespace : bool Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.FormattingOptions.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let insertFinalNewline_field = ref Option.None
       and insertSpaces_field = ref Option.None
       and tabSize_field = ref Option.None
       and trimFinalNewlines_field = ref Option.None
       and trimTrailingWhitespace_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "insertFinalNewline" -> (
             match Ppx_yojson_conv_lib.( ! ) insertFinalNewline_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               insertFinalNewline_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "insertSpaces" -> (
             match Ppx_yojson_conv_lib.( ! ) insertSpaces_field with
             | Option.None ->
               let fvalue = bool_of_yojson _field_yojson in
               insertSpaces_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "tabSize" -> (
             match Ppx_yojson_conv_lib.( ! ) tabSize_field with
             | Option.None ->
               let fvalue = int_of_yojson _field_yojson in
               tabSize_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "trimFinalNewlines" -> (
             match Ppx_yojson_conv_lib.( ! ) trimFinalNewlines_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               trimFinalNewlines_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "trimTrailingWhitespace" -> (
             match Ppx_yojson_conv_lib.( ! ) trimTrailingWhitespace_field with
             | Option.None ->
               let fvalue =
                 Json.Nullable_option.t_of_yojson bool_of_yojson _field_yojson
               in
               trimTrailingWhitespace_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | _ -> ());
           iter tail
         | [] -> ()
       in
       iter field_yojsons;
       match Ppx_yojson_conv_lib.( ! ) duplicates with
       | _ :: _ ->
         Ppx_yojson_conv_lib.Yojson_conv_error.record_duplicate_fields _tp_loc
           (Ppx_yojson_conv_lib.( ! ) duplicates)
           yojson
       | [] -> (
         match Ppx_yojson_conv_lib.( ! ) extra with
         | _ :: _ ->
           Ppx_yojson_conv_lib.Yojson_conv_error.record_extra_fields _tp_loc
             (Ppx_yojson_conv_lib.( ! ) extra)
             yojson
         | [] -> (
           match
             ( Ppx_yojson_conv_lib.( ! ) insertFinalNewline_field,
               Ppx_yojson_conv_lib.( ! ) insertSpaces_field,
               Ppx_yojson_conv_lib.( ! ) tabSize_field,
               Ppx_yojson_conv_lib.( ! ) trimFinalNewlines_field,
               Ppx_yojson_conv_lib.( ! ) trimTrailingWhitespace_field )
           with
           | ( insertFinalNewline_value,
               Option.Some insertSpaces_value,
               Option.Some tabSize_value,
               trimFinalNewlines_value,
               trimTrailingWhitespace_value ) ->
             { insertFinalNewline =
                 (match insertFinalNewline_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               insertSpaces = insertSpaces_value;
               tabSize = tabSize_value;
               trimFinalNewlines =
                 (match trimFinalNewlines_value with
                 | Option.None -> None
                 | Option.Some v -> v);
               trimTrailingWhitespace =
                 (match trimTrailingWhitespace_value with
                 | Option.None -> None
                 | Option.Some v -> v)
             }
           | _ ->
             Ppx_yojson_conv_lib.Yojson_conv_error.record_undefined_elements
               _tp_loc yojson
               [ ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) insertSpaces_field)
                     Option.None,
                   "insertSpaces" );
                 ( Ppx_yojson_conv_lib.poly_equal
                     (Ppx_yojson_conv_lib.( ! ) tabSize_field)
                     Option.None,
                   "tabSize" )
               ])))
     | _ as yojson ->
       Ppx_yojson_conv_lib.Yojson_conv_error.record_list_instead_atom _tp_loc
         yojson
      : Ppx_yojson_conv_lib.Yojson.Safe.t -> t)

  let _ = t_of_yojson

  let yojson_of_t =
    (function
     | { insertFinalNewline = v_insertFinalNewline;
         insertSpaces = v_insertSpaces;
         tabSize = v_tabSize;
         trimFinalNewlines = v_trimFinalNewlines;
         trimTrailingWhitespace = v_trimTrailingWhitespace
       } ->
       let bnds : (string * Ppx_yojson_conv_lib.Yojson.Safe.t) list = [] in
       let bnds =
         if None = v_trimTrailingWhitespace then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_trimTrailingWhitespace
           in
           let bnd = ("trimTrailingWhitespace", arg) in
           bnd :: bnds
       in
       let bnds =
         if None = v_trimFinalNewlines then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_trimFinalNewlines
           in
           let bnd = ("trimFinalNewlines", arg) in
           bnd :: bnds
       in
       let bnds =
         let arg = yojson_of_int v_tabSize in
         ("tabSize", arg) :: bnds
       in
       let bnds =
         let arg = yojson_of_bool v_insertSpaces in
         ("insertSpaces", arg) :: bnds
       in
       let bnds =
         if None = v_insertFinalNewline then bnds
         else
           let arg =
             (Json.Nullable_option.yojson_of_t yojson_of_bool)
               v_insertFinalNewline
           in
           let bnd = ("insertFinalNewline", arg) in
           bnd :: bnds
       in
       `Assoc bnds
      : t -> Ppx_yojson_conv_lib.Yojson.Safe.t)

  let _ = yojson_of_t

  [@@@end]

  let create ?(insertFinalNewline : bool option) ~(insertSpaces : bool)
      ~(tabSize : int) ?(trimFinalNewlines : bool option)
      ?(trimTrailingWhitespace : bool option) (() : unit) : t =
    { insertFinalNewline;
      insertSpaces;
      tabSize;
      trimFinalNewlines;
      trimTrailingWhitespace
    }
end

module DocumentFormattingParams = struct
  type t =
    { options : FormattingOptions.t;
      textDocument : TextDocumentIdentifier.t;
      workDoneToken : ProgressToken.t Json.Nullable_option.t
          [@default None] [@yojson_drop_default ( = )]
    }
  [@@deriving_inline yojson] [@@yojson.allow_extra_fields]

  let _ = fun (_ : t) -> ()

  let t_of_yojson =
    (let _tp_loc = "lsp/src/types.ml.DocumentFormattingParams.t" in
     function
     | `Assoc field_yojsons as yojson -> (
       let options_field = ref Option.None
       and textDocument_field = ref Option.None
       and workDoneToken_field = ref Option.None
       and duplicates = ref []
       and extra = ref [] in
       let rec iter = function
         | (field_name, _field_yojson) :: tail ->
           (match field_name with
           | "options" -> (
             match Ppx_yojson_conv_lib.( ! ) options_field with
             | Option.None ->
               let fvalue = FormattingOptions.t_of_yojson _field_yojson in
               options_field := Option.Some fvalue
             | Option.Some _ ->
               duplicates := field_name :: Ppx_yojson_conv_lib.( ! ) duplicates)
           | "textDocument" -> (
             match Ppx_yojson_conv_lib.( ! ) textDocument_field with
             | Option.None ->
               let fvalue = TextDocumentIdentifier.t_of_yojson _field_yojson in
               textDocument_field := Option.Some fvalue
             |