// Licensed under the Apache License, Version 2.0
// <LICENSE-APACHE or http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your option.
// All files in the project carrying such notice may not be copied, modified, or distributed
// except according to those terms.
//! Direct3D capabilities include file
use ctypes::{c_char, c_float, c_void};
use shared::basetsd::UINT64;
use shared::guiddef::GUID;
use shared::minwindef::{BOOL, BYTE, DWORD, FLOAT, INT, UINT, USHORT, WORD};
use shared::windef::HWND;
use um::winnt::{HANDLE, HRESULT, LARGE_INTEGER, LONG, SHORT};
pub type D3DCOLOR = DWORD;
#[inline]
pub fn D3DCOLOR_ARGB(a: DWORD, r: DWORD, g: DWORD, b: DWORD) -> D3DCOLOR {
    (((a & 0xff) << 24) | ((r & 0xff) << 16) | ((g & 0xff) << 8) | (b & 0xff)) as D3DCOLOR
}
#[inline]
pub fn D3DCOLOR_RGBA(r: DWORD, g: DWORD, b: DWORD, a: DWORD) -> D3DCOLOR {
    D3DCOLOR_ARGB(a, r, g, b)
}
#[inline]
pub fn D3DCOLOR_XRGB(r: DWORD, g: DWORD, b: DWORD) -> D3DCOLOR {
    D3DCOLOR_ARGB(0xff, r, g, b)
}
#[inline]
pub fn D3DCOLOR_XYUV(y: DWORD, u: DWORD, v: DWORD) -> D3DCOLOR {
    D3DCOLOR_ARGB(0xff, y, u, v)
}
#[inline]
pub fn D3DCOLOR_AYUV(a: DWORD, y: DWORD, u: DWORD, v: DWORD) -> D3DCOLOR {
    D3DCOLOR_ARGB(a, y, u, v)
}
#[inline]
pub fn D3DCOLOR_COLORVALUE(r: f32, g: f32, b: f32, a: f32) -> D3DCOLOR {
    D3DCOLOR_ARGB(
        (r * 255f32) as DWORD,
        (g * 255f32) as DWORD,
        (b * 255f32) as DWORD,
        (a * 255f32) as DWORD,
    )
}
STRUCT!{struct D3DVECTOR {
    x: c_float,
    y: c_float,
    z: c_float,
}}
STRUCT!{struct D3DCOLORVALUE {
    r: c_float,
    g: c_float,
    b: c_float,
    a: c_float,
}}
STRUCT!{struct D3DRECT {
    x1: LONG,
    y1: LONG,
    x2: LONG,
    y2: LONG,
}}
STRUCT!{struct D3DMATRIX {
    m: [[c_float; 4]; 4],
}}
STRUCT!{struct D3DVIEWPORT9 {
    X: DWORD,
    Y: DWORD,
    Width: DWORD,
    Height: DWORD,
    MinZ: c_float,
    MaxZ: c_float,
}}
pub const D3DMAXUSERCLIPPLANES: DWORD = 32;
pub const D3DCLIPPLANE0: DWORD = 1 << 0;
pub const D3DCLIPPLANE1: DWORD = 1 << 1;
pub const D3DCLIPPLANE2: DWORD = 1 << 2;
pub const D3DCLIPPLANE3: DWORD = 1 << 3;
pub const D3DCLIPPLANE4: DWORD = 1 << 4;
pub const D3DCLIPPLANE5: DWORD = 1 << 5;
pub const D3DCS_LEFT: DWORD = 0x00000001;
pub const D3DCS_RIGHT: DWORD = 0x00000002;
pub const D3DCS_TOP: DWORD = 0x00000004;
pub const D3DCS_BOTTOM: DWORD = 0x00000008;
pub const D3DCS_FRONT: DWORD = 0x00000010;
pub const D3DCS_BACK: DWORD = 0x00000020;
pub const D3DCS_PLANE0: DWORD = 0x00000040;
pub const D3DCS_PLANE1: DWORD = 0x00000080;
pub const D3DCS_PLANE2: DWORD = 0x00000100;
pub const D3DCS_PLANE3: DWORD = 0x00000200;
pub const D3DCS_PLANE4: DWORD = 0x00000400;
pub const D3DCS_PLANE5: DWORD = 0x00000800;
pub const D3DCS_ALL: DWORD = D3DCS_LEFT | D3DCS_RIGHT | D3DCS_TOP | D3DCS_BOTTOM | D3DCS_FRONT
    | D3DCS_BACK | D3DCS_PLANE0 | D3DCS_PLANE1 | D3DCS_PLANE2 | D3DCS_PLANE3 | D3DCS_PLANE4
    | D3DCS_PLANE5;
STRUCT!{struct D3DCLIPSTATUS9 {
    ClipUnion: DWORD,
    ClipIntersection: DWORD,
}}
STRUCT!{struct D3DMATERIAL9 {
    Diffuse: D3DCOLORVALUE,
    Ambient: D3DCOLORVALUE,
    Specular: D3DCOLORVALUE,
    Emissive: D3DCOLORVALUE,
    Power: c_float,
}}
ENUM!{enum D3DLIGHTTYPE {
    D3DLIGHT_POINT = 1,
    D3DLIGHT_SPOT = 2,
    D3DLIGHT_DIRECTIONAL = 3,
}}
STRUCT!{struct D3DLIGHT9 {
    Type: D3DLIGHTTYPE,
    Diffuse: D3DCOLORVALUE,
    Specular: D3DCOLORVALUE,
    Ambient: D3DCOLORVALUE,
    Position: D3DVECTOR,
    Direction: D3DVECTOR,
    Range: c_float,
    Falloff: c_float,
    Attenuation0: c_float,
    Attenuation1: c_float,
    Attenuation2: c_float,
    Theta: c_float,
    Phi: c_float,
}}
pub const D3DCLEAR_TARGET: DWORD = 0x00000001;
pub const D3DCLEAR_ZBUFFER: DWORD = 0x00000002;
pub const D3DCLEAR_STENCIL: DWORD = 0x00000004;
ENUM!{enum D3DSHADEMODE {
    D3DSHADE_FLAT = 1,
    D3DSHADE_GOURAUD = 2,
    D3DSHADE_PHONG = 3,
}}
ENUM!{enum D3DFILLMODE {
    D3DFILL_POINT = 1,
    D3DFILL_WIREFRAME = 2,
    D3DFILL_SOLID = 3,
}}
ENUM!{enum D3DBLEND {
    D3DBLEND_ZERO = 1,
    D3DBLEND_ONE = 2,
    D3DBLEND_SRCCOLOR = 3,
    D3DBLEND_INVSRCCOLOR = 4,
    D3DBLEND_SRCALPHA = 5,
    D3DBLEND_INVSRCALPHA = 6,
    D3DBLEND_DESTALPHA = 7,
    D3DBLEND_INVDESTALPHA = 8,
    D3DBLEND_DESTCOLOR = 9,
    D3DBLEND_INVDESTCOLOR = 10,
    D3DBLEND_SRCALPHASAT = 11,
    D3DBLEND_BOTHSRCALPHA = 12,
    D3DBLEND_BOTHINVSRCALPHA = 13,
    D3DBLEND_BLENDFACTOR = 14,
    D3DBLEND_INVBLENDFACTOR = 15,
    D3DBLEND_SRCCOLOR2 = 16,
    D3DBLEND_INVSRCCOLOR2 = 17,
}}
ENUM!{enum D3DBLENDOP {
    D3DBLENDOP_ADD = 1,
    D3DBLENDOP_SUBTRACT = 2,
    D3DBLENDOP_REVSUBTRACT = 3,
    D3DBLENDOP_MIN = 4,
    D3DBLENDOP_MAX = 5,
}}
ENUM!{enum D3DTEXTUREADDRESS {
    D3DTADDRESS_WRAP = 1,
    D3DTADDRESS_MIRROR = 2,
    D3DTADDRESS_CLAMP = 3,
    D3DTADDRESS_BORDER = 4,
    D3DTADDRESS_MIRRORONCE = 5,
}}
ENUM!{enum D3DCULL {
    D3DCULL_NONE = 1,
    D3DCULL_CW = 2,
    D3DCULL_CCW = 3,
}}
ENUM!{enum D3DCMPFUNC {
    D3DCMP_NEVER = 1,
    D3DCMP_LESS = 2,
    D3DCMP_EQUAL = 3,
    D3DCMP_LESSEQUAL = 4,
    D3DCMP_GREATER = 5,
    D3DCMP_NOTEQUAL = 6,
    D3DCMP_GREATEREQUAL = 7,
    D3DCMP_ALWAYS = 8,
}}
ENUM!{enum D3DSTENCILOP {
    D3DSTENCILOP_KEEP = 1,
    D3DSTENCILOP_ZERO = 2,
    D3DSTENCILOP_REPLACE = 3,
    D3DSTENCILOP_INCRSAT = 4,
    D3DSTENCILOP_DECRSAT = 5,
    D3DSTENCILOP_INVERT = 6,
    D3DSTENCILOP_INCR = 7,
    D3DSTENCILOP_DECR = 8,
}}
ENUM!{enum D3DFOGMODE {
    D3DFOG_NONE = 0,
    D3DFOG_EXP = 1,
    D3DFOG_EXP2 = 2,
    D3DFOG_LINEAR = 3,
}}
ENUM!{enum D3DZBUFFERTYPE {
    D3DZB_FALSE = 0,
    D3DZB_TRUE = 1,
    D3DZB_USEW = 2,
}}
ENUM!{enum D3DPRIMITIVETYPE {
    D3DPT_POINTLIST = 1,
    D3DPT_LINELIST = 2,
    D3DPT_LINESTRIP = 3,
    D3DPT_TRIANGLELIST = 4,
    D3DPT_TRIANGLESTRIP = 5,
    D3DPT_TRIANGLEFAN = 6,
}}
ENUM!{enum D3DTRANSFORMSTATETYPE {
    D3DTS_VIEW = 2,
    D3DTS_PROJECTION = 3,
    D3DTS_TEXTURE0 = 16,
    D3DTS_TEXTURE1 = 17,
    D3DTS_TEXTURE2 = 18,
    D3DTS_TEXTURE3 = 19,
    D3DTS_TEXTURE4 = 20,
    D3DTS_TEXTURE5 = 21,
    D3DTS_TEXTURE6 = 22,
    D3DTS_TEXTURE7 = 23,
}}
macro_rules! D3DTS_WORLDMATRIX {
    ($index:expr) => ($index + 256)
}
pub const D3DTS_WORLD: D3DTRANSFORMSTATETYPE = D3DTS_WORLDMATRIX!(0);
pub const D3DTS_WORLD1: D3DTRANSFORMSTATETYPE = D3DTS_WORLDMATRIX!(1);
pub const D3DTS_WORLD2: D3DTRANSFORMSTATETYPE = D3DTS_WORLDMATRIX!(2);
pub const D3DTS_WORLD3: D3DTRANSFORMSTATETYPE = D3DTS_WORLDMATRIX!(3);
ENUM!{enum D3DRENDERSTATETYPE {
    D3DRS_ZENABLE = 7,
    D3DRS_FILLMODE = 8,
    D3DRS_SHADEMODE = 9,
    D3DRS_ZWRITEENABLE = 14,
    D3DRS_ALPHATESTENABLE = 15,
    D3DRS_LASTPIXEL = 16,
    D3DRS_SRCBLEND = 19,
    D3DRS_DESTBLEND = 20,
    D3DRS_CULLMODE = 22,
    D3DRS_ZFUNC = 23,
    D3DRS_ALPHAREF = 24,
    D3DRS_ALPHAFUNC = 25,
    D3DRS_DITHERENABLE = 26,
    D3DRS_ALPHABLENDENABLE = 27,
    D3DRS_FOGENABLE = 28,
    D3DRS_SPECULARENABLE = 29,
    D3DRS_FOGCOLOR = 34,
    D3DRS_FOGTABLEMODE = 35,
    D3DRS_FOGSTART = 36,
    D3DRS_FOGEND = 37,
    D3DRS_FOGDENSITY = 38,
    D3DRS_RANGEFOGENABLE = 48,
    D3DRS_STENCILENABLE = 52,
    D3DRS_STENCILFAIL = 53,
    D3DRS_STENCILZFAIL = 54,
    D3DRS_STENCILPASS = 55,
    D3DRS_STENCILFUNC = 56,
    D3DRS_STENCILREF = 57,
    D3DRS_STENCILMASK = 58,
    D3DRS_STENCILWRITEMASK = 59,
    D3DRS_TEXTUREFACTOR = 60,
    D3DRS_WRAP0 = 128,
    D3DRS_WRAP1 = 129,
    D3DRS_WRAP2 = 130,
    D3DRS_WRAP3 = 131,
    D3DRS_WRAP4 = 132,
    D3DRS_WRAP5 = 133,
    D3DRS_WRAP6 = 134,
    D3DRS_WRAP7 = 135,
    D3DRS_CLIPPING = 136,
    D3DRS_LIGHTING = 137,
    D3DRS_AMBIENT = 139,
    D3DRS_FOGVERTEXMODE = 140,
    D3DRS_COLORVERTEX = 141,
    D3DRS_LOCALVIEWER = 142,
    D3DRS_NORMALIZENORMALS = 143,
    D3DRS_DIFFUSEMATERIALSOURCE = 145,
    D3DRS_SPECULARMATERIALSOURCE = 146,
    D3DRS_AMBIENTMATERIALSOURCE = 147,
    D3DRS_EMISSIVEMATERIALSOURCE = 148,
    D3DRS_VERTEXBLEND = 151,
    D3DRS_CLIPPLANEENABLE = 152,
    D3DRS_POINTSIZE = 154,
    D3DRS_POINTSIZE_MIN = 155,
    D3DRS_POINTSPRITEENABLE = 156,
    D3DRS_POINTSCALEENABLE = 157,
    D3DRS_POINTSCALE_A = 158,
    D3DRS_POINTSCALE_B = 159,
    D3DRS_POINTSCALE_C = 160,
    D3DRS_MULTISAMPLEANTIALIAS = 161,
    D3DRS_MULTISAMPLEMASK = 162,
    D3DRS_PATCHEDGESTYLE = 163,
    D3DRS_DEBUGMONITORTOKEN = 165,
    D3DRS_POINTSIZE_MAX = 166,
    D3DRS_INDEXEDVERTEXBLENDENABLE = 167,
    D3DRS_COLORWRITEENABLE = 168,
    D3DRS_TWEENFACTOR = 170,
    D3DRS_BLENDOP = 171,
    D3DRS_POSITIONDEGREE = 172,
    D3DRS_NORMALDEGREE = 173,
    D3DRS_SCISSORTESTENABLE = 174,
    D3DRS_SLOPESCALEDEPTHBIAS = 175,
    D3DRS_ANTIALIASEDLINEENABLE = 176,
    D3DRS_MINTESSELLATIONLEVEL = 178,
    D3DRS_MAXTESSELLATIONLEVEL = 179,
    D3DRS_ADAPTIVETESS_X = 180,
    D3DRS_ADAPTIVETESS_Y = 181,
    D3DRS_ADAPTIVETESS_Z = 182,
    D3DRS_ADAPTIVETESS_W = 183,
    D3DRS_ENABLEADAPTIVETESSELLATION = 184,
    D3DRS_TWOSIDEDSTENCILMODE = 185,
    D3DRS_CCW_STENCILFAIL = 186,
    D3DRS_CCW_STENCILZFAIL = 187,
    D3DRS_CCW_STENCILPASS = 188,
    D3DRS_CCW_STENCILFUNC = 189,
    D3DRS_COLORWRITEENABLE1 = 190,
    D3DRS_COLORWRITEENABLE2 = 191,
    D3DRS_COLORWRITEENABLE3 = 192,
    D3DRS_BLENDFACTOR = 193,
    D3DRS_SRGBWRITEENABLE = 194,
    D3DRS_DEPTHBIAS = 195,
    D3DRS_WRAP8 = 198,
    D3DRS_WRAP9 = 199,
    D3DRS_WRAP10 = 200,
    D3DRS_WRAP11 = 201,
    D3DRS_WRAP12 = 202,
    D3DRS_WRAP13 = 203,
    D3DRS_WRAP14 = 204,
    D3DRS_WRAP15 = 205,
    D3DRS_SEPARATEALPHABLENDENABLE = 206,
    D3DRS_SRCBLENDALPHA = 207,
    D3DRS_DESTBLENDALPHA = 208,
    D3DRS_BLENDOPALPHA = 209,
}}
pub const D3D_MAX_SIMULTANEOUS_RENDERTARGETS: DWORD = 4;
ENUM!{enum D3DMATERIALCOLORSOURCE {
    D3DMCS_MATERIAL = 0,
    D3DMCS_COLOR1 = 1,
    D3DMCS_COLOR2 = 2,
}}
pub const D3DRENDERSTATE_WRAPBIAS: DWORD = 128;
pub const D3DWRAP_U: DWORD = 0x00000001;
pub const D3DWRAP_V: DWORD = 0x00000002;
pub const D3DWRAP_W: DWORD = 0x00000004;
pub const D3DWRAPCOORD_0: DWORD = 0x00000001;
pub const D3DWRAPCOORD_1: DWORD = 0x00000002;
pub const D3DWRAPCOORD_2: DWORD = 0x00000004;
pub const D3DWRAPCOORD_3: DWORD = 0x00000008;
pub const D3DCOLORWRITEENABLE_RED: DWORD = 1 << 0;
pub const D3DCOLORWRITEENABLE_GREEN: DWORD = 1 << 1;
pub const D3DCOLORWRITEENABLE_BLUE: DWORD = 1 << 2;
pub const D3DCOLORWRITEENABLE_ALPHA: DWORD = 1 << 3;
ENUM!{enum D3DTEXTURESTAGESTATETYPE {
    D3DTSS_COLOROP = 1,
    D3DTSS_COLORARG1 = 2,
    D3DTSS_COLORARG2 = 3,
    D3DTSS_ALPHAOP = 4,
    D3DTSS_ALPHAARG1 = 5,
    D3DTSS_ALPHAARG2 = 6,
    D3DTSS_BUMPENVMAT00 = 7,
    D3DTSS_BUMPENVMAT01 = 8,
    D3DTSS_BUMPENVMAT10 = 9,
    D3DTSS_BUMPENVMAT11 = 10,
    D3DTSS_TEXCOORDINDEX = 11,
    D3DTSS_BUMPENVLSCALE = 22,
    D3DTSS_BUMPENVLOFFSET = 23,
    D3DTSS_TEXTURETRANSFORMFLAGS = 24,
    D3DTSS_COLORARG0 = 26,
    D3DTSS_ALPHAARG0 = 27,
    D3DTSS_RESULTARG = 28,
    D3DTSS_CONSTANT = 32,
}}
ENUM!{enum D3DSAMPLERSTATETYPE {
    D3DSAMP_ADDRESSU = 1,
    D3DSAMP_ADDRESSV = 2,
    D3DSAMP_ADDRESSW = 3,
    D3DSAMP_BORDERCOLOR = 4,
    D3DSAMP_MAGFILTER = 5,
    D3DSAMP_MINFILTER = 6,
    D3DSAMP_MIPFILTER = 7,
    D3DSAMP_MIPMAPLODBIAS = 8,
    D3DSAMP_MAXMIPLEVEL = 9,
    D3DSAMP_MAXANISOTROPY = 10,
    D3DSAMP_SRGBTEXTURE = 11,
    D3DSAMP_ELEMENTINDEX = 12,
    D3DSAMP_DMAPOFFSET = 13,
}}
pub const D3DDMAPSAMPLER: DWORD = 256;
pub const D3DVERTEXTEXTURESAMPLER0: DWORD = D3DDMAPSAMPLER + 1;
pub const D3DVERTEXTEXTURESAMPLER1: DWORD = D3DDMAPSAMPLER + 2;
pub const D3DVERTEXTEXTURESAMPLER2: DWORD = D3DDMAPSAMPLER + 3;
pub const D3DVERTEXTEXTURESAMPLER3: DWORD = D3DDMAPSAMPLER + 4;
pub const D3DTSS_TCI_PASSTHRU: DWORD = 0x00000000;
pub const D3DTSS_TCI_CAMERASPACENORMAL: DWORD = 0x00010000;
pub const D3DTSS_TCI_CAMERASPACEPOSITION: DWORD = 0x00020000;
pub const D3DTSS_TCI_CAMERASPACEREFLECTIONVECTOR: DWORD = 0x00030000;
pub const D3DTSS_TCI_SPHEREMAP: DWORD = 0x00040000;
ENUM!{enum D3DTEXTUREOP {
    D3DTOP_DISABLE = 1,
    D3DTOP_SELECTARG1 = 2,
    D3DTOP_SELECTARG2 = 3,
    D3DTOP_MODULATE = 4,
    D3DTOP_MODULATE2X = 5,
    D3DTOP_MODULATE4X = 6,
    D3DTOP_ADD = 7,
    D3DTOP_ADDSIGNED = 8,
    D3DTOP_ADDSIGNED2X = 9,
    D3DTOP_SUBTRACT = 10,
    D3DTOP_ADDSMOOTH = 11,
    D3DTOP_BLENDDIFFUSEALPHA = 12,
    D3DTOP_BLENDTEXTUREALPHA = 13,
    D3DTOP_BLENDFACTORALPHA = 14,
    D3DTOP_BLENDTEXTUREALPHAPM = 15,
    D3DTOP_BLENDCURRENTALPHA = 16,
    D3DTOP_PREMODULATE = 17,
    D3DTOP_MODULATEALPHA_ADDCOLOR = 18,
    D3DTOP_MODULATECOLOR_ADDALPHA = 19,
    D3DTOP_MODULATEINVALPHA_ADDCOLOR = 20,
    D3DTOP_MODULATEINVCOLOR_ADDALPHA = 21,
    D3DTOP_BUMPENVMAP = 22,
    D3DTOP_BUMPENVMAPLUMINANCE = 23,
    D3DTOP_DOTPRODUCT3 = 24,
    D3DTOP_MULTIPLYADD = 25,
    D3DTOP_LERP = 26,
}}
pub const D3DTA_SELECTMASK: DWORD = 0x0000000f;
pub const D3DTA_DIFFUSE: DWORD = 0x00000000;
pub const D3DTA_CURRENT: DWORD = 0x00000001;
pub const D3DTA_TEXTURE: DWORD = 0x00000002;
pub const D3DTA_TFACTOR: DWORD = 0x00000003;
pub const D3DTA_SPECULAR: DWORD = 0x00000004;
pub const D3DTA_TEMP: DWORD = 0x00000005;
pub const D3DTA_CONSTANT: DWORD = 0x00000006;
pub const D3DTA_COMPLEMENT: DWORD = 0x00000010;
pub const D3DTA_ALPHAREPLICATE: DWORD = 0x00000020;
ENUM!{enum D3DTEXTUREFILTERTYPE {
    D3DTEXF_NONE = 0,
    D3DTEXF_POINT = 1,
    D3DTEXF_LINEAR = 2,
    D3DTEXF_ANISOTROPIC = 3,
    D3DTEXF_PYRAMIDALQUAD = 6,
    D3DTEXF_GAUSSIANQUAD = 7,
    D3DTEXF_CONVOLUTIONMONO = 8,
}}
pub const D3DPV_DONOTCOPYDATA: DWORD = 1 << 0;
pub const D3DFVF_RESERVED0: DWORD = 0x001;
pub const D3DFVF_POSITION_MASK: DWORD = 0x400E;
pub const D3DFVF_XYZ: DWORD = 0x002;
pub const D3DFVF_XYZRHW: DWORD = 0x004;
pub const D3DFVF_XYZB1: DWORD = 0x006;
pub const D3DFVF_XYZB2: DWORD = 0x008;
pub const D3DFVF_XYZB3: DWORD = 0x00a;
pub const D3DFVF_XYZB4: DWORD = 0x00c;
pub const D3DFVF_XYZB5: DWORD = 0x00e;
pub const D3DFVF_XYZW: DWORD = 0x4002;
pub const D3DFVF_NORMAL: DWORD = 0x010;
pub const D3DFVF_PSIZE: DWORD = 0x020;
pub const D3DFVF_DIFFUSE: DWORD = 0x040;
pub const D3DFVF_SPECULAR: DWORD = 0x080;
pub const D3DFVF_TEXCOUNT_MASK: DWORD = 0xf00;
pub const D3DFVF_TEXCOUNT_SHIFT: DWORD = 8;
pub const D3DFVF_TEX0: DWORD = 0x000;
pub const D3DFVF_TEX1: DWORD = 0x100;
pub const D3DFVF_TEX2: DWORD = 0x200;
pub const D3DFVF_TEX3: DWORD = 0x300;
pub const D3DFVF_TEX4: DWORD = 0x400;
pub const D3DFVF_TEX5: DWORD = 0x500;
pub const D3DFVF_TEX6: DWORD = 0x600;
pub const D3DFVF_TEX7: DWORD = 0x700;
pub const D3DFVF_TEX8: DWORD = 0x800;
pub const D3DFVF_LASTBETA_UBYTE4: DWORD = 0x1000;
pub const D3DFVF_LASTBETA_D3DCOLOR: DWORD = 0x8000;
pub const D3DFVF_RESERVED2: DWORD = 0x6000;
ENUM!{enum D3DDECLUSAGE {
    D3DDECLUSAGE_POSITION = 0,
    D3DDECLUSAGE_BLENDWEIGHT,
    D3DDECLUSAGE_BLENDINDICES,
    D3DDECLUSAGE_NORMAL,
    D3DDECLUSAGE_PSIZE,
    D3DDECLUSAGE_TEXCOORD,
    D3DDECLUSAGE_TANGENT,
    D3DDECLUSAGE_BINORMAL,
    D3DDECLUSAGE_TESSFACTOR,
    D3DDECLUSAGE_POSITIONT,
    D3DDECLUSAGE_COLOR,
    D3DDECLUSAGE_FOG,
    D3DDECLUSAGE_DEPTH,
    D3DDECLUSAGE_SAMPLE,
}}
pub const MAXD3DDECLUSAGE: D3DDECLUSAGE = D3DDECLUSAGE_SAMPLE;
pub const MAXD3DDECLUSAGEINDEX: DWORD = 15;
pub const MAXD3DDECLLENGTH: DWORD = 64;
ENUM!{enum D3DDECLMETHOD {
    D3DDECLMETHOD_DEFAULT = 0,
    D3DDECLMETHOD_PARTIALU,
    D3DDECLMETHOD_PARTIALV,
    D3DDECLMETHOD_CROSSUV,
    D3DDECLMETHOD_UV,
    D3DDECLMETHOD_LOOKUP,
    D3DDECLMETHOD_LOOKUPPRESAMPLED,
}}
pub const MAXD3DDECLMETHOD: D3DDECLMETHOD = D3DDECLMETHOD_LOOKUPPRESAMPLED;
ENUM!{enum D3DDECLTYPE {
    D3DDECLTYPE_FLOAT1 = 0,
    D3DDECLTYPE_FLOAT2 = 1,
    D3DDECLTYPE_FLOAT3 = 2,
    D3DDECLTYPE_FLOAT4 = 3,
    D3DDECLTYPE_D3DCOLOR = 4,
    D3DDECLTYPE_UBYTE4 = 5,
    D3DDECLTYPE_SHORT2 = 6,
    D3DDECLTYPE_SHORT4 = 7,
    D3DDECLTYPE_UBYTE4N = 8,
    D3DDECLTYPE_SHORT2N = 9,
    D3DDECLTYPE_SHORT4N = 10,
    D3DDECLTYPE_USHORT2N = 11,
    D3DDECLTYPE_USHORT4N = 12,
    D3DDECLTYPE_UDEC3 = 13,
    D3DDECLTYPE_DEC3N = 14,
    D3DDECLTYPE_FLOAT16_2 = 15,
    D3DDECLTYPE_FLOAT16_4 = 16,
    D3DDECLTYPE_UNUSED = 17,
}}
pub const MAXD3DDECLTYPE: D3DDECLTYPE = D3DDECLTYPE_UNUSED;
STRUCT!{struct D3DVERTEXELEMENT9 {
    Stream: WORD,
    Offset: WORD,
    Type: BYTE,
    Method: BYTE,
    Usage: BYTE,
    UsageIndex: BYTE,
}}
pub type LPD3DVERTEXELEMENT9 = *mut D3DVERTEXELEMENT9;
pub const D3DDECL_END: D3DVERTEXELEMENT9 = D3DVERTEXELEMENT9 {
    Stream: 0xFF,
    Offset: 0,
    Type: D3DDECLTYPE_UNUSED as BYTE,
    Method: 0,
    Usage: 0,
    UsageIndex: 0,
};
pub const D3DDP_MAXTEXCOORD: DWORD = 8;
pub const D3DSTREAMSOURCE_INDEXEDDATA: DWORD = 1 << 30;
pub const D3DSTREAMSOURCE_INSTANCEDATA: DWORD = 2 << 30;
pub const D3DSI_OPCODE_MASK: DWORD = 0x0000FFFF;
pub const D3DSI_INSTLENGTH_MASK: DWORD = 0x0F000000;
pub const D3DSI_INSTLENGTH_SHIFT: DWORD = 24;
ENUM!{enum D3DSHADER_INSTRUCTION_OPCODE_TYPE {
    D3DSIO_NOP = 0,
    D3DSIO_MOV,
    D3DSIO_ADD,
    D3DSIO_SUB,
    D3DSIO_MAD,
    D3DSIO_MUL,
    D3DSIO_RCP,
    D3DSIO_RSQ,
    D3DSIO_DP3,
    D3DSIO_DP4,
    D3DSIO_MIN,
    D3DSIO_MAX,
    D3DSIO_SLT,
    D3DSIO_SGE,
    D3DSIO_EXP,
    D3DSIO_LOG,
    D3DSIO_LIT,
    D3DSIO_DST,
    D3DSIO_LRP,
    D3DSIO_FRC,
    D3DSIO_M4x4,
    D3DSIO_M4x3,
    D3DSIO_M3x4,
    D3DSIO_M3x3,
    D3DSIO_M3x2,
    D3DSIO_CALL,
    D3DSIO_CALLNZ,
    D3DSIO_LOOP,
    D3DSIO_RET,
    D3DSIO_ENDLOOP,
    D3DSIO_LABEL,
    D3DSIO_DCL,
    D3DSIO_POW,
    D3DSIO_CRS,
    D3DSIO_SGN,
    D3DSIO_ABS,
    D3DSIO_NRM,
    D3DSIO_SINCOS,
    D3DSIO_REP,
    D3DSIO_ENDREP,
    D3DSIO_IF,
    D3DSIO_IFC,
    D3DSIO_ELSE,
    D3DSIO_ENDIF,
    D3DSIO_BREAK,
    D3DSIO_BREAKC,
    D3DSIO_MOVA,
    D3DSIO_DEFB,
    D3DSIO_DEFI,
    D3DSIO_TEXCOORD = 64,
    D3DSIO_TEXKILL,
    D3DSIO_TEX,
    D3DSIO_TEXBEM,
    D3DSIO_TEXBEML,
    D3DSIO_TEXREG2AR,
    D3DSIO_TEXREG2GB,
    D3DSIO_TEXM3x2PAD,
    D3DSIO_TEXM3x2TEX,
    D3DSIO_TEXM3x3PAD,
    D3DSIO_TEXM3x3TEX,
    D3DSIO_RESERVED0,
    D3DSIO_TEXM3x3SPEC,
    D3DSIO_TEXM3x3VSPEC,
    D3DSIO_EXPP,
    D3DSIO_LOGP,
    D3DSIO_CND,
    D3DSIO_DEF,
    D3DSIO_TEXREG2RGB,
    D3DSIO_TEXDP3TEX,
    D3DSIO_TEXM3x2DEPTH,
    D3DSIO_TEXDP3,
    D3DSIO_TEXM3x3,
    D3DSIO_TEXDEPTH,
    D3DSIO_CMP,
    D3DSIO_BEM,
    D3DSIO_DP2ADD,
    D3DSIO_DSX,
    D3DSIO_DSY,
    D3DSIO_TEXLDD,
    D3DSIO_SETP,
    D3DSIO_TEXLDL,
    D3DSIO_BREAKP,
    D3DSIO_PHASE = 0xFFFD,
    D3DSIO_COMMENT = 0xFFFE,
    D3DSIO_END = 0xFFFF,
}}
pub const D3DSI_COISSUE: DWORD = 0x40000000;
pub const D3DSP_OPCODESPECIFICCONTROL_MASK: DWORD = 0x00ff0000;
pub const D3DSP_OPCODESPECIFICCONTROL_SHIFT: DWORD = 16;
pub const D3DSI_TEXLD_PROJECT: DWORD = 0x01 << D3DSP_OPCODESPECIFICCONTROL_SHIFT;
pub const D3DSI_TEXLD_BIAS: DWORD = 0x02 << D3DSP_OPCODESPECIFICCONTROL_SHIFT;
ENUM!{enum D3DSHADER_COMPARISON {
    D3DSPC_RESERVED0 = 0,
    D3DSPC_GT = 1,
    D3DSPC_EQ = 2,
    D3DSPC_GE = 3,
    D3DSPC_LT = 4,
    D3DSPC_NE = 5,
    D3DSPC_LE = 6,
    D3DSPC_RESERVED1 = 7,
}}
pub const D3DSHADER_COMPARISON_SHIFT: DWORD = D3DSP_OPCODESPECIFICCONTROL_SHIFT;
pub const D3DSHADER_COMPARISON_MASK: DWORD = 0x7 << D3DSHADER_COMPARISON_SHIFT;
pub const D3DSHADER_INSTRUCTION_PREDICATED: DWORD = 0x1 << 28;
pub const D3DSP_DCL_USAGE_SHIFT: DWORD = 0;
pub const D3DSP_DCL_USAGE_MASK: DWORD = 0x0000000f;
pub const D3DSP_DCL_USAGEINDEX_SHIFT: DWORD = 16;
pub const D3DSP_DCL_USAGEINDEX_MASK: DWORD = 0x000f0000;
pub const D3DSP_TEXTURETYPE_SHIFT: DWORD = 27;
pub const D3DSP_TEXTURETYPE_MASK: DWORD = 0x78000000;
ENUM!{enum D3DSAMPLER_TEXTURE_TYPE {
    D3DSTT_UNKNOWN = 0 << D3DSP_TEXTURETYPE_SHIFT,
    D3DSTT_2D = 2 << D3DSP_TEXTURETYPE_SHIFT,
    D3DSTT_CUBE = 3 << D3DSP_TEXTURETYPE_SHIFT,
    D3DSTT_VOLUME = 4 << D3DSP_TEXTURETYPE_SHIFT,
}}
pub const D3DSP_REGNUM_MASK: DWORD = 0x000007FF;
pub const D3DSP_WRITEMASK_0: DWORD = 0x00010000;
pub const D3DSP_WRITEMASK_1: DWORD = 0x00020000;
pub const D3DSP_WRITEMASK_2: DWORD = 0x00040000;
pub const D3DSP_WRITEMASK_3: DWORD = 0x00080000;
pub const D3DSP_WRITEMASK_ALL: DWORD = 0x000F0000;
pub const D3DSP_DSTMOD_SHIFT: DWORD = 20;
pub const D3DSP_DSTMOD_MASK: DWORD = 0x00F00000;
pub const D3DSPDM_NONE: DWORD = 0 << D3DSP_DSTMOD_SHIFT;
pub const D3DSPDM_SATURATE: DWORD = 1 << D3DSP_DSTMOD_SHIFT;
pub const D3DSPDM_PARTIALPRECISION: DWORD = 2 << D3DSP_DSTMOD_SHIFT;
pub const D3DSPDM_MSAMPCENTROID: DWORD = 4 << D3DSP_DSTMOD_SHIFT;
pub const D3DSP_DSTSHIFT_SHIFT: DWORD = 24;
pub const D3DSP_DSTSHIFT_MASK: DWORD = 0x0F000000;
pub const D3DSP_REGTYPE_SHIFT: DWORD = 28;
pub const D3DSP_REGTYPE_SHIFT2: DWORD = 8;
pub const D3DSP_REGTYPE_MASK: DWORD = 0x70000000;
pub const D3DSP_REGTYPE_MASK2: DWORD = 0x00001800;
ENUM!{enum D3DSHADER_PARAM_REGISTER_TYPE {
    D3DSPR_TEMP = 0,
    D3DSPR_INPUT = 1,
    D3DSPR_CONST = 2,
    D3DSPR_ADDR = 3,
    D3DSPR_TEXTURE = 3,
    D3DSPR_RASTOUT = 4,
    D3DSPR_ATTROUT = 5,
    D3DSPR_TEXCRDOUT = 6,
    D3DSPR_OUTPUT = 6,
    D3DSPR_CONSTINT = 7,
    D3DSPR_COLOROUT = 8,
    D3DSPR_DEPTHOUT = 9,
    D3DSPR_SAMPLER = 10,
    D3DSPR_CONST2 = 11,
    D3DSPR_CONST3 = 12,
    D3DSPR_CONST4 = 13,
    D3DSPR_CONSTBOOL = 14,
    D3DSPR_LOOP = 15,
    D3DSPR_TEMPFLOAT16 = 16,
    D3DSPR_MISCTYPE = 17,
    D3DSPR_LABEL = 18,
    D3DSPR_PREDICATE = 19,
}}
ENUM!{enum D3DSHADER_MISCTYPE_OFFSETS {
    D3DSMO_POSITION = 0,
    D3DSMO_FACE = 1,
}}
ENUM!{enum D3DVS_RASTOUT_OFFSETS {
    D3DSRO_POSITION = 0,
    D3DSRO_FOG,
    D3DSRO_POINT_SIZE,
}}
pub const D3DVS_ADDRESSMODE_SHIFT: DWORD = 13;
pub const D3DVS_ADDRESSMODE_MASK: DWORD = 1 << D3DVS_ADDRESSMODE_SHIFT;
ENUM!{enum D3DVS_ADDRESSMODE_TYPE {
    D3DVS_ADDRMODE_ABSOLUTE = 0 << D3DVS_ADDRESSMODE_SHIFT,
    D3DVS_ADDRMODE_RELATIVE = 1 << D3DVS_ADDRESSMODE_SHIFT,
}}
pub const D3DSHADER_ADDRESSMODE_SHIFT: DWORD = 13;
pub const D3DSHADER_ADDRESSMODE_MASK: DWORD = 1 << D3DSHADER_ADDRESSMODE_SHIFT;
ENUM!{enum D3DSHADER_ADDRESSMODE_TYPE {
    D3DSHADER_ADDRMODE_ABSOLUTE = 0 << D3DSHADER_ADDRESSMODE_SHIFT,
    D3DSHADER_ADDRMODE_RELATIVE = 1 << D3DSHADER_ADDRESSMODE_SHIFT,
}}
pub const D3DVS_SWIZZLE_SHIFT: DWORD = 16;
pub const D3DVS_SWIZZLE_MASK: DWORD = 0x00FF0000;
pub const D3DVS_X_X: DWORD = 0 << D3DVS_SWIZZLE_SHIFT;
pub const D3DVS_X_Y: DWORD = 1 << D3DVS_SWIZZLE_SHIFT;
pub const D3DVS_X_Z: DWORD = 2 << D3DVS_SWIZZLE_SHIFT;
pub const D3DVS_X_W: DWORD = 3 << D3DVS_SWIZZLE_SHIFT;
pub const D3DVS_Y_X: DWORD = 0 << (D3DVS_SWIZZLE_SHIFT + 2);
pub const D3DVS_Y_Y: DWORD = 1 << (D3DVS_SWIZZLE_SHIFT + 2);
pub const D3DVS_Y_Z: DWORD = 2 << (D3DVS_SWIZZLE_SHIFT + 2);
pub const D3DVS_Y_W: DWORD = 3 << (D3DVS_SWIZZLE_SHIFT + 2);
pub const D3DVS_Z_X: DWORD = 0 << (D3DVS_SWIZZLE_SHIFT + 4);
pub const D3DVS_Z_Y: DWORD = 1 << (D3DVS_SWIZZLE_SHIFT + 4);
pub const D3DVS_Z_Z: DWORD = 2 << (D3DVS_SWIZZLE_SHIFT + 4);
pub const D3DVS_Z_W: DWORD = 3 << (D3DVS_SWIZZLE_SHIFT + 4);
pub const D3DVS_W_X: DWORD = 0 << (D3DVS_SWIZZLE_SHIFT + 6);
pub const D3DVS_W_Y: DWORD = 1 << (D3DVS_SWIZZLE_SHIFT + 6);
pub const D3DVS_W_Z: DWORD = 2 << (D3DVS_SWIZZLE_SHIFT + 6);
pub const D3DVS_W_W: DWORD = 3 << (D3DVS_SWIZZLE_SHIFT + 6);
pub const D3DVS_NOSWIZZLE: DWORD = D3DVS_X_X | D3DVS_Y_Y | D3DVS_Z_Z | D3DVS_W_W;
pub const D3DSP_SWIZZLE_SHIFT: DWORD = 16;
pub const D3DSP_SWIZZLE_MASK: DWORD = 0x00FF0000;
pub const D3DSP_NOSWIZZLE: DWORD = (0 << (D3DSP_SWIZZLE_SHIFT + 0))
    | (1 << (D3DSP_SWIZZLE_SHIFT + 2)) | (2 << (D3DSP_SWIZZLE_SHIFT + 4))
    | (3 << (D3DSP_SWIZZLE_SHIFT + 6));
pub const D3DSP_REPLICATERED: DWORD = (0 << (D3DSP_SWIZZLE_SHIFT + 0))
    | (0 << (D3DSP_SWIZZLE_SHIFT + 2)) | (0 << (D3DSP_SWIZZLE_SHIFT + 4))
    | (0 << (D3DSP_SWIZZLE_SHIFT + 6));
pub const D3DSP_REPLICATEGREEN: DWORD = (1 << (D3DSP_SWIZZLE_SHIFT + 0))
    | (1 << (D3DSP_SWIZZLE_SHIFT + 2)) | (1 << (D3DSP_SWIZZLE_SHIFT + 4))
    | (1 << (D3DSP_SWIZZLE_SHIFT + 6));
pub const D3DSP_REPLICATEBLUE: DWORD = (2 << (D3DSP_SWIZZLE_SHIFT + 0))
    | (2 << (D3DSP_SWIZZLE_SHIFT + 2)) | (2 << (D3DSP_SWIZZLE_SHIFT + 4))
    | (2 << (D3DSP_SWIZZLE_SHIFT + 6));
pub const D3DSP_REPLICATEALPHA: DWORD = (3 << (D3DSP_SWIZZLE_SHIFT + 0))
    | (3 << (D3DSP_SWIZZLE_SHIFT + 2)) | (3 << (D3DSP_SWIZZLE_SHIFT + 4))
    | (3 << (D3DSP_SWIZZLE_SHIFT + 6));
pub const D3DSP_SRCMOD_SHIFT: DWORD = 24;
pub const D3DSP_SRCMOD_MASK: DWORD = 0x0F000000;
ENUM!{enum D3DSHADER_PARAM_SRCMOD_TYPE {
    D3DSPSM_NONE = 0 << D3DSP_SRCMOD_SHIFT,
    D3DSPSM_NEG = 1 << D3DSP_SRCMOD_SHIFT,
    D3DSPSM_BIAS = 2 << D3DSP_SRCMOD_SHIFT,
    D3DSPSM_BIASNEG = 3 << D3DSP_SRCMOD_SHIFT,
    D3DSPSM_SIGN = 4 << D3DSP_SRCMOD_SHIFT,
    D3DSPSM_SIGNNEG = 5 << D3DSP_SRCMOD_SHIFT,
    D3DSPSM_COMP = 6 << D3DSP_SRCMOD_SHIFT,
    D3DSPSM_X2 = 7 << D3DSP_SRCMOD_SHIFT,
    D3DSPSM_X2NEG = 8 << D3DSP_SRCMOD_SHIFT,
    D3DSPSM_DZ = 9 << D3DSP_SRCMOD_SHIFT,
    D3DSPSM_DW = 10 << D3DSP_SRCMOD_SHIFT,
    D3DSPSM_ABS = 11 << D3DSP_SRCMOD_SHIFT,
    D3DSPSM_ABSNEG = 12 << D3DSP_SRCMOD_SHIFT,
    D3DSPSM_NOT = 13 << D3DSP_SRCMOD_SHIFT,
}}
pub const D3DSP_MIN_PRECISION_SHIFT: DWORD = 14;
pub const D3DSP_MIN_PRECISION_MASK: DWORD = 0x0000C000;
ENUM!{enum D3DSHADER_MIN_PRECISION {
    D3DMP_DEFAULT = 0,
    D3DMP_16 = 1,
    D3DMP_2_8 = 2,
}}
pub const D3DSI_COMMENTSIZE_SHIFT: DWORD = 16;
pub const D3DSI_COMMENTSIZE_MASK: DWORD = 0x7FFF0000;
pub const D3DPS_END: DWORD = 0x0000FFFF;
pub const D3DVS_END: DWORD = 0x0000FFFF;
ENUM!{enum D3DBASISTYPE {
    D3DBASIS_BEZIER = 0,
    D3DBASIS_BSPLINE = 1,
    D3DBASIS_CATMULL_ROM = 2,
}}
ENUM!{enum D3DDEGREETYPE {
    D3DDEGREE_LINEAR = 1,
    D3DDEGREE_QUADRATIC = 2,
    D3DDEGREE_CUBIC = 3,
    D3DDEGREE_QUINTIC = 5,
}}
ENUM!{enum D3DPATCHEDGESTYLE {
    D3DPATCHEDGE_DISCRETE = 0,
    D3DPATCHEDGE_CONTINUOUS = 1,
}}
ENUM!{enum D3DSTATEBLOCKTYPE {
    D3DSBT_ALL = 1,
    D3DSBT_PIXELSTATE = 2,
    D3DSBT_VERTEXSTATE = 3,
}}
ENUM!{enum D3DVERTEXBLENDFLAGS {
    D3DVBF_DISABLE = 0,
    D3DVBF_1WEIGHTS = 1,
    D3DVBF_2WEIGHTS = 2,
    D3DVBF_3WEIGHTS = 3,
    D3DVBF_TWEENING = 255,
    D3DVBF_0WEIGHTS = 256,
}}
ENUM!{enum D3DTEXTURETRANSFORMFLAGS {
    D3DTTFF_DISABLE = 0,
    D3DTTFF_COUNT1 = 1,
    D3DTTFF_COUNT2 = 2,
    D3DTTFF_COUNT3 = 3,
    D3DTTFF_COUNT4 = 4,
    D3DTTFF_PROJECTED = 256,
}}
pub const D3DFVF_TEXTUREFORMAT2: DWORD = 0;
pub const D3DFVF_TEXTUREFORMAT1: DWORD = 3;
pub const D3DFVF_TEXTUREFORMAT3: DWORD = 1;
pub const D3DFVF_TEXTUREFORMAT4: DWORD = 2;
ENUM!{enum D3DDEVTYPE {
    D3DDEVTYPE_HAL = 1,
    D3DDEVTYPE_REF = 2,
    D3DDEVTYPE_SW = 3,
    D3DDEVTYPE_NULLREF = 4,
}}
ENUM!{enum D3DMULTISAMPLE_TYPE {
    D3DMULTISAMPLE_NONE = 0,
    D3DMULTISAMPLE_NONMASKABLE = 1,
    D3DMULTISAMPLE_2_SAMPLES = 2,
    D3DMULTISAMPLE_3_SAMPLES = 3,
    D3DMULTISAMPLE_4_SAMPLES = 4,
    D3DMULTISAMPLE_5_SAMPLES = 5,
    D3DMULTISAMPLE_6_SAMPLES = 6,
    D3DMULTISAMPLE_7_SAMPLES = 7,
    D3DMULTISAMPLE_8_SAMPLES = 8,
    D3DMULTISAMPLE_9_SAMPLES = 9,
    D3DMULTISAMPLE_10_SAMPLES = 10,
    D3DMULTISAMPLE_11_SAMPLES = 11,
    D3DMULTISAMPLE_12_SAMPLES = 12,
    D3DMULTISAMPLE_13_SAMPLES = 13,
    D3DMULTISAMPLE_14_SAMPLES = 14,
    D3DMULTISAMPLE_15_SAMPLES = 15,
    D3DMULTISAMPLE_16_SAMPLES = 16,
}}
ENUM!{enum D3DFORMAT {
    D3DFMT_UNKNOWN = 0,
    D3DFMT_R8G8B8 = 20,
    D3DFMT_A8R8G8B8 = 21,
    D3DFMT_X8R8G8B8 = 22,
    D3DFMT_R5G6B5 = 23,
    D3DFMT_X1R5G5B5 = 24,
    D3DFMT_A1R5G5B5 = 25,
    D3DFMT_A4R4G4B4 = 26,
    D3DFMT_R3G3B2 = 27,
    D3DFMT_A8 = 28,
    D3DFMT_A8R3G3B2 = 29,
    D3DFMT_X4R4G4B4 = 30,
    D3DFMT_A2B10G10R10 = 31,
    D3DFMT_A8B8G8R8 = 32,
    D3DFMT_X8B8G8R8 = 33,
    D3DFMT_G16R16 = 34,
    D3DFMT_A2R10G10B10 = 35,
    D3DFMT_A16B16G16R16 = 36,
    D3DFMT_A8P8 = 40,
    D3DFMT_P8 = 41,
    D3DFMT_L8 = 50,
    D3DFMT_A8L8 = 51,
    D3DFMT_A4L4 = 52,
    D3DFMT_V8U8 = 60,
    D3DFMT_L6V5U5 = 61,
    D3DFMT_X8L8V8U8 = 62,
    D3DFMT_Q8W8V8U8 = 63,
    D3DFMT_V16U16 = 64,
    D3DFMT_A2W10V10U10 = 67,
    D3DFMT_UYVY = MAKEFOURCC!(b'U', b'Y', b'V', b'Y'),
    D3DFMT_R8G8_B8G8 = MAKEFOURCC!(b'R', b'G', b'B', b'G'),
    D3DFMT_YUY2 = MAKEFOURCC!(b'Y', b'U', b'Y', b'2'),
    D3DFMT_G8R8_G8B8 = MAKEFOURCC!(b'G', b'R', b'G', b'B'),
    D3DFMT_DXT1 = MAKEFOURCC!(b'D', b'X', b'T', b'1'),
    D3DFMT_DXT2 = MAKEFOURCC!(b'D', b'X', b'T', b'2'),
    D3DFMT_DXT3 = MAKEFOURCC!(b'D', b'X', b'T', b'3'),
    D3DFMT_DXT4 = MAKEFOURCC!(b'D', b'X', b'T', b'4'),
    D3DFMT_DXT5 = MAKEFOURCC!(b'D', b'X', b'T', b'5'),
    D3DFMT_D16_LOCKABLE = 70,
    D3DFMT_D32 = 71,
    D3DFMT_D15S1 = 73,
    D3DFMT_D24S8 = 75,
    D3DFMT_D24X8 = 77,
    D3DFMT_D24X4S4 = 79,
    D3DFMT_D16 = 80,
    D3DFMT_D32F_LOCKABLE = 82,
    D3DFMT_D24FS8 = 83,
    D3DFMT_D32_LOCKABLE = 84,
    D3DFMT_S8_LOCKABLE = 85,
    D3DFMT_L16 = 81,
    D3DFMT_VERTEXDATA = 100,
    D3DFMT_INDEX16 = 101,
    D3DFMT_INDEX32 = 102,
    D3DFMT_Q16W16V16U16 = 110,
    D3DFMT_MULTI2_ARGB8 = MAKEFOURCC!(b'M', b'E', b'T', b'1'),
    D3DFMT_R16F = 111,
    D3DFMT_G16R16F = 112,
    D3DFMT_A16B16G16R16F = 113,
    D3DFMT_R32F = 114,
    D3DFMT_G32R32F = 115,
    D3DFMT_A32B32G32R32F = 116,
    D3DFMT_CxV8U8 = 117,
    D3DFMT_A1 = 118,
    D3DFMT_A2B10G10R10_XR_BIAS = 119,
    D3DFMT_BINARYBUFFER = 199,
}}
STRUCT!{struct D3DDISPLAYMODE {
    Width: UINT,
    Height: UINT,
    RefreshRate: UINT,
    Format: D3DFORMAT,
}}
STRUCT!{struct D3DDEVICE_CREATION_PARAMETERS {
    AdapterOrdinal: UINT,
    DeviceType: D3DDEVTYPE,
    hFocusWindow: HWND,
    BehaviorFlags: DWORD,
}}
ENUM!{enum D3DSWAPEFFECT {
    D3DSWAPEFFECT_DISCARD = 1,
    D3DSWAPEFFECT_FLIP = 2,
    D3DSWAPEFFECT_COPY = 3,
    D3DSWAPEFFECT_OVERLAY = 4,
    D3DSWAPEFFECT_FLIPEX = 5,
}}
ENUM!{enum D3DPOOL {
    D3DPOOL_DEFAULT = 0,
    D3DPOOL_MANAGED = 1,
    D3DPOOL_SYSTEMMEM = 2,
    D3DPOOL_SCRATCH = 3,
}}
pub const D3DPRESENT_RATE_DEFAULT: DWORD = 0x00000000;
STRUCT!{struct D3DPRESENT_PARAMETERS {
    BackBufferWidth: UINT,
    BackBufferHeight: UINT,
    BackBufferFormat: D3DFORMAT,
    BackBufferCount: UINT,
    MultiSampleType: D3DMULTISAMPLE_TYPE,
    MultiSampleQuality: DWORD,
    SwapEffect: D3DSWAPEFFECT,
    hDeviceWindow: HWND,
    Windowed: BOOL,
    EnableAutoDepthStencil: BOOL,
    AutoDepthStencilFormat: D3DFORMAT,
    Flags: DWORD,
    FullScreen_RefreshRateInHz: UINT,
    PresentationInterval: UINT,
}}
pub const D3DPRESENTFLAG_LOCKABLE_BACKBUFFER: DWORD = 0x00000001;
pub const D3DPRESENTFLAG_DISCARD_DEPTHSTENCIL: DWORD = 0x00000002;
pub const D3DPRESENTFLAG_DEVICECLIP: DWORD = 0x00000004;
pub const D3DPRESENTFLAG_VIDEO: DWORD = 0x00000010;
pub const D3DPRESENTFLAG_NOAUTOROTATE: DWORD = 0x00000020;
pub const D3DPRESENTFLAG_UNPRUNEDMODE: DWORD = 0x00000040;
pub const D3DPRESENTFLAG_OVERLAY_LIMITEDRGB: DWORD = 0x00000080;
pub const D3DPRESENTFLAG_OVERLAY_YCbCr_BT709: DWORD = 0x00000100;
pub const D3DPRESENTFLAG_OVERLAY_YCbCr_xvYCC: DWORD = 0x00000200;
pub const D3DPRESENTFLAG_RESTRICTED_CONTENT: DWORD = 0x00000400;
pub const D3DPRESENTFLAG_RESTRICT_SHARED_RESOURCE_DRIVER: DWORD = 0x00000800;
STRUCT!{struct D3DGAMMARAMP {
    red: [WORD; 256],
    green: [WORD; 256],
    blue: [WORD; 256],
}}
ENUM!{enum D3DBACKBUFFER_TYPE {
    D3DBACKBUFFER_TYPE_MONO = 0,
    D3DBACKBUFFER_TYPE_LEFT = 1,
    D3DBACKBUFFER_TYPE_RIGHT = 2,
}}
ENUM!{enum D3DRESOURCETYPE {
    D3DRTYPE_SURFACE = 1,
    D3DRTYPE_VOLUME = 2,
    D3DRTYPE_TEXTURE = 3,
    D3DRTYPE_VOLUMETEXTURE = 4,
    D3DRTYPE_CUBETEXTURE = 5,
    D3DRTYPE_VERTEXBUFFER = 6,
    D3DRTYPE_INDEXBUFFER = 7,
}}
pub const D3DUSAGE_RENDERTARGET: DWORD = 0x00000001;
pub const D3DUSAGE_DEPTHSTENCIL: DWORD = 0x00000002;
pub const D3DUSAGE_DYNAMIC: DWORD = 0x00000200;
pub const D3DUSAGE_NONSECURE: DWORD = 0x00800000;
pub const D3DUSAGE_AUTOGENMIPMAP: DWORD = 0x00000400;
pub const D3DUSAGE_DMAP: DWORD = 0x00004000;
pub const D3DUSAGE_QUERY_LEGACYBUMPMAP: DWORD = 0x00008000;
pub const D3DUSAGE_QUERY_SRGBREAD: DWORD = 0x00010000;
pub const D3DUSAGE_QUERY_FILTER: DWORD = 0x00020000;
pub const D3DUSAGE_QUERY_SRGBWRITE: DWORD = 0x00040000;
pub const D3DUSAGE_QUERY_POSTPIXELSHADER_BLENDING: DWORD = 0x00080000;
pub const D3DUSAGE_QUERY_VERTEXTEXTURE: DWORD = 0x00100000;
pub const D3DUSAGE_QUERY_WRAPANDMIP: DWORD = 0x00200000;
pub const D3DUSAGE_WRITEONLY: DWORD = 0x00000008;
pub const D3DUSAGE_SOFTWAREPROCESSING: DWORD = 0x00000010;
pub const D3DUSAGE_DONOTCLIP: DWORD = 0x00000020;
pub const D3DUSAGE_POINTS: DWORD = 0x00000040;
pub const D3DUSAGE_RTPATCHES: DWORD = 0x00000080;
pub const D3DUSAGE_NPATCHES: DWORD = 0x00000100;
pub const D3DUSAGE_TEXTAPI: DWORD = 0x10000000;
pub const D3DUSAGE_RESTRICTED_CONTENT: DWORD = 0x00000800;
pub const D3DUSAGE_RESTRICT_SHARED_RESOURCE: DWORD = 0x00002000;
pub const D3DUSAGE_RESTRICT_SHARED_RESOURCE_DRIVER: DWORD = 0x00001000;
ENUM!{enum D3DCUBEMAP_FACES {
    D3DCUBEMAP_FACE_POSITIVE_X = 0,
    D3DCUBEMAP_FACE_NEGATIVE_X = 1,
    D3DCUBEMAP_FACE_POSITIVE_Y = 2,
    D3DCUBEMAP_FACE_NEGATIVE_Y = 3,
    D3DCUBEMAP_FACE_POSITIVE_Z = 4,
    D3DCUBEMAP_FACE_NEGATIVE_Z = 5,
}}
pub const D3DLOCK_READONLY: DWORD = 0x00000010;
pub const D3DLOCK_DISCARD: DWORD = 0x00002000;
pub const D3DLOCK_NOOVERWRITE: DWORD = 0x00001000;
pub const D3DLOCK_NOSYSLOCK: DWORD = 0x00000800;
pub const D3DLOCK_DONOTWAIT: DWORD = 0x00004000;
pub const D3DLOCK_NO_DIRTY_UPDATE: DWORD = 0x00008000;
STRUCT!{struct D3DVERTEXBUFFER_DESC {
    Format: D3DFORMAT,
    Type: D3DRESOURCETYPE,
    Usage: DWORD,
    Pool: D3DPOOL,
    Size: UINT,
    FVF: DWORD,
}}
STRUCT!{struct D3DINDEXBUFFER_DESC {
    Format: D3DFORMAT,
    Type: D3DRESOURCETYPE,
    Usage: DWORD,
    Pool: D3DPOOL,
    Size: UINT,
}}
STRUCT!{struct D3DSURFACE_DESC {
    Format: D3DFORMAT,
    Type: D3DRESOURCETYPE,
    Usage: DWORD,
    Pool: D3DPOOL,
    MultiSampleType: D3DMULTISAMPLE_TYPE,
    MultiSampleQuality: DWORD,
    Width: UINT,
    Height: UINT,
}}
STRUCT!{struct D3DVOLUME_DESC {
    Format: D3DFORMAT,
    Type: D3DRESOURCETYPE,
    Usage: DWORD,
    Pool: D3DPOOL,
    Width: UINT,
    Height: UINT,
    Depth: UINT,
}}
STRUCT!{struct D3DLOCKED_RECT {
    Pitch: INT,
    pBits: *mut c_void,
}}
STRUCT!{struct D3DBOX {
    Left: UINT,
    Top: UINT,
    Right: UINT,
    Bottom: UINT,
    Front: UINT,
    Back: UINT,
}}
STRUCT!{struct D3DLOCKED_BOX {
    RowPitch: INT,
    SlicePitch: INT,
    pBits: *mut c_void,
}}
STRUCT!{struct D3DRANGE {
    Offset: UINT,
    Size: UINT,
}}
STRUCT!{struct D3DRECTPATCH_INFO {
    StartVertexOffsetWidth: UINT,
    StartVertexOffsetHeight: UINT,
    Width: UINT,
    Height: UINT,
    Stride: UINT,
    Basis: D3DBASISTYPE,
    Degree: D3DDEGREETYPE,
}}
STRUCT!{struct D3DTRIPATCH_INFO {
    StartVertexOffset: UINT,
    NumVertices: UINT,
    Basis: D3DBASISTYPE,
    Degree: D3DDEGREETYPE,
}}
pub const MAX_DEVICE_IDENTIFIER_STRING: usize = 512;
// FIXME packed(4)
STRUCT!{#[cfg_attr(target_arch = "x86", repr(packed))] struct D3DADAPTER_IDENTIFIER9 {
    Driver: [c_char; MAX_DEVICE_IDENTIFIER_STRING],
    Description: [c_char; MAX_DEVICE_IDENTIFIER_STRING],
    DeviceName: [c_char; 32],
    DriverVersion: LARGE_INTEGER,
    VendorId: DWORD,
    DeviceId: DWORD,
    SubSysId: DWORD,
    Revision: DWORD,
    DeviceIdentifier: GUID,
    WHQLLevel: DWORD,
}}
STRUCT!{struct D3DRASTER_STATUS {
    InVBlank: BOOL,
    ScanLine: UINT,
}}
ENUM!{enum D3DDEBUGMONITORTOKENS {
    D3DDMT_ENABLE = 0,
    D3DDMT_DISABLE = 1,
}}
ENUM!{enum D3DQUERYTYPE {
    D3DQUERYTYPE_VCACHE = 4,
    D3DQUERYTYPE_RESOURCEMANAGER = 5,
    D3DQUERYTYPE_VERTEXSTATS = 6,
    D3DQUERYTYPE_EVENT = 8,
    D3DQUERYTYPE_OCCLUSION = 9,
    D3DQUERYTYPE_TIMESTAMP = 10,
    D3DQUERYTYPE_TIMESTAMPDISJOINT = 11,
    D3DQUERYTYPE_TIMESTAMPFREQ = 12,
    D3DQUERYTYPE_PIPELINETIMINGS = 13,
    D3DQUERYTYPE_INTERFACETIMINGS = 14,
    D3DQUERYTYPE_VERTEXTIMINGS = 15,
    D3DQUERYTYPE_PIXELTIMINGS = 16,
    D3DQUERYTYPE_BANDWIDTHTIMINGS = 17,
    D3DQUERYTYPE_CACHEUTILIZATION = 18,
    D3DQUERYTYPE_MEMORYPRESSURE = 19,
}}
pub const D3DISSUE_END: DWORD = 1 << 0;
pub const D3DISSUE_BEGIN: DWORD = 1 << 1;
pub const D3DGETDATA_FLUSH: DWORD = 1 << 0;
STRUCT!{struct D3DRESOURCESTATS {
    bThrashing: BOOL,
    ApproxBytesDownloaded: DWORD,
    NumEvicts: DWORD,
    NumVidCreates: DWORD,
    LastPri: DWORD,
    NumUsed: DWORD,
    NumUsedInVidMem: DWORD,
    WorkingSet: DWORD,
    WorkingSetBytes: DWORD,
    TotalManaged: DWORD,
    TotalBytes: DWORD,
}}
pub const D3DRTYPECOUNT: usize = D3DRTYPE_INDEXBUFFER as usize + 1;
STRUCT!{struct D3DDEVINFO_RESOURCEMANAGER {
    stats: [D3DRESOURCESTATS; D3DRTYPECOUNT],
}}
pub type LPD3DDEVINFO_RESOURCEMANAGER = *mut D3DDEVINFO_RESOURCEMANAGER;
STRUCT!{struct D3DDEVINFO_D3DVERTEXSTATS {
    NumRenderedTriangles: DWORD,
    NumExtraClippingTriangles: DWORD,
}}
pub type LPD3DDEVINFO_D3DVERTEXSTATS = *mut D3DDEVINFO_D3DVERTEXSTATS;
STRUCT!{struct D3DDEVINFO_VCACHE {
    Pattern: DWORD,
    OptMethod: DWORD,
    CacheSize: DWORD,
    MagicNumber: DWORD,
}}
pub type LPD3DDEVINFO_VCACHE = *mut D3DDEVINFO_VCACHE;
STRUCT!{struct D3DDEVINFO_D3D9PIPELINETIMINGS {
    VertexProcessingTimePercent: FLOAT,
    PixelProcessingTimePercent: FLOAT,
    OtherGPUProcessingTimePercent: FLOAT,
    GPUIdleTimePercent: FLOAT,
}}
STRUCT!{struct D3DDEVINFO_D3D9INTERFACETIMINGS {
    WaitingForGPUToUseApplicationResourceTimePercent: FLOAT,
    WaitingForGPUToAcceptMoreCommandsTimePercent: FLOAT,
    WaitingForGPUToStayWithinLatencyTimePercent: FLOAT,
    WaitingForGPUExclusiveResourceTimePercent: FLOAT,
    WaitingForGPUOtherTimePercent: FLOAT,
}}
STRUCT!{struct D3DDEVINFO_D3D9STAGETIMINGS {
    MemoryProcessingPercent: FLOAT,
    ComputationProcessingPercent: FLOAT,
}}
STRUCT!{struct D3DDEVINFO_D3D9BANDWIDTHTIMINGS {
    MaxBandwidthUtilized: FLOAT,
    FrontEndUploadMemoryUtilizedPercent: FLOAT,
    VertexRateUtilizedPercent: FLOAT,
    TriangleSetupRateUtilizedPercent: FLOAT,
    FillRateUtilizedPercent: FLOAT,
}}
STRUCT!{struct D3DDEVINFO_D3D9CACHEUTILIZATION {
    TextureCacheHitRate: FLOAT,
    PostTransformVertexCacheHitRate: FLOAT,
}}
// FIXME packed(4)
STRUCT!{#[cfg_attr(target_arch = "x86", repr(packed))] struct D3DMEMORYPRESSURE {
    BytesEvictedFromProcess: UINT64,
    SizeOfInefficientAllocation: UINT64,
    LevelOfEfficiency: DWORD,
}}
ENUM!{enum D3DCOMPOSERECTSOP {
    D3DCOMPOSERECTS_COPY = 1,
    D3DCOMPOSERECTS_OR = 2,
    D3DCOMPOSERECTS_AND = 3,
    D3DCOMPOSERECTS_NEG = 4,
}}
STRUCT!{struct D3DCOMPOSERECTDESC {
    X: USHORT,
    Y: USHORT,
    Width: USHORT,
    Height: USHORT,
}}
STRUCT!{struct D3DCOMPOSERECTDESTINATION {
    SrcRectIndex: USHORT,
    Reserved: USHORT,
    X: SHORT,
    Y: SHORT,
}}
pub const D3DCOMPOSERECTS_MAXNUMRECTS: DWORD = 0xFFFF;
pub const D3DCONVOLUTIONMONO_MAXWIDTH: DWORD = 7;
pub const D3DCONVOLUTIONMONO_MAXHEIGHT: DWORD = D3DCONVOLUTIONMONO_MAXWIDTH;
pub const D3DFMT_A1_SURFACE_MAXWIDTH: DWORD = 8192;
pub const D3DFMT_A1_SURFACE_MAXHEIGHT: DWORD = 2048;
// FIXME packed(4)
STRUCT!{#[cfg_attr(target_arch = "x86", repr(packed))] struct D3DPRESENTSTATS {
    PresentCount: UINT,
    PresentRefreshCount: UINT,
    SyncRefreshCount: UINT,
    SyncQPCTime: LARGE_INTEGER,
    SyncGPUTime: LARGE_INTEGER,
}}
ENUM!{enum D3DSCANLINEORDERING {
    D3DSCANLINEORDERING_UNKNOWN = 0,
    D3DSCANLINEORDERING_PROGRESSIVE = 1,
    D3DSCANLINEORDERING_INTERLACED = 2,
}}
STRUCT!{struct D3DDISPLAYMODEEX {
    Size: UINT,
    Width: UINT,
    Height: UINT,
    RefreshRate: UINT,
    Format: D3DFORMAT,
    ScanLineOrdering: D3DSCANLINEORDERING,
}}
STRUCT!{struct D3DDISPLAYMODEFILTER {
    Size: UINT,
    Format: D3DFORMAT,
    ScanLineOrdering: D3DSCANLINEORDERING,
}}
ENUM!{enum D3DDISPLAYROTATION {
    D3DDISPLAYROTATION_IDENTITY = 1,
    D3DDISPLAYROTATION_90 = 2,
    D3DDISPLAYROTATION_180 = 3,
    D3DDISPLAYROTATION_270 = 4,
}}
pub const D3D9_RESOURCE_PRIORITY_MINIMUM: DWORD = 0x28000000;
pub const D3D9_RESOURCE_PRIORITY_LOW: DWORD = 0x50000000;
pub const D3D9_RESOURCE_PRIORITY_NORMAL: DWORD = 0x78000000;
pub const D3D9_RESOURCE_PRIORITY_HIGH: DWORD = 0xa0000000;
pub const D3D9_RESOURCE_PRIORITY_MAXIMUM: DWORD = 0xc8000000;
pub const D3D_OMAC_SIZE: usize = 16;
STRUCT!{struct D3D_OMAC {
    Omac: [BYTE; D3D_OMAC_SIZE],
}}
ENUM!{enum D3DAUTHENTICATEDCHANNELTYPE {
    D3DAUTHENTICATEDCHANNEL_D3D9 = 1,
    D3DAUTHENTICATEDCHANNEL_DRIVER_SOFTWARE = 2,
    D3DAUTHENTICATEDCHANNEL_DRIVER_HARDWARE = 3,
}}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERY_INPUT {
    QueryType: GUID,
    hChannel: HANDLE,
    SequenceNumber: UINT,
}}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT {
    omac: D3D_OMAC,
    QueryType: GUID,
    hChannel: HANDLE,
    SequenceNumber: UINT,
    ReturnCode: HRESULT,
}}
DEFINE_GUID!{D3DAUTHENTICATEDQUERY_PROTECTION,
    0xa84eb584, 0xc495, 0x48aa, 0xb9, 0x4d, 0x8b, 0xd2, 0xd6, 0xfb, 0xce, 0x5}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_PROTECTION_FLAGS {
    Value: UINT,
}}
BITFIELD!{D3DAUTHENTICATEDCHANNEL_PROTECTION_FLAGS Value: UINT [
    ProtectionEnabled set_ProtectionEnabled[0..1],
    OverlayOrFullscreenRequired set_OverlayOrFullscreenRequired[1..2],
]}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYPROTECTION_OUTPUT {
    Output: D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT,
    ProtectionFlags: D3DAUTHENTICATEDCHANNEL_PROTECTION_FLAGS,
}}
DEFINE_GUID!{D3DAUTHENTICATEDQUERY_CHANNELTYPE,
    0xbc1b18a5, 0xb1fb, 0x42ab, 0xbd, 0x94, 0xb5, 0x82, 0x8b, 0x4b, 0xf7, 0xbe}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYCHANNELTYPE_OUTPUT {
    Output: D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT,
    ChannelType: D3DAUTHENTICATEDCHANNELTYPE,
}}
DEFINE_GUID!{D3DAUTHENTICATEDQUERY_DEVICEHANDLE,
    0xec1c539d, 0x8cff, 0x4e2a, 0xbc, 0xc4, 0xf5, 0x69, 0x2f, 0x99, 0xf4, 0x80}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYDEVICEHANDLE_OUTPUT {
    Output: D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT,
    DeviceHandle: HANDLE,
}}
DEFINE_GUID!{D3DAUTHENTICATEDQUERY_CRYPTOSESSION,
    0x2634499e, 0xd018, 0x4d74, 0xac, 0x17, 0x7f, 0x72, 0x40, 0x59, 0x52, 0x8d}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYCRYPTOSESSION_INPUT {
    Input: D3DAUTHENTICATEDCHANNEL_QUERY_INPUT,
    DXVA2DecodeHandle: HANDLE,
}}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYCRYPTOSESSION_OUTPUT {
    Output: D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT,
    DXVA2DecodeHandle: HANDLE,
    CryptoSessionHandle: HANDLE,
    DeviceHandle: HANDLE,
}}
DEFINE_GUID!{D3DAUTHENTICATEDQUERY_RESTRICTEDSHAREDRESOURCEPROCESSCOUNT,
    0xdb207b3, 0x9450, 0x46a6, 0x82, 0xde, 0x1b, 0x96, 0xd4, 0x4f, 0x9c, 0xf2}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYRESTRICTEDSHAREDRESOURCEPROCESSCOUNT_OUTPUT {
    Output: D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT,
    NumRestrictedSharedResourceProcesses: UINT,
}}
DEFINE_GUID!{D3DAUTHENTICATEDQUERY_RESTRICTEDSHAREDRESOURCEPROCESS,
    0x649bbadb, 0xf0f4, 0x4639, 0xa1, 0x5b, 0x24, 0x39, 0x3f, 0xc3, 0xab, 0xac}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYRESTRICTEDSHAREDRESOURCEPROCESS_INPUT {
    Input: D3DAUTHENTICATEDCHANNEL_QUERY_INPUT,
    ProcessIndex: UINT,
}}
ENUM!{enum D3DAUTHENTICATEDCHANNEL_PROCESSIDENTIFIERTYPE {
    PROCESSIDTYPE_UNKNOWN = 0,
    PROCESSIDTYPE_DWM = 1,
    PROCESSIDTYPE_HANDLE = 2,
}}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYRESTRICTEDSHAREDRESOURCEPROCESS_OUTPUT {
    Output: D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT,
    ProcessIndex: UINT,
    ProcessIdentifer: D3DAUTHENTICATEDCHANNEL_PROCESSIDENTIFIERTYPE,
    ProcessHandle: HANDLE,
}}
DEFINE_GUID!{D3DAUTHENTICATEDQUERY_UNRESTRICTEDPROTECTEDSHAREDRESOURCECOUNT,
    0x12f0bd6, 0xe662, 0x4474, 0xbe, 0xfd, 0xaa, 0x53, 0xe5, 0x14, 0x3c, 0x6d}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYUNRESTRICTEDPROTECTEDSHAREDRESOURCECOUNT_OUTPUT {
    Output: D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT,
    NumUnrestrictedProtectedSharedResources: UINT,
}}
DEFINE_GUID!{D3DAUTHENTICATEDQUERY_OUTPUTIDCOUNT,
    0x2c042b5e, 0x8c07, 0x46d5, 0xaa, 0xbe, 0x8f, 0x75, 0xcb, 0xad, 0x4c, 0x31}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYOUTPUTIDCOUNT_INPUT {
    Input: D3DAUTHENTICATEDCHANNEL_QUERY_INPUT,
    DeviceHandle: HANDLE,
    CryptoSessionHandle: HANDLE,
}}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYOUTPUTIDCOUNT_OUTPUT {
    Output: D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT,
    DeviceHandle: HANDLE,
    CryptoSessionHandle: HANDLE,
    NumOutputIDs: UINT,
}}
DEFINE_GUID!{D3DAUTHENTICATEDQUERY_OUTPUTID,
    0x839ddca3, 0x9b4e, 0x41e4, 0xb0, 0x53, 0x89, 0x2b, 0xd2, 0xa1, 0x1e, 0xe7}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYOUTPUTID_INPUT {
    Input: D3DAUTHENTICATEDCHANNEL_QUERY_INPUT,
    DeviceHandle: HANDLE,
    CryptoSessionHandle: HANDLE,
    OutputIDIndex: UINT,
}}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYOUTPUTID_OUTPUT {
    Output: D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT,
    DeviceHandle: HANDLE,
    CryptoSessionHandle: HANDLE,
    OutputIDIndex: UINT,
    OutputID: UINT64,
}}
DEFINE_GUID!{D3DAUTHENTICATEDQUERY_ACCESSIBILITYATTRIBUTES,
    0x6214d9d2, 0x432c, 0x4abb, 0x9f, 0xce, 0x21, 0x6e, 0xea, 0x26, 0x9e, 0x3b}
ENUM!{enum D3DBUSTYPE {
    D3DBUSTYPE_OTHER = 0x00000000,
    D3DBUSTYPE_PCI = 0x00000001,
    D3DBUSTYPE_PCIX = 0x00000002,
    D3DBUSTYPE_PCIEXPRESS = 0x00000003,
    D3DBUSTYPE_AGP = 0x00000004,
    D3DBUSIMPL_MODIFIER_INSIDE_OF_CHIPSET = 0x00010000,
    MD3DBUSIMPL_ODIFIER_TRACKS_ON_MOTHER_BOARD_TO_CHIP = 0x00020000,
    D3DBUSIMPL_MODIFIER_TRACKS_ON_MOTHER_BOARD_TO_SOCKET = 0x00030000,
    D3DBUSIMPL_MODIFIER_DAUGHTER_BOARD_CONNECTOR = 0x00040000,
    D3DBUSIMPL_MODIFIER_DAUGHTER_BOARD_CONNECTOR_INSIDE_OF_NUAE = 0x00050000,
    D3DBUSIMPL_MODIFIER_NON_STANDARD = 0x80000000,
}}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYINFOBUSTYPE_OUTPUT {
    Output: D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT,
    BusType: D3DBUSTYPE,
    bAccessibleInContiguousBlocks: BOOL,
    bAccessibleInNonContiguousBlocks: BOOL,
}}
DEFINE_GUID!{D3DAUTHENTICATEDQUERY_ENCRYPTIONWHENACCESSIBLEGUIDCOUNT,
    0xb30f7066, 0x203c, 0x4b07, 0x93, 0xfc, 0xce, 0xaa, 0xfd, 0x61, 0x24, 0x1e}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYEVICTIONENCRYPTIONGUIDCOUNT_OUTPUT {
    Output: D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT,
    NumEncryptionGuids: UINT,
}}
DEFINE_GUID!{D3DAUTHENTICATEDQUERY_ENCRYPTIONWHENACCESSIBLEGUID,
    0xf83a5958, 0xe986, 0x4bda, 0xbe, 0xb0, 0x41, 0x1f, 0x6a, 0x7a, 0x1, 0xb7}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYEVICTIONENCRYPTIONGUID_INPUT {
    Input: D3DAUTHENTICATEDCHANNEL_QUERY_INPUT,
    EncryptionGuidIndex: UINT,
}}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYEVICTIONENCRYPTIONGUID_OUTPUT {
    Output: D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT,
    EncryptionGuidIndex: UINT,
    EncryptionGuid: GUID,
}}
DEFINE_GUID!{D3DAUTHENTICATEDQUERY_CURRENTENCRYPTIONWHENACCESSIBLE,
    0xec1791c7, 0xdad3, 0x4f15, 0x9e, 0xc3, 0xfa, 0xa9, 0x3d, 0x60, 0xd4, 0xf0}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_QUERYUNCOMPRESSEDENCRYPTIONLEVEL_OUTPUT {
    Output: D3DAUTHENTICATEDCHANNEL_QUERY_OUTPUT,
    EncryptionGuid: GUID,
}}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_CONFIGURE_INPUT {
    omac: D3D_OMAC,
    ConfigureType: GUID,
    hChannel: HANDLE,
    SequenceNumber: UINT,
}}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_CONFIGURE_OUTPUT {
    omac: D3D_OMAC,
    ConfigureType: GUID,
    hChannel: HANDLE,
    SequenceNumber: UINT,
    ReturnCode: HRESULT,
}}
DEFINE_GUID!{D3DAUTHENTICATEDCONFIGURE_INITIALIZE,
    0x6114bdb, 0x3523, 0x470a, 0x8d, 0xca, 0xfb, 0xc2, 0x84, 0x51, 0x54, 0xf0}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_CONFIGUREINITIALIZE {
    Parameters: D3DAUTHENTICATEDCHANNEL_CONFIGURE_INPUT,
    StartSequenceQuery: UINT,
    StartSequenceConfigure: UINT,
}}
DEFINE_GUID!{D3DAUTHENTICATEDCONFIGURE_PROTECTION,
    0x50455658, 0x3f47, 0x4362, 0xbf, 0x99, 0xbf, 0xdf, 0xcd, 0xe9, 0xed, 0x29}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_CONFIGUREPROTECTION {
    Parameters: D3DAUTHENTICATEDCHANNEL_CONFIGURE_INPUT,
    Protections: D3DAUTHENTICATEDCHANNEL_PROTECTION_FLAGS,
}}
DEFINE_GUID!{D3DAUTHENTICATEDCONFIGURE_CRYPTOSESSION,
    0x6346cc54, 0x2cfc, 0x4ad4, 0x82, 0x24, 0xd1, 0x58, 0x37, 0xde, 0x77, 0x0}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_CONFIGURECRYPTOSESSION {
    Parameters: D3DAUTHENTICATEDCHANNEL_CONFIGURE_INPUT,
    DXVA2DecodeHandle: HANDLE,
    CryptoSessionHandle: HANDLE,
    DeviceHandle: HANDLE,
}}
DEFINE_GUID!{D3DAUTHENTICATEDCONFIGURE_SHAREDRESOURCE,
    0x772d047, 0x1b40, 0x48e8, 0x9c, 0xa6, 0xb5, 0xf5, 0x10, 0xde, 0x9f, 0x1}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_CONFIGURESHAREDRESOURCE {
    Parameters: D3DAUTHENTICATEDCHANNEL_CONFIGURE_INPUT,
    ProcessIdentiferType: D3DAUTHENTICATEDCHANNEL_PROCESSIDENTIFIERTYPE,
    ProcessHandle: HANDLE,
    AllowAccess: BOOL,
}}
DEFINE_GUID!{D3DAUTHENTICATEDCONFIGURE_ENCRYPTIONWHENACCESSIBLE,
    0x41fff286, 0x6ae0, 0x4d43, 0x9d, 0x55, 0xa4, 0x6e, 0x9e, 0xfd, 0x15, 0x8a}
STRUCT!{struct D3DAUTHENTICATEDCHANNEL_CONFIGUREUNCOMPRESSEDENCRYPTION {
    Parameters: D3DAUTHENTICATEDCHANNEL_CONFIGURE_INPUT,
    EncryptionGuid: GUID,
}}
STRUCT!{struct D3DENCRYPTED_BLOCK_INFO {
    NumEncryptedBytesAtBeginning: UINT,
    NumBytesInSkipPattern: UINT,
    NumBytesInEncryptPattern: UINT,
}}
STRUCT!{struct D3DAES_CTR_IV {
    IV: UINT64,
    Count: UINT64,
}}
