#!/usr/bin/perl
#
# Special wrapper script to generate the actual pod2man script.  This is
# required for proper start-up code on non-UNIX platforms, and is used inside
# Perl core.

use 5.006;
use strict;
use warnings;

use Config qw(%Config);
use Cwd qw(cwd);
use File::Basename qw(basename dirname);

# List explicitly here the variables you want Configure to generate.
# Metaconfig only looks for shell variables, so you have to mention them as if
# they were shell variables, not %Config entries.  Thus you write
#  $startperl
# to ensure Configure will look for $Config{startperl}.

# This forces PL files to create target in same directory as PL file.
# This is so that make depend always knows where to find PL derivatives.
chdir(dirname($0)) or die "Cannot change directories: $!\n";
my $file = basename($0, '.PL');
if ($^O eq 'VMS') {
    $file .= '.com';
}

# Create the generated script.
## no critic (InputOutput::RequireBriefOpen)
## no critic (InputOutput::RequireCheckedSyscalls)
open(my $out, '>', $file) or die "Cannot create $file: $!\n";
print "Extracting $file (with variable substitutions)\n";
## use critic

# In this section, Perl variables will be expanded during extraction.  You can
# use $Config{...} to use Configure variables.
print {$out} <<"PREAMBLE" or die "Cannot write to $file: $!\n";
$Config{startperl}
    eval 'exec $Config{perlpath} -S \$0 \${1+"\$@"}'
        if \$running_under_some_shell;
PREAMBLE

# In the following, Perl variables are not expanded during extraction.
print {$out} <<'SCRIPT_BODY' or die "Cannot write to $file: $!\n";

# Convert POD data to formatted *roff input.
#
# The driver script for Pod::Man.
#
# SPDX-License-Identifier: GPL-1.0-or-later OR Artistic-1.0-Perl

use 5.006;
use strict;
use warnings;

use Getopt::Long qw(GetOptions);
use Pod::Man ();
use Pod::Usage qw(pod2usage);

use strict;

# Clean up $0 for error reporting.
$0 =~ s%.*/%%;

# Insert -- into @ARGV before any single dash argument to hide it from
# Getopt::Long; we want to interpret it as meaning stdin.
my $stdin;
@ARGV = map { $_ eq '-' && !$stdin++ ? ('--', $_) : $_ } @ARGV;

# Parse our options, trying to retain backward compatibility with pod2man but
# allowing short forms as well.  --lax is currently ignored.
my %options;
Getopt::Long::config ('bundling_override');
GetOptions (\%options, 'center|c=s', 'date|d=s', 'errors=s', 'fixed=s',
            'fixedbold=s', 'fixeditalic=s', 'fixedbolditalic=s', 'help|h',
            'lax|l', 'lquote=s', 'name|n=s', 'nourls', 'official|o',
            'quotes|q=s', 'release|r=s', 'rquote=s', 'section|s=s', 'stderr',
            'verbose|v', 'utf8|u')
    or exit 1;
pod2usage (0) if $options{help};

# Official sets --center, but don't override things explicitly set.
if ($options{official} && !defined $options{center}) {
    $options{center} = 'Perl Programmers Reference Guide';
}

# Verbose is only our flag, not a Pod::Man flag.
my $verbose = $options{verbose};
delete $options{verbose};

# This isn't a valid Pod::Man option and is only accepted for backward
# compatibility.
delete $options{lax};

# If neither stderr nor errors is set, default to errors = die.
if (!defined $options{stderr} && !defined $options{errors}) {
    $options{errors} = 'die';
}

# Initialize and run the formatter, pulling a pair of input and output off at
# a time.  For each file, we check whether the document was completely empty
# and, if so, will remove the created file and exit with a non-zero exit
# status.
my $parser = Pod::Man->new (%options);
my $status = 0;
my @files;
do {
    @files = splice (@ARGV, 0, 2);
    print "  $files[1]\n" if $verbose;
    $parser->parse_from_file (@files);
    if ($parser->{CONTENTLESS}) {
        $status = 1;
        if (defined $files[0]) {
            warn "$0: unable to format $files[0]\n";
        } else {
            warn "$0: unable to format standard input\n";
        }
        if (defined ($files[1]) and $files[1] ne '-') {
            unlink $files[1] unless (-s $files[1]);
        }
    }
} while (@ARGV);
exit $status;

__END__

=for stopwords
en em --stderr stderr --utf8 UTF-8 overdo markup MT-LEVEL Allbery Solaris URL
troff troff-specific formatters uppercased Christiansen --nourls UTC prepend
lquote rquote

=head1 NAME

pod2man - Convert POD data to formatted *roff input

=head1 SYNOPSIS

pod2man [B<--center>=I<string>] [B<--date>=I<string>] [B<--errors>=I<style>]
    [B<--fixed>=I<font>] [B<--fixedbold>=I<font>] [B<--fixeditalic>=I<font>]
    [B<--fixedbolditalic>=I<font>] [B<--name>=I<name>] [B<--nourls>]
    [B<--official>] [B<--release>=I<version>] [B<--section>=I<manext>]
    [B<--quotes>=I<quotes>] [B<--lquote>=I<quote>] [B<--rquote>=I<quote>]
    [B<--stderr>] [B<--utf8>] [B<--verbose>] [I<input> [I<output>] ...]

pod2man B<--help>

=head1 DESCRIPTION

B<pod2man> is a front-end for Pod::Man, using it to generate *roff input
from POD source.  The resulting *roff code is suitable for display on a
terminal using nroff(1), normally via man(1), or printing using troff(1).

I<input> is the file to read for POD source (the POD can be embedded in
code).  If I<input> isn't given, it defaults to C<STDIN>.  I<output>, if
given, is the file to which to write the formatted output.  If I<output>
isn't given, the formatted output is written to C<STDOUT>.  Several POD
files can be processed in the same B<pod2man> invocation (saving module
load and compile times) by providing multiple pairs of I<input> and
I<output> files on the command line.

B<--section>, B<--release>, B<--center>, B<--date>, and B<--official> can
be used to set the headers and footers to use; if not given, Pod::Man will
assume various defaults.  See below or L<Pod::Man> for details.

B<pod2man> assumes that your *roff formatters have a fixed-width font
named C<CW>.  If yours is called something else (like C<CR>), use
B<--fixed> to specify it.  This generally only matters for troff output
for printing.  Similarly, you can set the fonts used for bold, italic, and
bold italic fixed-width output.

Besides the obvious pod conversions, Pod::Man, and therefore pod2man also
takes care of formatting func(), func(n), and simple variable references
like $foo or @bar so you don't have to use code escapes for them; complex
expressions like C<$fred{'stuff'}> will still need to be escaped, though.
It also translates dashes that aren't used as hyphens into en dashes, makes
long dashes--like this--into proper em dashes, fixes "paired quotes," and
takes care of several other troff-specific tweaks.  See L<Pod::Man> for
complete information.

=head1 OPTIONS

=over 4

=item B<-c> I<string>, B<--center>=I<string>

Sets the centered page header for the C<.TH> macro to I<string>.  The
default is "User Contributed Perl Documentation", but also see
B<--official> below.

=item B<-d> I<string>, B<--date>=I<string>

Set the left-hand footer string for the C<.TH> macro to I<string>.  By
default, the modification date of the input file will be used, or the
current date if input comes from C<STDIN>, and will be based on UTC (so
that the output will be reproducible regardless of local time zone).

=item B<--errors>=I<style>

Set the error handling style.  C<die> says to throw an exception on any
POD formatting error.  C<stderr> says to report errors on standard error,
but not to throw an exception.  C<pod> says to include a POD ERRORS
section in the resulting documentation summarizing the errors.  C<none>
ignores POD errors entirely, as much as possible.

The default is C<die>.

=item B<--fixed>=I<font>

The fixed-width font to use for verbatim text and code.  Defaults to
C<CW>.  Some systems may want C<CR> instead.  Only matters for troff(1)
output.

=item B<--fixedbold>=I<font>

Bold version of the fixed-width font.  Defaults to C<CB>.  Only matters
for troff(1) output.

=item B<--fixeditalic>=I<font>

Italic version of the fixed-width font (actually, something of a misnomer,
since most fixed-width fonts only have an oblique version, not an italic
version).  Defaults to C<CI>.  Only matters for troff(1) output.

=item B<--fixedbolditalic>=I<font>

Bold italic (probably actually oblique) version of the fixed-width font.
Pod::Man doesn't assume you have this, and defaults to C<CB>.  Some
systems (such as Solaris) have this font available as C<CX>.  Only matters
for troff(1) output.

=item B<-h>, B<--help>

Print out usage information.

=item B<-l>, B<--lax>

No longer used.  B<pod2man> used to check its input for validity as a
manual page, but this should now be done by L<podchecker(1)> instead.
Accepted for backward compatibility; this option no longer does anything.

=item B<--lquote>=I<quote>

=item B<--rquote>=I<quote>

Sets the quote marks used to surround CE<lt>> text.  B<--lquote> sets the
left quote mark and B<--rquote> sets the right quote mark.  Either may also
be set to the special value C<none>, in which case no quote mark is added
on that side of CE<lt>> text (but the font is still changed for troff
output).

Also see the B<--quotes> option, which can be used to set both quotes at once.
If both B<--quotes> and one of the other options is set, B<--lquote> or
B<--rquote> overrides B<--quotes>.

=item B<-n> I<name>, B<--name>=I<name>

Set the name of the manual page for the C<.TH> macro to I<name>.  Without
this option, the manual name is set to the uppercased base name of the
file being converted unless the manual section is 3, in which case the
path is parsed to see if it is a Perl module path.  If it is, a path like
C<.../lib/Pod/Man.pm> is converted into a name like C<Pod::Man>.  This
option, if given, overrides any automatic determination of the name.

Although one does not have to follow this convention, be aware that the
convention for UNIX man pages for commands is for the man page title to be
in all-uppercase, even if the command isn't.

This option is probably not useful when converting multiple POD files at
once.

When converting POD source from standard input, the name will be set to
C<STDIN> if this option is not provided.  Providing this option is strongly
recommended to set a meaningful manual page name.

=item B<--nourls>

Normally, LZ<><> formatting codes with a URL but anchor text are formatted
to show both the anchor text and the URL.  In other words:

    L<foo|http://example.com/>

is formatted as:

    foo <http://example.com/>

This flag, if given, suppresses the URL when anchor text is given, so this
example would be formatted as just C<foo>.  This can produce less
cluttered output in cases where the URLs are not particularly important.

=item B<-o>, B<--official>

Set the default header to indicate that this page is part of the standard
Perl release, if B<--center> is not also given.

=item B<-q> I<quotes>, B<--quotes>=I<quotes>

Sets the quote marks used to surround CE<lt>> text to I<quotes>.  If
I<quotes> is a single character, it is used as both the left and right
quote.  Otherwise, it is split in half, and the first half of the string
is used as the left quote and the second is used as the right quote.

I<quotes> may also be set to the special value C<none>, in which case no
quote marks are added around CE<lt>> text (but the font is still changed for
troff output).

Also see the B<--lquote> and B<--rquote> options, which can be used to set the
left and right quotes independently.  If both B<--quotes> and one of the other
options is set, B<--lquote> or B<--rquote> overrides B<--quotes>.

=item B<-r> I<version>, B<--release>=I<version>

Set the centered footer for the C<.TH> macro to I<version>.  By default,
this is set to the version of Perl you run B<pod2man> under.  Setting this
to the empty string will cause some *roff implementations to use the
system default value.

Note that some system C<an> macro sets assume that the centered footer
will be a modification date and will prepend something like "Last
modified: ".  If this is the case for your target system, you may want to
set B<--release> to the last modified date and B<--date> to the version
number.

=item B<-s> I<string>, B<--section>=I<string>

Set the section for the C<.TH> macro.  The standard section numbering
convention is to use 1 for user commands, 2 for system calls, 3 for
functions, 4 for devices, 5 for file formats, 6 for games, 7 for
miscellaneous information, and 8 for administrator commands.  There is a lot
of variation here, however; some systems (like Solaris) use 4 for file
formats, 5 for miscellaneous information, and 7 for devices.  Still others
use 1m instead of 8, or some mix of both.  About the only section numbers
that are reliably consistent are 1, 2, and 3.

By default, section 1 will be used unless the file ends in C<.pm>, in
which case section 3 will be selected.

=item B<--stderr>

By default, B<pod2man> dies if any errors are detected in the POD input.
If B<--stderr> is given and no B<--errors> flag is present, errors are
sent to standard error, but B<pod2man> does not abort.  This is equivalent
to C<--errors=stderr> and is supported for backward compatibility.

=item B<-u>, B<--utf8>

By default, B<pod2man> produces the most conservative possible *roff
output to try to ensure that it will work with as many different *roff
implementations as possible.  Many *roff implementations cannot handle
non-ASCII characters, so this means all non-ASCII characters are converted
either to a *roff escape sequence that tries to create a properly accented
character (at least for troff output) or to C<X>.

This option says to instead output literal UTF-8 characters.  If your
*roff implementation can handle it, this is the best output format to use
and avoids corruption of documents containing non-ASCII characters.
However, be warned that *roff source with literal UTF-8 characters is not
supported by many implementations and may even result in segfaults and
other bad behavior.

Be aware that, when using this option, the input encoding of your POD
source should be properly declared unless it's US-ASCII.  Pod::Simple will
attempt to guess the encoding and may be successful if it's Latin-1 or
UTF-8, but it will warn, which by default results in a B<pod2man> failure.
Use the C<=encoding> command to declare the encoding.  See L<perlpod(1)>
for more information.

=item B<-v>, B<--verbose>

Print out the name of each output file as it is being generated.

=back

=head1 EXIT STATUS

As long as all documents processed result in some output, even if that
output includes errata (a C<POD ERRORS> section generated with
C<--errors=pod>), B<pod2man> will exit with status 0.  If any of the
documents being processed do not result in an output document, B<pod2man>
will exit with status 1.  If there are syntax errors in a POD document
being processed and the error handling style is set to the default of
C<die>, B<pod2man> will abort immediately with exit status 255.

=head1 DIAGNOSTICS

If B<pod2man> fails with errors, see L<Pod::Man> and L<Pod::Simple> for
information about what those errors might mean.

=head1 EXAMPLES

    pod2man program > program.1
    pod2man SomeModule.pm /usr/perl/man/man3/SomeModule.3
    pod2man --section=7 note.pod > note.7

If you would like to print out a lot of man page continuously, you probably
want to set the C and D registers to set contiguous page numbering and
even/odd paging, at least on some versions of man(7).

    troff -man -rC1 -rD1 perl.1 perldata.1 perlsyn.1 ...

To get index entries on C<STDERR>, turn on the F register, as in:

    troff -man -rF1 perl.1

The indexing merely outputs messages via C<.tm> for each major page,
section, subsection, item, and any C<XE<lt>E<gt>> directives.  See
L<Pod::Man> for more details.

=head1 BUGS

Lots of this documentation is duplicated from L<Pod::Man>.

=head1 AUTHOR

Russ Allbery <rra@cpan.org>, based I<very> heavily on the original
B<pod2man> by Larry Wall and Tom Christiansen.

=head1 COPYRIGHT AND LICENSE

Copyright 1999-2001, 2004, 2006, 2008, 2010, 2012-2018 Russ Allbery
<rra@cpan.org>

This program is free software; you may redistribute it and/or modify it
under the same terms as Perl itself.

=head1 SEE ALSO

L<Pod::Man>, L<Pod::Simple>, L<man(1)>, L<nroff(1)>, L<perlpod(1)>,
L<podchecker(1)>, L<perlpodstyle(1)>, L<troff(1)>, L<man(7)>

The man page documenting the an macro set may be L<man(5)> instead of
L<man(7)> on your system.

The current version of this script is always available from its web site at
L<https://www.eyrie.org/~eagle/software/podlators/>.  It is also part of the
Perl core distribution as of 5.6.0.

=cut
SCRIPT_BODY

# Finish the generation of the script.
close($out) or die "Cannot close $file: $!\n";
chmod(0755, $file) or die "Cannot reset permissions for $file: $!\n";
if ($Config{'eunicefix'} ne q{:}) {
    exec("$Config{'eunicefix'} $file");
}

# Local Variables:
# copyright-at-end-flag: t
# End:
