#!/usr/bin/env perl
use strict;
use warnings;

my $have_pem = eval { require Convert::PEM; 1; };

use Test::More;
use Crypt::DSA::GMP;
use Crypt::DSA::GMP::Key;

plan tests => 1 + 6+4 + 6 + 6 + 6 + 6;

my $dsa = Crypt::DSA::GMP->new;
ok($dsa, 'Created Crypt::DSA::GMP object');

# OpenSSH created a DSA key.  Here is the pivate key in PEM format.
SKIP: {
skip "Requires Convert::PEM",6+4 unless $have_pem;
my $key = Crypt::DSA::GMP::Key->new( Content => <<KEY, Type => 'PEM' );
-----BEGIN DSA PRIVATE KEY-----
MIIBvAIBAAKBgQDXlwsLmDAzu5cVJMQfuoAFN28gjdNhvlIzHsYj9DseX/fD0Ypd
P6gK3r1l+eaDxIUIGT0v0dEJ61LekVAlqu0UtY/0QkAiDvhKeu1Fd7BnfiFesy5K
k7uX+cAbF7zL56eSja+eM5Sdq8q08wfAh9Ofd0KIXi1ao94wl8x2M2HOaQIVAK8T
De65jZ05SIt23ggWDqPWj21PAoGBAL4bb+HlKFGojBz8nUfsnSyE1VmhcwoIzGZU
ufadOmpshw1k37DzMWRQFi0x42icyxDauteAL8XvRElDxYuIQtOkxftNGEbYke6+
P6yjZ9fBOqVioybhwMXP0sQWlJWrqXx87nsKvg8hpcRZPj/C6a6GaJR5z24RLVco
wLufgMjLAoGBALTFbsytXttpkJvq3e6AG2oI5Tqrf9r2lYXKHMR4ndOdYczR4HI9
hLpyPMxyApW8ZQFwGZZoF5xHJoDj4LHfXJLv7wYShH4Z5nib/aDnRhWQtTqCMvcH
d0uNsNeELjX2Z+p5Z0W6rpizqYRsDN6/DPBNeSjfQak8Tag1iQ07VY5UAhRnRZMB
cd3TdX8wCgDClkApLNw58Q==
-----END DSA PRIVATE KEY-----
KEY
ok($key, 'Parsed key');
is($key->p, '151392455091639550568519801075590101869183672615001863144196762530501805142441752890430572923230712442074549439023083734271076440248189311471188085820147173239683091599263983590135598056034358243091320683170507827529486050037512482517500113992421938792051259600522238259711697949742364166658967180391089294953', 'key->p is correct');
is($key->q, '999498312746506920601077474194896035390476479823', 'key->q is correct');
is($key->g, '133497798857568394194099413301265987491200143881902609816263281844830829416427484982073801731443860145137071571135988119085176779469466812747615469603239392070908171059761619938393293605853643146560513938619782396764590771298856859695216459238860439491655492556229944283438349271306614564566919174540312496331', 'key->g is correct');
is($key->priv_key, '589577606811222969696891048290367221342146410993', 'key->priv_key is correct');
is($key->pub_key, '126941868992180834474872247767556072910294796867573697528152183018789017019052523055878800156524070756872623139034881299297000219345826603242784660359025808275183547724659911147326557393069249502117387490666914263066320078405501107994312663273275470063902656808999323475519180778784699834704373133727881334356', 'key->pub_key is correct');
sign_and_verify($dsa, $key);
}

# Here is the public key.
{
my $key = Crypt::DSA::GMP::Key->new( Content => <<KEY, Type => 'SSH2' );
ssh-dss AAAAB3NzaC1kc3MAAACBANeXCwuYMDO7lxUkxB+6gAU3byCN02G+UjMexiP0Ox5f98PRil0/qArevWX55oPEhQgZPS/R0QnrUt6RUCWq7RS1j/RCQCIO+Ep67UV3sGd+IV6zLkqTu5f5wBsXvMvnp5KNr54zlJ2ryrTzB8CH0593QoheLVqj3jCXzHYzYc5pAAAAFQCvEw3uuY2dOUiLdt4IFg6j1o9tTwAAAIEAvhtv4eUoUaiMHPydR+ydLITVWaFzCgjMZlS59p06amyHDWTfsPMxZFAWLTHjaJzLENq614Avxe9ESUPFi4hC06TF+00YRtiR7r4/rKNn18E6pWKjJuHAxc/SxBaUlaupfHzuewq+DyGlxFk+P8LproZolHnPbhEtVyjAu5+AyMsAAACBALTFbsytXttpkJvq3e6AG2oI5Tqrf9r2lYXKHMR4ndOdYczR4HI9hLpyPMxyApW8ZQFwGZZoF5xHJoDj4LHfXJLv7wYShH4Z5nib/aDnRhWQtTqCMvcHd0uNsNeELjX2Z+p5Z0W6rpizqYRsDN6/DPBNeSjfQak8Tag1iQ07VY5U someone\@somewhere.com
KEY
ok($key, 'Parsed key');
is($key->p, '151392455091639550568519801075590101869183672615001863144196762530501805142441752890430572923230712442074549439023083734271076440248189311471188085820147173239683091599263983590135598056034358243091320683170507827529486050037512482517500113992421938792051259600522238259711697949742364166658967180391089294953', 'key->p is correct');
is($key->q, '999498312746506920601077474194896035390476479823', 'key->q is correct');
is($key->g, '133497798857568394194099413301265987491200143881902609816263281844830829416427484982073801731443860145137071571135988119085176779469466812747615469603239392070908171059761619938393293605853643146560513938619782396764590771298856859695216459238860439491655492556229944283438349271306614564566919174540312496331', 'key->g is correct');
ok(! defined $key->priv_key, 'key->priv_key is undefined in public key');
is($key->pub_key, '126941868992180834474872247767556072910294796867573697528152183018789017019052523055878800156524070756872623139034881299297000219345826603242784660359025808275183547724659911147326557393069249502117387490666914263066320078405501107994312663273275470063902656808999323475519180778784699834704373133727881334356', 'key->pub_key is correct');
}

# Take public key in dss-ssh format and convert via:
#    ssh-keygen -e -m PKCS8 -f id_dsa.pub
SKIP: {
skip "Requires Convert::PEM",6 unless $have_pem;
my $key = Crypt::DSA::GMP::Key->new( Content => <<KEY, Type => 'PEM' );
-----BEGIN PUBLIC KEY-----
MIIBuDCCASwGByqGSM44BAEwggEfAoGBANeXCwuYMDO7lxUkxB+6gAU3byCN02G+
UjMexiP0Ox5f98PRil0/qArevWX55oPEhQgZPS/R0QnrUt6RUCWq7RS1j/RCQCIO
+Ep67UV3sGd+IV6zLkqTu5f5wBsXvMvnp5KNr54zlJ2ryrTzB8CH0593QoheLVqj
3jCXzHYzYc5pAhUArxMN7rmNnTlIi3beCBYOo9aPbU8CgYEAvhtv4eUoUaiMHPyd
R+ydLITVWaFzCgjMZlS59p06amyHDWTfsPMxZFAWLTHjaJzLENq614Avxe9ESUPF
i4hC06TF+00YRtiR7r4/rKNn18E6pWKjJuHAxc/SxBaUlaupfHzuewq+DyGlxFk+
P8LproZolHnPbhEtVyjAu5+AyMsDgYUAAoGBALTFbsytXttpkJvq3e6AG2oI5Tqr
f9r2lYXKHMR4ndOdYczR4HI9hLpyPMxyApW8ZQFwGZZoF5xHJoDj4LHfXJLv7wYS
hH4Z5nib/aDnRhWQtTqCMvcHd0uNsNeELjX2Z+p5Z0W6rpizqYRsDN6/DPBNeSjf
Qak8Tag1iQ07VY5U
-----END PUBLIC KEY-----
KEY
ok($key, 'Parsed key');
is($key->p, '151392455091639550568519801075590101869183672615001863144196762530501805142441752890430572923230712442074549439023083734271076440248189311471188085820147173239683091599263983590135598056034358243091320683170507827529486050037512482517500113992421938792051259600522238259711697949742364166658967180391089294953', 'key->p is correct');
is($key->q, '999498312746506920601077474194896035390476479823', 'key->q is correct');
is($key->g, '133497798857568394194099413301265987491200143881902609816263281844830829416427484982073801731443860145137071571135988119085176779469466812747615469603239392070908171059761619938393293605853643146560513938619782396764590771298856859695216459238860439491655492556229944283438349271306614564566919174540312496331', 'key->g is correct');
ok(! defined $key->priv_key, 'key->priv_key is undefined in public key');
is($key->pub_key, '126941868992180834474872247767556072910294796867573697528152183018789017019052523055878800156524070756872623139034881299297000219345826603242784660359025808275183547724659911147326557393069249502117387490666914263066320078405501107994312663273275470063902656808999323475519180778784699834704373133727881334356', 'key->pub_key is correct');
}

# Convert via:
#   ssh-keygen -e -m RFC4716 -f id_dsa
{
my $key = Crypt::DSA::GMP::Key->new( Content => <<KEY, Type => 'SSH2' );
---- BEGIN SSH2 PUBLIC KEY ----
Comment: "1024-bit DSA, converted by danaj from OpenSSH"
AAAAB3NzaC1kc3MAAACBANeXCwuYMDO7lxUkxB+6gAU3byCN02G+UjMexiP0Ox5f98PRil
0/qArevWX55oPEhQgZPS/R0QnrUt6RUCWq7RS1j/RCQCIO+Ep67UV3sGd+IV6zLkqTu5f5
wBsXvMvnp5KNr54zlJ2ryrTzB8CH0593QoheLVqj3jCXzHYzYc5pAAAAFQCvEw3uuY2dOU
iLdt4IFg6j1o9tTwAAAIEAvhtv4eUoUaiMHPydR+ydLITVWaFzCgjMZlS59p06amyHDWTf
sPMxZFAWLTHjaJzLENq614Avxe9ESUPFi4hC06TF+00YRtiR7r4/rKNn18E6pWKjJuHAxc
/SxBaUlaupfHzuewq+DyGlxFk+P8LproZolHnPbhEtVyjAu5+AyMsAAACBALTFbsytXttp
kJvq3e6AG2oI5Tqrf9r2lYXKHMR4ndOdYczR4HI9hLpyPMxyApW8ZQFwGZZoF5xHJoDj4L
HfXJLv7wYShH4Z5nib/aDnRhWQtTqCMvcHd0uNsNeELjX2Z+p5Z0W6rpizqYRsDN6/DPBN
eSjfQak8Tag1iQ07VY5U
---- END SSH2 PUBLIC KEY ----
KEY
ok($key, 'Parsed key');
is($key->p, '151392455091639550568519801075590101869183672615001863144196762530501805142441752890430572923230712442074549439023083734271076440248189311471188085820147173239683091599263983590135598056034358243091320683170507827529486050037512482517500113992421938792051259600522238259711697949742364166658967180391089294953', 'key->p is correct');
is($key->q, '999498312746506920601077474194896035390476479823', 'key->q is correct');
is($key->g, '133497798857568394194099413301265987491200143881902609816263281844830829416427484982073801731443860145137071571135988119085176779469466812747615469603239392070908171059761619938393293605853643146560513938619782396764590771298856859695216459238860439491655492556229944283438349271306614564566919174540312496331', 'key->g is correct');
ok(! defined $key->priv_key, 'key->priv_key is undefined in public key');
is($key->pub_key, '126941868992180834474872247767556072910294796867573697528152183018789017019052523055878800156524070756872623139034881299297000219345826603242784660359025808275183547724659911147326557393069249502117387490666914263066320078405501107994312663273275470063902656808999323475519180778784699834704373133727881334356', 'key->pub_key is correct');
}

# From RFC 4716
{
my $key = Crypt::DSA::GMP::Key->new( Content => <<KEY, Type => 'SSH2' );
---- BEGIN SSH2 PUBLIC KEY ----
Comment: DSA Public Key for use with MyIsp
AAAAB3NzaC1kc3MAAACBAPY8ZOHY2yFSJA6XYC9HRwNHxaehvx5wOJ0rzZdzoSOXxbET
W6ToHv8D1UJ/z+zHo9Fiko5XybZnDIaBDHtblQ+Yp7StxyltHnXF1YLfKD1G4T6JYrdH
YI14Om1eg9e4NnCRleaqoZPF3UGfZia6bXrGTQf3gJq2e7Yisk/gF+1VAAAAFQDb8D5c
vwHWTZDPfX0D2s9Rd7NBvQAAAIEAlN92+Bb7D4KLYk3IwRbXblwXdkPggA4pfdtW9vGf
J0/RHd+NjB4eo1D+0dix6tXwYGN7PKS5R/FXPNwxHPapcj9uL1Jn2AWQ2dsknf+i/FAA
vioUPkmdMc0zuWoSOEsSNhVDtX3WdvVcGcBq9cetzrtOKWOocJmJ80qadxTRHtUAAACB
AN7CY+KKv1gHpRzFwdQm7HK9bb1LAo2KwaoXnadFgeptNBQeSXG1vO+JsvphVMBJc9HS
n24VYtYtsMu74qXviYjziVucWKjjKEb11juqnF0GDlB3VVmxHLmxnAz643WK42Z7dLM5
sY29ouezv4Xz2PuMch5VGPP+CDqzCM4loWgV
---- END SSH2 PUBLIC KEY ----
KEY
ok($key, 'Parsed key');
is($key->p, '172912739361858571093957685000871580557539145461488522157891963468542763919020113500844767617702521044388103869066028745497212242974201912472716856378733696030359824674747515672873581883380498813002364359956904882913735488304338320882737669402322515183802836774249104467384545037621295598395399358763696123221', 'key->p is correct');
is($key->q, '1255626590179665817295475654204371833735706001853', 'key->q is correct');
is($key->g, '104542111957051574443507210365575422382386284629635861863168872240203216074384359832907098765397916841041232274543207973430275092618267562344461747950207453700708157354355640359052702086950475523082766062911804129120196699631832685907397565903508001966306315006141446279152203060511039590076780999046480338645', 'key->g is correct');
ok(! defined $key->priv_key, 'key->priv_key is undefined in public key');
is($key->pub_key, '156426925849132582888923515013715685031112342417997332032001377184507622857046183855217375911279333442426508390421925803571145682125712749016671491849778185115184920702953440227844996385208790152029506883757564191763243611665708689363768384610087250847373692963122547164980532522704439627864339929607694280725', 'key->pub_key is correct');
}




sub sign_and_verify {
  my($dsa, $key) = @_;
  my $message = "This prime validation algorithm is used to validate that the integers p and q were generated by the prime generation algorithm given in Appendix A.1.1.2.  The validation algorithm requires the values of p, q, domain_parameter_seed and counter, which were output from the prime generation algorithm.  Let Hash() be the hash function used to generate p and q, and let outlen be its output block length.";
  {
    my $sig = $dsa->sign(
        Message => $message,
        Key => $key,
    );
    ok($sig, 'generated default signature');
    my $verified = $dsa->verify(
        Key       => $key,
        Message   => $message,
        Signature => $sig,
    );
    ok($verified, 'verified default signature');
  }
  {
    my $sig = $dsa->sign(
        Message => $message,
        Key => $key,
        Standard => "FIPS 186-4",
    );
    ok($sig, 'generated FIPS 186-4 signature');
    my $verified = $dsa->verify(
        Key       => $key,
        Message   => $message,
        Signature => $sig,
        Standard => "FIPS 186-4",
    );
    ok($verified, 'verified FIPS 186-4 signature');
  }
  1;
}
