﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/eventbridge/EventBridgeRequest.h>
#include <aws/eventbridge/EventBridge_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EventBridge {
namespace Model {

/**
 */
class CreatePartnerEventSourceRequest : public EventBridgeRequest {
 public:
  AWS_EVENTBRIDGE_API CreatePartnerEventSourceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreatePartnerEventSource"; }

  AWS_EVENTBRIDGE_API Aws::String SerializePayload() const override;

  AWS_EVENTBRIDGE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the partner event source. This name must be unique and must be in
   * the format <code> <i>partner_name</i>/<i>event_namespace</i>/<i>event_name</i>
   * </code>. The Amazon Web Services account that wants to use this partner event
   * source must create a partner event bus with a name that matches the name of the
   * partner event source.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreatePartnerEventSourceRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services account ID that is permitted to create a matching
   * partner event bus for this partner event source.</p>
   */
  inline const Aws::String& GetAccount() const { return m_account; }
  inline bool AccountHasBeenSet() const { return m_accountHasBeenSet; }
  template <typename AccountT = Aws::String>
  void SetAccount(AccountT&& value) {
    m_accountHasBeenSet = true;
    m_account = std::forward<AccountT>(value);
  }
  template <typename AccountT = Aws::String>
  CreatePartnerEventSourceRequest& WithAccount(AccountT&& value) {
    SetAccount(std::forward<AccountT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_account;
  bool m_nameHasBeenSet = false;
  bool m_accountHasBeenSet = false;
};

}  // namespace Model
}  // namespace EventBridge
}  // namespace Aws
