/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.notebook;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RawLocalFileSystem;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemStorage {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileSystemStorage.class);
    private static final String S3A = "s3a";
    private static final String FS_DEFAULTFS = "fs.defaultFS";
    private ZeppelinConfiguration zConf;
    private Configuration hadoopConf;
    private boolean isSecurityEnabled;
    private FileSystem fs;

    public FileSystemStorage(ZeppelinConfiguration zConf, String path) throws IOException {
        URI defaultFSURI;
        URI zepConfigURI;
        this.zConf = zConf;
        this.hadoopConf = new Configuration();
        try {
            zepConfigURI = new URI(path);
        }
        catch (URISyntaxException e) {
            LOGGER.error("Failed to get Zeppelin config URI");
            throw new IOException(e);
        }
        this.hadoopConf.setBoolean("fs.file.impl.disable.cache", true);
        String defaultFS = this.hadoopConf.get(FS_DEFAULTFS);
        try {
            defaultFSURI = new URI(defaultFS);
        }
        catch (URISyntaxException e) {
            LOGGER.error("Failed to get defaultFS URI");
            throw new IOException(e);
        }
        if (!this.isS3AFileSystem(defaultFSURI, zepConfigURI)) {
            this.hadoopConf.set("fs.file.impl", RawLocalFileSystem.class.getName());
        }
        this.isSecurityEnabled = UserGroupInformation.isSecurityEnabled();
        this.fs = FileSystem.get((URI)zepConfigURI, (Configuration)this.hadoopConf);
    }

    public boolean isS3AFileSystem(URI defaultFSURI, URI zepConfigURI) {
        return defaultFSURI.getScheme().equals(S3A) || StringUtils.isNotEmpty((CharSequence)zepConfigURI.getScheme()) && zepConfigURI.getScheme().equals(S3A);
    }

    public FileSystem getFs() {
        return this.fs;
    }

    public Path makeQualified(Path path) {
        return this.fs.makeQualified(path);
    }

    public boolean exists(final Path path) throws IOException {
        return this.callHdfsOperation(new HdfsOperation<Boolean>(){

            @Override
            public Boolean call() throws IOException {
                return FileSystemStorage.this.fs.exists(path);
            }
        });
    }

    public void tryMkDir(final Path dir) throws IOException {
        this.callHdfsOperation(new HdfsOperation<Void>(){

            @Override
            public Void call() throws IOException {
                if (!FileSystemStorage.this.fs.exists(dir)) {
                    FileSystemStorage.this.fs.mkdirs(dir);
                    LOGGER.info("Create dir {} in hdfs", (Object)dir);
                }
                if (FileSystemStorage.this.fs.getFileStatus(dir).isFile()) {
                    throw new IOException(dir.toString() + " is file instead of directory, please remove it or specify another directory");
                }
                FileSystemStorage.this.fs.mkdirs(dir);
                return null;
            }
        });
    }

    public List<Path> list(final Path path) throws IOException {
        return this.callHdfsOperation(new HdfsOperation<List<Path>>(){

            @Override
            public List<Path> call() throws IOException {
                ArrayList<Path> paths = new ArrayList<Path>();
                for (FileStatus status : FileSystemStorage.this.fs.globStatus(path)) {
                    paths.add(status.getPath());
                }
                return paths;
            }
        });
    }

    public List<Path> listAll(final Path path) throws IOException {
        return this.callHdfsOperation(new HdfsOperation<List<Path>>(){

            @Override
            public List<Path> call() throws IOException {
                ArrayList<Path> paths = new ArrayList<Path>();
                this.collectNoteFiles(path, paths);
                return paths;
            }

            private void collectNoteFiles(Path folder, List<Path> noteFiles) throws IOException {
                FileStatus[] paths;
                for (FileStatus path2 : paths = FileSystemStorage.this.fs.listStatus(folder)) {
                    if (path2.isDirectory()) {
                        this.collectNoteFiles(path2.getPath(), noteFiles);
                        continue;
                    }
                    if (path2.getPath().getName().endsWith(".zpln")) {
                        noteFiles.add(path2.getPath());
                        continue;
                    }
                    LOGGER.warn("Unknown file: {}", (Object)path2.getPath());
                }
            }
        });
    }

    public boolean delete(final Path path) throws IOException {
        return this.callHdfsOperation(new HdfsOperation<Boolean>(){

            @Override
            public Boolean call() throws IOException {
                return FileSystemStorage.this.fs.delete(path, true);
            }
        });
    }

    public String readFile(final Path file) throws IOException {
        return this.callHdfsOperation(new HdfsOperation<String>(){

            @Override
            public String call() throws IOException {
                LOGGER.debug("Read from file: {}", (Object)file);
                ByteArrayOutputStream noteBytes = new ByteArrayOutputStream();
                IOUtils.copyBytes((InputStream)FileSystemStorage.this.fs.open(file), (OutputStream)noteBytes, (Configuration)FileSystemStorage.this.hadoopConf);
                return noteBytes.toString(FileSystemStorage.this.zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_ENCODING));
            }
        });
    }

    public void writeFile(String content, Path file, boolean writeTempFileFirst) throws IOException {
        this.writeFile(content, file, writeTempFileFirst, null);
    }

    public void writeFile(final String content, final Path file, boolean writeTempFileFirst, Set<PosixFilePermission> permissions) throws IOException {
        final FsPermission fsPermission = permissions == null || permissions.isEmpty() ? FsPermission.getFileDefault() : FsPermission.valueOf((String)("-" + PosixFilePermissions.toString(permissions)));
        this.callHdfsOperation(new HdfsOperation<Void>(){

            @Override
            public Void call() throws IOException {
                ByteArrayInputStream in = new ByteArrayInputStream(content.getBytes(FileSystemStorage.this.zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_ENCODING)));
                Path tmpFile = new Path(file.toString() + ".tmp");
                IOUtils.copyBytes((InputStream)in, (OutputStream)FileSystemStorage.this.fs.create(tmpFile), (Configuration)FileSystemStorage.this.hadoopConf);
                FileSystemStorage.this.fs.setPermission(tmpFile, fsPermission);
                FileSystemStorage.this.fs.delete(file, true);
                FileSystemStorage.this.fs.rename(tmpFile, file);
                return null;
            }
        });
    }

    public void move(Path src, Path dest) throws IOException {
        this.callHdfsOperation(() -> {
            this.fs.rename(src, dest);
            return null;
        });
    }

    public synchronized <T> T callHdfsOperation(final HdfsOperation<T> func) throws IOException {
        if (this.isSecurityEnabled) {
            try {
                return (T)UserGroupInformation.getCurrentUser().doAs(new PrivilegedExceptionAction<T>(){

                    @Override
                    public T run() throws Exception {
                        return func.call();
                    }
                });
            }
            catch (InterruptedException e) {
                throw new IOException(e);
            }
        }
        return func.call();
    }

    static {
        if (UserGroupInformation.isSecurityEnabled()) {
            ZeppelinConfiguration zConf = ZeppelinConfiguration.create();
            String keytab = zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_SERVER_KERBEROS_KEYTAB);
            String principal = zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_SERVER_KERBEROS_PRINCIPAL);
            if (StringUtils.isBlank((CharSequence)keytab) || StringUtils.isBlank((CharSequence)principal)) {
                throw new RuntimeException("keytab and principal can not be empty, keytab: " + keytab + ", principal: " + principal);
            }
            try {
                UserGroupInformation.loginUserFromKeytab((String)principal, (String)keytab);
            }
            catch (IOException e) {
                throw new RuntimeException("Fail to login via keytab:" + keytab + ", principal:" + principal, e);
            }
        }
    }

    private static interface HdfsOperation<T> {
        public T call() throws IOException;
    }
}

