/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.search;

import com.google.common.collect.ImmutableMap;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.LongPoint;
import org.apache.lucene.document.StringField;
import org.apache.lucene.document.TextField;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.IndexableField;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryparser.classic.MultiFieldQueryParser;
import org.apache.lucene.queryparser.classic.ParseException;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.WildcardQuery;
import org.apache.lucene.search.highlight.Formatter;
import org.apache.lucene.search.highlight.Highlighter;
import org.apache.lucene.search.highlight.InvalidTokenOffsetsException;
import org.apache.lucene.search.highlight.QueryScorer;
import org.apache.lucene.search.highlight.Scorer;
import org.apache.lucene.search.highlight.SimpleHTMLFormatter;
import org.apache.lucene.search.highlight.TextFragment;
import org.apache.lucene.search.highlight.TokenSources;
import org.apache.lucene.store.ByteBuffersDirectory;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.notebook.Note;
import org.apache.zeppelin.notebook.Notebook;
import org.apache.zeppelin.notebook.Paragraph;
import org.apache.zeppelin.search.SearchService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LuceneSearch
extends SearchService {
    private static final Logger LOGGER = LoggerFactory.getLogger(LuceneSearch.class);
    private static final String SEARCH_FIELD_TEXT = "contents";
    private static final String SEARCH_FIELD_TITLE = "header";
    private static final String PARAGRAPH = "paragraph";
    private static final String ID_FIELD = "id";
    private final Directory indexDirectory;
    private final IndexWriter indexWriter;
    private final Notebook notebook;

    @Inject
    public LuceneSearch(ZeppelinConfiguration conf, Notebook notebook) throws IOException {
        super("LuceneSearch");
        this.notebook = notebook;
        if (conf.isZeppelinSearchUseDisk()) {
            try {
                Path indexPath = Paths.get(conf.getZeppelinSearchIndexPath(), new String[0]);
                this.indexDirectory = FSDirectory.open((Path)indexPath);
                LOGGER.info("Use {} for storing lucene search index", (Object)indexPath);
            }
            catch (IOException e) {
                throw new IOException("Failed to create index directory for search service.", e);
            }
        } else {
            this.indexDirectory = new ByteBuffersDirectory();
        }
        StandardAnalyzer analyzer = new StandardAnalyzer();
        IndexWriterConfig indexWriterConfig = new IndexWriterConfig((Analyzer)analyzer);
        try {
            this.indexWriter = new IndexWriter(this.indexDirectory, indexWriterConfig);
        }
        catch (IOException e) {
            throw new IOException("Failed to create new IndexWriter", e);
        }
        if (conf.isIndexRebuild()) {
            notebook.addInitConsumer(this::addNoteIndex);
        }
        this.notebook.addNotebookEventListener(this);
    }

    @Override
    public List<Map<String, String>> query(String queryStr) {
        if (null == this.indexDirectory) {
            throw new IllegalStateException("Something went wrong on instance creation time, index dir is null");
        }
        List<Map<String, String>> result = Collections.emptyList();
        try (DirectoryReader indexReader = DirectoryReader.open((Directory)this.indexDirectory);){
            IndexSearcher indexSearcher = new IndexSearcher((IndexReader)indexReader);
            StandardAnalyzer analyzer = new StandardAnalyzer();
            MultiFieldQueryParser parser = new MultiFieldQueryParser(new String[]{SEARCH_FIELD_TEXT, SEARCH_FIELD_TITLE}, (Analyzer)analyzer);
            Query query = parser.parse(queryStr);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Searching for: {}", (Object)query.toString(SEARCH_FIELD_TEXT));
            }
            SimpleHTMLFormatter htmlFormatter = new SimpleHTMLFormatter();
            Highlighter highlighter = new Highlighter((Formatter)htmlFormatter, (Scorer)new QueryScorer(query));
            result = this.doSearch(indexSearcher, query, (Analyzer)analyzer, highlighter);
        }
        catch (IOException e) {
            LOGGER.error("Failed to open index dir {}, make sure indexing finished OK", (Object)this.indexDirectory, (Object)e);
        }
        catch (ParseException e) {
            LOGGER.error("Failed to parse query {}", (Object)queryStr, (Object)e);
        }
        return result;
    }

    private List<Map<String, String>> doSearch(IndexSearcher searcher, Query query, Analyzer analyzer, Highlighter highlighter) {
        ArrayList<Map<String, String>> matchingParagraphs = new ArrayList<Map<String, String>>();
        try {
            ScoreDoc[] hits = searcher.search((Query)query, (int)20).scoreDocs;
            for (int i = 0; i < hits.length; ++i) {
                LOGGER.debug("doc={} score={}", (Object)hits[i].doc, (Object)Float.valueOf(hits[i].score));
                int id = hits[i].doc;
                Document doc = searcher.doc(id);
                String path = doc.get(ID_FIELD);
                if (path != null) {
                    TokenStream tokenTitle;
                    TextFragment[] frgTitle;
                    LOGGER.debug("{}. {}", (Object)(i + 1), (Object)path);
                    String title = doc.get("title");
                    if (title != null) {
                        LOGGER.debug("   Title: {}", (Object)doc.get("title"));
                    }
                    String text = doc.get(SEARCH_FIELD_TEXT);
                    String header = doc.get(SEARCH_FIELD_TITLE);
                    String fragment = "";
                    if (text != null) {
                        TokenStream tokenStream = TokenSources.getTokenStream((IndexReader)searcher.getIndexReader(), (int)id, (String)SEARCH_FIELD_TEXT, (Analyzer)analyzer);
                        TextFragment[] frags = highlighter.getBestTextFragments(tokenStream, text, true, 3);
                        LOGGER.debug("    {} fragments found for query '{}'", (Object)frags.length, (Object)query);
                        for (TextFragment frag : frags) {
                            if (frag == null || !(frag.getScore() > 0.0f)) continue;
                            LOGGER.debug("    Fragment: {}", (Object)frag);
                        }
                        String string = fragment = frags != null && frags.length > 0 ? frags[0].toString() : "";
                    }
                    header = header != null ? ((frgTitle = highlighter.getBestTextFragments(tokenTitle = TokenSources.getTokenStream((IndexReader)searcher.getIndexReader(), (int)id, (String)SEARCH_FIELD_TITLE, (Analyzer)analyzer), header, true, 3)) != null && frgTitle.length > 0 ? frgTitle[0].toString() : "") : "";
                    matchingParagraphs.add((Map<String, String>)ImmutableMap.of((Object)ID_FIELD, (Object)path, (Object)"name", (Object)title, (Object)"snippet", (Object)fragment, (Object)"text", (Object)text, (Object)SEARCH_FIELD_TITLE, (Object)header));
                    continue;
                }
                LOGGER.info("{}. No {} for this document", (Object)(i + 1), (Object)ID_FIELD);
            }
        }
        catch (IOException | InvalidTokenOffsetsException e) {
            LOGGER.error("Exception on searching for {}", (Object)query, (Object)e);
        }
        return matchingParagraphs;
    }

    @Override
    public void updateNoteIndex(String noteId) {
        this.updateIndexNoteName(noteId);
    }

    private void updateIndexNoteName(String noteId) {
        try {
            this.notebook.processNote(noteId, note -> {
                if (note != null) {
                    String noteName = note.getName();
                    LOGGER.debug("Update note index: {}, '{}'", (Object)noteId, (Object)noteName);
                    if (null == noteName || noteName.isEmpty()) {
                        LOGGER.debug("Skipping empty notebook name");
                        return null;
                    }
                    this.updateDoc(noteId, noteName, null);
                }
                return null;
            });
        }
        catch (IOException e) {
            LOGGER.error("Unable to update note {}", (Object)noteId, (Object)e);
        }
    }

    @Override
    public void updateParagraphIndex(String noteId, String paragraphId) {
        try {
            this.notebook.processNote(noteId, note -> {
                if (note != null) {
                    Paragraph p = note.getParagraph(paragraphId);
                    LOGGER.debug("Update paragraph index: {}", (Object)paragraphId);
                    this.updateDoc(noteId, note.getName(), p);
                }
                return null;
            });
        }
        catch (IOException e) {
            LOGGER.error("Unable to update paragraph {} of note {}", new Object[]{paragraphId, noteId, e});
        }
    }

    private void updateDoc(String noteId, String noteName, Paragraph p) throws IOException {
        String id = LuceneSearch.formatId(noteId, p);
        Document doc = this.newDocument(id, noteName, p);
        try {
            this.indexWriter.updateDocument(new Term(ID_FIELD, id), (Iterable)doc);
            this.indexWriter.commit();
        }
        catch (IOException e) {
            throw new IOException("Failed to update index of notebook " + noteId, e);
        }
    }

    static String formatId(String noteId, Paragraph p) {
        String id = noteId;
        if (null != p) {
            id = String.join((CharSequence)"/", id, PARAGRAPH, p.getId());
        }
        return id;
    }

    static String formatDeleteId(String noteId, String paragraphId) {
        String id = noteId;
        id = null != paragraphId ? String.join((CharSequence)"/", id, PARAGRAPH, paragraphId) : id + "*";
        return id;
    }

    private Document newDocument(String id, String noteName, Paragraph p) {
        Document doc = new Document();
        StringField pathField = new StringField(ID_FIELD, id, Field.Store.YES);
        doc.add((IndexableField)pathField);
        doc.add((IndexableField)new StringField("title", noteName, Field.Store.YES));
        if (null != p) {
            if (p.getText() != null) {
                doc.add((IndexableField)new TextField(SEARCH_FIELD_TEXT, p.getText(), Field.Store.YES));
            }
            if (p.getTitle() != null) {
                doc.add((IndexableField)new TextField(SEARCH_FIELD_TITLE, p.getTitle(), Field.Store.YES));
            }
            Date date = p.getDateStarted() != null ? p.getDateStarted() : p.getDateCreated();
            doc.add((IndexableField)new LongPoint("modified", new long[]{date.getTime()}));
        } else {
            doc.add((IndexableField)new TextField(SEARCH_FIELD_TEXT, noteName, Field.Store.YES));
        }
        return doc;
    }

    @Override
    public void addNoteIndex(String noteId) {
        try {
            this.notebook.processNote(noteId, note -> {
                if (note != null) {
                    this.addIndexDocAsync(note);
                }
                return null;
            });
            this.indexWriter.commit();
        }
        catch (IOException e) {
            LOGGER.error("Failed to add note {} to index", (Object)noteId, (Object)e);
        }
    }

    @Override
    public void addParagraphIndex(String noteId, String paragraphId) {
        try {
            this.notebook.processNote(noteId, note -> {
                if (note != null) {
                    Paragraph p = note.getParagraph(paragraphId);
                    this.updateDoc(noteId, note.getName(), p);
                }
                return null;
            });
        }
        catch (IOException e) {
            LOGGER.error("Failed to add paragraph {} of note {} to index", new Object[]{paragraphId, noteId, e});
        }
    }

    private void addIndexDocAsync(Note note) throws IOException {
        this.indexNoteName(note.getId(), note.getName());
        for (Paragraph paragraph : note.getParagraphs()) {
            this.updateDoc(note.getId(), note.getName(), paragraph);
        }
    }

    @Override
    public void deleteNoteIndex(String noteId) {
        try {
            this.deleteDoc(noteId, null);
            this.deleteParagraphIndex(noteId, null);
        }
        catch (IOException e) {
            LOGGER.error("Unable to delete note {}", (Object)noteId, (Object)e);
        }
    }

    @Override
    public void deleteParagraphIndex(String noteId, String paragraphId) {
        try {
            this.deleteDoc(noteId, paragraphId);
        }
        catch (IOException e) {
            LOGGER.error("Unable to delete paragraph {} of note {}", new Object[]{paragraphId, noteId, e});
        }
    }

    private void deleteDoc(String noteId, String paragraphId) throws IOException {
        String fullNoteOrJustParagraph = LuceneSearch.formatDeleteId(noteId, paragraphId);
        LOGGER.debug("Deleting note {}, out of: {}", (Object)noteId, (Object)this.indexWriter.getDocStats().numDocs);
        try {
            this.indexWriter.deleteDocuments(new Query[]{new WildcardQuery(new Term(ID_FIELD, fullNoteOrJustParagraph))});
            this.indexWriter.commit();
        }
        catch (IOException e) {
            throw new IOException("Failed to delete " + noteId + " from index by '" + fullNoteOrJustParagraph + "'", e);
        }
        LOGGER.debug("Done, index contains {} docs now", (Object)this.indexWriter.getDocStats().numDocs);
    }

    @Override
    @PreDestroy
    public void close() {
        super.close();
        try {
            this.indexWriter.close();
        }
        catch (IOException e) {
            LOGGER.error("Failed to close the notebook index", (Throwable)e);
        }
    }

    private void indexNoteName(String noteId, String noteName) throws IOException {
        LOGGER.debug("Indexing Notebook {}, '{}'", (Object)noteId, (Object)noteName);
        if (StringUtils.isBlank((CharSequence)noteName)) {
            LOGGER.debug("Skipping empty or blank notebook name");
            return;
        }
        this.updateDoc(noteId, noteName, null);
    }
}

