/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.security;

import java.io.ByteArrayInputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.hadoop.hdds.annotation.InterfaceAudience;
import org.apache.hadoop.hdds.annotation.InterfaceStability;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.security.OzoneSecretManager;
import org.apache.hadoop.hdds.security.SecurityConfig;
import org.apache.hadoop.hdds.security.x509.certificate.client.CertificateClient;
import org.apache.hadoop.hdds.security.x509.exception.CertificateException;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.ozone.om.OzoneManager;
import org.apache.hadoop.ozone.om.S3SecretManager;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.exceptions.OMLeaderNotReadyException;
import org.apache.hadoop.ozone.om.exceptions.OMNotLeaderException;
import org.apache.hadoop.ozone.protocol.proto.OzoneManagerProtocolProtos;
import org.apache.hadoop.ozone.security.AWSV4AuthValidator;
import org.apache.hadoop.ozone.security.OzoneSecretStore;
import org.apache.hadoop.ozone.security.OzoneTokenIdentifier;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.HadoopKerberosName;
import org.apache.hadoop.security.token.SecretManager;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class OzoneDelegationTokenSecretManager
extends OzoneSecretManager<OzoneTokenIdentifier> {
    private static final Logger LOG = LoggerFactory.getLogger(OzoneDelegationTokenSecretManager.class);
    private final Map<OzoneTokenIdentifier, OzoneTokenIdentifier.TokenInfo> currentTokens;
    private final OzoneSecretStore store;
    private final S3SecretManager s3SecretManager;
    private Thread tokenRemoverThread;
    private final long tokenRemoverScanInterval;
    private final String omServiceId;
    private final OzoneManager ozoneManager;
    private final Object noInterruptsLock = new Object();
    private final boolean isRatisEnabled;

    public OzoneDelegationTokenSecretManager(Builder b) throws IOException {
        super(new SecurityConfig((ConfigurationSource)b.ozoneConf), b.tokenMaxLifetime, b.tokenRenewInterval, b.service, LOG);
        this.setCertClient(b.certClient);
        this.omServiceId = b.omServiceId;
        this.currentTokens = new ConcurrentHashMap<OzoneTokenIdentifier, OzoneTokenIdentifier.TokenInfo>();
        this.tokenRemoverScanInterval = b.tokenRemoverScanInterval;
        this.s3SecretManager = b.s3SecretManager;
        this.ozoneManager = b.ozoneManager;
        this.store = new OzoneSecretStore(b.ozoneConf, this.ozoneManager.getMetadataManager());
        this.isRatisEnabled = b.ozoneConf.getBoolean("ozone.om.ratis.enable", true);
        this.loadTokenSecretState(this.store.loadState());
    }

    public OzoneTokenIdentifier createIdentifier() {
        return OzoneTokenIdentifier.newInstance();
    }

    public OzoneTokenIdentifier createIdentifier(Text owner, Text renewer, Text realUser) {
        return OzoneTokenIdentifier.newInstance((Text)owner, (Text)renewer, (Text)realUser);
    }

    public Token<OzoneTokenIdentifier> createToken(Text owner, Text renewer, Text realUser) throws IOException {
        OzoneTokenIdentifier identifier = this.createIdentifier(owner, renewer, realUser);
        this.updateIdentifierDetails(identifier);
        byte[] password = this.createPassword(identifier.getBytes(), this.getCurrentKey().getPrivateKey());
        long expiryTime = identifier.getIssueDate() + this.getTokenRenewInterval();
        if (!this.isRatisEnabled) {
            this.addToTokenStore(identifier, password, expiryTime);
        }
        Token token = new Token(identifier.getBytes(), password, identifier.getKind(), this.getService());
        if (LOG.isDebugEnabled()) {
            LOG.debug("Created delegation token: {}", (Object)token);
        }
        return token;
    }

    public long updateToken(Token<OzoneTokenIdentifier> token, OzoneTokenIdentifier ozoneTokenIdentifier, long tokenRenewInterval) {
        long renewTime = ozoneTokenIdentifier.getIssueDate() + tokenRenewInterval;
        OzoneTokenIdentifier.TokenInfo tokenInfo = new OzoneTokenIdentifier.TokenInfo(renewTime, token.getPassword(), ozoneTokenIdentifier.getTrackingId());
        this.currentTokens.put(ozoneTokenIdentifier, tokenInfo);
        return renewTime;
    }

    private void addToTokenStore(OzoneTokenIdentifier identifier, byte[] password, long renewTime) throws IOException {
        OzoneTokenIdentifier.TokenInfo tokenInfo = new OzoneTokenIdentifier.TokenInfo(renewTime, password, identifier.getTrackingId());
        this.currentTokens.put(identifier, tokenInfo);
        this.store.storeToken(identifier, tokenInfo.getRenewDate());
    }

    private void updateIdentifierDetails(OzoneTokenIdentifier identifier) {
        long now = Time.now();
        int sequenceNum = this.incrementDelegationTokenSeqNum();
        identifier.setIssueDate(now);
        identifier.setMasterKeyId(this.getCurrentKey().getKeyId());
        identifier.setSequenceNumber(sequenceNum);
        identifier.setMaxDate(now + this.getTokenMaxLifetime());
        identifier.setOmCertSerialId(this.getCertSerialId());
        identifier.setOmServiceId(this.getOmServiceId());
    }

    private String getOmServiceId() {
        return this.omServiceId;
    }

    public synchronized long renewToken(Token<OzoneTokenIdentifier> token, String renewer) throws IOException {
        ByteArrayInputStream buf = new ByteArrayInputStream(token.getIdentifier());
        DataInputStream in = new DataInputStream(buf);
        OzoneTokenIdentifier id = OzoneTokenIdentifier.readProtoBuf((DataInput)in);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Token renewal for identifier: {}, total currentTokens: {}", (Object)this.formatTokenId((TokenIdentifier)id), (Object)this.currentTokens.size());
        }
        long now = Time.now();
        if (id.getMaxDate() < now) {
            throw new OMException(String.valueOf(renewer) + " tried to renew an expired token " + this.formatTokenId((TokenIdentifier)id) + " max expiration date: " + Time.formatTime((long)id.getMaxDate()) + " currentTime: " + Time.formatTime((long)now), OMException.ResultCodes.TOKEN_EXPIRED);
        }
        this.validateToken(id);
        if (id.getRenewer() == null || id.getRenewer().toString().isEmpty()) {
            throw new AccessControlException(String.valueOf(renewer) + " tried to renew a token " + this.formatTokenId((TokenIdentifier)id) + " without a renewer");
        }
        if (!id.getRenewer().toString().equals(renewer)) {
            throw new AccessControlException(String.valueOf(renewer) + " tries to renew a token " + this.formatTokenId((TokenIdentifier)id) + " with non-matching renewer " + id.getRenewer());
        }
        long renewTime = Math.min(id.getMaxDate(), now + this.getTokenRenewInterval());
        if (!this.isRatisEnabled) {
            try {
                this.addToTokenStore(id, token.getPassword(), renewTime);
            }
            catch (IOException e) {
                LOG.error("Unable to update token " + id.getSequenceNumber(), (Throwable)e);
            }
        }
        return renewTime;
    }

    public void updateRenewToken(Token<OzoneTokenIdentifier> token, OzoneTokenIdentifier ozoneTokenIdentifier, long expiryTime) {
        OzoneTokenIdentifier.TokenInfo tokenInfo = new OzoneTokenIdentifier.TokenInfo(expiryTime, token.getPassword(), ozoneTokenIdentifier.getTrackingId());
        this.currentTokens.put(ozoneTokenIdentifier, tokenInfo);
    }

    public OzoneTokenIdentifier cancelToken(Token<OzoneTokenIdentifier> token, String canceller) throws IOException {
        OzoneTokenIdentifier id = OzoneTokenIdentifier.readProtoBuf((byte[])token.getIdentifier());
        if (LOG.isDebugEnabled()) {
            LOG.debug("Token cancellation requested for identifier: {}", (Object)this.formatTokenId((TokenIdentifier)id));
        }
        if (id.getUser() == null) {
            throw new SecretManager.InvalidToken("Token with no owner " + this.formatTokenId((TokenIdentifier)id));
        }
        String owner = id.getUser().getUserName();
        Text renewer = id.getRenewer();
        HadoopKerberosName cancelerKrbName = new HadoopKerberosName(canceller);
        String cancelerShortName = cancelerKrbName.getShortName();
        if (!(canceller.equals(owner) || renewer != null && !renewer.toString().isEmpty() && cancelerShortName.equals(renewer.toString()))) {
            throw new AccessControlException(String.valueOf(canceller) + " is not authorized to cancel the token " + this.formatTokenId((TokenIdentifier)id));
        }
        if (!this.isRatisEnabled) {
            try {
                this.store.removeToken(id);
            }
            catch (IOException e) {
                LOG.error("Unable to remove token " + id.getSequenceNumber(), (Throwable)e);
            }
            OzoneTokenIdentifier.TokenInfo info = this.currentTokens.remove(id);
            if (info == null) {
                throw new SecretManager.InvalidToken("Token not found " + this.formatTokenId((TokenIdentifier)id));
            }
        } else {
            OzoneTokenIdentifier.TokenInfo info = this.currentTokens.get(id);
            if (info == null) {
                throw new SecretManager.InvalidToken("Token not found in-memory map of tokens" + this.formatTokenId((TokenIdentifier)id));
            }
        }
        return id;
    }

    public void removeToken(OzoneTokenIdentifier ozoneTokenIdentifier) {
        this.currentTokens.remove(ozoneTokenIdentifier);
    }

    public byte[] retrievePassword(OzoneTokenIdentifier identifier) throws SecretManager.InvalidToken {
        try {
            this.ozoneManager.checkLeaderStatus();
        }
        catch (OMLeaderNotReadyException | OMNotLeaderException e) {
            SecretManager.InvalidToken wrappedStandby = new SecretManager.InvalidToken("IOException");
            wrappedStandby.initCause(e);
            throw wrappedStandby;
        }
        if (identifier.getTokenType().equals((Object)OzoneManagerProtocolProtos.OMTokenProto.Type.S3AUTHINFO)) {
            return this.validateS3AuthInfo(identifier);
        }
        return this.validateToken(identifier).getPassword();
    }

    private OzoneTokenIdentifier.TokenInfo validateToken(OzoneTokenIdentifier identifier) throws SecretManager.InvalidToken {
        OzoneTokenIdentifier.TokenInfo info = this.currentTokens.get(identifier);
        if (info == null) {
            throw new SecretManager.InvalidToken("token " + this.formatTokenId((TokenIdentifier)identifier) + " can't be found in cache");
        }
        long now = Time.now();
        if (info.getRenewDate() < now) {
            throw new SecretManager.InvalidToken("token " + this.formatTokenId((TokenIdentifier)identifier) + " is " + "expired, current time: " + Time.formatTime((long)now) + " expected renewal time: " + Time.formatTime((long)info.getRenewDate()));
        }
        if (!this.verifySignature(identifier, info.getPassword())) {
            throw new SecretManager.InvalidToken("Tampered/Invalid token.");
        }
        return info;
    }

    public boolean verifySignature(OzoneTokenIdentifier identifier, byte[] password) {
        X509Certificate signerCert;
        try {
            signerCert = this.getCertClient().getCertificate(identifier.getOmCertSerialId());
        }
        catch (CertificateException certificateException) {
            return false;
        }
        if (signerCert == null) {
            return false;
        }
        try {
            signerCert.checkValidity();
        }
        catch (CertificateExpiredException | CertificateNotYetValidException e) {
            LOG.error("signerCert {} is invalid", (Object)signerCert, (Object)e);
            return false;
        }
        try {
            return this.getCertClient().verifySignature(identifier.getBytes(), password, signerCert);
        }
        catch (CertificateException e) {
            LOG.error("verifySignature with signerCert {} failed", (Object)signerCert, (Object)e);
            return false;
        }
    }

    private byte[] validateS3AuthInfo(OzoneTokenIdentifier identifier) throws SecretManager.InvalidToken {
        String awsSecret;
        LOG.trace("Validating S3AuthInfo for identifier:{}", (Object)identifier);
        if (identifier.getOwner() == null) {
            throw new SecretManager.InvalidToken("Owner is missing from the S3 auth token");
        }
        if (!identifier.getOwner().toString().equals(identifier.getAwsAccessId())) {
            LOG.error("Owner and AWSAccessId is different in the S3 token. Possible  security attack: {}", (Object)identifier);
            throw new SecretManager.InvalidToken("Invalid S3 identifier: owner=" + identifier.getOwner() + ", awsAccessId=" + identifier.getAwsAccessId());
        }
        try {
            awsSecret = this.s3SecretManager.getSecretString(identifier.getAwsAccessId());
        }
        catch (IOException e) {
            LOG.error("Error while validating S3 identifier:{}", (Object)identifier, (Object)e);
            throw new SecretManager.InvalidToken("No S3 secret found for S3 identifier:" + identifier);
        }
        if (awsSecret == null) {
            throw new SecretManager.InvalidToken("No S3 secret found for S3 identifier:" + identifier);
        }
        if (AWSV4AuthValidator.validateRequest(identifier.getStrToSign(), identifier.getSignature(), awsSecret)) {
            return identifier.getSignature().getBytes(StandardCharsets.UTF_8);
        }
        throw new SecretManager.InvalidToken("Invalid S3 identifier:" + identifier);
    }

    private void loadTokenSecretState(OzoneSecretStore.OzoneManagerSecretState<OzoneTokenIdentifier> state) throws IOException {
        LOG.info("Loading token state into token manager.");
        for (Map.Entry<OzoneTokenIdentifier, Long> entry : state.getTokenState().entrySet()) {
            this.addPersistedDelegationToken(entry.getKey(), entry.getValue());
        }
    }

    private void addPersistedDelegationToken(OzoneTokenIdentifier identifier, long renewDate) throws IOException {
        if (this.isRunning()) {
            throw new IOException("Can't add persisted delegation token to a running SecretManager.");
        }
        byte[] password = this.createPassword(identifier.getBytes(), this.getCertClient().getPrivateKey());
        if (identifier.getSequenceNumber() > this.getDelegationTokenSeqNum()) {
            this.setDelegationTokenSeqNum(identifier.getSequenceNumber());
        }
        if (this.currentTokens.get(identifier) != null) {
            throw new IOException("Same delegation token being added twice: " + this.formatTokenId((TokenIdentifier)identifier));
        }
        this.currentTokens.put(identifier, new OzoneTokenIdentifier.TokenInfo(renewDate, password, identifier.getTrackingId()));
    }

    public synchronized void start(CertificateClient certClient) throws IOException {
        super.start(certClient);
        this.tokenRemoverThread = new Daemon((Runnable)new ExpiredTokenRemover());
        this.tokenRemoverThread.setName(String.valueOf(this.ozoneManager.getThreadNamePrefix()) + "ExpiredTokenRemover");
        this.tokenRemoverThread.start();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void stopThreads() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Stopping expired delegation token remover thread");
        }
        this.setIsRunning(false);
        if (this.tokenRemoverThread != null) {
            Object object = this.noInterruptsLock;
            synchronized (object) {
                this.tokenRemoverThread.interrupt();
            }
            try {
                this.tokenRemoverThread.join();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("Unable to join on token removal thread", e);
            }
        }
    }

    public void stop() throws IOException {
        super.stop();
        this.stopThreads();
        if (this.store != null) {
            this.store.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void removeExpiredToken() {
        long now = Time.now();
        Object object = this.noInterruptsLock;
        synchronized (object) {
            Iterator<Map.Entry<OzoneTokenIdentifier, OzoneTokenIdentifier.TokenInfo>> i = this.currentTokens.entrySet().iterator();
            while (i.hasNext()) {
                Map.Entry<OzoneTokenIdentifier, OzoneTokenIdentifier.TokenInfo> entry = i.next();
                long renewDate = entry.getValue().getRenewDate();
                if (renewDate >= now) continue;
                i.remove();
                try {
                    this.store.removeToken(entry.getKey());
                }
                catch (IOException iOException) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug("Failed to remove expired token {}", (Object)entry.getValue());
                }
            }
        }
    }

    public long getTokenRemoverScanInterval() {
        return this.tokenRemoverScanInterval;
    }

    public static class Builder {
        private OzoneConfiguration ozoneConf;
        private long tokenMaxLifetime;
        private long tokenRenewInterval;
        private long tokenRemoverScanInterval;
        private Text service;
        private S3SecretManager s3SecretManager;
        private CertificateClient certClient;
        private String omServiceId;
        private OzoneManager ozoneManager;

        public OzoneDelegationTokenSecretManager build() throws IOException {
            return new OzoneDelegationTokenSecretManager(this);
        }

        public Builder setConf(OzoneConfiguration conf) {
            this.ozoneConf = conf;
            return this;
        }

        public Builder setTokenMaxLifetime(long dtMaxLifetime) {
            this.tokenMaxLifetime = dtMaxLifetime;
            return this;
        }

        public Builder setTokenRenewInterval(long dtRenewInterval) {
            this.tokenRenewInterval = dtRenewInterval;
            return this;
        }

        public Builder setTokenRemoverScanInterval(long dtRemoverScanInterval) {
            this.tokenRemoverScanInterval = dtRemoverScanInterval;
            return this;
        }

        public Builder setService(Text dtService) {
            this.service = dtService;
            return this;
        }

        public Builder setS3SecretManager(S3SecretManager s3SecManager) {
            this.s3SecretManager = s3SecManager;
            return this;
        }

        public Builder setCertificateClient(CertificateClient certificateClient) {
            this.certClient = certificateClient;
            return this;
        }

        public Builder setOmServiceId(String serviceId) {
            this.omServiceId = serviceId;
            return this;
        }

        public Builder setOzoneManager(OzoneManager ozoneMgr) {
            this.ozoneManager = ozoneMgr;
            return this;
        }
    }

    private class ExpiredTokenRemover
    extends Thread {
        private long lastTokenCacheCleanup;

        private ExpiredTokenRemover() {
        }

        @Override
        public void run() {
            LOG.info("Starting expired delegation token remover thread, tokenRemoverScanInterval={} min(s)", (Object)(OzoneDelegationTokenSecretManager.this.getTokenRemoverScanInterval() / 60000L));
            try {
                while (OzoneDelegationTokenSecretManager.this.isRunning()) {
                    long now = Time.now();
                    if (this.lastTokenCacheCleanup + OzoneDelegationTokenSecretManager.this.getTokenRemoverScanInterval() < now) {
                        OzoneDelegationTokenSecretManager.this.removeExpiredToken();
                        this.lastTokenCacheCleanup = now;
                    }
                    Thread.sleep(Math.min(5000L, OzoneDelegationTokenSecretManager.this.getTokenRemoverScanInterval()));
                }
            }
            catch (InterruptedException ie) {
                LOG.info("ExpiredTokenRemover was interrupted.", (Throwable)ie);
                Thread.currentThread().interrupt();
            }
            catch (Exception t) {
                LOG.error("ExpiredTokenRemover thread received unexpected exception", (Throwable)t);
                Runtime.getRuntime().exit(-1);
            }
        }
    }
}

