/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.s3.signature;

import com.google.common.annotations.VisibleForTesting;
import java.io.UnsupportedEncodingException;
import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.util.Map;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.ozone.s3.signature.Credential;
import org.apache.hadoop.ozone.s3.signature.MalformedResourceException;
import org.apache.hadoop.ozone.s3.signature.SignatureInfo;
import org.apache.hadoop.ozone.s3.signature.SignatureParser;
import org.apache.hadoop.ozone.s3.signature.SignatureProcessor;
import org.apache.hadoop.ozone.s3.signature.StringToSignProducer;
import org.apache.hadoop.ozone.s3.util.S3Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AuthorizationV4QueryParser
implements SignatureParser {
    private static final Logger LOG = LoggerFactory.getLogger(AuthorizationV4QueryParser.class);
    private final Map<String, String> queryParameters;
    private static final Long X_AMZ_EXPIRES_MIN = 1L;
    private static final Long X_AMZ_EXPIRES_MAX = 604800L;
    private static final String AWS_REQUEST = "aws4_request";

    public AuthorizationV4QueryParser(Map<String, String> queryParameters) {
        this.queryParameters = queryParameters;
    }

    @Override
    public SignatureInfo parseSignature() throws MalformedResourceException {
        if (!this.queryParameters.containsKey("X-Amz-Signature")) {
            return null;
        }
        this.validateAlgorithm();
        try {
            this.validateDateAndExpires();
        }
        catch (DateTimeParseException ex) {
            throw new MalformedResourceException("Invalid X-Amz-Date format: " + this.queryParameters.get("X-Amz-Date"));
        }
        String rawCredential = this.queryParameters.get("X-Amz-Credential");
        Credential credential = null;
        try {
            credential = new Credential(S3Utils.urlDecode(rawCredential));
        }
        catch (UnsupportedEncodingException e) {
            throw new MalformedResourceException("X-Amz-Credential is not proper URL encoded rawCredential:" + rawCredential);
        }
        this.validateCredential(credential);
        this.validateSignedHeaders();
        this.validateSignature();
        return new SignatureInfo(SignatureInfo.Version.V4, credential.getDate(), this.queryParameters.get("X-Amz-Date"), credential.getAccessKeyID(), this.queryParameters.get("X-Amz-Signature"), this.queryParameters.get("X-Amz-SignedHeaders"), credential.createScope(), this.queryParameters.get("X-Amz-Algorithm"), false);
    }

    private void validateAlgorithm() throws MalformedResourceException {
        String algorithm = this.queryParameters.get("X-Amz-Algorithm");
        if (algorithm == null) {
            throw new MalformedResourceException("Unspecified signature algorithm.");
        }
        if (StringUtils.isEmpty((CharSequence)algorithm) || !algorithm.equals("AWS4-HMAC-SHA256")) {
            throw new MalformedResourceException("Unsupported signature algorithm: " + algorithm);
        }
    }

    @VisibleForTesting
    protected void validateDateAndExpires() throws MalformedResourceException, DateTimeParseException {
        String dateString = this.queryParameters.get("X-Amz-Date");
        String expiresString = this.queryParameters.get("X-Amz-Expires");
        if (dateString == null || expiresString == null || dateString.length() == 0 || expiresString.length() == 0) {
            throw new MalformedResourceException("dateString or expiresString are missing or empty.");
        }
        Long expires = Long.parseLong(expiresString);
        if (expires >= X_AMZ_EXPIRES_MIN && expires <= X_AMZ_EXPIRES_MAX) {
            if (ZonedDateTime.parse(dateString, StringToSignProducer.TIME_FORMATTER).plus(expires, ChronoUnit.SECONDS).isBefore(ZonedDateTime.now())) {
                throw new MalformedResourceException("Pre-signed S3 url is expired. dateString:" + dateString + " expiresString:" + expiresString);
            }
        } else {
            throw new MalformedResourceException("Invalid expiry duration. X-Amz-Expires should be between " + X_AMZ_EXPIRES_MIN + "and" + X_AMZ_EXPIRES_MAX + " expiresString:" + expiresString);
        }
    }

    private void validateCredential(Credential credential) throws MalformedResourceException {
        if (credential.getAccessKeyID().isEmpty()) {
            throw new MalformedResourceException("AWS access id is empty. credential: " + credential);
        }
        if (credential.getAwsRegion().isEmpty()) {
            throw new MalformedResourceException("AWS region is empty. credential: " + credential);
        }
        if (credential.getAwsRequest().isEmpty() || !credential.getAwsRequest().equals(AWS_REQUEST)) {
            throw new MalformedResourceException("AWS request is empty or invalid. credential:" + credential);
        }
        if (credential.getAwsService().isEmpty()) {
            throw new MalformedResourceException("AWS service is empty. credential:" + credential);
        }
        if (!credential.getDate().isEmpty()) {
            try {
                LocalDate.parse(credential.getDate(), SignatureProcessor.DATE_FORMATTER);
            }
            catch (DateTimeParseException ex) {
                throw new MalformedResourceException("AWS date format is invalid. credential:" + credential);
            }
        } else {
            throw new MalformedResourceException("AWS date is empty. credential:{}" + credential);
        }
    }

    private void validateSignedHeaders() throws MalformedResourceException {
        String signedHeadersStr = this.queryParameters.get("X-Amz-SignedHeaders");
        if (signedHeadersStr == null || StringUtils.isEmpty((CharSequence)signedHeadersStr) || signedHeadersStr.split(";").length == 0) {
            throw new MalformedResourceException("No signed headers found.");
        }
    }

    private void validateSignature() throws MalformedResourceException {
        String signature = this.queryParameters.get("X-Amz-Signature");
        if (StringUtils.isEmpty((CharSequence)signature)) {
            throw new MalformedResourceException("Signature is empty.");
        }
        try {
            Hex.decodeHex((String)signature);
        }
        catch (DecoderException e) {
            throw new MalformedResourceException("Signature:" + signature + " should be in hexa-decimal encoding.");
        }
    }
}

