/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.pipeline;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;
import java.util.Collection;
import java.util.List;
import java.util.NavigableSet;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.hdds.client.ReplicationConfig;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.SCMRatisProtocol;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.ha.SCMHAInvocationHandler;
import org.apache.hadoop.hdds.scm.ha.SCMRatisServer;
import org.apache.hadoop.hdds.scm.metadata.DBTransactionBuffer;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.pipeline.PipelineNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.PipelineStateManager;
import org.apache.hadoop.hdds.scm.pipeline.PipelineStateMap;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.hdds.utils.db.TableIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PipelineStateManagerImpl
implements PipelineStateManager {
    private static final Logger LOG = LoggerFactory.getLogger(PipelineStateManagerImpl.class);
    private PipelineStateMap pipelineStateMap;
    private final NodeManager nodeManager;
    private Table<PipelineID, Pipeline> pipelineStore;
    private final DBTransactionBuffer transactionBuffer;
    private final ReadWriteLock lock = new ReentrantReadWriteLock();

    public PipelineStateManagerImpl(Table<PipelineID, Pipeline> pipelineStore, NodeManager nodeManager, DBTransactionBuffer buffer) throws IOException {
        this.pipelineStateMap = new PipelineStateMap();
        this.nodeManager = nodeManager;
        this.pipelineStore = pipelineStore;
        this.transactionBuffer = buffer;
        this.initialize();
    }

    private void initialize() throws IOException {
        if (this.pipelineStore == null || this.nodeManager == null) {
            throw new IOException("PipelineStore cannot be null");
        }
        if (this.pipelineStore.isEmpty()) {
            LOG.info("No pipeline exists in current db");
            return;
        }
        try (TableIterator iterator = this.pipelineStore.iterator();){
            while (iterator.hasNext()) {
                Pipeline pipeline = (Pipeline)((Table.KeyValue)iterator.next()).getValue();
                this.pipelineStateMap.addPipeline(pipeline);
                this.nodeManager.addPipeline(pipeline);
            }
        }
    }

    @Override
    public void addPipeline(HddsProtos.Pipeline pipelineProto) throws IOException {
        Pipeline pipeline = Pipeline.getFromProtobuf((HddsProtos.Pipeline)pipelineProto);
        this.lock.writeLock().lock();
        try {
            if (this.pipelineStore != null) {
                this.pipelineStateMap.addPipeline(pipeline);
                this.nodeManager.addPipeline(pipeline);
                this.transactionBuffer.addToBuffer(this.pipelineStore, (Object)pipeline.getId(), (Object)pipeline);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public void addContainerToPipeline(PipelineID pipelineId, ContainerID containerID) throws IOException {
        this.lock.writeLock().lock();
        try {
            this.pipelineStateMap.addContainerToPipeline(pipelineId, containerID);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public void addContainerToPipelineForce(PipelineID pipelineId, ContainerID containerID) throws IOException {
        this.lock.writeLock().lock();
        try {
            this.pipelineStateMap.addContainerToPipelineSCMStart(pipelineId, containerID);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public Pipeline getPipeline(PipelineID pipelineID) throws PipelineNotFoundException {
        this.lock.readLock().lock();
        try {
            Pipeline pipeline = this.pipelineStateMap.getPipeline(pipelineID);
            return pipeline;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    @Override
    public List<Pipeline> getPipelines() {
        this.lock.readLock().lock();
        try {
            List<Pipeline> list = this.pipelineStateMap.getPipelines();
            return list;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    @Override
    public List<Pipeline> getPipelines(ReplicationConfig replicationConfig) {
        this.lock.readLock().lock();
        try {
            List<Pipeline> list = this.pipelineStateMap.getPipelines(replicationConfig);
            return list;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Pipeline> getPipelines(ReplicationConfig replicationConfig, Pipeline.PipelineState state) {
        this.lock.readLock().lock();
        try {
            List<Pipeline> list = this.pipelineStateMap.getPipelines(replicationConfig, state);
            return list;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Pipeline> getPipelines(ReplicationConfig replicationConfig, Pipeline.PipelineState state, Collection<DatanodeDetails> excludeDns, Collection<PipelineID> excludePipelines) {
        this.lock.readLock().lock();
        try {
            List<Pipeline> list = this.pipelineStateMap.getPipelines(replicationConfig, state, excludeDns, excludePipelines);
            return list;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int getPipelineCount(ReplicationConfig replicationConfig, Pipeline.PipelineState state) {
        this.lock.readLock().lock();
        try {
            int n = this.pipelineStateMap.getPipelineCount(replicationConfig, state);
            return n;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    @Override
    public NavigableSet<ContainerID> getContainers(PipelineID pipelineID) throws IOException {
        this.lock.readLock().lock();
        try {
            NavigableSet<ContainerID> navigableSet = this.pipelineStateMap.getContainers(pipelineID);
            return navigableSet;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    @Override
    public int getNumberOfContainers(PipelineID pipelineID) throws IOException {
        this.lock.readLock().lock();
        try {
            int n = this.pipelineStateMap.getNumberOfContainers(pipelineID);
            return n;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void removePipeline(HddsProtos.PipelineID pipelineIDProto) throws IOException {
        PipelineID pipelineID = PipelineID.getFromProtobuf((HddsProtos.PipelineID)pipelineIDProto);
        try {
            this.lock.writeLock().lock();
            try {
                Pipeline pipeline = this.pipelineStateMap.removePipeline(pipelineID);
                this.nodeManager.removePipeline(pipeline);
                if (this.pipelineStore != null) {
                    this.transactionBuffer.removeFromBuffer(this.pipelineStore, (Object)pipelineID);
                }
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }
        catch (PipelineNotFoundException pnfe) {
            LOG.warn("Pipeline {} is not found in the pipeline Map. Pipeline may have been deleted already.", (Object)pipelineIDProto.getId());
        }
    }

    @Override
    public void removeContainerFromPipeline(PipelineID pipelineID, ContainerID containerID) throws IOException {
        try {
            this.lock.writeLock().lock();
            try {
                this.pipelineStateMap.removeContainerFromPipeline(pipelineID, containerID);
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }
        catch (PipelineNotFoundException pnfe) {
            LOG.info("Pipeline {} is not found in the pipeline2ContainerMap. Pipeline may have been closed already.", (Object)pipelineID);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void updatePipelineState(HddsProtos.PipelineID pipelineIDProto, HddsProtos.PipelineState newState) throws IOException {
        PipelineID pipelineID = PipelineID.getFromProtobuf((HddsProtos.PipelineID)pipelineIDProto);
        Pipeline.PipelineState newPipelineState = Pipeline.PipelineState.fromProtobuf((HddsProtos.PipelineState)newState);
        try {
            this.lock.writeLock().lock();
            try {
                if (this.pipelineStore != null) {
                    this.pipelineStateMap.updatePipelineState(pipelineID, newPipelineState);
                    this.transactionBuffer.addToBuffer(this.pipelineStore, (Object)pipelineID, (Object)this.getPipeline(pipelineID));
                }
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }
        catch (PipelineNotFoundException pnfe) {
            LOG.warn("Pipeline {} is not found in the pipeline Map. Pipeline may have been deleted already.", (Object)pipelineID);
        }
        catch (IOException ex) {
            LOG.error("Pipeline {} state update failed", (Object)pipelineID);
            throw ex;
        }
    }

    @Override
    public void close() throws Exception {
        this.lock.writeLock().lock();
        try {
            if (this.pipelineStore != null) {
                this.pipelineStore.close();
                this.pipelineStore = null;
            }
        }
        catch (Exception ex) {
            LOG.error("Pipeline store close failed", (Throwable)ex);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public void reinitialize(Table<PipelineID, Pipeline> store) throws IOException {
        this.lock.writeLock().lock();
        try {
            this.pipelineStore.close();
            this.pipelineStateMap = new PipelineStateMap();
            this.pipelineStore = store;
            this.initialize();
        }
        catch (Exception ex) {
            LOG.error("PipelineManager reinitialization close failed", (Throwable)ex);
            throw new IOException(ex);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {
        private Table<PipelineID, Pipeline> pipelineStore;
        private NodeManager nodeManager;
        private SCMRatisServer scmRatisServer;
        private DBTransactionBuffer transactionBuffer;

        public Builder setSCMDBTransactionBuffer(DBTransactionBuffer buffer) {
            this.transactionBuffer = buffer;
            return this;
        }

        public Builder setRatisServer(SCMRatisServer ratisServer) {
            this.scmRatisServer = ratisServer;
            return this;
        }

        public Builder setNodeManager(NodeManager scmNodeManager) {
            this.nodeManager = scmNodeManager;
            return this;
        }

        public Builder setPipelineStore(Table<PipelineID, Pipeline> pipelineTable) {
            this.pipelineStore = pipelineTable;
            return this;
        }

        public PipelineStateManager build() throws IOException {
            Preconditions.checkNotNull(this.pipelineStore);
            PipelineStateManagerImpl pipelineStateManager = new PipelineStateManagerImpl(this.pipelineStore, this.nodeManager, this.transactionBuffer);
            SCMHAInvocationHandler invocationHandler = new SCMHAInvocationHandler(SCMRatisProtocol.RequestType.PIPELINE, pipelineStateManager, this.scmRatisServer);
            return (PipelineStateManager)Proxy.newProxyInstance(SCMHAInvocationHandler.class.getClassLoader(), new Class[]{PipelineStateManager.class}, (InvocationHandler)invocationHandler);
        }
    }
}

