/****************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one   *
 * or more contributor license agreements.  See the NOTICE file *
 * distributed with this work for additional information        *
 * regarding copyright ownership.  The ASF licenses this file   *
 * to you under the Apache License, Version 2.0 (the            *
 * "License"); you may not use this file except in compliance   *
 * with the License.  You may obtain a copy of the License at   *
 *                                                              *
 *   http://www.apache.org/licenses/LICENSE-2.0                 *
 *                                                              *
 * Unless required by applicable law or agreed to in writing,   *
 * software distributed under the License is distributed on an  *
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY       *
 * KIND, either express or implied.  See the License for the    *
 * specific language governing permissions and limitations      *
 * under the License.                                           *
 ****************************************************************/

package org.apache.james.webadmin.service;

import static org.apache.james.JsonSerializationVerifier.recursiveComparisonConfiguration;
import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.mock;

import java.util.Optional;

import org.apache.james.JsonSerializationVerifier;
import org.apache.james.json.JsonGenericSerializer;
import org.apache.james.mailbox.model.MailboxConstants;
import org.apache.james.util.ClassLoaderUtils;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

class ExpireMailboxTaskSerializationTest {
    private ExpireMailboxService expireMailboxService;

    @BeforeEach
    void setUp() {
        expireMailboxService = mock(ExpireMailboxService.class);
    }

    @Test
    void shouldMatchJsonSerializationContractWithOlderThan() throws Exception {
        JsonSerializationVerifier.dtoModule(ExpireMailboxDTO.module(expireMailboxService))
            .bean(new ExpireMailboxTask(expireMailboxService, new ExpireMailboxService.RunningOptions(1, MailboxConstants.INBOX,
                Optional.of(false), Optional.empty(), false, Optional.of("90d"))))
            .json(ClassLoaderUtils.getSystemResourceAsString("json/expireMailbox.age.task.v2.json"))
            .verify();
    }

    @Test
    void shouldMatchJsonSerializationContractWithExpiresHeader() throws Exception {
        JsonSerializationVerifier.dtoModule(ExpireMailboxDTO.module(expireMailboxService))
            .bean(new ExpireMailboxTask(expireMailboxService, new ExpireMailboxService.RunningOptions(1, MailboxConstants.INBOX,
                Optional.of(true), Optional.of("bob"), true, Optional.empty())))
            .json(ClassLoaderUtils.getSystemResourceAsString("json/expireMailbox.header.task.v2.json"))
            .verify();
    }

    @Test
    void shouldDeserializeLegacyWhenHeader() throws Exception {
        ExpireMailboxTask actual = JsonGenericSerializer
            .forModules(ExpireMailboxDTO.module(expireMailboxService))
            .withoutNestedType()
            .deserialize(ClassLoaderUtils.getSystemResourceAsString("json/expireMailbox.age.task.json"));

        assertThat(actual)
            .usingRecursiveComparison(recursiveComparisonConfiguration)
            .isEqualTo(new ExpireMailboxTask(expireMailboxService, new ExpireMailboxService.RunningOptions(1, MailboxConstants.INBOX,
                Optional.empty(), Optional.empty(), false, Optional.of("90d"))));
    }

    @Test
    void shouldDeserializeLegacy() throws Exception {
        ExpireMailboxTask actual = JsonGenericSerializer
            .forModules(ExpireMailboxDTO.module(expireMailboxService))
            .withoutNestedType()
            .deserialize(ClassLoaderUtils.getSystemResourceAsString("json/expireMailbox.header.task.json"));

        assertThat(actual)
            .usingRecursiveComparison(recursiveComparisonConfiguration)
            .isEqualTo(new ExpireMailboxTask(expireMailboxService, new ExpireMailboxService.RunningOptions(1, MailboxConstants.INBOX,
                Optional.empty(), Optional.empty(), true, Optional.empty())));
    }
}
