/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.baremaps.openstreetmap;

import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;
import org.apache.baremaps.openstreetmap.pbf.PbfBlockReader;
import org.apache.baremaps.openstreetmap.pbf.PbfEntityReader;
import org.apache.baremaps.openstreetmap.state.StateReader;
import org.apache.baremaps.openstreetmap.xml.XmlChangeReader;
import org.apache.baremaps.openstreetmap.xml.XmlEntityReader;
import org.locationtech.jts.geom.Coordinate;

/**
 * A utility class for reading OpenStreetMap files.
 */
public class OpenStreetMapFormat {

  private OpenStreetMapFormat() {}

  /**
   * Returns a reader of OpenStreetMap states.
   * 
   * @return a reader
   */
  static StateReader stateReader() {
    return new StateReader();
  }

  /**
   * Returns a reader of OpenStreetMap pbf blocks.
   * 
   * @return a reader
   */
  static PbfBlockReader pbfBlockReader() {
    return new PbfBlockReader();
  }

  /**
   * Returns a reader of OpenStreetMap pbf entities.
   * 
   * @return a reader
   */
  static PbfEntityReader pbfEntityReader() {
    return new PbfEntityReader();
  }

  /**
   * Returns a reader of OpenStreetMap xml entities.
   * 
   * @return a reader
   */
  static XmlEntityReader xmlEntityReader() {
    return new XmlEntityReader();
  }

  /**
   * Returns a reader of OpenStreetMap xml changes.
   * 
   * @return a reader
   */
  static XmlChangeReader xmlChangeReader() {
    return new XmlChangeReader();
  }

  /**
   * A reader of objects of type T from an input stream.
   * 
   * @param <T> the type of the object
   */
  public interface Reader<T> {

    T read(InputStream inputStream);

  }

  /**
   * A reader of OpenStreetMap entities from an input stream.
   *
   * @param <T> the type of the object
   */
  public interface EntityReader<T> extends Reader<Stream<T>> {

    /**
     * Gets the flag enabling the generation of geometries.
     *
     * @return the value of the flag
     */
    boolean getGeometries();

    /**
     * Sets the flag enabling the generation of geometries.
     *
     * @param geometries the value of the flag
     * @return the reader
     */
    EntityReader<T> setGeometries(boolean geometries);

    /**
     * Gets the projection of the geometries generated by this parser.
     *
     * @return the projection of the geometries
     */
    int getSrid();

    /**
     * Sets the projection of the geometries generated by this parser.
     *
     * @param srid the projection of the geometries
     * @return the reader
     */
    EntityReader<T> setSrid(int srid);

    /**
     * Gets the map used to store coordinates for generating geometries.
     *
     * @return the map of coordinates
     */
    Map<Long, Coordinate> getCoordinateMap();

    /**
     * Sets the map used to store coordinates for generating geometries.
     *
     * @param coordinateMap the map of coordinates
     * @return the reader
     */
    EntityReader<T> setCoordinateMap(Map<Long, Coordinate> coordinateMap);

    /**
     * Gets the map used to store references for generating geometries.
     *
     * @return the map of references
     */
    Map<Long, List<Long>> getReferenceMap();

    /**
     * Sets the map used to store references for generating geometries.
     *
     * @param referenceMap the map of references
     * @return the reader
     */
    EntityReader<T> setReferenceMap(Map<Long, List<Long>> referenceMap);

  }
}
