/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.fs;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.Objects;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.HBaseIOException;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.util.CommonFSUtils;
import org.apache.hadoop.hbase.util.ReflectionUtils;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public final class ErasureCodingUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ErasureCodingUtils.class);

    private ErasureCodingUtils() {
    }

    public static void verifySupport(Configuration conf, String policy) throws HBaseIOException {
        DistributedFileSystem dfs = ErasureCodingUtils.getDfs(conf);
        ErasureCodingUtils.checkAvailable(dfs, policy);
        Path globalTempDir = new Path(conf.get("hbase.rootdir"), ".tmp");
        Path currentTempDir = ErasureCodingUtils.createTempDir((FileSystem)dfs, globalTempDir);
        try {
            ErasureCodingUtils.setPolicy((FileSystem)dfs, currentTempDir, policy);
            try (FSDataOutputStream out = dfs.create(new Path(currentTempDir, "test.out"));){
                out.writeUTF("Testing " + policy);
            }
        }
        catch (IOException e) {
            throw new DoNotRetryIOException("Failed write test for EC policy. Check cause or logs", e);
        }
        finally {
            try {
                dfs.delete(currentTempDir, true);
            }
            catch (IOException e) {
                LOG.warn("Failed to delete temp path for ec test", (Throwable)e);
            }
        }
    }

    private static Path createTempDir(FileSystem fs, Path tempDir) throws HBaseIOException {
        Path currentTempDir = new Path(tempDir, "ec-test-" + System.currentTimeMillis());
        try {
            fs.mkdirs(currentTempDir);
            fs.deleteOnExit(currentTempDir);
        }
        catch (IOException e) {
            throw new HBaseIOException("Failed to create test dir for EC write test", e);
        }
        return currentTempDir;
    }

    private static void checkAvailable(DistributedFileSystem dfs, String requestedPolicy) throws HBaseIOException {
        Collection policies;
        try {
            policies = (Collection)ErasureCodingUtils.callDfsMethod(dfs, "getAllErasureCodingPolicies", new Object[0]);
        }
        catch (IOException e) {
            throw new HBaseIOException("Failed to check for Erasure Coding policy: " + requestedPolicy, e);
        }
        for (Object policyInfo : policies) {
            if (!ErasureCodingUtils.checkPolicyMatch(policyInfo, requestedPolicy)) continue;
            return;
        }
        throw new DoNotRetryIOException("Cannot set Erasure Coding policy: " + requestedPolicy + ". Policy not found. Available policies are: " + ErasureCodingUtils.getPolicyNames(policies));
    }

    private static boolean checkPolicyMatch(Object policyInfo, String requestedPolicy) throws DoNotRetryIOException {
        try {
            String policyName = ErasureCodingUtils.getPolicyNameFromInfo(policyInfo);
            if (requestedPolicy.equals(policyName)) {
                boolean isEnabled = (Boolean)ErasureCodingUtils.callObjectMethod(policyInfo, "isEnabled", new Object[0]);
                if (!isEnabled) {
                    throw new DoNotRetryIOException("Cannot set Erasure Coding policy: " + requestedPolicy + ". The policy must be enabled, but has state " + ErasureCodingUtils.callObjectMethod(policyInfo, "getState", new Object[0]));
                }
                return true;
            }
        }
        catch (DoNotRetryIOException e) {
            throw e;
        }
        catch (IOException e) {
            throw new DoNotRetryIOException("Unable to check for match of Erasure Coding Policy " + policyInfo, e);
        }
        return false;
    }

    private static String getPolicyNameFromInfo(Object policyInfo) throws IOException {
        Object policy = ErasureCodingUtils.callObjectMethod(policyInfo, "getPolicy", new Object[0]);
        return (String)ErasureCodingUtils.callObjectMethod(policy, "getName", new Object[0]);
    }

    private static String getPolicyNames(Collection<Object> policyInfos) {
        return policyInfos.stream().map(p -> {
            try {
                return ErasureCodingUtils.getPolicyNameFromInfo(p);
            }
            catch (IOException e) {
                LOG.warn("Could not extract policy name from {}", p, (Object)e);
                return "unknown";
            }
        }).collect(Collectors.joining(", "));
    }

    public static boolean needsSync(TableDescriptor oldDescriptor, TableDescriptor newDescriptor) {
        String newPolicy = oldDescriptor.getErasureCodingPolicy();
        String oldPolicy = newDescriptor.getErasureCodingPolicy();
        return !Objects.equals(oldPolicy, newPolicy);
    }

    public static void sync(FileSystem fs, Path rootDir, TableDescriptor newDescriptor) throws IOException {
        String newPolicy = newDescriptor.getErasureCodingPolicy();
        if (newPolicy == null) {
            ErasureCodingUtils.unsetPolicy(fs, rootDir, newDescriptor.getTableName());
        } else {
            ErasureCodingUtils.setPolicy(fs, rootDir, newDescriptor.getTableName(), newPolicy);
        }
    }

    public static void setPolicy(FileSystem fs, Path rootDir, TableName tableName, String policy) throws IOException {
        Path path = CommonFSUtils.getTableDir(rootDir, tableName);
        ErasureCodingUtils.setPolicy(fs, path, policy);
    }

    public static void setPolicy(FileSystem fs, Path path, String policy) throws IOException {
        ErasureCodingUtils.callDfsMethod(ErasureCodingUtils.getDfs(fs), "setErasureCodingPolicy", path, policy);
    }

    public static void unsetPolicy(FileSystem fs, Path rootDir, TableName tableName) throws IOException {
        Path path;
        DistributedFileSystem dfs = ErasureCodingUtils.getDfs(fs);
        if (ErasureCodingUtils.getPolicyNameForPath(dfs, path = CommonFSUtils.getTableDir(rootDir, tableName)) == null) {
            LOG.warn("No EC policy set for path {}, nothing to unset", (Object)path);
            return;
        }
        ErasureCodingUtils.callDfsMethod(dfs, "unsetErasureCodingPolicy", path);
    }

    public static void enablePolicy(FileSystem fs, String policy) throws IOException {
        ErasureCodingUtils.callDfsMethod(ErasureCodingUtils.getDfs(fs), "enableErasureCodingPolicy", policy);
    }

    private static DistributedFileSystem getDfs(Configuration conf) throws HBaseIOException {
        try {
            return ErasureCodingUtils.getDfs(FileSystem.get((Configuration)conf));
        }
        catch (DoNotRetryIOException e) {
            throw e;
        }
        catch (IOException e) {
            throw new HBaseIOException("Failed to get FileSystem from conf", e);
        }
    }

    private static DistributedFileSystem getDfs(FileSystem fs) throws DoNotRetryIOException {
        if (!(fs instanceof DistributedFileSystem)) {
            throw new DoNotRetryIOException("Cannot manage Erasure Coding policy. Erasure Coding is only available on HDFS, but fs is " + fs.getClass().getSimpleName());
        }
        return (DistributedFileSystem)fs;
    }

    public static String getPolicyNameForPath(DistributedFileSystem dfs, Path path) throws IOException {
        Object policy = ErasureCodingUtils.callDfsMethod(dfs, "getErasureCodingPolicy", path);
        if (policy == null) {
            return null;
        }
        return (String)ErasureCodingUtils.callObjectMethod(policy, "getName", new Object[0]);
    }

    private static <T> T callDfsMethod(DistributedFileSystem dfs, String name, Object ... params) throws IOException {
        return ErasureCodingUtils.callObjectMethod(dfs, name, params);
    }

    private static <T> T callObjectMethod(Object object, String name, Object ... params) throws IOException {
        return ErasureCodingUtils.unwrapInvocationException(() -> ReflectionUtils.invokeMethod(object, name, params));
    }

    private static <T> T unwrapInvocationException(ThrowingObjectSupplier runnable) throws IOException {
        try {
            return (T)runnable.run();
        }
        catch (UnsupportedOperationException e) {
            if (e.getCause() instanceof InvocationTargetException) {
                Throwable cause = e.getCause().getCause();
                if (cause instanceof IOException) {
                    throw (IOException)cause;
                }
                if (cause instanceof RuntimeException) {
                    throw (RuntimeException)cause;
                }
            }
            throw e;
        }
    }

    private static interface ThrowingObjectSupplier {
        public Object run() throws IOException;
    }
}

