/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.security.token.block;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import java.io.ByteArrayInputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.IOException;
import java.security.MessageDigest;
import java.security.SecureRandom;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.crypto.SecretKey;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.datatransfer.InvalidEncryptionKeyException;
import org.apache.hadoop.hdfs.security.token.block.BlockKey;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenIdentifier;
import org.apache.hadoop.hdfs.security.token.block.DataEncryptionKey;
import org.apache.hadoop.hdfs.security.token.block.ExportedBlockKeys;
import org.apache.hadoop.io.WritableUtils;
import org.apache.hadoop.ipc.Server;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.SecretManager;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.Timer;

@InterfaceAudience.Private
public class BlockTokenSecretManager
extends SecretManager<BlockTokenIdentifier> {
    public static final Log LOG = LogFactory.getLog(BlockTokenSecretManager.class);
    public static final Token<BlockTokenIdentifier> DUMMY_TOKEN = new Token();
    private final boolean isMaster;
    private long keyUpdateInterval;
    private volatile long tokenLifetime;
    private int serialNo;
    private BlockKey currentKey;
    private BlockKey nextKey;
    private final Map<Integer, BlockKey> allKeys;
    private String blockPoolId;
    private final String encryptionAlgorithm;
    private final int intRange;
    private final int nnRangeStart;
    private final boolean shouldWrapQOP;
    private final SecureRandom nonceGenerator = new SecureRandom();
    private Timer timer;

    public BlockTokenSecretManager(long keyUpdateInterval, long tokenLifetime, String blockPoolId, String encryptionAlgorithm) {
        this(false, keyUpdateInterval, tokenLifetime, blockPoolId, encryptionAlgorithm, 0, 1, false);
    }

    public BlockTokenSecretManager(long keyUpdateInterval, long tokenLifetime, int nnIndex, int numNNs, String blockPoolId, String encryptionAlgorithm, boolean shouldWrapQOP) {
        this(true, keyUpdateInterval, tokenLifetime, blockPoolId, encryptionAlgorithm, nnIndex, numNNs, shouldWrapQOP);
        Preconditions.checkArgument((nnIndex >= 0 ? 1 : 0) != 0);
        Preconditions.checkArgument((numNNs > 0 ? 1 : 0) != 0);
    }

    public BlockTokenSecretManager(long keyUpdateInterval, long tokenLifetime, int nnIndex, int numNNs, String blockPoolId, String encryptionAlgorithm) {
        this(keyUpdateInterval, tokenLifetime, nnIndex, numNNs, blockPoolId, encryptionAlgorithm, false);
    }

    private BlockTokenSecretManager(boolean isMaster, long keyUpdateInterval, long tokenLifetime, String blockPoolId, String encryptionAlgorithm, int nnIndex, int numNNs, boolean shouldWrapQOP) {
        this.intRange = Integer.MAX_VALUE / numNNs;
        this.nnRangeStart = this.intRange * nnIndex;
        this.isMaster = isMaster;
        this.keyUpdateInterval = keyUpdateInterval;
        this.tokenLifetime = tokenLifetime;
        this.allKeys = new HashMap<Integer, BlockKey>();
        this.blockPoolId = blockPoolId;
        this.encryptionAlgorithm = encryptionAlgorithm;
        this.shouldWrapQOP = shouldWrapQOP;
        this.timer = new Timer();
        this.setSerialNo(new SecureRandom().nextInt(Integer.MAX_VALUE));
        LOG.info((Object)("Block token key range: [" + this.nnRangeStart + ", " + (this.nnRangeStart + this.intRange) + ")"));
        this.generateKeys();
    }

    @VisibleForTesting
    public synchronized void setSerialNo(int nextNo) {
        this.serialNo = nextNo % this.intRange + this.nnRangeStart;
        assert (this.serialNo >= this.nnRangeStart && this.serialNo < this.nnRangeStart + this.intRange) : "serialNo " + this.serialNo + " is not in the designated range: [" + this.nnRangeStart + ", " + (this.nnRangeStart + this.intRange) + ")";
    }

    public void setBlockPoolId(String blockPoolId) {
        this.blockPoolId = blockPoolId;
    }

    private synchronized void generateKeys() {
        if (!this.isMaster) {
            return;
        }
        this.setSerialNo(this.serialNo + 1);
        this.currentKey = new BlockKey(this.serialNo, this.timer.now() + 2L * this.keyUpdateInterval + this.tokenLifetime, this.generateSecret());
        this.setSerialNo(this.serialNo + 1);
        this.nextKey = new BlockKey(this.serialNo, this.timer.now() + 3L * this.keyUpdateInterval + this.tokenLifetime, this.generateSecret());
        this.allKeys.put(this.currentKey.getKeyId(), this.currentKey);
        this.allKeys.put(this.nextKey.getKeyId(), this.nextKey);
    }

    public synchronized ExportedBlockKeys exportKeys() {
        if (!this.isMaster) {
            return null;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)"Exporting access keys");
        }
        return new ExportedBlockKeys(true, this.keyUpdateInterval, this.tokenLifetime, this.currentKey, this.allKeys.values().toArray(new BlockKey[0]));
    }

    private synchronized void removeExpiredKeys() {
        long now = this.timer.now();
        Iterator<Map.Entry<Integer, BlockKey>> it = this.allKeys.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<Integer, BlockKey> e = it.next();
            if (e.getValue().getExpiryDate() >= now) continue;
            it.remove();
        }
    }

    public synchronized void addKeys(ExportedBlockKeys exportedKeys) throws IOException {
        if (this.isMaster || exportedKeys == null) {
            return;
        }
        LOG.info((Object)"Setting block keys");
        this.removeExpiredKeys();
        this.currentKey = exportedKeys.getCurrentKey();
        BlockKey[] receivedKeys = exportedKeys.getAllKeys();
        for (int i = 0; i < receivedKeys.length; ++i) {
            if (receivedKeys[i] == null) continue;
            this.allKeys.put(receivedKeys[i].getKeyId(), receivedKeys[i]);
        }
    }

    public synchronized boolean updateKeys(long updateTime) throws IOException {
        if (updateTime > this.keyUpdateInterval) {
            return this.updateKeys();
        }
        return false;
    }

    synchronized boolean updateKeys() throws IOException {
        if (!this.isMaster) {
            return false;
        }
        LOG.info((Object)"Updating block keys");
        this.removeExpiredKeys();
        this.allKeys.put(this.currentKey.getKeyId(), new BlockKey(this.currentKey.getKeyId(), this.timer.now() + this.keyUpdateInterval + this.tokenLifetime, this.currentKey.getKey()));
        this.currentKey = new BlockKey(this.nextKey.getKeyId(), this.timer.now() + 2L * this.keyUpdateInterval + this.tokenLifetime, this.nextKey.getKey());
        this.allKeys.put(this.currentKey.getKeyId(), this.currentKey);
        this.setSerialNo(this.serialNo + 1);
        this.nextKey = new BlockKey(this.serialNo, this.timer.now() + 3L * this.keyUpdateInterval + this.tokenLifetime, this.generateSecret());
        this.allKeys.put(this.nextKey.getKeyId(), this.nextKey);
        return true;
    }

    public Token<BlockTokenIdentifier> generateToken(ExtendedBlock block, EnumSet<BlockTokenIdentifier.AccessMode> modes) throws IOException {
        UserGroupInformation ugi = UserGroupInformation.getCurrentUser();
        String userID = ugi == null ? null : ugi.getShortUserName();
        return this.generateToken(userID, block, modes);
    }

    public Token<BlockTokenIdentifier> generateToken(String userId, ExtendedBlock block, EnumSet<BlockTokenIdentifier.AccessMode> modes) throws IOException {
        String qop;
        BlockTokenIdentifier id = new BlockTokenIdentifier(userId, block.getBlockPoolId(), block.getBlockId(), modes);
        if (this.shouldWrapQOP && (qop = Server.getAuxiliaryPortEstablishedQOP()) != null) {
            id.setHandshakeMsg(qop.getBytes(Charsets.UTF_8));
        }
        return new Token((TokenIdentifier)id, (SecretManager)this);
    }

    public void checkAccess(BlockTokenIdentifier id, String userId, ExtendedBlock block, BlockTokenIdentifier.AccessMode mode) throws SecretManager.InvalidToken {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Checking access for user=" + userId + ", block=" + block + ", access mode=" + mode + " using " + id.toString()));
        }
        if (userId != null && !userId.equals(id.getUserId())) {
            throw new SecretManager.InvalidToken("Block token with " + id.toString() + " doesn't belong to user " + userId);
        }
        if (!id.getBlockPoolId().equals(block.getBlockPoolId())) {
            throw new SecretManager.InvalidToken("Block token with " + id.toString() + " doesn't apply to block " + block);
        }
        if (id.getBlockId() != block.getBlockId()) {
            throw new SecretManager.InvalidToken("Block token with " + id.toString() + " doesn't apply to block " + block);
        }
        if (BlockTokenSecretManager.isExpired(id.getExpiryDate())) {
            throw new SecretManager.InvalidToken("Block token with " + id.toString() + " is expired.");
        }
        if (!id.getAccessModes().contains(mode)) {
            throw new SecretManager.InvalidToken("Block token with " + id.toString() + " doesn't have " + mode + " permission");
        }
    }

    public void checkAccess(Token<BlockTokenIdentifier> token, String userId, ExtendedBlock block, BlockTokenIdentifier.AccessMode mode) throws SecretManager.InvalidToken {
        BlockTokenIdentifier id = new BlockTokenIdentifier();
        try {
            id.readFields((DataInput)new DataInputStream(new ByteArrayInputStream(token.getIdentifier())));
        }
        catch (IOException e) {
            throw new SecretManager.InvalidToken("Unable to de-serialize block token identifier for user=" + userId + ", block=" + block + ", access mode=" + mode);
        }
        this.checkAccess(id, userId, block, mode);
        if (!MessageDigest.isEqual(this.retrievePassword(id), token.getPassword())) {
            throw new SecretManager.InvalidToken("Block token with " + id + " doesn't have the correct token password");
        }
    }

    private static boolean isExpired(long expiryDate) {
        return Time.now() > expiryDate;
    }

    static boolean isTokenExpired(Token<BlockTokenIdentifier> token) throws IOException {
        ByteArrayInputStream buf = new ByteArrayInputStream(token.getIdentifier());
        DataInputStream in = new DataInputStream(buf);
        long expiryDate = WritableUtils.readVLong((DataInput)in);
        return BlockTokenSecretManager.isExpired(expiryDate);
    }

    public void setTokenLifetime(long tokenLifetime) {
        this.tokenLifetime = tokenLifetime;
    }

    public BlockTokenIdentifier createIdentifier() {
        return new BlockTokenIdentifier();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected byte[] createPassword(BlockTokenIdentifier identifier) {
        BlockKey key = null;
        BlockTokenSecretManager blockTokenSecretManager = this;
        synchronized (blockTokenSecretManager) {
            key = this.currentKey;
        }
        if (key == null) {
            throw new IllegalStateException("currentKey hasn't been initialized.");
        }
        identifier.setExpiryDate(this.timer.now() + this.tokenLifetime);
        identifier.setKeyId(key.getKeyId());
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Generating block token for " + identifier.toString()));
        }
        return BlockTokenSecretManager.createPassword((byte[])identifier.getBytes(), (SecretKey)key.getKey());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public byte[] retrievePassword(BlockTokenIdentifier identifier) throws SecretManager.InvalidToken {
        if (BlockTokenSecretManager.isExpired(identifier.getExpiryDate())) {
            throw new SecretManager.InvalidToken("Block token with " + identifier.toString() + " is expired.");
        }
        BlockKey key = null;
        BlockTokenSecretManager blockTokenSecretManager = this;
        synchronized (blockTokenSecretManager) {
            key = this.allKeys.get(identifier.getKeyId());
        }
        if (key == null) {
            throw new SecretManager.InvalidToken("Can't re-compute password for " + identifier.toString() + ", since the required block key (keyID=" + identifier.getKeyId() + ") doesn't exist.");
        }
        return BlockTokenSecretManager.createPassword((byte[])identifier.getBytes(), (SecretKey)key.getKey());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DataEncryptionKey generateDataEncryptionKey() {
        byte[] nonce = new byte[8];
        this.nonceGenerator.nextBytes(nonce);
        BlockKey key = null;
        BlockTokenSecretManager blockTokenSecretManager = this;
        synchronized (blockTokenSecretManager) {
            key = this.currentKey;
        }
        byte[] encryptionKey = BlockTokenSecretManager.createPassword((byte[])nonce, (SecretKey)key.getKey());
        return new DataEncryptionKey(key.getKeyId(), this.blockPoolId, nonce, encryptionKey, this.timer.now() + this.tokenLifetime, this.encryptionAlgorithm);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public byte[] retrieveDataEncryptionKey(int keyId, byte[] nonce) throws InvalidEncryptionKeyException {
        BlockKey key = null;
        BlockTokenSecretManager blockTokenSecretManager = this;
        synchronized (blockTokenSecretManager) {
            key = this.allKeys.get(keyId);
            if (key == null) {
                throw new InvalidEncryptionKeyException("Can't re-compute encryption key for nonce, since the required block key (keyID=" + keyId + ") doesn't exist. Current key: " + this.currentKey.getKeyId());
            }
        }
        return BlockTokenSecretManager.createPassword((byte[])nonce, (SecretKey)key.getKey());
    }

    public BlockKey getCurrentKey() {
        return this.currentKey;
    }

    @VisibleForTesting
    public synchronized void setKeyUpdateIntervalForTesting(long millis) {
        this.keyUpdateInterval = millis;
    }

    @VisibleForTesting
    public void clearAllKeysForTesting() {
        this.allKeys.clear();
    }

    @VisibleForTesting
    public synchronized boolean hasKey(int keyId) {
        BlockKey key = this.allKeys.get(keyId);
        return key != null;
    }

    @VisibleForTesting
    public synchronized int getSerialNoForTesting() {
        return this.serialNo;
    }
}

