// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

import org.codehaus.groovy.runtime.IOGroovyMethods

suite("test_index_compaction_unique_keys_arr", "array_contains_inverted_index, nonConcurrent") {
    // here some variable to control inverted index query
    sql """ set enable_profile=true"""
    sql """ set enable_pipeline_x_engine=true;"""
    sql """ set enable_inverted_index_query=true"""
    sql """ set enable_common_expr_pushdown=true """
    sql """ set enable_common_expr_pushdown_for_inverted_index=true """

    def isCloudMode = isCloudMode()
    def tableName = "test_index_compaction_unique_keys_arr"
    def backendId_to_backendIP = [:]
    def backendId_to_backendHttpPort = [:]
    getBackendIpHttpPort(backendId_to_backendIP, backendId_to_backendHttpPort);

    boolean disableAutoCompaction = false

    def set_be_config = { key, value ->
        for (String backend_id: backendId_to_backendIP.keySet()) {
            def (code, out, err) = update_be_config(backendId_to_backendIP.get(backend_id), backendId_to_backendHttpPort.get(backend_id), key, value)
            logger.info("update config: code=" + code + ", out=" + out + ", err=" + err)
        }
    }

    def get_rowset_count = { tablets ->
        int rowsetCount = 0
        for (def tablet in tablets) {
            def (code, out, err) = curl("GET", tablet.CompactionStatus)
            logger.info("Show tablets status: code=" + code + ", out=" + out + ", err=" + err)
            assertEquals(code, 0)
            def tabletJson = parseJson(out.trim())
            assert tabletJson.rowsets instanceof List
            rowsetCount +=((List<String>) tabletJson.rowsets).size()
        }
        return rowsetCount
    }

    def check_config = { String key, String value ->
        for (String backend_id: backendId_to_backendIP.keySet()) {
            def (code, out, err) = show_be_config(backendId_to_backendIP.get(backend_id), backendId_to_backendHttpPort.get(backend_id))
            logger.info("Show config: code=" + code + ", out=" + out + ", err=" + err)
            assertEquals(code, 0)
            def configList = parseJson(out.trim())
            assert configList instanceof List
            for (Object ele in (List) configList) {
                assert ele instanceof List<String>
                if (((List<String>) ele)[0] == key) {
                    assertEquals(value, ((List<String>) ele)[2])
                }
            }
        }
    }

    def run_test = { table_name ->

        sql """ INSERT INTO ${table_name} VALUES (1, "andy", "andy love apple", 100); """
        sql """ INSERT INTO ${table_name} VALUES (1, "bason", "bason hate pear", 99); """
        sql """ INSERT INTO ${table_name} VALUES (2, "andy", "andy love apple", 100); """
        sql """ INSERT INTO ${table_name} VALUES (2, "bason", "bason hate pear", 99); """
        sql """ INSERT INTO ${table_name} VALUES (3, "andy", "andy love apple", 100); """
        sql """ INSERT INTO ${table_name} VALUES (3, "bason", "bason hate pear", 99); """

        qt_sql """ select * from ${table_name} order by id, name, hobbies, score """
        qt_sql """ select * from ${table_name} where name match "andy" order by id, name, hobbies, score """
        qt_sql """ select * from ${table_name} where hobbies match "pear" order by id, name, hobbies, score """
        qt_sql """ select * from ${table_name} where score < 100 order by id, name, hobbies, score """

        //TabletId,ReplicaId,BackendId,SchemaHash,Version,LstSuccessVersion,LstFailedVersion,LstFailedTime,LocalDataSize,RemoteDataSize,RowCount,State,LstConsistencyCheckTime,CheckVersion,VersionCount,PathHash,MetaUrl,CompactionStatus
        def tablets = sql_return_maparray """ show tablets from ${table_name}; """
        def dedup_tablets = deduplicate_tablets(tablets)

        // In the p0 testing environment, there are no expected operations such as scaling down BE (backend) services
        // if tablets or dedup_tablets is empty, exception is thrown, and case fail
        int replicaNum = Math.floor(tablets.size() / dedup_tablets.size())
        if (replicaNum != 1 && replicaNum != 3)
        {
            assert(false);
        }

        // before full compaction, there are 7 rowsets.
        int rowsetCount = get_rowset_count.call(tablets);
        assert (rowsetCount == 7 * replicaNum)

        // trigger full compactions for all tablets in ${table_name}
        trigger_and_wait_compaction(table_name, "full")

        // after full compaction, there is only 1 rowset.
        rowsetCount = get_rowset_count.call(tablets);
        if (isCloudMode) {
            assert (rowsetCount == (1 + 1) * replicaNum)
        } else {
            assert (rowsetCount == 1 * replicaNum)
        }

        qt_sql """ select * from ${table_name} order by id, name, hobbies, score """
        qt_sql """ select * from ${table_name} where name match "andy" order by id, name, hobbies, score """
        qt_sql """ select * from ${table_name} where hobbies match "pear" order by id, name, hobbies, score """
        qt_sql """ select * from ${table_name} where score < 100 order by id, name, hobbies, score """

        // insert more data and trigger full compaction again
        sql """ INSERT INTO ${table_name} VALUES (1, "andy", "andy love apple", 100); """
        sql """ INSERT INTO ${table_name} VALUES (1, "bason", "bason hate pear", 99); """
        sql """ INSERT INTO ${table_name} VALUES (2, "andy", "andy love apple", 100); """
        sql """ INSERT INTO ${table_name} VALUES (2, "bason", "bason hate pear", 99); """
        sql """ INSERT INTO ${table_name} VALUES (3, "andy", "andy love apple", 100); """
        sql """ INSERT INTO ${table_name} VALUES (3, "bason", "bason hate pear", 99); """

        qt_sql """ select * from ${table_name} order by id, name, hobbies, score """
        qt_sql """ select * from ${table_name} where name match "andy" order by id, name, hobbies, score """
        qt_sql """ select * from ${table_name} where hobbies match "pear" order by id, name, hobbies, score """
        qt_sql """ select * from ${table_name} where score < 100 order by id, name, hobbies, score """

        rowsetCount = get_rowset_count.call(tablets);
        if (isCloudMode) {
            assert (rowsetCount == (7 + 1) * replicaNum)
        } else {
            assert (rowsetCount == 7 * replicaNum)
        }

        // trigger full compactions for all tablets in ${table_name}
        trigger_and_wait_compaction(table_name, "full")

        // after full compaction, there is only 1 rowset.
        rowsetCount = get_rowset_count.call(tablets);
        if (isCloudMode) {
            assert (rowsetCount == (1 + 1) * replicaNum)
        } else {
            assert (rowsetCount == 1 * replicaNum)
        }

        qt_sql """ select * from ${table_name} order by id, name, hobbies, score """
        qt_sql """ select * from ${table_name} where name match "andy" order by id, name, hobbies, score """
        qt_sql """ select * from ${table_name} where hobbies match "pear" order by id, name, hobbies, score """
        qt_sql """ select * from ${table_name} where score < 100 order by id, name, hobbies, score """
    }

    boolean invertedIndexCompactionEnable = false
    boolean has_update_be_config = false
    try {
        String backend_id;
        backend_id = backendId_to_backendIP.keySet()[0]
        def (code, out, err) = show_be_config(backendId_to_backendIP.get(backend_id), backendId_to_backendHttpPort.get(backend_id))

        logger.info("Show config: code=" + code + ", out=" + out + ", err=" + err)
        assertEquals(code, 0)
        def configList = parseJson(out.trim())
        assert configList instanceof List

        for (Object ele in (List) configList) {
            assert ele instanceof List<String>
            if (((List<String>) ele)[0] == "inverted_index_compaction_enable") {
                invertedIndexCompactionEnable = Boolean.parseBoolean(((List<String>) ele)[2])
                logger.info("inverted_index_compaction_enable: ${((List<String>) ele)[2]}")
            }
            if (((List<String>) ele)[0] == "disable_auto_compaction") {
                disableAutoCompaction = Boolean.parseBoolean(((List<String>) ele)[2])
                logger.info("disable_auto_compaction: ${((List<String>) ele)[2]}")
            }
        }
        set_be_config.call("inverted_index_compaction_enable", "true")
        has_update_be_config = true
        // check updated config
        check_config.call("inverted_index_compaction_enable", "true");

        sql """ DROP TABLE IF EXISTS ${tableName}; """
        sql """
            CREATE TABLE ${tableName} (
                `id` int(11) NULL,
                `name` varchar(255) NULL,
                `hobbies` text NULL,
                `score` int(11) NULL,
                index index_name (name) using inverted,
                index index_hobbies (hobbies) using inverted properties("support_phrase" = "true", "parser" = "english", "lower_case" = "true"),
                index index_score (score) using inverted
            ) ENGINE=OLAP
            UNIQUE KEY(`id`)
            COMMENT 'OLAP'
            DISTRIBUTED BY HASH(`id`) BUCKETS 1
            PROPERTIES (
                "replication_num" = "1",
                "disable_auto_compaction" = "true",
                "enable_unique_key_merge_on_write" = "true"
            );
        """

        run_test.call(tableName)

        // cluster key is not supported in cloud mode at branch-3.0
        if (false) {
            tableName = "test_index_compaction_unique_keys_arr_cluster_key"
            sql """ DROP TABLE IF EXISTS ${tableName}; """
            sql """
                CREATE TABLE ${tableName} (
                    `id` int(11) NULL,
                    `name` varchar(255) NULL,
                    `hobbies` text NULL,
                    `score` int(11) NULL,
                    index index_name (name) using inverted,
                    index index_hobbies (hobbies) using inverted properties("parser"="english"),
                    index index_score (score) using inverted
                ) ENGINE=OLAP
                UNIQUE KEY(`id`)
                CLUSTER BY (`score`)
                COMMENT 'OLAP'
                DISTRIBUTED BY HASH(`id`) BUCKETS 1
                PROPERTIES ( 
                    "replication_num" = "1",
                    "disable_auto_compaction" = "true",
                    "enable_unique_key_merge_on_write" = "true"
                );
            """
            run_test.call(tableName)
        }
    } finally {
        if (has_update_be_config) {
            set_be_config.call("inverted_index_compaction_enable", invertedIndexCompactionEnable.toString())
        }
    }
}
